# Copyright (c) 2020 Luca Koczula
# Copyright (c) 2020 Tobias Wagner


# Genetic Algorithm based on the paper by Li et al. (2014)

from random import choice, choices, sample
from operator import itemgetter
from inform_or_gabm_objects.chromosome import Chromosome
from .best_match_heuristic import best_match_heuristic
import time


def crossover(mating_pool, current_generation, items):
    """ This function handles the crossover/mutation of two chromosomes of the mating pool.

    Parameters
    ----------
    mating_pool
        The list of chromosomes to mutate/crossover.

    current_generation
        Number specifying the current generation.

    items
        The items of the current order.

    Returns
    -------
        A list containing two children chromosomes.
    """
    cutting_points = sample(range(0,len(items)), k=2)
    cutting_points.sort()

    # parents for the two offsprings
    children = []
    for parent in mating_pool:
        position1 = cutting_points[0]
        if mating_pool.index(parent) == 0:
            parent1 = mating_pool[0]
            parent2 = mating_pool[1]
        else:
            parent1 = mating_pool[1]
            parent2 = mating_pool[0]

        # generate empty lists for children
        bps = []
        cls = []
        for _ in range(len(items)):
            bps.append('')
            cls.append('')

        # copy genes between cutting_points
        while position1 <= cutting_points[1]:
            bps[position1] = parent1.BPS[position1]
            cls[position1] = parent1.CLS[position1]
            position1 += 1

        # Filling the open position with the missing genes (circularly from second cutting point)
        for element in parent2.BPS:
            if position1 >= len(items):
                position1 = 0
            if element not in bps:
                bps[position1] = element
                cls[position1] = parent2.CLS[parent2.BPS.index(element)]
                position1 += 1

        # renumbering the boxes (first box - 1, last box - len(items))
        for box in range(len(cls)):
            cls[box] = (cls[box][0], box+1)
        
        children.append(Chromosome(bps, cls, current_generation))

    return children


def selection(generation, current_generation, items, prob_c, prob_t, E):
    """ This function selects the chromosomes of the new generation.

    Parameters
    ----------
    generation
        List of all chromosomes in the current generation.

    current_generation
        Number specifying the current generation.

    items
        List of all items in the current order.

    prob_c
        Probability prob_c.

    prob_t
        Probability prob_t.

    E
        Number of best chromosomes that directly advance to the next generation.

    Returns
    -------
        Next generation of chromosomes.
    """
    next_generation = generation[0:E]
    del generation[0:E]

    # selection for rest generation
    while generation != []:
        # generating mating pool
        mating_pool = []
        if len(generation) <= 2:
            mating_pool = generation.copy()
            del generation[:]
        else:
            j = 0
            while j <= 1:
                tournament = sample(generation, k=2)
                y = choices(tournament, weights=[prob_t, 1-prob_t], k=1)[0]
                mating_pool.append(y)
                generation.remove(y)
                j += 1

        # Mutation: 'no' or 'yes'
        decision = choices(['no', 'yes'], weights=[prob_c, 1-prob_c], k=1)[0]
        if decision == 'no':
            # chosen chromosomes in mating_pool directly proceed into next generation
            next_generation = next_generation + mating_pool
        else:
            # chosen chromosomes in mating_pool are generating two offsprings by crossover/mutation
            children = crossover(mating_pool, current_generation, items)
            next_generation = next_generation + children.copy()
            del children[:]

    generation = next_generation.copy()
    del next_generation[:]

    print("PROCESSING: generation: " + str(current_generation) + ', # of chromosomes: ' + str(len(generation)))

    return generation


def population_initialization(items, order, population_size, boxes, orders, length_item, width_item, height_item, weight_item):
    """ This function is used to initialise the first generation of chromosomes.

    Parameters
    ----------
    items
        List of all items to be packed in the current order.

    order
        ID of the current order to be packed.

    population_size
        Number of chromosomes in each generation.

    boxes
        List of all available box types to pack items.

    orders
        Dictionary containing all orders.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    weight_item
        Dictionary containing the weight of all items.

    Returns
    -------
        The first generation of chromosomes for the optimisation process.
        Number specifying the first generation.
    """
    # Initialization of first generation 5 ordered instances, 95 random instances
    current_generation = 1
    
    generation = []
    e = 1
    while e <= population_size:
        cls = []
        if e == 1:
            # instances ordered by length, width, height, volume, weight
            sortedLength, sortedWidth, sortedHeight, sortedVolume, sortedWeight = ordering(order=order, orders=orders, length_item=length_item, width_item=width_item, height_item=height_item, weight_item=weight_item)
            sortedLists = [sortedLength, sortedWidth, sortedHeight, sortedVolume, sortedWeight]
            for currList in sortedLists:
                cls = []
                for j in range(1, len(items)+1):
                    cls_box = choice(boxes)
                    cls.append((cls_box, j))
                generation.append(Chromosome(currList, cls, current_generation))
            e = len(sortedLists) + 1
            pass
        else:
            # random instances
            bps = sample(items, k=len(items))
            for j in range(1, len(items)+1):
                    cls_box = choice(boxes)
                    cls.append((cls_box, j))
            generation.append(Chromosome(bps, cls, current_generation))
            e += 1

    return generation, current_generation


def ordering(order, orders, length_item, width_item, height_item, weight_item):
    """ This function creates five sorted lists of items to initialise the first generation of chromosomes.
        The sorted criteria for the lists are either length, width, height, volume or weight.

    Parameters
    ----------
    order
        The order to be considered in the following optimisation.

    orders
        Dictionary containing all orders.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    weight_item
        Dictionary containing the weight of all items.

    Returns
    -------
        Five lists of items sorted by either their length, width, height, volume or weight.
    """
    # create list itemsOfOrder containing all items of order 'order' including their itemID, length, width, height, volume and weight
    itemsOfOrder = []
    for currentItem in orders[order]:
        itemID = currentItem[0]
        itemVolumme = length_item[itemID] * width_item[itemID] * height_item[itemID]
        for item in range(currentItem[1]):
            itemsOfOrder.append((itemID, length_item[itemID], width_item[itemID], height_item[itemID], itemVolumme, weight_item[itemID]))

    sortedLength = []
    sortedWidth = []
    sortedHeight = []
    sortedVolume = []
    sortedWeight = []
    
    # sort lists by lenght, width, height, volume and weight, format them appropriately and add a counter for each item
    sortedLists = [sortedLength, sortedWidth, sortedHeight, sortedVolume, sortedWeight]
    index = 1
    currentItemID = ''
    for currentList in sortedLists:
        counter = 1
        itemsOfOrder.sort(key=itemgetter(index), reverse=True)
        tmp = itemsOfOrder[:]
        index += 1
        for item in tmp:
            if currentItemID != item[0]:
                currentItemID = item[0]
                counter = 1
            currentList.append((item[0], counter))
            counter += 1
    
    return sortedLength, sortedWidth, sortedHeight, sortedVolume, sortedWeight


def genetic_algorithm(generations, population_size, probc, probt, E, considered_items, consideredEMS, order, items,
                      boxes, length_item, width_item, height_item, length_box, width_box, height_box, weight_item, orders):
    """ This function is the main part of the genetic algorithm and is used to start the Best Match Heuristic.

    Parameters
    ----------
    generations
        The number of generations created by the genetic algorithm.

    population_size
        Number of chromosomes for each generation.

    probc
        Probability prob_c.

    probt
        Probability prob_t.

    E
        Number of best chromosomes that directly advance to the next generation.

    considered_items
        Number of items considered in each step of the heuristic.

    consideredEMS
        Number of EMS considered in each iteration.

    order
        ID of the order to be optimised.

    items
        Items of the current order.

    boxes
        List of all available boxes.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    length_box
        Dictionary containing the length of all boxes.

    width_box
        Dictionary containing the width of all boxes.

    height_box
        Dictionary containing the height of all boxes.

    weight_item
        Dictionary containing the weight of all items.

    orders
        Dictionary containing all orders.

    Returns
    -------
        Best possible solution found over all chromosomes.
        List of all items packed.
        List of all boxes used.
        Time needed to optimise the current optimisation problem.
    """
    start_time = time.time()

    generation, current_generation = population_initialization(items=items,
                                                               order=order,
                                                               population_size=population_size,
                                                               boxes=boxes,
                                                               orders=orders,
                                                               length_item=length_item,
                                                               width_item=width_item,
                                                               height_item=height_item,
                                                               weight_item=weight_item)
    generation = best_match_heuristic(generation=generation,
                                      considered_items=considered_items,
                                      considered_EMS=consideredEMS,
                                      length_item=length_item,
                                      width_item=width_item,
                                      height_item=height_item,
                                      length_box=length_box,
                                      width_box=width_box,
                                      height_box=height_box,
                                      weight_item=weight_item,
                                      best_chromosome=False)

    while current_generation <= generations:
        generation = selection(generation=generation,
                               current_generation=current_generation,
                               items=items,
                               prob_c=probc,
                               prob_t=probt,
                               E=E)
        generation = best_match_heuristic(generation=generation,
                                          considered_items=considered_items,
                                          considered_EMS=consideredEMS,
                                          length_item=length_item,
                                          width_item=width_item,
                                          height_item=height_item,
                                          length_box=length_box,
                                          width_box=width_box,
                                          height_box=height_box,
                                          weight_item=weight_item,
                                          best_chromosome=False)
        current_generation += 1
    
    elapsed_time = time.time() - start_time
    last_chromosome = generation[:1]
    best_chromosome, item_list, box_list = best_match_heuristic(generation=last_chromosome,
                                                                considered_items=considered_items,
                                                                considered_EMS=consideredEMS,
                                                                length_item=length_item,
                                                                width_item=width_item,
                                                                height_item=height_item,
                                                                length_box=length_box,
                                                                width_box=width_box,
                                                                height_box=height_box,
                                                                weight_item=weight_item,
                                                                best_chromosome=True)

    return best_chromosome, item_list, box_list, elapsed_time
