# -*- coding: utf-8 -*-
"""Extreme Points-based heuristic."""

#  Copyright (c) 2020. Yordan Manolov <yordan DOT manolov AT rwth DASH aachen DOT de >

from random import choice, randint
from argparse import Namespace
from typing import List

from inform_or_utility import sort_items
from inform_or_ep_objects.box import Box, MAX_BOX_ID_NUMBER, STANDARD_SIZES
from inform_or_ep_objects.item import Item
from inform_or_ep_objects.order import Order


def extreme_points(order: Order, packing_instructions_file: str, args: Namespace) -> List[Box]:
    """
    This is the entry point function in the extreme points heuristic module.
    It sorts all items from an incoming order, then
    sends the sorted list for the calculation of the item -> box
    mapping to the desired function (first fit or best fit variant).
    This function also outputs the convention to follow
    when reading the packing instructions.

    Parameters
    ----------
    order: The current order for packaging.
    args: Additional arguments such as desired sorting criterion,
    desired merit function (if best-fit desired).
    packing_instructions_file: The file to which to write packing instructions.

    Returns
    -------
    A list of boxes required to pack all items from the current order.
    """

    generate_packing_instructions_per_order(order, packing_instructions_file)

    if args.sorting == "volume":
        sorted_items = sort_items.item_sort_volume(order.itemlist)
    elif args.sorting == "volume_height":
        sorted_items = sort_items.item_sort_volume_height(order.itemlist)
    elif args.sorting == "height_volume":
        sorted_items = sort_items.item_sort_height_volume(order.itemlist)
    elif args.sorting == "area":
        sorted_items = sort_items.item_sort_area(order.itemlist)
    elif args.sorting == "area_height":
        sorted_items = sort_items.item_sort_area_height(order.itemlist)
    elif args.sorting == "height_area":
        sorted_items = sort_items.item_sort_height_area(order.itemlist)
    elif args.sorting == "height":
        sorted_items = sort_items.item_sort_height(order.itemlist)

    if args.merit_function is None:
        boxes = ep_first_fit_decreasing(sorted_items)
    else:
        boxes = ep_best_fit_decreasing(sorted_items, args.merit_function)
    for box in boxes:
        box.generate_packing_instructions_per_box(packing_instructions_file)
    return boxes


def ep_best_fit_decreasing(items: List[Item], merit_function: str) -> List[Box]:
    """
    The best-fit variant of the heuristic.

    Parameters
    ----------
    merit_function : the merit function to use for choosing best boxes.
    TODO: reimplement f_rs exactly as in paper sketch
    items: A list of items to pack into boxes

    Returns
    -------
    A list of boxes required to pack the items
    """

    boxes = [(grab_box(items[0]))]
    for item in items:
        for remainder in range(item.count):
            boxes_fitting = [b for b in boxes
                             if b.test_fits(item.length, item.width, item.height)]
            if len(boxes_fitting) < 1:
                boxes.append(grab_box(item))
                boxes_fitting.append(boxes[-1])

            if merit_function == "fv":
                boxes_optimal = merit_fv(boxes_fitting, item)
            elif merit_function == "lev":
                boxes_optimal = merit_lev(boxes_fitting, item)
            elif merit_function == "mp":
                boxes_optimal = merit_mp(boxes_fitting, item)
            elif merit_function == "rs":
                boxes_optimal = merit_rs(boxes_fitting, item)
            boxes_optimal[0].place_item(item)
            boxes_optimal[0].update_eps(item.length, item.width, item.height)
    return boxes


def ep_first_fit_decreasing(items: List[Item]) -> List[Box]:
    """
    The first-fit variant of the heuristic.

    Parameters
    ----------
    items A list of items to pack into boxes

    Returns
    -------
A list of boxes required to pack the items
    """

    boxes = [grab_box(items[0])]
    for item in items:
        for remainder in range(item.count):
            boxes_fitting = list(filter(lambda b: b.test_fits(item.length,
                                                              item.width,
                                                              item.height),
                                        boxes))
            if len(boxes_fitting) < 1:
                boxes.append(grab_box(item))
                boxes_fitting = boxes
            first = boxes_fitting[-1]
            first.place_item(item)
            first.update_eps(item.length, item.width, item.height)
    return boxes


def generate_packing_instructions_per_order(order: Order, output_file: str) -> None:
    """

    Parameters
    ----------
    order The order for which to write an ID in the packing instructions
    output_file The file which should contain the packing instructions

    Returns
    -------

    """
    with open(output_file, "a") as output:
        output.writelines("Processing order no. {} \n".format(order.identifier))


def grab_box(for_item: Item) -> Box:
    """
    Selects a new box for some item.

    Parameters
    ----------
    for_item: The item for which a new box is picked. Used for checking
    whether box and item dimensions match

    Returns
    -------
    A Box object of the specified size
    """
    fitting_size = grab_box_size(for_item)
    new_box = Box("B" + str(randint(0, MAX_BOX_ID_NUMBER)),
                  standard_size=fitting_size)
    return new_box


def grab_box_size(i: Item) -> str:
    """
    Determine one std size which can accommodate the input item. Starts with
    the biggest available size

    Parameters
    ----------
    i The item for which an appropriate box size is determined

    Returns
    -------

    """
    for key, dimension_vector in sorted(STANDARD_SIZES.items(), reverse=True):
        comparison_dimensions = zip(dimension_vector,
                                    [i.length, i.width, i.height])
        if all(x >= y for (x, y) in comparison_dimensions):
            return key
        elif key == list(STANDARD_SIZES.keys())[0]:
            raise KeyError("Don't put an elephant in a boat;"
                           " item too big for any box.")


def merit_fv(boxes: List[Box], item: Item) -> List[Box]:
    """
    Merit function free volume.

    Parameters
    ----------
    boxes: A list of non-full boxes capable of accommodating the current item
    item: The current item to be placed in a box

    Returns
    -------
    A list of the boxes maximizing the merit function
    """
    free_volume_left = [b.get_free_volume() - item.volume for b in boxes]
    return [b for b in boxes
            if b.get_free_volume() - item.volume == min(free_volume_left)]


def merit_mp(boxes: List[Box], item: Item) -> List[Box]:
    """
    Merit function packing size.

    Parameters
    ----------
    boxes: A list of non-full boxes capable of accommodating the current item
    item: The current item to be placed in a box

    Returns
    -------
    A list of the boxes maximizing the merit function
    """
    packing_size = item.length * item.width
    maximum_packing_size = [(b.name, (b.current_ep[0] * b.current_ep[1] + packing_size)) for b in boxes]
    optimum = min(maximum_packing_size)
    return [b for b in boxes if b.name == optimum[0]]


def merit_lev(boxes: List[Box], item: Item) -> List[Box]:
    """
    Merit function leveled free volume.

    Parameters
    ----------
    boxes: A list of non-full boxes capable of accommodating the current item
    item: The current item to be placed in a box

    Returns
    -------
    A list of the boxes maximizing the merit function
    """

    new_level = item.length * item.height
    maximum_packing_size_leveled = [(b.name, b.current_ep[1] * b.current_ep[2] + new_level) for b in boxes]
    optimum = min(maximum_packing_size_leveled)
    return [b for b in boxes if b.name == optimum[0]]


def merit_rs(boxes: List[Box], item: Item) -> List[Box]:
    """
    Merit function residual spaces.

    Parameters
    ----------
    boxes: A list of non-full boxes capable of accommodating the current item
    item: The current item to be placed in a box

    Returns
    -------
    A list of the boxes maximizing the merit function
    """

    maximum_merit = dict()
    merits = list()
    residual_space = dict()
    rs_after_insertion = dict()
    for box in boxes:
        maximum_merit[box] = max(e for e in box.current_ep)
        for point in box.current_ep:
            residual_space[point] = [box.length - box.current_ep[0],
                                     box.width - box.current_ep[1],
                                     box.height - box.current_ep[2]]
            rs_after_insertion[point] = [
                item.length - residual_space[point][0],
                item.width - residual_space[point][1],
                item.height - residual_space[point][2]]
            maximum_merit[box] = min(maximum_merit[box],
                                     min(rs_after_insertion[point]))
            merits.append(maximum_merit[box])
    return [b for b in boxes if maximum_merit[b] == min(merits)]


def print_instruction_description_to_file(packing_instructions_file: str) \
        -> None:
    """

    Parameters
    ----------
    packing_instructions_file The file containing the packing instructions
    when the optimal mapping is known

    Returns
    -------

    """
    with open(packing_instructions_file, "a") as output:
        output.writelines("Convention of packing instructions: order, box, "
                          "(item ID of the item for the box, "
                          "number of items of the same ID). \n")
        output.writelines("The order of the items in the output is equivalent "
                          "to the actual packing order. \n")
