# Copyright (c) 2020 Luca Koczula
# Copyright (c) 2020 Tobias Wagner

# Best Match Heuristic based on the paper by Li et al. (2014)

from inform_or_gabm_objects.ems import EMS
from inform_or_gabm_objects.item import Item


# list for all ems (first element refers to box)
ems_list = []
item_list = []
box_list = []


def initialize_EMS(box, length_box, width_box, height_box):
    """ This function is used to initialise the first EMS of a box.
        This EMS has the exact same dimensions as the (empty) box.

    Parameters
    ----------
    box
        The box to initialise the EMS for.

    length_box
        Dictionary containing the length of all boxes.

    width_box
        Dictionary containing the width of all boxes.

    height_box
        Dictionary containing the height of all boxes.

    Returns
    -------
    An EMS with the dimensions of the box containing the EMS.
    """
    global boxcounter
    boxcounter += 1
    return EMS(boxcounter, length_box[box[0]], width_box[box[0]], height_box[box[0]], 0, 0, 0)


def delete_empty_EMS():
    """ This function will delete all EMS that got created with a volume <= 0.

    Returns
    -------
    """
    for ems in ems_list:
        if ems.volume <= 0:
            ems_list.remove(ems)
            

def check_EMS_intersect(curr_ems, change_x, change_y, change_z):
    """ This function updates EMS that are intersected by a newly placed item.

    Parameters
    ----------
    curr_ems
        The current EMS, that was used to place the new item.

    change_x
        Dimension of the item in x direction.

    change_y
        Dimension of the item in y direction.

    change_z
        Dimension of the item in z direction.

    Returns
    -------
    """
    item_max_x = curr_ems.min_x + change_x
    item_max_y = curr_ems.min_y + change_y
    item_max_z = curr_ems.min_z + change_z

    ems_list_copy = ems_list[:]
    for ems in ems_list_copy:
        if ems != curr_ems:
            if ems.box == curr_ems.box:
                if (ems.min_x < item_max_x <= ems.max_x) and (ems.min_y < item_max_y <= ems.max_y) and (ems.min_z < item_max_z <= ems.max_z):
                    # six new EMS, limited by the different item dimension
                    create_new_EMS(curr_ems, ems, item_max_x, item_max_y, item_max_z)
                    ems_list.remove(ems)

                if (ems.min_x <= curr_ems.min_x < ems.max_x) and (ems.min_y <= curr_ems.min_y < ems.max_y) and (ems.min_z <= curr_ems.min_z < ems.max_z):
                    # six new EMS, limited by the different item dimension
                    create_new_EMS(curr_ems, ems, item_max_x, item_max_y, item_max_z)
                    if ems in ems_list:
                        ems_list.remove(ems)

                delete_empty_EMS()


def create_new_EMS(curr_ems, ems, item_max_x, item_max_y, item_max_z):
    """ Creates 6 new EMS, limited by the different item dimensions.

    Parameters
    ----------
    curr_ems
        The current EMS, that was used to place the new item.

    ems
        List of all EMS.

    item_max_x
        New maximum x dimension.

    item_max_y
        New maximum y dimension.

    item_max_z
        New maximum z dimension.

    Returns
    -------
    """
    # min_x
    ems_list.append(EMS(ems.box, curr_ems.min_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
    # min_y
    ems_list.append(EMS(ems.box, ems.max_x, curr_ems.min_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
    # min_z
    ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, curr_ems.min_z, ems.min_x, ems.min_y, ems.min_z))
    # max_x
    ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, item_max_x, ems.min_y, ems.min_z))
    # max_y
    ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, item_max_y, ems.min_z))
    # max_z
    ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, item_max_z))


def check_EMS_pierce(curr_ems, change_x, change_y, change_z):
    """ This function updates EMS that are pierced by newly placed items.

    Parameters
    ----------
    curr_ems
        The current EMS, that was used to place the new item.

    change_x
        Dimension of the item in x direction.

    change_y
        Dimension of the item in y direction.

    change_z
        Dimension of the item in z direction.

    Returns
    -------
    """
    item_max_x = curr_ems.min_x + change_x
    item_max_y = curr_ems.min_y + change_y
    item_max_z = curr_ems.min_z + change_z

    ems_list_copy = ems_list[:]
    for ems in ems_list_copy:
        if ems != curr_ems:
            if ems.box == curr_ems.box:
                # min_coordinate
                if ems.min_x <= curr_ems.min_x < ems.max_x and ems.min_y <= curr_ems.min_y < ems.max_y:
                    if curr_ems.min_z <= ems.min_z < item_max_z:
                        # three new EMS, limited by the different item dimension
                        # max_z (behind)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, item_max_z))
                        # min_y (left)
                        ems_list.append(EMS(ems.box, ems.max_x, curr_ems.min_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
                        # min_x (below)
                        ems_list.append(EMS(ems.box, curr_ems.min_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
                        # delete EMS
                        if ems in ems_list:
                            ems_list.remove(ems)

                if ems.min_x <= curr_ems.min_x < ems.max_x and ems.min_z <= curr_ems.min_z < ems.max_z:
                    if curr_ems.min_y <= ems.min_y < item_max_y:
                        # three new EMS, limited by the different item dimension
                        # max_y (behind)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, item_max_y, ems.min_z))
                        # min_x (left)
                        ems_list.append(EMS(ems.box, curr_ems.min_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
                        # min_z (below)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, curr_ems.min_z, ems.min_x, ems.min_y, ems.min_z))
                        # delete EMS
                        if ems in ems_list:
                            ems_list.remove(ems)

                if ems.min_y <= curr_ems.min_y < ems.max_y and ems.min_z <= curr_ems.min_z < ems.max_z:
                    if curr_ems.min_x <= ems.min_x < item_max_x:
                        # three new EMS, limited by the different item dimension
                        # max_x (behind)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, item_max_x, ems.min_y, ems.min_z))
                        # min_y (left)
                        ems_list.append(EMS(ems.box, ems.max_x, curr_ems.min_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
                        # min_z (below)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, curr_ems.min_z, ems.min_x, ems.min_y, ems.min_z))
                        # delete EMS
                        if ems in ems_list:
                            ems_list.remove(ems)

                # max_coordinate
                if ems.min_x < item_max_x <= ems.max_x and ems.min_y < item_max_y <= ems.max_y:
                    if curr_ems.min_z < ems.max_z <= item_max_z:
                        # three new EMS, limited by the different item dimension
                        # min_z (in front)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, curr_ems.min_z, ems.min_x, ems.min_y, ems.min_z))
                        # max_y (right)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, item_max_y, ems.min_z))
                        # max_x (on top)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, item_max_x, ems.min_y, ems.min_z))
                        # delete EMS
                        if ems in ems_list:
                            ems_list.remove(ems)

                if ems.min_x < item_max_x <= ems.max_x and ems.min_z < item_max_z <= ems.max_z:
                    if curr_ems.min_y < ems.max_y <= item_max_y:
                        # three new EMS, limited by the different item dimension
                        # min_y (in front)
                        ems_list.append(EMS(ems.box, ems.max_x, curr_ems.min_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
                        # max_x (right)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, item_max_x, ems.min_y, ems.min_z))
                        # max_z (on top)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, item_max_z))
                        # delete EMS
                        if ems in ems_list:
                            ems_list.remove(ems)

                if ems.min_y < item_max_y <= ems.max_y and ems.min_z < item_max_z <= ems.max_z:
                    if curr_ems.min_x < ems.max_x <= item_max_x:
                        # three new EMS, limited by the different item dimension
                        # min_x (in front)
                        ems_list.append(EMS(ems.box, curr_ems.min_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, ems.min_z))
                        # max_y (right)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, item_max_y, ems.min_z))
                        # max_z (on top)
                        ems_list.append(EMS(ems.box, ems.max_x, ems.max_y, ems.max_z, ems.min_x, ems.min_y, item_max_z))
                        # delete EMS
                        if ems in ems_list:
                            ems_list.remove(ems)
                
                delete_empty_EMS()
                

def check_EMS_inside():
    """ This function checks if one EMS is completely inside another EMS.
        If this is true the smaller EMS gets deleted.

    Returns
    -------
    """
    for ems1 in ems_list:
        for ems2 in ems_list:
            if ems1 != ems2:
                if ems1.min_x <= ems2.min_x <= ems1.max_x and ems1.min_x <= ems2.max_x <= ems1.max_x:
                    if ems1.min_y <= ems2.min_y <= ems1.max_y and ems1.min_y <= ems2.max_y <= ems1.max_y:
                        if ems1.min_z <= ems2.min_z <= ems1.max_z and ems1.min_z <= ems2.max_z <= ems1.max_z:
                            ems_list.remove(ems2)


def update_EMS(curr_ems, coor, change):
    """ Calculate new EMS dimensions based on the given EMS.
        Coordinates length decrease (change) in that direction.

    Parameters
    ----------
    curr_ems
        EMS that will be updated.
    coor
        Dimension in which the EMS should get updated.

    change
        Length by which the EMS will get changed.

    Returns
    -------
        New EMS object with updated dimensions.
    """
    tmpmax_x = curr_ems.max_x
    tmpmax_y = curr_ems.max_y
    tmpmax_z = curr_ems.max_z
    if coor == 'x':
        tmpmin_x = curr_ems.min_x + change
        tmpmin_y = curr_ems.min_y
        tmpmin_z = curr_ems.min_z
    elif coor == 'y':
        tmpmin_x = curr_ems.min_x
        tmpmin_y = curr_ems.min_y + change
        tmpmin_z = curr_ems.min_z
    elif coor == 'z':
        tmpmin_x = curr_ems.min_x
        tmpmin_y = curr_ems.min_y
        tmpmin_z = curr_ems.min_z + change
    
    # returns initialization of new EMS, output needs to be saved as a new EMS object
    return EMS(curr_ems.box, tmpmax_x, tmpmax_y, tmpmax_z, tmpmin_x, tmpmin_y, tmpmin_z)
    

def check_orientations(orientation, ems, item, length_item, width_item, height_item):
    """ Checks whether the item fits in the EMS with the current orientation.

    Parameters
    ----------
    orientation
        Orientation to check.

    ems
        EMS to place the item.

    item
        Item to be placed in the EMS.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.
    Returns
    -------
        A tuple containing the item, EMS, orientation, calculated volume ratio and the minimal margin.
    """
    if orientation == 1:
        x = length_item[item[0]]
        y = width_item[item[0]]
        z = height_item[item[0]]
    elif orientation == 2:
        x = length_item[item[0]]
        y = height_item[item[0]]
        z = width_item[item[0]]
    elif orientation == 3:
        x = width_item[item[0]]
        y = length_item[item[0]]
        z = height_item[item[0]]
    elif orientation == 4:
        x = width_item[item[0]]
        y = height_item[item[0]]
        z = length_item[item[0]]
    elif orientation == 5:
        x = height_item[item[0]]
        y = length_item[item[0]]
        z = width_item[item[0]]
    else:
        x = height_item[item[0]]
        y = width_item[item[0]]
        z = length_item[item[0]]

    # check if item fits into EMS. If yes, return item, ems, orientation, volume ratio and the min margin, otherwise pass
    if ems.min_x + x <= ems.max_x:
        if ems.min_y + y <= ems.max_y:
            if ems.min_z + z <= ems.max_z:
                # calculate volume ratio (volume_item/volume_ems)
                volume_item = x * y * z
                volume_ems = ems.volume
                volume_ratio = volume_item/volume_ems

                # calculate margin in each direction and determine the smallest margin (min_margin)
                x_margin = ems.length - x
                y_margin = ems.width - y
                z_margin = ems.height - z

                min_margin = min(x_margin, y_margin, z_margin)
                return (item, ems, orientation, volume_ratio, min_margin)


def orientation_check(orientation, item, length_item, width_item, height_item):
    """ 'Standardize' dimensions of items to allow easier calculations.

    Parameters
    ----------
    orientation
        Orientation in which to place the item.

    item
        Item to place.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    Returns
    -------
        Standardized dimensions in x, y and z direction.
    """
    if orientation == 1:
        change_x = length_item[item]
        change_y = width_item[item]
        change_z = height_item[item]
        length = 'L'
        width = 'W'
        height = 'H'
    elif orientation == 2:
        change_x = length_item[item]
        change_y = height_item[item]
        change_z = width_item[item]
        length = 'L'
        width = 'H'
        height = 'W'
    elif orientation == 3:
        change_x = width_item[item]
        change_y = length_item[item]
        change_z = height_item[item]
        length = 'W'
        width = 'L'
        height = 'H'
    elif orientation == 4:
        change_x = width_item[item]
        change_y = height_item[item]
        change_z = length_item[item]
        length = 'W'
        width = 'H'
        height = 'L'
    elif orientation == 5:
        change_x = height_item[item]
        change_y = length_item[item]
        change_z = width_item[item]
        length = 'H'
        width = 'L'
        height = 'W'
    else:
        change_x = height_item[item]
        change_y = width_item[item]
        change_z = length_item[item]
        length = 'H'
        width = 'W'
        height = 'L'

    return change_x, change_y, change_z, length, width, height


def best_match_heuristic(generation, considered_items, considered_EMS, length_item, width_item, height_item, length_box,
                         width_box, height_box, weight_item, best_chromosome=False):
    """ This function is the main part of the best match heuristic.

    Parameters
    ----------
    generation
        Current generation that should be evaluated.

    considered_items
        Number of items considered in each step of the heuristic.

    considered_EMS
        Number of EMS considered in each Iteration.

    length_item
        Dictionary containing the length of all items.

    width_item
        Dictionary containing the width of all items.

    height_item
        Dictionary containing the height of all items.

    length_box
        Dictionary containing the length of all boxes.

    width_box
        Dictionary containing the width of all boxes.

    height_box
        Dictionary containing the height of all boxes.

    weight_item
        Dictionary containing the weight of all items.

    best_chromosome
        Indicates whether the list of items and list of boxes should be returned

    Returns
    -------
        Generation with appended fitness value for each chromosome.
        If best_chromosome is set, a list of the packed items and used boxes is returned as well.
    """
    # check each chromosome in a given generation
    for chromosome in generation:
        del ems_list[:]
        global boxcounter
        boxcounter = 0
        if chromosome.fitness == None or best_chromosome == True:
            OC = [] # list of opened containers
            P = [] # packing sequence
            BPS = chromosome.BPS[:]
            CLS = chromosome.CLS[:]
            boxpacking = []
            boxplaced = False
            while BPS != []:
                boxplaced = False
                # check each opened box
                for c in OC:
                    # list of EMSs in the current box c
                    curr_box_ems_list = []
                    for element in ems_list:
                        if element.box == c[1]:
                            curr_box_ems_list.append(element)
                    curr_box_ems_list.sort(key=lambda x: x.min_vertex)
                    j = 1
                    while j <= len(curr_box_ems_list) and boxplaced == False:
                        k = j + considered_EMS
                        while j < k and j <= len(curr_box_ems_list):
                            for i in range(0, considered_items):
                                if i <= len(BPS)-1:
                                    for l in range(1,7):
                                        # check if placement is possible. If yes, place item with feasible orientation and min. margin into the box, otherwise pass
                                        possiblePlacement = check_orientations(orientation=l, ems=curr_box_ems_list[j - 1], item=BPS[i], length_item=length_item, width_item=width_item, height_item=height_item)
                                        if possiblePlacement != None:
                                            P.append(possiblePlacement)
                            j += 1
                        if P != []:
                            # sort P in terms of volume_ratio and min_margin, pack the first (best) possiblePlacement, get orientation and remove item from BPS list
                            P.sort(key = lambda y: (-y[3], y[4]))
                            boxpacking.append((P[0][0], P[0][1].box))
                            change_x, change_y, change_z, length, width, height = orientation_check(orientation=P[0][2], item=P[0][0][0], length_item=length_item, width_item=width_item, height_item=height_item)
                            BPS.remove(P[0][0])
                            
                            # save coordinates for best chromosome in last generation for visualization
                            if best_chromosome == True:
                                item_list.append(Item(P[0][0], P[0][1].box, P[0][1].min_x, P[0][1].min_y, P[0][1].min_z, change_x, change_y, change_z, length, width, height))

                            # update the EMS in which the current item is placed in. This creates three new EMSs
                            ems_list.append(update_EMS(P[0][1], 'x', change_x))
                            ems_list.append(update_EMS(P[0][1], 'y', change_y))
                            ems_list.append(update_EMS(P[0][1], 'z', change_z))
                            
                            # check if new EMS cut any other EMSs, if yes update these EMSs
                            check_EMS_intersect(P[0][1], change_x, change_y, change_z)
                            check_EMS_pierce(P[0][1], change_x, change_y, change_z)
                            ems_list.remove(P[0][1])
                            # check if one EMS is completely inside an other EMS and if any empty EMSs (volume = 0) exist
                            check_EMS_inside()
                            delete_empty_EMS()

                            # delete list of possible Placements, set boxplaced to True
                            del P[:]
                            boxplaced = True

                    if boxplaced == True:
                        break

                while CLS != [] and boxplaced == False:
                    # open a new box
                    ems_list.append(initialize_EMS(box=CLS[0], length_box=length_box, width_box=width_box, height_box=height_box))
                    OC.append(CLS[0])
                    CLS.pop(0)
                    # try to pack the first kb boxes into the new box
                    for i in range(0, considered_items):
                        if i <= len(BPS)-1:
                            for l in range(1,7):
                                # check if placement is possible. If yes, place item with feasible orientation and min. margin into the box, otherwise pass
                                possiblePlacement = check_orientations(orientation=l, ems=ems_list[-1], item=BPS[i], length_item=length_item, width_item=width_item, height_item=height_item)
                                if possiblePlacement != None:
                                    P.append(possiblePlacement)

                    if P != []:
                        # sort P in terms of volume_ratio and min_margin, pack the first (best) possiblePlacement, get orientation and remove item from BPS list
                        P.sort(key = lambda y: (-y[3], y[4]))
                        boxpacking.append((P[0][0], (P[0][1].box)))
                        change_x, change_y, change_z, length, width, height = orientation_check(orientation=P[0][2], item=P[0][0][0], length_item=length_item, width_item=width_item, height_item=height_item)
                        BPS.remove(P[0][0])

                        # save coordinates for best chromosome in last generation for visualization
                        if best_chromosome == True:
                            item_list.append(Item(P[0][0], P[0][1].box, P[0][1].min_x, P[0][1].min_y, P[0][1].min_z, change_x, change_y, change_z, length, width, height))
                        
                        # update the EMS in which the current item is placed in. This creates three new EMSs
                        ems_list.append(update_EMS(P[0][1], 'x', change_x))
                        ems_list.append(update_EMS(P[0][1], 'y', change_y))
                        ems_list.append(update_EMS(P[0][1], 'z', change_z))
                        
                        # check if new EMS cut any other EMSs, if yes update these EMSs
                        check_EMS_intersect(P[0][1], change_x, change_y, change_z)
                        check_EMS_pierce(P[0][1], change_x, change_y, change_z)
                        ems_list.remove(P[0][1])
                        # check if one EMS is completely inside an other EMS and if any empty EMSs (volume = 0) exist
                        check_EMS_inside()
                        delete_empty_EMS()

                        # delete list of possible Placements, set boxplaced to True
                        del P[:]
                        boxplaced = True

                if boxplaced == False:
                    chromosome.set_fitness(0)
                    break

            # calculation of fitness (volume ratio of cumulated items and boxes)
            vol_items = 0
            vol_boxes = 0
            used_boxes = []
            # determine cumulated volume of items (vol_items) and boxes (vol_boxes) 
            for item in boxpacking:
                vol_items += (length_item[item[0][0]] * width_item[item[0][0]] * height_item[item[0][0]])
                used_boxes.append(item[1])
            for box in OC:
                if box[1] in used_boxes:
                    vol_boxes += (length_box[box[0]] * width_box[box[0]] * height_box[box[0]]) 
                    if best_chromosome == True:
                        box_list.append(box)

            # calculate fitness as the volume ratio and set the EMSs object attributes fitness and packing_sequence
            if chromosome.fitness != 0:
                fitness = vol_items/vol_boxes
                chromosome.set_packing_sequence(boxpacking)

                itemsPerBox = {}
                weightBoxes = {}

                # calculate total box weights
                for i in range(len(chromosome.packing_sequence)):
                
                    # create dictionary with all items in each used box
                    if chromosome.packing_sequence[i][1] in itemsPerBox:
                        itemsPerBox[chromosome.packing_sequence[i][1]].append(chromosome.packing_sequence[i][0][0])
                    else:
                        itemsPerBox[chromosome.packing_sequence[i][1]] = [chromosome.packing_sequence[i][0][0]]

                for box in itemsPerBox:
                    boxtuple = chromosome.CLS[box-1]
                    boxweight = 0
                    for i in itemsPerBox[box]:
                        boxweight += weight_item[i]

                    weightBoxes[boxtuple] = boxweight

                totalCost = 0
                for box in weightBoxes: 
                    if box[0] == 'B0' or box[0] == 'B1':
                        if weightBoxes[box] <= 2000:
                            totalCost += 3.79
                        elif weightBoxes[box] <= 5000:
                            totalCost += 5.99
                        elif weightBoxes[box] <= 10000:
                            totalCost += 8.49
                        elif weightBoxes[box] <= 31500:
                            totalCost += 16.08
                        else:
                            totalCost += 25
                    else:
                        if weightBoxes[box] <= 5000:
                            totalCost += 5.99
                        elif weightBoxes[box] <= 10000:
                            totalCost += 8.49
                        elif weightBoxes[box] <= 31500:
                            totalCost += 16.08
                        else:
                            totalCost += 25

                chromosome.set_costs(totalCost)
                chromosome.set_fitness(fitness)

        else:
            # no packing needed, because already done before (e.g. unchanged chromosome)
            pass
        
    generation.sort(key=lambda x: x.fitness, reverse=True)
    
    if best_chromosome == True:
        return generation, item_list, box_list
    else:
        return generation
