# Copyright (c) 2020 Moritz Dederichs
# Copyright (c) 2020 Janos Piddubnij

from gurobipy import Model, GRB, quicksum
from typing import Dict, Any


def __compute_item_distribution(model: Model, boxes: list, box_indices: range, items: list) -> Dict[Any, list]:
    """ This function constructs a dictionary assigning all items to their boxes as computed by the gurobi model.

    Parameters
    ----------
    model: gurobipy.Model
        The optimisation model that computed the item assignment to boxes.

    boxes: gurobipy.tuplelist
        The list of all box types.

    box_indices: range
        The indices for all boxes of one type.

    items: list
        The list of items that have to be packed.
        Assumed list element Tuple[item name, item index].

    Returns
    -------
    Dict[Any, list]
        A list assigning for each box the items that have to be packed inside.
        The dict key has the type Tuple[box_type, box_index].
    """
    distribution = dict()
    if model.status in (GRB.status.OPTIMAL, GRB.status.INTERRUPTED):
        for b in boxes:
            for i in box_indices:
                k = model.getVarByName(f'k_{b}_{i}')
                if k.x > 0.5:
                    distribution[b, i] = list()
                    for j in items:
                        v = model.getVarByName(f'v_{j}_{b}_{i}')
                        if v.x > 0.5:
                            distribution[b, i].append(j)
    return distribution


def assign_items_by_volume(boxes: list, width_boxes: dict, height_boxes: dict, length_boxes: dict,
                           items: list, width_items: Dict[str, int], height_items: Dict[str, int],
                           length_items: Dict[str, int], volume_ratio: float, num_boxes: int = None) -> Dict[Any, list]:
    """ This model assigns a given set of items to the minimal number of boxes such that the sum of all item volumes
        within one box is not more than volume_ratio * box volume.

    Parameters
    ----------
    boxes: gurobipy.tuplelist
        The set of box types that can be used to pack items.

    width_boxes: gurobipy.tupledict
        A dictionary defining the width of each box type in 'boxes'.

    height_boxes: gurobipy.tupledict
        A dictionary defining the height of each box type in 'boxes'.

    length_boxes: gurobipy.tupledict
        A dictionary defining the length of each box type in 'boxes'.

    items: list
        A list containing all items that have to be picked.
        Assumed list element Tuple[item name, item index].

    width_items: Dict[str, int]
        A dictionary defining the width of each item type.

    height_items: Dict[str, int]
        A dictionary defining the height of each item type.

    length_items: Dict[str, int]
        A dictionary defining the length of each item type.

    volume_ratio: float
        The maximal volume ratio between all packed items and the containing box.

    num_boxes: int
        The minimal number of boxes to use.

    Returns
    -------
    Dict[Any, list]
        A dictionary assigning each item to a specific box defined by the dict key (box_type, box_index).
    """
    model = Model('Volume_INFORM')
    model.params.TimeLimit = 60

    # Big M as an upper bound of boxes needed for each box type
    big_m = len(items)
    box_indices = range(big_m)
    box_min_dim = dict()
    box_med_dim = dict()
    box_max_dim = dict()
    for b in boxes:
        box_min_dim[b], box_med_dim[b], box_max_dim[b] = sorted([length_boxes[b], width_boxes[b], height_boxes[b]])

    item_min_dim = dict()
    item_med_dim = dict()
    item_max_dim = dict()
    for i in items:
        item_min_dim[i], item_med_dim[i], item_max_dim[i] = sorted([length_items[i[0]], width_items[i[0]], height_items[i[0]]])

    # Variables

    # Variable k_b_i = 1 if box i in [0, big_m) of type b in B is used
    k = {}
    for b in boxes:
        for i in box_indices:
            k[b, i] = model.addVar(name=f'k_{b}_{i}', vtype=GRB.BINARY)

    # Variable o_b_i = 1 if box i in [0, big_m) of type b in B is packed more than volume_ratio
    o = {}
    for b in boxes:
        for i in box_indices:
            o[b, i] = model.addVar(name=f'o_{b}_{i}', vtype=GRB.BINARY)

    # Variable v_j_b_i = 1 if item j is placed in box i of type b
    v = {}
    for j in items:
        for b in boxes:
            for i in box_indices:
                v[j, b, i] = model.addVar(name=f'v_{j}_{b}_{i}', vtype=GRB.BINARY)

    # Constants
    item_volume = dict()
    for j in items:
        item_volume[j] = length_items[j[0]] * width_items[j[0]] * height_items[j[0]]

    box_volume = dict()
    for b in boxes:
        box_volume[b] = length_boxes[b] * width_boxes[b] * height_boxes[b]

    # Constraints

    # If this is a restart, use at least one more box than the number of boxes that the items were assigned to previously and did not fit in
    if num_boxes is not None:
        model.addConstr(quicksum(k[b, i] for b in boxes for i in box_indices) >= num_boxes)

    # All items j have to be packed in exactly one box
    for j in items:
        model.addConstr(quicksum(v[j, b, i] for b in boxes for i in box_indices) == 1)

    # An item j may only be packed into a box i of type b if this box is used. Also, make sure that the item can fit into the box
    for j in items:
        for b in boxes:
            for i in box_indices:
                model.addConstr(v[j, b, i] * item_min_dim[j] <= k[b, i] * box_min_dim[b])
                model.addConstr(v[j, b, i] * item_med_dim[j] <= k[b, i] * box_med_dim[b])
                model.addConstr(v[j, b, i] * item_max_dim[j] <= k[b, i] * box_max_dim[b])

    # The combined volume of all items j within box i of type b may not be greater than 'volume_ratio' * volume of the box
    for b in boxes:
        for i in box_indices:
            model.addConstr(
                quicksum(v[j, b, i] * item_volume[j] for j in items) <= (volume_ratio + (1 - volume_ratio) * o[b, i]) *
                box_volume[b])

    model.setObjective(quicksum(k[b, i] + 1.5 * o[b, i] for b in boxes for i in box_indices), GRB.MINIMIZE)

    model._last_sol_time = None
    model._first_sol_found = False

    # Default optimization sense is minimise
    model.optimize(__termination_criterion)

    # Return the distribution
    return __compute_item_distribution(model, boxes, box_indices, items)


def __termination_criterion(model, where):
    """ This termination criterion ensures that the optimizer will not get stuck trying all solutions if an
    adequate number of boxes has been found.

    Parameters
    ----------
    model
    where

    Returns
    -------
    """
    if where == GRB.Callback.MIP:
        if model._last_sol_time is not None:
            now = model.cbGet(GRB.Callback.RUNTIME)
            if now - model._last_sol_time > 5:
                model.terminate()

    if where == GRB.Callback.MIPSOL:
        if not model._first_sol_found:
            model._first_sol_found = True
        else:
            now = model.cbGet(GRB.Callback.RUNTIME)
            model._last_sol_time = now