# Copyright (c) 2020 Moritz Dederichs

from inform_or_models.peak_filling_slice_push_heuristic import PeakFillingSlicePush
from inform_or_models.assign_items_by_volume import assign_items_by_volume
from inform_or_parser.parse_generated_csv import parse_generated_csv
from inform_or_parser.parse_boxes import parse_boxes
from typing import Dict


class SplitModelPFSP:
    def __init__(self, input_file: str, boxes_file: str):
        """ Creates an instance of the Split Model with integrated Peak-Filling Slice Push approach.

        Parameters
        ----------
        input_file: str
            Path to the file containing all orders to be packed

        boxes_file: str
            Path to the csv file containing all box dimensions.
        """
        self.orders, self.length_items, self.width_items, self.height_items, self.weight_items = parse_generated_csv(
            input_file)
        boxes_dict = parse_boxes(boxes_file)
        self.boxes = list(boxes_dict.keys())
        self.length_boxes = {k: v[0] for k, v in boxes_dict.items()}
        self.width_boxes = {k: v[1] for k, v in boxes_dict.items()}
        self.height_boxes = {k: v[2] for k, v in boxes_dict.items()}
        self.item_sizes = {key: (self.length_items[key], self.width_items[key], self.height_items[key]) for key in
                           self.length_items.keys()}
        self.pfsp_scenarios = {box: PeakFillingSlicePush(self.item_sizes, self.length_boxes[box], self.width_boxes[box],
                                                         self.height_boxes[box], box) for box in self.boxes}

    def process(self, order: str) -> Dict[str, dict]:
        """ Solves the optimisation problem to pack all items of an order in as few boxes as possible.

        Parameters
        ----------
        order: str
            The order to be packed.

        Returns
        -------
        Dict[str, dict]
            The solution computed by the optimisation process.
            Describes for all boxes the items to be placed and their specific location.
        """
        items = list()
        for item in self.orders[order]:
            for index in range(item[1]):
                items.append((item[0], index + 1))

        # Find an acceptable distribution of items into boxes
        distribution = assign_items_by_volume(self.boxes, self.width_boxes, self.height_boxes,
                                              self.length_boxes, items, self.width_items, self.height_items,
                                              self.length_items, 0.8)

        solution = dict()
        running_number = 1
        for box, items in distribution.items():
            box = box[0]
            print(f'Start packing boxes of type {box}')
            print(f'Packing the following items')
            item_list = [item[0] for item in items]
            print(f'{item_list}')
            pfsp = self.pfsp_scenarios.get(box)
            packing = pfsp.process(item_list)
            print(f'Finished packing the items into {len(packing)} boxes.')
            packing = {(k[0], int(k[1]) + running_number): v for k, v in packing.items()}
            running_number += len(packing)
            solution.update(packing)

        return solution

