# Copyright (c) 2020 Janos Piddubnij 
# Copyright (c) 2020 Moritz Dederichs


from inform_or_models.assign_items_by_volume import assign_items_by_volume
from inform_or_parser.parse_generated_csv import parse_generated_csv
from inform_or_models.place_items_in_box import place_items_in_box
from inform_or_parser.parse_boxes import parse_boxes
from typing import Dict, Tuple
import sys

class SplitModel:
    def __init__(self, input_file: str, boxes_file: str):
        """ Creates an instance of the SplitModel approach.

        Parameters
        ----------
        input_file: str
            The CSV file created by the INFORM-Generator to be processed.
        """
        self.orders, self.length_items, self.width_items, self.height_items, self.weight_items = parse_generated_csv(input_file)
        boxes_dict = parse_boxes(boxes_file)
        self.boxes = list(boxes_dict.keys())
        self.length_boxes = {k: v[0] for k, v in boxes_dict.items()}
        self.width_boxes = {k: v[1] for k, v in boxes_dict.items()}
        self.height_boxes = {k: v[2] for k, v in boxes_dict.items()}

    def process(self, order: str) -> Dict[str, dict]:
        """ Solves the optimisation problem for the given order.

        Parameters
        ----------
        order: str
            The order to be processed.

        Returns
        -------
        Dict[str, dict]
            A dictionary stating for each item in which item it has to be packed as well as the exact
            placing coordinates and the item orientation.
        """

        items = list()
        for item in self.orders[order]:
            for index in range(item[1]):
                items.append((item[0], index + 1))

        # Find an acceptable distribution of items into boxes
        distribution = assign_items_by_volume(self.boxes, self.width_boxes, self.height_boxes,
                                              self.length_boxes, items, self.width_items, self.height_items,
                                              self.length_items, 0.8)

        # Place items optimally in each box
        packed_boxes = dict()
        for box in list(distribution.keys()):
            b = box[0]
            print('Starting Place_Items_In_Box')
            item_placements = place_items_in_box(distribution[box], self.width_items, self.height_items,
                                                 self.length_items, self.width_boxes[b], self.height_boxes[b],
                                                 self.length_boxes[b])
            print('Ended Place_Items_In_Box')
            packed_boxes[box] = item_placements

        # If there are empty boxes (the assigned items did not fit), restart the procedure with the requirement to pack
        # these items into at least 2 boxes

        # Keep restarting until all items are packed
        distribution, packed_boxes, check_needed = self.restart(distribution, packed_boxes)

        while check_needed:
            distribution, packed_boxes, check_needed = self.restart(distribution, packed_boxes)

        return packed_boxes

    def restart(self, distribution: dict, packed_boxes: dict) -> Tuple[dict, dict, bool]:
        """ Restarts the procedure with the requirement to repack the items that did not fit in the previously assigned boxes

        Parameters
        ----------
        distribution: dict
            The dictionary containing the original distribution

        packed_boxes: dict
            The dictionary containing the packed boxes (including the empty ones)

        Returns
        -------
        Tuple[dict, dict, bool]
            A tuple containing the updated distribution, dictionary of packed boxes and a boolean determining whether a
            new check for empty boxes is needed
        """
        boxes_to_add = dict()
        unpacked_boxes = list()
        unpacked_items = list()
        check_needed = False
        for packed_box, values in packed_boxes.items():
            if not values:
                unpacked_boxes.append(packed_box)
                unpacked_items.extend(distribution[packed_box])
        
        # Determine the new distribution of previously unpacked items
        if len(unpacked_boxes) > 0:
            distribution = assign_items_by_volume(self.boxes, self.width_boxes, self.height_boxes, self.length_boxes,
                                                  unpacked_items, self.width_items, self.height_items,
                                                  self.length_items, 0.8, len(unpacked_boxes) + 1)
                
            # Calculate the exact placements
            for box in list(distribution.keys()):
                b = box[0]
                item_placements = place_items_in_box(distribution[box], self.width_items, self.height_items,
                                                     self.length_items, self.width_boxes[b], self.height_boxes[b],
                                                     self.length_boxes[b])
                boxes_to_add[box] = item_placements

            check_needed = True

        # Remove the empty boxes
        for b in unpacked_boxes:
            packed_boxes.pop(b, None)

        # Add the boxes packed after restart
        packed_boxes.update(boxes_to_add)

        return distribution, packed_boxes, check_needed
