# -*- coding: utf-8 -*-
"""A class describing the box data type."""

#  Copyright (c) 2020.  Yordan Manolov <yordan DOT manolov AT rwth DASH aachen DOT de >
#

import functools
import operator
from collections import Counter
from typing import List, Tuple

from .item import Item

BOX_S_LENGTH = 60
BOX_M_LENGTH = 120
BOX_L_LENGTH = 120
BOX_XL_LENGTH = 120
BOX_S_WIDTH = 30
BOX_M_WIDTH = 60
BOX_L_WIDTH = 60
BOX_XL_WIDTH = 60
BOX_S_HEIGHT = 15
BOX_M_HEIGHT = 60
BOX_L_HEIGHT = 60
BOX_XL_HEIGHT = 60


# MAX_BOX_DIM = BOX_XL_LENGTH
# MIN_BOX_DIM = BOX_S_HEIGHT
MAX_BOX_DIM = 50
MAX_BOX_ID_NUMBER = 10000


# STANDARD_SIZES = {"S": [BOX_S_LENGTH, BOX_S_WIDTH, BOX_S_HEIGHT],
#                   "M": [BOX_M_LENGTH, BOX_M_WIDTH, BOX_M_HEIGHT],
#                   "L": [BOX_L_LENGTH, BOX_L_WIDTH, BOX_L_HEIGHT],
#                   "XL": [BOX_XL_LENGTH, BOX_XL_WIDTH, BOX_XL_HEIGHT]}

STANDARD_SIZES = {'B0': [20, 10, 10],
                  'B1': [20, 20, 10],
                  'B2': [20, 20, 20],
                  'B3': [20, 40, 20],
                  'B4': [20, 40, 40],
                  'B5': [30, 40, 40],
                  'B6': [40, 40, 40],
                  'B7': [40, 50, 30]}


class Box:
    """
    Box custom data type
    """

    def __init__(self, name: str, length: int = 0, width: int = 0, height: int = 0, standard_size: str = "B4"):
        self.name = name
        if standard_size in STANDARD_SIZES.keys():
            self.length = STANDARD_SIZES[standard_size][0]
            self.width = STANDARD_SIZES[standard_size][1]
            self.height = STANDARD_SIZES[standard_size][2]
            self.standard_size = standard_size
        else:
            self.length = length
            self.width = width
            self.height = height
            self.standard_size = "custom"

        self.items = list()
        self.weight = 0
        self.current_ep = [0, 0, 0]
        self.volume = self.length * self.width * self.height
        self.area = self.length * self.width

    def generate_packing_instructions_per_box(self, output_file: str) -> None:
        """

        Parameters
        ----------
        output_file:  File to write instructions to

        Returns
        -------

        """
        with open(output_file, "a") as output:
            output.writelines(
                "For box {} of size {},"
                " place the following items:"
                " {} \n".format(self.name, self.standard_size, str(self.get_items_simplified())))

    def generate_packing_instructions_per_item(self, item_id: str, output_file: str) -> str:
        """

        Parameters
        ----------
        item_id ID of the item in the box
        output_file File to write instructions to

        Returns
        -------
        One line of instructions
        """
        with open(output_file, "a") as output:
            output.writelines("Place item {} into box {} \n".format(item_id, self.name))
        return """Place item {} into box {}""".format(item_id, self.name)

    def get_items(self) -> List[Item]:
        """

        Returns
        -------
        A list of items currently packed in the box
        """
        return self.items

    def get_items_simplified(self) -> List[Tuple[str, str]]:
        """

        Returns
        -------
        The item list as an adjacency list. Each element of the list is of the
        form (id, count), where both are strings. Intended only for writing
        packing instructions; do not use to extract/manipulate data.

        """
        aggregated = Counter(self.get_items()).items()
        human_readable = [(x.identifier, y) for x, y in aggregated]
        return human_readable

    def get_free_area(self) -> float:
        """

        Parameters
        ----------

        Returns
        -------
     The area of the box not occupied by any items yet
        """
        return (self.area -
                self.current_ep[0] * self.current_ep[1])

    def get_free_volume(self) -> float:
        """

        Parameters
        ----------

        Returns
        -------
        The difference between the box's volume
        and the volume of the envelope defined by the extreme points
        """

        return (self.volume -
                functools.reduce(operator.mul, self.current_ep, 1))

    def get_free_volume_ratio(self) -> float:
        """

        Returns
        -------
        The free volume of the box in percent

        """
        return 100 * (1 - self.get_free_volume() / self.volume)


    def _optimal_eps(self, item_length: int, item_width: int, item_height: int) -> List[int]:
        """
        Calculates the optimal extreme points for a potential new item.
        Note: does not check whether the item would
        actually fit, this is done in a separate function

        Parameters
        ----------
        item_length Length of the new potential item
        item_width Width of the new potential item
        item_height Height of the new potential item

        Returns
        -------
        The minimum extreme points after the item's insertion into the box
        """

        item_dimensions = [item_length, item_width, item_height]
        if self.current_ep == [0, 0, 0]:
            return item_dimensions
        minimum_increase = max(self.current_ep, item_dimensions,
                               key=lambda x: sum(x))
        best_placement = [0 if x > min(item_dimensions) else x for x in
                          item_dimensions]
        return [x + y for x, y in zip(minimum_increase, best_placement)]

    def place_item(self, item: Item) -> bool:
        """
        Puts an item into the current box.
        Parameters
        ----------
        item The new item to be placed into the box

        Returns
        -------
        Success code
        """

        self.items.append(item)
        return True

    def test_fits(self, candidate_item_length: int, candidate_item_width: int, candidate_item_height: int) -> bool:
        """
        Tests whether the new item for the box would fit
        under the restriction of optimal extreme points.
        Note: if the item would fit with suboptimal extreme points,
        the result is still negative.

        Parameters
        ---
        candidate_item_length: Length of the new potential item
        candidate_item_width: Width of the new potential item
        candidate_item_height: Height of the new potential item

        Returns
        ---
        Whether or not an item fits (with best possible EPs)
        """
        box_dimensions = [self.length, self.width, self.height]
        potential_eps = self._optimal_eps(candidate_item_length, candidate_item_width, candidate_item_height)
        return all(x <= y for x, y in zip(potential_eps, box_dimensions))

    def update_eps(self, item_length: int, item_width: int, item_height: int) -> List[int]:
        """
        Updates the most recent extreme points of the box.
        Parameters
        ---
        item_length: Length of the new potential item
        item_width: Width of the new potential item
        item_height: Height of the new potential item

        Returns
        ---
        A list with the newly calculated EPs
        """
        self.current_ep = self._optimal_eps(item_length, item_width, item_height)
        return self.current_ep
