/*
**  bifst_a.c
**  bif-c
**
**  Created by Joel Rees on 2009/07/25.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIFST/A, as mechanically as possible.
**
*/


#include <stddef.h>
#include <signal.h>	/* For BREAK key handling. */
#include <string.h> /* For debug prints. */

#include <stdio.h>	/* For handling stdio. Also for debugging purposes, since XCODE gets so confused. */


#include "bifst_a.h"
#include "bif_m.h"
#include "bifdp_a.h"

#if defined DEBUGGING && DEBUGGING > 0 && DEBUGGING < 0x10
#include "bif1_a.h"	/* For SEMIS and DOT in debugging. */
#include "bif2_a.h" /* For constants, user constants, user variables, etc. in debugging. */
#include "bif3b_a.h" /* For TYPE */
#include "bif5_a.h" /* For DOT */
#endif /* defined DEBUGGING && DEBUGGING > 0 && DEBUGGING < 0x10 */

#include "bif7b_a.h"


static byte_t memoryImage[ ILIM ] = { 0 };	/* Initialize it to keep the optimizer from discarding it. */
/* For bifst_a.h:
00000010 * Start up routines for BIF
00020 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
00100 *
00105 ILIM	EQU $8000 memory
00110 IBUFF	EQU ILIM-BCT*(BWID+4)
00120 IPAD	EQU IBUFF-TWID max
00130 IHASH	EQU IPAD-34
00140 IWPAD	EQU IHASH-NLMASK-2
00150 ITIB	EQU IWPAD-TWID
00160 IUSER	EQU ITIB-UEND
00170 IRP0	EQU IUSER-2
00180 ISP0	EQU IRP0-258
00190 * dictionary below here
00400 *
*/

/* This is not organized well, at all. I'm just going to use it as it is, for now.
** 
00410 ORIG	EQU * initials for task 0, offsets
00420 	NOP 0
00430 	JMP COLD 1
00440 	NOP 4
00450 	JMP WARM 5
00460 	FDB $6809 cpu 8
00470 	FDB 0 rev $0A
00480 	FDB FORGET-CFAOFF CURRENT for COLD $0C
00490 	FDB 8 backspace $0E
00500 	FDB IUSER task 0 user area $10
00510 	FDB ISP0 $12
00520 	FDB IRP0 $14
00530 	FDB ITIB $16
00540 	FDB NLMASK not used $18
00550 	FDB 1 WARNING $1A
00560 	FDB FOLLOW-1 FENCE for COLD $1C
00570 	FDB FOLLOW DP for COLD $1E
00580 	FDB BIF+2 defs root $20
00590 	FDB IPAD $22
00600 	FDB IWPAD $24
00610 	FDB 32 terminal columns $26
00620 	FDB IBUFF $28
00630 	FDB ILIM $2A
01000 *
*/
boot_constants_s	ORIG =
{	COLD,
	WARM,
	{ 0x43 },	/* CPUid is 'C' */
	{ 0 },	/* version */
	{ (natural_t) &hFORGET },	/* initial current */
	{ '\b' },
	{ (natural_t) ( memoryImage + IUSER ) },	/* This cast is not the best idea, but should get me running. */
	{ (natural_t) ( memoryImage + ISP0 ) },
	{ (natural_t) ( memoryImage + IRP0 ) },
	{ (natural_t) ( memoryImage + ITIB ) },
	{ NLMASK },
	{ 1 },	/* disk on-line flag */
	/* Either I change the fence to see a flag or a link, or I copy the headers into memoryImage. */
	{ (natural_t) memoryImage /* allocatable area;  was FOLLOW - 1 */ },	/* FENCE for FORGET */
	{ (natural_t) ( memoryImage ) /* was FOLLOW */ },	/* initial allocation pointer. */
	{ (natural_t) &hBIF },	/* initial context root, both contexts. */
	{ (natural_t) ( memoryImage + IPAD ) },
	{ (natural_t) ( memoryImage + IWPAD ) },
	{ 32 },
	{ (natural_t) ( memoryImage + IBUFF ) },
	{ (natural_t) ( memoryImage + ILIM ) },
	{ (natural_t) &hFORGET /* since the headers move, end of the linked list;  was marked by FENCE. */ },
	{ 0 /* Assume standard library echos. */ }
};


/* Initially, one big (so to speak) array of bytes, following the 6809 code (so to speak). 
*/


/* Need to do BREAK key handling, will do it strictly by the K&R book example.
*/
static sig_atomic_t got_break = 0;

static void catch_break( int siggie /* Shouldn't be much we need to look at here, anyway. Nothing, actually. */ )
{	signal( SIGINT, catch_break );
	got_break = 1;
	return;
}

int break_pressed(void)
{	int breakseen = ( got_break != 0 );
	got_break = 0;
	return breakseen;
}

/* Could declare the functions here, but want to follow the 6809 code order where I can. */
static void DOREGS(void);
static void PRUNE(void);
static void MARK(void);
static void CHECK(void);


/* Need to handle stdin, stdout, stderr, and "stdlog". 
// This is not really what I want to do, even in the FIG bootstrap.
// But, since stdout is apparently not an lvalue, we need to indirect.
*/
FILE * standardInput = NULL;
FILE * standardOutput = NULL;
FILE * standardError = NULL;
FILE * standardLog = NULL;
/* For the BIF/6809 compatible bootstrap, all I really want is a kind of "stdlog".
// That can be done in the same file as >PRT and >VID.
*/


/*
01010 	FCC 'COLD'
01020 	FCB 4
01030 	FCB MFORE
01040 	FDB 0
01050 	FDB BIF+2
01060 	FDB 0
01070 	FDB 0
*/
static character_t sCOLD[] = "\x4" "COLD";
definition_header_s hCOLD =	
{	{ (natural_t) sCOLD },
	{ 0 },
	{ 0 },	/* This is the very first definition in the initial base vocabulary. */
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },	/* Full leaf node. */
	{ 0 },
	{ (natural_t) COLD }
};
/* HEADER( COLD, 0 / * no special modes. * /, WARM, BIF, COLD, 0 ); */
snatural_t COLD( definition_header_s * wp )
{
	standardInput = stdin;
	standardOutput = stdout;
	standardError = stderr;
	signal( SIGINT, catch_break );	/* prepare for BREAK key handling. */
/* 01080 COLD	LEAY DOREGS,PCR */
/* 01090 	EXG PC,Y call */ /* Set up the essential registers, called without stack. */
#if defined DEBUGGING
fprintf( standardError, "memory image at: %p, ORIG at: %p, IUSER == %lx\n", memoryImage, &ORIG, IUSER );
#endif /* defined DEBUGGING */
	DOREGS();	/* Points W to the initialization table instead of Y, use UP instead of X. */
/* 01100 	SETDP VDP */
/* 01110 	CLR $71 for BASIC reset */ /* But we aren't taking over the whole machine. */
/* 01120 	LDD -4,Y */
/* 01130 	STD UCURR,X */
	UP.task->lastDefined = W.bootConstantsp->initialCurrent;
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial current's name: %s\n", UP.task->lastDefined.definitionp->nameLink.chString + 1 );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
	/* work from here, fixing the initialization table as I go */
/* 01140 	LDD $0C,Y */
/* 01150 	STD UFENCE,X */	
	UP.task->forgetFence = W.bootConstantsp->initialForgetFence;
	/* But the FENCE won't work like this for built-ins. Not that it needs to. */
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial FENCE: %lx\n", UP.task->forgetFence.integer );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
	UP.task->tailFence = W.bootConstantsp->initialTailFence;
	/* For FENCEing by mark. */
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial TAIL: %lx\n", UP.task->tailFence.integer );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
/* 01160 	LDD $0E,Y */
/* 01170 	STD UDP,X */
	UP.task->dictionaryAllocationPointer = W.bootConstantsp->initialDictionaryAllocationPointer;
#if defined DEBUGGING
fprintf( standardError, "initial allocation pointer: %p\n", UP.task->dictionaryAllocationPointer.cellp );
#endif /* defined DEBUGGING */
/* 01180 	LDD $10,Y */
/* 01190 	STD UROOT,X */
/* 01200 	STD UDROOT,X */
	UP.task->definitionContextRoot = UP.task->searchContextRoot = W.bootConstantsp->initialDefinitionContextRoot;
#if defined DEBUGGING && DEBUGGING <= 3
fprintf( standardError, "initial context's name: %s\n", UP.task->definitionContextRoot.definitionp->nameLink.chString + 1 );
#endif  /* defined DEBUGGING && DEBUGGING <= 3 */
	/* Currently hand-building the tree. Would it save some object space to explicitly build it? */
/* 01210 	PSHU D     Initial vocabularies */
/* 01220 	BSR PRUNE  must ALL be PRUNEd! */
	MARK();	/* Just because we can? */
	( * --SP ) = W.bootConstantsp->initialDefinitionContextRoot;
	PRUNE();
/* 01230 	LDD #EDITOR+2 */
/* 01240 	PSHU D */
/* 01250 	BSR PRUNE */
	( * --SP ).definitionp = &hEDITOR;
	PRUNE();
/* 01260 	LDD #ASMBLR+2 */
/* 01270 	PSHU D */
/* 01280 	BSR PRUNE */
	( * --SP ).definitionp = &hASMBLR;
	PRUNE();
	( * --SP ).definitionp = &hUTIL;	/* bug in BIF6809 */
	PRUNE();
/* 01290 	JMP WARM+5 */
	CHECK();
	WARM();
}


static void dumpHeader( definition_header_s * header, int linksFlag )
{	fprintf( standardError, "hd:%p", header );
	if ( header != ( definition_header_s * ) 0 )
	{	character_t * name = header->nameLink.chString;
		if ( linksFlag )
		{	fprintf( standardError, "\tup:%p", header->vocabLink.definitionp );
		}
		fprintf( standardError, "\tnm:%p", name );
		if ( name != (character_t *) 0 )
		{	unsigned long length = (unsigned long) strlen( name + 1 );
			unsigned lengthByte = (unsigned char) name[ 0 ];
			fprintf( standardError, "\t%s\t%d:%lu%c", name + 1, 
					 lengthByte, length, (lengthByte != length) ? '*' : ' '
				   );
		}
		else
		{	fputs( "\t*-none-*\t0:0!", standardError );
		}
		fprintf( standardError, "\tim:%04lx\tam:%04lx\tal:%p", 
				 header->interpMode.integer, 
				 header->allocMode.integer,
				 header->allocLink.definitionp 
			   );
		if ( linksFlag )
		{	fprintf( standardError, "\tl:%p\tr:%p\tcd:%p\tp[0]:%08lx", 
					 header->leftLink.definitionp, 
					 header->rightLink.definitionp,
					 header->codeLink.icodep, 
					 header->parameterLink[ 0 ].integer 
				   );
		}
	}
	fputc( '\n', standardError );
}


static void MARK(void)
{	definition_header_s * tracer = UP.task->tailFence.definitionp;
	while ( tracer != (definition_header_s *) 0 )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		tracer->allocMode.integer &= ~VOC_MEMO_MARK;
		tracer->allocMode.integer |= BOOT_VOC_MARK;
		tracer = tracer->allocLink.definitionp;
	}
}


/* 01300 	SETDP 0 */
/* 01310 * Watch stack (vocabulary) depth! */
/* 01315 	SETDP VDP */
/* 01320 PRUNE	PSHS D,X,Y */
/* 01330 	LDY <UP */
/* 01340 	LDX ,U++ vocab */
/* 01344 	BEQ PRUNEX+2 */
/* 01348 	LDD ,X root */
/* 01352 	BEQ PRUNEX+2 */
/* 01356 	CMPD UFENCE,Y */
/* 01360 	BLS PRUNSK */
/* 01364 	LDD #0 */
/* 01368 	STD ,X */
/* 01372 	BRA PRUNEX+2 */
/* 01376 PRUNSK	TFR D,X */
/* 01380 	LDD #0 */
/* 01386 	PSHS D mark */
/* 01390 PRUNL	LDD RTOFF,X */
/* 01400 	BEQ PRUNLF */
/* 01410 	CMPD UFENCE,Y */
/* 01420 	BLS PRUNLF-2 */
/* 01430 	LDD #0 make leaf */
/* 01440 	STD RTOFF,X */
/* 01450 	BRA PRUNLF */
/* 01460 	PSHS D for later */
/* 01470 PRUNLF	LDD LFTOFF,X */
/* 01480 	BEQ PRUNEX-2 */
/* 01490 	CMPD UFENCE,Y */
/* 01500 	BHI PRUNL0 */
/* 01510 	TFR D,X go left */
/* 01520 	BRA PRUNL */
/* 01530 PRUNL0	LDD #0 make leaf */
/* 01540 	STD LFTOFF,X */
/* 01550 	LDX ,S++ go right? */
/* 01560 PRUNEX	BNE PRUNL */
/* 01570 	PULS D,X,Y,PC */
/* 01580 	SETDP 0 */
/* 01590 * 
*/

static void PRUNE(void)	/* Static binary tree, not balanced at run-time. */
{	definition_header_s * tracer = ( * SP++ ).definitionp;
	definition_header_s * probe;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "Starting the prune.\n", standardError );
fprintf( standardError, "allocation area limits, bottom:%p, top:%p\n", 
		 UP.task->forgetFence.definitionp, W.bootConstantsp->initialLimitOfBufferRAM.definitionp );
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	if ( ( tracer == (definition_header_s *) 0 )
		 || ( tracer->codeLink.icode != XVOC ) 
		 || ( tracer->parameterLink[ 0 ].cellp == (cell_u *) 0 )
	   )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( standardError, "Early exit on %p\n", tracer );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		return;
	}
	probe = tracer->parameterLink[ 0 ].definitionp;
	/* If I start balancing the tree, I'll need to go to the leaf nodes before pruning. */
	if ( ( probe > UP.task->forgetFence.definitionp ) 
		 && ( probe < W.bootConstantsp->initialLimitOfBufferRAM.definitionp ) )
	{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fprintf( standardError, "Whole tree lopped under %p\n", tracer );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		tracer->parameterLink[ 0 ].integer = 0;	/* Assume the whole vocabulary can be lopped. */
		return;
	}
	( * --SP ).integer = 0;	/* Mark our stopping point with a NULL. (On the parameter stack instead of the return stack.) */
	/* Look right and go left. */
	while ( probe != (definition_header_s *) 0 )
	{
		if ( ( probe->allocMode.integer & VOC_MEMO_MARK ) != 0 )
		{
#if defined DEBUGGING
fputs( "** Retracing our steps! **", standardError );
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING */
			break;
		}

		probe->allocMode.integer |= VOC_MEMO_MARK;
		tracer = probe;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
dumpHeader( tracer, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		probe = tracer->rightLink.definitionp;
		if ( probe != (definition_header_s *) 0 )
		{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "looking right, at ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
			if ( ( probe > UP.task->forgetFence.definitionp ) 
				 && ( probe < W.bootConstantsp->initialLimitOfBufferRAM.definitionp ) )
			{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "pruned right \n", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
				tracer->rightLink.definitionp = (definition_header_s *) 0;	/* Assume the whole subtree can be lopped. */
			}
			else
			{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "saving right branch to ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
				( * --SP ).definitionp = probe;	/* Save it to come back to later. */
			}
		}
		probe = tracer->leftLink.definitionp;
		if ( probe != (definition_header_s *) 0 )
		{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "looking left, at ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
			if ( ( probe > UP.task->forgetFence.definitionp ) 
				  && ( probe < W.bootConstantsp->initialLimitOfBufferRAM.definitionp ) )
			{
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "pruned left", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
				probe = tracer->leftLink.definitionp = (definition_header_s *) 0;	/* Assume the whole subtree can be lopped. */
				/* Leave probe as a flag for descent or no descent. */
			}
		}
		if ( probe == (definition_header_s *) 0 )	/* Might have been cleared above. */
		{
			probe = ( * SP++ ).definitionp;
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
fputs( "moving right to ", standardError );
dumpHeader( probe, 1 );
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
		}
#if defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE
		else
		{
fputs( "descending left to", standardError );
dumpHeader( probe, 1 );
		}
#endif  /* defined DEBUGGING && defined DBG_SHOW_SYMBOL_TABLE */
	}
}


static void CHECK(void)
{	definition_header_s * tracer = UP.task->tailFence.definitionp;
	while ( tracer != (definition_header_s *) 0 )
	{	natural_t flags = tracer->allocMode.integer;
		if ( ( flags & VOC_MEMO_MARK ) == 0 )
		{	fputs( "** !!Broken Symbol Table Links!! **: ", standardError );
			dumpHeader( tracer, 1 );
		}
		tracer->allocMode.integer &= ~VOC_MEMO_MARK;
		tracer = tracer->allocLink.definitionp;
	}
}


/* Point the essential registers to real/available memory.
*/
/* 02000 	SETDP 0 */
static void DOREGS(void)
{
/* 02010 DOREGS	LDS #IRP0 */
	/* These are to make the CPU sane. Need to get these from the initializations table, instead? */
	RP = (cell_u *) ( memoryImage + IRP0 );	/* woops! Don't make that 4 times allocation. */
/* 02020 	PSHS CC save */ /* Not real registers, no need to mask interrupts. */
/* 02030 	ORCC #$50 mask ints */
/* 02040 	PSHS Y return adr */ /* Using a standard C call, no need to manipulate the return address. */
/* 02050 	LDD #DPAGE */
/* 02060 	TFR A,DP */
/* 02070 	SETDP VDP */ /* Not messing with short (DP) addressing. */
/* 02080 	LDY #ORIG+$10 */ /* We'll point at with W, instead. */
	W.bytep = (byte_p) &ORIG;
/* 02090 	LDX ,Y */
/* 02100 	STX <UP */
	UP = W.bootConstantsp->initialTaskBase;
/* 02110 	LDU #ISP0 */
	SP = (cell_u *) ( memoryImage + ISP0 );
/* 02120 	LDD #0 */
/* 02130 	STD >0 trap [NULL]s */
	/* Can't really do this portably, need to conditionally compile for architectures that don't trap NULL refs. */
#if !defined SYSTEM_TRAPS_NULLS
	* ( (natural_t * ) 0 ) = 0;
#endif
	sysSIG.integer = SYS_SIG_NORMAL;	/* This has been turned into the hard exception flag. */
/* 02140 	RTS */
#if defined DEBUGGING
fprintf( standardError, "RP: %p, SP: %p, UP.bytep: %p, UP.integer: %lx, W.bootConstantsp: %p, W.integer: %lx\n", RP, SP, UP.bytep, UP.integer, W.bootConstantsp, W.integer );
#endif /* defined DEBUGGING */
}
/* 02150 	SETDP 0 */
/* 02160 * 
*/


#if defined DEBUGGING && defined DBG_BOOTSTRAP_TESTING
static definition_header_s hiBOOTSTRAP_TEST;
#endif /* defined DEBUGGING && defined DBG_BOOTSTRAP_TESTING */
/*
02230 	FCC 'WARM'
02240 	FCB 4
02250 	FCB MFORE
02260 	FDB COLD-CFAOFF
02270 	FDB BIF+2
02280 	FDB VLIST-CFAOFF
02290 	FDB WARN-CFAOFF
*/
static character_t sWARM[] = "\x4" "WARM";
definition_header_s hWARM =	
{	{ (natural_t) sWARM },
	{ 0 },
	{ (natural_t) &hCOLD },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hVLIST },
	{ (natural_t) &hWARN },
	{ (natural_t) WARM }
};
snatural_t WARM( definition_header_s * wp )
{
//HEADER( WARM, 0 /* no special modes. */, COLD, BIF, WARM, 0 );
/* 02300 WARM	LEAY <DOREGS,PCR */
/* 02310 	EXG PC,Y call */
	DOREGS();
/* 02320 	SETDP VDP */
/* 02360 	LDD 2,Y */
/* 02370 	STD US0,X */
	UP.task->dataStackBase = W.bootConstantsp->initialParameterStackBase;
#if defined DEBUGGING
fprintf( standardError, "initial parameter stack base: %p\n", UP.task->dataStackBase.cellp );
#endif /* defined DEBUGGING */
/* 02380 	STD UCSP,X */
	UP.task->compilerStackMarker = W.bootConstantsp->initialParameterStackBase;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial compile stack mark: %p\n", UP.task->compilerStackMarker.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02390 	LDD 4,Y */
/* 02400 	STD UR0,X */
	UP.task->returnStackBase = W.bootConstantsp->initialReturnStackBase;
#if defined DEBUGGING
fprintf( standardError, "initial return stack base: %p\n", UP.task->returnStackBase.cellp );
#endif /* defined DEBUGGING */
/* 02410 	LDD -2,Y */
/* 02420 	STD UBS,X */
	UP.task->backSpaceConstant = W.bootConstantsp->terminalBackSpace;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial backspace: %lx\n", UP.task->backSpaceConstant.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02430 	LDD 6,Y */
/* 02440 	STD UTIB,X */
	UP.task->terminalInputBuffer = W.bootConstantsp->initialTerminalInputBuffer;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial terminal input buffer: %p\n", UP.task->terminalInputBuffer.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02450 	LDD $0A,Y */
/* 02460 	STD UWARN,X */
	UP.task->diskOnLine = W.bootConstantsp->initialDiskOnLine;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial disk on line flag: %lx\n", UP.task->diskOnLine.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02470 	LEAY $10,Y */
/* 02480 	LDD 2,Y */
/* 02490 	STD UPAD,X */
/* 02500 	STD UHLD,X */
	UP.task->numericConversionScratchPad = UP.task->padMarker = W.bootConstantsp->initialNumericConversionScratchPad;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial scratchpad & mark: %p\n", UP.task->numericConversionScratchPad.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02510 	LDD 4,Y */
/* 02520 	STD UWP,X */
	UP.task->wordBufferPointer = W.bootConstantsp->initialWordBufferPointer;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial word buffer pointer: %p\n", UP.task->wordBufferPointer.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02530 	LDD 6,Y */
/* 02540 	STD UCOLUM,X */
	UP.task->terminalColumns = W.bootConstantsp->initialTerminalColumns;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial terminal width: %lx\n", UP.task->terminalColumns.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02550 	LDD 8,Y */
/* 02560 	STD UFIRST,X */
/* 02570 	STD UUSE,X */
/* 02580 	STD UPREV,X */
	UP.task->firstByteOfBuffers = UP.task->leastRecentBuffer = UP.task->mostRecentBuffer = W.bootConstantsp->initialDiskBuffers;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial disk buffers: %p\n", UP.task->firstByteOfBuffers.cellp );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02590 	LDD $0A,Y */
/* 02600 	STD ULIMIT,X */
	UP.task->limitOfBufferRAM = W.bootConstantsp->initialLimitOfBufferRAM;
#if defined DEBUGGING
fprintf( standardError, "initial RAM limit: %p\n", UP.task->limitOfBufferRAM.cellp );
#endif /* defined DEBUGGING */
/* 02610 	LDD #16 */
/* 02620 	STD UBASE,X */
	UP.task->numericBase.integer = 16;
#if defined DEBUGGING && DEBUGGING <= 2
fprintf( standardError, "initial numeric base: %lx\n", UP.task->numericBase.integer );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02630 	LDD #0 */
/* 02640 	STD [UR0,X] hole */
	( * UP.task->returnStackBase.cellp ).integer = 0;
/* 02650 	STD [US0,X] hole */
	( * UP.task->dataStackBase.cellp ).integer = 0;
#if defined DEBUGGING
fprintf( standardError, "hit the holes?: %lx %lx (zeros?)\n", SP[ 0 ].integer, RP[ 0 ].integer );
#endif /* defined DEBUGGING */
	{
/* 02660 	LDA #(UEND-UIN) */
		snatural_t count = UEND - UIN;
/* 02670 	LEAY UIN,X */
		byte_t * ptr = UP.bytep + UIN;
#if defined DEBUGGING && DEBUGGING <= 2
/* Watch this. Some compilers think size_t is unsigned long, some think it is int, etc. */
fprintf( standardError, "UP.task: %p UEND: %ld, UIN: %ld\n", UP.task, (natural_t) UEND, (natural_t) UIN );
fprintf( standardError, "one past end should be %p == %p\n", UP.task->spare + UP_SPARE_COUNT, ptr + count );
fprintf( standardError, "clearing %ld bytes from: %p, UIN should be at: %p\n", count, ptr, &( UP.task->bufferInputOffset ) );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
/* 02680 WARMLY	STB ,Y+ */ /* was safe because I knew it was non-zero. */
/* 02690 	DECA */
		while ( --count >= 0 )
		{
#if defined DEBUGGING && DEBUGGING <= 2
/* dbg */ if ( ptr >= UP.bytep + UEND - 1 ) fprintf( standardError, "clear final at %p\n", ptr );
/* dbg */ if ( ptr >= (byte_p) ( UP.task->spare + UP_SPARE_COUNT ) ) fprintf( standardError, "overrun at %p\n", ptr );
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
			* ptr++ = 0;	/* Clear out the rest. Bytewise is not as fast, fix it later. */
/* 02700 	BNE WARMLY */
		}
	}
	/* Until I decide to brave curses or whatever. */
	UP.task->terminalEcho = W.bootConstantsp->initialTerminalEcho;
	/* Patching i-code list tail robbing branches, since C requires initializations to be constant. 
	** Not really a good thing to do, refactoring would be better, but I need to get it running first.
	*/
	hDIFIND.parameterLink[ 3 ].integer 
		=  sizeof (cell_u) * ( ( hDDFIND.parameterLink + 2 ) - ( hDIFIND.parameterLink + 4 ) );
	
/* 02740 	PULS CC */	/* Not messing with interrupts. */
/* 02745 	ANDCC #$EF enable IRQ (disc) */
	/* Call high level code. Maybe I want to do this a different way, maybe I don't. 
	*/
	/* This wasn't what I wanted:
	** W.cellp = iWARM + 1;	/ * Since DOCOL doesn't grab it from the stack, here, ... */
	DOCOL( &hIWARM);
	/* ( * --RP ).definitionp = &hWARM; This didn't make a good buffer, anyway. */
	/* IP = iWARM;	// * As if we just executed the first definition. Might want to make a macro for shifting gears? */
	/* And/or, absorb NEXT into the WARM boot code? No. Put it back.
	** If anything, it should be absorbed into DOCOL, but not that, either, until I check DOES(), etc.
	XNEXT();
	*/
	/*
	while ( sysSIG.integer == 0 )
	{	W = ( * IP++ );
		( * ( * W.definitionp ).codeLink.icode )();
	}
	*/
}
/* 02750 	DOCOL */
static character_t sIWARM[] = "\x5" "IWARM";
definition_header_s hIWARM =	
{	{ (natural_t) sIWARM },
	{ 0 },
	{ (natural_t) &hWARM },
	{ MFORE },
	{ (natural_t) &hUTIL },
	{ (natural_t) 0 },
	{ (natural_t) 0 },
	{ (natural_t) XCOL },
	{
#if defined DBG_BOOTSTRAP_TESTING
		{ (natural_t) &hiBOOTSTRAP_TEST },
#endif /* defined DBG_BOOTSTRAP_TESTING */
	/* ABORT Could be a security hole to (C subroutine) call from here. 
	*/
/* 02760 	FDB EMTBUF */	/* bif4/a -- EMTBUF was assembly language in my 6809 version */
		{ (natural_t) &hEMTBUF },
/* 02770 	FDB ABORT */	/* bif7b/a --> many, of course */
		{ (natural_t) &hPHABORT }	/* This must be the one that actually does the setjmp(); */
	}
};
/* 02780 	SETDP 0 */
/* 02890 * */


#if defined DEBUGGING


#	if defined DBG_BOOTSTRAP_TESTING
#include "bif_a.h"	/* for testing */
#include "bifb_a.h"	/* for testing */
static character_t siBOOTSTRAP_TEST[] = "\xf" "iBOOTSTRAP_TEST";
static definition_header_s hiBOOTSTRAP_TEST =	
{	{ (natural_t) siBOOTSTRAP_TEST },
	{ 0 },
	{ 0 },	/* no allocation link */
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },	/* Fake the link to the dummy vocabulary, just in case. (what?) */
	{ 0 },
	{ 0 },
	{ (natural_t) XCOL },
	{
#if defined DEBUGGING && DEBUGGING <= 2
		/* { (natural_t) COLD },	/ * guard-rail flag for first trip through the debugger */
		/* Fake the outer interpreter "loop" for a while. */
		/* Actually, put some test stuff in here. Probably needs an endless loop, though, or some reasonable way to cut loose. */
		/* So, say hello the hard way, tests NEXT, EMIT, and LIT: */
		{ (natural_t) &hLIT },
		{ 'h' /* 0xFACEFA4E */ }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ 'i' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ '1' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ '\n' }, 
		{ (natural_t) &hEMIT },
		/* Now test some basic stuff, FETCH and STORE: */
		{ (natural_t) &hLIT },
		{ 0xFACEFA4E }, 
		{ (natural_t) &hLIT },
		{ (natural_t) memoryImage },	/* Use the heap without managing it yet. */
		{ (natural_t) &hSTORE },
		{ (natural_t) &hLIT },
		{ (natural_t) memoryImage },	/* Check the results with the debugger. */
		{ (natural_t) &hFETCH },
		{ (natural_t) &hLIT },
		{ (natural_t) memoryImage + sizeof (cell_u) },
		{ (natural_t) &hSTORE },
		/* At this point, I only have code in bif_m, bifu_i, bifdp_a, bifst_a, bif_a, bifb_a, 
		// bif5b_a, bif6b_a, and bif7b_a. 
		*/
		/* Test shifting between leaf and non-leaf, DOCOL, DOCON, : */
		/* loop for a while. */
#	if defined DEBUGGING && DEBUGGING < 2
		{ (natural_t) &hBRANCH },
		{ -25 * sizeof (cell_u) },
#	endif /* defined DEBUGGING && DEBUGGING < 2 */
#endif /* defined DEBUGGING && DEBUGGING <= 2 */
#if defined DEBUGGING && DEBUGGING >= 2 && DEBUGGING <= 3 
		{ (natural_t) &hDUMMYCOL },
		{ (natural_t) &hDUMMYCOL1 },
		{ (natural_t) &hDUMMYFIND },
		{ (natural_t) &hBRANCH },
		{ -5 * sizeof (cell_u) },
		/* To do the colon definition, I also need SEMIS, from bif1_a . */
#endif /* defined DEBUGGING && DEBUGGING >= 2 && DEBUGGING <= 3 */
#if defined DEBUGGING && DEBUGGING > 0 && DEBUGGING < 0x10
		{ (natural_t) COLD },	/* bumper */
		{ (natural_t) COLD },	/* bumper */
		{ (natural_t) COLD }	/* bumper */
	}
};


#endif /* defined DEBUGGING && DEBUGGING > 0 && DEBUGGING < 0x10 */


#if defined DEBUGGING && DEBUGGING >= 2 && DEBUGGING <= 3

static character_t sDUMMYVOC[] = "\x8" "DUMMYVOC";
definition_header_s hDUMMYVOC =
{	{ (natural_t) sDUMMYVOC },
	{ MIMM },
	{ 0 },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XVOC }, 
	{ { (natural_t) &hDUMMYCOL } }
};

static character_t DUMMYCOLstring[] = "world\nis_crazy\n";	/* 6 to "is_crazy", 15 total */
static character_t sDUMMYCOL[] = "\x8" "DUMMYCOL";
definition_header_s hDUMMYCOL =	
{	{ (natural_t) sDUMMYCOL },
	{ 0 },
	{ (natural_t) &hDUMMYVOC },
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },
	{ (natural_t) &hDUMMYAVAR },
	{ (natural_t) &hDUMMYCOL1 },
	{ (natural_t) XCOL },
/* icode list starts here: */
	{	{ (natural_t) &hLIT },
		{ 'h' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ 'e' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ 'l' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ 'l' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ 'o' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ '2' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ '\n' }, 
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ (natural_t) DUMMYCOLstring }, 
		{ (natural_t) &hLIT },
		{ 6 }, 
		{ (natural_t) &hTYPE },
		{ (natural_t) &hSEMIS }
	}
};


static character_t BASEstring[] = "numeric output base: "; /* 21 total */
static character_t CPERLstring[] = "columns per line: "; /* 18 total */

static character_t sDUMMYCOL1[] = "\x9" "DUMMYCOL1";
definition_header_s hDUMMYCOL1 =	
{	{ (natural_t) sDUMMYCOL1 },
	{ 0 },
	{ (natural_t) &hDUMMYCOL },
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },
	{ 0 },
	{ (natural_t) &hSPARE1 },
	{ (natural_t) XCOL },
/* icode list starts here:  (Check it with the debugger.) */
	{	{ (natural_t) &hDUMMYAVAR },
		{ (natural_t) &hFETCH },
		{ (natural_t) &hDUP },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '\n' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ (natural_t) BASEstring },
		{ (natural_t) &hLIT },
		{ 21 },
		{ (natural_t) &hTYPE },
		{ (natural_t) &hBASE },
		{ (natural_t) &hFETCH },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '\n' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ (natural_t) CPERLstring },
		{ (natural_t) &hLIT },
		{ 18 },
		{ (natural_t) &hTYPE },
		{ (natural_t) &hCPERL },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '\n' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hLIT },
		{ (natural_t) memoryImage + 2 * sizeof (cell_u) },
		{ (natural_t) &hSTORE },
		{ (natural_t) &hDUMMYACON },
		{ (natural_t) &hDUMMYAVAR },
		{ (natural_t) &hSTORE },
		{ (natural_t) &hSEMIS }
	}
};

static character_t sDUMMYAVAR[] = "\x9" "DUMMYAVAR";
definition_header_s hDUMMYAVAR =	
{	{ (natural_t) sDUMMYAVAR },
	{ 0 },
	{ (natural_t) &hDUMMYCOL1 },
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },
	{ (natural_t) &hDUMMYACON },
	{ 0 },
	{ (natural_t) XVAR },
/* variable space here (But global variables should not really be used, especially in C code.): */
	{	{ 0xce11c0de }	}	/* And C is initializing it while it reserves it. */
};

static character_t sDUMMYACON[] = "\x9" "DUMMYACON";
definition_header_s hDUMMYACON =	
{	{ (natural_t) sDUMMYACON },
	{ 0 },
	{ (natural_t) &hDUMMYAVAR },
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },
	{ 0 },
	{ 0 },
	{ (natural_t) XCON },
/* space for constant value here: */
	{	{ 0xdeafbead }	}	/* And C is initializing it while it reserves it. */
};

static character_t sSPARE1[] = "\x6" "SPARE1";
definition_header_s hSPARE1 =
{	{ (natural_t) sSPARE1 },
	{ 0 },
	{ (natural_t) &hDUMMYAVAR },
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },
	{ (natural_t) &hDUMMYFIND },
	{ 0 },
	{ (natural_t) XUSER },
	{	{ USPARE }	}
};

static character_t sFAKEBUFFERgood[] = "DUMMYFIND";
static character_t sFAKEBUFFERbad[] = "DUMMYFINE";
static character_t sFOUND[] = " search result: ";	/* 16 total */

/* Test the search/find stuff with this dummy vocabulary. */
static character_t sDUMMYFIND[] = "\x9" "DUMMYFIND";
definition_header_s hDUMMYFIND =	
{	{ (natural_t) sDUMMYFIND },
	{ 0 },
	{ (natural_t) &hSPARE1 },
	{ MFORE },
	{ (natural_t) &hDUMMYVOC },
	{ 0 },
	{ 0 },
	{ (natural_t) XCOL },
/* icode list starts here: */
	{	
		{ (natural_t) &hLIT },
		{ (natural_t) &hDUMMYFIND },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ (natural_t) sFOUND },
		{ (natural_t) &hLIT },
		{ 16 },
		{ (natural_t) &hTYPE },
		
		{ (natural_t) &hLIT },
		{ (natural_t) sFAKEBUFFERgood },
		{ (natural_t) &hLIT },
		{ (natural_t) &( hDUMMYVOC.parameterLink ) },
		{ (natural_t) &hPREF },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '@' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hDUP },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ ':' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hFETCH },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '*' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '\n' },
		{ (natural_t) &hEMIT },
		
		{ (natural_t) &hLIT },
		{ (natural_t) sFAKEBUFFERbad },
		{ (natural_t) &hLIT },
		{ (natural_t) &( hDUMMYVOC.parameterLink ) },
		{ (natural_t) &hPREF },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '@' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hDUP },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ ':' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hFETCH },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '*' },
		{ (natural_t) &hEMIT },
		{ (natural_t) &hDOT },
		{ (natural_t) &hLIT },
		{ '\n' },
		{ (natural_t) &hEMIT },

		{ (natural_t) &hSEMIS }
	}
};


#endif /* defined DEBUGGING && DEBUGGING >= 2 && DEBUGGING <= 3  */

#	endif /* defined DBG_BOOTSTRAP_TESTING */

#endif /* defined DEBUGGING */
