
// CodeMirror version 2.3
//
// All functions that need access to the editor's state live inside
// the CodeMirror function. Below that, at the bottom of the file,
// some utilities are defined.

// CodeMirror is the only global var we claim
var CodeMirror = (function() {
  // This is the function that produces an editor instance. Its
  // closure is used to store the editor state.
  function CodeMirror(place, givenOptions) {
    // Determine effective options based on given values and defaults.
    var options = {}, defaults = CodeMirror.defaults;
    for (var opt in defaults)
      if (defaults.hasOwnProperty(opt))
        options[opt] = (givenOptions && givenOptions.hasOwnProperty(opt) ? givenOptions : defaults)[opt];

    // The element in which the editor lives.
    var wrapper = document.createElement("div");
    wrapper.className = "CodeMirror" + (options.lineWrapping ? " CodeMirror-wrap" : "");
    // This mess creates the base DOM structure for the editor.
    wrapper.innerHTML =
      '<div style="overflow: hidden; position: relative; width: 3px; height: 0px;">' + // Wraps and hides input textarea
        '<textarea style="position: absolute; padding: 0; width: 1px; height: 1em" wrap="off" ' +
          'autocorrect="off" autocapitalize="off"></textarea></div>' +
      '<div class="CodeMirror-scrollbar">' + // The vertical scrollbar. Horizontal scrolling is handled by the scroller itself.
        '<div class="CodeMirror-scrollbar-inner">' + // The empty scrollbar content, used solely for managing the scrollbar thumb.
      '</div></div>' + // This must be before the scroll area because it's float-right.
      '<div class="CodeMirror-scroll" tabindex="-1">' +
        '<div style="position: relative">' + // Set to the height of the text, causes scrolling
          '<div style="position: relative">' + // Moved around its parent to cover visible view
            '<div class="CodeMirror-gutter"><div class="CodeMirror-gutter-text"></div></div>' +
            // Provides positioning relative to (visible) text origin
            '<div class="CodeMirror-lines"><div style="position: relative; z-index: 0">' +
              // Used to measure text size
              '<div style="position: absolute; width: 100%; height: 0; overflow: hidden; visibility: hidden;"></div>' +
              '<pre class="CodeMirror-cursor">&#160;</pre>' + // Absolutely positioned blinky cursor
              '<pre class="CodeMirror-cursor" style="visibility: hidden">&#160;</pre>' + // Used to force a width
              '<div style="position: relative; z-index: -1"></div><div></div>' + // DIVs containing the selection and the actual code
            '</div></div></div></div></div>';
    if (place.appendChild) place.appendChild(wrapper); else place(wrapper);
    // I've never seen more elegant code in my life.
    var inputDiv = wrapper.firstChild, input = inputDiv.firstChild,
        scroller = wrapper.lastChild, code = scroller.firstChild,
        mover = code.firstChild, gutter = mover.firstChild, gutterText = gutter.firstChild,
        lineSpace = gutter.nextSibling.firstChild, measure = lineSpace.firstChild,
        cursor = measure.nextSibling, widthForcer = cursor.nextSibling,
        selectionDiv = widthForcer.nextSibling, lineDiv = selectionDiv.nextSibling,
        scrollbar = inputDiv.nextSibling, scrollbarInner = scrollbar.firstChild;
    themeChanged(); keyMapChanged();
    // Needed to hide big blue blinking cursor on Mobile Safari
    if (ios) input.style.width = "0px";
    if (!webkit) scroller.draggable = true;
    lineSpace.style.outline = "none";
    if (options.tabindex != null) input.tabIndex = options.tabindex;
    if (options.autofocus) focusInput();
    if (!options.gutter && !options.lineNumbers) gutter.style.display = "none";
    // Needed to handle Tab key in KHTML
    if (khtml) inputDiv.style.height = "1px", inputDiv.style.position = "absolute";

    // Check for OS X >= 10.7. If so, we need to force a width on the scrollbar, and 
    // make it overlap the content. (But we only do this if the scrollbar doesn't already
    // have a natural width. If the mouse is plugged in or the user sets the system pref
    // to always show scrollbars, the scrollbar shouldn't overlap.)
    if (mac_geLion) {
      scrollbar.className += (overlapScrollbars() ? " cm-sb-overlap" : " cm-sb-nonoverlap");
    } else if (ie_lt8) {
      // Need to set a minimum width to see the scrollbar on IE7 (but must not set it on IE8).
      scrollbar.className += " cm-sb-ie7";
    }

    // Check for problem with IE innerHTML not working when we have a
    // P (or similar) parent node.
    try { stringWidth("x"); }
    catch (e) {
      if (e.message.match(/runtime/i))
        e = new Error("A CodeMirror inside a P-style element does not work in Internet Explorer. (innerHTML bug)");
      throw e;
    }

    // Delayed object wrap timeouts, making sure only one is active. blinker holds an interval.
    var poll = new Delayed(), highlight = new Delayed(), blinker;

    // mode holds a mode API object. doc is the tree of Line objects,
    // work an array of lines that should be parsed, and history the
    // undo history (instance of History constructor).
    var mode, doc = new BranchChunk([new LeafChunk([new Line("")])]), work, focused;
    loadMode();
    // The selection. These are always maintained to point at valid
    // positions. Inverted is used to remember that the user is
    // selecting bottom-to-top.
    var sel = {from: {line: 0, ch: 0}, to: {line: 0, ch: 0}, inverted: false};
    // Selection-related flags. shiftSelecting obviously tracks
    // whether the user is holding shift.
    var shiftSelecting, lastClick, lastDoubleClick, lastScrollTop = 0, lastScrollLeft = 0, draggingText,
        overwrite = false, suppressEdits = false;
    // Variables used by startOperation/endOperation to track what
    // happened during the operation.
    var updateInput, userSelChange, changes, textChanged, selectionChanged, leaveInputAlone,
        gutterDirty, callbacks;
    // Current visible range (may be bigger than the view window).
    var displayOffset = 0, showingFrom = 0, showingTo = 0, lastSizeC = 0;
    // bracketHighlighted is used to remember that a bracket has been
    // marked.
    var bracketHighlighted;
    // Tracks the maximum line length so that the horizontal scrollbar
    // can be kept static when scrolling.
    var maxLine = "", updateMaxLine = false, maxLineChanged = true;
    var tabCache = {};

    // Initialize the content.
    operation(function(){setValue(options.value || ""); updateInput = false;})();
    var history = new History();

    // Register our event handlers.
    connect(scroller, "mousedown", operation(onMouseDown));
    connect(scroller, "dblclick", operation(onDoubleClick));
    connect(lineSpace, "selectstart", e_preventDefault);
    // Gecko browsers fire contextmenu *after* opening the menu, at
    // which point we can't mess with it anymore. Context menu is
    // handled in onMouseDown for Gecko.
    if (!gecko) connect(scroller, "contextmenu", onContextMenu);
    connect(scroller, "scroll", onScroll);
    connect(scrollbar, "scroll", onScroll);
    connect(scrollbar, "mousedown", function() {setTimeout(focusInput, 0);});
    connect(scroller, "mousewheel", onMouseWheel);
    connect(scroller, "DOMMouseScroll", onMouseWheel);
    connect(window, "resize", function() {updateDisplay(true);});
    connect(input, "keyup", operation(onKeyUp));
    connect(input, "input", fastPoll);
    connect(input, "keydown", operation(onKeyDown));
    connect(input, "keypress", operation(onKeyPress));
    connect(input, "focus", onFocus);
    connect(input, "blur", onBlur);

    if (options.dragDrop) {
      connect(scroller, "dragstart", onDragStart);
      function drag_(e) {
        if (options.onDragEvent && options.onDragEvent(instance, addStop(e))) return;
        e_stop(e);
      }
      connect(scroller, "dragenter", drag_);
      connect(scroller, "dragover", drag_);
      connect(scroller, "drop", operation(onDrop));
    }
    connect(scroller, "paste", function(){focusInput(); fastPoll();});
    connect(input, "paste", fastPoll);
    connect(input, "cut", operation(function(){
      if (!options.readOnly) replaceSelection("");
    }));

    // Needed to handle Tab key in KHTML
    if (khtml) connect(code, "mouseup", function() {
        if (document.activeElement == input) input.blur();
        focusInput();
    });

    // IE throws unspecified error in certain cases, when
    // trying to access activeElement before onload
    var hasFocus; try { hasFocus = (document.activeElement == input); } catch(e) { }
    if (hasFocus || options.autofocus) setTimeout(onFocus, 20);
    else onBlur();

    function isLine(l) {return l >= 0 && l < doc.size;}
    // The instance object that we'll return. Mostly calls out to
    // local functions in the CodeMirror function. Some do some extra
    // range checking and/or clipping. operation is used to wrap the
    // call so that changes it makes are tracked, and the display is
    // updated afterwards.
    var instance = wrapper.CodeMirror = {
      getValue: getValue,
      setValue: operation(setValue),
      getSelection: getSelection,
      replaceSelection: operation(replaceSelection),
      focus: function(){window.focus(); focusInput(); onFocus(); fastPoll();},
      setOption: function(option, value) {
        var oldVal = options[option];
        options[option] = value;
        if (option == "mode" || option == "indentUnit") loadMode();
        else if (option == "readOnly" && value == "nocursor") {onBlur(); input.blur();}
        else if (option == "readOnly" && !value) {resetInput(true);}
        else if (option == "theme") themeChanged();
        else if (option == "lineWrapping" && oldVal != value) operation(wrappingChanged)();
        else if (option == "tabSize") updateDisplay(true);
        else if (option == "keyMap") keyMapChanged();
        if (option == "lineNumbers" || option == "gutter" || option == "firstLineNumber" || option == "theme") {
          gutterChanged();
          updateDisplay(true);
        }
      },
      getOption: function(option) {return options[option];},
      undo: operation(undo),
      redo: operation(redo),
      indentLine: operation(function(n, dir) {
        if (typeof dir != "string") {
          if (dir == null) dir = options.smartIndent ? "smart" : "prev";
          else dir = dir ? "add" : "subtract";
        }
        if (isLine(n)) indentLine(n, dir);
      }),
      indentSelection: operation(indentSelected),
      historySize: function() {return {undo: history.done.length, redo: history.undone.length};},
      clearHistory: function() {history = new History();},
      matchBrackets: operation(function(){matchBrackets(true);}),
      getTokenAt: operation(function(pos) {
        pos = clipPos(pos);
        return getLine(pos.line).getTokenAt(mode, getStateBefore(pos.line), pos.ch);
      }),
      getStateAfter: function(line) {
        line = clipLine(line == null ? doc.size - 1: line);
        return getStateBefore(line + 1);
      },
      cursorCoords: function(start, mode) {
        if (start == null) start = sel.inverted;
        return this.charCoords(start ? sel.from : sel.to, mode);
      },
      charCoords: function(pos, mode) {
        pos = clipPos(pos);
        if (mode == "local") return localCoords(pos, false);
        if (mode == "div") return localCoords(pos, true);
        return pageCoords(pos);
      },
      coordsChar: function(coords) {
        var off = eltOffset(lineSpace);
        return coordsChar(coords.x - off.left, coords.y - off.top);
      },
      markText: operation(markText),
      setBookmark: setBookmark,
      findMarksAt: findMarksAt,
      setMarker: operation(addGutterMarker),
      clearMarker: operation(removeGutterMarker),
      setLineClass: operation(setLineClass),
      hideLine: operation(function(h) {return setLineHidden(h, true);}),
      showLine: operation(function(h) {return setLineHidden(h, false);}),
      onDeleteLine: function(line, f) {
        if (typeof line == "number") {
          if (!isLine(line)) return null;
          line = getLine(line);
        }
        (line.handlers || (line.handlers = [])).push(f);
        return line;
      },
      lineInfo: lineInfo,
      addWidget: function(pos, node, scroll, vert, horiz) {
        pos = localCoords(clipPos(pos));
        var top = pos.yBot, left = pos.x;
        node.style.position = "absolute";
        code.appendChild(node);
        if (vert == "over") top = pos.y;
        else if (vert == "near") {
          var vspace = Math.max(scroller.offsetHeight, doc.height * textHeight()),
              hspace = Math.max(code.clientWidth, lineSpace.clientWidth) - paddingLeft();
          if (pos.yBot + node.offsetHeight > vspace && pos.y > node.offsetHeight)
            top = pos.y - node.offsetHeight;
          if (left + node.offsetWidth > hspace)
            left = hspace - node.offsetWidth;
        }
        node.style.top = (top + paddingTop()) + "px";
        node.style.left = node.style.right = "";
        if (horiz == "right") {
          left = code.clientWidth - node.offsetWidth;
          node.style.right = "0px";
        } else {
          if (horiz == "left") left = 0;
          else if (horiz == "middle") left = (code.clientWidth - node.offsetWidth) / 2;
          node.style.left = (left + paddingLeft()) + "px";
        }
        if (scroll)
          scrollIntoView(left, top, left + node.offsetWidth, top + node.offsetHeight);
      },

      lineCount: function() {return doc.size;},
      clipPos: clipPos,
      getCursor: function(start) {
        if (start == null) start = sel.inverted;
        return copyPos(start ? sel.from : sel.to);
      },
      somethingSelected: function() {return !posEq(sel.from, sel.to);},
      setCursor: operation(function(line, ch, user) {
        if (ch == null && typeof line.line == "number") setCursor(line.line, line.ch, user);
        else setCursor(line, ch, user);
      }),
      setSelection: operation(function(from, to, user) {
        (user ? setSelectionUser : setSelection)(clipPos(from), clipPos(to || from));
      }),
      getLine: function(line) {if (isLine(line)) return getLine(line).text;},
      getLineHandle: function(line) {if (isLine(line)) return getLine(line);},
      setLine: operation(function(line, text) {
        if (isLine(line)) replaceRange(text, {line: line, ch: 0}, {line: line, ch: getLine(line).text.length});
      }),
      removeLine: operation(function(line) {
        if (isLine(line)) replaceRange("", {line: line, ch: 0}, clipPos({line: line+1, ch: 0}));
      }),
      replaceRange: operation(replaceRange),
      getRange: function(from, to) {return getRange(clipPos(from), clipPos(to));},

      triggerOnKeyDown: operation(onKeyDown),
      execCommand: function(cmd) {return commands[cmd](instance);},
      // Stuff used by commands, probably not much use to outside code.
      moveH: operation(moveH),
      deleteH: operation(deleteH),
      moveV: operation(moveV),
      toggleOverwrite: function() {
        if(overwrite){
          overwrite = false;
          cursor.className = cursor.className.replace(" CodeMirror-overwrite", "");
        } else {
          overwrite = true;
          cursor.className += " CodeMirror-overwrite";
        }
      },

      posFromIndex: function(off) {
        var lineNo = 0, ch;
        doc.iter(0, doc.size, function(line) {
          var sz = line.text.length + 1;
          if (sz > off) { ch = off; return true; }
          off -= sz;
          ++lineNo;
        });
        return clipPos({line: lineNo, ch: ch});
      },
      indexFromPos: function (coords) {
        if (coords.line < 0 || coords.ch < 0) return 0;
        var index = coords.ch;
        doc.iter(0, coords.line, function (line) {
          index += line.text.length + 1;
        });
        return index;
      },
      scrollTo: function(x, y) {
        if (x != null) scroller.scrollLeft = x;
        if (y != null) scrollbar.scrollTop = y;
        updateDisplay([]);
      },
      getScrollInfo: function() {
        return {x: scroller.scrollLeft, y: scrollbar.scrollTop,
                height: scrollbar.scrollHeight, width: scroller.scrollWidth};
      },

      operation: function(f){return operation(f)();},
      compoundChange: function(f){return compoundChange(f);},
      refresh: function(){
        updateDisplay(true);
        if (scrollbar.scrollHeight > lastScrollTop)
          scrollbar.scrollTop = lastScrollTop;
      },
      getInputField: function(){return input;},
      getWrapperElement: function(){return wrapper;},
      getScrollerElement: function(){return scroller;},
      getGutterElement: function(){return gutter;}
    };

    function getLine(n) { return getLineAt(doc, n); }
    function updateLineHeight(line, height) {
      gutterDirty = true;
      var diff = height - line.height;
      for (var n = line; n; n = n.parent) n.height += diff;
    }

    function setValue(code) {
      var top = {line: 0, ch: 0};
      updateLines(top, {line: doc.size - 1, ch: getLine(doc.size-1).text.length},
                  splitLines(code), top, top);
      updateInput = true;
    }
    function getValue() {
      var text = [];
      doc.iter(0, doc.size, function(line) { text.push(line.text); });
      return text.join("\n");
    }

    function onScroll(e) {
      if (lastScrollTop != scrollbar.scrollTop || lastScrollLeft != scroller.scrollLeft) {
        lastScrollTop = scrollbar.scrollTop;
        lastScrollLeft = scroller.scrollLeft;
        updateDisplay([]);
        if (options.fixedGutter) gutter.style.left = scroller.scrollLeft + "px";
        if (options.onScroll) options.onScroll(instance);
      }
    }

    function onMouseDown(e) {
      setShift(e_prop(e, "shiftKey"));
      // Check whether this is a click in a widget
      for (var n = e_target(e); n != wrapper; n = n.parentNode)
        if (n.parentNode == code && n != mover) return;

      // See if this is a click in the gutter
      for (var n = e_target(e); n != wrapper; n = n.parentNode)
        if (n.parentNode == gutterText) {
          if (options.onGutterClick)
            options.onGutterClick(instance, indexOf(gutterText.childNodes, n) + showingFrom, e);
          return e_preventDefault(e);
        }

      var start = posFromMouse(e);

      switch (e_button(e)) {
      case 3:
        if (gecko && !mac) onContextMenu(e);
        return;
      case 2:
        if (start) setCursor(start.line, start.ch, true);
        setTimeout(focusInput, 20);
        e_preventDefault(e);
        return;
      }
      // For button 1, if it was clicked inside the editor
      // (posFromMouse returning non-null), we have to adjust the
      // selection.
      if (!start) {if (e_target(e) == scroller) e_preventDefault(e); return;}

      if (!focused) onFocus();

      var now = +new Date;
      if (lastDoubleClick && lastDoubleClick.time > now - 400 && posEq(lastDoubleClick.pos, start)) {
        e_preventDefault(e);
        setTimeout(focusInput, 20);
        return selectLine(start.line);
      } else if (lastClick && lastClick.time > now - 400 && posEq(lastClick.pos, start)) {
        lastDoubleClick = {time: now, pos: start};
        e_preventDefault(e);
        return selectWordAt(start);
      } else { lastClick = {time: now, pos: start}; }

      var last = start, going;
      if (options.dragDrop && dragAndDrop && !options.readOnly && !posEq(sel.from, sel.to) &&
          !posLess(start, sel.from) && !posLess(sel.to, start)) {
        // Let the drag handler handle this.
        if (webkit) scroller.draggable = true;
        function dragEnd(e2) {
          if (webkit) scroller.draggable = false;
          draggingText = false;
          up(); drop();
          if (Math.abs(e.clientX - e2.clientX) + Math.abs(e.clientY - e2.clientY) < 10) {
            e_preventDefault(e2);
            setCursor(start.line, start.ch, true);
            focusInput();
          }
        }
        var up = connect(document, "mouseup", operation(dragEnd), true);
        var drop = connect(scroller, "drop", operation(dragEnd), true);
        draggingText = true;
        // IE's approach to draggable
        if (scroller.dragDrop) scroller.dragDrop();
        return;
      }
      e_preventDefault(e);
      setCursor(start.line, start.ch, true);

      function extend(e) {
        var cur = posFromMouse(e, true);
        if (cur && !posEq(cur, last)) {
          if (!focused) onFocus();
          last = cur;
          setSelectionUser(start, cur);
          updateInput = false;
          var visible = visibleLines();
          if (cur.line >= visible.to || cur.line < visible.from)
            going = setTimeout(operation(function(){extend(e);}), 150);
        }
      }

      function done(e) {
        clearTimeout(going);
        var cur = posFromMouse(e);
        if (cur) setSelectionUser(start, cur);
        e_preventDefault(e);
        focusInput();
        updateInput = true;
        move(); up();
      }
      var move = connect(document, "mousemove", operation(function(e) {
        clearTimeout(going);
        e_preventDefault(e);
        if (!ie && !e_button(e)) done(e);
        else extend(e);
      }), true);
      var up = connect(document, "mouseup", operation(done), true);
    }
    function onDoubleClick(e) {
      for (var n = e_target(e); n != wrapper; n = n.parentNode)
        if (n.parentNode == gutterText) return e_preventDefault(e);
      var start = posFromMouse(e);
      if (!start) return;
      lastDoubleClick = {time: +new Date, pos: start};
      e_preventDefault(e);
      selectWordAt(start);
    }
    function onDrop(e) {
      if (options.onDragEvent && options.onDragEvent(instance, addStop(e))) return;
      e.preventDefault();
      var pos = posFromMouse(e, true), files = e.dataTransfer.files;
      if (!pos || options.readOnly) return;
      if (files && files.length && window.FileReader && window.File) {
        function loadFile(file, i) {
          var reader = new FileReader;
          reader.onload = function() {
            text[i] = reader.result;
            if (++read == n) {
              pos = clipPos(pos);
              operation(function() {
                var end = replaceRange(text.join(""), pos, pos);
                setSelectionUser(pos, end);
              })();
            }
          };
          reader.readAsText(file);
        }
        var n = files.length, text = Array(n), read = 0;
        for (var i = 0; i < n; ++i) loadFile(files[i], i);
      } else {
        // Don't do a replace if the drop happened inside of the selected text.
        if (draggingText && !(posLess(pos, sel.from) || posLess(sel.to, pos))) return;
        try {
          var text = e.dataTransfer.getData("Text");
          if (text) {
            compoundChange(function() {
              var curFrom = sel.from, curTo = sel.to;
              setSelectionUser(pos, pos);
              if (draggingText) replaceRange("", curFrom, curTo);
              replaceSelection(text);
              focusInput();
            });
          }
        }
        catch(e){}
      }
    }
    function onDragStart(e) {
      var txt = getSelection();
      e.dataTransfer.setData("Text", txt);
      
      // Use dummy image instead of default browsers image.
      if (gecko || chrome || opera) {
        var img = document.createElement('img');
        img.scr = 'data:image/gif;base64,R0lGODdhAgACAIAAAAAAAP///ywAAAAAAgACAAACAoRRADs='; //1x1 image
        e.dataTransfer.setDragImage(img, 0, 0);
      }
    }

    function doHandleBinding(bound, dropShift) {
      if (typeof bound == "string") {
        bound = commands[bound];
        if (!bound) return false;
      }
      var prevShift = shiftSelecting;
      try {
        if (options.readOnly) suppressEdits = true;
        if (dropShift) shiftSelecting = null;
        bound(instance);
      } catch(e) {
        if (e != Pass) throw e;
        return false;
      } finally {
        shiftSelecting = prevShift;
        suppressEdits = false;
      }
      return true;
    }
    function handleKeyBinding(e) {
      // Handle auto keymap transitions
      var startMap = getKeyMap(options.keyMap), next = startMap.auto;
      clearTimeout(maybeTransition);
      if (next && !isModifierKey(e)) maybeTransition = setTimeout(function() {
        if (getKeyMap(options.keyMap) == startMap) {
          options.keyMap = (next.call ? next.call(null, instance) : next);
        }
      }, 50);

      var name = keyNames[e_prop(e, "keyCode")], handled = false;
      if (name == null || e.altGraphKey) return false;
      if (e_prop(e, "altKey")) name = "Alt-" + name;
      if (e_prop(e, "ctrlKey")) name = "Ctrl-" + name;
      if (e_prop(e, "metaKey")) name = "Cmd-" + name;

      var stopped = false;
      function stop() { stopped = true; }

      if (e_prop(e, "shiftKey")) {
        handled = lookupKey("Shift-" + name, options.extraKeys, options.keyMap,
                            function(b) {return doHandleBinding(b, true);}, stop)
               || lookupKey(name, options.extraKeys, options.keyMap, function(b) {
                 if (typeof b == "string" && /^go[A-Z]/.test(b)) return doHandleBinding(b);
               }, stop);
      } else {
        handled = lookupKey(name, options.extraKeys, options.keyMap, doHandleBinding, stop);
      }
      if (stopped) handled = false;
      if (handled) {
        e_preventDefault(e);
        restartBlink();
        if (ie) { e.oldKeyCode = e.keyCode; e.keyCode = 0; }
      }
      return handled;
    }
    function handleCharBinding(e, ch) {
      var handled = lookupKey("'" + ch + "'", options.extraKeys,
                              options.keyMap, function(b) { return doHandleBinding(b, true); });
      if (handled) {
        e_preventDefault(e);
        restartBlink();
      }
      return handled;
    }

    var lastStoppedKey = null, maybeTransition;
    function onKeyDown(e) {
      if (!focused) onFocus();
      if (ie && e.keyCode == 27) { e.returnValue = false; }
      if (pollingFast) { if (readInput()) pollingFast = false; }
      if (options.onKeyEvent && options.onKeyEvent(instance, addStop(e))) return;
      var code = e_prop(e, "keyCode");
      // IE does strange things with escape.
      setShift(code == 16 || e_prop(e, "shiftKey"));
      // First give onKeyEvent option a chance to handle this.
      var handled = handleKeyBinding(e);
      if (opera) {
        lastStoppedKey = handled ? code : null;
        // Opera has no cut event... we try to at least catch the key combo
        if (!handled && code == 88 && e_prop(e, mac ? "metaKey" : "ctrlKey"))
          replaceSelection("");
      }
    }
    function onKeyPress(e) {
      if (pollingFast) readInput();
      if (options.onKeyEvent && options.onKeyEvent(instance, addStop(e))) return;
      var keyCode = e_prop(e, "keyCode"), charCode = e_prop(e, "charCode");
      if (opera && keyCode == lastStoppedKey) {lastStoppedKey = null; e_preventDefault(e); return;}
      if (((opera && (!e.which || e.which < 10)) || khtml) && handleKeyBinding(e)) return;
      var ch = String.fromCharCode(charCode == null ? keyCode : charCode);
      if (options.electricChars && mode.electricChars && options.smartIndent && !options.readOnly) {
        if (mode.electricChars.indexOf(ch) > -1)
          setTimeout(operation(function() {indentLine(sel.to.line, "smart");}), 75);
      }
      if (handleCharBinding(e, ch)) return;
      fastPoll();
    }
    function onKeyUp(e) {
      if (options.onKeyEvent && options.onKeyEvent(instance, addStop(e))) return;
      if (e_prop(e, "keyCode") == 16) shiftSelecting = null;
    }

    function onFocus() {
      if (options.readOnly == "nocursor") return;
      if (!focused) {
        if (options.onFocus) options.onFocus(instance);
        focused = true;
        if (scroller.className.search(/\bCodeMirror-focused\b/) == -1)
          scroller.className += " CodeMirror-focused";
        if (!leaveInputAlone) resetInput(true);
      }
      slowPoll();
      restartBlink();
    }
    function onBlur() {
      if (focused) {
        if (options.onBlur) options.onBlur(instance);
        focused = false;
        if (bracketHighlighted)
          operation(function(){
            if (bracketHighlighted) { bracketHighlighted(); bracketHighlighted = null; }
          })();
        scroller.className = scroller.className.replace(" CodeMirror-focused", "");
      }
      clearInterval(blinker);
      setTimeout(function() {if (!focused) shiftSelecting = null;}, 150);
    }

    function chopDelta(delta) {
      // Make sure we always scroll a little bit for any nonzero delta.
      if (delta > 0.0 && delta < 1.0) return 1;
      else if (delta > -1.0 && delta < 0.0) return -1;
      else return Math.round(delta);
    }

    function onMouseWheel(e) {
      var deltaX = 0, deltaY = 0;
      if (e.type == "DOMMouseScroll") { // Firefox
        var delta = -e.detail * 8.0;
        if (e.axis == e.HORIZONTAL_AXIS) deltaX = delta;
        else if (e.axis == e.VERTICAL_AXIS) deltaY = delta;
      } else if (e.wheelDeltaX !== undefined && e.wheelDeltaY !== undefined) { // WebKit
        deltaX = e.wheelDeltaX / 3.0;
        deltaY = e.wheelDeltaY / 3.0;
      } else if (e.wheelDelta !== undefined) { // IE or Opera
        deltaY = e.wheelDelta / 3.0;
      }

      var scrolled = false;
      deltaX = chopDelta(deltaX);
      deltaY = chopDelta(deltaY);
      if ((deltaX > 0 && scroller.scrollLeft > 0) ||
          (deltaX < 0 && scroller.scrollLeft + scroller.clientWidth < scroller.scrollWidth)) {
        scroller.scrollLeft -= deltaX;
        scrolled = true;
      }
      if ((deltaY > 0 && scrollbar.scrollTop > 0) ||
          (deltaY < 0 && scrollbar.scrollTop + scrollbar.clientHeight < scrollbar.scrollHeight)) {
        scrollbar.scrollTop -= deltaY;
        scrolled = true;
      }
      if (scrolled) e_stop(e);
    }

    // Replace the range from from to to by the strings in newText.
    // Afterwards, set the selection to selFrom, selTo.
    function updateLines(from, to, newText, selFrom, selTo) {
      if (suppressEdits) return;
      if (history) {
        var old = [];
        doc.iter(from.line, to.line + 1, function(line) { old.push(line.text); });
        history.addChange(from.line, newText.length, old);
        while (history.done.length > options.undoDepth) history.done.shift();
      }
      updateLinesNoUndo(from, to, newText, selFrom, selTo);
    }
    function unredoHelper(from, to) {
      if (!from.length) return;
      var set = from.pop(), out = [];
      for (var i = set.length - 1; i >= 0; i -= 1) {
        var change = set[i];
        var replaced = [], end = change.start + change.added;
        doc.iter(change.start, end, function(line) { replaced.push(line.text); });
        out.push({start: change.start, added: change.old.length, old: replaced});
        var pos = {line: change.start + change.old.length - 1,
                   ch: editEnd(replaced[replaced.length-1], change.old[change.old.length-1])};
        updateLinesNoUndo({line: change.start, ch: 0}, {line: end - 1, ch: getLine(end-1).text.length}, change.old, pos, pos);
      }
      updateInput = true;
      to.push(out);
    }
    function undo() {unredoHelper(history.done, history.undone);}
    function redo() {unredoHelper(history.undone, history.done);}

    function updateLinesNoUndo(from, to, newText, selFrom, selTo) {
      if (suppressEdits) return;
      var recomputeMaxLength = false, maxLineLength = maxLine.length;
      if (!options.lineWrapping)
        doc.iter(from.line, to.line + 1, function(line) {
          if (!line.hidden && line.text.length == maxLineLength) {recomputeMaxLength = true; return true;}
        });
      if (from.line != to.line || newText.length > 1) gutterDirty = true;

      var nlines = to.line - from.line, firstLine = getLine(from.line), lastLine = getLine(to.line);
      // First adjust the line structure, taking some care to leave highlighting intact.
      if (from.ch == 0 && to.ch == 0 && newText[newText.length - 1] == "") {
        // This is a whole-line replace. Treated specially to make
        // sure line objects move the way they are supposed to.
        var added = [], prevLine = null;
        if (from.line) {
          prevLine = getLine(from.line - 1);
          prevLine.fixMarkEnds(lastLine);
        } else lastLine.fixMarkStarts();
        for (var i = 0, e = newText.length - 1; i < e; ++i)
          added.push(Line.inheritMarks(newText[i], prevLine));
        if (nlines) doc.remove(from.line, nlines, callbacks);
        if (added.length) doc.insert(from.line, added);
      } else if (firstLine == lastLine) {
        if (newText.length == 1)
          firstLine.replace(from.ch, to.ch, newText[0]);
        else {
          lastLine = firstLine.split(to.ch, newText[newText.length-1]);
          firstLine.replace(from.ch, null, newText[0]);
          firstLine.fixMarkEnds(lastLine);
          var added = [];
          for (var i = 1, e = newText.length - 1; i < e; ++i)
            added.push(Line.inheritMarks(newText[i], firstLine));
          added.push(lastLine);
          doc.insert(from.line + 1, added);
        }
      } else if (newText.length == 1) {
        firstLine.replace(from.ch, null, newText[0]);
        lastLine.replace(null, to.ch, "");
        firstLine.append(lastLine);
        doc.remove(from.line + 1, nlines, callbacks);
      } else {
        var added = [];
        firstLine.replace(from.ch, null, newText[0]);
        lastLine.replace(null, to.ch, newText[newText.length-1]);
        firstLine.fixMarkEnds(lastLine);
        for (var i = 1, e = newText.length - 1; i < e; ++i)
          added.push(Line.inheritMarks(newText[i], firstLine));
        if (nlines > 1) doc.remove(from.line + 1, nlines - 1, callbacks);
        doc.insert(from.line + 1, added);
      }
      if (options.lineWrapping) {
        var perLine = Math.max(5, scroller.clientWidth / charWidth() - 3);
        doc.iter(from.line, from.line + newText.length, function(line) {
          if (line.hidden) return;
          var guess = Math.ceil(line.text.length / perLine) || 1;
          if (guess != line.height) updateLineHeight(line, guess);
        });
      } else {
        doc.iter(from.line, from.line + newText.length, function(line) {
          var l = line.text;
          if (!line.hidden && l.length > maxLineLength) {
            maxLine = l; maxLineLength = l.length; maxLineChanged = true;
            recomputeMaxLength = false;
          }
        });
        if (recomputeMaxLength) updateMaxLine = true;
      }

      // Add these lines to the work array, so that they will be
      // highlighted. Adjust work lines if lines were added/removed.
      var newWork = [], lendiff = newText.length - nlines - 1;
      for (var i = 0, l = work.length; i < l; ++i) {
        var task = work[i];
        if (task < from.line) newWork.push(task);
        else if (task > to.line) newWork.push(task + lendiff);
      }
      var hlEnd = from.line + Math.min(newText.length, 500);
      highlightLines(from.line, hlEnd);
      newWork.push(hlEnd);
      work = newWork;
      startWorker(100);
      // Remember that these lines changed, for updating the display
      changes.push({from: from.line, to: to.line + 1, diff: lendiff});
      var changeObj = {from: from, to: to, text: newText};
      if (textChanged) {
        for (var cur = textChanged; cur.next; cur = cur.next) {}
        cur.next = changeObj;
      } else textChanged = changeObj;

      // Update the selection
      function updateLine(n) {return n <= Math.min(to.line, to.line + lendiff) ? n : n + lendiff;}
      setSelection(clipPos(selFrom), clipPos(selTo),
                   updateLine(sel.from.line), updateLine(sel.to.line));
    }

    function updateVerticalScroll(scrollTop) {
      var th = textHeight(), virtualHeight = Math.floor(doc.height * th + 2 * paddingTop()), scrollbarHeight = scroller.clientHeight;
      scrollbar.style.height = scrollbarHeight + "px";
      if (scroller.clientHeight)
        scrollbarInner.style.height = virtualHeight + "px";
      // Position the mover div to align with the current virtual scroll position
      if (scrollTop != null) scrollbar.scrollTop = scrollTop;
      mover.style.top = (displayOffset * th - scrollbar.scrollTop) + "px";
      scrollbar.style.display = (virtualHeight > scrollbarHeight) ? "block" : "none";
    }
  
    // On Mac OS X Lion and up, detect whether the mouse is plugged in by measuring 
    // the width of a div with a scrollbar in it. If the width is <= 1, then
    // the mouse isn't plugged in and scrollbars should overlap the content.
    function overlapScrollbars() {
      var tmpSb = document.createElement('div'),
          tmpSbInner = document.createElement('div');
      tmpSb.className = "CodeMirror-scrollbar";
      tmpSb.style.cssText = "position: absolute; left: -9999px; height: 100px;";
      tmpSbInner.className = "CodeMirror-scrollbar-inner";
      tmpSbInner.style.height = "200px";
      tmpSb.appendChild(tmpSbInner);

      document.body.appendChild(tmpSb);
      var result = (tmpSb.offsetWidth <= 1);
      document.body.removeChild(tmpSb);
      return result;
    }

    function computeMaxLength() {
      var maxLineLength = 0; 
      maxLine = ""; maxLineChanged = true;
      doc.iter(0, doc.size, function(line) {
        var l = line.text;
        if (!line.hidden && l.length > maxLineLength) {
          maxLineLength = l.length; maxLine = l;
        }
      });
      updateMaxLine = false;
    }

    function replaceRange(code, from, to) {
      from = clipPos(from);
      if (!to) to = from; else to = clipPos(to);
      code = splitLines(code);
      function adjustPos(pos) {
        if (posLess(pos, from)) return pos;
        if (!posLess(to, pos)) return end;
        var line = pos.line + code.length - (to.line - from.line) - 1;
        var ch = pos.ch;
        if (pos.line == to.line)
          ch += code[code.length-1].length - (to.ch - (to.line == from.line ? from.ch : 0));
        return {line: line, ch: ch};
      }
      var end;
      replaceRange1(code, from, to, function(end1) {
        end = end1;
        return {from: adjustPos(sel.from), to: adjustPos(sel.to)};
      });
      return end;
    }
    function replaceSelection(code, collapse) {
      replaceRange1(splitLines(code), sel.from, sel.to, function(end) {
        if (collapse == "end") return {from: end, to: end};
        else if (collapse == "start") return {from: sel.from, to: sel.from};
        else return {from: sel.from, to: end};
      });
    }
    function replaceRange1(code, from, to, computeSel) {
      var endch = code.length == 1 ? code[0].length + from.ch : code[code.length-1].length;
      var newSel = computeSel({line: from.line + code.length - 1, ch: endch});
      updateLines(from, to, code, newSel.from, newSel.to);
    }

    function getRange(from, to) {
      var l1 = from.line, l2 = to.line;
      if (l1 == l2) return getLine(l1).text.slice(from.ch, to.ch);
      var code = [getLine(l1).text.slice(from.ch)];
      doc.iter(l1 + 1, l2, function(line) { code.push(line.text); });
      code.push(getLine(l2).text.slice(0, to.ch));
      return code.join("\n");
    }
    function getSelection() {
      return getRange(sel.from, sel.to);
    }

    var pollingFast = false; // Ensures slowPoll doesn't cancel fastPoll
    function slowPoll() {
      if (pollingFast) return;
      poll.set(options.pollInterval, function() {
        startOperation();
        readInput();
        if (focused) slowPoll();
        endOperation();
      });
    }
    function fastPoll() {
      var missed = false;
      pollingFast = true;
      function p() {
        startOperation();
        var changed = readInput();
        if (!changed && !missed) {missed = true; poll.set(60, p);}
        else {pollingFast = false; slowPoll();}
        endOperation();
      }
      poll.set(20, p);
    }

    // Previnput is a hack to work with IME. If we reset the textarea
    // on every change, that breaks IME. So we look for changes
    // compared to the previous content instead. (Modern browsers have
    // events that indicate IME taking place, but these are not widely
    // supported or compatible enough yet to rely on.)
    var prevInput = "";
    function readInput() {
      if (leaveInputAlone || !focused || hasSelection(input) || options.readOnly) return false;
      var text = input.value;
      if (text == prevInput) return false;
      shiftSelecting = null;
      var same = 0, l = Math.min(prevInput.length, text.length);
      while (same < l && prevInput[same] == text[same]) ++same;
      if (same < prevInput.length)
        sel.from = {line: sel.from.line, ch: sel.from.ch - (prevInput.length - same)};
      else if (overwrite && posEq(sel.from, sel.to))
        sel.to = {line: sel.to.line, ch: Math.min(getLine(sel.to.line).text.length, sel.to.ch + (text.length - same))};
      replaceSelection(text.slice(same), "end");
      if (text.length > 1000) { input.value = prevInput = ""; }
      else prevInput = text;
      return true;
    }
    function resetInput(user) {
      if (!posEq(sel.from, sel.to)) {
        prevInput = "";
        input.value = getSelection();
        selectInput(input);
      } else if (user) prevInput = input.value = "";
    }

    function focusInput() {
      if (options.readOnly != "nocursor") input.focus();
    }

    function scrollEditorIntoView() {
      if (!cursor.getBoundingClientRect) return;
      var rect = cursor.getBoundingClientRect();
      // IE returns bogus coordinates when the instance sits inside of an iframe and the cursor is hidden
      if (ie && rect.top == rect.bottom) return;
      var winH = window.innerHeight || Math.max(document.body.offsetHeight, document.documentElement.offsetHeight);
      if (rect.top < 0 || rect.bottom > winH) scrollCursorIntoView();
    }
    function scrollCursorIntoView() {
      var coords = calculateCursorCoords();
      return scrollIntoView(coords.x, coords.y, coords.x, coords.yBot);
    }
    function calculateCursorCoords() {
      var cursor = localCoords(sel.inverted ? sel.from : sel.to);
      var x = options.lineWrapping ? Math.min(cursor.x, lineSpace.offsetWidth) : cursor.x;
      return {x: x, y: cursor.y, yBot: cursor.yBot};
    }
    function scrollIntoView(x1, y1, x2, y2) {
      var scrollPos = calculateScrollPos(x1, y1, x2, y2), scrolled = false;
      if (scrollPos.scrollLeft != null) {scroller.scrollLeft = scrollPos.scrollLeft; scrolled = true;}
      if (scrollPos.scrollTop != null) {scrollbar.scrollTop = scrollPos.scrollTop; scrolled = true;}
      if (scrolled && options.onScroll) options.onScroll(instance);
    }
    function calculateScrollPos(x1, y1, x2, y2) {
      var pl = paddingLeft(), pt = paddingTop();
      y1 += pt; y2 += pt; x1 += pl; x2 += pl;
      var screen = scroller.clientHeight, screentop = scrollbar.scrollTop, result = {};
      var atTop = y1 < paddingTop() + 10;
      if (y1 < screentop) result.scrollTop = atTop ? 0 : Math.max(0, y1);
      else if (y2 > screentop + screen) result.scrollTop = y2 - screen;

      var screenw = scroller.clientWidth, screenleft = scroller.scrollLeft;
      var gutterw = options.fixedGutter ? gutter.clientWidth : 0;
      var atLeft = x1 < gutterw + pl + 10;
      if (x1 < screenleft + gutterw || atLeft) {
        if (atLeft) x1 = 0;
        result.scrollLeft = Math.max(0, x1 - 10 - gutterw);
      } else if (x2 > screenw + screenleft - 3) {
        result.scrollLeft = x2 + 10 - screenw;
      }
      return result;
    }

    function visibleLines(scrollTop) {
      var lh = textHeight(), top = (scrollTop != null ? scrollTop : scrollbar.scrollTop) - paddingTop();
      var fromHeight = Math.max(0, Math.floor(top / lh));
      var toHeight = Math.ceil((top + scroller.clientHeight) / lh);
      return {from: lineAtHeight(doc, fromHeight),
              to: lineAtHeight(doc, toHeight)};
    }
    // Uses a set of changes plus the current scroll position to
    // determine which DOM updates have to be made, and makes the
    // updates.
    function updateDisplay(changes, suppressCallback, scrollTop) {
      if (!scroller.clientWidth) {
        showingFrom = showingTo = displayOffset = 0;
        return;
      }
      // Compute the new visible window
      // If scrollTop is specified, use that to determine which lines
      // to render instead of the current scrollbar position.
      var visible = visibleLines(scrollTop);
      // Bail out if the visible area is already rendered and nothing changed.
      if (changes !== true && changes.length == 0 && visible.from > showingFrom && visible.to < showingTo) {
        updateVerticalScroll(scrollTop);
        return;
      }
      var from = Math.max(visible.from - 100, 0), to = Math.min(doc.size, visible.to + 100);
      if (showingFrom < from && from - showingFrom < 20) from = showingFrom;
      if (showingTo > to && showingTo - to < 20) to = Math.min(doc.size, showingTo);

      // Create a range of theoretically intact lines, and punch holes
      // in that using the change info.
      var intact = changes === true ? [] :
        computeIntact([{from: showingFrom, to: showingTo, domStart: 0}], changes);
      // Clip off the parts that won't be visible
      var intactLines = 0;
      for (var i = 0; i < intact.length; ++i) {
        var range = intact[i];
        if (range.from < from) {range.domStart += (from - range.from); range.from = from;}
        if (range.to > to) range.to = to;
        if (range.from >= range.to) intact.splice(i--, 1);
        else intactLines += range.to - range.from;
      }
      if (intactLines == to - from && from == showingFrom && to == showingTo) {
        updateVerticalScroll(scrollTop);
        return;
      }
      intact.sort(function(a, b) {return a.domStart - b.domStart;});

      var th = textHeight(), gutterDisplay = gutter.style.display;
      lineDiv.style.display = "none";
      patchDisplay(from, to, intact);
      lineDiv.style.display = gutter.style.display = "";

      var different = from != showingFrom || to != showingTo || lastSizeC != scroller.clientHeight + th;
      // This is just a bogus formula that detects when the editor is
      // resized or the font size changes.
      if (different) lastSizeC = scroller.clientHeight + th;
      showingFrom = from; showingTo = to;
      displayOffset = heightAtLine(doc, from);

      // Since this is all rather error prone, it is honoured with the
      // only assertion in the whole file.
      if (lineDiv.childNodes.length != showingTo - showingFrom)
        throw new Error("BAD PATCH! " + JSON.stringify(intact) + " size=" + (showingTo - showingFrom) +
                        " nodes=" + lineDiv.childNodes.length);

      function checkHeights() {
        var curNode = lineDiv.firstChild, heightChanged = false;
        doc.iter(showingFrom, showingTo, function(line) {
          if (!line.hidden) {
            var height = Math.round(curNode.offsetHeight / th) || 1;
            if (line.height != height) {
              updateLineHeight(line, height);
              gutterDirty = heightChanged = true;
            }
          }
          curNode = curNode.nextSibling;
        });
        return heightChanged;
      }

      if (options.lineWrapping) {
        // Guess whether we're going to need the scrollbar, so that we don't end up changing the linewrapping
        // after the scrollbar appears (during updateVerticalScroll()). Only do this if the scrollbar is
        // appearing (if it's disappearing, we don't have to worry about the scroll position, and there are
        // issues on IE7 if we turn it off too early).
        var virtualHeight = Math.floor(doc.height * th + 2 * paddingTop()), scrollbarHeight = scroller.clientHeight;
        if (virtualHeight > scrollbarHeight) scrollbar.style.display = "block";
        checkHeights();
      }

      gutter.style.display = gutterDisplay;
      if (different || gutterDirty) {
        // If the gutter grew in size, re-check heights. If those changed, re-draw gutter.
        updateGutter() && options.lineWrapping && checkHeights() && updateGutter();
      }
      updateSelection();
      updateVerticalScroll(scrollTop);
      if (!suppressCallback && options.onUpdate) options.onUpdate(instance);
      return true;
    }

    function computeIntact(intact, changes) {
      for (var i = 0, l = changes.length || 0; i < l; ++i) {
        var change = changes[i], intact2 = [], diff = change.diff || 0;
        for (var j = 0, l2 = intact.length; j < l2; ++j) {
          var range = intact[j];
          if (change.to <= range.from && change.diff)
            intact2.push({from: range.from + diff, to: range.to + diff,
                          domStart: range.domStart});
          else if (change.to <= range.from || change.from >= range.to)
            intact2.push(range);
          else {
            if (change.from > range.from)
              intact2.push({from: range.from, to: change.from, domStart: range.domStart});
            if (change.to < range.to)
              intact2.push({from: change.to + diff, to: range.to + diff,
                            domStart: range.domStart + (change.to - range.from)});
          }
        }
        intact = intact2;
      }
      return intact;
    }

    function patchDisplay(from, to, intact) {
      // The first pass removes the DOM nodes that aren't intact.
      if (!intact.length) lineDiv.innerHTML = "";
      else {
        function killNode(node) {
          var tmp = node.nextSibling;
          node.parentNode.removeChild(node);
          return tmp;
        }
        var domPos = 0, curNode = lineDiv.firstChild, n;
        for (var i = 0; i < intact.length; ++i) {
          var cur = intact[i];
          while (cur.domStart > domPos) {curNode = killNode(curNode); domPos++;}
          for (var j = 0, e = cur.to - cur.from; j < e; ++j) {curNode = curNode.nextSibling; domPos++;}
        }
        while (curNode) curNode = killNode(curNode);
      }
      // This pass fills in the lines that actually changed.
      var nextIntact = intact.shift(), curNode = lineDiv.firstChild, j = from;
      var scratch = document.createElement("div");
      doc.iter(from, to, function(line) {
        if (nextIntact && nextIntact.to == j) nextIntact = intact.shift();
        if (!nextIntact || nextIntact.from > j) {
          if (line.hidden) var html = scratch.innerHTML = "<pre></pre>";
          else {
            var html = '<pre' + (line.className ? ' class="' + line.className + '"' : '') + '>'
              + line.getHTML(makeTab) + '</pre>';
            // Kludge to make sure the styled element lies behind the selection (by z-index)
            if (line.bgClassName)
              html = '<div style="position: relative"><pre class="' + line.bgClassName +
              '" style="position: absolute; left: 0; right: 0; top: 0; bottom: 0; z-index: -2">&#160;</pre>' + html + "</div>";
          }
          scratch.innerHTML = html;
          lineDiv.insertBefore(scratch.firstChild, curNode);
        } else {
          curNode = curNode.nextSibling;
        }
        ++j;
      });
    }

    function updateGutter() {
      if (!options.gutter && !options.lineNumbers) return;
      var hText = mover.offsetHeight, hEditor = scroller.clientHeight;
      gutter.style.height = (hText - hEditor < 2 ? hEditor : hText) + "px";
      var html = [], i = showingFrom, normalNode;
      doc.iter(showingFrom, Math.max(showingTo, showingFrom + 1), function(line) {
        if (line.hidden) {
          html.push("<pre></pre>");
        } else {
          var marker = line.gutterMarker;
          var text = options.lineNumbers ? i + options.firstLineNumber : null;
          if (marker && marker.text)
            text = marker.text.replace("%N%", text != null ? text : "");
          else if (text == null)
            text = "\u00a0";
          html.push((marker && marker.style ? '<pre class="' + marker.style + '">' : "<pre>"), text);
          for (var j = 1; j < line.height; ++j) html.push("<br/>&#160;");
          html.push("</pre>");
          if (!marker) normalNode = i;
        }
        ++i;
      });
      gutter.style.display = "none";
      gutterText.innerHTML = html.join("");
      // Make sure scrolling doesn't cause number gutter size to pop
      if (normalNode != null && options.lineNumbers) {
        var node = gutterText.childNodes[normalNode - showingFrom];
        var minwidth = String(doc.size).length, val = eltText(node.firstChild), pad = "";
        while (val.length + pad.length < minwidth) pad += "\u00a0";
        if (pad) node.insertBefore(document.createTextNode(pad), node.firstChild);
      }
      gutter.style.display = "";
      var resized = Math.abs((parseInt(lineSpace.style.marginLeft) || 0) - gutter.offsetWidth) > 2;
      lineSpace.style.marginLeft = gutter.offsetWidth + "px";
      gutterDirty = false;
      return resized;
    }
    function updateSelection() {
      var collapsed = posEq(sel.from, sel.to);
      var fromPos = localCoords(sel.from, true);
      var toPos = collapsed ? fromPos : localCoords(sel.to, true);
      var headPos = sel.inverted ? fromPos : toPos, th = textHeight();
      var wrapOff = eltOffset(wrapper), lineOff = eltOffset(lineDiv);
      inputDiv.style.top = Math.max(0, Math.min(scroller.offsetHeight, headPos.y + lineOff.top - wrapOff.top)) + "px";
      inputDiv.style.left = Math.max(0, Math.min(scroller.offsetWidth, headPos.x + lineOff.left - wrapOff.left)) + "px";
      if (collapsed) {
        cursor.style.top = headPos.y + "px";
        cursor.style.left = (options.lineWrapping ? Math.min(headPos.x, lineSpace.offsetWidth) : headPos.x) + "px";
        cursor.style.display = "";
        selectionDiv.style.display = "none";
      } else {
        var sameLine = fromPos.y == toPos.y, html = "";
        var clientWidth = lineSpace.clientWidth || lineSpace.offsetWidth;
        var clientHeight = lineSpace.clientHeight || lineSpace.offsetHeight;
        function add(left, top, right, height) {
          var rstyle = quirksMode ? "width: " + (!right ? clientWidth : clientWidth - right - left) + "px"
                                  : "right: " + right + "px";
          html += '<div class="CodeMirror-selected" style="position: absolute; left: ' + left +
            'px; top: ' + top + 'px; ' + rstyle + '; height: ' + height + 'px"></div>';
        }
        if (sel.from.ch && fromPos.y >= 0) {
          var right = sameLine ? clientWidth - toPos.x : 0;
          add(fromPos.x, fromPos.y, right, th);
        }
        var middleStart = Math.max(0, fromPos.y + (sel.from.ch ? th : 0));
        var middleHeight = Math.min(toPos.y, clientHeight) - middleStart;
        if (middleHeight > 0.2 * th)
          add(0, middleStart, 0, middleHeight);
        if ((!sameLine || !sel.from.ch) && toPos.y < clientHeight - .5 * th)
          add(0, toPos.y, clientWidth - toPos.x, th);
        selectionDiv.innerHTML = html;
        cursor.style.display = "none";
        selectionDiv.style.display = "";
      }
    }

    function setShift(val) {
      if (val) shiftSelecting = shiftSelecting || (sel.inverted ? sel.to : sel.from);
      else shiftSelecting = null;
    }
    function setSelectionUser(from, to) {
      var sh = shiftSelecting && clipPos(shiftSelecting);
      if (sh) {
        if (posLess(sh, from)) from = sh;
        else if (posLess(to, sh)) to = sh;
      }
      setSelection(from, to);
      userSelChange = true;
    }
    // Update the selection. Last two args are only used by
    // updateLines, since they have to be expressed in the line
    // numbers before the update.
    function setSelection(from, to, oldFrom, oldTo) {
      goalColumn = null;
      if (oldFrom == null) {oldFrom = sel.from.line; oldTo = sel.to.line;}
      if (posEq(sel.from, from) && posEq(sel.to, to)) return;
      if (posLess(to, from)) {var tmp = to; to = from; from = tmp;}

      // Skip over hidden lines.
      if (from.line != oldFrom) {
        var from1 = skipHidden(from, oldFrom, sel.from.ch);
        // If there is no non-hidden line left, force visibility on current line
        if (!from1) setLineHidden(from.line, false);
        else from = from1;
      }
      if (to.line != oldTo) to = skipHidden(to, oldTo, sel.to.ch);

      if (posEq(from, to)) sel.inverted = false;
      else if (posEq(from, sel.to)) sel.inverted = false;
      else if (posEq(to, sel.from)) sel.inverted = true;

      if (options.autoClearEmptyLines && posEq(sel.from, sel.to)) {
        var head = sel.inverted ? from : to;
        if (head.line != sel.from.line && sel.from.line < doc.size) {
          var oldLine = getLine(sel.from.line);
          if (/^\s+$/.test(oldLine.text))
            setTimeout(operation(function() {
              if (oldLine.parent && /^\s+$/.test(oldLine.text)) {
                var no = lineNo(oldLine);
                replaceRange("", {line: no, ch: 0}, {line: no, ch: oldLine.text.length});
              }
            }, 10));
        }
      }

      sel.from = from; sel.to = to;
      selectionChanged = true;
    }
    function skipHidden(pos, oldLine, oldCh) {
      function getNonHidden(dir) {
        var lNo = pos.line + dir, end = dir == 1 ? doc.size : -1;
        while (lNo != end) {
          var line = getLine(lNo);
          if (!line.hidden) {
            var ch = pos.ch;
            if (toEnd || ch > oldCh || ch > line.text.length) ch = line.text.length;
            return {line: lNo, ch: ch};
          }
          lNo += dir;
        }
      }
      var line = getLine(pos.line);
      var toEnd = pos.ch == line.text.length && pos.ch != oldCh;
      if (!line.hidden) return pos;
      if (pos.line >= oldLine) return getNonHidden(1) || getNonHidden(-1);
      else return getNonHidden(-1) || getNonHidden(1);
    }
    function setCursor(line, ch, user) {
      var pos = clipPos({line: line, ch: ch || 0});
      (user ? setSelectionUser : setSelection)(pos, pos);
    }

    function clipLine(n) {return Math.max(0, Math.min(n, doc.size-1));}
    function clipPos(pos) {
      if (pos.line < 0) return {line: 0, ch: 0};
      if (pos.line >= doc.size) return {line: doc.size-1, ch: getLine(doc.size-1).text.length};
      var ch = pos.ch, linelen = getLine(pos.line).text.length;
      if (ch == null || ch > linelen) return {line: pos.line, ch: linelen};
      else if (ch < 0) return {line: pos.line, ch: 0};
      else return pos;
    }

    function findPosH(dir, unit) {
      var end = sel.inverted ? sel.from : sel.to, line = end.line, ch = end.ch;
      var lineObj = getLine(line);
      function findNextLine() {
        for (var l = line + dir, e = dir < 0 ? -1 : doc.size; l != e; l += dir) {
          var lo = getLine(l);
          if (!lo.hidden) { line = l; lineObj = lo; return true; }
        }
      }
      function moveOnce(boundToLine) {
        if (ch == (dir < 0 ? 0 : lineObj.text.length)) {
          if (!boundToLine && findNextLine()) ch = dir < 0 ? lineObj.text.length : 0;
          else return false;
        } else ch += dir;
        return true;
      }
      if (unit == "char") moveOnce();
      else if (unit == "column") moveOnce(true);
      else if (unit == "word") {
        var sawWord = false;
        for (;;) {
          if (dir < 0) if (!moveOnce()) break;
          if (isWordChar(lineObj.text.charAt(ch))) sawWord = true;
          else if (sawWord) {if (dir < 0) {dir = 1; moveOnce();} break;}
          if (dir > 0) if (!moveOnce()) break;
        }
      }
      return {line: line, ch: ch};
    }
    function moveH(dir, unit) {
      var pos = dir < 0 ? sel.from : sel.to;
      if (shiftSelecting || posEq(sel.from, sel.to)) pos = findPosH(dir, unit);
      setCursor(pos.line, pos.ch, true);
    }
    function deleteH(dir, unit) {
      if (!posEq(sel.from, sel.to)) replaceRange("", sel.from, sel.to);
      else if (dir < 0) replaceRange("", findPosH(dir, unit), sel.to);
      else replaceRange("", sel.from, findPosH(dir, unit));
      userSelChange = true;
    }
    var goalColumn = null;
    function moveV(dir, unit) {
      var dist = 0, pos = localCoords(sel.inverted ? sel.from : sel.to, true);
      if (goalColumn != null) pos.x = goalColumn;
      if (unit == "page") dist = Math.min(scroller.clientHeight, window.innerHeight || document.documentElement.clientHeight);
      else if (unit == "line") dist = textHeight();
      var target = coordsChar(pos.x, pos.y + dist * dir + 2);
      if (unit == "page") scrollbar.scrollTop += localCoords(target, true).y - pos.y;
      setCursor(target.line, target.ch, true);
      goalColumn = pos.x;
    }

    function selectWordAt(pos) {
      var line = getLine(pos.line).text;
      var start = pos.ch, end = pos.ch;
      while (start > 0 && isWordChar(line.charAt(start - 1))) --start;
      while (end < line.length && isWordChar(line.charAt(end))) ++end;
      setSelectionUser({line: pos.line, ch: start}, {line: pos.line, ch: end});
    }
    function selectLine(line) {
      setSelectionUser({line: line, ch: 0}, clipPos({line: line + 1, ch: 0}));
    }
    function indentSelected(mode) {
      if (posEq(sel.from, sel.to)) return indentLine(sel.from.line, mode);
      var e = sel.to.line - (sel.to.ch ? 0 : 1);
      for (var i = sel.from.line; i <= e; ++i) indentLine(i, mode);
    }

    function indentLine(n, how) {
      if (!how) how = "add";
      if (how == "smart") {
        if (!mode.indent) how = "prev";
        else var state = getStateBefore(n);
      }

      var line = getLine(n), curSpace = line.indentation(options.tabSize),
          curSpaceString = line.text.match(/^\s*/)[0], indentation;
      if (how == "smart") {
        indentation = mode.indent(state, line.text.slice(curSpaceString.length), line.text);
        if (indentation == Pass) how = "prev";
      }
      if (how == "prev") {
        if (n) indentation = getLine(n-1).indentation(options.tabSize);
        else indentation = 0;
      }
      else if (how == "add") indentation = curSpace + options.indentUnit;
      else if (how == "subtract") indentation = curSpace - options.indentUnit;
      indentation = Math.max(0, indentation);
      var diff = indentation - curSpace;

      if (!diff) {
        if (sel.from.line != n && sel.to.line != n) return;
        var indentString = curSpaceString;
      } else {
        var indentString = "", pos = 0;
        if (options.indentWithTabs)
          for (var i = Math.floor(indentation / options.tabSize); i; --i) {pos += options.tabSize; indentString += "\t";}
        while (pos < indentation) {++pos; indentString += " ";}
      }

      replaceRange(indentString, {line: n, ch: 0}, {line: n, ch: curSpaceString.length});
    }

    function loadMode() {
      mode = CodeMirror.getMode(options, options.mode);
      doc.iter(0, doc.size, function(line) { line.stateAfter = null; });
      work = [0];
      startWorker();
    }
    function gutterChanged() {
      var visible = options.gutter || options.lineNumbers;
      gutter.style.display = visible ? "" : "none";
      if (visible) gutterDirty = true;
      else lineDiv.parentNode.style.marginLeft = 0;
    }
    function wrappingChanged(from, to) {
      if (options.lineWrapping) {
        wrapper.className += " CodeMirror-wrap";
        var perLine = scroller.clientWidth / charWidth() - 3;
        doc.iter(0, doc.size, function(line) {
          if (line.hidden) return;
          var guess = Math.ceil(line.text.length / perLine) || 1;
          if (guess != 1) updateLineHeight(line, guess);
        });
        lineSpace.style.width = code.style.width = "";
        widthForcer.style.left = "";
      } else {
        wrapper.className = wrapper.className.replace(" CodeMirror-wrap", "");
        maxLine = ""; maxLineChanged = true;
        doc.iter(0, doc.size, function(line) {
          if (line.height != 1 && !line.hidden) updateLineHeight(line, 1);
          if (line.text.length > maxLine.length) maxLine = line.text;
        });
      }
      changes.push({from: 0, to: doc.size});
    }
    function makeTab(col) {
      var w = options.tabSize - col % options.tabSize, cached = tabCache[w];
      if (cached) return cached;
      for (var str = '<span class="cm-tab">', i = 0; i < w; ++i) str += " ";
      return (tabCache[w] = {html: str + "</span>", width: w});
    }
    function themeChanged() {
      scroller.className = scroller.className.replace(/\s*cm-s-\S+/g, "") +
        options.theme.replace(/(^|\s)\s*/g, " cm-s-");
    }
    function keyMapChanged() {
      var style = keyMap[options.keyMap].style;
      wrapper.className = wrapper.className.replace(/\s*cm-keymap-\S+/g, "") +
        (style ? " cm-keymap-" + style : "");
    }

    function TextMarker() { this.set = []; }
    TextMarker.prototype.clear = operation(function() {
      var min = Infinity, max = -Infinity;
      for (var i = 0, e = this.set.length; i < e; ++i) {
        var line = this.set[i], mk = line.marked;
        if (!mk || !line.parent) continue;
        var lineN = lineNo(line);
        min = Math.min(min, lineN); max = Math.max(max, lineN);
        for (var j = 0; j < mk.length; ++j)
          if (mk[j].marker == this) mk.splice(j--, 1);
      }
      if (min != Infinity)
        changes.push({from: min, to: max + 1});
    });
    TextMarker.prototype.find = function() {
      var from, to;
      for (var i = 0, e = this.set.length; i < e; ++i) {
        var line = this.set[i], mk = line.marked;
        for (var j = 0; j < mk.length; ++j) {
          var mark = mk[j];
          if (mark.marker == this) {
            if (mark.from != null || mark.to != null) {
              var found = lineNo(line);
              if (found != null) {
                if (mark.from != null) from = {line: found, ch: mark.from};
                if (mark.to != null) to = {line: found, ch: mark.to};
              }
            }
          }
        }
      }
      return {from: from, to: to};
    };

    function markText(from, to, className) {
      from = clipPos(from); to = clipPos(to);
      var tm = new TextMarker();
      if (!posLess(from, to)) return tm;
      function add(line, from, to, className) {
        getLine(line).addMark(new MarkedText(from, to, className, tm));
      }
      if (from.line == to.line) add(from.line, from.ch, to.ch, className);
      else {
        add(from.line, from.ch, null, className);
        for (var i = from.line + 1, e = to.line; i < e; ++i)
          add(i, null, null, className);
        add(to.line, null, to.ch, className);
      }
      changes.push({from: from.line, to: to.line + 1});
      return tm;
    }

    function setBookmark(pos) {
      pos = clipPos(pos);
      var bm = new Bookmark(pos.ch);
      getLine(pos.line).addMark(bm);
      return bm;
    }

    function findMarksAt(pos) {
      pos = clipPos(pos);
      var markers = [], marked = getLine(pos.line).marked;
      if (!marked) return markers;
      for (var i = 0, e = marked.length; i < e; ++i) {
        var m = marked[i];
        if ((m.from == null || m.from <= pos.ch) &&
            (m.to == null || m.to >= pos.ch))
          markers.push(m.marker || m);
      }
      return markers;
    }

    function addGutterMarker(line, text, className) {
      if (typeof line == "number") line = getLine(clipLine(line));
      line.gutterMarker = {text: text, style: className};
      gutterDirty = true;
      return line;
    }
    function removeGutterMarker(line) {
      if (typeof line == "number") line = getLine(clipLine(line));
      line.gutterMarker = null;
      gutterDirty = true;
    }

    function changeLine(handle, op) {
      var no = handle, line = handle;
      if (typeof handle == "number") line = getLine(clipLine(handle));
      else no = lineNo(handle);
      if (no == null) return null;
      if (op(line, no)) changes.push({from: no, to: no + 1});
      else return null;
      return line;
    }
    function setLineClass(handle, className, bgClassName) {
      return changeLine(handle, function(line) {
        if (line.className != className || line.bgClassName != bgClassName) {
          line.className = className;
          line.bgClassName = bgClassName;
          return true;
        }
      });
    }
    function setLineHidden(handle, hidden) {
      return changeLine(handle, function(line, no) {
        if (line.hidden != hidden) {
          line.hidden = hidden;
          if (!options.lineWrapping) {
            var l = line.text;
            if (hidden && l.length == maxLine.length) {
              updateMaxLine = true;
            } else if (!hidden && l.length > maxLine.length) {
              maxLine = l; maxWidth = null; updateMaxLine = false;
            }
          }
          updateLineHeight(line, hidden ? 0 : 1);
          var fline = sel.from.line, tline = sel.to.line;
          if (hidden && (fline == no || tline == no)) {
            var from = fline == no ? skipHidden({line: fline, ch: 0}, fline, 0) : sel.from;
            var to = tline == no ? skipHidden({line: tline, ch: 0}, tline, 0) : sel.to;
            // Can't hide the last visible line, we'd have no place to put the cursor
            if (!to) return;
            setSelection(from, to);
          }
          return (gutterDirty = true);
        }
      });
    }

    function lineInfo(line) {
      if (typeof line == "number") {
        if (!isLine(line)) return null;
        var n = line;
        line = getLine(line);
        if (!line) return null;
      } else {
        var n = lineNo(line);
        if (n == null) return null;
      }
      var marker = line.gutterMarker;
      return {line: n, handle: line, text: line.text, markerText: marker && marker.text,
              markerClass: marker && marker.style, lineClass: line.className, bgClass: line.bgClassName};
    }

    function stringWidth(str) {
      measure.innerHTML = "<pre><span>x</span></pre>";
      measure.firstChild.firstChild.firstChild.nodeValue = str;
      return measure.firstChild.firstChild.offsetWidth || 10;
    }
    // These are used to go from pixel positions to character
    // positions, taking varying character widths into account.
    function charFromX(line, x) {
      if (x <= 0) return 0;
      var lineObj = getLine(line), text = lineObj.text;
      function getX(len) {
        return measureLine(lineObj, len).left;
      }
      var from = 0, fromX = 0, to = text.length, toX;
      // Guess a suitable upper bound for our search.
      var estimated = Math.min(to, Math.ceil(x / charWidth()));
      for (;;) {
        var estX = getX(estimated);
        if (estX <= x && estimated < to) estimated = Math.min(to, Math.ceil(estimated * 1.2));
        else {toX = estX; to = estimated; break;}
      }
      if (x > toX) return to;
      // Try to guess a suitable lower bound as well.
      estimated = Math.floor(to * 0.8); estX = getX(estimated);
      if (estX < x) {from = estimated; fromX = estX;}
      // Do a binary search between these bounds.
      for (;;) {
        if (to - from <= 1) return (toX - x > x - fromX) ? from : to;
        var middle = Math.ceil((from + to) / 2), middleX = getX(middle);
        if (middleX > x) {to = middle; toX = middleX;}
        else {from = middle; fromX = middleX;}
      }
    }

    var tempId = "CodeMirror-temp-" + Math.floor(Math.random() * 0xffffff).toString(16);
    function measureLine(line, ch) {
      if (ch == 0) return {top: 0, left: 0};
      var wbr = options.lineWrapping && ch < line.text.length &&
                spanAffectsWrapping.test(line.text.slice(ch - 1, ch + 1));
      measure.innerHTML = "<pre>" + line.getHTML(makeTab, ch, tempId, wbr) + "</pre>";
      var elt = document.getElementById(tempId);
      var top = elt.offsetTop, left = elt.offsetLeft;
      // Older IEs report zero offsets for spans directly after a wrap
      if (ie && top == 0 && left == 0) {
        var backup = document.createElement("span");
        backup.innerHTML = "x";
        elt.parentNode.insertBefore(backup, elt.nextSibling);
        top = backup.offsetTop;
      }
      return {top: top, left: left};
    }
    function localCoords(pos, inLineWrap) {
      var x, lh = textHeight(), y = lh * (heightAtLine(doc, pos.line) - (inLineWrap ? displayOffset : 0));
      if (pos.ch == 0) x = 0;
      else {
        var sp = measureLine(getLine(pos.line), pos.ch);
        x = sp.left;
        if (options.lineWrapping) y += Math.max(0, sp.top);
      }
      return {x: x, y: y, yBot: y + lh};
    }
    // Coords must be lineSpace-local
    function coordsChar(x, y) {
      if (y < 0) y = 0;
      var th = textHeight(), cw = charWidth(), heightPos = displayOffset + Math.floor(y / th);
      var lineNo = lineAtHeight(doc, heightPos);
      if (lineNo >= doc.size) return {line: doc.size - 1, ch: getLine(doc.size - 1).text.length};
      var lineObj = getLine(lineNo), text = lineObj.text;
      var tw = options.lineWrapping, innerOff = tw ? heightPos - heightAtLine(doc, lineNo) : 0;
      if (x <= 0 && innerOff == 0) return {line: lineNo, ch: 0};
      function getX(len) {
        var sp = measureLine(lineObj, len);
        if (tw) {
          var off = Math.round(sp.top / th);
          return Math.max(0, sp.left + (off - innerOff) * scroller.clientWidth);
        }
        return sp.left;
      }
      var from = 0, fromX = 0, to = text.length, toX;
      // Guess a suitable upper bound for our search.
      var estimated = Math.min(to, Math.ceil((x + innerOff * scroller.clientWidth * .9) / cw));
      for (;;) {
        var estX = getX(estimated);
        if (estX <= x && estimated < to) estimated = Math.min(to, Math.ceil(estimated * 1.2));
        else {toX = estX; to = estimated; break;}
      }
      if (x > toX) return {line: lineNo, ch: to};
      // Try to guess a suitable lower bound as well.
      estimated = Math.floor(to * 0.8); estX = getX(estimated);
      if (estX < x) {from = estimated; fromX = estX;}
      // Do a binary search between these bounds.
      for (;;) {
        if (to - from <= 1) return {line: lineNo, ch: (toX - x > x - fromX) ? from : to};
        var middle = Math.ceil((from + to) / 2), middleX = getX(middle);
        if (middleX > x) {to = middle; toX = middleX;}
        else {from = middle; fromX = middleX;}
      }
    }
    function pageCoords(pos) {
      var local = localCoords(pos, true), off = eltOffset(lineSpace);
      return {x: off.left + local.x, y: off.top + local.y, yBot: off.top + local.yBot};
    }

    var cachedHeight, cachedHeightFor, measureText;
    function textHeight() {
      if (measureText == null) {
        measureText = "<pre>";
        for (var i = 0; i < 49; ++i) measureText += "x<br/>";
        measureText += "x</pre>";
      }
      var offsetHeight = lineDiv.clientHeight;
      if (offsetHeight == cachedHeightFor) return cachedHeight;
      cachedHeightFor = offsetHeight;
      measure.innerHTML = measureText;
      cachedHeight = measure.firstChild.offsetHeight / 50 || 1;
      measure.innerHTML = "";
      return cachedHeight;
    }
    var cachedWidth, cachedWidthFor = 0;
    function charWidth() {
      if (scroller.clientWidth == cachedWidthFor) return cachedWidth;
      cachedWidthFor = scroller.clientWidth;
      return (cachedWidth = stringWidth("x"));
    }
    function paddingTop() {return lineSpace.offsetTop;}
    function paddingLeft() {return lineSpace.offsetLeft;}

    function posFromMouse(e, liberal) {
      var offW = eltOffset(scroller, true), x, y;
      // Fails unpredictably on IE[67] when mouse is dragged around quickly.
      try { x = e.clientX; y = e.clientY; } catch (e) { return null; }
      // This is a mess of a heuristic to try and determine whether a
      // scroll-bar was clicked or not, and to return null if one was
      // (and !liberal).
      if (!liberal && (x - offW.left > scroller.clientWidth || y - offW.top > scroller.clientHeight))
        return null;
      var offL = eltOffset(lineSpace, true);
      return coordsChar(x - offL.left, y - offL.top);
    }
    function onContextMenu(e) {
      var pos = posFromMouse(e), scrollPos = scrollbar.scrollTop;
      if (!pos || opera) return; // Opera is difficult.
      if (posEq(sel.from, sel.to) || posLess(pos, sel.from) || !posLess(pos, sel.to))
        operation(setCursor)(pos.line, pos.ch);

      var oldCSS = input.style.cssText;
      inputDiv.style.position = "absolute";
      input.style.cssText = "position: fixed; width: 30px; height: 30px; top: " + (e.clientY - 5) +
        "px; left: " + (e.clientX - 5) + "px; z-index: 1000; background: white; " +
        "border-width: 0; outline: none; overflow: hidden; opacity: .05; filter: alpha(opacity=5);";
      leaveInputAlone = true;
      var val = input.value = getSelection();
      focusInput();
      selectInput(input);
      function rehide() {
        var newVal = splitLines(input.value).join("\n");
        if (newVal != val && !options.readOnly) operation(replaceSelection)(newVal, "end");
        inputDiv.style.position = "relative";
        input.style.cssText = oldCSS;
        if (ie_lt9) scrollbar.scrollTop = scrollPos;
        leaveInputAlone = false;
        resetInput(true);
        slowPoll();
      }

      if (gecko) {
        e_stop(e);
        var mouseup = connect(window, "mouseup", function() {
          mouseup();
          setTimeout(rehide, 20);
        }, true);
      } else {
        setTimeout(rehide, 50);
      }
    }

    // Cursor-blinking
    function restartBlink() {
      clearInterval(blinker);
      var on = true;
      cursor.style.visibility = "";
      blinker = setInterval(function() {
        cursor.style.visibility = (on = !on) ? "" : "hidden";
      }, 650);
    }

    var matching = {"(": ")>", ")": "(<", "[": "]>", "]": "[<", "{": "}>", "}": "{<"};
    function matchBrackets(autoclear) {
      var head = sel.inverted ? sel.from : sel.to, line = getLine(head.line), pos = head.ch - 1;
      var match = (pos >= 0 && matching[line.text.charAt(pos)]) || matching[line.text.charAt(++pos)];
      if (!match) return;
      var ch = match.charAt(0), forward = match.charAt(1) == ">", d = forward ? 1 : -1, st = line.styles;
      for (var off = pos + 1, i = 0, e = st.length; i < e; i+=2)
        if ((off -= st[i].length) <= 0) {var style = st[i+1]; break;}

      var stack = [line.text.charAt(pos)], re = /[(){}[\]]/;
      function scan(line, from, to) {
        if (!line.text) return;
        var st = line.styles, pos = forward ? 0 : line.text.length - 1, cur;
        for (var i = forward ? 0 : st.length - 2, e = forward ? st.length : -2; i != e; i += 2*d) {
          var text = st[i];
          if (st[i+1] != style) {pos += d * text.length; continue;}
          for (var j = forward ? 0 : text.length - 1, te = forward ? text.length : -1; j != te; j += d, pos+=d) {
            if (pos >= from && pos < to && re.test(cur = text.charAt(j))) {
              var match = matching[cur];
              if (match.charAt(1) == ">" == forward) stack.push(cur);
              else if (stack.pop() != match.charAt(0)) return {pos: pos, match: false};
              else if (!stack.length) return {pos: pos, match: true};
            }
          }
        }
      }
      for (var i = head.line, e = forward ? Math.min(i + 100, doc.size) : Math.max(-1, i - 100); i != e; i+=d) {
        var line = getLine(i), first = i == head.line;
        var found = scan(line, first && forward ? pos + 1 : 0, first && !forward ? pos : line.text.length);
        if (found) break;
      }
      if (!found) found = {pos: null, match: false};
      var style = found.match ? "CodeMirror-matchingbracket" : "CodeMirror-nonmatchingbracket";
      var one = markText({line: head.line, ch: pos}, {line: head.line, ch: pos+1}, style),
          two = found.pos != null && markText({line: i, ch: found.pos}, {line: i, ch: found.pos + 1}, style);
      var clear = operation(function(){one.clear(); two && two.clear();});
      if (autoclear) setTimeout(clear, 800);
      else bracketHighlighted = clear;
    }

    // Finds the line to start with when starting a parse. Tries to
    // find a line with a stateAfter, so that it can start with a
    // valid state. If that fails, it returns the line with the
    // smallest indentation, which tends to need the least context to
    // parse correctly.
    function findStartLine(n) {
      var minindent, minline;
      for (var search = n, lim = n - 40; search > lim; --search) {
        if (search == 0) return 0;
        var line = getLine(search-1);
        if (line.stateAfter) return search;
        var indented = line.indentation(options.tabSize);
        if (minline == null || minindent > indented) {
          minline = search - 1;
          minindent = indented;
        }
      }
      return minline;
    }
    function getStateBefore(n) {
      var start = findStartLine(n), state = start && getLine(start-1).stateAfter;
      if (!state) state = startState(mode);
      else state = copyState(mode, state);
      doc.iter(start, n, function(line) {
        line.highlight(mode, state, options.tabSize);
        line.stateAfter = copyState(mode, state);
      });
      if (start < n) changes.push({from: start, to: n});
      if (n < doc.size && !getLine(n).stateAfter) work.push(n);
      return state;
    }
    function highlightLines(start, end) {
      var state = getStateBefore(start);
      doc.iter(start, end, function(line) {
        line.highlight(mode, state, options.tabSize);
        line.stateAfter = copyState(mode, state);
      });
    }
    function highlightWorker() {
      var end = +new Date + options.workTime;
      var foundWork = work.length;
      while (work.length) {
        if (!getLine(showingFrom).stateAfter) var task = showingFrom;
        else var task = work.pop();
        if (task >= doc.size) continue;
        var start = findStartLine(task), state = start && getLine(start-1).stateAfter;
        if (state) state = copyState(mode, state);
        else state = startState(mode);

        var unchanged = 0, compare = mode.compareStates, realChange = false,
            i = start, bail = false;
        doc.iter(i, doc.size, function(line) {
          var hadState = line.stateAfter;
          if (+new Date > end) {
            work.push(i);
            startWorker(options.workDelay);
            if (realChange) changes.push({from: task, to: i + 1});
            return (bail = true);
          }
          var changed = line.highlight(mode, state, options.tabSize);
          if (changed) realChange = true;
          line.stateAfter = copyState(mode, state);
          var done = null;
          if (compare) {
            var same = hadState && compare(hadState, state);
            if (same != Pass) done = !!same;
          }
          if (done == null) {
            if (changed !== false || !hadState) unchanged = 0;
            else if (++unchanged > 3 && (!mode.indent || mode.indent(hadState, "") == mode.indent(state, "")))
              done = true;
          }
          if (done) return true;
          ++i;
        });
        if (bail) return;
        if (realChange) changes.push({from: task, to: i + 1});
      }
      if (foundWork && options.onHighlightComplete)
        options.onHighlightComplete(instance);
    }
    function startWorker(time) {
      if (!work.length) return;
      highlight.set(time, operation(highlightWorker));
    }

    // Operations are used to wrap changes in such a way that each
    // change won't have to update the cursor and display (which would
    // be awkward, slow, and error-prone), but instead updates are
    // batched and then all combined and executed at once.
    function startOperation() {
      updateInput = userSelChange = textChanged = null;
      changes = []; selectionChanged = false; callbacks = [];
    }
    function endOperation() {
      if (updateMaxLine) computeMaxLength();
      if (maxLineChanged && !options.lineWrapping) {
        widthForcer.style.left = stringWidth(maxLine) + "px";
        maxLineChanged = false;
      }
      var newScrollPos, updated;
      if (selectionChanged) {
        var coords = calculateCursorCoords();
        newScrollPos = calculateScrollPos(coords.x, coords.y, coords.x, coords.yBot);
      }
      if (changes.length) updated = updateDisplay(changes, true, (newScrollPos ? newScrollPos.scrollTop : null));
      else {
        if (selectionChanged) updateSelection();
        if (gutterDirty) updateGutter();
      }
      if (newScrollPos) scrollCursorIntoView();
      if (selectionChanged) {scrollEditorIntoView(); restartBlink();}

      if (focused && !leaveInputAlone &&
          (updateInput === true || (updateInput !== false && selectionChanged)))
        resetInput(userSelChange);

      if (selectionChanged && options.matchBrackets)
        setTimeout(operation(function() {
          if (bracketHighlighted) {bracketHighlighted(); bracketHighlighted = null;}
          if (posEq(sel.from, sel.to)) matchBrackets(false);
        }), 20);
      var sc = selectionChanged, cbs = callbacks; // these can be reset by callbacks
      if (textChanged && options.onChange && instance)
        options.onChange(instance, textChanged);
      if (sc && options.onCursorActivity)
        options.onCursorActivity(instance);
      for (var i = 0; i < cbs.length; ++i) cbs[i](instance);
      if (updated && options.onUpdate) options.onUpdate(instance);
    }
    var nestedOperation = 0;
    function operation(f) {
      return function() {
        if (!nestedOperation++) startOperation();
        try {var result = f.apply(this, arguments);}
        finally {if (!--nestedOperation) endOperation();}
        return result;
      };
    }

    function compoundChange(f) {
      history.startCompound();
      try { return f(); } finally { history.endCompound(); }
    }

    for (var ext in extensions)
      if (extensions.propertyIsEnumerable(ext) &&
          !instance.propertyIsEnumerable(ext))
        instance[ext] = extensions[ext];
    return instance;
  } // (end of function CodeMirror)

  // The default configuration options.
  CodeMirror.defaults = {
    value: "",
    mode: null,
    theme: "default",
    indentUnit: 2,
    indentWithTabs: false,
    smartIndent: true,
    tabSize: 4,
    keyMap: "default",
    extraKeys: null,
    electricChars: true,
    autoClearEmptyLines: false,
    onKeyEvent: null,
    onDragEvent: null,
    lineWrapping: false,
    lineNumbers: false,
    gutter: false,
    fixedGutter: false,
    firstLineNumber: 1,
    readOnly: false,
    dragDrop: true,
    onChange: null,
    onCursorActivity: null,
    onGutterClick: null,
    onHighlightComplete: null,
    onUpdate: null,
    onFocus: null, onBlur: null, onScroll: null,
    matchBrackets: false,
    workTime: 100,
    workDelay: 200,
    pollInterval: 100,
    undoDepth: 40,
    tabindex: null,
    autofocus: null
  };

  var ios = /AppleWebKit/.test(navigator.userAgent) && /Mobile\/\w+/.test(navigator.userAgent);
  var mac = ios || /Mac/.test(navigator.platform);
  var win = /Win/.test(navigator.platform);

  // Known modes, by name and by MIME
  var modes = CodeMirror.modes = {}, mimeModes = CodeMirror.mimeModes = {};
  CodeMirror.defineMode = function(name, mode) {
    if (!CodeMirror.defaults.mode && name != "null") CodeMirror.defaults.mode = name;
    if (arguments.length > 2) {
      mode.dependencies = [];
      for (var i = 2; i < arguments.length; ++i) mode.dependencies.push(arguments[i]);
    }
    modes[name] = mode;
  };
  CodeMirror.defineMIME = function(mime, spec) {
    mimeModes[mime] = spec;
  };
  CodeMirror.resolveMode = function(spec) {
    if (typeof spec == "string" && mimeModes.hasOwnProperty(spec))
      spec = mimeModes[spec];
    else if (typeof spec == "string" && /^[\w\-]+\/[\w\-]+\+xml$/.test(spec))
      return CodeMirror.resolveMode("application/xml");
    if (typeof spec == "string") return {name: spec};
    else return spec || {name: "null"};
  };
  CodeMirror.getMode = function(options, spec) {
    var spec = CodeMirror.resolveMode(spec);
    var mfactory = modes[spec.name];
    if (!mfactory) return CodeMirror.getMode(options, "text/plain");
    return mfactory(options, spec);
  };
  CodeMirror.listModes = function() {
    var list = [];
    for (var m in modes)
      if (modes.propertyIsEnumerable(m)) list.push(m);
    return list;
  };
  CodeMirror.listMIMEs = function() {
    var list = [];
    for (var m in mimeModes)
      if (mimeModes.propertyIsEnumerable(m)) list.push({mime: m, mode: mimeModes[m]});
    return list;
  };

  var extensions = CodeMirror.extensions = {};
  CodeMirror.defineExtension = function(name, func) {
    extensions[name] = func;
  };

  var commands = CodeMirror.commands = {
    selectAll: function(cm) {cm.setSelection({line: 0, ch: 0}, {line: cm.lineCount() - 1});},
    killLine: function(cm) {
      var from = cm.getCursor(true), to = cm.getCursor(false), sel = !posEq(from, to);
      if (!sel && cm.getLine(from.line).length == from.ch) cm.replaceRange("", from, {line: from.line + 1, ch: 0});
      else cm.replaceRange("", from, sel ? to : {line: from.line});
    },
    deleteLine: function(cm) {var l = cm.getCursor().line; cm.replaceRange("", {line: l, ch: 0}, {line: l});},
    undo: function(cm) {cm.undo();},
    redo: function(cm) {cm.redo();},
    goDocStart: function(cm) {cm.setCursor(0, 0, true);},
    goDocEnd: function(cm) {cm.setSelection({line: cm.lineCount() - 1}, null, true);},
    goLineStart: function(cm) {cm.setCursor(cm.getCursor().line, 0, true);},
    goLineStartSmart: function(cm) {
      var cur = cm.getCursor();
      var text = cm.getLine(cur.line), firstNonWS = Math.max(0, text.search(/\S/));
      cm.setCursor(cur.line, cur.ch <= firstNonWS && cur.ch ? 0 : firstNonWS, true);
    },
    goLineEnd: function(cm) {cm.setSelection({line: cm.getCursor().line}, null, true);},
    goLineUp: function(cm) {cm.moveV(-1, "line");},
    goLineDown: function(cm) {cm.moveV(1, "line");},
    goPageUp: function(cm) {cm.moveV(-1, "page");},
    goPageDown: function(cm) {cm.moveV(1, "page");},
    goCharLeft: function(cm) {cm.moveH(-1, "char");},
    goCharRight: function(cm) {cm.moveH(1, "char");},
    goColumnLeft: function(cm) {cm.moveH(-1, "column");},
    goColumnRight: function(cm) {cm.moveH(1, "column");},
    goWordLeft: function(cm) {cm.moveH(-1, "word");},
    goWordRight: function(cm) {cm.moveH(1, "word");},
    delCharLeft: function(cm) {cm.deleteH(-1, "char");},
    delCharRight: function(cm) {cm.deleteH(1, "char");},
    delWordLeft: function(cm) {cm.deleteH(-1, "word");},
    delWordRight: function(cm) {cm.deleteH(1, "word");},
    indentAuto: function(cm) {cm.indentSelection("smart");},
    indentMore: function(cm) {cm.indentSelection("add");},
    indentLess: function(cm) {cm.indentSelection("subtract");},
    insertTab: function(cm) {cm.replaceSelection("\t", "end");},
    defaultTab: function(cm) {
      if (cm.somethingSelected()) cm.indentSelection("add");
      else cm.replaceSelection("\t", "end");
    },
    transposeChars: function(cm) {
      var cur = cm.getCursor(), line = cm.getLine(cur.line);
      if (cur.ch > 0 && cur.ch < line.length - 1)
        cm.replaceRange(line.charAt(cur.ch) + line.charAt(cur.ch - 1),
                        {line: cur.line, ch: cur.ch - 1}, {line: cur.line, ch: cur.ch + 1});
    },
    newlineAndIndent: function(cm) {
      cm.replaceSelection("\n", "end");
      cm.indentLine(cm.getCursor().line);
    },
    toggleOverwrite: function(cm) {cm.toggleOverwrite();}
  };

  var keyMap = CodeMirror.keyMap = {};
  keyMap.basic = {
    "Left": "goCharLeft", "Right": "goCharRight", "Up": "goLineUp", "Down": "goLineDown",
    "End": "goLineEnd", "Home": "goLineStartSmart", "PageUp": "goPageUp", "PageDown": "goPageDown",
    "Delete": "delCharRight", "Backspace": "delCharLeft", "Tab": "defaultTab", "Shift-Tab": "indentAuto",
    "Enter": "newlineAndIndent", "Insert": "toggleOverwrite"
  };
  // Note that the save and find-related commands aren't defined by
  // default. Unknown commands are simply ignored.
  keyMap.pcDefault = {
    "Ctrl-A": "selectAll", "Ctrl-D": "deleteLine", "Ctrl-Z": "undo", "Shift-Ctrl-Z": "redo", "Ctrl-Y": "redo",
    "Ctrl-Home": "goDocStart", "Alt-Up": "goDocStart", "Ctrl-End": "goDocEnd", "Ctrl-Down": "goDocEnd",
    "Ctrl-Left": "goWordLeft", "Ctrl-Right": "goWordRight", "Alt-Left": "goLineStart", "Alt-Right": "goLineEnd",
    "Ctrl-Backspace": "delWordLeft", "Ctrl-Delete": "delWordRight", "Ctrl-S": "save", "Ctrl-F": "find",
    "Ctrl-G": "findNext", "Shift-Ctrl-G": "findPrev", "Shift-Ctrl-F": "replace", "Shift-Ctrl-R": "replaceAll",
    "Ctrl-[": "indentLess", "Ctrl-]": "indentMore",
    fallthrough: "basic"
  };
  keyMap.macDefault = {
    "Cmd-A": "selectAll", "Cmd-D": "deleteLine", "Cmd-Z": "undo", "Shift-Cmd-Z": "redo", "Cmd-Y": "redo",
    "Cmd-Up": "goDocStart", "Cmd-End": "goDocEnd", "Cmd-Down": "goDocEnd", "Alt-Left": "goWordLeft",
    "Alt-Right": "goWordRight", "Cmd-Left": "goLineStart", "Cmd-Right": "goLineEnd", "Alt-Backspace": "delWordLeft",
    "Ctrl-Alt-Backspace": "delWordRight", "Alt-Delete": "delWordRight", "Cmd-S": "save", "Cmd-F": "find",
    "Cmd-G": "findNext", "Shift-Cmd-G": "findPrev", "Cmd-Alt-F": "replace", "Shift-Cmd-Alt-F": "replaceAll",
    "Cmd-[": "indentLess", "Cmd-]": "indentMore",
    fallthrough: ["basic", "emacsy"]
  };
  keyMap["default"] = mac ? keyMap.macDefault : keyMap.pcDefault;
  keyMap.emacsy = {
    "Ctrl-F": "goCharRight", "Ctrl-B": "goCharLeft", "Ctrl-P": "goLineUp", "Ctrl-N": "goLineDown",
    "Alt-F": "goWordRight", "Alt-B": "goWordLeft", "Ctrl-A": "goLineStart", "Ctrl-E": "goLineEnd",
    "Ctrl-V": "goPageUp", "Shift-Ctrl-V": "goPageDown", "Ctrl-D": "delCharRight", "Ctrl-H": "delCharLeft",
    "Alt-D": "delWordRight", "Alt-Backspace": "delWordLeft", "Ctrl-K": "killLine", "Ctrl-T": "transposeChars"
  };

  function getKeyMap(val) {
    if (typeof val == "string") return keyMap[val];
    else return val;
  }
  function lookupKey(name, extraMap, map, handle, stop) {
    function lookup(map) {
      map = getKeyMap(map);
      var found = map[name];
      if (found != null && handle(found)) return true;
      if (map.nofallthrough) {
        if (stop) stop();
        return true;
      }
      var fallthrough = map.fallthrough;
      if (fallthrough == null) return false;
      if (Object.prototype.toString.call(fallthrough) != "[object Array]")
        return lookup(fallthrough);
      for (var i = 0, e = fallthrough.length; i < e; ++i) {
        if (lookup(fallthrough[i])) return true;
      }
      return false;
    }
    if (extraMap && lookup(extraMap)) return true;
    return lookup(map);
  }
  function isModifierKey(event) {
    var name = keyNames[e_prop(event, "keyCode")];
    return name == "Ctrl" || name == "Alt" || name == "Shift" || name == "Mod";
  }

  CodeMirror.fromTextArea = function(textarea, options) {
    if (!options) options = {};
    options.value = textarea.value;
    if (!options.tabindex && textarea.tabindex)
      options.tabindex = textarea.tabindex;
    if (options.autofocus == null && textarea.getAttribute("autofocus") != null)
      options.autofocus = true;

    function save() {textarea.value = instance.getValue();}
    if (textarea.form) {
      // Deplorable hack to make the submit method do the right thing.
      var rmSubmit = connect(textarea.form, "submit", save, true);
      if (typeof textarea.form.submit == "function") {
        var realSubmit = textarea.form.submit;
        function wrappedSubmit() {
          save();
          textarea.form.submit = realSubmit;
          textarea.form.submit();
          textarea.form.submit = wrappedSubmit;
        }
        textarea.form.submit = wrappedSubmit;
      }
    }

    textarea.style.display = "none";
    var instance = CodeMirror(function(node) {
      textarea.parentNode.insertBefore(node, textarea.nextSibling);
    }, options);
    instance.save = save;
    instance.getTextArea = function() { return textarea; };
    instance.toTextArea = function() {
      save();
      textarea.parentNode.removeChild(instance.getWrapperElement());
      textarea.style.display = "";
      if (textarea.form) {
        rmSubmit();
        if (typeof textarea.form.submit == "function")
          textarea.form.submit = realSubmit;
      }
    };
    return instance;
  };

  // Utility functions for working with state. Exported because modes
  // sometimes need to do this.
  function copyState(mode, state) {
    if (state === true) return state;
    if (mode.copyState) return mode.copyState(state);
    var nstate = {};
    for (var n in state) {
      var val = state[n];
      if (val instanceof Array) val = val.concat([]);
      nstate[n] = val;
    }
    return nstate;
  }
  CodeMirror.copyState = copyState;
  function startState(mode, a1, a2) {
    return mode.startState ? mode.startState(a1, a2) : true;
  }
  CodeMirror.startState = startState;

  // The character stream used by a mode's parser.
  function StringStream(string, tabSize) {
    this.pos = this.start = 0;
    this.string = string;
    this.tabSize = tabSize || 8;
  }
  StringStream.prototype = {
    eol: function() {return this.pos >= this.string.length;},
    sol: function() {return this.pos == 0;},
    peek: function() {return this.string.charAt(this.pos);},
    next: function() {
      if (this.pos < this.string.length)
        return this.string.charAt(this.pos++);
    },
    eat: function(match) {
      var ch = this.string.charAt(this.pos);
      if (typeof match == "string") var ok = ch == match;
      else var ok = ch && (match.test ? match.test(ch) : match(ch));
      if (ok) {++this.pos; return ch;}
    },
    eatWhile: function(match) {
      var start = this.pos;
      while (this.eat(match)){}
      return this.pos > start;
    },
    eatSpace: function() {
      var start = this.pos;
      while (/[\s\u00a0]/.test(this.string.charAt(this.pos))) ++this.pos;
      return this.pos > start;
    },
    skipToEnd: function() {this.pos = this.string.length;},
    skipTo: function(ch) {
      var found = this.string.indexOf(ch, this.pos);
      if (found > -1) {this.pos = found; return true;}
    },
    backUp: function(n) {this.pos -= n;},
    column: function() {return countColumn(this.string, this.start, this.tabSize);},
    indentation: function() {return countColumn(this.string, null, this.tabSize);},
    match: function(pattern, consume, caseInsensitive) {
      if (typeof pattern == "string") {
        function cased(str) {return caseInsensitive ? str.toLowerCase() : str;}
        if (cased(this.string).indexOf(cased(pattern), this.pos) == this.pos) {
          if (consume !== false) this.pos += pattern.length;
          return true;
        }
      } else {
        var match = this.string.slice(this.pos).match(pattern);
        if (match && consume !== false) this.pos += match[0].length;
        return match;
      }
    },
    current: function(){return this.string.slice(this.start, this.pos);}
  };
  CodeMirror.StringStream = StringStream;

  function MarkedText(from, to, className, marker) {
    this.from = from; this.to = to; this.style = className; this.marker = marker;
  }
  MarkedText.prototype = {
    attach: function(line) { this.marker.set.push(line); },
    detach: function(line) {
      var ix = indexOf(this.marker.set, line);
      if (ix > -1) this.marker.set.splice(ix, 1);
    },
    split: function(pos, lenBefore) {
      if (this.to <= pos && this.to != null) return null;
      var from = this.from < pos || this.from == null ? null : this.from - pos + lenBefore;
      var to = this.to == null ? null : this.to - pos + lenBefore;
      return new MarkedText(from, to, this.style, this.marker);
    },
    dup: function() { return new MarkedText(null, null, this.style, this.marker); },
    clipTo: function(fromOpen, from, toOpen, to, diff) {
      if (fromOpen && to > this.from && (to < this.to || this.to == null))
        this.from = null;
      else if (this.from != null && this.from >= from)
        this.from = Math.max(to, this.from) + diff;
      if (toOpen && (from < this.to || this.to == null) && (from > this.from || this.from == null))
        this.to = null;
      else if (this.to != null && this.to > from)
        this.to = to < this.to ? this.to + diff : from;
    },
    isDead: function() { return this.from != null && this.to != null && this.from >= this.to; },
    sameSet: function(x) { return this.marker == x.marker; }
  };

  function Bookmark(pos) {
    this.from = pos; this.to = pos; this.line = null;
  }
  Bookmark.prototype = {
    attach: function(line) { this.line = line; },
    detach: function(line) { if (this.line == line) this.line = null; },
    split: function(pos, lenBefore) {
      if (pos < this.from) {
        this.from = this.to = (this.from - pos) + lenBefore;
        return this;
      }
    },
    isDead: function() { return this.from > this.to; },
    clipTo: function(fromOpen, from, toOpen, to, diff) {
      if ((fromOpen || from < this.from) && (toOpen || to > this.to)) {
        this.from = 0; this.to = -1;
      } else if (this.from > from) {
        this.from = this.to = Math.max(to, this.from) + diff;
      }
    },
    sameSet: function(x) { return false; },
    find: function() {
      if (!this.line || !this.line.parent) return null;
      return {line: lineNo(this.line), ch: this.from};
    },
    clear: function() {
      if (this.line) {
        var found = indexOf(this.line.marked, this);
        if (found != -1) this.line.marked.splice(found, 1);
        this.line = null;
      }
    }
  };

  // Line objects. These hold state related to a line, including
  // highlighting info (the styles array).
  function Line(text, styles) {
    this.styles = styles || [text, null];
    this.text = text;
    this.height = 1;
    this.marked = this.gutterMarker = this.className = this.bgClassName = this.handlers = null;
    this.stateAfter = this.parent = this.hidden = null;
  }
  Line.inheritMarks = function(text, orig) {
    var ln = new Line(text), mk = orig && orig.marked;
    if (mk) {
      for (var i = 0; i < mk.length; ++i) {
        if (mk[i].to == null && mk[i].style) {
          var newmk = ln.marked || (ln.marked = []), mark = mk[i];
          var nmark = mark.dup(); newmk.push(nmark); nmark.attach(ln);
        }
      }
    }
    return ln;
  }
  Line.prototype = {
    // Replace a piece of a line, keeping the styles around it intact.
    replace: function(from, to_, text) {
      var st = [], mk = this.marked, to = to_ == null ? this.text.length : to_;
      copyStyles(0, from, this.styles, st);
      if (text) st.push(text, null);
      copyStyles(to, this.text.length, this.styles, st);
      this.styles = st;
      this.text = this.text.slice(0, from) + text + this.text.slice(to);
      this.stateAfter = null;
      if (mk) {
        var diff = text.length - (to - from);
        for (var i = 0; i < mk.length; ++i) {
          var mark = mk[i];
          mark.clipTo(from == null, from || 0, to_ == null, to, diff);
          if (mark.isDead()) {mark.detach(this); mk.splice(i--, 1);}
        }
      }
    },
    // Split a part off a line, keeping styles and markers intact.
    split: function(pos, textBefore) {
      var st = [textBefore, null], mk = this.marked;
      copyStyles(pos, this.text.length, this.styles, st);
      var taken = new Line(textBefore + this.text.slice(pos), st);
      if (mk) {
        for (var i = 0; i < mk.length; ++i) {
          var mark = mk[i];
          var newmark = mark.split(pos, textBefore.length);
          if (newmark) {
            if (!taken.marked) taken.marked = [];
            taken.marked.push(newmark); newmark.attach(taken);
            if (newmark == mark) mk.splice(i--, 1);
          }
        }
      }
      return taken;
    },
    append: function(line) {
      var mylen = this.text.length, mk = line.marked, mymk = this.marked;
      this.text += line.text;
      copyStyles(0, line.text.length, line.styles, this.styles);
      if (mymk) {
        for (var i = 0; i < mymk.length; ++i)
          if (mymk[i].to == null) mymk[i].to = mylen;
      }
      if (mk && mk.length) {
        if (!mymk) this.marked = mymk = [];
        outer: for (var i = 0; i < mk.length; ++i) {
          var mark = mk[i];
          if (!mark.from) {
            for (var j = 0; j < mymk.length; ++j) {
              var mymark = mymk[j];
              if (mymark.to == mylen && mymark.sameSet(mark)) {
                mymark.to = mark.to == null ? null : mark.to + mylen;
                if (mymark.isDead()) {
                  mymark.detach(this);
                  mk.splice(i--, 1);
                }
                continue outer;
              }
            }
          }
          mymk.push(mark);
          mark.attach(this);
          mark.from += mylen;
          if (mark.to != null) mark.to += mylen;
        }
      }
    },
    fixMarkEnds: function(other) {
      var mk = this.marked, omk = other.marked;
      if (!mk) return;
      for (var i = 0; i < mk.length; ++i) {
        var mark = mk[i], close = mark.to == null;
        if (close && omk) {
          for (var j = 0; j < omk.length; ++j)
            if (omk[j].sameSet(mark)) {close = false; break;}
        }
        if (close) mark.to = this.text.length;
      }
    },
    fixMarkStarts: function() {
      var mk = this.marked;
      if (!mk) return;
      for (var i = 0; i < mk.length; ++i)
        if (mk[i].from == null) mk[i].from = 0;
    },
    addMark: function(mark) {
      mark.attach(this);
      if (this.marked == null) this.marked = [];
      this.marked.push(mark);
      this.marked.sort(function(a, b){return (a.from || 0) - (b.from || 0);});
    },
    // Run the given mode's parser over a line, update the styles
    // array, which contains alternating fragments of text and CSS
    // classes.
    highlight: function(mode, state, tabSize) {
      var stream = new StringStream(this.text, tabSize), st = this.styles, pos = 0;
      var changed = false, curWord = st[0], prevWord;
      if (this.text == "" && mode.blankLine) mode.blankLine(state);
      while (!stream.eol()) {
        var style = mode.token(stream, state);
        var substr = this.text.slice(stream.start, stream.pos);
        stream.start = stream.pos;
        if (pos && st[pos-1] == style)
          st[pos-2] += substr;
        else if (substr) {
          if (!changed && (st[pos+1] != style || (pos && st[pos-2] != prevWord))) changed = true;
          st[pos++] = substr; st[pos++] = style;
          prevWord = curWord; curWord = st[pos];
        }
        // Give up when line is ridiculously long
        if (stream.pos > 5000) {
          st[pos++] = this.text.slice(stream.pos); st[pos++] = null;
          break;
        }
      }
      if (st.length != pos) {st.length = pos; changed = true;}
      if (pos && st[pos-2] != prevWord) changed = true;
      // Short lines with simple highlights return null, and are
      // counted as changed by the driver because they are likely to
      // highlight the same way in various contexts.
      return changed || (st.length < 5 && this.text.length < 10 ? null : false);
    },
    // Fetch the parser token for a given character. Useful for hacks
    // that want to inspect the mode state (say, for completion).
    getTokenAt: function(mode, state, ch) {
      var txt = this.text, stream = new StringStream(txt);
      while (stream.pos < ch && !stream.eol()) {
        stream.start = stream.pos;
        var style = mode.token(stream, state);
      }
      return {start: stream.start,
              end: stream.pos,
              string: stream.current(),
              className: style || null,
              state: state};
    },
    indentation: function(tabSize) {return countColumn(this.text, null, tabSize);},
    // Produces an HTML fragment for the line, taking selection,
    // marking, and highlighting into account.
    getHTML: function(makeTab, wrapAt, wrapId, wrapWBR) {
      var html = [], first = true, col = 0;
      function span_(text, style) {
        if (!text) return;
        // Work around a bug where, in some compat modes, IE ignores leading spaces
        if (first && ie && text.charAt(0) == " ") text = "\u00a0" + text.slice(1);
        first = false;
        if (text.indexOf("\t") == -1) {
          col += text.length;
          var escaped = htmlEscape(text);
        } else {
          var escaped = "";
          for (var pos = 0;;) {
            var idx = text.indexOf("\t", pos);
            if (idx == -1) {
              escaped += htmlEscape(text.slice(pos));
              col += text.length - pos;
              break;
            } else {
              col += idx - pos;
              var tab = makeTab(col);
              escaped += htmlEscape(text.slice(pos, idx)) + tab.html;
              col += tab.width;
              pos = idx + 1;
            }
          }
        }
        if (style) html.push('<span class="', style, '">', escaped, "</span>");
        else html.push(escaped);
      }
      var span = span_;
      if (wrapAt != null) {
        var outPos = 0, open = "<span id=\"" + wrapId + "\">";
        span = function(text, style) {
          var l = text.length;
          if (wrapAt >= outPos && wrapAt < outPos + l) {
            if (wrapAt > outPos) {
              span_(text.slice(0, wrapAt - outPos), style);
              // See comment at the definition of spanAffectsWrapping
              if (wrapWBR) html.push("<wbr>");
            }
            html.push(open);
            var cut = wrapAt - outPos;
            span_(opera ? text.slice(cut, cut + 1) : text.slice(cut), style);
            html.push("</span>");
            if (opera) span_(text.slice(cut + 1), style);
            wrapAt--;
            outPos += l;
          } else {
            outPos += l;
            span_(text, style);
            // Output empty wrapper when at end of line
            if (outPos == wrapAt && outPos == len) html.push(open + " </span>");
            // Stop outputting HTML when gone sufficiently far beyond measure
            else if (outPos > wrapAt + 10 && /\s/.test(text)) span = function(){};
          }
        }
      }

      var st = this.styles, allText = this.text, marked = this.marked;
      var len = allText.length;
      function styleToClass(style) {
        if (!style) return null;
        return "cm-" + style.replace(/ +/g, " cm-");
      }

      if (!allText && wrapAt == null) {
        span(" ");
      } else if (!marked || !marked.length) {
        for (var i = 0, ch = 0; ch < len; i+=2) {
          var str = st[i], style = st[i+1], l = str.length;
          if (ch + l > len) str = str.slice(0, len - ch);
          ch += l;
          span(str, styleToClass(style));
        }
      } else {
        var pos = 0, i = 0, text = "", style, sg = 0;
        var nextChange = marked[0].from || 0, marks = [], markpos = 0;
        function advanceMarks() {
          var m;
          while (markpos < marked.length &&
                 ((m = marked[markpos]).from == pos || m.from == null)) {
            if (m.style != null) marks.push(m);
            ++markpos;
          }
          nextChange = markpos < marked.length ? marked[markpos].from : Infinity;
          for (var i = 0; i < marks.length; ++i) {
            var to = marks[i].to || Infinity;
            if (to == pos) marks.splice(i--, 1);
            else nextChange = Math.min(to, nextChange);
          }
        }
        var m = 0;
        while (pos < len) {
          if (nextChange == pos) advanceMarks();
          var upto = Math.min(len, nextChange);
          while (true) {
            if (text) {
              var end = pos + text.length;
              var appliedStyle = style;
              for (var j = 0; j < marks.length; ++j)
                appliedStyle = (appliedStyle ? appliedStyle + " " : "") + marks[j].style;
              span(end > upto ? text.slice(0, upto - pos) : text, appliedStyle);
              if (end >= upto) {text = text.slice(upto - pos); pos = upto; break;}
              pos = end;
            }
            text = st[i++]; style = styleToClass(st[i++]);
          }
        }
      }
      return html.join("");
    },
    cleanUp: function() {
      this.parent = null;
      if (this.marked)
        for (var i = 0, e = this.marked.length; i < e; ++i) this.marked[i].detach(this);
    }
  };
  // Utility used by replace and split above
  function copyStyles(from, to, source, dest) {
    for (var i = 0, pos = 0, state = 0; pos < to; i+=2) {
      var part = source[i], end = pos + part.length;
      if (state == 0) {
        if (end > from) dest.push(part.slice(from - pos, Math.min(part.length, to - pos)), source[i+1]);
        if (end >= from) state = 1;
      } else if (state == 1) {
        if (end > to) dest.push(part.slice(0, to - pos), source[i+1]);
        else dest.push(part, source[i+1]);
      }
      pos = end;
    }
  }

  // Data structure that holds the sequence of lines.
  function LeafChunk(lines) {
    this.lines = lines;
    this.parent = null;
    for (var i = 0, e = lines.length, height = 0; i < e; ++i) {
      lines[i].parent = this;
      height += lines[i].height;
    }
    this.height = height;
  }
  LeafChunk.prototype = {
    chunkSize: function() { return this.lines.length; },
    remove: function(at, n, callbacks) {
      for (var i = at, e = at + n; i < e; ++i) {
        var line = this.lines[i];
        this.height -= line.height;
        line.cleanUp();
        if (line.handlers)
          for (var j = 0; j < line.handlers.length; ++j) callbacks.push(line.handlers[j]);
      }
      this.lines.splice(at, n);
    },
    collapse: function(lines) {
      lines.splice.apply(lines, [lines.length, 0].concat(this.lines));
    },
    insertHeight: function(at, lines, height) {
      this.height += height;
      this.lines = this.lines.slice(0, at).concat(lines).concat(this.lines.slice(at));
      for (var i = 0, e = lines.length; i < e; ++i) lines[i].parent = this;
    },
    iterN: function(at, n, op) {
      for (var e = at + n; at < e; ++at)
        if (op(this.lines[at])) return true;
    }
  };
  function BranchChunk(children) {
    this.children = children;
    var size = 0, height = 0;
    for (var i = 0, e = children.length; i < e; ++i) {
      var ch = children[i];
      size += ch.chunkSize(); height += ch.height;
      ch.parent = this;
    }
    this.size = size;
    this.height = height;
    this.parent = null;
  }
  BranchChunk.prototype = {
    chunkSize: function() { return this.size; },
    remove: function(at, n, callbacks) {
      this.size -= n;
      for (var i = 0; i < this.children.length; ++i) {
        var child = this.children[i], sz = child.chunkSize();
        if (at < sz) {
          var rm = Math.min(n, sz - at), oldHeight = child.height;
          child.remove(at, rm, callbacks);
          this.height -= oldHeight - child.height;
          if (sz == rm) { this.children.splice(i--, 1); child.parent = null; }
          if ((n -= rm) == 0) break;
          at = 0;
        } else at -= sz;
      }
      if (this.size - n < 25) {
        var lines = [];
        this.collapse(lines);
        this.children = [new LeafChunk(lines)];
        this.children[0].parent = this;
      }
    },
    collapse: function(lines) {
      for (var i = 0, e = this.children.length; i < e; ++i) this.children[i].collapse(lines);
    },
    insert: function(at, lines) {
      var height = 0;
      for (var i = 0, e = lines.length; i < e; ++i) height += lines[i].height;
      this.insertHeight(at, lines, height);
    },
    insertHeight: function(at, lines, height) {
      this.size += lines.length;
      this.height += height;
      for (var i = 0, e = this.children.length; i < e; ++i) {
        var child = this.children[i], sz = child.chunkSize();
        if (at <= sz) {
          child.insertHeight(at, lines, height);
          if (child.lines && child.lines.length > 50) {
            while (child.lines.length > 50) {
              var spilled = child.lines.splice(child.lines.length - 25, 25);
              var newleaf = new LeafChunk(spilled);
              child.height -= newleaf.height;
              this.children.splice(i + 1, 0, newleaf);
              newleaf.parent = this;
            }
            this.maybeSpill();
          }
          break;
        }
        at -= sz;
      }
    },
    maybeSpill: function() {
      if (this.children.length <= 10) return;
      var me = this;
      do {
        var spilled = me.children.splice(me.children.length - 5, 5);
        var sibling = new BranchChunk(spilled);
        if (!me.parent) { // Become the parent node
          var copy = new BranchChunk(me.children);
          copy.parent = me;
          me.children = [copy, sibling];
          me = copy;
        } else {
          me.size -= sibling.size;
          me.height -= sibling.height;
          var myIndex = indexOf(me.parent.children, me);
          me.parent.children.splice(myIndex + 1, 0, sibling);
        }
        sibling.parent = me.parent;
      } while (me.children.length > 10);
      me.parent.maybeSpill();
    },
    iter: function(from, to, op) { this.iterN(from, to - from, op); },
    iterN: function(at, n, op) {
      for (var i = 0, e = this.children.length; i < e; ++i) {
        var child = this.children[i], sz = child.chunkSize();
        if (at < sz) {
          var used = Math.min(n, sz - at);
          if (child.iterN(at, used, op)) return true;
          if ((n -= used) == 0) break;
          at = 0;
        } else at -= sz;
      }
    }
  };

  function getLineAt(chunk, n) {
    while (!chunk.lines) {
      for (var i = 0;; ++i) {
        var child = chunk.children[i], sz = child.chunkSize();
        if (n < sz) { chunk = child; break; }
        n -= sz;
      }
    }
    return chunk.lines[n];
  }
  function lineNo(line) {
    if (line.parent == null) return null;
    var cur = line.parent, no = indexOf(cur.lines, line);
    for (var chunk = cur.parent; chunk; cur = chunk, chunk = chunk.parent) {
      for (var i = 0, e = chunk.children.length; ; ++i) {
        if (chunk.children[i] == cur) break;
        no += chunk.children[i].chunkSize();
      }
    }
    return no;
  }
  function lineAtHeight(chunk, h) {
    var n = 0;
    outer: do {
      for (var i = 0, e = chunk.children.length; i < e; ++i) {
        var child = chunk.children[i], ch = child.height;
        if (h < ch) { chunk = child; continue outer; }
        h -= ch;
        n += child.chunkSize();
      }
      return n;
    } while (!chunk.lines);
    for (var i = 0, e = chunk.lines.length; i < e; ++i) {
      var line = chunk.lines[i], lh = line.height;
      if (h < lh) break;
      h -= lh;
    }
    return n + i;
  }
  function heightAtLine(chunk, n) {
    var h = 0;
    outer: do {
      for (var i = 0, e = chunk.children.length; i < e; ++i) {
        var child = chunk.children[i], sz = child.chunkSize();
        if (n < sz) { chunk = child; continue outer; }
        n -= sz;
        h += child.height;
      }
      return h;
    } while (!chunk.lines);
    for (var i = 0; i < n; ++i) h += chunk.lines[i].height;
    return h;
  }

  // The history object 'chunks' changes that are made close together
  // and at almost the same time into bigger undoable units.
  function History() {
    this.time = 0;
    this.done = []; this.undone = [];
    this.compound = 0;
    this.closed = false;
  }
  History.prototype = {
    addChange: function(start, added, old) {
      this.undone.length = 0;
      var time = +new Date, cur = this.done[this.done.length - 1], last = cur && cur[cur.length - 1];
      var dtime = time - this.time;

      if (this.compound && cur && !this.closed) {
        cur.push({start: start, added: added, old: old});
      } else if (dtime > 400 || !last || this.closed ||
                 last.start > start + old.length || last.start + last.added < start) {
        this.done.push([{start: start, added: added, old: old}]);
        this.closed = false;
      } else {
        var startBefore = Math.max(0, last.start - start),
            endAfter = Math.max(0, (start + old.length) - (last.start + last.added));
        for (var i = startBefore; i > 0; --i) last.old.unshift(old[i - 1]);
        for (var i = endAfter; i > 0; --i) last.old.push(old[old.length - i]);
        if (startBefore) last.start = start;
        last.added += added - (old.length - startBefore - endAfter);
      }
      this.time = time;
    },
    startCompound: function() {
      if (!this.compound++) this.closed = true;
    },
    endCompound: function() {
      if (!--this.compound) this.closed = true;
    }
  };

  function stopMethod() {e_stop(this);}
  // Ensure an event has a stop method.
  function addStop(event) {
    if (!event.stop) event.stop = stopMethod;
    return event;
  }

  function e_preventDefault(e) {
    if (e.preventDefault) e.preventDefault();
    else e.returnValue = false;
  }
  function e_stopPropagation(e) {
    if (e.stopPropagation) e.stopPropagation();
    else e.cancelBubble = true;
  }
  function e_stop(e) {e_preventDefault(e); e_stopPropagation(e);}
  CodeMirror.e_stop = e_stop;
  CodeMirror.e_preventDefault = e_preventDefault;
  CodeMirror.e_stopPropagation = e_stopPropagation;

  function e_target(e) {return e.target || e.srcElement;}
  function e_button(e) {
    if (e.which) return e.which;
    else if (e.button & 1) return 1;
    else if (e.button & 2) return 3;
    else if (e.button & 4) return 2;
  }

  // Allow 3rd-party code to override event properties by adding an override
  // object to an event object.
  function e_prop(e, prop) {
    var overridden = e.override && e.override.hasOwnProperty(prop);
    return overridden ? e.override[prop] : e[prop];
  }

  // Event handler registration. If disconnect is true, it'll return a
  // function that unregisters the handler.
  function connect(node, type, handler, disconnect) {
    if (typeof node.addEventListener == "function") {
      node.addEventListener(type, handler, false);
      if (disconnect) return function() {node.removeEventListener(type, handler, false);};
    } else {
      var wrapHandler = function(event) {handler(event || window.event);};
      node.attachEvent("on" + type, wrapHandler);
      if (disconnect) return function() {node.detachEvent("on" + type, wrapHandler);};
    }
  }
  CodeMirror.connect = connect;

  function Delayed() {this.id = null;}
  Delayed.prototype = {set: function(ms, f) {clearTimeout(this.id); this.id = setTimeout(f, ms);}};

  var Pass = CodeMirror.Pass = {toString: function(){return "CodeMirror.Pass";}};

  var gecko = /gecko\/\d{7}/i.test(navigator.userAgent);
  var ie = /MSIE \d/.test(navigator.userAgent);
  var ie_lt8 = /MSIE [1-7]\b/.test(navigator.userAgent);
  var ie_lt9 = /MSIE [1-8]\b/.test(navigator.userAgent);
  var quirksMode = ie && document.documentMode == 5;
  var webkit = /WebKit\//.test(navigator.userAgent);
  var chrome = /Chrome\//.test(navigator.userAgent);
  var opera = /Opera\//.test(navigator.userAgent);
  var safari = /Apple Computer/.test(navigator.vendor);
  var khtml = /KHTML\//.test(navigator.userAgent);
  var mac_geLion = /Mac OS X 10\D([7-9]|\d\d)\D/.test(navigator.userAgent);

  // Detect drag-and-drop
  var dragAndDrop = function() {
    // There is *some* kind of drag-and-drop support in IE6-8, but I
    // couldn't get it to work yet.
    if (ie_lt9) return false;
    var div = document.createElement('div');
    return "draggable" in div || "dragDrop" in div;
  }();

  // Feature-detect whether newlines in textareas are converted to \r\n
  var lineSep = function () {
    var te = document.createElement("textarea");
    te.value = "foo\nbar";
    if (te.value.indexOf("\r") > -1) return "\r\n";
    return "\n";
  }();

  // For a reason I have yet to figure out, some browsers disallow
  // word wrapping between certain characters *only* if a new inline
  // element is started between them. This makes it hard to reliably
  // measure the position of things, since that requires inserting an
  // extra span. This terribly fragile set of regexps matches the
  // character combinations that suffer from this phenomenon on the
  // various browsers.
  var spanAffectsWrapping = /^$/; // Won't match any two-character string
  if (gecko) spanAffectsWrapping = /$'/;
  else if (safari) spanAffectsWrapping = /\-[^ \-?]|\?[^ !'\"\),.\-\/:;\?\]\}]/;
  else if (chrome) spanAffectsWrapping = /\-[^ \-\.?]|\?[^ \-\.?\]\}:;!'\"\),\/]|[\.!\"#&%\)*+,:;=>\]|\}~][\(\{\[<]|\$'/;

  // Counts the column offset in a string, taking tabs into account.
  // Used mostly to find indentation.
  function countColumn(string, end, tabSize) {
    if (end == null) {
      end = string.search(/[^\s\u00a0]/);
      if (end == -1) end = string.length;
    }
    for (var i = 0, n = 0; i < end; ++i) {
      if (string.charAt(i) == "\t") n += tabSize - (n % tabSize);
      else ++n;
    }
    return n;
  }

  function computedStyle(elt) {
    if (elt.currentStyle) return elt.currentStyle;
    return window.getComputedStyle(elt, null);
  }

  // Find the position of an element by following the offsetParent chain.
  // If screen==true, it returns screen (rather than page) coordinates.
  function eltOffset(node, screen) {
    var bod = node.ownerDocument.body;
    var x = 0, y = 0, skipBody = false;
    for (var n = node; n; n = n.offsetParent) {
      var ol = n.offsetLeft, ot = n.offsetTop;
      // Firefox reports weird inverted offsets when the body has a border.
      if (n == bod) { x += Math.abs(ol); y += Math.abs(ot); }
      else { x += ol, y += ot; }
      if (screen && computedStyle(n).position == "fixed")
        skipBody = true;
    }
    var e = screen && !skipBody ? null : bod;
    for (var n = node.parentNode; n != e; n = n.parentNode)
      if (n.scrollLeft != null) { x -= n.scrollLeft; y -= n.scrollTop;}
    return {left: x, top: y};
  }
  // Use the faster and saner getBoundingClientRect method when possible.
  if (document.documentElement.getBoundingClientRect != null) eltOffset = function(node, screen) {
    // Take the parts of bounding client rect that we are interested in so we are able to edit if need be,
    // since the returned value cannot be changed externally (they are kept in sync as the element moves within the page)
    try { var box = node.getBoundingClientRect(); box = { top: box.top, left: box.left }; }
    catch(e) { box = {top: 0, left: 0}; }
    if (!screen) {
      // Get the toplevel scroll, working around browser differences.
      if (window.pageYOffset == null) {
        var t = document.documentElement || document.body.parentNode;
        if (t.scrollTop == null) t = document.body;
        box.top += t.scrollTop; box.left += t.scrollLeft;
      } else {
        box.top += window.pageYOffset; box.left += window.pageXOffset;
      }
    }
    return box;
  };

  // Get a node's text content.
  function eltText(node) {
    return node.textContent || node.innerText || node.nodeValue || "";
  }
  function selectInput(node) {
    if (ios) { // Mobile Safari apparently has a bug where select() is broken.
      node.selectionStart = 0;
      node.selectionEnd = node.value.length;
    } else node.select();
  }

  // Operations on {line, ch} objects.
  function posEq(a, b) {return a.line == b.line && a.ch == b.ch;}
  function posLess(a, b) {return a.line < b.line || (a.line == b.line && a.ch < b.ch);}
  function copyPos(x) {return {line: x.line, ch: x.ch};}

  var escapeElement = document.createElement("pre");
  function htmlEscape(str) {
    escapeElement.textContent = str;
    return escapeElement.innerHTML;
  }
  // Recent (late 2011) Opera betas insert bogus newlines at the start
  // of the textContent, so we strip those.
  if (htmlEscape("a") == "\na") {
    htmlEscape = function(str) {
      escapeElement.textContent = str;
      return escapeElement.innerHTML.slice(1);
    };
  // Some IEs don't preserve tabs through innerHTML
  } else if (htmlEscape("\t") != "\t") {
    htmlEscape = function(str) {
      escapeElement.innerHTML = "";
      escapeElement.appendChild(document.createTextNode(str));
      return escapeElement.innerHTML;
    };
  }
  CodeMirror.htmlEscape = htmlEscape;

  // Used to position the cursor after an undo/redo by finding the
  // last edited character.
  function editEnd(from, to) {
    if (!to) return 0;
    if (!from) return to.length;
    for (var i = from.length, j = to.length; i >= 0 && j >= 0; --i, --j)
      if (from.charAt(i) != to.charAt(j)) break;
    return j + 1;
  }

  function indexOf(collection, elt) {
    if (collection.indexOf) return collection.indexOf(elt);
    for (var i = 0, e = collection.length; i < e; ++i)
      if (collection[i] == elt) return i;
    return -1;
  }
  function isWordChar(ch) {
    return /\w/.test(ch) || ch.toUpperCase() != ch.toLowerCase();
  }

  // See if "".split is the broken IE version, if so, provide an
  // alternative way to split lines.
  var splitLines = "\n\nb".split(/\n/).length != 3 ? function(string) {
    var pos = 0, nl, result = [];
    while ((nl = string.indexOf("\n", pos)) > -1) {
      result.push(string.slice(pos, string.charAt(nl-1) == "\r" ? nl - 1 : nl));
      pos = nl + 1;
    }
    result.push(string.slice(pos));
    return result;
  } : function(string){return string.split(/\r?\n/);};
  CodeMirror.splitLines = splitLines;

  var hasSelection = window.getSelection ? function(te) {
    try { return te.selectionStart != te.selectionEnd; }
    catch(e) { return false; }
  } : function(te) {
    try {var range = te.ownerDocument.selection.createRange();}
    catch(e) {}
    if (!range || range.parentElement() != te) return false;
    return range.compareEndPoints("StartToEnd", range) != 0;
  };

  CodeMirror.defineMode("null", function() {
    return {token: function(stream) {stream.skipToEnd();}};
  });
  CodeMirror.defineMIME("text/plain", "null");

  var keyNames = {3: "Enter", 8: "Backspace", 9: "Tab", 13: "Enter", 16: "Shift", 17: "Ctrl", 18: "Alt",
                  19: "Pause", 20: "CapsLock", 27: "Esc", 32: "Space", 33: "PageUp", 34: "PageDown", 35: "End",
                  36: "Home", 37: "Left", 38: "Up", 39: "Right", 40: "Down", 44: "PrintScrn", 45: "Insert",
                  46: "Delete", 59: ";", 91: "Mod", 92: "Mod", 93: "Mod", 109: "-", 107: "=", 127: "Delete",
                  186: ";", 187: "=", 188: ",", 189: "-", 190: ".", 191: "/", 192: "`", 219: "[", 220: "\\",
                  221: "]", 222: "'", 63276: "PageUp", 63277: "PageDown", 63275: "End", 63273: "Home",
                  63234: "Left", 63232: "Up", 63235: "Right", 63233: "Down", 63302: "Insert", 63272: "Delete"};
  CodeMirror.keyNames = keyNames;
  (function() {
    // Number keys
    for (var i = 0; i < 10; i++) keyNames[i + 48] = String(i);
    // Alphabetic keys
    for (var i = 65; i <= 90; i++) keyNames[i] = String.fromCharCode(i);
    // Function keys
    for (var i = 1; i <= 12; i++) keyNames[i + 111] = keyNames[i + 63235] = "F" + i;
  })();

  return CodeMirror;
})();
// Define search commands. Depends on dialog.js or another
// implementation of the openDialog method.

// Replace works a little oddly -- it will do the replace on the next
// Ctrl-G (or whatever is bound to findNext) press. You prevent a
// replace by making sure the match is no longer selected when hitting
// Ctrl-G.

(function() {
	function SearchState() {
		this.posFrom = this.posTo = this.query = null;
		this.marked = [];
	}
	function getSearchState(cm) {
		return cm._searchState || (cm._searchState = new SearchState());
	}
	function getSearchCursor(cm, query, pos) {
		// Heuristic: if the query string is all lowercase, do a case insensitive search.
		return cm.getSearchCursor(query, pos, typeof query == "string" && query == query.toLowerCase());
	}
	function dialog(cm, text, shortText, f) {
		if (cm.openDialog) cm.openDialog(text, f);
		else f(prompt(shortText, ""));
	}
	function confirmDialog(cm, text, shortText, fs) {
		if (cm.openConfirm) cm.openConfirm(text, fs);
		else if (confirm(shortText)) fs[0]();
	}
	function parseQuery(query) {
		var isRE = query.match(/^\/(.*)\/([a-z]*)$/);
		return isRE ? new RegExp(isRE[1], isRE[2].indexOf("i") == -1 ? "" : "i") : query;
	}
	
	var queryDialog = 'Search: <input type="text" style="width: 10em"/> <span style="color: #888">(Use /re/ syntax for regexp search)</span>';
	
	function doSearch(query , cm, rev) {
		var state = getSearchState(cm);
		if (state.query) return findNext(cm, rev);
		cm.operation(function() {
			if (!query || state.query) return;
			state.query = parseQuery(query);
			if (cm.lineCount() < 2000) { // This is too expensive on big documents.
				for (var cursor = getSearchCursor(cm, query); cursor.findNext();)
					state.marked.push(cm.markText(cursor.from(), cursor.to(), "CodeMirror-searching"));
			}
			state.posFrom = state.posTo = cm.getCursor();
			findNext(cm, rev);
		});
		
	}
	function findNext(cm, rev) {cm.operation(function() {
		var state = getSearchState(cm);
		var cursor = getSearchCursor(cm, state.query, rev ? state.posFrom : state.posTo);
		if (!cursor.find(rev)) {
			cursor = getSearchCursor(cm, state.query, rev ? {line: cm.lineCount() - 1} : {line: 0, ch: 0});
			if (!cursor.find(rev)) return;
		}
		cm.setSelection(cursor.from(), cursor.to());
		state.posFrom = cursor.from(); state.posTo = cursor.to();
	})}
	function clearSearch(cm) {cm.operation(function() {
		var state = getSearchState(cm);
		if (!state.query) return;
		state.query = null;
		for (var i = 0; i < state.marked.length; ++i) state.marked[i].clear();
		state.marked.length = 0;
	})}
	var replaceQueryDialog = 'Replace: <input type="text" style="width: 10em"/> <span style="color: #888">(Use /re/ syntax for regexp search)</span>';
	var replacementQueryDialog = 'With: <input type="text" style="width: 10em"/>';
	var doReplaceConfirm = "Replace? <button>Yes</button> <button>No</button> <button>Stop</button>";
	function replace(cm, all) {
		var query = $("#search_query_for_replace").val() ;
		var text = $("#replace_query").val() ;
		if (!query) return;
		var cm = editor ;
		query = parseQuery(query);
		
		if (all) {
			cm.compoundChange(function() { cm.operation(function() {
				for (var cursor = getSearchCursor(cm, query); cursor.findNext();) {
					if (typeof query != "string") {
						var match = cm.getRange(cursor.from(), cursor.to()).match(query);
						cursor.replace(text.replace(/\$(\d)/, function(w, i) {return match[i];}));
					} else cursor.replace(text);
				}
			})});
		} else {
			clearSearch(cm);
			var cursor = getSearchCursor(cm, query, cm.getCursor());
			function advance() {
				var start = cursor.from(), match;
				if (!(match = cursor.findNext())) {
					cursor = getSearchCursor(cm, query);
					if (!(match = cursor.findNext()) || (start && cursor.from().line == start.line && cursor.from().ch == start.ch)) 
						return;
				}
				cm.setSelection(cursor.from(), cursor.to());
				doReplace(match) ;
			}
			function doReplace(match) {
				cursor.replace(typeof query == "string" ? text : text.replace(/\$(\d)/, function(w, i) {return match[i];}));
				advance();
			}
			advance();
		}
	}

	CodeMirror.commands.find = function(query , cm) {clearSearch(cm); doSearch(query , cm);};
	CodeMirror.commands.findNext = doSearch;
	CodeMirror.commands.findPrev = function(cm) {doSearch(cm, true);};
	CodeMirror.commands.clearSearch = clearSearch;
	CodeMirror.commands.replace = replace;
	CodeMirror.commands.replaceAll = function(cm) {replace(cm, true);};
})();
(function(){
  function SearchCursor(cm, query, pos, caseFold) {
    this.atOccurrence = false; this.cm = cm;
    if (caseFold == null && typeof query == "string") caseFold = false;

    pos = pos ? cm.clipPos(pos) : {line: 0, ch: 0};
    this.pos = {from: pos, to: pos};

    // The matches method is filled in based on the type of query.
    // It takes a position and a direction, and returns an object
    // describing the next occurrence of the query, or null if no
    // more matches were found.
    if (typeof query != "string") // Regexp match
      this.matches = function(reverse, pos) {
        if (reverse) {
          var line = cm.getLine(pos.line).slice(0, pos.ch), match = line.match(query), start = 0;
          while (match) {
            var ind = line.indexOf(match[0]);
            start += ind;
            line = line.slice(ind + 1);
            var newmatch = line.match(query);
            if (newmatch) match = newmatch;
            else break;
            start++;
          }
        }
        else {
          var line = cm.getLine(pos.line).slice(pos.ch), match = line.match(query),
          start = match && pos.ch + line.indexOf(match[0]);
        }
        if (match)
          return {from: {line: pos.line, ch: start},
                  to: {line: pos.line, ch: start + match[0].length},
                  match: match};
      };
    else { // String query
      if (caseFold) query = query.toLowerCase();
      var fold = caseFold ? function(str){return str.toLowerCase();} : function(str){return str;};
      var target = query.split("\n");
      // Different methods for single-line and multi-line queries
      if (target.length == 1)
        this.matches = function(reverse, pos) {
          var line = fold(cm.getLine(pos.line)), len = query.length, match;
          if (reverse ? (pos.ch >= len && (match = line.lastIndexOf(query, pos.ch - len)) != -1)
              : (match = line.indexOf(query, pos.ch)) != -1)
            return {from: {line: pos.line, ch: match},
                    to: {line: pos.line, ch: match + len}};
        };
      else
        this.matches = function(reverse, pos) {
          var ln = pos.line, idx = (reverse ? target.length - 1 : 0), match = target[idx], line = fold(cm.getLine(ln));
          var offsetA = (reverse ? line.indexOf(match) + match.length : line.lastIndexOf(match));
          if (reverse ? offsetA >= pos.ch || offsetA != match.length
              : offsetA <= pos.ch || offsetA != line.length - match.length)
            return;
          for (;;) {
            if (reverse ? !ln : ln == cm.lineCount() - 1) return;
            line = fold(cm.getLine(ln += reverse ? -1 : 1));
            match = target[reverse ? --idx : ++idx];
            if (idx > 0 && idx < target.length - 1) {
              if (line != match) return;
              else continue;
            }
            var offsetB = (reverse ? line.lastIndexOf(match) : line.indexOf(match) + match.length);
            if (reverse ? offsetB != line.length - match.length : offsetB != match.length)
              return;
            var start = {line: pos.line, ch: offsetA}, end = {line: ln, ch: offsetB};
            return {from: reverse ? end : start, to: reverse ? start : end};
          }
        };
    }
  }

  SearchCursor.prototype = {
    findNext: function() {return this.find(false);},
    findPrevious: function() {return this.find(true);},

    find: function(reverse) {
      var self = this, pos = this.cm.clipPos(reverse ? this.pos.from : this.pos.to);
      function savePosAndFail(line) {
        var pos = {line: line, ch: 0};
        self.pos = {from: pos, to: pos};
        self.atOccurrence = false;
        return false;
      }

      for (;;) {
        if (this.pos = this.matches(reverse, pos)) {
          this.atOccurrence = true;
          return this.pos.match || true;
        }
        if (reverse) {
          if (!pos.line) return savePosAndFail(0);
          pos = {line: pos.line-1, ch: this.cm.getLine(pos.line-1).length};
        }
        else {
          var maxLine = this.cm.lineCount();
          if (pos.line == maxLine - 1) return savePosAndFail(maxLine);
          pos = {line: pos.line+1, ch: 0};
        }
      }
    },

    from: function() {if (this.atOccurrence) return this.pos.from;},
    to: function() {if (this.atOccurrence) return this.pos.to;},

    replace: function(newText) {
      var self = this;
      if (this.atOccurrence)
        self.pos.to = this.cm.replaceRange(newText, self.pos.from, self.pos.to);
    }
  };

  CodeMirror.defineExtension("getSearchCursor", function(query, pos, caseFold) {
    return new SearchCursor(this, query, pos, caseFold);
  });
})();
// TODO number prefixes
(function() {
  // Really primitive kill-ring implementation.
  var killRing = [];
  function addToRing(str) {
    killRing.push(str);
    if (killRing.length > 50) killRing.shift();
  }
  function getFromRing() { return killRing[killRing.length - 1] || ""; }
  function popFromRing() { if (killRing.length > 1) killRing.pop(); return getFromRing(); }

  CodeMirror.keyMap.emacs = {
    "Ctrl-X": function(cm) {cm.setOption("keyMap", "emacs-Ctrl-X");},
    "Ctrl-W": function(cm) {addToRing(cm.getSelection()); cm.replaceSelection("");},
    "Ctrl-Alt-W": function(cm) {addToRing(cm.getSelection()); cm.replaceSelection("");},
    "Alt-W": function(cm) {addToRing(cm.getSelection());},
    "Ctrl-Y": function(cm) {cm.replaceSelection(getFromRing());},
    "Alt-Y": function(cm) {cm.replaceSelection(popFromRing());},
    "Ctrl-/": "undo", "Shift-Ctrl--": "undo", "Shift-Alt-,": "goDocStart", "Shift-Alt-.": "goDocEnd",
    "Ctrl-S": "findNext", "Ctrl-R": "findPrev", "Ctrl-G": "clearSearch", "Shift-Alt-5": "replace",
    "Ctrl-Z": "undo", "Cmd-Z": "undo", "Alt-/": "autocomplete",
    fallthrough: ["basic", "emacsy"]
  };

  CodeMirror.keyMap["emacs-Ctrl-X"] = {
    "Ctrl-S": "save", "Ctrl-W": "save", "S": "saveAll", "F": "open", "U": "undo", "K": "close",
    auto: "emacs", nofallthrough: true
  };
})();﻿

var zen_editor = (function(){
/**
 * Zen Coding settings
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
var zen_settings = {
	/** 
	 * Variables that can be placed inside snippets or abbreviations as ${variable}
	 * ${child} variable is reserved, don't use it 
	 */
	'variables': {
		'lang': 'ja',
		'locale': 'ja-JP',
		'charset': 'Shift_JIS',
		
		/** Inner element indentation */
		'indentation': '\t',
		
		// newline variables, useful for wrapping
		'newline': '\n',
		'nl': '\n'
	},
	
	'css': {
		'filters': 'html,css',
		'snippets': {
			"@i": "@import url(|);",
			"@m": "@media print {\n\t|\n}",
			"@f": "@font-face {\n\tfont-family:|;\n\tsrc:url(|);\n}",
			"!": "!important",
			"pos": "position:|;",
			"pos:s": "position:static;",
			"pos:a": "position:absolute;",
			"pos:r": "position:relative;",
			"pos:f": "position:fixed;",
			"t": "top:|;",
			"t:a": "top:auto;",
			"r": "right:|;",
			"r:a": "right:auto;",
			"b": "bottom:|;",
			"b:a": "bottom:auto;",
			"brad": "-webkit-border-radius: ${1:radius};\n-moz-border-radius: $1;\n-ms-border-radius: $1;\nborder-radius: $1;",
			"bsha": "-webkit-box-shadow: ${1:hoff} ${2:voff} ${3:blur} ${4:rgba(0,0,0,0.5)};\n-moz-box-shadow: $1 $2 $3 $4;\n-ms-box-shadow: $1 $2 $3 $4;\nbox-shadow: $1 $2 $3 $4;",
			"l": "left:|;",
			"l:a": "left:auto;",
			"z": "z-index:|;",
			"z:a": "z-index:auto;",
			"fl": "float:|;",
			"fl:n": "float:none;",
			"fl:l": "float:left;",
			"fl:r": "float:right;",
			"cl": "clear:|;",
			"cl:n": "clear:none;",
			"cl:l": "clear:left;",
			"cl:r": "clear:right;",
			"cl:b": "clear:both;",
			"d": "display:|;",
			"d:n": "display:none;",
			"d:b": "display:block;",
			"d:i": "display:inline;",
			"d:ib": "display:inline-block;",
			"d:li": "display:list-item;",
			"d:ri": "display:run-in;",
			"d:cp": "display:compact;",
			"d:tb": "display:table;",
			"d:itb": "display:inline-table;",
			"d:tbcp": "display:table-caption;",
			"d:tbcl": "display:table-column;",
			"d:tbclg": "display:table-column-group;",
			"d:tbhg": "display:table-header-group;",
			"d:tbfg": "display:table-footer-group;",
			"d:tbr": "display:table-row;",
			"d:tbrg": "display:table-row-group;",
			"d:tbc": "display:table-cell;",
			"d:rb": "display:ruby;",
			"d:rbb": "display:ruby-base;",
			"d:rbbg": "display:ruby-base-group;",
			"d:rbt": "display:ruby-text;",
			"d:rbtg": "display:ruby-text-group;",
			"v": "visibility:|;",
			"v:v": "visibility:visible;",
			"v:h": "visibility:hidden;",
			"v:c": "visibility:collapse;",
			"ov": "overflow:|;",
			"ov:v": "overflow:visible;",
			"ov:h": "overflow:hidden;",
			"ov:s": "overflow:scroll;",
			"ov:a": "overflow:auto;",
			"ovx": "overflow-x:|;",
			"ovx:v": "overflow-x:visible;",
			"ovx:h": "overflow-x:hidden;",
			"ovx:s": "overflow-x:scroll;",
			"ovx:a": "overflow-x:auto;",
			"ovy": "overflow-y:|;",
			"ovy:v": "overflow-y:visible;",
			"ovy:h": "overflow-y:hidden;",
			"ovy:s": "overflow-y:scroll;",
			"ovy:a": "overflow-y:auto;",
			"ovs": "overflow-style:|;",
			"ovs:a": "overflow-style:auto;",
			"ovs:s": "overflow-style:scrollbar;",
			"ovs:p": "overflow-style:panner;",
			"ovs:m": "overflow-style:move;",
			"ovs:mq": "overflow-style:marquee;",
			"zoo": "zoom:1;",
			"cp": "clip:|;",
			"cp:a": "clip:auto;",
			"cp:r": "clip:rect(|);",
			"bxz": "box-sizing:|;",
			"bxz:cb": "box-sizing:content-box;",
			"bxz:bb": "box-sizing:border-box;",
			"bxsh": "box-shadow:|;",
			"bxsh:n": "box-shadow:none;",
			"bxsh:w": "-webkit-box-shadow:0 0 0 #000;",
			"bxsh:m": "-moz-box-shadow:0 0 0 0 #000;",
			"m": "margin:|;",
			"m:a": "margin:auto;",
			"m:0": "margin:0;",
			"m:2": "margin:0 0;",
			"m:3": "margin:0 0 0;",
			"m:4": "margin:0 0 0 0;",
			"mt": "margin-top:|;",
			"mt:a": "margin-top:auto;",
			"mr": "margin-right:|;",
			"mr:a": "margin-right:auto;",
			"mb": "margin-bottom:|;",
			"mb:a": "margin-bottom:auto;",
			"ml": "margin-left:|;",
			"ml:a": "margin-left:auto;",
			"p": "padding:|;",
			"p:0": "padding:0;",
			"p:2": "padding:0 0;",
			"p:3": "padding:0 0 0;",
			"p:4": "padding:0 0 0 0;",
			"pt": "padding-top:|;",
			"pr": "padding-right:|;",
			"pb": "padding-bottom:|;",
			"pl": "padding-left:|;",
			"w": "width:|;",
			"w:a": "width:auto;",
			"h": "height:|;",
			"h:a": "height:auto;",
			"maw": "max-width:|;",
			"maw:n": "max-width:none;",
			"mah": "max-height:|;",
			"mah:n": "max-height:none;",
			"miw": "min-width:|;",
			"mih": "min-height:|;",
			"o": "outline:|;",
			"o:n": "outline:none;",
			"oo": "outline-offset:|;",
			"ow": "outline-width:|;",
			"os": "outline-style:|;",
			"oc": "outline-color:#000;",
			"oc:i": "outline-color:invert;",
			"bd": "border:|;",
			"bd+": "border:1px solid #000;",
			"bd:n": "border:none;",
			"bdbk": "border-break:|;",
			"bdbk:c": "border-break:close;",
			"bdcl": "border-collapse:|;",
			"bdcl:c": "border-collapse:collapse;",
			"bdcl:s": "border-collapse:separate;",
			"bdc": "border-color:#000;",
			"bdi": "border-image:url(|);",
			"bdi:n": "border-image:none;",
			"bdi:w": "-webkit-border-image:url(|) 0 0 0 0 stretch stretch;",
			"bdi:m": "-moz-border-image:url(|) 0 0 0 0 stretch stretch;",
			"bdti": "border-top-image:url(|);",
			"bdti:n": "border-top-image:none;",
			"bdri": "border-right-image:url(|);",
			"bdri:n": "border-right-image:none;",
			"bdbi": "border-bottom-image:url(|);",
			"bdbi:n": "border-bottom-image:none;",
			"bdli": "border-left-image:url(|);",
			"bdli:n": "border-left-image:none;",
			"bdci": "border-corner-image:url(|);",
			"bdci:n": "border-corner-image:none;",
			"bdci:c": "border-corner-image:continue;",
			"bdtli": "border-top-left-image:url(|);",
			"bdtli:n": "border-top-left-image:none;",
			"bdtli:c": "border-top-left-image:continue;",
			"bdtri": "border-top-right-image:url(|);",
			"bdtri:n": "border-top-right-image:none;",
			"bdtri:c": "border-top-right-image:continue;",
			"bdbri": "border-bottom-right-image:url(|);",
			"bdbri:n": "border-bottom-right-image:none;",
			"bdbri:c": "border-bottom-right-image:continue;",
			"bdbli": "border-bottom-left-image:url(|);",
			"bdbli:n": "border-bottom-left-image:none;",
			"bdbli:c": "border-bottom-left-image:continue;",
			"bdf": "border-fit:|;",
			"bdf:c": "border-fit:clip;",
			"bdf:r": "border-fit:repeat;",
			"bdf:sc": "border-fit:scale;",
			"bdf:st": "border-fit:stretch;",
			"bdf:ow": "border-fit:overwrite;",
			"bdf:of": "border-fit:overflow;",
			"bdf:sp": "border-fit:space;",
			"bdl": "border-length:|;",
			"bdl:a": "border-length:auto;",
			"bdsp": "border-spacing:|;",
			"bds": "border-style:|;",
			"bds:n": "border-style:none;",
			"bds:h": "border-style:hidden;",
			"bds:dt": "border-style:dotted;",
			"bds:ds": "border-style:dashed;",
			"bds:s": "border-style:solid;",
			"bds:db": "border-style:double;",
			"bds:dtds": "border-style:dot-dash;",
			"bds:dtdtds": "border-style:dot-dot-dash;",
			"bds:w": "border-style:wave;",
			"bds:g": "border-style:groove;",
			"bds:r": "border-style:ridge;",
			"bds:i": "border-style:inset;",
			"bds:o": "border-style:outset;",
			"bdw": "border-width:|;",
			"bdt": "border-top:|;",
			"bdt+": "border-top:1px solid #000;",
			"bdt:n": "border-top:none;",
			"bdtw": "border-top-width:|;",
			"bdts": "border-top-style:|;",
			"bdts:n": "border-top-style:none;",
			"bdtc": "border-top-color:#000;",
			"bdr": "border-right:|;",
			"bdr+": "border-right:1px solid #000;",
			"bdr:n": "border-right:none;",
			"bdrw": "border-right-width:|;",
			"bdrs": "border-right-style:|;",
			"bdrs:n": "border-right-style:none;",
			"bdrc": "border-right-color:#000;",
			"bdb": "border-bottom:|;",
			"bdb+": "border-bottom:1px solid #000;",
			"bdb:n": "border-bottom:none;",
			"bdbw": "border-bottom-width:|;",
			"bdbs": "border-bottom-style:|;",
			"bdbs:n": "border-bottom-style:none;",
			"bdbc": "border-bottom-color:#000;",
			"bdl": "border-left:|;",
			"bdl+": "border-left:1px solid #000;",
			"bdl:n": "border-left:none;",
			"bdlw": "border-left-width:|;",
			"bdls": "border-left-style:|;",
			"bdls:n": "border-left-style:none;",
			"bdlc": "border-left-color:#000;",
			"bdrs": "border-radius:|;",
			"bdtrrs": "border-top-right-radius:|;",
			"bdtlrs": "border-top-left-radius:|;",
			"bdbrrs": "border-bottom-right-radius:|;",
			"bdblrs": "border-bottom-left-radius:|;",
			"bg": "background:|;",
			"bg+": "background:#FFF url(|) 0 0 no-repeat;",
			"bg:n": "background:none;",
			"bg:ie": "filter:progid:DXImageTransform.Microsoft.AlphaImageLoader(src='${1:x}.png',sizingMethod='${2:crop}');",
			"bgc": "background-color:#FFF;",
			"bgi": "background-image:url(|);",
			"bgi:n": "background-image:none;",
			"bgr": "background-repeat:|;",
			"bgr:n": "background-repeat:no-repeat;",
			"bgr:x": "background-repeat:repeat-x;",
			"bgr:y": "background-repeat:repeat-y;",
			"bga": "background-attachment:|;",
			"bga:f": "background-attachment:fixed;",
			"bga:s": "background-attachment:scroll;",
			"bgp": "background-position:0 0;",
			"bgpx": "background-position-x:|;",
			"bgpy": "background-position-y:|;",
			"bgbk": "background-break:|;",
			"bgbk:bb": "background-break:bounding-box;",
			"bgbk:eb": "background-break:each-box;",
			"bgbk:c": "background-break:continuous;",
			"bgcp": "background-clip:|;",
			"bgcp:bb": "background-clip:border-box;",
			"bgcp:pb": "background-clip:padding-box;",
			"bgcp:cb": "background-clip:content-box;",
			"bgcp:nc": "background-clip:no-clip;",
			"bgo": "background-origin:|;",
			"bgo:pb": "background-origin:padding-box;",
			"bgo:bb": "background-origin:border-box;",
			"bgo:cb": "background-origin:content-box;",
			"bgz": "background-size:|;",
			"bgz:a": "background-size:auto;",
			"bgz:ct": "background-size:contain;",
			"bgz:cv": "background-size:cover;",
			"c": "color:#000;",
			"tbl": "table-layout:|;",
			"tbl:a": "table-layout:auto;",
			"tbl:f": "table-layout:fixed;",
			"cps": "caption-side:|;",
			"cps:t": "caption-side:top;",
			"cps:b": "caption-side:bottom;",
			"ec": "empty-cells:|;",
			"ec:s": "empty-cells:show;",
			"ec:h": "empty-cells:hide;",
			"lis": "list-style:|;",
			"lis:n": "list-style:none;",
			"lisp": "list-style-position:|;",
			"lisp:i": "list-style-position:inside;",
			"lisp:o": "list-style-position:outside;",
			"list": "list-style-type:|;",
			"list:n": "list-style-type:none;",
			"list:d": "list-style-type:disc;",
			"list:c": "list-style-type:circle;",
			"list:s": "list-style-type:square;",
			"list:dc": "list-style-type:decimal;",
			"list:dclz": "list-style-type:decimal-leading-zero;",
			"list:lr": "list-style-type:lower-roman;",
			"list:ur": "list-style-type:upper-roman;",
			"lisi": "list-style-image:|;",
			"lisi:n": "list-style-image:none;",
			"q": "quotes:|;",
			"q:n": "quotes:none;",
			"q:ru": "quotes:'\00AB' '\00BB' '\201E' '\201C';",
			"q:en": "quotes:'\201C' '\201D' '\2018' '\2019';",
			"ct": "content:|;",
			"ct:n": "content:normal;",
			"ct:oq": "content:open-quote;",
			"ct:noq": "content:no-open-quote;",
			"ct:cq": "content:close-quote;",
			"ct:ncq": "content:no-close-quote;",
			"ct:a": "content:attr(|);",
			"ct:c": "content:counter(|);",
			"ct:cs": "content:counters(|);",
			"coi": "counter-increment:|;",
			"cor": "counter-reset:|;",
			"va": "vertical-align:|;",
			"va:sup": "vertical-align:super;",
			"va:t": "vertical-align:top;",
			"va:tt": "vertical-align:text-top;",
			"va:m": "vertical-align:middle;",
			"va:bl": "vertical-align:baseline;",
			"va:b": "vertical-align:bottom;",
			"va:tb": "vertical-align:text-bottom;",
			"va:sub": "vertical-align:sub;",
			"ta": "text-align:|;",
			"ta:l": "text-align:left;",
			"ta:c": "text-align:center;",
			"ta:r": "text-align:right;",
			"tal": "text-align-last:|;",
			"tal:a": "text-align-last:auto;",
			"tal:l": "text-align-last:left;",
			"tal:c": "text-align-last:center;",
			"tal:r": "text-align-last:right;",
			"td": "text-decoration:|;",
			"td:n": "text-decoration:none;",
			"td:u": "text-decoration:underline;",
			"td:o": "text-decoration:overline;",
			"td:l": "text-decoration:line-through;",
			"te": "text-emphasis:|;",
			"te:n": "text-emphasis:none;",
			"te:ac": "text-emphasis:accent;",
			"te:dt": "text-emphasis:dot;",
			"te:c": "text-emphasis:circle;",
			"te:ds": "text-emphasis:disc;",
			"te:b": "text-emphasis:before;",
			"te:a": "text-emphasis:after;",
			"th": "text-height:|;",
			"th:a": "text-height:auto;",
			"th:f": "text-height:font-size;",
			"th:t": "text-height:text-size;",
			"th:m": "text-height:max-size;",
			"ti": "text-indent:|;",
			"ti:-": "text-indent:-9999px;",
			"tj": "text-justify:|;",
			"tj:a": "text-justify:auto;",
			"tj:iw": "text-justify:inter-word;",
			"tj:ii": "text-justify:inter-ideograph;",
			"tj:ic": "text-justify:inter-cluster;",
			"tj:d": "text-justify:distribute;",
			"tj:k": "text-justify:kashida;",
			"tj:t": "text-justify:tibetan;",
			"to": "text-outline:|;",
			"to+": "text-outline:0 0 #000;",
			"to:n": "text-outline:none;",
			"tr": "text-replace:|;",
			"tr:n": "text-replace:none;",
			"tt": "text-transform:|;",
			"tt:n": "text-transform:none;",
			"tt:c": "text-transform:capitalize;",
			"tt:u": "text-transform:uppercase;",
			"tt:l": "text-transform:lowercase;",
			"tw": "text-wrap:|;",
			"tw:n": "text-wrap:normal;",
			"tw:no": "text-wrap:none;",
			"tw:u": "text-wrap:unrestricted;",
			"tw:s": "text-wrap:suppress;",
			"tsh": "text-shadow:|;",
			"tsh+": "text-shadow:0 0 0 #000;",
			"tsh:n": "text-shadow:none;",
			"lh": "line-height:|;",
			"whs": "white-space:|;",
			"whs:n": "white-space:normal;",
			"whs:p": "white-space:pre;",
			"whs:nw": "white-space:nowrap;",
			"whs:pw": "white-space:pre-wrap;",
			"whs:pl": "white-space:pre-line;",
			"whsc": "white-space-collapse:|;",
			"whsc:n": "white-space-collapse:normal;",
			"whsc:k": "white-space-collapse:keep-all;",
			"whsc:l": "white-space-collapse:loose;",
			"whsc:bs": "white-space-collapse:break-strict;",
			"whsc:ba": "white-space-collapse:break-all;",
			"wob": "word-break:|;",
			"wob:n": "word-break:normal;",
			"wob:k": "word-break:keep-all;",
			"wob:l": "word-break:loose;",
			"wob:bs": "word-break:break-strict;",
			"wob:ba": "word-break:break-all;",
			"wos": "word-spacing:|;",
			"wow": "word-wrap:|;",
			"wow:nm": "word-wrap:normal;",
			"wow:n": "word-wrap:none;",
			"wow:u": "word-wrap:unrestricted;",
			"wow:s": "word-wrap:suppress;",
			"lts": "letter-spacing:|;",
			"f": "font:|;",
			"f+": "font:1em Arial,sans-serif;",
			"fw": "font-weight:|;",
			"fw:n": "font-weight:normal;",
			"fw:b": "font-weight:bold;",
			"fw:br": "font-weight:bolder;",
			"fw:lr": "font-weight:lighter;",
			"fs": "font-style:|;",
			"fs:n": "font-style:normal;",
			"fs:i": "font-style:italic;",
			"fs:o": "font-style:oblique;",
			"fv": "font-variant:|;",
			"fv:n": "font-variant:normal;",
			"fv:sc": "font-variant:small-caps;",
			"fz": "font-size:|;",
			"fza": "font-size-adjust:|;",
			"fza:n": "font-size-adjust:none;",
			"ff": "font-family:|;",
			"ff:s": "font-family:serif;",
			"ff:ss": "font-family:sans-serif;",
			"ff:c": "font-family:cursive;",
			"ff:f": "font-family:fantasy;",
			"ff:m": "font-family:monospace;",
			"fef": "font-effect:|;",
			"fef:n": "font-effect:none;",
			"fef:eg": "font-effect:engrave;",
			"fef:eb": "font-effect:emboss;",
			"fef:o": "font-effect:outline;",
			"fem": "font-emphasize:|;",
			"femp": "font-emphasize-position:|;",
			"femp:b": "font-emphasize-position:before;",
			"femp:a": "font-emphasize-position:after;",
			"fems": "font-emphasize-style:|;",
			"fems:n": "font-emphasize-style:none;",
			"fems:ac": "font-emphasize-style:accent;",
			"fems:dt": "font-emphasize-style:dot;",
			"fems:c": "font-emphasize-style:circle;",
			"fems:ds": "font-emphasize-style:disc;",
			"fsm": "font-smooth:|;",
			"fsm:a": "font-smooth:auto;",
			"fsm:n": "font-smooth:never;",
			"fsm:aw": "font-smooth:always;",
			"fst": "font-stretch:|;",
			"fst:n": "font-stretch:normal;",
			"fst:uc": "font-stretch:ultra-condensed;",
			"fst:ec": "font-stretch:extra-condensed;",
			"fst:c": "font-stretch:condensed;",
			"fst:sc": "font-stretch:semi-condensed;",
			"fst:se": "font-stretch:semi-expanded;",
			"fst:e": "font-stretch:expanded;",
			"fst:ee": "font-stretch:extra-expanded;",
			"fst:ue": "font-stretch:ultra-expanded;",
			"op": "opacity:|;",
			"op:ie": "filter:progid:DXImageTransform.Microsoft.Alpha(Opacity=100);",
			"op:ms": "-ms-filter:'progid:DXImageTransform.Microsoft.Alpha(Opacity=100)';",
			"rz": "resize:|;",
			"rz:n": "resize:none;",
			"rz:b": "resize:both;",
			"rz:h": "resize:horizontal;",
			"rz:v": "resize:vertical;",
			"cur": "cursor:|;",
			"cur:a": "cursor:auto;",
			"cur:d": "cursor:default;",
			"cur:c": "cursor:crosshair;",
			"cur:ha": "cursor:hand;",
			"cur:he": "cursor:help;",
			"cur:m": "cursor:move;",
			"cur:p": "cursor:pointer;",
			"cur:t": "cursor:text;",
			"pgbb": "page-break-before:|;",
			"pgbb:au": "page-break-before:auto;",
			"pgbb:al": "page-break-before:always;",
			"pgbb:l": "page-break-before:left;",
			"pgbb:r": "page-break-before:right;",
			"pgbi": "page-break-inside:|;",
			"pgbi:au": "page-break-inside:auto;",
			"pgbi:av": "page-break-inside:avoid;",
			"pgba": "page-break-after:|;",
			"pgba:au": "page-break-after:auto;",
			"pgba:al": "page-break-after:always;",
			"pgba:l": "page-break-after:left;",
			"pgba:r": "page-break-after:right;",
			"orp": "orphans:|;",
			"wid": "widows:|;"
		}
	},
	
	'html': {
		'filters': 'html',
		'snippets': {
			'cc:ie6': '<!--[if lte IE 6]>\n\t${child}|\n<![endif]-->',
			'cc:ie': '<!--[if IE]>\n\t${child}|\n<![endif]-->',
			'cc:noie': '<!--[if !IE]><!-->\n\t${child}|\n<!--<![endif]-->',
			'html:4t': '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">\n' +
					'<html lang="${lang}">\n' +
					'<head>\n' +
					'	<meta http-equiv="Content-Type" content="text/html;charset=${charset}">\n' +
					'	<title></title>\n' +
					'</head>\n' +
					'<body>\n\t${child}|\n</body>\n' +
					'</html>',
			
			'html:4s': '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">\n' +
					'<html lang="${lang}">\n' +
					'<head>\n' +
					'	<meta http-equiv="Content-Type" content="text/html;charset=${charset}">\n' +
					'	<title></title>\n' +
					'</head>\n' +
					'<body>\n\t${child}|\n</body>\n' +
					'</html>',
			
			'html:xt': '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">\n' +
					'<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="${lang}">\n' +
					'<head>\n' +
					'	<meta http-equiv="Content-Type" content="text/html;charset=${charset}" />\n' +
					'	<title></title>\n' +
					'</head>\n' +
					'<body>\n\t${child}|\n</body>\n' +
					'</html>',
			
			'html:xs': '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">\n' +
					'<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="${lang}">\n' +
					'<head>\n' +
					'	<meta http-equiv="Content-Type" content="text/html;charset=${charset}" />\n' +
					'	<title></title>\n' +
					'</head>\n' +
					'<body>\n\t${child}|\n</body>\n' +
					'</html>',
			
			'html:xxs': '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n' +
					'<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="${lang}">\n' +
					'<head>\n' +
					'	<meta http-equiv="Content-Type" content="text/html;charset=${charset}" />\n' +
					'	<title></title>\n' +
					'</head>\n' +
					'<body>\n\t${child}|\n</body>\n' +
					'</html>',
			
			'html:5': '<!DOCTYPE HTML>\n' +
					'<html lang="${lang}">\n' +
					'<head>\n' +
					'	<meta charset="${charset}">\n' +
					'	<title></title>\n' +
					'</head>\n' +
					'<body>\n\t${child}|\n</body>\n' +
					'</html>'
		},
		
		'abbreviations': {
			'a': '<a href="">',
			'a:link': '<a href="http://|">',
			'a:mail': '<a href="mailto:|">',
			'abbr': '<abbr title="">',
			'acronym': '<acronym title="">',
			'base': '<base href="" />',
			'bdo': '<bdo dir="">',
			'bdo:r': '<bdo dir="rtl">',
			'bdo:l': '<bdo dir="ltr">',
			'link:css': '<link rel="stylesheet" type="text/css" href="${1:style}.css" media="all" />',
			'link:print': '<link rel="stylesheet" type="text/css" href="|print.css" media="print" />',
			'link:favicon': '<link rel="shortcut icon" type="image/x-icon" href="|favicon.ico" />',
			'link:touch': '<link rel="apple-touch-icon" href="|favicon.png" />',
			'link:rss': '<link rel="alternate" type="application/rss+xml" title="RSS" href="|rss.xml" />',
			'link:atom': '<link rel="alternate" type="application/atom+xml" title="Atom" href="atom.xml" />',
			'meta:utf': '<meta http-equiv="Content-Type" content="text/html;charset=UTF-8" />',
			'meta:win': '<meta http-equiv="Content-Type" content="text/html;charset=windows-1251" />',
			'meta:compat': '<meta http-equiv="X-UA-Compatible" content="IE=7" />',
			'style': '<style type="text/css">',
			'script': '<script type="text/javascript">',
			'script:src': '<script type="text/javascript" src="">',
			'img': '<img src="" alt="" />',
			'iframe': '<iframe src="" frameborder="0">',
			'embed': '<embed src="" type="" />',
			'object': '<object data="" type="">',
			'param': '<param name="" value="" />',
			'map': '<map name="">',
			'area': '<area shape="" coords="" href="" alt="" />',
			'area:d': '<area shape="default" href="" alt="" />',
			'area:c': '<area shape="circle" coords="" href="" alt="" />',
			'area:r': '<area shape="rect" coords="" href="" alt="" />',
			'area:p': '<area shape="poly" coords="" href="" alt="" />',
			'link': '<link rel="stylesheet" href="" />',
			'form': '<form action="">',
			'form:get': '<form action="" method="get">',
			'form:post': '<form action="" method="post">',
			'label': '<label for="">',
			'input': '<input type="" />',
			'input:hidden': '<input type="hidden" name="" />',
			'input:h': '<input type="hidden" name="" />',
			'input:text': '<input type="text" name="" id="" />',
			'input:t': '<input type="text" name="" id="" />',
			'input:search': '<input type="search" name="" id="" />',
			'input:email': '<input type="email" name="" id="" />',
			'input:url': '<input type="url" name="" id="" />',
			'input:password': '<input type="password" name="" id="" />',
			'input:p': '<input type="password" name="" id="" />',
			'input:datetime': '<input type="datetime" name="" id="" />',
			'input:date': '<input type="date" name="" id="" />',
			'input:datetime-local': '<input type="datetime-local" name="" id="" />',
			'input:month': '<input type="month" name="" id="" />',
			'input:week': '<input type="week" name="" id="" />',
			'input:time': '<input type="time" name="" id="" />',
			'input:number': '<input type="number" name="" id="" />',
			'input:color': '<input type="color" name="" id="" />',
			'input:checkbox': '<input type="checkbox" name="" id="" />',
			'input:c': '<input type="checkbox" name="" id="" />',
			'input:radio': '<input type="radio" name="" id="" />',
			'input:r': '<input type="radio" name="" id="" />',
			'input:range': '<input type="range" name="" id="" />',
			'input:file': '<input type="file" name="" id="" />',
			'input:f': '<input type="file" name="" id="" />',
			'input:submit': '<input type="submit" value="" />',
			'input:s': '<input type="submit" value="" />',
			'input:image': '<input type="image" src="" alt="" />',
			'input:i': '<input type="image" src="" alt="" />',
			'input:reset': '<input type="reset" value="" />',
			'input:button': '<input type="button" value="" />',
			'input:b': '<input type="button" value="" />',
			'select': '<select name="" id=""></select>',
			'option': '<option value=""></option>',
			'textarea': '<textarea name="" id="" cols="30" rows="10">',
			'menu:context': '<menu type="context">',
			'menu:c': '<menu type="context">',
			'menu:toolbar': '<menu type="toolbar">',
			'menu:t': '<menu type="toolbar">',
			'video': '<video src="">',
			'audio': '<audio src="">',
			'html:xml': '<html xmlns="http://www.w3.org/1999/xhtml">',
			'bq': '<blockquote>',
			'acr': '<acronym>',
			'fig': '<figure>',
			'ifr': '<iframe>',
			'emb': '<embed>',
			'obj': '<object>',
			'src': '<source>',
			'cap': '<caption>',
			'colg': '<colgroup>',
			'fst': '<fieldset>',
			'btn': '<button>',
			'optg': '<optgroup>',
			'opt': '<option>',
			'tarea': '<textarea>',
			'leg': '<legend>',
			'sect': '<section>',
			'art': '<article>',
			'hdr': '<header>',
			'ftr': '<footer>',
			'adr': '<address>',
			'dlg': '<dialog>',
			'str': '<strong>',
			'prog': '<progress>',
			'fset': '<fieldset>',
			'datag': '<datagrid>',
			'datal': '<datalist>',
			'kg': '<keygen>',
			'out': '<output>',
			'det': '<details>',
			'cmd': '<command>',
			
			// expandos
			'ol+': 'ol>li',
			'ul+': 'ul>li',
			'dl+': 'dl>dt+dd',
			'map+': 'map>area',
			'table+': 'table>tr>td',
			'colgroup+': 'colgroup>col',
			'colg+': 'colgroup>col',
			'tr+': 'tr>td',
			'select+': 'select>option',
			'optgroup+': 'optgroup>option',
			'optg+': 'optgroup>option'

		},
		
		'element_types': {
			'empty': 'area,base,basefont,br,col,frame,hr,img,input,isindex,link,meta,param,embed,keygen,command',
			'block_level': 'address,applet,blockquote,button,center,dd,del,dir,div,dl,dt,fieldset,form,frameset,hr,iframe,ins,isindex,li,link,map,menu,noframes,noscript,object,ol,p,pre,script,table,tbody,td,tfoot,th,thead,tr,ul,h1,h2,h3,h4,h5,h6',
			'inline_level': 'a,abbr,acronym,applet,b,basefont,bdo,big,br,button,cite,code,del,dfn,em,font,i,iframe,img,input,ins,kbd,label,map,object,q,s,samp,select,small,span,strike,strong,sub,sup,textarea,tt,u,var'
		}
	},
	
	'xml': {
		'extends': 'html',
		'filters': 'html'
	},
	
	'xsl': {
		'extends': 'html',
		'filters': 'html, xsl',
		'abbreviations': {
			'tm': '<xsl:template match="" mode="">',
			'tmatch': 'tm',
			'tn': '<xsl:template name="">',
			'tname': 'tn',
			'xsl:when': '<xsl:when test="">',
			'wh': 'xsl:when',
			'var': '<xsl:variable name="">',
			'vare': '<xsl:variable name="" select=""/>',
			'if': '<xsl:if test="">',
			'call': '<xsl:call-template name=""/>',
			'attr': '<xsl:attribute name="">',
			'wp': '<xsl:with-param name="" select=""/>',
			'par': '<xsl:param name="" select=""/>',
			'val': '<xsl:value-of select=""/>',
			'co': '<xsl:copy-of select=""/>',
			'each': '<xsl:for-each select="">',
			'for': 'each',
			'ap': '<xsl:apply-templates select="" mode=""/>',
			
			//expandos
			'choose+': 'xsl:choose>xsl:when+xsl:otherwise'
		}
	},
	
	'haml': {
		'filters': 'haml',
		'extends': 'html'
	}
};/**
 * Parsed resources (snippets, abbreviations, variables, etc.) for Zen Coding.
 * Contains convenient method to get access for snippets with respect of 
 * inheritance. Also provides abilitity to store data in different vocabularies
 * ('system' and 'user') for fast and safe resurce update
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
var zen_resources = (function(){
	var TYPE_ABBREVIATION = 'zen-tag',
		TYPE_EXPANDO = 'zen-expando',
	
		/** Reference to another abbreviation or tag */
		TYPE_REFERENCE = 'zen-reference',
		
		VOC_SYSTEM = 'system',
		VOC_USER = 'user',
		
		/** Regular expression for XML tag matching */
		re_tag = /^<(\w+\:?[\w\-]*)((?:\s+[\w\:\-]+\s*=\s*(['"]).*?\3)*)\s*(\/?)>/,
		re_attrs = /([\w\-]+)\s*=\s*(['"])(.*?)\2/g,
		
		system_settings = {},
		user_settings = {};
		
	/**
	 * Trim whitespace from string
	 * @param {String} text
	 * @return {String}
	 */
	function trim(text) {
		return (text || "").replace( /^\s+|\s+$/g, "" );
	}
		
	/**
	 * Check if specified resource is parsed by Zen Coding
	 * @param {Object} obj
	 * @return {Boolean}
	 */
	function isParsed(obj) {
		return obj && obj.__zen_parsed__;
	}
	
	/**
	 * Marks object as parsed by Zen Coding
	 * @param {Object}
	 */
	function setParsed(obj) {
		obj.__zen_parsed__ = true;
	}
	
	/**
	 * Returns resource vocabulary by its name
	 * @param {String} name Vocabulary name ('system' or 'user')
	 */
	function getVocabulary(name) {
		return name == VOC_SYSTEM ? system_settings : user_settings;
	}
		
	/**
	 * Helper function that transforms string into hash
	 * @return {Object}
	 */
	function stringToHash(str){
		var obj = {}, items = str.split(",");
		for ( var i = 0; i < items.length; i++ )
			obj[ items[i] ] = true;
		return obj;
	}
	
	/**
	 * Creates resource inheritance chain for lookups
	 * @param {String} vocabulary Resource vocabulary
	 * @param {String} syntax Syntax name
	 * @param {String} name Resource name
	 * @return {Array}
	 */
	function createResourceChain(vocabulary, syntax, name) {
		var voc = getVocabulary(vocabulary),
			result = [],
			resource;
		
		if (voc && syntax in voc) {
			resource = voc[syntax];
			if (name in resource)
				result.push(resource[name]);
		}
		
		// get inheritance definition
		// in case of user-defined vocabulary, resource dependency
		// may be defined in system vocabulary only, so we have to correctly
		// handle this case
		var chain_source;
		if (resource && 'extends' in resource)
			chain_source = resource;
		else if (vocabulary == VOC_USER && syntax in system_settings 
			&& 'extends' in system_settings[syntax] )
			chain_source = system_settings[syntax];
			
		if (chain_source) {
			if (!isParsed(chain_source['extends'])) {
				var ar = chain_source['extends'].split(',');
				for (var i = 0; i < ar.length; i++) 
					ar[i] = trim(ar[i]);
				chain_source['extends'] = ar;
				setParsed(chain_source['extends']);
			}
			
			// find resource in ancestors
			for (var i = 0; i < chain_source['extends'].length; i++) {
				var type = chain_source['extends'][i];
				if (voc[type] && voc[type][name])
					result.push(voc[type][name]);
			}
		}
		
		return result;
	}
	
	/**
	 * Get resource collection from settings vocbulary for specified syntax. 
	 * It follows inheritance chain if resource wasn't directly found in
	 * syntax settings
	 * @param {String} vocabulary Resource vocabulary
	 * @param {String} syntax Syntax name
	 * @param {String} name Resource name
	 */
	function getSubset(vocabulary, syntax, name) {
		var chain = createResourceChain(vocabulary, syntax, name);
		return chain[0];
	}
	
	/**
	 * Returns parsed item located in specified vocabulary by its syntax and
	 * name
	 * @param {String} vocabulary Resource vocabulary
	 * @param {String} syntax Syntax name
	 * @param {String} name Resource name ('abbreviation', 'snippet')
	 * @param {String} item Abbreviation or snippet name
	 * @return {Object|null}
	 */
	function getParsedItem(vocabulary, syntax, name, item) {
		var chain = createResourceChain(vocabulary, syntax, name),
			result = null,
			res;
			
		for (var i = 0, il = chain.length; i < il; i++) {
			res = chain[i];
			if (item in res) {
				if (name == 'abbreviations' && !isParsed(res[item])) {
					// parse abbreviation
					var value = res[item];
					res[item] = parseAbbreviation(item, value);
					res[item].__ref = value;
					setParsed(res[item]);
				}
				
				result = res[item];
				break;
			}
		}
		
		return result;
	}
	
	/**
	 * Unified object for parsed data
	 */
	function entry(type, key, value) {
		return {
			type: type,
			key: key,
			value: value
		};
	}
	
	/**
	 * Make expando from string
	 * @param {String} key
	 * @param {String} value
	 * @return {Object}
	 */
	function makeExpando(key, value) {
		return entry(TYPE_EXPANDO, key, value);
	}
	
	/**
	 * Make abbreviation from string
	 * @param {String} key Abbreviation key
	 * @param {String} tag_name Expanded element's tag name
	 * @param {String} attrs Expanded element's attributes
	 * @param {Boolean} is_empty Is expanded element empty or not
	 * @return {Object}
	 */
	function makeAbbreviation(key, tag_name, attrs, is_empty) {
		var result = {
			name: tag_name,
			is_empty: !!is_empty
		};
		
		if (attrs) {
			var m;
			result.attributes = [];
			while (m = re_attrs.exec(attrs)) {
				result.attributes.push({
					name: m[1],
					value: m[3]
				});
			}
		}
		
		return entry(TYPE_ABBREVIATION, key, result);
	}
	
	/**
	 * Parses single abbreviation
	 * @param {String} key Abbreviation name
	 * @param {String} value = Abbreviation value
	 * @return {Object}
	 */
	function parseAbbreviation(key, value) {
		key = trim(key);
		var m;
		if (key.substr(-1) == '+') {
			// this is expando, leave 'value' as is
			return makeExpando(key, value);
		} else if (m = re_tag.exec(value)) {
			return makeAbbreviation(key, m[1], m[2], m[4] == '/');
		} else {
			// assume it's reference to another abbreviation
			return entry(TYPE_REFERENCE, key, value);
		}
	}
	
	return {
		/**
		 * Sets new unparsed data for specified settings vocabulary
		 * @param {Object} data
		 * @param {String} type Vocabulary type ('system' or 'user')
		 */
		setVocabulary: function(data, type) {
			if (type == VOC_SYSTEM)
				system_settings = data;
			else
				user_settings = data;
		},
		
		/**
		 * Get data from specified vocabulary. Can contain parsed entities
		 * @param {String} name Vocabulary type ('system' or 'user')
		 * @return {Object}
		 */
		getVocabulary: getVocabulary,
		
		/**
		 * Returns resource value from data set with respect of inheritance
		 * @param {String} syntax Resource syntax (html, css, ...)
		 * @param {String} name Resource name ('snippets' or 'abbreviation')
		 * @param {String} abbr Abbreviation name
		 * @return {Object|null}
		 */
		getResource: function(syntax, name, item) {
			return getParsedItem(VOC_USER, syntax, name, item) 
				|| getParsedItem(VOC_SYSTEM, syntax, name, item);
		},
		
		/**
		 * Returns abbreviation value from data set
		 * @param {String} type Resource type (html, css, ...)
		 * @param {String} name Abbreviation name
		 * @return {Object|null}
		 */
		getAbbreviation: function(type, name) {
			return this.getResource(type, 'abbreviations', name) 
				|| this.getResource(type, 'abbreviations', name.replace(/\-/g, ':'));
		},
		
		/**
		 * Returns snippet value from data set
		 * @param {String} type Resource type (html, css, ...)
		 * @param {String} name Snippet name
		 * @return {Object|null}
		 */
		getSnippet: function(type, name) {
			return this.getResource(type, 'snippets', name)
				|| this.getResource(type, 'snippets', name.replace(/\-/g, ':'));
		},
		
		/**
		 * Returns variable value
		 * @return {String}
		 */
		getVariable: function(name) {
			return getSubset(VOC_USER, 'variables', name) 
				|| getSubset(VOC_SYSTEM, 'variables', name);
		},
		
		/**
		 * Returns resource subset from settings vocabulary
		 * @param {String} syntax Syntax name
		 * @param {String} name Resource name
		 * @return {Object}
		 */
		getSubset: function(syntax, name) {
			return getSubset(VOC_USER, syntax, name) 
				|| getSubset(VOC_SYSTEM, syntax, name);
		},
		
		/**
		 * Check if specified item exists in specified resource collection
		 * (like 'empty', 'block_level')
		 * @param {String} syntax 
		 * @param {String} collection Collection name
		 * @param {String} item Item name
		 */
		isItemInCollection: function(syntax, collection, item) {
			return item in this.getElementsCollection(getVocabulary(VOC_USER)[syntax], collection)
				|| item in this.getElementsCollection(getVocabulary(VOC_SYSTEM)[syntax], collection);
		},
		
		/**
		 * Returns specified elements collection (like 'empty', 'block_level') from
		 * <code>resource</code>. If collections wasn't found, returns empty object
		 * @param {Object} resource
		 * @param {String} type
		 * @return {Object}
		 */
		getElementsCollection: function(resource, type) {
			if (resource && resource.element_types) {
				// if it's not parsed yet – do it
				var res = resource.element_types;
				if (!isParsed(res)) {
					for (var p in res) 
						res[p] = stringToHash(res[p]);
						
					setParsed(res);
				}
				return res[type] || {}
			}
			else
				return {};
		},
		
		/**
		 * Check if there are resources for specified syntax
		 * @param {String} syntax
		 * @return {Boolean}
		 */
		hasSyntax: function(syntax) {
			return syntax in getVocabulary(VOC_USER) 
				|| syntax in getVocabulary(VOC_SYSTEM);
		}
	}
})();

try {
	zen_resources.setVocabulary(zen_settings, 'system');
	zen_resources.setVocabulary(my_zen_settings, 'user');
} catch(e) {}/**
 * Class that parses abbreviation into tree with respect of groups, attributes
 * and text nodes
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 * 
 * @include "zen_coding.js"
 */
var zen_parser = (function(){
	
	var re_valid_name = /^[\w\d\-_\$\:@!]+\+?$/i;
	
	/**
	 * @class
	 */
	function TreeNode(parent) {
		this.abbreviation = '';
		/** @type {TreeNode} */
		this.parent = null;
		this.children = [];
		this.count = 1;
		this.name = null;
		this.text = null;
		this.attributes = [];
		this.is_repeating = false;
		this.has_implict_name = false;
	}
	
	TreeNode.prototype = {
		/**
		 * Adds passed or creates new child
		 * @param {TreeNode} [child]
		 * @return {TreeNode}
		 */
		addChild: function(child) {
			child = child || new TreeNode;
			child.parent = this;
			this.children.push(child);
			return child;
		},
		
		/**
		 * Replace current node in parent's child list with another node
		 * @param {TreeNode} node
		 */
		replace: function(node) {
			if (this.parent) {
				var children = this.parent.children;
				for (var i = 0, il = children.length; i < il; i++) {
					if (children[i] === this) {
						children[i] = node;
						this.parent = null;
						return;
					}
				}
			}
		},
		
		/**
		 * Sets abbreviation that belongs to current node
		 * @param {String} abbr
		 */
		setAbbreviation: function(abbr) {
			this.abbreviation = abbr;
			var m = abbr.match(/\*(\d+)?$/);
			if (m) {
				this.count = parseInt(m[1] || 1, 10);
				this.is_repeating = !m[1];
				abbr = abbr.substr(0, abbr.length - m[0].length);
			}
			
			if (abbr) {
				var name_text = splitExpression(abbr);
				var name = name_text[0];
				if (name_text.length == 2)
					this.text = name_text[1];
					
				if (name) {
					var attr_result = parseAttributes(name);
					this.name = attr_result[0] || 'div';
					this.has_implict_name = !attr_result[0];
					this.attributes = attr_result[1];
				}
			}
			
			// validate name
			if (this.name && !re_valid_name.test(this.name)) {
				throw new Error('InvalidAbbreviation');
			}
		},
		
		/**
		 * @return {String}
		 */
		getAbbreviation: function() {
			return this.expr;
		},
		
		/**
		 * Dump current tree node into a foramtted string
		 * @return {String}
		 */
		toString: function(level) {
			level = level || 0;
			var output = '(empty)';
			if (this.abbreviation) {
				output = '';
				if (this.name)
					output = this.name;
					
				if (this.text !== null)
					output += (output ? ' ' : '') + '{text: "' + this.text + '"}';
					
				if (this.attributes.length) {
					var attrs = [];
					for (var i = 0, il = this.attributes.length; i < il; i++) {
						attrs.push(this.attributes[i].name + '="' + this.attributes[i].value + '"'); 
					}
					output += ' [' + attrs.join(', ') + ']';
				}
			}
			var result = zen_coding.repeatString('-', level)
				+ output 
				+ '\n';
			for (var i = 0, il = this.children.length; i < il; i++) {
				result += this.children[i].toString(level + 1);
			}
			
			return result;
		},
		
		/**
		 * Check if current node contains children with empty <code>expr</code>
		 * property
		 * @return {Boolean}
		 */
		hasEmptyChildren: function() {
			for (var i = 0, il = this.children.length; i < il; i++) {
				if (this.children[i].isEmpty())
					return true;
			}
			
			return false;
		},
		
		/**
		 * @return {Boolean}
		 */
		isEmpty: function() {
			return !this.abbreviation;
		},
		
		/**
		 * Check if current node is a text-only node
		 * @return {Boolean}
		 */
		isTextNode: function() {
			return !this.name && this.text;
		}
	};
	
	/**
	 * Check if character is numeric
	 * @requires {Stirng} ch
	 * @return {Boolean}
	 */
	function isNumeric(ch) {
		if (typeof(ch) == 'string')
			ch = ch.charCodeAt(0);
			
		return (ch && ch > 47 && ch < 58);
	}
	
	/**
	 * Optimizes tree node: replaces empty nodes with their children
	 * @param {TreeNode} node
	 * @return {TreeNode}
	 */
	function squash(node) {
		for (var i = node.children.length - 1; i >=0; i--) {
			/** @type {TreeNode} */
			var n = node.children[i];
			if (n.isEmpty()) {
				var args = [i, 1];
				for (var j = 0, jl = n.children.length; j < jl; j++) {
					args.push(n.children[j]);
				}
				
				Array.prototype.splice.apply(node.children, args);
			}
		}
		
		return node;
	}
	
	/**
	 * Trim whitespace from string
	 * @param {String} text
	 * @return {String}
	 */
	function trim(text) {
		return (text || "").replace( /^\s+|\s+$/g, "" );
	}
	
	/**
	 * Get word, starting at <code>ix</code> character of <code>str</code>
	 */
	function getWord(ix, str) {
		var m = str.substring(ix).match(/^[\w\-:\$]+/);
		return m ? m[0] : '';
	}
	
	/**
	 * Extract attributes and their values from attribute set 
	 * @param {String} attr_set
	 */
	function extractAttributes(attr_set) {
		attr_set = trim(attr_set);
		var loop_count = 100, // endless loop protection
			re_string = /^(["'])((?:(?!\1)[^\\]|\\.)*)\1/,
			result = [],
			attr;
			
		while (attr_set && loop_count--) {
			var attr_name = getWord(0, attr_set);
			attr = null;
			if (attr_name) {
				attr = {name: attr_name, value: ''};
//				result[attr_name] = '';
				// let's see if attribute has value
				var ch = attr_set.charAt(attr_name.length);
				switch (ch) {
					case '=':
						var ch2 = attr_set.charAt(attr_name.length + 1);
						if (ch2 == '"' || ch2 == "'") {
							// we have a quoted string
							var m = attr_set.substring(attr_name.length + 1).match(re_string);
							if (m) {
								attr.value = m[2];
								attr_set = trim(attr_set.substring(attr_name.length + m[0].length + 1));
							} else {
								// something wrong, break loop
								attr_set = '';
							}
						} else {
							// unquoted string
							var m = attr_set.substring(attr_name.length + 1).match(/(.+?)(\s|$)/);
							if (m) {
								attr.value = m[1];
								attr_set = trim(attr_set.substring(attr_name.length + m[1].length + 1));
							} else {
								// something wrong, break loop
								attr_set = '';
							}
						}
						break;
					default:
						attr_set = trim(attr_set.substring(attr_name.length));
						break;
				}
			} else {
				// something wrong, can't extract attribute name
				break;
			}
			
			if (attr) result.push(attr);
		}
		return result;
	}
	
	/**
	 * Parses tag attributes extracted from abbreviation
	 * @param {String} str
	 */
	function parseAttributes(str) {
		/*
		 * Example of incoming data:
		 * #header
		 * .some.data
		 * .some.data#header
		 * [attr]
		 * #item[attr=Hello other="World"].class
		 */
		var result = [],
			name = '',
			collect_name = true,
			class_name,
			char_map = {'#': 'id', '.': 'class'};
		
		// walk char-by-char
		var i = 0,
			il = str.length,
			val;
			
		while (i < il) {
			var ch = str.charAt(i);
			switch (ch) {
				case '#': // id
					val = getWord(i, str.substring(1));
					result.push({name: char_map[ch], value: val});
					i += val.length + 1;
					collect_name = false;
					break;
				case '.': // class
					val = getWord(i, str.substring(1));
					if (!class_name) {
						// remember object pointer for value modification
						class_name = {name: char_map[ch], value: ''};
						result.push(class_name);
					}
					
					class_name.value += ((class_name.value) ? ' ' : '') + val;
					i += val.length + 1;
					collect_name = false;
					break;
				case '[': //begin attribute set
					// search for end of set
					var end_ix = str.indexOf(']', i);
					if (end_ix == -1) {
						// invalid attribute set, stop searching
						i = str.length;
					} else {
						var attrs = extractAttributes(str.substring(i + 1, end_ix));
						for (var j = 0, jl = attrs.length; j < jl; j++) {
							result.push(attrs[j]);
						}
						i = end_ix;
					}
					collect_name = false;
					break;
				default:
					if (collect_name)
						name += ch;
					i++;
			}
		}
		
		return [name, result];
	}
	
	/**
	 * @param {TreeNode} node
	 * @return {TreeNode}
	 */
	function optimizeTree(node) {
		while (node.hasEmptyChildren())
			squash(node);
			
		for (var i = 0, il = node.children.length; i < il; i++) {
			optimizeTree(node.children[i]);
		}
		
		return node;
	}
	
	/**
	 * Split expression by node name and its content, if exists. E.g. if we pass
	 * <code>a{Text}</code> expression, it will be splitted into <code>a</code>
	 * and <code>Text</code>
	 * @param {String} expr
	 * @return {Array} Result with one or two elements (if expression contains
	 * text node)
	 */
	function splitExpression(expr) {
		// fast test on text node
		if (expr.indexOf('{') == -1)
			return [expr];
			
		var attr_lvl = 0,
			text_lvl = 0,
			brace_stack = [],
			i = 0,
			il = expr.length,
			ch;
			
		while (i < il) {
			ch = expr.charAt(i);
			switch (ch) {
				case '[':
					if (!text_lvl)
						attr_lvl++;
					break;
				case ']':
					if (!text_lvl)
						attr_lvl--;
					break;
				case '{':
					if (!attr_lvl) {
						text_lvl++;
						brace_stack.push(i);
					}
					break;
				case '}':
					if (!attr_lvl) {
						text_lvl--;
						var brace_start = brace_stack.pop();
						if (text_lvl === 0) {
							// found braces bounds
							return [
								expr.substring(0, brace_start),
								expr.substring(brace_start + 1, i)
							];
						}
					}
					break;
			}
			i++;
		}
		
		// if we are here, then no valid text node found
		return [expr];
	}
	
	
	return {
		/**
		 * Parses abbreviation into tree with respect of groups, 
		 * text nodes and attributes. Each node of the tree is a single 
		 * abbreviation. Tree represents actual structure of the outputted 
		 * result
		 * @param {String} abbr Abbreviation to parse
		 * @return {TreeNode}
		 */
		parse: function(abbr) {
			var root = new TreeNode,
				context = root.addChild(),
				i = 0,
				il = abbr.length,
				text_lvl = 0,
				attr_lvl = 0,
				group_lvl = 0,
				group_stack = [root],
				ch, prev_ch,
				token = '';
				
			group_stack.last = function() {
				return this[this.length - 1];
			};
			
			var dumpToken = function() {
				if (token)
					context.setAbbreviation(token);
				token = '';
			};
				
			while (i < il) {
				ch = abbr.charAt(i);
				prev_ch = i ? abbr.charAt(i - 1) : '';
				switch (ch) {
					case '{':
						if (!attr_lvl)
							text_lvl++;
						token += ch;
						break;
					case '}':
						if (!attr_lvl)
							text_lvl--;
						token += ch;
						break;
					case '[':
						if (!text_lvl)
							attr_lvl++;
						token += ch;
						break;
					case ']':
						if (!text_lvl)
							attr_lvl--;
						token += ch;
						break;
					case '(':
						if (!text_lvl && !attr_lvl) {
							// beginning of the new group
							dumpToken();
							
							if (prev_ch != '+' && prev_ch != '>') {
								// previous char is not an operator, assume it's
								// a sibling
								context = context.parent.addChild();
							}
							
							group_stack.push(context);
							context = context.addChild();
						} else {
							token += ch;
						}
						break;
					case ')':
						if (!text_lvl && !attr_lvl) {
							// end of the group, pop stack
							dumpToken();
							context = group_stack.pop();
							
							if (i < il - 1 && abbr.charAt(i + 1) == '*') {
								// group multiplication
								var group_mul = '', n_ch;
								for (var j = i + 2; j < il; j++) {
									n_ch = abbr.charAt(j);
									if (isNumeric(n_ch))
										group_mul += n_ch;
									else 
										break;
								}
								
								i += group_mul.length + 1;
								group_mul = parseInt(group_mul || 1, 10);
								while (1 < group_mul--)
									context.parent.addChild(context);
//									last_parent.addChild(cur_item);
							}
							
						} else {
							token += ch;
						}
						break;
					case '+': // sibling operator
						if (!text_lvl && !attr_lvl && i != il - 1 /* expando? */) {
							dumpToken();
							context = context.parent.addChild();
						} else {
							token += ch;
						}
						break;
					case '>': // child operator
						if (!text_lvl && !attr_lvl) {
							dumpToken();
							context = context.addChild();
						} else {
							token += ch;
						}
						break;
					default:
						token += ch;
				}
				
				i++;
			}
			// put the final token
			dumpToken();
			
			return optimizeTree(root);
		},
		
		TreeNode: TreeNode,
		optimizeTree: optimizeTree
	}
})();/**
 * Core library that do all Zen Coding magic
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 * @include "settings.js"
 * @include "zen_parser.js"
 * @include "zen_resources.js"
 */
var zen_coding = (function(){
	var re_tag = /<\/?[\w:\-]+(?:\s+[\w\-:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*\s*(\/?)>$/,
	
		caret_placeholder = '{%::zen-caret::%}',
		newline = '\n',
		
		/** List of registered filters */
		filters = {},
		/** Filters that will be applied for unknown syntax */
		basic_filters = 'html',
		
		profiles = {},
		default_profile = {
			tag_case: 'lower',
			attr_case: 'lower',
			attr_quotes: 'double',
			
			// each tag on new line
			tag_nl: 'decide',
			
			place_cursor: true,
			
			// indent tags
			indent: true,
			
			// how many inline elements should be to force line break 
			// (set to 0 to disable)
			inline_break: 3,
			
			// use self-closing style for writing empty elements, e.g. <br /> or <br>
			self_closing_tag: 'xhtml',
			
			// Profile-level output filters, re-defines syntax filters 
			filters: ''
		};
	
	function isNumeric(ch) {
		if (typeof(ch) == 'string')
			ch = ch.charCodeAt(0);
			
		return (ch && ch > 47 && ch < 58);
	}
	
	/**
	 * Проверяет, является ли символ допустимым в аббревиатуре
	 * @param {String} ch
	 * @return {Boolean}
	 */
	function isAllowedChar(ch) {
		ch = String(ch); // convert Java object to JS
		var char_code = ch.charCodeAt(0),
			special_chars = '#.>+*:$-_!@[]()|';
		
		return (char_code > 64 && char_code < 91)       // uppercase letter
				|| (char_code > 96 && char_code < 123)  // lowercase letter
				|| isNumeric(ch)                        // number
				|| special_chars.indexOf(ch) != -1;     // special character
	}
	
	/**
	 * Возвращает символ перевода строки, используемый в редакторе
	 * @return {String}
	 */
	function getNewline() {
		return zen_coding.getNewline();
	}
	
	/**
	 * Returns caret placeholder
	 * @return {String}
	 */
	function getCaretPlaceholder() {
		return (typeof(caret_placeholder) != 'string') 
			? caret_placeholder()
			: caret_placeholder
	}
	
	/**
	 * Split text into lines. Set <code>remove_empty</code> to true to filter
	 * empty lines
	 * @param {String} text
	 * @param {Boolean} [remove_empty]
	 * @return {Array}
	 */
	function splitByLines(text, remove_empty) {
		// IE fails to split string by regexp, 
		// need to normalize newlines first
		// Also, Mozilla's Rhiho JS engine has a wierd newline bug
		var nl = getNewline();
		var lines = (text || '')
			.replace(/\r\n/g, '\n')
			.replace(/\n\r/g, '\n')
			.replace(/\r/g, '\n')
			.replace(/\n/g, nl)
			.split(nl);
		
		if (remove_empty) {
			for (var i = lines.length; i >= 0; i--) {
				if (!trim(lines[i]))
					lines.splice(i, 1);
			}
		}
		
		return lines;
	}
	
	/**
	 * Trim whitespace from string
	 * @param {String} text
	 * @return {String}
	 */
	function trim(text) {
		return (text || "").replace( /^\s+|\s+$/g, "" );
	}
	
	function createProfile(options) {
		var result = {};
		for (var p in default_profile)
			result[p] = (p in options) ? options[p] : default_profile[p];
		
		return result;
	}
	
	function setupProfile(name, options) {
		profiles[name.toLowerCase()] = createProfile(options || {});
	}
	
	/**
	 * Repeats string <code>how_many</code> times
	 * @param {String} str
	 * @param {Number} how_many
	 * @return {String}
	 */
	function repeatString(str, how_many) {
		var result = '';
		for (var i = 0; i < how_many; i++) 
			result += str;
			
		return result;
	}
	
	/**
	 * Indents text with padding
	 * @param {String} text Text to indent
	 * @param {String|Number} pad Padding size (number) or padding itself (string)
	 * @return {String}
	 */
	function padString(text, pad) {
		var pad_str = (typeof(pad) == 'number') 
				? repeatString(getIndentation(), pad) 
				: pad, 
			result = '';
		
		var lines = splitByLines(text),
			nl = getNewline();
			
		result += lines[0];
		for (var j = 1; j < lines.length; j++) 
			result += nl + pad_str + lines[j];
			
		return result;
	}
	
	/**
	 * Class inheritance method
	 * @param {Function} derived Derived class
	 * @param {Function} from Base class
	 */
	function inherit(derived, from) {
		var Inheritance = function(){};
	
		Inheritance.prototype = from.prototype;
	
		derived.prototype = new Inheritance();
		derived.prototype.constructor = derived;
		derived.baseConstructor = from;
		derived.superClass = from.prototype;
	};
	
	/**
	 * Check if passed abbreviation is snippet
	 * @param {String} abbr
	 * @param {String} type
	 * @return {Boolean}
	 */
	function isShippet(abbr, type) {
		return getSnippet(type, filterNodeName(abbr)) ? true : false;
	}
	
	/**
	 * Test if passed string ends with XHTML tag. This method is used for testing
	 * '>' character: it belongs to tag or it's a part of abbreviation? 
	 * @param {String} str
	 * @return {Boolean}
	 */
	function isEndsWithTag(str) {
		return re_tag.test(str);
	}
	
	/**
	 * Replace variables like ${var} in string
	 * @param {String} str
	 * @param {Object|Function} [vars] Variable set (default is <code>zen_settings.variables</code>) 
	 * @return {String}
	 */
	function replaceVariables(str, vars) {
		var callback;
		
		if (typeof vars == 'function')
			callback = vars;
		else if (vars)
			callback = function(str, p1) {
				return (p1 in vars) ? vars[p1] : str;
			};
		else 
			callback = function(str, p1) {
				var v = getVariable(p1);
				return (v !== null && typeof v != 'undefined') ? v : str;
			}
		
		return str.replace(/\$\{([\w\-]+)\}/g, callback);
	}
	
	/**
	 * Removes any unnecessary characters from node name
	 * @param {String} name
	 * @return {String}
	 */
	function filterNodeName(name) {
		return (name || '').replace(/(.+)\!$/, '$1');
	}
	
	/**
	 * Test if text contains output placeholder $#
	 * @param {String} text
	 * @return {Boolean}
	 */
	function hasOutputPlaceholder(/* String */ text) {
		for (var i = 0, il = text.length; i < il; i++) {
			var ch = text.charAt(i);
			if (ch == '\\') { // escaped char
				i++;
				continue;
			} else if (ch == '$' && text.charAt(i + 1) == '#') {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Tag
	 * @class
	 * @param {zen_parser.TreeNode} node Parsed tree node
	 * @param {String} type Tag type (html, xml)
	 */
	function Tag(node, type) {
		type = type || 'html';
		
		var abbr = null;
		if (node.name) {
			abbr = getAbbreviation(type, filterNodeName(node.name));
			if (abbr && abbr.type == 'zen-reference')
				abbr = getAbbreviation(type, filterNodeName(abbr.value));
		}
		
		this.name = (abbr) ? abbr.value.name : node.name;
		this.real_name = node.name;
		this.count = node.count || 1;
		this._abbr = abbr;
		this.syntax = type;
		this._content = '';
		this._paste_content = '';
		this.repeat_by_lines = node.is_repeating;
		this.is_repeating = node && node.count > 1;
		this.parent = null;
		this.has_implicit_name = node.has_implict_name;
		
		this.setContent(node.text);
		
		// add default attributes
		if (this._abbr)
			this.copyAttributes(this._abbr.value);
		
		this.copyAttributes(node);
	}
	
	Tag.prototype = {
		/**
		 * Adds new child tag to current one
		 * @param {Tag} tag
		 */
		addChild: function(tag) {
			if (!this.children)
				this.children = [];
				
			tag.parent = this;
			this.children.push(tag);
		},
		
		/**
		 * Adds new attribute
		 * @param {String} name Attribute's name
		 * @param {String} value Attribute's value
		 */
		addAttribute: function(name, value) {
			if (!this.attributes)
				this.attributes = [];
				
			if (!this._attr_hash)
				this._attr_hash = {};
			
			// escape pipe (caret) character with internal placeholder
			value = replaceUnescapedSymbol(value, '|', getCaretPlaceholder());
			
			var a;
			if (name in this._attr_hash) {
				// attribute already exists, decide what to do
				a = this._attr_hash[name];
				if (name == 'class') {
					// 'class' is a magic attribute
					a.value += ((a.value) ? ' ' : '') + value;
				} else {
					a.value = value;
				}
			} else {
				a = {name: name, value: value};
				this._attr_hash[name] = a
				this.attributes.push(a);
			}
		},
		
		/**
		 * Copy attributes from parsed node
		 */
		copyAttributes: function(node) {
			if (node && node.attributes)
				for (var i = 0, il = node.attributes.length; i < il; i++) {
					var attr = node.attributes[i];
					this.addAttribute(attr.name, attr.value);
				}
		},
		
		/**
		 * This function tests if current tags' content contains xHTML tags. 
		 * This function is mostly used for output formatting
		 */
		hasTagsInContent: function() {
			return this.getContent() && re_tag.test(this.getContent());
		},
		
		/**
		 * Set textual content for tag
		 * @param {String} str Tag's content
		 */
		setContent: function(str) {
			this._content = replaceUnescapedSymbol(str || '', '|', getCaretPlaceholder());
		},
		
		/**
		 * Returns tag's textual content
		 * @return {String}
		 */
		getContent: function() {
			return this._content || '';
		},
		
		/**
		 * Set content that should be pasted to the output
		 * @param {String} val
		 */
		setPasteContent: function(val) {
			this._paste_content = zen_coding.escapeText(val);
		},
		
		/**
		 * Get content that should be pasted to the output
		 * @return {String}
		 */
		getPasteContent: function() {
			return this._paste_content;
		},
		
		/**
		 * Search for deepest and latest child of current element
		 * @return {Tag|null} Returns null if there's no children
		 */
		findDeepestChild: function() {
			if (!this.children || !this.children.length)
				return null;
				
			var deepest_child = this;
			while (true) {
				deepest_child = deepest_child.children[ deepest_child.children.length - 1 ];
				if (!deepest_child.children || !deepest_child.children.length)
					break;
			}
			
			return deepest_child;
		}
	};
	
	/**
	 * Snippet
	 * @param {zen_parser.TreeNode} node
	 * @param {String} type Tag type (html, xml)
	 */
	function Snippet(node, type) {
		/** @type {String} */
		this.name = filterNodeName(node.name);
		this.real_name = node.name;
		this.count = node.count;
		this.children = [];
		this._content = node.text || '';
		this.repeat_by_lines = node.is_repeating;
		this.is_repeating = node && node.count > 1;
		this.attributes = [];
		this.value = replaceUnescapedSymbol(getSnippet(type, this.name), '|', getCaretPlaceholder());
		this.parent = null;
		this.syntax = type;
		
		this.addAttribute('id', getCaretPlaceholder());
		this.addAttribute('class', getCaretPlaceholder());
		this.copyAttributes(node);
	}
	
	inherit(Snippet, Tag);
	
	/**
	 * Returns abbreviation value from data set
	 * @param {String} type Resource type (html, css, ...)
	 * @param {String} abbr Abbreviation name
	 * @return {Object|null}
	 */
	function getAbbreviation(type, abbr) {
		return zen_resources.getAbbreviation(type, abbr);
	}
	
	/**
	 * Returns snippet value from data set
	 * @param {String} type Resource type (html, css, ...)
	 * @param {String} snippet_name Snippet name
	 * @return {Object|null}
	 */
	function getSnippet(type, snippet_name) {
		return zen_resources.getSnippet(type, snippet_name);
	}
	
	/**
	 * Returns variable value
	 * @return {String}
	 */
	function getVariable(name) {
		return zen_resources.getVariable(name);
	}
	
	/**
	 * Returns indentation string
	 * @return {String}
	 */
	function getIndentation() {
		return getVariable('indentation');
	}
	
	/**
	 * @class
	 * Creates simplified tag from Zen Coding tag
	 * @param {Tag} tag
	 */
	function ZenNode(tag) {
		this.type = (tag instanceof Snippet) ? 'snippet' : 'tag';
		this.name = tag.name;
		this.real_name = tag.real_name;
		this.children = [];
		this.counter = 1;
		this.is_repeating = tag.is_repeating;
		this.repeat_by_lines = tag.repeat_by_lines;
		this.has_implicit_name = this.type == 'tag' && tag.has_implicit_name;
		
		// create deep copy of attribute list so we can change
		// their values in runtime without affecting other nodes
		// created from the same tag
		this.attributes = [];
		if (tag.attributes) {
			for (var i = 0, il = tag.attributes.length; i < il; i++) {
				var a =  tag.attributes[i];
				this.attributes.push({
					name: a.name,
					value: a.value
				});
			}
		}
		
		/** @type {Tag} Source element from which current tag was created */
		this.source = tag;
		
		// relations
		/** @type {ZenNode} */
		this.parent = null;
		/** @type {ZenNode} */
		this.nextSibling = null;
		/** @type {ZenNode} */
		this.previousSibling = null;
		
		// output params
		this.start = '';
		this.end = '';
		this.content = tag.getContent() || '';
		this.padding = '';
	}
	
	ZenNode.prototype = {
		/**
		 * @type {ZenNode} tag
		 */
		addChild: function(tag) {
			tag.parent = this;
			
			// check for implicit name
			if (tag.has_implicit_name && this.isInline())
				tag.name = 'span';
			
			var last_child = this.children[this.children.length - 1];
			if (last_child) {
				tag.previousSibling = last_child;
				last_child.nextSibling = tag;
			}
			
			this.children.push(tag);
		},
		
		/**
		 * Get attribute's value.
		 * @param {String} name
		 * @return {String|null} Returns <code>null</code> if attribute wasn't found
		 */
		getAttribute: function(name) {
			name = name.toLowerCase();
			for (var i = 0, il = this.attributes.length; i < il; i++) {
				if (this.attributes[i].name.toLowerCase() == name)
					return this.attributes[i].value;
			}
			
			return null;
		},
		
		/**
		 * Test if current tag is unary (no closing tag)
		 * @return {Boolean}
		 */
		isUnary: function() {
			if (this.type == 'snippet')
				return false;
				
			return (this.source._abbr && this.source._abbr.value.is_empty) 
				|| zen_resources.isItemInCollection(this.source.syntax, 'empty', this.name);
		},
		
		/**
		 * Test if current tag is inline-level (like &lt;strong&gt;, &lt;img&gt;)
		 * @return {Boolean}
		 */
		isInline: function() {
			return this.type == 'text' 
				|| zen_resources.isItemInCollection(this.source.syntax, 'inline_level', this.name);
		},
		
		/**
		 * Test if current element is block-level
		 * @return {Boolean}
		 */
		isBlock: function() {
			return this.type == 'snippet' || !this.isInline();
		},
		
		/**
		 * This function tests if current tags' content contains xHTML tags. 
		 * This function is mostly used for output formatting
		 */
		hasTagsInContent: function() {
			return this.content && re_tag.test(this.content);
		},
		
		/**
		 * Check if tag has child elements
		 * @return {Boolean}
		 */
		hasChildren: function() {
			return !!this.children.length;
		},
		
		/**
		 * Test if current tag contains block-level children
		 * @return {Boolean}
		 */
		hasBlockChildren: function() {
			if (this.hasTagsInContent() && this.isBlock()) {
				return true;
			}
			
			for (var i = 0; i < this.children.length; i++) {
				if (this.children[i].isBlock())
					return true;
			}
			
			return false;
		},
		
		/**
		 * Search for deepest and latest child of current element
		 * @return {ZenNode|null} Returns <code>null</code> if there's no children
		 */
		findDeepestChild: function() {
			if (!this.children.length)
				return null;
				
			var deepest_child = this;
			while (true) {
				deepest_child = deepest_child.children[ deepest_child.children.length - 1 ];
				if (!deepest_child.children.length)
					break;
			}
			
			return deepest_child;
		},
		
		/**
		 * @return {String}
		 */
		toString: function() {
			var content = '';
			for (var i = 0, il = this.children.length; i < il; i++) {
				content += this.children[i].toString();
			}
			
			return this.start + this.content + content + this.end;
		},
		
		/**
		 * Test if current element contains output placeholder (aka $#)
		 * @return {Boolean}
		 */
		hasOutputPlaceholder: function() {
			if (hasOutputPlaceholder(this.content)) {
				return true;
			} else {
				// search inside attributes
				for (var i = 0, il = this.attributes.length; i < il; i++) {
					if (hasOutputPlaceholder(this.attributes[i].value))
						return true;
				}
			}
			
			return false;
		},
		
		/**
		 * Recursively search for elements with output placeholders (aka $#)
		 * inside current element (not included in result)
		 * @param {Array} _arr
		 * @return {Array} Array of elements with output placeholders.  
		 */
		findElementsWithOutputPlaceholder: function(_arr) {
			_arr = _arr || [];
			for (var i = 0, il = this.children.length; i < il; i++) {
				if (this.children[i].hasOutputPlaceholder()) {
					_arr.push(this.children[i]);
				}
				this.children[i].findElementsWithOutputPlaceholder(_arr);
			}
			return _arr;
		},
		
		/**
		 * Paste content in context of current node. Pasting is a special case
		 * of recursive adding content in node. 
		 * This function will try to find $# placeholder inside node's 
		 * attributes and text content and replace in with <code>text</code>.
		 * If it doesn't find $# placeholder, it will put <code>text</code>
		 * value as the deepest child content
		 * @param {String} text Text to paste
		 */
		pasteContent: function(text) {
			var symbol = '$#',
				r = [symbol, text],
				replace_fn = function() {return r;},
				/** @type {ZenNode[]} */
				items = [];
				
			if (this.hasOutputPlaceholder())
				items.push(this);
				
			items = items.concat(this.findElementsWithOutputPlaceholder());
			
			if (items.length) {
				for (var i = 0, il = items.length; i < il; i++) {
					/** @type {ZenNode} */
					var item = items[i];
					item.content = replaceUnescapedSymbol(item.content, symbol, replace_fn);
					for (var j = 0, jl = item.attributes.length; j < jl; j++) {
						var a = item.attributes[j];
						a.value = replaceUnescapedSymbol(a.value, symbol, replace_fn);
					}
				}
			} else {
				// no placeholders found, add content to the deepest child
				var child = this.findDeepestChild() || this;
				child.content += text;
			}
		}
	};
	
	/**
	 * Roll outs basic Zen Coding tree into simplified, DOM-like tree.
	 * The simplified tree, for example, represents each multiplied element 
	 * as a separate element sets with its own content, if exists.
	 * 
	 * The simplified tree element contains some meta info (tag name, attributes, 
	 * etc.) as well as output strings, which are exactly what will be outputted
	 * after expanding abbreviation. This tree is used for <i>filtering</i>:
	 * you can apply filters that will alter output strings to get desired look
	 * of expanded abbreviation.
	 * 
	 * @param {Tag} tree
	 * @param {ZenNode} [parent]
	 */
	function rolloutTree(tree, parent) {
		parent = parent || new ZenNode(tree);
		
		var how_many = 1,
			tag_content = '';
			
		if (tree.children) {
			for (var i = 0, il = tree.children.length; i < il; i++) {
				/** @type {Tag} */
				var child = tree.children[i];
				how_many = child.count;
				
				if (child.repeat_by_lines) {
					// it's a repeating element
					tag_content = splitByLines(child.getPasteContent(), true);
					how_many = Math.max(tag_content.length, 1);
				} else {
					tag_content = child.getPasteContent();
				}
				
				for (var j = 0; j < how_many; j++) {
					var tag = new ZenNode(child);
					parent.addChild(tag);
					tag.counter = j + 1;
					
					if (child.children && child.children.length)
						rolloutTree(child, tag);
						
					if (tag_content) {
						var text = (typeof(tag_content) == 'string') 
							? tag_content 
							: (tag_content[j] || '');
						tag.pasteContent(trim(text));
					}
				}
			}
		}
		
		return parent;
	}
	
	/**
	 * Runs filters on tree
	 * @param {ZenNode} tree
	 * @param {String|Object} profile
	 * @param {String[]|String} filter_list
	 * @return {ZenNode}
	 */
	function runFilters(tree, profile, filter_list) {
		profile = processProfile(profile);
		
		if (typeof(filter_list) == 'string')
			filter_list = filter_list.split(/[\|,]/g);
			
		for (var i = 0, il = filter_list.length; i < il; i++) {
			var name = trim(filter_list[i].toLowerCase());
			if (name && name in filters) {
				tree = filters[name](tree, profile);
			}
		}
		
		return tree;
	}
	
	/**
	 * Transforms abbreviation into a primary internal tree. This tree should'n 
	 * be used ouside of this scope
	 * @param {zen_parser.TreeNode} node Parsed tree node
	 * @param {String} [type] Document type (xsl, html, etc.)
	 * @return {Tag}
	 */
	function transformTreeNode(node, type) {
		type = type || 'html';
		if (node.isEmpty()) return null;
		
		return isShippet(node.name, type) 
				? new Snippet(node, type)
				: new Tag(node, type);
	}
	
	/**
	 * Process single tree node: expand it and its children 
	 * @param {zen_parser.TreeNode} node
	 * @param {String} type
	 * @param {Tag} parent
	 */
	function processParsedNode(node, type, parent) {
		var t_node = transformTreeNode(node, type);
		parent.addChild(t_node);
			
		// set repeating element to the topmost node
		var root = parent;
		while (root.parent)
			root = root.parent;
		
		root.last = t_node;
		if (t_node.repeat_by_lines)
			root.multiply_elem = t_node;
			
		// process child groups
		for (var j = 0, jl = node.children.length; j < jl; j++) {
			processParsedNode(node.children[j], type, t_node);
		}
	}
	
	/**
	 * Replaces expando nodes by its parsed content
	 * @param {zen_parser.TreeNode} node
	 * @param {String} type
	 */
	function replaceExpandos(node, type) {
		for (var i = 0, il = node.children.length; i < il; i++) {
			var n = node.children[i];
			if (!n.isEmpty() && !n.isTextNode() && n.name.indexOf('+') != -1) {
				// it's expando
				var a = getAbbreviation(type, n.name);
				if (a)
					node.children[i] = zen_parser.parse(a.value);
			}
			replaceExpandos(node.children[i], type);
		}
	}
	
	/**
	 * Replaces expandos and optimizes tree structure by removing empty nodes
	 * @param {zen_parser.TreeNode} tree
	 * @param {String} type
	 */
	function preprocessParsedTree(tree, type) {
		replaceExpandos(tree, type);
		return zen_parser.optimizeTree(tree);
	}
	
	/**
	 * Pad string with zeroes
	 * @param {String} str
	 * @param {Number} pad
	 */
	function zeroPadString(str, pad) {
		var padding = '', 
			il = str.length;
			
		while (pad > il++) padding += '0';
		return padding + str; 
	}
	
	/**
	 * Replaces unescaped symbols in <code>str</code>. For example, the '$' symbol
	 * will be replaced in 'item$count', but not in 'item\$count'.
	 * @param {String} str Original string
	 * @param {String} symbol Symbol to replace
	 * @param {String|Function} replace Symbol replacement
	 * @return {String}
	 */
	function replaceUnescapedSymbol(str, symbol, replace) {
		var i = 0,
			il = str.length,
			sl = symbol.length,
			match_count = 0;
			
		while (i < il) {
			if (str.charAt(i) == '\\') {
				// escaped symbol, skip next character
				i += sl + 1;
			} else if (str.substr(i, sl) == symbol) {
				// have match
				var cur_sl = sl;
				match_count++;
				var new_value = replace;
				if (typeof(replace) !== 'string') {
					var replace_data = replace(str, symbol, i, match_count);
					if (replace_data) {
						cur_sl = replace_data[0].length;
						new_value = replace_data[1];
					} else {
						new_value = false;
					}
				}
				
				if (new_value === false) { // skip replacement
					i++;
					continue;
				}
				
				str = str.substring(0, i) + new_value + str.substring(i + cur_sl);
				// adjust indexes
				il = str.length;
				i += new_value.length;
			} else {
				i++;
			}
		}
		
		return str;
	}
	
	/**
	 * Processes profile argument, returning, if possible, profile object
	 */
	function processProfile(profile) {
		var _profile = profile;
		if (typeof(profile) == 'string' && profile in profiles)
			_profile = profiles[profile];
		
		if (!_profile)
			_profile = profiles['plain'];
			
		return _profile;
	}
	
	// create default profiles
	setupProfile('xhtml');
	setupProfile('html', {self_closing_tag: false});
	setupProfile('xml', {self_closing_tag: true, tag_nl: true});
	setupProfile('plain', {tag_nl: false, indent: false, place_cursor: false});
	
	
	return {
		/** Hash of all available actions */
		actions: {},
		
		/**
		 * Adds new Zen Coding action. This action will be available in
		 * <code>zen_settings.actions</code> object.
		 * @param {String} name Action's name
		 * @param {Function} fn Action itself. The first argument should be
		 * <code>zen_editor</code> instance.
		 */
		registerAction: function(name, fn) {
			this.actions[name.toLowerCase()] = fn;
		},
		
		/**
		 * Runs Zen Coding action. For list of available actions and their
		 * arguments see <code>zen_actions.js</code> file.
		 * @param {String} name Action name 
		 * @param {Array} args Additional arguments. It may be array of arguments
		 * or inline arguments. The first argument should be <code>zen_editor</code> instance
		 * @example
		 * zen_coding.runActions('expand_abbreviation', zen_editor);  
		 * zen_coding.runActions('wrap_with_abbreviation', [zen_editor, 'div']);  
		 */
		runAction: function(name, args) {
			if (!(args instanceof Array))
				args = Array.prototype.slice.call(arguments, 1);
				
			name = name.toLowerCase();
			if (name in this.actions)
				return this.actions[name].apply(this, args);
//			try {
//			} catch(e){
//				if (window && window.console)
//					console.error(e);
//				return false; 
//			}
		},
		
		expandAbbreviation: function(abbr, type, profile) {
			type = type || 'html';
			var parsed_tree = this.parseIntoTree(abbr, type);
			
			if (parsed_tree) {
				var tree = rolloutTree(parsed_tree);
				this.applyFilters(tree, type, profile, parsed_tree.filters);
				return replaceVariables(tree.toString());
			}
			
			return '';
		},
		
		/**
		 * Extracts abbreviations from text stream, starting from the end
		 * @param {String} str
		 * @return {String} Abbreviation or empty string
		 */
		extractAbbreviation: function(str) {
			var cur_offset = str.length,
				start_index = -1,
				group_count = 0,
				brace_count = 0,
				text_count = 0;
			
			while (true) {
				cur_offset--;
				if (cur_offset < 0) {
					// moved to the beginning of the line
					start_index = 0;
					break;
				}
				
				var ch = str.charAt(cur_offset);
				
				if (ch == ']') {
					brace_count++;
				} else if (ch == '[') {
					if (!brace_count) { // unexpected brace
						start_index = cur_offset + 1;
						break;
					}
					brace_count--;
				} else if (ch == '}') {
					text_count++;
				} else if (ch == '{') {
					if (!text_count) { // unexpected brace
						start_index = cur_offset + 1;
						break;
					}
					text_count--;
				} else if (ch == ')') {
					group_count++;
				} else if (ch == '(') {
					if (!group_count) { // unexpected brace
						start_index = cur_offset + 1;
						break;
					}
					group_count--;
				} else {
					if (brace_count || text_count) 
						// respect all characters inside attribute sets or text nodes
						continue;
					else if (!isAllowedChar(ch) || (ch == '>' && isEndsWithTag(str.substring(0, cur_offset + 1)))) {
						// found stop symbol
						start_index = cur_offset + 1;
						break;
					}
				}
			}
			
			if (start_index != -1 && !text_count && !brace_count && !group_count) 
				// found something, return abbreviation
				return str.substring(start_index);
			else
				return '';
		},
		
		/**
		 * Parses abbreviation into a node set
		 * @param {String} abbr Abbreviation
		 * @param {String} type Document type (xsl, html, etc.)
		 * @return {Tag}
		 */
		parseIntoTree: function(abbr, type) {
			type = type || 'html';
			// remove filters from abbreviation
			var filter_list = '';
			abbr = abbr.replace(/\|([\w\|\-]+)$/, function(str, p1){
				filter_list = p1;
				return '';
			});
			
			// try to parse abbreviation
			try {
				var abbr_tree = zen_parser.parse(abbr),
					tree_root = new Tag({}, type);
					
				abbr_tree = preprocessParsedTree(abbr_tree, type);
			} catch(e) {
				if (e.message == "InvalidAbbreviation")
					return null;
			}
				
			// then recursively expand each group item
			for (var i = 0, il = abbr_tree.children.length; i < il; i++) {
				processParsedNode(abbr_tree.children[i], type, tree_root);
			}
			
			tree_root.filters = filter_list;
			return tree_root;
		},
		
		/**
		 * Indents text with padding
		 * @param {String} text Text to indent
		 * @param {String|Number} pad Padding size (number) or padding itself (string)
		 * @return {String}
		 */
		padString: padString,
		setupProfile: setupProfile,
		getNewline: function(){
			return newline;
		},
		
		setNewline: function(str) {
			newline = str;
			this.setVariable('newline', str);
			this.setVariable('nl', str);
		},
		
		/**
		 * Wraps passed text with abbreviation. Text will be placed inside last
		 * expanded element
		 * @param {String} abbr Abbreviation
		 * @param {String} text Text to wrap
		 * @param {String} [type] Document type (html, xml, etc.). Default is 'html'
		 * @param {String} [profile] Output profile's name. Default is 'plain'
		 * @return {String}
		 */
		wrapWithAbbreviation: function(abbr, text, type, profile) {
			type = type || 'html';
			var tree_root = this.parseIntoTree(abbr, type),
				pasted = false;
				
			if (tree_root) {
				if (tree_root.multiply_elem) {
					// we have a repeating element, put content in
					tree_root.multiply_elem.setPasteContent(text);
					tree_root.multiply_elem.repeat_by_lines = pasted = true;
				}
				
				var tree = rolloutTree(tree_root);
				
				if (!pasted) 
					tree.pasteContent(text);
				
				this.applyFilters(tree, type, profile, tree_root.filters);
				return replaceVariables(tree.toString());
			}
			
			return null;
		},
		
		splitByLines: splitByLines,
		
		/**
		 * Check if cursor is placed inside xHTML tag
		 * @param {String} html Contents of the document
		 * @param {Number} cursor_pos Current caret position inside tag
		 * @return {Boolean}
		 */
		isInsideTag: function(html, cursor_pos) {
			var re_tag = /^<\/?\w[\w\:\-]*.*?>/;
			
			// search left to find opening brace
			var pos = cursor_pos;
			while (pos > -1) {
				if (html.charAt(pos) == '<') 
					break;
				pos--;
			}
			
			if (pos != -1) {
				var m = re_tag.exec(html.substring(pos));
				if (m && cursor_pos > pos && cursor_pos < pos + m[0].length)
					return true;
			}
			
			return false;
		},
		
		/**
		 * Returns caret placeholder
		 * @return {String}
		 */
		getCaretPlaceholder: getCaretPlaceholder,
		
		/**
		 * Set caret placeholder: a string (like '|') or function.
		 * You may use a function as a placeholder generator. For example,
		 * TextMate uses ${0}, ${1}, ..., ${n} natively for quick Tab-switching
		 * between them.
		 * @param {String|Function} value
		 */
		setCaretPlaceholder: function(value) {
			caret_placeholder = value;
		},
		
		rolloutTree: rolloutTree,
		
		/**
		 * Register new filter
		 * @param {String} name Filter name
		 * @param {Function} fn Filter function
		 */
		registerFilter: function(name, fn) {
			filters[name] = fn;
		},
		
		/**
		 * Factory method that produces <code>ZenNode</code> instance
		 * @param {String} name Node name
		 * @param {Array} [attrs] Array of attributes as key/value objects  
		 * @return {ZenNode}
		 */
		nodeFactory: function(name, attrs) {
			return new ZenNode({name: name, attributes: attrs || []});
		},
		
		/**
		 * Applies filters to tree according to syntax
		 * @param {ZenNode} tree Tag tree to apply filters to
		 * @param {String} syntax Syntax name ('html', 'css', etc.)
		 * @param {String|Object} profile Profile or profile's name
		 * @param {String|Array} [additional_filters] List or pipe-separated 
		 * string of additional filters to apply
		 * 
		 * @return {ZenNode}
		 */
		applyFilters: function(tree, syntax, profile, additional_filters) {
			profile = processProfile(profile);
			var _filters = profile.filters;
			if (!_filters)
				_filters = zen_resources.getSubset(syntax, 'filters') || basic_filters;
				
			if (additional_filters)
				_filters += '|' + ((typeof(additional_filters) == 'string') 
					? additional_filters 
					: additional_filters.join('|'));
				
			if (!_filters)
				// looks like unknown syntax, apply basic filters
				_filters = basic_filters;
				
			return runFilters(tree, profile, _filters);
		},
		
		runFilters: runFilters,
		
		repeatString: repeatString,
		getVariable: getVariable,
		/**
		 * Store runtime variable in user storage
		 * @param {String} name Variable name
		 * @param {String} value Variable value
		 */
		setVariable: function(name, value){
			var voc = zen_resources.getVocabulary('user') || {};
			if (!('varaibles' in voc))
				voc.variables = {};
				
			voc.variables[name] = value;
			zen_resources.setVocabulary(voc, 'user');
		},
		replaceVariables: replaceVariables,
		
		/**
		 * Escapes special characters used in Zen Coding, like '$', '|', etc.
		 * Use this method before passing to actions like "Wrap with Abbreviation"
		 * to make sure that existing spacial characters won't be altered
		 * @param {String} text
		 * @return {String}
		 */
		escapeText: function(text) {
			return text.replace(/([\$\|\\])/g, '\\$1');
		},
		
		/**
		 * Unescapes special characters used in Zen Coding, like '$', '|', etc.
		 * @param {String} text
		 * @return {String}
		 */
		unescapeText: function(text) {
			return text.replace(/\\(.)/g, '$1');
		},
		
		/**
		 * Replaces '$' character in string assuming it might be escaped with '\'
		 * @param {String} str
		 * @param {String|Number} value
		 * @return {String}
		 */
		replaceCounter: function(str, value) {
			var symbol = '$';
			value = String(value);
			return replaceUnescapedSymbol(str, symbol, function(str, symbol, pos, match_num){
				if (str.charAt(pos + 1) == '{' || isNumeric(str.charAt(pos + 1)) ) {
					// it's a variable, skip it
					return false;
				}
				
				// replace sequense of $ symbols with padded number  
				var j = pos + 1;
				while(str.charAt(j) == '$' && str.charAt(j + 1) != '{') j++;
				return [str.substring(pos, j), zeroPadString(value, j - pos)];
			});
		},
		
		isNumeric: isNumeric,
		
		/**
		 * Upgrades tabstops in zen node in order to prevent naming conflicts
		 * @param {ZenNode} node
		 * @param {Number} offset Tab index offset
		 * @returns {Number} Maximum tabstop index in element
		 */
		upgradeTabstops: function(node, offset) {
			var max_num = 0,
				props = ['start', 'end', 'content'],
				escape_fn = function(ch){ return '\\' + ch; },
				tabstop_fn = function(i, num, value) {
					num = parseInt(num);
					if (num > max_num) max_num = num;
						
					if (value)
						return '${' + (num + offset) + ':' + value + '}';
					else
						return '$' + (num + offset);
				};
				
			for (var i = 0, il = props.length; i < il; i++)
				node[props[i]] = this.processTextBeforePaste(node[props[i]], escape_fn, tabstop_fn);
			
			return max_num;
		},
		
		/**
		 * Get profile by it's name. If profile wasn't found, returns 'plain'
		 * profile
		 */
		getProfile: function(name) {
			return (name in profiles) ? profiles[name] : profiles['plain'];
		},
		
		/**
		 * Gets image size from image byte stream.
		 * @author http://romeda.org/rePublish/
		 * @param {String} stream Image byte stream (use <code>zen_file.read()</code>)
		 * @return {Object} Object with <code>width</code> and <code>height</code> properties
		 */
		getImageSize: function(stream) {
			var pngMagicNum = "\211PNG\r\n\032\n",
				jpgMagicNum = "\377\330",
				gifMagicNum = "GIF8",
				nextByte = function() {
					return stream.charCodeAt(pos++);
				};
		
			if (stream.substr(0, 8) === pngMagicNum) {
				// PNG. Easy peasy.
				var pos = stream.indexOf('IHDR') + 4;
			
				return { width:  (nextByte() << 24) | (nextByte() << 16) |
								 (nextByte() <<  8) | nextByte(),
						 height: (nextByte() << 24) | (nextByte() << 16) |
								 (nextByte() <<  8) | nextByte() };
			
			} else if (stream.substr(0, 4) === gifMagicNum) {
				pos = 6;
			
				return {
					width:  nextByte() | (nextByte() << 8),
					height: nextByte() | (nextByte() << 8)
				};
			
			} else if (stream.substr(0, 2) === jpgMagicNum) {
				// TODO need testing
				pos = 2;
			
				var l = stream.length;
				while (pos < l) {
					if (nextByte() != 0xFF) return;
				
					var marker = nextByte();
					if (marker == 0xDA) break;
				
					var size = (nextByte() << 8) | nextByte();
				
					if (marker >= 0xC0 && marker <= 0xCF && !(marker & 0x4) && !(marker & 0x8)) {
						pos += 1;
						return { height:  (nextByte() << 8) | nextByte(),
								 width: (nextByte() << 8) | nextByte() };
				
					} else {
						pos += size - 2;
					}
				}
			}
		},
		
		/**
		 * Returns context-aware node counter
		 * @param {node} ZenNode
		 * @return {Number}
		 */
		getCounterForNode: function(node) {
			// find nearest repeating parent
			var counter = node.counter;
			if (!node.is_repeating && !node.repeat_by_lines) {
				while (node = node.parent) {
					if (node.is_repeating || node.repeat_by_lines)
						return node.counter;
				}
			}
			
			return counter;
		},
		
		/**
		 * Process text that should be pasted into editor: clear escaped text and
		 * handle tabstops
		 * @param {String} text
		 * @param {Function} escape_fn Handle escaped character. Must return
		 * replaced value
		 * @param {Function} tabstop_fn Callback function that will be called on every
		 * tabstob occurance, passing <b>index</b>, <code>number</code> and 
		 * <b>value</b> (if exists) arguments. This function must return 
		 * replacement value
		 * @return {String} 
		 */
		processTextBeforePaste: function(text, escape_fn, tabstop_fn) {
			var i = 0, il = text.length, start_ix, _i,
				str_builder = [];
				
			var nextWhile = function(ix, fn) {
				while (ix < il) if (!fn(text.charAt(ix++))) break;
				return ix - 1;
			};
			
			while (i < il) {
				var ch = text.charAt(i);
				if (ch == '\\' && i + 1 < il) {
					// handle escaped character
					str_builder.push(escape_fn(text.charAt(i + 1)));
					i += 2;
					continue;
				} else if (ch == '$') {
					// looks like a tabstop
					var next_ch = text.charAt(i + 1) || '';
					_i = i;
					if (this.isNumeric(next_ch)) {
						// $N placeholder
						start_ix = i + 1;
						i = nextWhile(start_ix, this.isNumeric);
						if (start_ix < i) {
							str_builder.push(tabstop_fn(_i, text.substring(start_ix, i)));
							continue;
						}
					} else if (next_ch == '{') {
						// ${N:value} or ${N} placeholder
						var brace_count = 1;
						start_ix = i + 2;
						i = nextWhile(start_ix, this.isNumeric);
						
						if (i > start_ix) {
							if (text.charAt(i) == '}') {
								str_builder.push(tabstop_fn(_i, text.substring(start_ix, i)));
								i++; // handle closing brace
								continue;
							} else if (text.charAt(i) == ':') {
								var val_start = i + 2;
								i = nextWhile(val_start, function(c) {
									if (c == '{') brace_count++;
									else if (c == '}') brace_count--;
									return !!brace_count;
								});
								str_builder.push(tabstop_fn(_i, text.substring(start_ix, val_start - 2), text.substring(val_start - 1, i)));
								i++; // handle closing brace
								continue;
							}
						}
					}
					i = _i;
				}
				
				// push current character to stack
				str_builder.push(ch);
				i++;
			}
			
			return str_builder.join('');
		}
	}
})();/**
 * Middleware layer that communicates between editor and Zen Coding.
 * This layer describes all available Zen Coding actions, like 
 * "Expand Abbreviation".
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 * 
 * @include "zen_editor.js"
 * @include "html_matcher.js"
 * @include "zen_coding.js"
 * @include "zen_file.js"
 * @include "base64.js"
 */

/**
 * Search for abbreviation in editor from current caret position
 * @param {zen_editor} editor Editor instance
 * @return {String|null}
 */
function findAbbreviation(editor) {
	var range = editor.getSelectionRange(),
		content = String(editor.getContent());
	if (range.start != range.end) {
		// abbreviation is selected by user
		return content.substring(range.start, range.end);
	}
	
	// search for new abbreviation from current caret position
	var cur_line = editor.getCurrentLineRange();
	return zen_coding.extractAbbreviation(content.substring(cur_line.start, range.start));
}

/**
 * Find from current caret position and expand abbreviation in editor
 * @param {zen_editor} editor Editor instance
 * @param {String} [syntax] Syntax type (html, css, etc.)
 * @param {String} [profile_name] Output profile name (html, xml, xhtml)
 * @return {Boolean} Returns <code>true</code> if abbreviation was expanded 
 * successfully
 */
function expandAbbreviation(editor, syntax, profile_name) {
	syntax = String(syntax || editor.getSyntax());
	profile_name = String(profile_name || editor.getProfileName());
	
	var caret_pos = editor.getSelectionRange().end,
		abbr,
		content = '';
		
	if ( (abbr = findAbbreviation(editor)) ) {
		content = zen_coding.expandAbbreviation(abbr, syntax, profile_name);
		if (content) {
			editor.replaceContent(content, caret_pos - abbr.length, caret_pos);
			return true;
		}
	}
	
	return false;
}

/**
 * A special version of <code>expandAbbreviation</code> function: if it can't
 * find abbreviation, it will place Tab character at caret position
 * @param {zen_editor} editor Editor instance
 * @param {String} syntax Syntax type (html, css, etc.)
 * @param {String} profile_name Output profile name (html, xml, xhtml)
 */
function expandAbbreviationWithTab(editor, syntax, profile_name) {
	syntax = String(syntax || editor.getSyntax());
	profile_name = String(profile_name || editor.getProfileName());
	if (!expandAbbreviation(editor, syntax, profile_name))
		editor.replaceContent(zen_coding.getVariable('indentation'), editor.getCaretPos());
}

/**
 * Find and select HTML tag pair
 * @param {zen_editor} editor Editor instance
 * @param {String} [direction] Direction of pair matching: 'in' or 'out'. 
 * Default is 'out'
 */
function matchPair(editor, direction, syntax) {
	direction = String((direction || 'out').toLowerCase());
	syntax = String(syntax || editor.getProfileName());
	
	var range = editor.getSelectionRange(),
		cursor = range.end,
		range_start = range.start, 
		range_end = range.end,
//		content = zen_coding.splitByLines(editor.getContent()).join('\n'),
		content = String(editor.getContent()),
		range = null,
		_r,
	
		old_open_tag = zen_coding.html_matcher.last_match['opening_tag'],
		old_close_tag = zen_coding.html_matcher.last_match['closing_tag'];
		
	if (direction == 'in' && old_open_tag && range_start != range_end) {
//		user has previously selected tag and wants to move inward
		if (!old_close_tag) {
//			unary tag was selected, can't move inward
			return false;
		} else if (old_open_tag.start == range_start) {
			if (content.charAt(old_open_tag.end) == '<') {
//				test if the first inward tag matches the entire parent tag's content
				_r = zen_coding.html_matcher.find(content, old_open_tag.end + 1, syntax);
				if (_r[0] == old_open_tag.end && _r[1] == old_close_tag.start) {
					range = zen_coding.html_matcher(content, old_open_tag.end + 1, syntax);
				} else {
					range = [old_open_tag.end, old_close_tag.start];
				}
			} else {
				range = [old_open_tag.end, old_close_tag.start];
			}
		} else {
			var new_cursor = content.substring(0, old_close_tag.start).indexOf('<', old_open_tag.end);
			var search_pos = new_cursor != -1 ? new_cursor + 1 : old_open_tag.end;
			range = zen_coding.html_matcher(content, search_pos, syntax);
		}
	} else {
		range = zen_coding.html_matcher(content, cursor, syntax);
	}
	
	if (range !== null && range[0] != -1) {
		editor.createSelection(range[0], range[1]);
		return true;
	} else {
		return false;
	}
}

/**
 * Narrow down text indexes, adjusting selection to non-space characters
 * @param {String} text
 * @param {Number} start
 * @param {Number} end
 * @return {Array}
 */
function narrowToNonSpace(text, start, end) {
	// narrow down selection until first non-space character
	var re_space = /\s|\n|\r/;
	function isSpace(ch) {
		return re_space.test(ch);
	}
	
	while (start < end) {
		if (!isSpace(text.charAt(start)))
			break;
			
		start++;
	}
	
	while (end > start) {
		end--;
		if (!isSpace(text.charAt(end))) {
			end++;
			break;
		}
	}
	
	return [start, end];
}

/**
 * Wraps content with abbreviation
 * @param {zen_editor} Editor instance
 * @param {String} abbr Abbreviation to wrap with
 * @param {String} [syntax] Syntax type (html, css, etc.)
 * @param {String} [profile_name] Output profile name (html, xml, xhtml)
 */
function wrapWithAbbreviation(editor, abbr, syntax, profile_name) {
	syntax = String(syntax || editor.getSyntax());
	profile_name = String(profile_name || editor.getProfileName());
	abbr = abbr || editor.prompt("Enter abbreviation");
	
	var range = editor.getSelectionRange(),
		start_offset = range.start,
		end_offset = range.end,
		content = String(editor.getContent());
		
	if (!abbr || typeof abbr == 'undefined')
		return null; 
		
	abbr = String(abbr);
	
	if (start_offset == end_offset) {
		// no selection, find tag pair
		range = zen_coding.html_matcher(content, start_offset, profile_name);
		
		if (!range || range[0] == -1) // nothing to wrap
			return null;
		
		var narrowed_sel = narrowToNonSpace(content, range[0], range[1]);
		
		start_offset = narrowed_sel[0];
		end_offset = narrowed_sel[1];
	}
	
	var new_content = zen_coding.escapeText(content.substring(start_offset, end_offset)),
		result = zen_coding.wrapWithAbbreviation(abbr, unindent(editor, new_content), syntax, profile_name);
	
	if (result) {
		editor.setCaretPos(end_offset);
		editor.replaceContent(result, start_offset, end_offset);
	}
}

/**
 * Unindent content, thus preparing text for tag wrapping
 * @param {zen_editor} editor Editor instance
 * @param {String} text
 * @return {String}
 */
function unindent(editor, text) {
	return unindentText(text, getCurrentLinePadding(editor));
}

/**
 * Removes padding at the beginning of each text's line
 * @param {String} text
 * @param {String} pad
 */
function unindentText(text, pad) {
	var lines = zen_coding.splitByLines(text);
	for (var i = 0; i < lines.length; i++) {
		if (lines[i].search(pad) == 0)
			lines[i] = lines[i].substr(pad.length);
	}
	
	return lines.join(zen_coding.getNewline());
}

/**
 * Returns padding of current editor's line
 * @param {zen_editor} Editor instance
 * @return {String}
 */
function getCurrentLinePadding(editor) {
	return getLinePadding(editor.getCurrentLine());
}

/**
 * Returns line padding
 * @param {String} line
 * @return {String}
 */
function getLinePadding(line) {
	return (line.match(/^(\s+)/) || [''])[0];
}

/**
 * Search for new caret insertion point
 * @param {zen_editor} editor Editor instance
 * @param {Number} inc Search increment: -1 — search left, 1 — search right
 * @param {Number} offset Initial offset relative to current caret position
 * @return {Number} Returns -1 if insertion point wasn't found
 */
function findNewEditPoint(editor, inc, offset) {
	inc = inc || 1;
	offset = offset || 0;
	var cur_point = editor.getCaretPos() + offset,
		content = String(editor.getContent()),
		max_len = content.length,
		next_point = -1,
		re_empty_line = /^\s+$/;
	
	function ch(ix) {
		return content.charAt(ix);
	}
	
	function getLine(ix) {
		var start = ix;
		while (start >= 0) {
			var c = ch(start);
			if (c == '\n' || c == '\r')
				break;
			start--;
		}
		
		return content.substring(start, ix);
	}
		
	while (cur_point < max_len && cur_point > 0) {
		cur_point += inc;
		var cur_char = ch(cur_point),
			next_char = ch(cur_point + 1),
			prev_char = ch(cur_point - 1);
			
		switch (cur_char) {
			case '"':
			case '\'':
				if (next_char == cur_char && prev_char == '=') {
					// empty attribute
					next_point = cur_point + 1;
				}
				break;
			case '>':
				if (next_char == '<') {
					// between tags
					next_point = cur_point + 1;
				}
				break;
			case '\n':
			case '\r':
				// empty line
				if (re_empty_line.test(getLine(cur_point - 1))) {
					next_point = cur_point;
				}
				break;
		}
		
		if (next_point != -1)
			break;
	}
	
	return next_point;
}

/**
 * Move caret to previous edit point
 * @param {zen_editor} editor Editor instance
 */
function prevEditPoint(editor) {
	var cur_pos = editor.getCaretPos(),
		new_point = findNewEditPoint(editor, -1);
		
	if (new_point == cur_pos)
		// we're still in the same point, try searching from the other place
		new_point = findNewEditPoint(editor, -1, -2);
	
	if (new_point != -1) 
		editor.setCaretPos(new_point);
}

/**
 * Move caret to next edit point
 * @param {zen_editor} editor Editor instance
 */
function nextEditPoint(editor) {
	var new_point = findNewEditPoint(editor, 1);
	if (new_point != -1)
		editor.setCaretPos(new_point);
}

/**
 * Inserts newline character with proper indentation in specific positions only.
 * @param {zen_editor} editor
 * @return {Boolean} Returns <code>true</code> if line break was inserted 
 */
function insertFormattedNewlineOnly(editor) {
	var caret_pos = editor.getCaretPos(),
		content = String(editor.getContent()),
		nl = zen_coding.getNewline(),
		pad = zen_coding.getVariable('indentation'),
		syntax = String(editor.getSyntax());
		
	if (syntax == 'html') {
		// let's see if we're breaking newly created tag
		var pair = zen_coding.html_matcher.getTags(content, caret_pos, String(editor.getProfileName()));
		
		if (pair[0] && pair[1] && pair[0].type == 'tag' && pair[0].end == caret_pos && pair[1].start == caret_pos) {
			editor.replaceContent(nl + pad + zen_coding.getCaretPlaceholder() + nl, caret_pos);
			return true;
		}
	} else if (syntax == 'css') {
		if (caret_pos && content.charAt(caret_pos - 1) == '{') {
			// look ahead for a closing brace
			for (var i = caret_pos, il = content.length, ch; i < il; i++) {
				ch = content.charAt(i);
				if (ch == '}') return false;
				if (ch == '{') break;
			}
			
			// defining rule set
			var ins_value = nl + pad + zen_coding.getCaretPlaceholder() + nl,
				has_close_brace = caret_pos < content.length && content.charAt(caret_pos) == '}';
				
			var user_close_brace = zen_coding.getVariable('close_css_brace');
			if (user_close_brace) {
				// user defined how close brace should look like
				ins_value += zen_coding.replaceVariables(user_close_brace);
			} else if (!has_close_brace) {
				ins_value += '}';
			}
			
			editor.replaceContent(ins_value, caret_pos, caret_pos + (has_close_brace ? 1 : 0));
			return true;
		}
	}
		
	return false;
}

/**
 * Inserts newline character with proper indentation. This action is used in
 * editors that doesn't have indentation control (like textarea element) to 
 * provide proper indentation
 * @param {zen_editor} editor Editor instance
 */
function insertFormattedNewline(editor) {
	if (!insertFormattedNewlineOnly(editor)) {
		var cur_padding = getCurrentLinePadding(editor),
			content = String(editor.getContent()),
			caret_pos = editor.getCaretPos(),
			c_len = content.length,
			nl = zen_coding.getNewline();
			
		// check out next line padding
		var line_range = editor.getCurrentLineRange(),
			next_padding = '';
			
		for (var i = line_range.end + 1, ch; i < c_len; i++) {
			ch = content.charAt(i);
			if (ch == ' ' || ch == '\t')
				next_padding += ch;
			else
				break;
		}
		
		if (next_padding.length > cur_padding.length)
			editor.replaceContent(nl + next_padding, caret_pos, caret_pos, true);
		else
			editor.replaceContent(nl, caret_pos);
	}
}

/**
 * Select line under cursor
 * @param {zen_editor} editor Editor instance
 */
function selectLine(editor) {
	var range = editor.getCurrentLineRange();
	editor.createSelection(range.start, range.end);
}

/**
 * Moves caret to matching opening or closing tag
 * @param {zen_editor} editor
 */
function goToMatchingPair(editor) {
	var content = String(editor.getContent()),
		caret_pos = editor.getCaretPos();
	
	if (content.charAt(caret_pos) == '<') 
		// looks like caret is outside of tag pair  
		caret_pos++;
		
	var tags = zen_coding.html_matcher.getTags(content, caret_pos, String(editor.getProfileName()));
		
	if (tags && tags[0]) {
		// match found
		var open_tag = tags[0],
			close_tag = tags[1];
			
		if (close_tag) { // exclude unary tags
			if (open_tag.start <= caret_pos && open_tag.end >= caret_pos)
				editor.setCaretPos(close_tag.start);
			else if (close_tag.start <= caret_pos && close_tag.end >= caret_pos)
				editor.setCaretPos(open_tag.start);
		}
	}
}

/**
 * Merge lines spanned by user selection. If there's no selection, tries to find
 * matching tags and use them as selection
 * @param {zen_editor} editor
 */
function mergeLines(editor) {
	var selection = editor.getSelectionRange();
	if (selection.start == selection.end) {
		// find matching tag
		var pair = zen_coding.html_matcher(String(editor.getContent()), editor.getCaretPos(), String(editor.getProfileName()));
		if (pair) {
			selection.start = pair[0];
			selection.end = pair[1];
		}
	}
	
	if (selection.start != selection.end) {
		// got range, merge lines
		var text = String(editor.getContent()).substring(selection.start, selection.end),
			old_length = text.length;
		var lines =  zen_coding.splitByLines(text);
		
		for (var i = 1; i < lines.length; i++) {
			lines[i] = lines[i].replace(/^\s+/, '');
		}
		
		text = lines.join('').replace(/\s{2,}/, ' ');
		editor.replaceContent(text, selection.start, selection.end);
		editor.createSelection(selection.start, selection.start + text.length);
	}
}

/**
 * Toggle comment on current editor's selection or HTML tag/CSS rule
 * @param {zen_editor} editor
 */
function toggleComment(editor) {
	var syntax = String(editor.getSyntax());
	if (syntax == 'css') {
		// in case out editor is good enough and can recognize syntax from 
		// current token, we have to make sure that cursor is not inside
		// 'style' attribute of html element
		var caret_pos = editor.getCaretPos();
		var pair = zen_coding.html_matcher.getTags(String(editor.getContent()), caret_pos);
		if (pair && pair[0] && pair[0].type == 'tag' && 
				pair[0].start <= caret_pos && pair[0].end >= caret_pos) {
			syntax = 'html';
		}
	}
	
	switch (syntax) {
		case 'css':
			return toggleCSSComment(editor);
		default:
			return toggleHTMLComment(editor);
	}
}

/**
 * Toggle HTML comment on current selection or tag
 * @param {zen_editor} editor
 * @return {Boolean} Returns <code>true</code> if comment was toggled
 */
function toggleHTMLComment(editor) {
	var rng = editor.getSelectionRange(),
		content = String(editor.getContent());
		
	if (rng.start == rng.end) {
		// no selection, find matching tag
		var pair = zen_coding.html_matcher.getTags(content, editor.getCaretPos(), String(editor.getProfileName()));
		if (pair && pair[0]) { // found pair
			rng.start = pair[0].start;
			rng.end = pair[1] ? pair[1].end : pair[0].end;
		}
	}
	
	return genericCommentToggle(editor, '<!--', '-->', rng.start, rng.end);
}

/**
 * Simple CSS commenting
 * @param {zen_editor} editor
 * @return {Boolean} Returns <code>true</code> if comment was toggled
 */
function toggleCSSComment(editor) {
	var rng = editor.getSelectionRange();
		
	if (rng.start == rng.end) {
		// no selection, get current line
		rng = editor.getCurrentLineRange();

		// adjust start index till first non-space character
		var _r = narrowToNonSpace(String(editor.getContent()), rng.start, rng.end);
		rng.start = _r[0];
		rng.end = _r[1];
	}
	
	return genericCommentToggle(editor, '/*', '*/', rng.start, rng.end);
}

/**
 * Search for nearest comment in <code>str</code>, starting from index <code>from</code>
 * @param {String} text Where to search
 * @param {Number} from Search start index
 * @param {String} start_token Comment start string
 * @param {String} end_token Comment end string
 * @return {Array|null} Returns null if comment wasn't found
 */
function searchComment(text, from, start_token, end_token) {
	var start_ch = start_token.charAt(0),
		end_ch = end_token.charAt(0),
		comment_start = -1,
		comment_end = -1;
	
	function hasMatch(str, start) {
		return text.substr(start, str.length) == str;
	}
		
	// search for comment start
	while (from--) {
		if (text.charAt(from) == start_ch && hasMatch(start_token, from)) {
			comment_start = from;
			break;
		}
	}
	
	if (comment_start != -1) {
		// search for comment end
		from = comment_start;
		var content_len = text.length;
		while (content_len >= from++) {
			if (text.charAt(from) == end_ch && hasMatch(end_token, from)) {
				comment_end = from + end_token.length;
				break;
			}
		}
	}
	
	return (comment_start != -1 && comment_end != -1) 
		? [comment_start, comment_end] 
		: null;
}

/**
 * Escape special regexp chars in string, making it usable for creating dynamic
 * regular expressions
 * @param {String} str
 * @return {String}
 */
function escapeForRegexp(str) {
  var specials = new RegExp("[.*+?|()\\[\\]{}\\\\]", "g"); // .*+?|()[]{}\
  return str.replace(specials, "\\$&");
}

/**
 * Generic comment toggling routine
 * @param {zen_editor} editor
 * @param {String} comment_start Comment start token
 * @param {String} comment_end Comment end token
 * @param {Number} range_start Start selection range
 * @param {Number} range_end End selection range
 * @return {Boolean}
 */
function genericCommentToggle(editor, comment_start, comment_end, range_start, range_end) {
	var content = String(editor.getContent()),
		caret_pos = editor.getCaretPos(),
		new_content = null;
		
	/**
	 * Remove comment markers from string
	 * @param {Sting} str
	 * @return {String}
	 */
	function removeComment(str) {
		return str
			.replace(new RegExp('^' + escapeForRegexp(comment_start) + '\\s*'), function(str){
				caret_pos -= str.length;
				return '';
			}).replace(new RegExp('\\s*' + escapeForRegexp(comment_end) + '$'), '');
	}
	
	function hasMatch(str, start) {
		return content.substr(start, str.length) == str;
	}
		
	// first, we need to make sure that this substring is not inside 
	// comment
	var comment_range = searchComment(content, caret_pos, comment_start, comment_end);
	
	if (comment_range && comment_range[0] <= range_start && comment_range[1] >= range_end) {
		// we're inside comment, remove it
		range_start = comment_range[0];
		range_end = comment_range[1];
		
		new_content = removeComment(content.substring(range_start, range_end));
	} else {
		// should add comment
		// make sure that there's no comment inside selection
		new_content = comment_start + ' ' + 
			content.substring(range_start, range_end)
				.replace(new RegExp(escapeForRegexp(comment_start) + '\\s*|\\s*' + escapeForRegexp(comment_end), 'g'), '') +
			' ' + comment_end;
			
		// adjust caret position
		caret_pos += comment_start.length + 1;
	}

	// replace editor content
	if (new_content !== null) {
		editor.setCaretPos(range_start);
		editor.replaceContent(unindent(editor, new_content), range_start, range_end);
		editor.setCaretPos(caret_pos);
		return true;
	}
	
	return false;
}

/**
 * Splits or joins tag, e.g. transforms it into a short notation and vice versa:<br>
 * &lt;div&gt;&lt;/div&gt; → &lt;div /&gt; : join<br>
 * &lt;div /&gt; → &lt;div&gt;&lt;/div&gt; : split
 * @param {zen_editor} editor Editor instance
 * @param {String} [profile_name] Profile name
 */
function splitJoinTag(editor, profile_name) {
	var caret_pos = editor.getCaretPos(),
		profile = zen_coding.getProfile(String(profile_name || editor.getProfileName())),
		caret = zen_coding.getCaretPlaceholder();

	// find tag at current position
	var pair = zen_coding.html_matcher.getTags(String(editor.getContent()), caret_pos, String(editor.getProfileName()));
	if (pair && pair[0]) {
		var new_content = pair[0].full_tag;
		
		if (pair[1]) { // join tag
			var closing_slash = ' /';
			if (profile.self_closing_tag === true)
				closing_slash = '/';
				
			new_content = new_content.replace(/\s*>$/, closing_slash + '>');
			
			// add caret placeholder
			if (new_content.length + pair[0].start < caret_pos)
				new_content += caret;
			else {
				var d = caret_pos - pair[0].start;
				new_content = new_content.substring(0, d) + caret + new_content.substring(d);
			}
			
			editor.replaceContent(new_content, pair[0].start, pair[1].end);
		} else { // split tag
			var nl = zen_coding.getNewline(),
				pad = zen_coding.getVariable('indentation');
			
			// define tag content depending on profile
			var tag_content = (profile.tag_nl === true)
					? nl + pad +caret + nl
					: caret;
					
			new_content = new_content.replace(/\s*\/>$/, '>') + tag_content + '</' + pair[0].name + '>';
			editor.replaceContent(new_content, pair[0].start, pair[0].end);
		}
		
		return true;
	} else {
		return false;
	}
}

/**
 * Returns line bounds for specific character position
 * @param {String} text
 * @param {Number} from Where to start searching
 * @return {Object}
 */
function getLineBounds(text, from) {
	var len = text.length,
		start = 0,
		end = len - 1;
	
	// search left
	for (var i = from - 1; i > 0; i--) {
		var ch = text.charAt(i);
		if (ch == '\n' || ch == '\r') {
			start = i + 1;
			break;
		}
	}
	// search right
	for (var j = from; j < len; j++) {
		var ch = text.charAt(j);
		if (ch == '\n' || ch == '\r') {
			end = j;
			break;
		}
	}
	
	return {start: start, end: end};
}

/**
 * Gracefully removes tag under cursor
 * @param {zen_editor} editor
 */
function removeTag(editor) {
	var caret_pos = editor.getCaretPos(),
		content = String(editor.getContent());
		
	// search for tag
	var pair = zen_coding.html_matcher.getTags(content, caret_pos, String(editor.getProfileName()));
	if (pair && pair[0]) {
		if (!pair[1]) {
			// simply remove unary tag
			editor.replaceContent(zen_coding.getCaretPlaceholder(), pair[0].start, pair[0].end);
		} else {
			var tag_content_range = narrowToNonSpace(content, pair[0].end, pair[1].start),
				start_line_bounds = getLineBounds(content, tag_content_range[0]),
				start_line_pad = getLinePadding(content.substring(start_line_bounds.start, start_line_bounds.end)),
				tag_content = content.substring(tag_content_range[0], tag_content_range[1]);
				
			tag_content = unindentText(tag_content, start_line_pad);
			editor.replaceContent(zen_coding.getCaretPlaceholder() + tag_content, pair[0].start, pair[1].end);
		}
		
		return true;
	} else {
		return false;
	}
}

/**
 * Test if <code>text</code> starts with <code>token</code> at <code>pos</code>
 * position. If <code>pos</code> is ommited, search from beginning of text 
 * @param {String} token Token to test
 * @param {String} text Where to search
 * @param {Number} pos Position where to start search
 * @return {Boolean}
 * @since 0.65
 */
function startsWith(token, text, pos) {
	pos = pos || 0;
	return text.charAt(pos) == token.charAt(0) && text.substr(pos, token.length) == token;
}

/**
 * Encodes/decodes image under cursor to/from base64
 * @param {zen_editor} editor
 * @since 0.65
 */
function encodeDecodeBase64(editor) {
	var data = String(editor.getSelection()),
		caret_pos = editor.getCaretPos();
		
	if (!data) {
		// no selection, try to find image bounds from current caret position
		var text = String(editor.getContent()),
			ch, 
			m;
		while (caret_pos-- >= 0) {
			if (startsWith('src=', text, caret_pos)) { // found <img src="">
				if (m = text.substr(caret_pos).match(/^(src=(["'])?)([^'"<>\s]+)\1?/)) {
					data = m[3];
					caret_pos += m[1].length;
				}
				break;
			} else if (startsWith('url(', text, caret_pos)) { // found CSS url() pattern
				if (m = text.substr(caret_pos).match(/^(url\((['"])?)([^'"\)\s]+)\1?/)) {
					data = m[3];
					caret_pos += m[1].length;
				}
				break;
			}
		}
	}
	
	if (data) {
		if (startsWith('data:', data))
			return decodeFromBase64(editor, data, caret_pos);
		else
			return encodeToBase64(editor, data, caret_pos);
	} else {
		return false;
	}
}

/**
 * Encodes image to base64
 * @requires zen_file
 * 
 * @param {zen_editor} editor
 * @param {String} img_path Path to image
 * @param {Number} pos Caret position where image is located in the editor
 * @return {Boolean}
 */
function encodeToBase64(editor, img_path, pos) {
	var editor_file = editor.getFilePath(),
		default_mime_type = 'application/octet-stream';
		
	if (editor_file === null) {
		throw "You should save your file before using this action";
	}
	
	// locate real image path
	var real_img_path = zen_file.locateFile(editor_file, img_path);
	if (real_img_path === null) {
		throw "Can't find " + img_path + ' file';
	}
	
	var b64 = base64.encode(String(zen_file.read(real_img_path)));
	if (!b64) {
		throw "Can't encode file content to base64";
	}
	
	b64 = 'data:' + (base64.mime_types[String(zen_file.getExt(real_img_path))] || default_mime_type) +
		';base64,' + b64;
		
	editor.replaceContent('$0' + b64, pos, pos + img_path.length);
	return true;
}

/**
 * Decodes base64 string back to file.
 * @requires zen_editor.prompt
 * @requires zen_file
 * 
 * @param {zen_editor} editor
 * @param {String} data Base64-encoded file content
 * @param {Number} pos Caret position where image is located in the editor
 */
function decodeFromBase64(editor, data, pos) {
	// ask user to enter path to file
	var file_path = String(editor.prompt('Enter path to file (absolute or relative)'));
	if (!file_path)
		return false;
		
	var abs_path = zen_file.createPath(editor.getFilePath(), file_path);
	if (!abs_path) {
		throw "Can't save file";
	}
	
	zen_file.save(abs_path, base64.decode( data.replace(/^data\:.+?;.+?,/, '') ));
	editor.replaceContent('$0' + file_path, pos, pos + data.length);
	return true;
}

/**
 * Make decimal number look good: convert it to fixed precision end remove
 * traling zeroes 
 * @param {Number} num
 * @param {Number} [fracion] Fraction numbers (default is 2)
 * @return {String}
 */
function prettifyNumber(num, fraction) {
	return num.toFixed(typeof fraction == 'undefined' ? 2 : fraction).replace(/\.?0+$/, '');
}

/**
 * Find expression bounds in current editor at caret position. 
 * On each character a <code>fn</code> function will be caller which must 
 * return <code>true</code> if current character meets requirements, 
 * <code>false</code> otherwise
 * @param {zen_editor} editor
 * @param {Function} fn Function to test each character of expression
 * @return {Array} If expression found, returns array with start and end 
 * positions 
 */
function findExpressionBounds(editor, fn) {
	var content = String(editor.getContent()),
		il = content.length,
		expr_start = editor.getCaretPos() - 1,
		expr_end = expr_start + 1;
		
	// start by searching left
	while (expr_start >= 0 && fn(content.charAt(expr_start), expr_start, content)) expr_start--;
	
	// then search right
	while (expr_end < il && fn(content.charAt(expr_end), expr_end, content)) expr_end++;
	
	return expr_end > expr_start ? [++expr_start, expr_end] : null;
}

/**
 * Extract number from current caret position of the <code>editor</code> and
 * increment it by <code>step</code>
 * @param {zen_editor} editor
 * @param {Number} step Increment step (may be negative)
 */
function incrementNumber(editor, step) {
	var content = String(editor.getContent()),
		has_sign = false,
		has_decimal = false;
		
	var r = findExpressionBounds(editor, function(ch) {
		if (zen_coding.isNumeric(ch))
			return true;
		if (ch == '.')
			return has_decimal ? false : has_decimal = true;
		if (ch == '-')
			return has_sign ? false : has_sign = true;
			
		return false;
	});
		
	if (r) {
		var num = parseFloat(content.substring(r[0], r[1]));
		if (!isNaN(num)) {
			num = prettifyNumber(num + step);
			editor.replaceContent(num, r[0], r[1]);
			editor.createSelection(r[0], r[0] + num.length);
			return true;
		}
	}
	
	return false;
}

/**
 * Evaluates simple math expresison under caret
 * @param {zen_editor} editor
 */
function evaluateMathExpression(editor) {
	var content = String(editor.getContent()),
		chars = '.+-*/\\';
		
	var r = findExpressionBounds(editor, function(ch) {
		return zen_coding.isNumeric(ch) || chars.indexOf(ch) != -1;
	});
		
	if (r) {
		var expr = content.substring(r[0], r[1]);
		
		// replace integral division: 11\2 => Math.round(11/2) 
		expr = expr.replace(/([\d\.\-]+)\\([\d\.\-]+)/g, 'Math.round($1/$2)');
		
		try {
			var result = new Function('return ' + expr)();
			result = prettifyNumber(result);
			editor.replaceContent(result, r[0], r[1]);
			editor.setCaretPos(r[0] + result.length);
			return true;
		} catch (e) {}
	}
	
	return false;
}

// register all actions
zen_coding.registerAction('expand_abbreviation', expandAbbreviation);
zen_coding.registerAction('expand_abbreviation_with_tab', expandAbbreviationWithTab);
zen_coding.registerAction('match_pair', matchPair);
zen_coding.registerAction('match_pair_inward', function(editor){
	matchPair(editor, 'in');
});

zen_coding.registerAction('match_pair_outward', function(editor){
	matchPair(editor, 'out');
});
zen_coding.registerAction('wrap_with_abbreviation', wrapWithAbbreviation);
zen_coding.registerAction('prev_edit_point', prevEditPoint);
zen_coding.registerAction('next_edit_point', nextEditPoint);
zen_coding.registerAction('insert_formatted_line_break', insertFormattedNewline);
zen_coding.registerAction('insert_formatted_line_break_only', insertFormattedNewlineOnly);
zen_coding.registerAction('select_line', selectLine);
zen_coding.registerAction('matching_pair', goToMatchingPair);
zen_coding.registerAction('merge_lines', mergeLines);
zen_coding.registerAction('toggle_comment', toggleComment);
zen_coding.registerAction('split_join_tag', splitJoinTag);
zen_coding.registerAction('remove_tag', removeTag);
zen_coding.registerAction('encode_decode_data_url', encodeDecodeBase64);
//zen_coding.registerAction('update_image_size', updateImageSize);

zen_coding.registerAction('increment_number_by_1', function(editor) {
	return incrementNumber(editor, 1);
});

zen_coding.registerAction('decrement_number_by_1', function(editor) {
	return incrementNumber(editor, -1);
});

zen_coding.registerAction('increment_number_by_10', function(editor) {
	return incrementNumber(editor, 10);
});

zen_coding.registerAction('decrement_number_by_10', function(editor) {
	return incrementNumber(editor, -10);
});

zen_coding.registerAction('increment_number_by_01', function(editor) {
	return incrementNumber(editor, 0.1);
});

zen_coding.registerAction('decrement_number_by_01', function(editor) {
	return incrementNumber(editor, -0.1);
});

zen_coding.registerAction('evaluate_math_expression', evaluateMathExpression);
/**
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	// Regular Expressions for parsing tags and attributes
	var start_tag = /^<([\w\:\-]+)((?:\s+[\w\-:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*)\s*(\/?)>/,
		end_tag = /^<\/([\w\:\-]+)[^>]*>/,
		attr = /([\w\-:]+)(?:\s*=\s*(?:(?:"((?:\\.|[^"])*)")|(?:'((?:\\.|[^'])*)')|([^>\s]+)))?/g;
		
	// Empty Elements - HTML 4.01
	var empty = makeMap("area,base,basefont,br,col,frame,hr,img,input,isindex,link,meta,param,embed");

	// Block Elements - HTML 4.01
	var block = makeMap("address,applet,blockquote,button,center,dd,dir,div,dl,dt,fieldset,form,frameset,hr,iframe,isindex,li,map,menu,noframes,noscript,object,ol,p,pre,script,table,tbody,td,tfoot,th,thead,tr,ul");

	// Inline Elements - HTML 4.01
	var inline = makeMap("a,abbr,acronym,applet,b,basefont,bdo,big,br,button,cite,code,del,dfn,em,font,i,iframe,img,input,ins,kbd,label,map,object,q,s,samp,select,small,span,strike,strong,sub,sup,textarea,tt,u,var");

	// Elements that you can, intentionally, leave open
	// (and which close themselves)
	var close_self = makeMap("colgroup,dd,dt,li,options,p,td,tfoot,th,thead,tr");
	
	/** Current matching mode */
	var cur_mode = 'xhtml';
	
	/** Last matched HTML pair */
	var last_match = {
		opening_tag: null, // tag() or comment() object
		closing_tag: null, // tag() or comment() object
		start_ix: -1,
		end_ix: -1
	};
	
	function setMode(new_mode) {
		if (!new_mode || new_mode != 'html')
			new_mode = 'xhtml';
			
		cur_mode = new_mode;
	}
	
	function tag(match, ix) {
		var name = match[1].toLowerCase();
		return  {
			name: name,
			full_tag: match[0],
			start: ix,
			end: ix + match[0].length,
			unary: Boolean(match[3]) || (name in empty && cur_mode == 'html'),
			has_close: Boolean(match[3]),
			type: 'tag',
			close_self: (name in close_self && cur_mode == 'html')
		};
	}
	
	function comment(start, end) {
		return {
			start: start,
			end: end,
			type: 'comment'
		};
	}
	
	function makeMap(str){
		var obj = {}, items = str.split(",");
		for ( var i = 0; i < items.length; i++ )
			obj[ items[i] ] = true;
		return obj;
	}
	
	/**
	 * Makes selection ranges for matched tag pair
	 * @param {tag} opening_tag
	 * @param {tag} closing_tag
	 * @param {Number} ix
	 */
	function makeRange(opening_tag, closing_tag, ix) {
		ix = ix || 0;
		
		var start_ix = -1, 
			end_ix = -1;
		
		if (opening_tag && !closing_tag) { // unary element
			start_ix = opening_tag.start;
			end_ix = opening_tag.end;
		} else if (opening_tag && closing_tag) { // complete element
			if (
				(opening_tag.start < ix && opening_tag.end > ix) || 
				(closing_tag.start <= ix && closing_tag.end > ix)
			) {
				start_ix = opening_tag.start;
				end_ix = closing_tag.end;
			} else {
				start_ix = opening_tag.end;
				end_ix = closing_tag.start;
			}
		}
		
		return [start_ix, end_ix];
	}
	
	/**
	 * Save matched tag for later use and return found indexes
	 * @param {tag} opening_tag
	 * @param {tag} closing_tag
	 * @param {Number} ix
	 * @return {Array}
	 */
	function saveMatch(opening_tag, closing_tag, ix) {
		ix = ix || 0;
		last_match.opening_tag = opening_tag; 
		last_match.closing_tag = closing_tag;
		
		var range = makeRange(opening_tag, closing_tag, ix);
		last_match.start_ix = range[0];
		last_match.end_ix = range[1];
		
		return last_match.start_ix != -1 ? [last_match.start_ix, last_match.end_ix] : null;
	}
	
	/**
	 * Handle unary tag: find closing tag if needed
	 * @param {String} text
	 * @param {Number} ix
	 * @param {tag} open_tag
	 * @return {tag|null} Closing tag (or null if not found) 
	 */
	function handleUnaryTag(text, ix, open_tag) {
		if (open_tag.has_close)
			return null;
		else {
			// TODO finish this method
		}
	}
	
	/**
	 * Search for matching tags in <code>html</code>, starting from 
	 * <code>start_ix</code> position
	 * @param {String} html Code to search
	 * @param {Number} start_ix Character index where to start searching pair 
	 * (commonly, current caret position)
	 * @param {Function} action Function that creates selection range
	 * @return {Array|null}
	 */
	function findPair(html, start_ix, mode, action) {
		action = action || makeRange;
		setMode(mode);
		
		var forward_stack = [],
			backward_stack = [],
			/** @type {tag()} */
			opening_tag = null,
			/** @type {tag()} */
			closing_tag = null,
			range = null,
			html_len = html.length,
			m,
			ix,
			tmp_tag;
			
		forward_stack.last = backward_stack.last = function() {
			return this[this.length - 1];
		}
		
		function hasMatch(str, start) {
			if (arguments.length == 1)
				start = ix;
			return html.substr(start, str.length) == str;
		}
		
		function searchCommentStart(from) {
			while (from--) {
				if (html.charAt(from) == '<' && hasMatch('<!--', from))
					break;
			}
			
			return from;
		}
		
		// find opening tag
		ix = start_ix;
		while (ix-- && ix >= 0) {
			var ch = html.charAt(ix);
			if (ch == '<') {
				var check_str = html.substring(ix, html_len);
				
				if ( (m = check_str.match(end_tag)) ) { // found closing tag
					tmp_tag = tag(m, ix);
					if (tmp_tag.start < start_ix && tmp_tag.end > start_ix) // direct hit on searched closing tag
						closing_tag = tmp_tag;
					else
						backward_stack.push(tmp_tag);
				} else if ( (m = check_str.match(start_tag)) ) { // found opening tag
					tmp_tag = tag(m, ix);
					
					if (tmp_tag.unary) {
						if (tmp_tag.start < start_ix && tmp_tag.end > start_ix) // exact match
							// TODO handle unary tag 
							return action(tmp_tag, null, start_ix);
					} else if (backward_stack.last() && backward_stack.last().name == tmp_tag.name) {
						backward_stack.pop();
					} else { // found nearest unclosed tag
						opening_tag = tmp_tag;
						break;
					}
				} else if (check_str.indexOf('<!--') == 0) { // found comment start
					var end_ix = check_str.search('-->') + ix + 3;
					if (ix < start_ix && end_ix >= start_ix)
						return action( comment(ix, end_ix) );
				}
			} else if (ch == '-' && hasMatch('-->')) { // found comment end
				// search left until comment start is reached
				ix = searchCommentStart(ix);
			}
		}
		
		if (!opening_tag)
			return action(null);
		
		// find closing tag
		if (!closing_tag) {
			for (ix = start_ix; ix < html_len; ix++) {
				var ch = html.charAt(ix);
				if (ch == '<') {
					var check_str = html.substring(ix, html_len);
					
					if ( (m = check_str.match(start_tag)) ) { // found opening tag
						tmp_tag = tag(m, ix);
						if (!tmp_tag.unary)
							forward_stack.push( tmp_tag );
					} else if ( (m = check_str.match(end_tag)) ) { // found closing tag
						var tmp_tag = tag(m, ix);
						if (forward_stack.last() && forward_stack.last().name == tmp_tag.name)
							forward_stack.pop();
						else { // found matched closing tag
							closing_tag = tmp_tag;
							break;
						}
					} else if (hasMatch('<!--')) { // found comment
						ix += check_str.search('-->') + 2;
					}
				} else if (ch == '-' && hasMatch('-->')) {
					// looks like cursor was inside comment with invalid HTML
					if (!forward_stack.last() || forward_stack.last().type != 'comment') {
						var end_ix = ix + 3;
						return action(comment( searchCommentStart(ix), end_ix ));
					}
				}
			}
		}
		
		return action(opening_tag, closing_tag, start_ix);
	}
	
	/**
	 * Search for matching tags in <code>html</code>, starting 
	 * from <code>start_ix</code> position. The result is automatically saved in 
	 * <code>last_match</code> property
	 * 
	 * @return {Array|null}
	 */
	var HTMLPairMatcher = function(/* String */ html, /* Number */ start_ix, /*  */ mode){
		return findPair(html, start_ix, mode, saveMatch);
	}
	
	HTMLPairMatcher.start_tag = start_tag;
	HTMLPairMatcher.end_tag = end_tag;
	
	/**
	 * Search for matching tags in <code>html</code>, starting from 
	 * <code>start_ix</code> position. The difference between 
	 * <code>HTMLPairMatcher</code> function itself is that <code>find</code> 
	 * method doesn't save matched result in <code>last_match</code> property.
	 * This method is generally used for lookups 
	 */
	HTMLPairMatcher.find = function(html, start_ix, mode) {
		return findPair(html, start_ix, mode);
	};
	
	/**
	 * Search for matching tags in <code>html</code>, starting from 
	 * <code>start_ix</code> position. The difference between 
	 * <code>HTMLPairMatcher</code> function itself is that <code>getTags</code> 
	 * method doesn't save matched result in <code>last_match</code> property 
	 * and returns array of opening and closing tags
	 * This method is generally used for lookups 
	 */
	HTMLPairMatcher.getTags = function(html, start_ix, mode) {
		return findPair(html, start_ix, mode, function(opening_tag, closing_tag){
			return [opening_tag, closing_tag];
		});
	};
	
	HTMLPairMatcher.last_match = last_match;
	
	try {
		zen_coding.html_matcher = HTMLPairMatcher;
	} catch(e){}
	
})();/**
 * Comment important tags (with 'id' and 'class' attributes)
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	/**
	 * Add comments to tag
	 * @param {ZenNode} node
	 */
	function addComments(node, i) {
		var id_attr = node.getAttribute('id'),
			class_attr = node.getAttribute('class'),
			nl = zen_coding.getNewline();
			
		if (id_attr || class_attr) {
			var comment_str = '',
				padding = (node.parent) ? node.parent.padding : '';
			if (id_attr) comment_str += '#' + id_attr;
			if (class_attr) comment_str += '.' + class_attr;
			
			node.start = node.start.replace(/</, '<!-- ' + comment_str + ' -->' + nl + padding + '<');
			node.end = node.end.replace(/>/, '>' + nl + padding + '<!-- /' + comment_str + ' -->');
			
			// replace counters
			var counter = zen_coding.getCounterForNode(node);
			node.start = zen_coding.replaceCounter(node.start, counter);
			node.end = zen_coding.replaceCounter(node.end, counter);
		}
	}
	
	function process(tree, profile) {
		if (profile.tag_nl === false)
			return tree;
			
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			
			if (item.isBlock())
				addComments(item, i);
			
			process(item, profile);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('c', process);
})();/**
 * Process CSS properties: replaces snippets, augumented with ! char, with 
 * <em>!important</em> suffix 
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	var re_important = /(.+)\!$/;
	function process(tree, profile) {
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			
			// CSS properties are always snippets
			if (item.type == 'snippet' && re_important.test(item.real_name)) {
				item.start = item.start.replace(/(;?)$/, ' !important$1');
			}
			
			process(item, profile);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('css', process);
})();/**
 * Filter for escaping unsafe XML characters: <, >, &
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	var char_map = {
		'<': '&lt;',
		'>': '&gt;',
		'&': '&amp;'
	}
	
	function escapeChars(str) {
		return str.replace(/([<>&])/g, function(str, p1){
			return char_map[p1];
		});
	}
	
	function process(tree, profile, level) {
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			
			item.start = escapeChars(item.start);
			item.end = escapeChars(item.end);
			
			process(item);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('e', process);
})();/**
 * Format CSS properties: add space after property name:
 * padding:0; → padding: 0;
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	function process(tree, profile) {
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			
			// CSS properties are always snippets 
			if (item.type == 'snippet') {
				item.start = item.start.replace(/([\w\-]+\s*:)(?!:)\s*/, '$1 ');
			}
			
			process(item, profile);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('fc', process);
})();/**
 * Generic formatting filter: creates proper indentation for each tree node,
 * placing "%s" placeholder where the actual output should be. You can use
 * this filter to preformat tree and then replace %s placeholder to whatever you
 * need. This filter should't be called directly from editor as a part 
 * of abbreviation.
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 * 
 * @include "../zen_coding.js"
 */
(function(){
	var child_token = '${child}',
		placeholder = '%s';
	
	function getNewline() {
		return zen_coding.getNewline();
	}
	
	function getIndentation() {
		return zen_resources.getVariable('indentation');
	}
	
	/**
	 * Test if passed node has block-level sibling element
	 * @param {ZenNode} item
	 * @return {Boolean}
	 */
	function hasBlockSibling(item) {
		return (item.parent && item.parent.hasBlockChildren());
	}
	
	/**
	 * Test if passed itrem is very first child of the whole tree
	 * @param {ZenNode} tree
	 */
	function isVeryFirstChild(item) {
		return item.parent && !item.parent.parent && !item.previousSibling;
	}
	
	/**
	 * Need to add line break before element
	 * @param {ZenNode} node
	 * @param {Object} profile
	 * @return {Boolean}
	 */
	function shouldBreakLine(node, profile) {
		if (!profile.inline_break)
			return false;
			
		// find toppest non-inline sibling
		while (node.previousSibling && node.previousSibling.isInline())
			node = node.previousSibling;
		
		if (!node.isInline())
			return false;
			
		// calculate how many inline siblings we have
		var node_count = 1;
		while (node = node.nextSibling) {
			if (node.type == 'text' || !node.isInline())
				node_count = 0;
			else if (node.isInline())
				node_count++;
		}
		
		return node_count >= profile.inline_break;
	}
	
	/**
	 * Need to add newline because <code>item</code> has too many inline children
	 * @param {ZenNode} node
	 * @param {Object} profile
	 */
	function shouldBreakChild(node, profile) {
		// we need to test only one child element, because 
		// hasBlockChildren() method will do the rest
		return (node.children.length && shouldBreakLine(node.children[0], profile));
	}
	
	/**
	 * Processes element with <code>snippet</code> type
	 * @param {ZenNode} item
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function processSnippet(item, profile, level) {
		var data = item.source.value;
			
		if (!data)
			// snippet wasn't found, process it as tag
			return processTag(item, profile, level);
			
		item.start = item.end = placeholder;
		
		var padding = (item.parent) 
			? item.parent.padding
			: zen_coding.repeatString(getIndentation(), level);
		
		if (!isVeryFirstChild(item)) {
			item.start = getNewline() + padding + item.start;
		}
		
		// adjust item formatting according to last line of <code>start</code> property
		var parts = data.split(child_token),
			lines = zen_coding.splitByLines(parts[0] || ''),
			padding_delta = getIndentation();
			
		if (lines.length > 1) {
			var m = lines[lines.length - 1].match(/^(\s+)/);
			if (m)
				padding_delta = m[1];
		}
		
		item.padding = padding + padding_delta;
		
		return item;
	}
	
	/**
	 * Processes element with <code>tag</code> type
	 * @param {ZenNode} item
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function processTag(item, profile, level) {
		if (!item.name)
			// looks like it's a root element
			return item;
		
		item.start = item.end = placeholder;
		
		var is_unary = (item.isUnary() && !item.children.length);
			
		// formatting output
		if (profile.tag_nl !== false) {
			var padding = (item.parent) 
					? item.parent.padding
					: zen_coding.repeatString(getIndentation(), level),
				force_nl = (profile.tag_nl === true),
				should_break = shouldBreakLine(item, profile);
			
			// formatting block-level elements
			if (item.type != 'text') {
				if (( (item.isBlock() || should_break) && item.parent) || force_nl) {
					// snippet children should take different formatting
					if (!item.parent || (item.parent.type != 'snippet' && !isVeryFirstChild(item)))
						item.start = getNewline() + padding + item.start;
						
					if (item.hasBlockChildren() || shouldBreakChild(item, profile) || (force_nl && !is_unary))
						item.end = getNewline() + padding + item.end;
						
					if (item.hasTagsInContent() || (force_nl && !item.hasChildren() && !is_unary))
						item.start += getNewline() + padding + getIndentation();
					
				} else if (item.isInline() && hasBlockSibling(item) && !isVeryFirstChild(item)) {
					item.start = getNewline() + padding + item.start;
				} else if (item.isInline() && item.hasBlockChildren()) {
					item.end = getNewline() + padding + item.end;
				}
				
				item.padding = padding + getIndentation();
			}
		}
		
		return item;
	}
	
	/**
	 * Processes simplified tree, making it suitable for output as HTML structure
	 * @param {ZenNode} tree
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function process(tree, profile, level) {
		level = level || 0;
		
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			item = (item.type == 'tag') 
				? processTag(item, profile, level) 
				: processSnippet(item, profile, level);
				
			if (item.content)
				item.content = zen_coding.padString(item.content, item.padding);
				
			process(item, profile, level + 1);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('_format', process);
})();/**
 * Filter that produces HAML tree
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 * 
 * @include "../zen_coding.js"
 */
(function(){
	var child_token = '${child}';
	
	/**
	 * Creates HTML attributes string from tag according to profile settings
	 * @param {ZenNode} tag
	 * @param {default_profile} profile
	 */
	function makeAttributesString(tag, profile) {
		// make attribute string
		var attrs = '',
			attr_quote = profile.attr_quotes == 'single' ? "'" : '"',
			cursor = profile.place_cursor ? zen_coding.getCaretPlaceholder() : '',
			attr_name, 
			i,
			a;
			
		// use short notation for ID and CLASS attributes
		for (i = 0; i < tag.attributes.length; i++) {
			a = tag.attributes[i];
			switch (a.name.toLowerCase()) {
				case 'id':
					attrs += '#' + (a.value || cursor);
					break;
				case 'class':
					attrs += '.' + (a.value || cursor);
					break;
			}
		}
		
		var other_attrs = [];
		
		// process other attributes
		for (i = 0; i < tag.attributes.length; i++) {
			a = tag.attributes[i];
			var attr_name_lower = a.name.toLowerCase();
			if (attr_name_lower != 'id' && attr_name_lower != 'class') {
				attr_name = (profile.attr_case == 'upper') ? a.name.toUpperCase() : attr_name_lower;
				other_attrs.push(':' +attr_name + ' => ' + attr_quote + (a.value || cursor) + attr_quote);
			}
		}
		
		if (other_attrs.length)
			attrs += '{' + other_attrs.join(', ') + '}';
		
		return attrs;
	}
	
	/**
	 * Processes element with <code>snippet</code> type
	 * @param {ZenNode} item
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function processSnippet(item, profile, level) {
		var data = item.source.value;
			
		if (!data)
			// snippet wasn't found, process it as tag
			return processTag(item, profile, level);
			
		var parts = data.split(child_token),
			start = parts[0] || '',
			end = parts[1] || '',
			padding = item.parent ? item.parent.padding : '';
			
		item.start = item.start.replace('%s', zen_coding.padString(start, padding));
		item.end = item.end.replace('%s', zen_coding.padString(end, padding));
		
		// replace variables ID and CLASS
		var cb = function(str, var_name) {
			if (var_name == 'id' || var_name == 'class')
				return item.getAttribute(var_name);
			else
				return str;
		};
		item.start = zen_coding.replaceVariables(item.start, cb);
		item.end = zen_coding.replaceVariables(item.end, cb);
		
		return item;
	}
	
	/**
	 * Test if passed node has block-level sibling element
	 * @param {ZenNode} item
	 * @return {Boolean}
	 */
	function hasBlockSibling(item) {
		return (item.parent && item.parent.hasBlockChildren());
	}
	
	/**
	 * Processes element with <code>tag</code> type
	 * @param {ZenNode} item
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function processTag(item, profile, level) {
		if (!item.name)
			// looks like it's root element
			return item;
		
		var attrs = makeAttributesString(item, profile), 
			content = '', 
			cursor = profile.place_cursor ? zen_coding.getCaretPlaceholder() : '',
			self_closing = '',
			is_unary = (item.isUnary() && !item.children.length),
			start= '',
			end = '';
		
		if (profile.self_closing_tag && is_unary)
			self_closing = '/';
			
		// define tag name
		var tag_name = '%' + ((profile.tag_case == 'upper') ? item.name.toUpperCase() : item.name.toLowerCase());
		if (tag_name.toLowerCase() == '%div' && attrs && attrs.indexOf('{') == -1)
			// omit div tag
			tag_name = '';
			
		item.end = '';
		start = tag_name + attrs + self_closing;
		
		var placeholder = '%s';
		// We can't just replace placeholder with new value because
		// JavaScript will treat double $ character as a single one, assuming
		// we're using RegExp literal. 
		var pos = item.start.indexOf(placeholder);
		item.start = item.start.substring(0, pos) + start + item.start.substring(pos + placeholder.length);
		
		if (!item.children.length && !is_unary)
			item.start += cursor;
		
		return item;
	}
	
	/**
	 * Processes simplified tree, making it suitable for output as HTML structure
	 * @param {ZenNode} tree
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function process(tree, profile, level) {
		level = level || 0;
		if (level == 0)
			// preformat tree
			tree = zen_coding.runFilters(tree, profile, '_format');
		
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			item = (item.type == 'tag') 
				? processTag(item, profile, level) 
				: processSnippet(item, profile, level);
			
			// replace counters
			var counter = zen_coding.getCounterForNode(item);
			item.start = zen_coding.unescapeText(zen_coding.replaceCounter(item.start, counter));
			item.end = zen_coding.unescapeText(zen_coding.replaceCounter(item.end, counter));
			
			process(item, profile, level + 1);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('haml', process);
})();/**
 * Filter that produces HTML tree
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 * 
 * @include "../zen_coding.js"
 */
(function(){
	var child_token = '${child}',
		tabstops = 0;
		
	/**
	 * Returns proper string case, depending on profile value
	 * @param {String} val String to process
	 * @param {String} case_param Profile's case value ('lower', 'upper', 'leave')
	 */
	function processStringCase(val, case_param) {
		switch (String(case_param || '').toLowerCase()) {
			case 'lower':
				return val.toLowerCase();
			case 'upper':
				return val.toUpperCase();
		}
		
		return val;
	}
	
	/**
	 * Creates HTML attributes string from tag according to profile settings
	 * @param {ZenNode} tag
	 * @param {default_profile} profile
	 */
	function makeAttributesString(tag, profile) {
		// make attribute string
		var attrs = '',
			attr_quote = profile.attr_quotes == 'single' ? "'" : '"',
			cursor = profile.place_cursor ? zen_coding.getCaretPlaceholder() : '',
			attr_name;
			
		for (var i = 0; i < tag.attributes.length; i++) {
			var a = tag.attributes[i];
			attr_name = processStringCase(a.name, profile.attr_case);
			attrs += ' ' + attr_name + '=' + attr_quote + (a.value || cursor) + attr_quote;
		}
		
		return attrs;
	}
	
	/**
	 * Processes element with <code>snippet</code> type
	 * @param {ZenNode} item
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function processSnippet(item, profile, level) {
		var data = item.source.value;
			
		if (!data)
			// snippet wasn't found, process it as tag
			return processTag(item, profile, level);
			
		var parts = data.split(child_token),
			start = parts[0] || '',
			end = parts[1] || '',
			padding = item.parent ? item.parent.padding : '';
			
			
		item.start = item.start.replace('%s', zen_coding.padString(start, padding));
		item.end = item.end.replace('%s', zen_coding.padString(end, padding));
		
		// replace variables ID and CLASS
		var cb = function(str, var_name) {
			if (var_name == 'id' || var_name == 'class')
				return item.getAttribute(var_name);
			else
				return str;
		};
		item.start = zen_coding.replaceVariables(item.start, cb);
		item.end = zen_coding.replaceVariables(item.end, cb);
		
		return item;
	}
	
	/**
	 * Test if passed node has block-level sibling element
	 * @param {ZenNode} item
	 * @return {Boolean}
	 */
	function hasBlockSibling(item) {
		return (item.parent && item.parent.hasBlockChildren());
	}
	
	/**
	 * Processes element with <code>tag</code> type
	 * @param {ZenNode} item
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function processTag(item, profile, level) {
		if (!item.name)
			// looks like it's root element
			return item;
		
		var attrs = makeAttributesString(item, profile), 
			content = '', 
			cursor = profile.place_cursor ? zen_coding.getCaretPlaceholder() : '',
			self_closing = '',
			is_unary = (item.isUnary() && !item.children.length),
			start= '',
			end = '';
		
		if (profile.self_closing_tag == 'xhtml')
			self_closing = ' /';
		else if (profile.self_closing_tag === true)
			self_closing = '/';
			
		// define opening and closing tags
		if (item.type != 'text') {
			var tag_name = processStringCase(item.name, profile.tag_case);
			if (is_unary) {
				start = '<' + tag_name + attrs + self_closing + '>';
				item.end = '';
			} else {
				start = '<' + tag_name + attrs + '>';
				end = '</' + tag_name + '>';
			}
		}
		
		var placeholder = '%s';
		// We can't just replace placeholder with new value because
		// JavaScript will treat double $ character as a single one, assuming
		// we're using RegExp literal. 
		var pos = item.start.indexOf(placeholder);
		item.start = item.start.substring(0, pos) + start + item.start.substring(pos + placeholder.length);
		
		pos = item.end.indexOf(placeholder);
		item.end = item.end.substring(0, pos) + end + item.end.substring(pos + placeholder.length);
		
		if (!item.children.length && !is_unary && item.content.indexOf(cursor) == -1)
			item.start += cursor;
		
		return item;
	}
	
	/**
	 * Processes simplified tree, making it suitable for output as HTML structure
	 * @param {ZenNode} tree
	 * @param {Object} profile
	 * @param {Number} [level] Depth level
	 */
	function process(tree, profile, level) {
		level = level || 0;
		if (level == 0) {
			tree = zen_coding.runFilters(tree, profile, '_format');
			tabstops = 0;
		}
		
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
	
			var item = tree.children[i];
			item = (item.type == 'tag') 
				? processTag(item, profile, level) 
				: processSnippet(item, profile, level);
			
			// replace counters
			var counter = zen_coding.getCounterForNode(item);
			item.start = zen_coding.unescapeText(zen_coding.replaceCounter(item.start, counter));
			item.end = zen_coding.unescapeText(zen_coding.replaceCounter(item.end, counter));
			item.content = zen_coding.unescapeText(zen_coding.replaceCounter(item.content, counter));
			
			tabstops += zen_coding.upgradeTabstops(item, tabstops) + 1;
			
			process(item, profile, level + 1);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('html', process);
})();/**
 * Output abbreviation on a single line (i.e. no line breaks)
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	function process(tree, profile, level) {
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			if (item.type == 'tag') {
				// remove padding from item 
				var re_pad = /^\s+/;
				item.start = item.start.replace(re_pad, '');
				item.end = item.end.replace(re_pad, '');
			}
			
			// remove newlines 
			var re_nl = /[\n\r]/g;
			item.start = item.start.replace(re_nl, '');
			item.end = item.end.replace(re_nl, '');
			item.content = item.content.replace(re_nl, '');
			
			process(item);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('s', process);
})();
/**
 * Trim filter: removes characters at the beginning of the text
 *  content that indicates lists: numbers, #, *, -, etc.
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	function process(tree, profile, level) {
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			
			if (item.content)
				item.content = item.content.replace(/^([\s|\u00a0])?[\d|#|\-|\*|\u2022]+\.?\s*/, '$1');
			
			process(item);
		}
		
		return tree;
	}
	
	zen_coding.registerFilter('t', process);
})();
/**
 * Filter for trimming "select" attributes from some tags that contains
 * child elements
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
(function(){
	var tags = {
		'xsl:variable': 1,
		'xsl:with-param': 1
	};
	
	/**
	 * Removes "select" attribute from node
	 * @param {ZenNode} node
	 */
	function trimAttribute(node) {
		node.start = node.start.replace(/\s+select\s*=\s*(['"]).*?\1/, '');
	}
	
	function process(tree) {
		for (var i = 0, il = tree.children.length; i < il; i++) {
			/** @type {ZenNode} */
			var item = tree.children[i];
			if (item.type == 'tag' && item.name.toLowerCase() in tags && item.children.length)
				trimAttribute(item);
			process(item);
		}
	}
	
	zen_coding.registerFilter('xsl', process);
})();/**
 * Tests if passed keydown/keypress event corresponds to defied shortcut 
 * 
 * Based on http://www.openjs.com/scripts/events/keyboard_shortcuts/
 * By Binny V A
 * License : BSD
 */
var shortcut = (function(){
	var is_opera = !!window.opera,
		is_mac = /mac\s+os/i.test(navigator.userAgent),
		//Work around for stupid Shift key bug created by using lowercase - as a result the shift+num combination was broken
		shift_nums = {
			"`":"~",
			"1":"!",
			"2":"@",
			"3":"#",
			"4":"$",
			"5":"%",
			"6":"^",
			"7":"&",
			"8":"*",
			"9":"(",
			"0":")",
			"-":"_",
			"=":"+",
			";":":",
			"'":"\"",
			",":"<",
			".":">",
			"/":"?",
			"\\":"|"
		},
		
		//Special Keys - and their codes
		special_keys = {
			'esc':27,
			'escape':27,
			'tab':9,
			'space':32,
			'return':13,
			'enter':13,
			'backspace':8,

			'scrolllock':145,
			'scroll_lock':145,
			'scroll':145,
			'capslock':20,
			'caps_lock':20,
			'caps':20,
			'numlock':144,
			'num_lock':144,
			'num':144,
			
			'pause':19,
			'break':19,
			
			'insert':45,
			'home':36,
			'delete':46,
			'end':35,
			
			'pageup':33,
			'page_up':33,
			'pu':33,

			'pagedown':34,
			'page_down':34,
			'pd':34,
			
			'plus': 187,
			'minus': 189,

			'left':37,
			'up':38,
			'right':39,
			'down':40,

			'f1':112,
			'f2':113,
			'f3':114,
			'f4':115,
			'f5':116,
			'f6':117,
			'f7':118,
			'f8':119,
			'f9':120,
			'f10':121,
			'f11':122,
			'f12':123
		},
		
		mac_char_map = {
			'ctrl': '⌃',
			'control': '⌃',
			'meta': '⌘',
			'shift': '⇧',
			'alt': '⌥',
			'enter': '⏎',
			'tab': '⇥',
			'left': '←',
			'right': '→',
			'up': '↑',
			'down': '↓'
		},
		
		pc_char_map = {
			'meta': 'Ctrl',
			'control': 'Ctrl',
			'left': '←',
			'right': '→',
			'up': '↑',
			'down': '↓'
		},
		
		MODIFIERS = {
			SHIFT: 1,
			CTRL:  2,
			ALT:   4,
			META:  8
		};
		
	/**
	 * Makes first letter of string in uppercase
	 * @param {String} str
	 */
	function capitalize(str) {
		return str.charAt().toUpperCase() + str.substring(1);
	}
		
	return {
		/**
		 * Compile keyboard combination for faster tests
		 * @param {String|Object} combination
		 */
		compile: function(combination) {
			if (typeof combination != 'string') //already compiled
				return combination;
				
			var mask = 0,
				keys = combination.toLowerCase().split('+'), 
				key,
				k;
				
			for(var i = 0, il = keys.length; i < il; i++) {
				k = keys[i];
				// Due to stupid Opera bug I have to swap Ctrl and Meta keys
				if (is_mac && is_opera) {
					if (k == 'ctrl' || k == 'control')
						k = 'meta';
					else if (k == 'meta')
						k = 'ctrl';
				} else if (!is_mac && k == 'meta') {
					k = 'ctrl';
				}
				
				//Modifiers
				if(k == 'ctrl' || k == 'control')
					mask |= MODIFIERS.CTRL;
				else if (k == 'shift')
					mask |= MODIFIERS.SHIFT;
				else if (k == 'alt')
					mask |= MODIFIERS.ALT;
				else if (k == 'meta')
					mask |= MODIFIERS.META;
				else
					key = k;
			}
			
			return {
				mask: mask,
				key: key
			};
		},
		
		/**
		 * Test shortcut combination against event
		 * @param {String} combination Keyboard shortcut
		 * @param {Event} evt
		 */
		test: function(combination, evt) {
			var mask = 0,
				ccomb = this.compile(combination);
			
			if (evt.ctrlKey)  mask |= MODIFIERS.CTRL;
			if (evt.shiftKey) mask |= MODIFIERS.SHIFT;
			if (evt.altKey)   mask |= MODIFIERS.ALT;
			if (evt.metaKey)  mask |= MODIFIERS.META;
			
			var code = evt.keyCode ? evt.keyCode : evt.which,
				character = String.fromCharCode(code).toLowerCase();
			
			// if mask doesn't match, no need to test further
			if (mask !== ccomb.mask) return false;
			
			if (ccomb.key.length > 1) { //If it is a special key
				return special_keys[ccomb.key] == code;
			} else { //The special keys did not match
				if(code == 188) character = ","; //If the user presses , when the type is onkeydown
				if(code == 190) character = ".";
				if(code == 191) character = "/";
				
				if (character == ccomb.key) return true;
				if (evt.shiftKey && shift_nums[character]) //Stupid Shift key bug created by using lowercase
					return shift_nums[character] == ccomb.key;
			}
			
			return false;
		},
		
		/**
		 * Format keystroke for better readability, considering current platform
		 * mnemonics
		 * @param {String} keystroke
		 * @return {String}
		 */
		format: function(keystroke) {
			var char_map = is_mac ? mac_char_map : pc_char_map,
				glue = is_mac ? '' : '+',
				keys = keystroke.toLowerCase().split('+'),
				ar = [],
				key;
				
			for (var i = 0; i < keys.length; i++) {
				key = keys[i];
				ar.push(key in char_map ? char_map[key] : capitalize(key));
			}
			
			return ar.join(glue);
		}
	};
})();/**
 * High-level editor interface that communicates with underlying editor (like 
 * TinyMCE, CKEditor, etc.) or browser.
 * Basically, you should call <code>zen_editor.setContext(obj)</code> method to
 * set up undelying editor context before using any other method.
 * 
 * This interface is used by <i>zen_actions.js</i> for performing different 
 * actions like <b>Expand abbreviation</b>  
 * 
 * @example
 * var textarea = document.getElemenetsByTagName('textarea')[0];
 * zen_editor.setContext(textarea);
 * //now you are ready to use editor object
 * zen_editor.getSelectionRange();
 * 
 * @author Sergey Chikuyonok (serge.che@gmail.com)
 * @link http://chikuyonok.ru
 */
var zen_editor = (function(){
	/** @type {CodeMirror} */
	var mirror,
		
		/** Actions aliases */
		aliases = {
			balance_tag_inward: 'match_pair_inward',
			balance_tag_outward: 'match_pair_outward',
			previous_edit_point: 'prev_edit_point',
			pretty_break: 'insert_formatted_line_break'
		},
		
		shortcuts = {};
	
	/**
	 * Returns whitrespace padding of string
	 * @param {String} str String line
	 * @return {String}
	 */
	function getStringPadding(str) {
		return (str.match(/^(\s+)/) || [''])[0];
	}
	
	/**
	 * Handle tab-stops (like $1 or ${1:label}) inside text: find first tab-stop,
	 * marks it as selection, remove the rest. If tab-stop wasn't found, search
	 * for caret placeholder and use it as selection
	 * @param {String} text
	 * @return {Array} Array with new text and selection indexes (['...', -1,-1] 
	 * if there's no selection)
	 */
	function handleTabStops(text) {
		var selection_len = 0,
			caret_pos = text.indexOf(zen_coding.getCaretPlaceholder()),
			placeholders = {};
			
		// find caret position
		if (caret_pos != -1) {
			text = text.split(zen_coding.getCaretPlaceholder()).join('');
		} else {
			caret_pos = text.length;
		}
		
		text = zen_coding.processTextBeforePaste(text, 
			function(ch){ return ch; }, 
			function(i, num, val) {
				if (val) placeholders[num] = val;
				
				if (i < caret_pos) {
					caret_pos = i;
					if (val)
						selection_len = val.length;
				}
					
				return placeholders[num] || '';
			});
		
		return [text, caret_pos, caret_pos + selection_len];
	}
	
	function keysFilter(evt) {
		// test if occured event corresponds to one of the defined shortcut
		var sh, name, result;
		for (var s in shortcuts) if (shortcuts.hasOwnProperty(s)) {
			sh = shortcuts[s];
			if (shortcut.test(sh.compiled, evt)) {
				evt.preventDefault();
				name = aliases[sh.action] || sh.action;
				result = zen_coding.runAction(name, [zen_editor]);
				return (name == 'expand_abbreviation') ? result : true;
			}
		}
		
		return false;
	}
	
	/**
	 * Returns normalized action name
	 * @param {String} name Action name (like 'Expand Abbreviation')
	 * @return Normalized name for coding (like 'expand_abbreviation')
	 */
	function normalizeActionName(name) {
		return name
			.replace(/(^\s+|\s+$)/g, '') // remove trailing spaces
			.replace(/[\s\\\/]+/g, '_')
			.replace(/\./g, '')
			.toLowerCase();
	}
	
	/**
	 * Bind shortcut to Zen Coding action
	 * @param {String} keystroke
	 * @param {String} label
	 * @param {String} action_name
	 */
	function addShortcut(keystroke, label, action_name) {
		shortcuts[keystroke.toLowerCase()] = {
			compiled: shortcut.compile(keystroke),
			label: label,
			action: normalizeActionName(action_name || label)
		};
	}
	
	// add default shortcuts
	addShortcut('Meta+E', 'Expand Abbreviation');
	addShortcut('Tab', 'Expand Abbreviation');
	addShortcut('Meta+D', 'Balance Tag Outward');
	addShortcut('Shift+Meta+D', 'Balance Tag inward');
	addShortcut('Shift+Meta+A', 'Wrap with Abbreviation');
	addShortcut('Ctrl+Alt+RIGHT', 'Next Edit Point');
	addShortcut('Ctrl+Alt+LEFT', 'Previous Edit Point');
	addShortcut('Meta+L', 'Select Line');
	addShortcut('Meta+Shift+M', 'Merge Lines');
	addShortcut('Meta+/', 'Toggle Comment');
	addShortcut('Meta+J', 'Split/Join Tag');
	addShortcut('Meta+K', 'Remove Tag');
	addShortcut('Meta+Shift+Y', 'Evaluate Math Expression');
	
	addShortcut('Ctrl+UP', 'Increment number by 1');
	addShortcut('Ctrl+DOWN', 'Decrement number by 1');
	addShortcut('Alt+UP', 'Increment number by 0.1');
	addShortcut('Alt+DOWN', 'Decrement number by 0.1');
	addShortcut('Ctrl+Alt+UP', 'Increment number by 10');
	addShortcut('Ctrl+Alt+DOWN', 'Decrement number by 10');
	
	addShortcut('Meta+.', 'Select Next Item');
	addShortcut('Meta+,', 'Select Previous Item');
	addShortcut('Meta+B', 'Reflect CSS Value');
	
	return {
		/**
		 * Key handle for CodeMirror key events. You should pass this method as
		 * <code>onKeyEvent</code> handler in order to add Zen Coding 
		 * functionality to CodeMirror editor instance
		 */
		handleKeyEvent: function(editor, evt) {
			if (evt.type == 'keydown') {
				this.setContext(editor);
				return keysFilter(evt);
			}
		},
		
		/**
		 * Setup underlying editor context. You should call this method 
		 * <code>before</code> using any Zen Coding action.
		 * @param {CodeMirror} context
		 */
		setContext: function(context) {
			mirror = context;
			zen_coding.setVariable('indentation', zen_coding.repeatString(' ', mirror.getOption('indentUnit')));
		},
		
		/**
		 * Returns character indexes of selected text: object with <code>start</code>
		 * and <code>end</code> properties. If there's no selection, should return 
		 * object with <code>start</code> and <code>end</code> properties referring
		 * to current caret position
		 * @return {Object}
		 * @example
		 * var selection = zen_editor.getSelectionRange();
		 * alert(selection.start + ', ' + selection.end); 
		 */
		getSelectionRange: function() {
			var caret_pos = this.getCaretPos();
			return {
				start: caret_pos,
				end: caret_pos + this.getSelection().length
			};
		},
		
		/**
		 * Creates selection from <code>start</code> to <code>end</code> character
		 * indexes. If <code>end</code> is ommited, this method should place caret 
		 * and <code>start</code> index
		 * @param {Number} start
		 * @param {Number} [end]
		 * @example
		 * zen_editor.createSelection(10, 40);
		 * 
		 * //move caret to 15th character
		 * zen_editor.createSelection(15);
		 */
		createSelection: function(start, end) {
			if (start == end) {
				mirror.setCursor(mirror.posFromIndex(start));
			} else {
				mirror.setSelection(mirror.posFromIndex(start), mirror.posFromIndex(end));
			}
		},
		
		/**
		 * Returns current line's start and end indexes as object with <code>start</code>
		 * and <code>end</code> properties
		 * @return {Object}
		 * @example
		 * var range = zen_editor.getCurrentLineRange();
		 * alert(range.start + ', ' + range.end);
		 */
		getCurrentLineRange: function() {
			var caret = mirror.getCursor(true);
			
			return {
				start: mirror.indexFromPos({line: caret.line, ch: 0}),
				end: mirror.indexFromPos({line: caret.line, ch: mirror.getLine(caret.line).length})
			};
		},
		
		/**
		 * Returns current caret position
		 * @return {Number|null}
		 */
		getCaretPos: function(){
			return mirror.indexFromPos(mirror.getCursor(true));
		},
		
		/**
		 * Set new caret position
		 * @param {Number} pos Caret position
		 */
		setCaretPos: function(pos){
//			mirror.setCursor(mirror.posFromIndex(pos));
			this.createSelection(pos, pos);
		},
		
		/**
		 * Returns content of current line
		 * @return {String}
		 */
		getCurrentLine: function() {
			return mirror.getLine( mirror.getCursor(true).line ) || '';
		},
		
		/**
		 * Replace editor's content or it's part (from <code>start</code> to 
		 * <code>end</code> index). If <code>value</code> contains 
		 * <code>caret_placeholder</code>, the editor will put caret into 
		 * this position. If you skip <code>start</code> and <code>end</code>
		 * arguments, the whole target's content will be replaced with 
		 * <code>value</code>. 
		 * 
		 * If you pass <code>start</code> argument only,
		 * the <code>value</code> will be placed at <code>start</code> string 
		 * index of current content. 
		 * 
		 * If you pass <code>start</code> and <code>end</code> arguments,
		 * the corresponding substring of current target's content will be 
		 * replaced with <code>value</code>. 
		 * @param {String} value Content you want to paste
		 * @param {Number} [start] Start index of editor's content
		 * @param {Number} [end] End index of editor's content
		 */
		replaceContent: function(value, start, end) {
			var caret_pos = this.getCaretPos(),
				start = typeof(start) !== 'undefined' ? start : 0,
				end = typeof(end) !== 'undefined' ? end : this.getContent().length;
				
			// indent new value
			value = zen_coding.padString(value, getStringPadding(this.getCurrentLine()));
			
//			if (mirror.win.asEditorLines) {
//				value = mirror.win.asEditorLines(value).join(zen_coding.getNewline());
//			}
			
			// find new caret position
			var tabstop_res = handleTabStops(value);
			value = tabstop_res[0];
			
			start = start || 0;
			if (tabstop_res[1] !== -1) {
				tabstop_res[1] += start;
				tabstop_res[2] += start;
			} else {
				tabstop_res[1] = tabstop_res[2] = value.length + start;
			}
			
			try {
				mirror.replaceRange(value, mirror.posFromIndex(start), mirror.posFromIndex(end));
				// this.createSelection(start, end);
				// mirror.replaceSelection(value);
				this.createSelection(tabstop_res[1], tabstop_res[2]);
			} catch(e){}
		},
		
		/**
		 * Returns editor's content
		 * @return {String}
		 */
		getContent: function(){
			return mirror.getValue();
		},
		
		/**
		 * Returns current editor's syntax mode
		 * @return {String}
		 */
		getSyntax: function() {
			var syntax = mirror.getOption('syntax') ||  '',
				caret_pos = this.getCaretPos();
				
			if (!zen_resources.hasSyntax(syntax))
				syntax = 'html';
				
			if (syntax == 'html') {
				// get the context tag
				var pair = zen_coding.html_matcher.getTags(this.getContent(), caret_pos);
				if (pair && pair[0] && pair[0].type == 'tag' && pair[0].name.toLowerCase() == 'style') {
					// check that we're actually inside the tag
					if (pair[0].end <= caret_pos && pair[1].start >= caret_pos)
						syntax = 'css';
				}
			}
			
			return syntax;
		},
		
		/**
		 * Returns current output profile name (@see zen_coding#setupProfile)
		 * @return {String}
		 */
		getProfileName: function() {
			if (mirror.getOption('profile'))
				return mirror.getOption('profile');
				
			switch(this.getSyntax()) {
				 case 'xml':
				 case 'xsl':
				 	return 'xml';
				 case 'html':
				 	var profile = zen_coding.getVariable('profile');
				 	if (!profile) { // no forced profile, guess from content
					 	// html or xhtml?
				 		profile = this.getContent().search(/<!DOCTYPE[^>]+XHTML/) != -1 ? 'xhtml': 'html';
				 	}
				 	
				 	return profile;
			}
			
			return 'xhtml';
		},
		
		/**
		 * Ask user to enter something
		 * @param {String} title Dialog title
		 * @return {String} Entered data
		 * @since 0.65
		 */
		prompt: function(title) {
			return prompt(title);
		},
		
		/**
		 * Returns current selection
		 * @return {String}
		 * @since 0.65
		 */
		getSelection: function() {
			return mirror.getSelection() || '';
		},
		
		/**
		 * Returns current editor's file path
		 * @return {String}
		 * @since 0.65 
		 */
		getFilePath: function() {
			return location.href;
		},
		
		shortcut: addShortcut,
		
		/**
		 * Removes shortcut binding
		 * @param {String} keystroke
		 */
		unbindShortcut: function(keystroke) {
			keystroke = keystroke.toLowerCase();
			if (keystroke in shortcuts)
				delete shortcuts[keystroke];
		},
				
		/**
		 * Returns array of binded actions and their keystrokes
		 * @return {Array}
		 */
		getShortcuts: function() {
			var result = [], lp;
			
			for (var p in shortcuts) if (shortcuts.hasOwnProperty(p)) {
				lp = p.toLowerCase();
				
				// skip some internal bindings
				if (lp == 'tab' || lp == 'enter')
					continue;
					
				result.push({
					keystroke: shortcut.format(p), 
					compiled: shortcuts[p].compiled,
					label: shortcuts[p].label,
					action: shortcuts[p].action
				});
			}
			
			return result;
		},
		
		getInfo: function() {
			var message = 'This CodeMirror editor is powered by Zen Coding project: ' +
					'a set of tools for fast HTML coding.\n\n' +
					'Available shortcuts:\n';
					
			var sh = this.getShortcuts(),
				actions = [];
				
			for (var i = 0; i < sh.length; i++) {
				actions.push(sh[i].keystroke + ' — ' + sh[i].label)
			}
			
			message += actions.join('\n') + '\n\n';
			message += 'More info on http://code.google.com/p/zen-coding/';
			
			return message;
		},
		
		/**
		 * Show info window about Zen Coding
		 */
		showInfo: function() {
			alert(this.getInfo());
		},
		
		// expose some core Zen Coding objects
		
		/**
		 * Returns core Zen Codind object
		 */
		getCore: function() {
			return zen_coding;
		},
		
		/**
		 * Returns Zen Coding resource manager. You can add new snippets and 
		 * abbreviations with this manager, as well as modify ones.<br><br>
		 * 
		 * Zen Coding stores settings in two separate vocabularies: 'system' 
		 * and 'user'. The ultimate solution to add new abbreviations and
		 * snippets is to setup a 'user' vocabulary, like this:
		 * 
		 * @example
		 * var my_settings = {
		 * 	html: {
		 * 		abbreviations: {
		 * 			'tag': '<div class="mytag">'
		 * 		}
		 * 	}
		 * };
		 * zen_editor.getResourceManager().setVocabulary(my_settings, 'user')
		 * 
		 * @see zen_resources.js
		 */
		getResourceManager: function() {
			return zen_resources;
		}
	}
})();return zen_editor;	
})();(function(){function h(a,b,c){if(a===b)return a!==0||1/a==1/b;if(a==null||b==null)return a===b;if(a._chain)a=a._wrapped;if(b._chain)b=b._wrapped;if(a.isEqual&&j.isFunction(a.isEqual))return a.isEqual(b);if(b.isEqual&&j.isFunction(b.isEqual))return b.isEqual(a);var e=i.call(a);if(e!=i.call(b))return!1;switch(e){case "[object String]":return a==String(b);case "[object Number]":return a!=+a?b!=+b:a==0?1/a==1/b:a==+b;case "[object Date]":case "[object Boolean]":return+a==+b;case "[object RegExp]":return a.source==
b.source&&a.global==b.global&&a.multiline==b.multiline&&a.ignoreCase==b.ignoreCase}if(typeof a!="object"||typeof b!="object")return!1;for(var d=c.length;d--;)if(c[d]==a)return!0;c.push(a);var d=0,f=!0;if(e=="[object Array]"){if(d=a.length,f=d==b.length)for(;d--;)if(!(f=d in a==d in b&&h(a[d],b[d],c)))break}else{if("constructor"in a!="constructor"in b||a.constructor!=b.constructor)return!1;for(var m in a)if(j.has(a,m)&&(d++,!(f=j.has(b,m)&&h(a[m],b[m],c))))break;if(f){for(m in b)if(j.has(b,m)&&!d--)break;
f=!d}}c.pop();return f}var d=this,g=d._,f={},b=Array.prototype,c=Object.prototype,a=b.slice,e=b.unshift,i=c.toString,k=c.hasOwnProperty,l=b.forEach,n=b.map,o=b.reduce,q=b.reduceRight,r=b.filter,t=b.every,m=b.some,y=b.indexOf,w=b.lastIndexOf,c=Array.isArray,z=Object.keys,A=Function.prototype.bind,j=function(a){return new B(a)};if(typeof exports!=="undefined"){if(typeof module!=="undefined"&&module.exports)exports=module.exports=j;exports._=j}else d._=j;j.VERSION="1.3.3";var u=j.each=j.forEach=function(a,
b,c){if(a!=null)if(l&&a.forEach===l)a.forEach(b,c);else if(a.length===+a.length)for(var e=0,d=a.length;e<d;e++){if(e in a&&b.call(c,a[e],e,a)===f)break}else for(e in a)if(j.has(a,e)&&b.call(c,a[e],e,a)===f)break};j.map=j.collect=function(a,b,c){var e=[];if(a==null)return e;if(n&&a.map===n)return a.map(b,c);u(a,function(a,d,i){e[e.length]=b.call(c,a,d,i)});if(a.length===+a.length)e.length=a.length;return e};j.reduce=j.foldl=j.inject=function(a,b,c,e){var d=arguments.length>2;a==null&&(a=[]);if(o&&
a.reduce===o)return e&&(b=j.bind(b,e)),d?a.reduce(b,c):a.reduce(b);u(a,function(a,i,f){d?c=b.call(e,c,a,i,f):(c=a,d=!0)});if(!d)throw new TypeError("Reduce of empty array with no initial value");return c};j.reduceRight=j.foldr=function(a,b,c,e){var d=arguments.length>2;a==null&&(a=[]);if(q&&a.reduceRight===q)return e&&(b=j.bind(b,e)),d?a.reduceRight(b,c):a.reduceRight(b);var i=j.toArray(a).reverse();e&&!d&&(b=j.bind(b,e));return d?j.reduce(i,b,c,e):j.reduce(i,b)};j.find=j.detect=function(a,b,c){var e;
F(a,function(a,d,i){if(b.call(c,a,d,i))return e=a,!0});return e};j.filter=j.select=function(a,b,c){var e=[];if(a==null)return e;if(r&&a.filter===r)return a.filter(b,c);u(a,function(a,d,i){b.call(c,a,d,i)&&(e[e.length]=a)});return e};j.reject=function(a,b,c){var e=[];if(a==null)return e;u(a,function(a,d,i){b.call(c,a,d,i)||(e[e.length]=a)});return e};j.every=j.all=function(a,b,c){var e=!0;if(a==null)return e;if(t&&a.every===t)return a.every(b,c);u(a,function(a,d,i){if(!(e=e&&b.call(c,a,d,i)))return f});
return!!e};var F=j.some=j.any=function(a,b,c){b||(b=j.identity);var e=!1;if(a==null)return e;if(m&&a.some===m)return a.some(b,c);u(a,function(a,d,i){if(e||(e=b.call(c,a,d,i)))return f});return!!e};j.include=j.contains=function(a,b){var c=!1;return a==null?c:y&&a.indexOf===y?a.indexOf(b)!=-1:c=F(a,function(a){return a===b})};j.invoke=function(b,c){var e=a.call(arguments,2);return j.map(b,function(a){return(j.isFunction(c)?c||a:a[c]).apply(a,e)})};j.pluck=function(a,b){return j.map(a,function(a){return a[b]})};
j.max=function(a,b,c){if(!b&&j.isArray(a)&&a[0]===+a[0])return Math.max.apply(Math,a);if(!b&&j.isEmpty(a))return-Infinity;var e={computed:-Infinity};u(a,function(a,d,i){d=b?b.call(c,a,d,i):a;d>=e.computed&&(e={value:a,computed:d})});return e.value};j.min=function(a,b,c){if(!b&&j.isArray(a)&&a[0]===+a[0])return Math.min.apply(Math,a);if(!b&&j.isEmpty(a))return Infinity;var e={computed:Infinity};u(a,function(a,d,i){d=b?b.call(c,a,d,i):a;d<e.computed&&(e={value:a,computed:d})});return e.value};j.shuffle=
function(a){var b=[],c;u(a,function(a,e){c=Math.floor(Math.random()*(e+1));b[e]=b[c];b[c]=a});return b};j.sortBy=function(a,b,c){var e=j.isFunction(b)?b:function(a){return a[b]};return j.pluck(j.map(a,function(a,b,d){return{value:a,criteria:e.call(c,a,b,d)}}).sort(function(a,b){var c=a.criteria,e=b.criteria;return c===void 0?1:e===void 0?-1:c<e?-1:c>e?1:0}),"value")};j.groupBy=function(a,b){var c={},e=j.isFunction(b)?b:function(a){return a[b]};u(a,function(a,b){var d=e(a,b);(c[d]||(c[d]=[])).push(a)});
return c};j.sortedIndex=function(a,b,c){c||(c=j.identity);for(var e=0,d=a.length;e<d;){var i=e+d>>1;c(a[i])<c(b)?e=i+1:d=i}return e};j.toArray=function(b){return!b?[]:j.isArray(b)?a.call(b):j.isArguments(b)?a.call(b):b.toArray&&j.isFunction(b.toArray)?b.toArray():j.values(b)};j.size=function(a){return j.isArray(a)?a.length:j.keys(a).length};j.first=j.head=j.take=function(b,c,e){return c!=null&&!e?a.call(b,0,c):b[0]};j.initial=function(b,c,e){return a.call(b,0,b.length-(c==null||e?1:c))};j.last=function(b,
c,e){return c!=null&&!e?a.call(b,Math.max(b.length-c,0)):b[b.length-1]};j.rest=j.tail=function(b,c,e){return a.call(b,c==null||e?1:c)};j.compact=function(a){return j.filter(a,function(a){return!!a})};j.flatten=function(a,b){return j.reduce(a,function(a,c){if(j.isArray(c))return a.concat(b?c:j.flatten(c));a[a.length]=c;return a},[])};j.without=function(b){return j.difference(b,a.call(arguments,1))};j.uniq=j.unique=function(a,b,c){var c=c?j.map(a,c):a,e=[];a.length<3&&(b=!0);j.reduce(c,function(c,d,
i){if(b?j.last(c)!==d||!c.length:!j.include(c,d))c.push(d),e.push(a[i]);return c},[]);return e};j.union=function(){return j.uniq(j.flatten(arguments,!0))};j.intersection=j.intersect=function(b){var c=a.call(arguments,1);return j.filter(j.uniq(b),function(a){return j.every(c,function(b){return j.indexOf(b,a)>=0})})};j.difference=function(b){var c=j.flatten(a.call(arguments,1),!0);return j.filter(b,function(a){return!j.include(c,a)})};j.zip=function(){for(var b=a.call(arguments),c=j.max(j.pluck(b,"length")),
e=Array(c),d=0;d<c;d++)e[d]=j.pluck(b,""+d);return e};j.indexOf=function(a,b,c){if(a==null)return-1;var e;if(c)return c=j.sortedIndex(a,b),a[c]===b?c:-1;if(y&&a.indexOf===y)return a.indexOf(b);c=0;for(e=a.length;c<e;c++)if(c in a&&a[c]===b)return c;return-1};j.lastIndexOf=function(a,b){if(a==null)return-1;if(w&&a.lastIndexOf===w)return a.lastIndexOf(b);for(var c=a.length;c--;)if(c in a&&a[c]===b)return c;return-1};j.range=function(a,b,c){arguments.length<=1&&(b=a||0,a=0);for(var c=arguments[2]||1,
e=Math.max(Math.ceil((b-a)/c),0),d=0,i=Array(e);d<e;)i[d++]=a,a+=c;return i};var G=function(){};j.bind=function(b,c){var e,d;if(b.bind===A&&A)return A.apply(b,a.call(arguments,1));if(!j.isFunction(b))throw new TypeError;d=a.call(arguments,2);return e=function(){if(!(this instanceof e))return b.apply(c,d.concat(a.call(arguments)));G.prototype=b.prototype;var i=new G,f=b.apply(i,d.concat(a.call(arguments)));return Object(f)===f?f:i}};j.bindAll=function(b){var c=a.call(arguments,1);c.length==0&&(c=j.functions(b));
u(c,function(a){b[a]=j.bind(b[a],b)});return b};j.memoize=function(a,b){var c={};b||(b=j.identity);return function(){var e=b.apply(this,arguments);return j.has(c,e)?c[e]:c[e]=a.apply(this,arguments)}};j.delay=function(b,c){var e=a.call(arguments,2);return setTimeout(function(){return b.apply(null,e)},c)};j.defer=function(b){return j.delay.apply(j,[b,1].concat(a.call(arguments,1)))};j.throttle=function(a,b){var c,e,d,i,f,m,k=j.debounce(function(){f=i=!1},b);return function(){c=this;e=arguments;var g;
d||(d=setTimeout(function(){d=null;f&&a.apply(c,e);k()},b));i?f=!0:m=a.apply(c,e);k();i=!0;return m}};j.debounce=function(a,b,c){var e;return function(){var d=this,i=arguments;c&&!e&&a.apply(d,i);clearTimeout(e);e=setTimeout(function(){e=null;c||a.apply(d,i)},b)}};j.once=function(a){var b=!1,c;return function(){if(b)return c;b=!0;return c=a.apply(this,arguments)}};j.wrap=function(b,c){return function(){var e=[b].concat(a.call(arguments,0));return c.apply(this,e)}};j.compose=function(){var a=arguments;
return function(){for(var b=arguments,c=a.length-1;c>=0;c--)b=[a[c].apply(this,b)];return b[0]}};j.after=function(a,b){return a<=0?b():function(){if(--a<1)return b.apply(this,arguments)}};j.keys=z||function(a){if(a!==Object(a))throw new TypeError("Invalid object");var b=[],c;for(c in a)j.has(a,c)&&(b[b.length]=c);return b};j.values=function(a){return j.map(a,j.identity)};j.functions=j.methods=function(a){var b=[],c;for(c in a)j.isFunction(a[c])&&b.push(c);return b.sort()};j.extend=function(b){u(a.call(arguments,
1),function(a){for(var c in a)b[c]=a[c]});return b};j.pick=function(b){var c={};u(j.flatten(a.call(arguments,1)),function(a){a in b&&(c[a]=b[a])});return c};j.defaults=function(b){u(a.call(arguments,1),function(a){for(var c in a)b[c]==null&&(b[c]=a[c])});return b};j.clone=function(a){return!j.isObject(a)?a:j.isArray(a)?a.slice():j.extend({},a)};j.tap=function(a,b){b(a);return a};j.isEqual=function(a,b){return h(a,b,[])};j.isEmpty=function(a){if(a==null)return!0;if(j.isArray(a)||j.isString(a))return a.length===
0;for(var b in a)if(j.has(a,b))return!1;return!0};j.isElement=function(a){return!!(a&&a.nodeType==1)};j.isArray=c||function(a){return i.call(a)=="[object Array]"};j.isObject=function(a){return a===Object(a)};j.isArguments=function(a){return i.call(a)=="[object Arguments]"};if(!j.isArguments(arguments))j.isArguments=function(a){return!(!a||!j.has(a,"callee"))};j.isFunction=function(a){return i.call(a)=="[object Function]"};j.isString=function(a){return i.call(a)=="[object String]"};j.isNumber=function(a){return i.call(a)==
"[object Number]"};j.isFinite=function(a){return j.isNumber(a)&&isFinite(a)};j.isNaN=function(a){return a!==a};j.isBoolean=function(a){return a===!0||a===!1||i.call(a)=="[object Boolean]"};j.isDate=function(a){return i.call(a)=="[object Date]"};j.isRegExp=function(a){return i.call(a)=="[object RegExp]"};j.isNull=function(a){return a===null};j.isUndefined=function(a){return a===void 0};j.has=function(a,b){return k.call(a,b)};j.noConflict=function(){d._=g;return this};j.identity=function(a){return a};
j.times=function(a,b,c){for(var e=0;e<a;e++)b.call(c,e)};j.escape=function(a){return(""+a).replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;").replace(/"/g,"&quot;").replace(/'/g,"&#x27;").replace(/\//g,"&#x2F;")};j.result=function(a,b){if(a==null)return null;var c=a[b];return j.isFunction(c)?c.call(a):c};j.mixin=function(a){u(j.functions(a),function(b){H(b,j[b]=a[b])})};var s=0;j.uniqueId=function(a){var b=s++;return a?a+b:b};j.templateSettings={evaluate:/<%([\s\S]+?)%>/g,interpolate:/<%=([\s\S]+?)%>/g,
escape:/<%-([\s\S]+?)%>/g};var v=/.^/,x={"\\":"\\","'":"'",r:"\r",n:"\n",t:"\t",u2028:"\u2028",u2029:"\u2029"},C;for(C in x)x[x[C]]=C;var I=/\\|'|\r|\n|\t|\u2028|\u2029/g,J=/\\(\\|'|r|n|t|u2028|u2029)/g,D=function(a){return a.replace(J,function(a,b){return x[b]})};j.template=function(a,b,c){c=j.defaults(c||{},j.templateSettings);a="__p+='"+a.replace(I,function(a){return"\\"+x[a]}).replace(c.escape||v,function(a,b){return"'+\n_.escape("+D(b)+")+\n'"}).replace(c.interpolate||v,function(a,b){return"'+\n("+
D(b)+")+\n'"}).replace(c.evaluate||v,function(a,b){return"';\n"+D(b)+"\n;__p+='"})+"';\n";c.variable||(a="with(obj||{}){\n"+a+"}\n");var a="var __p='';var print=function(){__p+=Array.prototype.join.call(arguments, '')};\n"+a+"return __p;\n",e=new Function(c.variable||"obj","_",a);if(b)return e(b,j);b=function(a){return e.call(this,a,j)};b.source="function("+(c.variable||"obj")+"){\n"+a+"}";return b};j.chain=function(a){return j(a).chain()};var B=function(a){this._wrapped=a};j.prototype=B.prototype;
var E=function(a,b){return b?j(a).chain():a},H=function(b,c){B.prototype[b]=function(){var b=a.call(arguments);e.call(b,this._wrapped);return E(c.apply(j,b),this._chain)}};j.mixin(j);u(["pop","push","reverse","shift","sort","splice","unshift"],function(a){var c=b[a];B.prototype[a]=function(){var b=this._wrapped;c.apply(b,arguments);var e=b.length;(a=="shift"||a=="splice")&&e===0&&delete b[0];return E(b,this._chain)}});u(["concat","join","slice"],function(a){var c=b[a];B.prototype[a]=function(){return E(c.apply(this._wrapped,
arguments),this._chain)}});B.prototype.chain=function(){this._chain=!0;return this};B.prototype.value=function(){return this._wrapped}}).call(this);
(function(h,d){function g(a,c,i){var f;f=c&&c.hasOwnProperty("constructor")?c.constructor:function(){a.apply(this,arguments)};d.extend(f,a);b.prototype=a.prototype;f.prototype=new b;c&&d.extend(f.prototype,c);i&&d.extend(f,i);f.prototype.constructor=f;f.__super__=a.prototype;return f}var f={_:d},b=function(){},c=null;h.zen_coding={define:function(a,b){a in f||(f[a]=d.isFunction(b)?this.exec(b):b)},require:function(a){!(a in f)&&c&&c(a);return f[a]},exec:function(a,b){return a.call(b||h,d.bind(this.require,
this),d,this)},extend:function(a,b){var c=g(this,a,b);c.extend=this.extend;return c},expandAbbreviation:function(a,b,c,d){if(!a)return"";var b=b||"html",c=c||"plain",f=this.require("filters"),g=this.require("utils"),h=this.require("abbreviationParser"),c=this.require("profile").get(c,b);this.require("tabStops").resetTabstopIndex();a=f.extractFromAbbreviation(a);d=h.parse(a[0],{syntax:b,contextNode:d});b=f.composeList(b,c,a[1]);f.apply(d,b,c);return g.replaceVariables(d.toString())},defaultSyntax:function(){return"html"},
defaultProfile:function(){return"plain"},log:function(){h.console&&h.console.log&&h.console.log.apply(h.console,arguments)},setModuleLoader:function(a){c=a}}})(this,_);
zen_coding.define("abbreviationParser",function(h,d){function g(){this.parent=null;this.children=[];this._attributes=[];this.abbreviation="";this.counter=1;this._name=null;this._text="";this.repeatCount=1;this.hasImplicitRepeat=!1;this._data={};this.padding=this.content=this.end=this.start=""}function f(a){return a.substring(1,a.length-1)}function b(a){for(var a=h("utils").trim(a),c=new g,e=c.addChild(),i,m=h("stringStream").create(a),a=1E3,k;!m.eol()&&--a>0;)switch(i=m.peek(),i){case "(":m.start=
m.pos;if(m.skipToPair("(",")"))i=b(f(m.current())),(k=m.match(/^\*(\d+)?/,!0))&&e._setRepeat(k[1]),d.each(i.children,function(a){e.addChild(a)});else throw'Invalid abbreviation: mo matching ")" found for character at '+m.pos;break;case ">":e=e.addChild();m.next();break;case "+":e=e.parent.addChild();m.next();break;case "^":i=e.parent||e;e=(i.parent||i).addChild();m.next();break;default:m.start=m.pos,m.eatWhile(function(a){if(a=="["||a=="{"){if(m.skipToPair(a,r[a]))return m.backUp(1),!0;throw'Invalid abbreviation: mo matching "'+
r[a]+'" found for character at '+m.pos;}return a=="+"?(m.next(),a=m.eol()||~"+>^*".indexOf(m.peek()),m.backUp(1),a):a!="("&&n(a)}),e.setAbbreviation(m.current()),m.start=m.pos}if(a<1)throw"Endless loop detected";return c}function c(a){var a=h("utils").trim(a),b=[],a=h("stringStream").create(a);for(a.eatSpace();!a.eol();)if(a.start=a.pos,a.eatWhile(q)){var c=a.current(),e="";if(a.peek()=="="){a.next();a.start=a.pos;var d=a.peek();if(d=='"'||d=="'"){a.next();a:{for(var e=a,i=void 0;i=e.next();)if(i===
d){e=!0;break a}e=!1}if(e)e=a.current(),e=e.substring(1,e.length-1);else throw"Invalid attribute value";}else if(a.eatWhile(/[^\s\]]/))e=a.current();else throw"Invalid attribute value";}b.push({name:c,value:e});a.eatSpace()}else break;return b}function a(a){for(var b=[],d={"#":"id",".":"class"},i=null,m=h("stringStream").create(a);!m.eol();)switch(m.peek()){case "#":case ".":if(i===null)i=m.pos;var g=d[m.peek()];m.next();m.start=m.pos;m.eatWhile(q);b.push({name:g,value:m.current()});break;case "[":if(i===
null)i=m.pos;m.start=m.pos;if(!m.skipToPair("[","]"))throw"Invalid attribute set definition";b=b.concat(c(f(m.current())));break;default:m.next()}return!b.length?null:{element:a.substring(0,i),attributes:e(b)}}function e(a){var a=d.map(a,function(a){return d.clone(a)}),b={};return d.filter(a,function(a){if(!(a.name in b))return b[a.name]=a;var c=b[a.name];a.name.toLowerCase()=="class"?c.value+=(c.value.length?" ":"")+a.value:c.value=a.value;return!1})}function i(a){if(!~a.indexOf("{"))return null;
for(var b=h("stringStream").create(a);!b.eol();)switch(b.peek()){case "[":case "(":b.skipToPair(b.peek(),r[b.peek()]);break;case "{":return b.start=b.pos,b.skipToPair("{","}"),{element:a.substring(0,b.start),text:f(b.current())};default:b.next()}}function k(a){for(var b=a.children.length-1,c,e;b>=0;b--)if(e=a.children[b],e.isRepeating()){c=e.repeatCount;e.repeatCount=1;for(e.updateProperty("counter",1);--c>0;)e.parent.addChild(e.clone(),b+1).updateProperty("counter",c+1)}d.each(a.children,k);return a}
function l(a){for(var b=a.children.length-1;b>=0;b--){var c=a.children[b];c.isGroup()?c.replace(l(c).children):c.isEmpty()&&c.remove()}d.each(a.children,l);return a}function n(a){var b=a.charCodeAt(0);return b>64&&b<91||b>96&&b<123||b>47&&b<58||"#.*:$-_!@|".indexOf(a)!=-1}var o=/^[\w\-\$\:@\!]+\+?$/i,q=/[\w\-:\$]/,r={"[":"]","(":")","{":"}"},t=Array.prototype.splice,m=[],y=[],w=[];g.prototype={addChild:function(a,b){a=a||new g;a.parent=this;d.isUndefined(b)?this.children.push(a):this.children.splice(b,
0,a);return a},clone:function(){var a=new g;d.each(["abbreviation","counter","_name","_text","repeatCount","hasImplicitRepeat","start","end","content","padding"],function(b){a[b]=this[b]},this);a._attributes=d.map(this._attributes,function(a){return d.clone(a)});a._data=d.clone(this._data);a.children=d.map(this.children,function(b){b=b.clone();b.parent=a;return b});return a},remove:function(){if(this.parent)this.parent.children=d.without(this.parent.children,this);return this},replace:function(){var a=
this.parent,b=d.indexOf(a.children,this),c=d.flatten(arguments);t.apply(a.children,[b,1].concat(c));d.each(c,function(b){b.parent=a})},updateProperty:function(a,b){this[a]=b;d.each(this.children,function(c){c.updateProperty(a,b)})},find:function(a){if(!d.isFunction(a))var b=a.toLowerCase(),a=function(a){return a.name().toLowerCase()==b};var c=null;d.find(this.children,function(b){return a(b)?c=b:c=b.find(a)});return c},findAll:function(a){if(!d.isFunction(a))var b=a.toLowerCase(),a=function(a){return a.name().toLowerCase()==
b};var c=[];d.each(this.children,function(b){a(b)&&c.push(b);c=c.concat(b.findAll(a))});return d.compact(c)},data:function(a,b){if(arguments.length==2&&(this._data[a]=b,a=="resource"&&h("elements").is(b,"snippet")&&(this.content=b.data,this._text)))this.content=h("abbreviationUtils").insertChildContent(b.data,this._text);return this._data[a]},name:function(){var a=this.matchedResource();return h("elements").is(a,"element")?a.name:this._name},attributeList:function(){var a=[],b=this.matchedResource();
h("elements").is(b,"element")&&d.isArray(b.attributes)&&(a=a.concat(b.attributes));return e(a.concat(this._attributes))},attribute:function(a,b){if(arguments.length==2){var c=d.indexOf(d.pluck(this._attributes,"name"),a.toLowerCase());if(~c)this._attributes[c].value=b}return(d.find(this.attributeList(),function(b){return b.name==a})||{}).value},matchedResource:function(){return this.data("resource")},index:function(){return this.parent?d.indexOf(this.parent.children,this):-1},_setRepeat:function(a){a?
this.repeatCount=parseInt(a,10)||1:this.hasImplicitRepeat=!0},setAbbreviation:function(b){var c=this;this.abbreviation=b=(b||"").replace(/\*(\d+)?$/,function(a,b){c._setRepeat(b);return""});var e=i(b);if(e)b=e.element,this.content=this._text=e.text;if(e=a(b))b=e.element,this._attributes=e.attributes;if((this._name=b)&&!o.test(this._name))throw"Invalid abbreviation";},toString:function(){var a=h("utils"),b=this.start,c=this.end,e=this.content,i=this;d.each(w,function(a){b=a(b,i,"start");e=a(e,i,"content");
c=a(c,i,"end")});var m=d.map(this.children,function(a){return a.toString()}).join(""),e=h("abbreviationUtils").insertChildContent(e,m,{keepVariable:!1});return b+a.padString(e,this.padding)+c},hasEmptyChildren:function(){return!!d.find(this.children,function(a){return a.isEmpty()})},hasImplicitName:function(){return!this._name&&!this.isTextNode()},isGroup:function(){return!this.abbreviation},isEmpty:function(){return!this.abbreviation&&!this.children.length},isRepeating:function(){return this.repeatCount>
1||this.hasImplicitRepeat},isTextNode:function(){return!this.name()&&!this.attributeList().length},isElement:function(){return!this.isEmpty()&&!this.isTextNode()},deepestChild:function(){if(!this.children.length)return null;for(var a=this;a.children.length;)a=d.last(a.children);return a}};w.push(function(a,b){return h("utils").replaceCounter(a,b.counter)});return{parse:function(a,c){var c=c||{},e=b(a);if(c.contextNode){e._name=c.contextNode.name;var i={};d.each(e._attributes,function(a){i[a.name]=
a});d.each(c.contextNode.attributes,function(a){a.name in i?i[a.name].value=a.value:(a=d.clone(a),e._attributes.push(a),i[a.name]=a)})}d.each(m,function(a){a(e,c)});e=l(k(e));d.each(y,function(a){a(e,c)});return e},AbbreviationNode:g,addPreprocessor:function(a){d.include(m,a)||m.push(a)},removeFilter:function(a){preprocessor=d.without(m,a)},addPostprocessor:function(a){d.include(y,a)||y.push(a)},removePostprocessor:function(a){y=d.without(y,a)},addOutputProcessor:function(a){d.include(w,a)||w.push(a)},
removeOutputProcessor:function(a){w=d.without(w,a)},isAllowedChar:function(a){a=String(a);return n(a)||~">+^[](){}".indexOf(a)}}});
zen_coding.exec(function(h,d){function g(f,b){var c=h("resources"),a=h("elements"),e=h("abbreviationParser");d.each(d.clone(f.children),function(i){var f=c.getMatchedResource(i,b);if(d.isString(f))i.data("resource",a.create("snippet",f));else if(a.is(f,"reference")){var f=e.parse(f.data,{syntax:b}),h=f.deepestChild();h&&d.each(i.children,function(a){h.addChild(a)});i.replace(f.children)}else i.data("resource",f);g(i,b)})}h("abbreviationParser").addPreprocessor(function(d,b){var c=b.syntax||zen_coding.defaultSyntax();
g(d,c)})});
zen_coding.exec(function(h,d){function g(a){for(var b=h("range"),c=[],a=h("stringStream").create(a);!a.eol();){if(a.peek()=="\\")a.next();else if(a.start=a.pos,a.match(e,!0)){c.push(b.create(a.start,e));continue}a.next()}return c}function f(a,b){var c=h("utils"),e=g(a);e.reverse();d.each(e,function(e){a=c.replaceSubstring(a,b,e)});return a}function b(a){return g(a.content).length?!0:!!d.find(a.attributeList(),function(a){return!!g(a.value).length})}function c(a,c){var e=a.findAll(function(a){return b(a)});b(a)&&
e.unshift(a);e.length?d.each(e,function(a){a.content=f(a.content,c);d.each(a._attributes,function(a){a.value=f(a.value,c)})}):(e=a.deepestChild()||a,e.content=h("abbreviationUtils").insertChildContent(e.content,c))}var a=h("abbreviationParser"),e="$#";a.addPreprocessor(function(a,b){if(b.pastedContent){var c=h("utils").splitByLines(b.pastedContent,!0);a.findAll(function(a){if(a.hasImplicitRepeat)return a.data("paste",c),a.repeatCount=c.length})}});a.addPostprocessor(function(a,b){!a.findAll(function(a){var b=
a.data("paste"),e="";d.isArray(b)?e=b[a.counter-1]:d.isFunction(b)?e=b(a.counter-1,a.content):b&&(e=b);e&&c(a,e);a.data("paste",null);return!d.isUndefined(b)}).length&&b.pastedContent&&c(a,b.pastedContent)})});zen_coding.exec(function(h,d){function g(f){var b=h("tagName");d.each(f.children,function(c){if(c.hasImplicitName()||c.data("forceNameResolving"))c._name=b.resolve(c.parent.name());g(c)});return f}h("abbreviationParser").addPostprocessor(g)});
zen_coding.define("cssParser",function(h,d){function g(a){return typeof a!=="undefined"}function f(){return{"char":i.chnum,line:i.linenum}}function b(a,b,c){var e=i,c=c||{};k.push({charstart:g(c["char"])?c["char"]:e.chnum,charend:g(c.charend)?c.charend:e.chnum,linestart:g(c.line)?c.line:e.linenum,lineend:g(c.lineend)?c.lineend:e.linenum,value:a,type:b||a})}function c(a,b){var c=i,e=b||{},d=g(e["char"])?e["char"]:c.chnum,e=g(e.line)?e.line:c.linenum;return{name:"ParseError",message:a+" at line "+(e+
1)+" char "+(d+1),walker:c,tokens:k}}function a(a){var c=i,e=c.ch,d=f(),g=a?a+e:e,e=c.nextChar();for(a&&(d["char"]-=a.length);n(e)||o(e);)g+=e,e=c.nextChar();b(g,"identifier",d)}function e(){var e=i.ch;if(e===" "||e==="\t"){for(var d=i.ch,g="",m=f();d===" "||d==="\t";)g+=d,d=i.nextChar();b(g,"white",m)}else{if(e==="/"){var d=i,e=m=d.ch,h,k=f();h=d.nextChar();if(h!=="*")k.charend=k["char"],k.lineend=k.line,g=b(e,e,k);else{for(;!(m==="*"&&h==="/");)e+=h,m=h,h=d.nextChar();e+=h;d.nextChar();b(e,"comment",
k)}return g}if(e==='"'||e==="'"){d=i;e=m=g=d.ch;k=f();for(g=d.nextChar();g!==m;){if(g==="\n")if(h=d.nextChar(),h==="\\")e+=g+h;else throw c("Unterminated string",k);else e+=g==="\\"?g+d.nextChar():g;g=d.nextChar()}e+=g;d.nextChar();b(e,"string",k)}else if(e==="("){d=i;g=d.ch;m=0;e=g;h=f();for(g=d.nextChar();g!==")"&&!m;){if(g==="(")m++;else if(g===")")m--;else if(g===!1)throw c("Unterminated brace",h);e+=g;g=d.nextChar()}e+=g;d.nextChar();b(e,"brace",h)}else{if(e==="-"||e==="."||o(e)){g=i;m=g.ch;
e=f();h=m;var k=h===".",z,m=g.nextChar();z=!o(m);if(k&&z)e.charend=e["char"],e.lineend=e.line,d=b(h,".",e);else if(h==="-"&&z)d=a("-");else{for(;m!==!1&&(o(m)||!k&&m===".");)m==="."&&(k=!0),h+=m,m=g.nextChar();b(h,"number",e)}return d}if(n(e))return a();if(l(e))return d=i,e=d.ch,g=f(),h=d.nextChar(),h==="="&&l(e,!0)?(e+=h,b(e,"match",g),d.nextChar(),m=void 0):(g.charend=g["char"]+1,g.lineend=g.line,b(e,e,g)),m;if(e==="\n")b("line"),i.nextChar();else throw c("Unrecognized character");}}}var i,k=[],
l,n,o;i={lines:null,total_lines:0,linenum:-1,line:"",ch:"",chnum:-1,init:function(a){var b=i;b.lines=a.replace(/\r\n/g,"\n").replace(/\r/g,"\n").split("\n");b.total_lines=b.lines.length;b.chnum=-1;b.linenum=-1;b.ch="";b.line="";b.nextLine();b.nextChar()},nextLine:function(){this.linenum+=1;this.line=this.total_lines<=this.linenum?!1:this.lines[this.linenum];if(this.chnum!==-1)this.chnum=0;return this.line},nextChar:function(){for(this.chnum+=1;this.line.charAt(this.chnum)==="";){if(this.nextLine()===
!1)return this.ch=!1;this.chnum=-1;return this.ch="\n"}return this.ch=this.line.charAt(this.chnum)},peek:function(){return this.line.charAt(this.chnum+1)}};n=function(a){return a==="_"||a==="-"||a>="a"&&a<="z"||a>="A"&&a<="Z"};o=function(a){return a!==!1&&a>="0"&&a<="9"};l=function(){for(var a="{}[]()+*=.,;:>~|\\%$#@^!".split(""),b="*^|$~".split(""),c={},e={},d=0;d<a.length;d+=1)c[a[d]]=!0;for(d=0;d<b.length;d+=1)e[b[d]]=!0;return function(a,b){return b?!!e[a]:!!c[a]}}();return{lex:function(a){i.init(a);
for(k=[];i.ch!==!1;)e();return k},parse:function(a){var b=0;return d.map(this.lex(a),function(c){if(c.type=="line")c.value=a.charAt(b)=="\r"&&a.charAt(b+1)=="\n"?"\r\n":a.charAt(b);return{type:c.type,start:b,end:b+=c.value.length}})},toSource:function(a){for(var b=0,c=a.length,e,d="";b<c;b+=1)e=a[b],d+=e.type==="line"?"\n":e.value;return d}}});
zen_coding.define("xmlParser",function(h){function d(a,e){function d(b){e.tokenize=b;return b(a,e)}var i=a.next();if(i=="<")if(a.eat("!"))return a.eat("[")?a.match("CDATA[")?d(b("atom","]]\>")):null:a.match("--")?d(b("comment","--\>")):a.match("DOCTYPE",!0,!0)?(a.eatWhile(/[\w\._\-]/),d(c(1))):null;else if(a.eat("?"))return a.eatWhile(/[\w\._\-]/),e.tokenize=b("meta","?>"),"meta";else{A=a.eat("/")?"closeTag":"openTag";a.eatSpace();for(z="";i=a.eat(/[^\s\u00a0=<>\"\'\/?]/);)z+=i;e.tokenize=g;return"tag"}else return i==
"&"?(a.eat("#")?a.eat("x")?a.eatWhile(/[a-fA-F\d]/)&&a.eat(";"):a.eatWhile(/[\d]/)&&a.eat(";"):a.eatWhile(/[\w\.\-:]/)&&a.eat(";"))?"atom":"error":(a.eatWhile(/[^&<]/),"text")}function g(a,b){var c=a.next();return c==">"||c=="/"&&a.eat(">")?(b.tokenize=d,A=c==">"?"endTag":"selfcloseTag","tag"):c=="="?(A="equals",null):/[\'\"]/.test(c)?(b.tokenize=f(c),b.tokenize(a,b)):(a.eatWhile(/[^\s\u00a0=<>\"\'\/?]/),"word")}function f(a){return function(b,c){for(;!b.eol();)if(b.next()==a){c.tokenize=g;break}return"string"}}
function b(a,b){return function(c,e){for(;!c.eol();){if(c.match(b)){e.tokenize=d;break}c.next()}return a}}function c(a){return function(b,e){for(var i;(i=b.next())!=null;)if(i=="<")return e.tokenize=c(a+1),e.tokenize(b,e);else if(i==">")if(a==1){e.tokenize=d;break}else return e.tokenize=c(a-1),e.tokenize(b,e);return"meta"}}function a(){for(var a=arguments.length-1;a>=0;a--)j.cc.push(arguments[a])}function e(){a.apply(null,arguments);return!0}function i(){if(j.context)j.context=j.context.prev}function k(a){if(a==
"openTag")return j.tagName=z,e(q,l(j.startOfLine));else if(a=="closeTag")return a=!1,j.context?j.context.tagName!=z&&(w.implicitlyClosed.hasOwnProperty(j.context.tagName.toLowerCase())&&i(),a=!j.context||j.context.tagName!=z):a=!0,a&&(u="error"),e(n(a));return e()}function l(a){return function(b){if(b=="selfcloseTag"||b=="endTag"&&w.autoSelfClosers.hasOwnProperty(j.tagName.toLowerCase()))return o(j.tagName.toLowerCase()),e();if(b=="endTag"){o(j.tagName.toLowerCase());var b=j.tagName,c=w.doNotIndent.hasOwnProperty(b)||
j.context&&j.context.noIndent;j.context={prev:j.context,tagName:b,indent:j.indented,startOfLine:a,noIndent:c}}return e()}}function n(a){return function(b){a&&(u="error");if(b=="endTag")return i(),e();u="error";return e(arguments.callee)}}function o(a){for(var b;;){if(!j.context)break;b=j.context.tagName.toLowerCase();if(!w.contextGrabbers.hasOwnProperty(b)||!w.contextGrabbers[b].hasOwnProperty(a))break;i()}}function q(b){if(b=="word")return u="attribute",e(r,q);if(b=="endTag"||b=="selfcloseTag")return a();
u="error";return e(q)}function r(b){if(b=="equals")return e(t,q);w.allowMissing||(u="error");return b=="endTag"||b=="selfcloseTag"?a():e()}function t(b){if(b=="string")return e(m);if(b=="word"&&w.allowUnquoted)return u="string",e();u="error";return b=="endTag"||b=="selfCloseTag"?a():e()}function m(b){return b=="string"?e(m):a()}function y(a,b){if(a.sol())b.startOfLine=!0,b.indented=0;if(a.eatSpace())return null;u=A=z=null;var c=b.tokenize(a,b);b.type=A;if((c||A)&&c!="comment")for(j=b;;)if((b.cc.pop()||
k)(A||c))break;b.startOfLine=!1;return u||c}var w={autoSelfClosers:{},implicitlyClosed:{},contextGrabbers:{},doNotIndent:{},allowUnquoted:!0,allowMissing:!0},z=null,A=null,j=null,u;return{parse:function(a,b){for(var b=b||0,c={tokenize:d,cc:[],indented:0,startOfLine:!0,tagName:null,context:null},e=h("stringStream").create(a),i=[];!e.eol();)i.push({type:y(e,c),start:e.start+b,end:e.pos+b}),e.start=e.pos;return i}}});
zen_coding.define("utils",function(h,d){function g(b){this._data=[];this.length=0;b&&this.append(b)}var f="${0}";g.prototype={append:function(b){this._data.push(b);this.length+=b.length},toString:function(){return this._data.join("")},valueOf:function(){return this.toString()}};return{reTag:/<\/?[\w:\-]+(?:\s+[\w\-:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*\s*(\/?)>$/,endsWithTag:function(b){return this.reTag.test(b)},isNumeric:function(b){typeof b=="string"&&(b=b.charCodeAt(0));return b&&
b>47&&b<58},trim:function(b){return(b||"").replace(/^\s+|\s+$/g,"")},getNewline:function(){var b=h("resources");if(!b)return"\n";b=b.getVariable("newline");return d.isString(b)?b:"\n"},setNewline:function(b){var c=h("resources");c.setVariable("newline",b);c.setVariable("nl",b)},splitByLines:function(b,c){var a=this.getNewline(),a=(b||"").replace(/\r\n/g,"\n").replace(/\n\r/g,"\n").replace(/\r/g,"\n").replace(/\n/g,a).split(a);c&&(a=d.filter(a,function(a){return a.length&&!!this.trim(a)},this));return a},
normalizeNewline:function(b){return this.splitByLines(b).join(this.getNewline())},repeatString:function(b,c){for(var a=[],e=0;e<c;e++)a.push(b);return a.join("")},padString:function(b,c){var a=d.isNumber(c)?this.repeatString(h("resources").getVariable("indentation")||"\t",c):c,e=[],i=this.splitByLines(b),f=this.getNewline();e.push(i[0]);for(var g=1;g<i.length;g++)e.push(f+a+i[g]);return e.join("")},zeroPadString:function(b,c){for(var a="",e=b.length;c>e++;)a+="0";return a+b},unindentString:function(b,
c){for(var a=this.splitByLines(b),e=0;e<a.length;e++)a[e].search(c)==0&&(a[e]=a[e].substr(c.length));return a.join(this.getNewline())},replaceUnescapedSymbol:function(b,c,a){for(var e=0,i=b.length,f=c.length,g=0;e<i;)if(b.charAt(e)=="\\")e+=f+1;else if(b.substr(e,f)==c){var h=f;g++;var o=a;if(d.isFunction(a))(o=a(b,c,e,g))?(h=o[0].length,o=o[1]):o=!1;o===!1?e++:(b=b.substring(0,e)+o+b.substring(e+h),i=b.length,e+=o.length)}else e++;return b},replaceVariables:function(b,c){var c=c||{},a=d.isFunction(c)?
c:function(a,b){return b in c?c[b]:null},e=h("resources");return h("tabStops").processText(b,{variable:function(b){var c=a(b.token,b.name,b);c===null&&(c=e.getVariable(b.name));if(c===null||d.isUndefined(c))c=b.token;return c}})},replaceCounter:function(b,c){var c=String(c),a=this;return this.replaceUnescapedSymbol(String(b),"$",function(b,d,f){if(b.charAt(f+1)=="{"||a.isNumeric(b.charAt(f+1)))return!1;for(d=f+1;b.charAt(d)=="$"&&b.charAt(d+1)!="{";)d++;return[b.substring(f,d),a.zeroPadString(c,d-
f)]})},matchesTag:function(b){return this.reTag.test(b||"")},escapeText:function(b){return b.replace(/([\$\\])/g,"\\$1")},unescapeText:function(b){return b.replace(/\\(.)/g,"$1")},getCaretPlaceholder:function(){return d.isFunction(f)?f.apply(this,arguments):f},setCaretPlaceholder:function(b){f=b},getLinePadding:function(b){return(b.match(/^(\s+)/)||[""])[0]},getLinePaddingFromPosition:function(b,c){return this.getLinePadding(this.findNewlineBounds(b,c).substring(b))},escapeForRegexp:function(b){return b.replace(RegExp("[.*+?|()\\[\\]{}\\\\]",
"g"),"\\$&")},prettifyNumber:function(b,c){return b.toFixed(typeof c=="undefined"?2:c).replace(/\.?0+$/,"")},stringBuilder:function(b){return new g(b)},replaceSubstring:function(b,c,a,e){if(d.isObject(a)&&"end"in a)e=a.end,a=a.start;d.isString(e)&&(e=a+e.length);d.isUndefined(e)&&(e=a);return a<0||a>b.length?b:b.substring(0,a)+c+b.substring(e)},narrowToNonSpace:function(b,c,a){c=h("range").create(c,a);for(a=/[\s\n\r\u00a0]/;c.start<c.end;){if(!a.test(b.charAt(c.start)))break;c.start++}for(;c.end>
c.start;)if(c.end--,!a.test(b.charAt(c.end))){c.end++;break}return c},findNewlineBounds:function(b,c){for(var a=b.length,e=0,d=a-1,f=c-1;f>0;f--){var g=b.charAt(f);if(g=="\n"||g=="\r"){e=f+1;break}}for(f=c;f<a;f++)if(g=b.charAt(f),g=="\n"||g=="\r"){d=f;break}return h("range").create(e,d-e)},deepMerge:function(){var b,c,a,e,f,g=arguments[0]||{},h=1,n=arguments.length;for(!d.isObject(g)&&!d.isFunction(g)&&(g={});h<n;h++)if((b=arguments[h])!=null)for(c in b)a=g[c],e=b[c],g!==e&&(e&&(d.isObject(e)||(f=
d.isArray(e)))?(f?(f=!1,a=a&&d.isArray(a)?a:[]):a=a&&d.isObject(a)?a:{},g[c]=this.deepMerge(a,e)):e!==void 0&&(g[c]=e));return g}}});
zen_coding.define("range",function(h,d){function g(f,b){d.isObject(f)&&"start"in f?(this.start=Math.min(f.start,f.end),this.end=Math.max(f.start,f.end)):d.isArray(f)?(this.start=f[0],this.end=f[1]):(b=d.isString(b)?b.length:+b,this.start=f,this.end=f+b)}g.prototype={length:function(){return Math.abs(this.end-this.start)},equal:function(d){return this.start===d.start&&this.end===d.end},shift:function(d){this.start+=d;this.end+=d;return this},overlap:function(d){return d.start<=this.end&&d.end>=this.start},
intersection:function(d){if(this.overlap(d)){var b=Math.max(d.start,this.start);return new g(b,Math.min(d.end,this.end)-b)}return null},union:function(d){if(this.overlap(d)){var b=Math.min(d.start,this.start);return new g(b,Math.max(d.end,this.end)-b)}return null},inside:function(d){return this.start<=d&&this.end>d},include:function(d){return this.start<=d.start&&this.end>=d.end},substring:function(d){return this.length()>0?d.substring(this.start,this.end):""},clone:function(){return new g(this.start,
this.length())},toArray:function(){return[this.start,this.end]},toString:function(){return"{"+this.start+", "+this.length()+"}"}};return{create:function(f,b){return d.isUndefined(f)||f===null?null:f instanceof g?f:new g(f,b)},create2:function(f,b){d.isNumber(f)&&d.isNumber(b)&&(b-=f);return this.create(f,b)}}});
zen_coding.define("handlerList",function(h,d){function g(){this._list=[]}g.prototype={add:function(f,b){this._list.push(d.extend({order:0},b||{},{fn:f}))},remove:function(f){this._list=d.without(this._list,d.find(this._list,function(b){return b.fn===f}))},list:function(){return d.sortBy(this._list,"order").reverse()},listFn:function(){return d.pluck(this.list(),"fn")},exec:function(f,b){var b=b||[],c=null;d.find(this.list(),function(a){c=a.fn.apply(a,b);if(c!==f)return!0});return c}};return{create:function(){return new g}}});
zen_coding.define("tokenIterator",function(h,d){function g(d){this.tokens=d;this._position=0;this.reset()}g.prototype={next:function(){if(this.hasNext()){var d=this.tokens[++this._i];this._position=d.start;return d}return null},current:function(){return this.tokens[this._i]},position:function(){return this._position},hasNext:function(){return this._i<this._il-1},reset:function(){this._i=-1;this._il=this.tokens.length},item:function(){return this.tokens[this._i]},itemNext:function(){return this.tokens[this._i+
1]},itemPrev:function(){return this.tokens[this._i-1]},nextUntil:function(f,b){for(var c,a=d.isString(f)?function(a){return a.type==f}:f;c=this.next();)if(b&&b.call(this,c),a.call(this,c))break}};return{create:function(d){return new g(d)}}});
zen_coding.define("stringStream",function(){function h(d){this.pos=this.start=0;this.string=d}h.prototype={eol:function(){return this.pos>=this.string.length},sol:function(){return this.pos==0},peek:function(){return this.string.charAt(this.pos)},next:function(){if(this.pos<this.string.length)return this.string.charAt(this.pos++)},eat:function(d){var g=this.string.charAt(this.pos);if(typeof d=="string"?g==d:g&&(d.test?d.test(g):d(g)))return++this.pos,g},eatWhile:function(d){for(var g=this.pos;this.eat(d););
return this.pos>g},eatSpace:function(){for(var d=this.pos;/[\s\u00a0]/.test(this.string.charAt(this.pos));)++this.pos;return this.pos>d},skipToEnd:function(){this.pos=this.string.length},skipTo:function(d){d=this.string.indexOf(d,this.pos);if(d>-1)return this.pos=d,!0},skipToPair:function(d,g){for(var f=0,b,c=this.pos,a=this.string.length;c<a;)if(b=this.string.charAt(c++),b==d)f++;else if(b==g&&(f--,f<1))return this.pos=c,!0;return!1},backUp:function(d){this.pos-=d},match:function(d,g,f){if(typeof d==
"string"){if(f=f?function(b){return b.toLowerCase()}:function(b){return b},f(this.string).indexOf(f(d),this.pos)==this.pos)return g!==!1&&(this.pos+=d.length),!0}else return(d=this.string.slice(this.pos).match(d))&&g!==!1&&(this.pos+=d[0].length),d},current:function(){return this.string.slice(this.start,this.pos)}};return{create:function(d){return new h(d)}}});
zen_coding.define("resources",function(h,d){function g(a){return a==c?i:k}function f(b,c,e){var d=g(b),f=[],m=null;d&&c in d&&(m=d[c],e in m&&f.push(m[e]));var k=null;m&&"extends"in m?k=m:b==a&&c in i&&"extends"in i[c]&&(k=i[c]);if(k){if(!k["extends"]||!k["extends"].__zen_parsed__){c=k["extends"].split(",");m=h("utils");for(b=0;b<c.length;b++)c[b]=m.trim(c[b]);k["extends"]=c;k["extends"].__zen_parsed__=!0}for(b=0;b<k["extends"].length;b++)c=k["extends"][b],d[c]&&d[c][e]&&f.push(d[c][e])}return f}
function b(a,b,c,d){for(var b=f(a,b,c),i=null,g=h("elements"),k=0,l=b.length;k<l;k++)if(a=b[k],d in a){if(!a[d]||!a[d].__zen_parsed__){b=a[d];i=h("utils");b=i.replaceUnescapedSymbol(b,"|",i.getCaretPlaceholder());switch(c){case "abbreviations":c=a;k=g=d;i=b;h("utils").trim(k);l=h("elements");k=void 0;i=(k=e.exec(i))?l.create("element",k[1],k[2],k[4]=="/"):l.create("reference",i);c[g]=i;a[d].__ref=b;break;case "snippets":a[d]=g.create("snippet",b)}a[d].__zen_parsed__=!0}i=a[d];break}return i}var c=
"system",a="user",e=/^<(\w+\:?[\w\-]*)((?:\s+[\w\:\-]+\s*=\s*(['"]).*?\3)*)\s*(\/?)>/,i={},k={},l=h("handlerList").create();return{setVocabulary:function(a,b){b==c?i=a:k=a},getVocabulary:g,getResource:function(e,d,i){return b(a,e,d,i)||b(c,e,d,i)},getAbbreviation:function(a,b){b=b||"";return this.getResource(a,"abbreviations",b)||this.getResource(a,"abbreviations",b.replace(/\-/g,":"))},getSnippet:function(a,b){b=b||"";return this.getResource(a,"snippets",b)||this.getResource(a,"snippets",b.replace(/\-/g,
":"))},getMatchedResource:function(a,b){return l.exec(null,d.toArray(arguments))||this.getAbbreviation(b,a.name())||this.getSnippet(b,a.name())},getVariable:function(b){return f(a,"variables",b)[0]||f(c,"variables",b)[0]},setVariable:function(a,b){var c=g("user")||{};if(!("variables"in c))c.variables={};c.variables[a]=b;this.setVocabulary(c,"user")},getSubset:function(b,e){return f(a,b,e)[0]||f(c,b,e)[0]},hasSyntax:function(b){return b in g(a)||b in g(c)},addResolver:function(a,b){l.add(a,b)},removeResolver:function(a){l.remove(a)}}});
zen_coding.define("actions",function(h,d){function g(b){return h("utils").trim(b.charAt(0).toUpperCase()+b.substring(1).replace(/_[a-z]/g,function(b){return" "+b.charAt(1).toUpperCase()}))}var f={};return{add:function(b,c,a){b=b.toLowerCase();f[b]={name:b,fn:c,options:a||{}}},get:function(b){return f[b.toLowerCase()]},run:function(b,c){d.isArray(c)||(c=d.rest(arguments));var a=this.get(b);return a?a.fn.apply(zen_coding,c):(zen_coding.log('Action "%s" is not defined',b),!1)},getAll:function(){return f},
getList:function(){return d.values(this.getAll())},getMenu:function(b){var c=[],b=b||[];d.each(this.getList(),function(a){if(!a.options.hidden&&!d.include(b,a.name)){var e=g(a.name),i=c;if(a.options.label)for(var f=a.options.label.split("/"),e=f.pop(),h,n;h=f.shift();)n=d.find(i,function(a){return a.type=="submenu"&&a.name==h}),n||(n={name:h,type:"submenu",items:[]},i.push(n)),i=n.items;i.push({type:"action",name:a.name,label:e})}});return c},getActionNameForMenuTitle:function(b,c){var a=null;d.find(c||
this.getMenu(),function(c){if(c.type=="action"){if(c.label==b||c.name==b)return a=c.name}else return a=this.getActionNameForMenuTitle(b,c.items)},this);return a||null}}});
zen_coding.define("profile",function(h,d){function g(b){d.extend(this,a,b)}function f(a,b){switch(String(b||"").toLowerCase()){case "lower":return a.toLowerCase();case "upper":return a.toUpperCase()}return a}function b(a,b){return c[a.toLowerCase()]=new g(b)}var c={},a={tag_case:"asis",attr_case:"asis",attr_quotes:"double",tag_nl:"decide",place_cursor:!0,indent:!0,inline_break:3,self_closing_tag:"xhtml",filters:""};g.prototype={tagName:function(a){return f(a,this.tag_case)},attributeName:function(a){return f(a,
this.attr_case)},attributeQuote:function(){return this.attr_quotes=="single"?"'":'"'},selfClosing:function(){return this.self_closing_tag=="xhtml"?" /":this.self_closing_tag===!0?"/":""},cursor:function(){return this.place_cursor?h("utils").getCaretPlaceholder():""}};b("xhtml");b("html",{self_closing_tag:!1});b("xml",{self_closing_tag:!0,tag_nl:!0});b("plain",{tag_nl:!1,indent:!1,place_cursor:!1});return{create:function(c,i){return arguments.length==2?b(c,i):new g(d.defaults(c||{},a))},get:function(a,
b){if(b&&d.isString(a)){var f=h("resources").getSubset(b,"profile");f&&(a=f)}return d.isString(a)&&a.toLowerCase()in c?c[a.toLowerCase()]:a&&"tag_case"in a?this.create(a):c.plain},remove:function(a){a=(a||"").toLowerCase();a in c&&delete c[a]},stringCase:f,quote:function(a){console.log("deprecated");return a=="single"?"'":'"'},selfClosing:function(a){console.log("deprecated");return a=="xhtml"?" /":a===!0?"/":""}}});
zen_coding.define("editorUtils",function(h){return{isInsideTag:function(d,g){for(var f=/^<\/?\w[\w\:\-]*.*?>/,b=g;b>-1;){if(d.charAt(b)=="<")break;b--}return b!=-1&&(f=f.exec(d.substring(b)))&&g>b&&g<b+f[0].length?!0:!1},outputInfo:function(d,g,f){return{syntax:String(g||d.getSyntax()),profile:String(f||d.getProfileName()),content:String(d.getContent())}},unindent:function(d,g){return h("utils").unindentString(g,this.getCurrentLinePadding(d))},getCurrentLinePadding:function(d){return h("utils").getLinePadding(d.getCurrentLine())}}});
zen_coding.define("actionUtils",function(h){return{mimeTypes:{gif:"image/gif",png:"image/png",jpg:"image/jpeg",jpeg:"image/jpeg",svg:"image/svg+xml",html:"text/html",htm:"text/html"},extractAbbreviation:function(d){for(var g=d.length,f=-1,b=0,c=0,a=0,e=h("utils"),i=h("abbreviationParser");;){g--;if(g<0){f=0;break}var k=d.charAt(g);if(k=="]")c++;else if(k=="["){if(!c){f=g+1;break}c--}else if(k=="}")a++;else if(k=="{"){if(!a){f=g+1;break}a--}else if(k==")")b++;else if(k=="("){if(!b){f=g+1;break}b--}else if(!c&&
!a&&(!i.isAllowedChar(k)||k==">"&&e.endsWithTag(d.substring(0,g+1)))){f=g+1;break}}return f!=-1&&!a&&!c&&!b?d.substring(f):""},getImageSize:function(d){var g=function(){return d.charCodeAt(f++)};if(d.substr(0,8)==="\u0089PNG\r\n\u001a\n"){var f=d.indexOf("IHDR")+4;return{width:g()<<24|g()<<16|g()<<8|g(),height:g()<<24|g()<<16|g()<<8|g()}}else if(d.substr(0,4)==="GIF8")return f=6,{width:g()|g()<<8,height:g()|g()<<8};else if(d.substr(0,2)==="\u00ff\u00d8")for(var f=2,b=d.length;f<b;){if(g()!=255)break;
var c=g();if(c==218)break;var a=g()<<8|g();if(c>=192&&c<=207&&!(c&4)&&!(c&8))return f+=1,{height:g()<<8|g(),width:g()<<8|g()};else f+=a-2}},captureContext:function(d){if(String(d.getSyntax())in{html:1,xml:1,xsl:1}){var g=h("html_matcher").getTags(String(d.getContent()),d.getCaretPos(),String(d.getProfileName()));if(g&&g[0]&&g[0].type=="tag"){for(var d=/([\w\-:]+)(?:\s*=\s*(?:(?:"((?:\\.|[^"])*)")|(?:'((?:\\.|[^'])*)')|([^>\s]+)))?/g,f=g[0],g=f.full_tag.replace(/^<[\w\-\:]+/,""),f={name:f.name,attributes:[]},
b;b=d.exec(g);)f.attributes.push({name:b[1],value:b[2]});return f}}return null},findExpressionBounds:function(d,g){for(var f=String(d.getContent()),b=f.length,c=d.getCaretPos()-1,a=c+1;c>=0&&g(f.charAt(c),c,f);)c--;for(;a<b&&g(f.charAt(a),a,f);)a++;if(a>c)return h("range").create([++c,a])},compoundUpdate:function(d,g){if(g){var f=d.getSelectionRange();d.replaceContent(g.data,g.start,g.end,!0);d.createSelection(g.caret,g.caret+f.end-f.start);return!0}return!1}}});
zen_coding.define("abbreviationUtils",function(h,d){return{isSnippet:function(d){return h("elements").is(d.matchedResource(),"snippet")},isUnary:function(d){var f=d.matchedResource();return d.children.length||this.isSnippet(d)?!1:f&&f.is_empty||h("tagName").isEmptyElement(d.name())},isInline:function(d){return d.isTextNode()||!d.name()||h("tagName").isInlineLevel(d.name())},isBlock:function(d){return h("elements").is(d.matchedResource(),"snippet")||!this.isInline(d)},hasTagsInContent:function(d){return h("utils").matchesTag(d.content)},
hasBlockChildren:function(g){return this.hasTagsInContent(g)&&this.isBlock(g)||d.any(g.children,function(d){return this.isBlock(d)},this)},insertChildContent:function(g,f,b){var b=d.extend({keepVariable:!0,appendIfNoChild:!0},b||{}),c=!1,a=h("utils"),g=a.replaceVariables(g,function(e,d,h){var l=e;d=="child"&&(l=a.padString(f,a.getLinePaddingFromPosition(g,h.start)),c=!0,b.keepVariable&&(l+=e));return l});!c&&b.appendIfNoChild&&(g+=f);return g}}});
zen_coding.define("base64",function(){return{encode:function(h){for(var d=[],g,f,b,c,a,e,i=0,k=h.length;i<k;)c=h.charCodeAt(i++),a=h.charCodeAt(i++),e=h.charCodeAt(i++),g=c&255,f=a&255,b=e&255,c=g>>2,g=(g&3)<<4|f>>4,f=(f&15)<<2|b>>6,b&=63,isNaN(a)?f=b=64:isNaN(e)&&(b=64),d.push("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".charAt(c)+"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".charAt(g)+"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".charAt(f)+
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".charAt(b));return d.join("")},decode:function(h){var d,g,f,b,c,a=0,e=0,i=[],k=h.length;if(!h)return h;h+="";do d="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".indexOf(h.charAt(a++)),g="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".indexOf(h.charAt(a++)),b="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".indexOf(h.charAt(a++)),c="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".indexOf(h.charAt(a++)),
f=d<<18|g<<12|b<<6|c,d=f>>16&255,g=f>>8&255,f&=255,b==64?i[e++]=String.fromCharCode(d):c==64?i[e++]=String.fromCharCode(d,g):i[e++]=String.fromCharCode(d,g,f);while(a<k);return i.join("")}}});
(function(){function h(a){if(!a||a!="html")a="xhtml";n=a}function d(a,b){var c=a[1].toLowerCase();return{name:c,full_tag:a[0],start:b,end:b+a[0].length,unary:Boolean(a[3])||c in k&&n=="html",has_close:Boolean(a[3]),type:"tag",close_self:c in l&&n=="html"}}function g(a,b){return{start:a,end:b,type:"comment"}}function f(a){for(var b={},a=a.split(","),c=0;c<a.length;c++)b[a[c]]=!0;return b}function b(a,b,c){var c=c||0,e=-1,d=-1;if(a&&!b)e=a.start,d=a.end;else if(a&&b)a.start<c&&a.end>c||b.start<=c&&
b.end>c?(e=a.start,d=b.end):(e=a.end,d=b.start);return[e,d]}function c(a,c,e){o.opening_tag=a;o.closing_tag=c;a=b(a,c,e||0);o.start_ix=a[0];o.end_ix=a[1];return o.start_ix!=-1?[o.start_ix,o.end_ix]:null}function a(a,c,f,k){function n(b,c){arguments.length==1&&(c=v);return a.substr(c,b.length)==b}function l(b){for(;b--;)if(a.charAt(b)=="<"&&n("<\!--",b))break;return b}k=k||b;h(f);var f=[],j=[],o=null,r=null,q=a.length,s,v;f.last=j.last=function(){return this[this.length-1]};for(v=c;v--&&v>=0;)if(s=
a.charAt(v),s=="<"){var x=a.substring(v,q);if(s=x.match(i))s=d(s,v),s.start<c&&s.end>c?r=s:j.push(s);else if(s=x.match(e))if(s=d(s,v),s.unary){if(s.start<c&&s.end>c)return k(s,null,c)}else if(j.last()&&j.last().name==s.name)j.pop();else{o=s;break}else if(x.indexOf("<\!--")==0&&(s=x.search("--\>")+v+3,v<c&&s>=c))return k(g(v,s))}else s=="-"&&n("--\>")&&(v=l(v));if(!o)return k(null);if(!r)for(v=c;v<q;v++)if(s=a.charAt(v),s=="<")if(x=a.substring(v,q),s=x.match(e))s=d(s,v),s.unary||f.push(s);else if(s=
x.match(i))if(s=d(s,v),f.last()&&f.last().name==s.name)f.pop();else{r=s;break}else n("<\!--")&&(v+=x.search("--\>")+2);else if(s=="-"&&n("--\>")&&(!f.last()||f.last().type!="comment"))return s=v+3,k(g(l(v),s));return k(o,r,c)}var e=/^<([\w\:\-]+)((?:\s+[\w\-:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*)\s*(\/?)>/,i=/^<\/([\w\:\-]+)[^>]*>/,k=f("area,base,basefont,br,col,frame,hr,img,input,isindex,link,meta,param,embed");f("address,applet,blockquote,button,center,dd,dir,div,dl,dt,fieldset,form,frameset,hr,iframe,isindex,li,map,menu,noframes,noscript,object,ol,p,pre,script,table,tbody,td,tfoot,th,thead,tr,ul");
f("a,abbr,acronym,applet,b,basefont,bdo,big,br,button,cite,code,del,dfn,em,font,i,iframe,img,input,ins,kbd,label,map,object,q,s,samp,select,small,span,strike,strong,sub,sup,textarea,tt,u,var");var l=f("colgroup,dd,dt,li,options,p,td,tfoot,th,thead,tr"),n="xhtml",o={opening_tag:null,closing_tag:null,start_ix:-1,end_ix:-1},q=function(b,e,d){return a(b,e,d,c)};q.start_tag=e;q.end_tag=i;q.find=function(b,c,e){return a(b,c,e)};q.getTags=function(b,c,e){return a(b,c,e,function(a,b){return[a,b]})};q.last_match=
o;try{zen_coding.define("html_matcher",function(){return q})}catch(r){}})();
zen_coding.define("tabStops",function(h,d){var g=100,f=0,b={replaceCarets:!1,escape:function(b){return"\\"+b},tabstop:function(b){return b.token},variable:function(b){return b.token}};h("abbreviationParser").addOutputProcessor(function(b,a){var e=0,d=h("tabStops"),g=h("utils"),b=d.processText(b,{tabstop:function(a){var b=parseInt(a.group);if(b==0)return"${0}";b>e&&(e=b);return a.placeholder?"${"+(b+f)+":"+a.placeholder+"}":"${"+(b+f)+"}"}}),b=g.replaceVariables(b,d.variablesResolver(a));f+=e+1;return b});
return{extract:function(c,a){var e=h("utils"),i={carets:""},f=[],a=d.extend({},b,a,{tabstop:function(a){var b=a.token,c="";if(a.placeholder=="cursor")f.push({start:a.start,end:a.start+b.length,group:"carets",value:""});else{if("placeholder"in a)i[a.group]=a.placeholder;a.group in i&&(c=i[a.group]);f.push({start:a.start,end:a.start+b.length,group:a.group,value:c})}return b}});a.replaceCarets&&(c=c.replace(RegExp(e.escapeForRegexp(e.getCaretPlaceholder()),"g"),"${0:cursor}"));var c=this.processText(c,
a),g=e.stringBuilder(),n=0,e=d.map(f,function(a){g.append(c.substring(n,a.start));var b=g.length,e=i[a.group]||"";g.append(e);n=a.end;return{group:a.group,start:b,end:b+e.length}});g.append(c.substring(n));return{text:g.toString(),tabstops:d.sortBy(e,"start")}},processText:function(c,a){for(var a=d.extend({},b,a),e=h("utils").stringBuilder(),f=h("stringStream").create(c),g,l;g=f.next();)if(g=="\\"&&!f.eol())e.append(a.escape(f.next()));else{l=g;if(g=="$")if(f.start=f.pos-1,f.match(/^[0-9]+/))l=a.tabstop({start:e.length,
group:f.current().substr(1),token:f.current()});else if(g=f.match(/^\{([a-z_\-][\w\-]*)\}/))l=a.variable({start:e.length,name:g[1],token:f.current()});else if(g=f.match(/^\{([0-9]+)(:.+?)?\}/)){l={start:e.length,group:g[1],token:f.current()};if(g[2])l.placeholder=g[2].substr(1);l=a.tabstop(l)}e.append(l)}return e.toString()},upgrade:function(b,a){var e=0,f={tabstop:function(b){var c=parseInt(b.group);c>e&&(e=c);return b.placeholder?"${"+(c+a)+":"+b.placeholder+"}":"${"+(c+a)+"}"}};d.each(["start",
"end","content"],function(a){b[a]=this.processText(b[a],f)},this);return e},variablesResolver:function(b){var a={},e=h("resources");return function(f,k){if(k=="child")return f;if(k=="cursor")return h("utils").getCaretPlaceholder();var l=b.attribute(k);if(!d.isUndefined(l))return l;if(l=e.getVariable(k))return l;a[k]||(a[k]=g++);return"${"+a[k]+":"+k+"}"}},resetPlaceholderCounter:function(){console.log("deprecated");g=100},resetTabstopIndex:function(){f=0;g=100}}});
zen_coding.define("preferences",function(h,d){var g={},f={},b=null,c=null;return{define:function(a,b,c){var g=a;d.isString(a)&&(g={},g[a]={value:b,description:c});d.each(g,function(a,b){f[b]=d.isObject(a)&&"value"in a&&d.keys(a).length<3?a:{value:a}})},set:function(a,b){var c=a;d.isString(a)&&(c={},c[a]=b);d.each(c,function(a,b){if(!(b in f))throw'Property "'+b+'" is not defined. You should define it first with `define` method of current module';a!==f[b].value?g[b]=a:b in g&&delete g[p]})},get:function(a){if(a in
g)return g[a];if(a in f)return f[a].value},getArray:function(a){a=this.get(a);d.isUndefined(a)||(a=d.map(a.split(","),h("utils").trim),a.length||(a=null));return a},description:function(a){return a in f?f[a].description:void 0},remove:function(a){d.isArray(a)||(a=[a]);d.each(a,function(a){a in g&&delete g[a];a in f&&delete f[a]})},list:function(){return d.map(d.keys(f).sort(),function(a){return{name:a,value:this.get(a),description:f[a].description}},this)},load:function(a){d.each(a,function(a,b){this.set(b,
a)},this)},reset:function(){g={}},_startTest:function(){b=f;c=g;f={};g={}},_stopTest:function(){f=b;g=c}}});
zen_coding.define("filters",function(h,d){function g(b){return!b?[]:d.isString(b)?b.split(/[\|,]/g):b}var f={};return{add:function(b,c){f[b]=c},apply:function(b,c,a){var e=h("utils"),a=h("profile").get(a);d.each(g(c),function(c){(c=e.trim(c.toLowerCase()))&&c in f&&(b=f[c](b,a))});return b},composeList:function(b,c,a){c=h("profile").get(c);b=g(c.filters||h("resources").getSubset(b,"filters")||"html");a&&(b=b.concat(g(a)));if(!b||!b.length)b=g("html");return b},extractFromAbbreviation:function(b){var c=
"",b=b.replace(/\|([\w\|\-]+)$/,function(a,b){c=b;return""});return[b,g(c)]}}});
zen_coding.define("elements",function(h,d){function g(a){return{data:a}}var f={},b=/([\w\-]+)\s*=\s*(['"])(.*?)\2/g,c={add:function(a,b){var c=this;f[a]=function(){var d=b.apply(c,arguments);if(d)d.type=a;return d}},get:function(a){return f[a]},create:function(a){var b=[].slice.call(arguments,1),c=this.get(a);return c?c.apply(this,b):null},is:function(a,b){return a&&a.type===b}};c.add("element",function(a,c,f){var g={name:a,is_empty:!!f};if(c)if(g.attributes=[],d.isArray(c))g.attributes=c;else if(d.isString(c))for(;a=
b.exec(c);)g.attributes.push({name:a[1],value:a[3]});else d.each(c,function(a,b){g.attributes.push({name:b,value:a})});return g});c.add("snippet",g);c.add("reference",g);c.add("empty",function(){return{}});return c});
zen_coding.define("editTree",function(h,d,g){function f(a,b){this.options=d.extend({offset:0},b);this.source=a;this._children=[];this._positions={name:0};this.initialize.apply(this,arguments)}function b(a,b,c){this.parent=a;this._name=b.value;this._value=c?c.value:"";this._positions={name:b.start,value:c?c.start:-1};this.initialize.apply(this,arguments)}var c=h("range").create;f.extend=g.extend;f.prototype={initialize:function(){},_updateSource:function(a,b,f){var g=c(b,d.isUndefined(f)?0:f-b),l=
a.length-g.length(),n=function(a){d.each(a,function(b,c){b>=g.end&&(a[c]+=l)})};n(this._positions);d.each(this.list(),function(a){n(a._positions)});this.source=h("utils").replaceSubstring(this.source,a,g)},add:function(a,c){var d=new b(a,c);this._children.push(d);return d},get:function(a){return d.isNumber(a)?this.list()[a]:d.isString(a)?d.find(this.list(),function(b){return b.name()===a}):a},getAll:function(a){d.isArray(a)||(a=[a]);var b=[],c=[];d.each(a,function(a){d.isString(a)?b.push(a):d.isNumber(a)&&
c.push(a)});return d.filter(this.list(),function(a,f){return d.include(c,f)||d.include(b,a.name())})},value:function(a,b,c){var f=this.get(a);if(f)return f.value(b);if(!d.isUndefined(b))return this.add(a,b,c)},values:function(a){return d.map(this.getAll(a),function(a){return a.value()})},remove:function(a){if(a=this.get(a))this._updateSource("",a.fullRange()),this._children=d.without(this._children,a)},list:function(){return this._children},indexOf:function(a){return d.indexOf(this.list(),this.get(a))},
name:function(a){if(!d.isUndefined(a)&&this._name!==(a=String(a)))this._updateSource(a,this._positions.name,this._positions.name+this._name.length),this._name=a;return this._name},nameRange:function(a){return c(this._positions.name+(a?this.options.offset:0),this.name())},range:function(a){return c(a?this.options.offset:0,this.toString())},itemFromPosition:function(a,b){return d.find(this.list(),function(c){return c.range(b).inside(a)})},toString:function(){return this.source}};b.extend=g.extend;b.prototype=
{initialize:function(){},_pos:function(a,b){return a+(b?this.parent.options.offset:0)},value:function(a){if(!d.isUndefined(a)&&this._value!==(a=String(a)))this.parent._updateSource(a,this.valueRange()),this._value=a;return this._value},name:function(a){if(!d.isUndefined(a)&&this._name!==(a=String(a)))this.parent._updateSource(a,this.nameRange()),this._name=a;return this._name},namePosition:function(a){return this._pos(this._positions.name,a)},valuePosition:function(a){return this._pos(this._positions.value,
a)},range:function(a){return c(this.namePosition(a),this.toString())},fullRange:function(a){return this.range(a)},nameRange:function(a){return c(this.namePosition(a),this.name())},valueRange:function(a){return c(this.valuePosition(a),this.value())},toString:function(){return this.name()+this.value()},valueOf:function(){return this.toString()}};return{EditContainer:f,EditElement:b,createToken:function(a,b,c){a={start:a||0,value:b||"",type:c};a.end=a.start+a.value.length;return a}}});
zen_coding.define("cssEditTree",function(h,d){function g(a,b){return h("range").create(a,b)}function f(a,b){var b=b||e|i,c=["white","line"];if((b&i)==i)for(;a.length&&d.include(c,d.last(a).type);)a.pop();if((b&e)==e)for(;a.length&&d.include(c,a[0].type);)a.shift();return a}function b(a){var b=["white","line",":"],c=[],h,k;a.nextUntil(function(){return!d.include(b,this.itemNext().type)});for(k=a.current().end;h=a.next();){if(h.type=="}"||h.type==";")return f(c,e|(h.type=="}"?i:0)),c.length?(k=c[0].start,
a=d.last(c).end):a=k,g(k,a-k);c.push(h)}if(c.length)return g(c[0].start,d.last(c).end-c[0].start)}function c(a){var b=h("stringStream").create(a),c=[],e=/[\s\u00a0,]/,f=function(){b.next();c.push(g(b.start,b.current()));b.start=b.pos};b.eatSpace();for(b.start=b.pos;a=b.next();)if(a=='"'||a=="'"){b.next();if(!b.skipTo(a))break;f()}else if(a=="("){b.backUp(1);if(!b.skipToPair("(",")"))break;b.backUp(1);f()}else if(e.test(a))c.push(g(b.start,b.current().length-1)),b.eatWhile(e),b.start=b.pos;f();return d.chain(c).filter(function(a){return!!a.length()}).uniq(!1,
function(a){return a.toString()}).value()}var a={styleBefore:"\n\t",styleSeparator:": ",offset:0},e=1,i=2,k=h("editTree").EditContainer.extend({initialize:function(c){d.defaults(this.options,a);var e=h("editTree"),i=h("tokenIterator").create(h("cssParser").parse(c)),k,t=[],m;for(k=i.position();m=i.next();){if(m.type=="{")break;t.push(m)}f(t);t.length?(k=t[0].start,t=d.last(t).end):t=k;k=g(k,t-k);this._positions.name=k.start;this._name=k.substring(c);if(!i.current()||i.current().type!="{")throw"Invalid CSS rule";
for(this._positions.contentStart=i.position()+1;k=i.next();)k.type=="identifier"&&(k=g(k),t=b(i),m=i.current()&&i.current().type==";"?g(i.current()):g(t.end,0),this._children.push(new l(this,e.createToken(k.start,k.substring(c)),e.createToken(t.start,t.substring(c)),e.createToken(m.start,m.substring(c)))));this._saveStyle()},_saveStyle:function(){var a=this._positions.contentStart,b=this.source,c=h("utils");d.each(this.list(),function(e){e.styleBefore=b.substring(a,e.namePosition());var f=c.splitByLines(e.styleBefore);
if(f.length>1)e.styleBefore="\n"+d.last(f);e.styleSeparator=b.substring(e.nameRange().end,e.valuePosition());e.styleBefore=d.last(e.styleBefore.split("*/"));e.styleSeparator=e.styleSeparator.replace(/\/\*.*?\*\//g,"");a=e.range().end})},add:function(a,b,c){var e=this.list(),f=this._positions.contentStart,g=d.pick(this.options,"styleBefore","styleSeparator"),i=h("editTree");if(d.isUndefined(c))c=e.length;var k=e[c];if(k)f=k.fullRange().start;else if(k=e[c-1])k.end(";"),f=k.range().end;k&&(g=d.pick(k,
"styleBefore","styleSeparator"));a=i.createToken(f+g.styleBefore.length,a);b=i.createToken(a.end+g.styleSeparator.length,b);i=new l(this,a,b,i.createToken(b.end,";"));d.extend(i,g);this._updateSource(i.styleBefore+i.toString(),f);this._children.splice(c,0,i);return i}}),l=h("editTree").EditElement.extend({initialize:function(a,b,c,e){this.styleBefore=a.options.styleBefore;this.styleSeparator=a.options.styleSeparator;this._end=e.value;this._positions.end=e.start},valueParts:function(a){var b=c(this.value());
if(a){var e=this.valuePosition(!0);d.each(b,function(a){a.shift(e)})}return b},end:function(a){if(!d.isUndefined(a)&&this._end!==a)this.parent._updateSource(a,this._positions.end,this._positions.end+this._end.length),this._end=a;return this._end},fullRange:function(a){a=this.range(a);a.start-=this.styleBefore.length;return a},toString:function(){return this.name()+this.styleSeparator+this.value()+this.end()}});return{parse:function(a,b){return new k(a,b)},parseFromPosition:function(a,b,c){c=this.extractRule(a,
b,c);return!c||!c.inside(b)?null:this.parse(c.substring(a),{offset:c.start})},extractRule:function(a,b,c){for(var e="",d=a.length,f=-1,g;b>=0;){g=a.charAt(b);if(g=="{"){f=b;break}else if(g=="}"&&!c){b++;break}b--}for(;b<d;){g=a.charAt(b);if(g=="{")f=b;else if(g=="}"){f!=-1&&(e=a.substring(f,b+1));break}b++}if(e){b=f-1;for(c="";b>=0;){g=a.charAt(b);if("{}/\\<>".indexOf(g)!=-1)break;b--}c=a.substring(b+1,f).replace(/^[\s\n\r]+/m,"");return h("range").create(f-c.length,e.length+c.length)}return null},
baseName:function(a){return a.replace(/^\s*\-\w+\-/,"")},findParts:c}});
zen_coding.define("xmlEditTree",function(h,d){var g={styleBefore:" ",styleSeparator:"=",styleQuote:'"',offset:0},f=/^<([\w\:\-]+)((?:\s+[\w\-:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*)\s*(\/?)>/m,b=h("editTree").EditContainer.extend({initialize:function(a){d.defaults(this.options,g);this._positions.name=1;var b=null,f=h("xmlParser").parse(a),k=h("range");d.each(f,function(d){d.value=k.create(d).substring(a);switch(d.type){case "tag":if(/^<[^\/]+/.test(d.value))this._name=d.value.substring(1);
break;case "attribute":b&&this._children.push(new c(this,b));b=d;break;case "string":this._children.push(new c(this,b,d)),b=null}},this);b&&this._children.push(new c(this,b));this._saveStyle()},_saveStyle:function(){var a=this.nameRange().end,b=this.source;d.each(this.list(),function(c){c.styleBefore=b.substring(a,c.namePosition());if(c.valuePosition()!==-1)c.styleSeparator=b.substring(c.namePosition()+c.name().length,c.valuePosition()-c.styleQuote.length);a=c.range().end})},add:function(a,b,f){var g=
this.list(),l=this.nameRange().end,n=h("editTree"),o=d.pick(this.options,"styleBefore","styleSeparator","styleQuote");if(d.isUndefined(f))f=g.length;var q=g[f];if(q)l=q.fullRange().start;else if(q=g[f-1])l=q.range().end;q&&(o=d.pick(q,"styleBefore","styleSeparator","styleQuote"));b=o.styleQuote+b+o.styleQuote;a=new c(this,n.createToken(l+o.styleBefore.length,a),n.createToken(l+o.styleBefore.length+a.length+o.styleSeparator.length,b));d.extend(a,o);this._updateSource(a.styleBefore+a.toString(),l);
this._children.splice(f,0,a);return a}}),c=h("editTree").EditElement.extend({initialize:function(a,b,c){this.styleBefore=a.options.styleBefore;this.styleSeparator=a.options.styleSeparator;b="";a=a.options.styleQuote;if(c)b=c.value,a=b.charAt(0),a=='"'||a=="'"?b=b.substring(1):a="",a&&b.charAt(b.length-1)==a&&(b=b.substring(0,b.length-1));this.styleQuote=a;this._value=b;this._positions.value=c?c.start+a.length:-1},fullRange:function(a){a=this.range(a);a.start-=this.styleBefore.length;return a},toString:function(){return this.name()+
this.styleSeparator+this.styleQuote+this.value()+this.styleQuote}});return{parse:function(a,c){return new b(a,c)},parseFromPosition:function(a,b,c){c=this.extractTag(a,b,c);return!c||!c.inside(b)?null:this.parse(c.substring(a),{offset:c.start})},extractTag:function(a,b,c){var d=a.length,g,n=h("range"),o=Math.min(2E3,d),q=null,r=function(b){var c;if(a.charAt(b)=="<"&&(c=a.substr(b,o).match(f)))return n.create(b,c[0])};for(g=b;g>=0;g--)if(q=r(g))break;if(q&&(q.inside(b)||c))return q;if(!q&&c)return null;
for(g=b;g<d;g++)if(q=r(g))return q}}});
zen_coding.define("expandAbbreviation",function(h,d){var g=h("handlerList").create(),f=h("actions");f.add("expand_abbreviation",function(b,c,a){return g.exec(!1,d.toArray(arguments))});f.add("expand_abbreviation_with_tab",function(b,c,a){f.run("expand_abbreviation",b,c,a)||b.replaceContent(h("resources").getVariable("indentation"),b.getCaretPos())},{hidden:!0});g.add(function(b,c,a){var c=h("editorUtils").outputInfo(b,c,a),a=b.getSelectionRange().end,d="",f;d=h("range").create(b.getSelectionRange());
f=String(b.getContent());if(d.length())f=d.substring(f);else{var g=b.getCurrentLineRange();f=h("actionUtils").extractAbbreviation(f.substring(g.start,d.start))}return f&&(d=zen_coding.expandAbbreviation(f,c.syntax,c.profile,h("actionUtils").captureContext(b)))?(b.replaceContent(d,a-f.length,a),!0):!1},{order:-1});return{addHandler:function(b,c){g.add(b,c)},removeHandler:function(b){g.remove(b,options)}}});
zen_coding.define("wrapWithAbbreviation",function(h){h("actions").add("wrap_with_abbreviation",function(d,g,f,b){var b=h("editorUtils").outputInfo(d,f,b),c=h("utils"),a=h("editorUtils"),e=h("html_matcher"),g=g||d.prompt("Enter abbreviation");if(!g)return null;var g=String(g),i=d.getSelectionRange(),f=i.start,i=i.end;if(f==i){i=e(b.content,f,b.profile);if(!i||i[0]==-1)return!1;i=c.narrowToNonSpace(b.content,i[0],i[1]-i[0]);f=i.start;i=i.end}c=c.escapeText(b.content.substring(f,i));return(g=h("wrapWithAbbreviation").wrap(g,
a.unindent(d,c),b.syntax,b.profile))?(d.replaceContent(g,f,i),!0):!1});return{wrap:function(d,g,f,b){var c=h("filters"),a=h("utils"),f=f||zen_coding.defaultSyntax(),b=b||zen_coding.defaultProfile();h("tabStops").resetTabstopIndex();d=c.extractFromAbbreviation(d);return(g=h("abbreviationParser").parse(d[0],{syntax:f,pastedContent:g}))?(f=c.composeList(f,b,d[1]),c.apply(g,f,b),a.replaceVariables(g.toString())):null}}});
zen_coding.exec(function(h,d){function g(b,a){var e=a-(b.options.offset||0),f=/^[\s\n\r]/;return d.find(b.list(),function(a){return a.range().end===e?f.test(b.source.charAt(e)):a.range().inside(e)})}function f(b,a,d,f){for(var g=-1,l=-1;a--;)if(b.substr(a,d.length)==d){g=a;break}if(g!=-1){a=g;for(d=b.length;d>=a++;)if(b.substr(a,f.length)==f){l=a+f.length;break}}return g!=-1&&l!=-1?h("range").create(g,l-g):null}function b(b,a,d,g){function k(b){return b.replace(RegExp("^"+r.escapeForRegexp(a)+"\\s*"),
function(a){o-=a.length;return""}).replace(RegExp("\\s*"+r.escapeForRegexp(d)+"$"),"")}var l=h("editorUtils"),n=l.outputInfo(b).content,o=b.getCaretPos(),q=null,r=h("utils");(q=f(n,o,a,d))&&q.overlap(g)?(g=q,q=k(g.substring(n))):(q=a+" "+g.substring(n).replace(RegExp(r.escapeForRegexp(a)+"\\s*|\\s*"+r.escapeForRegexp(d),"g"),"")+" "+d,o+=a.length+1);return q!==null?(b.setCaretPos(g.start),b.replaceContent(l.unindent(b,q),g.start,g.end),b.setCaretPos(o),!0):!1}h("actions").add("toggle_comment",function(c){var a=
h("editorUtils").outputInfo(c);if(a.syntax=="css"){var d=c.getCaretPos(),f=h("html_matcher").getTags(a.content,d);if(f&&f[0]&&f[0].type=="tag"&&f[0].start<=d&&f[0].end>=d)a.syntax="html"}if(a.syntax=="css"){f=h("range").create(c.getSelectionRange());a=h("editorUtils").outputInfo(c);if(!f.length()&&(d=h("cssEditTree").parseFromPosition(a.content,c.getCaretPos())))f=(f=g(d,c.getCaretPos()))?f.range(!0):h("range").create(d.nameRange(!0).start,d.source);f.length()||(f=h("range").create(c.getCurrentLineRange()),
h("utils").narrowToNonSpace(a.content,f));c=b(c,"/*","*/",f)}else{a=h("range").create(c.getSelectionRange());d=h("editorUtils").outputInfo(c);if(!a.length()&&(d=h("html_matcher").getTags(d.content,c.getCaretPos(),d.profile))&&d[0])a.start=d[0].start,a.end=d[1]?d[1].end:d[0].end;c=b(c,"<\!--","--\>",a)}return c})});
zen_coding.exec(function(h){function d(d,f,b){function c(b){for(var c=b;c>=0;){var d=a.charAt(c);if(d=="\n"||d=="\r")break;c--}return a.substring(c,b)}for(var f=f||1,b=d.getCaretPos()+(b||0),a=String(d.getContent()),d=a.length,e=-1,h=/^\s+$/;b<=d&&b>=0;){b+=f;var k=a.charAt(b),l=a.charAt(b+1),n=a.charAt(b-1);switch(k){case '"':case "'":l==k&&n=="="&&(e=b+1);break;case ">":l=="<"&&(e=b+1);break;case "\n":case "\r":h.test(c(b-1))&&(e=b)}if(e!=-1)break}return e}h=h("actions");h.add("prev_edit_point",
function(g){var f=g.getCaretPos(),b=d(g,-1);b==f&&(b=d(g,-1,-2));return b!=-1?(g.setCaretPos(b),!0):!1},{label:"Previous Edit Point"});h.add("next_edit_point",function(g){var f=d(g,1);f!=-1&&g.setCaretPos(f)})});
zen_coding.exec(function(h,d){function g(a,b,c,d){var e=h("range"),f=h("editorUtils").outputInfo(a).content,g=f.length,i,k=e.create(-1,0),l=e.create(a.getSelectionRange());i=l.start;for(var n=1E5;i>=0&&i<g&&--n>0;){if(e=c(f,i,b)){if(k.equal(e))break;k=e.clone();if(i=d(e.substring(f),e.start,l.clone()))return a.createSelection(i.start,i.end),!0;else i=b?e.start:e.end-1}i+=b?-1:1}return!1}function f(a){var b=!0;return g(a,!1,function(a,c){if(b){b=!1;var d;a:{d=c;for(var e;d>=0;){if(e=i(a,d)){d=e;break a}d--}d=
null}return d}else return i(a,c)},function(a,b,c){return e(a,b,c,!1)})}function b(a){return g(a,!0,i,function(a,b,c){return e(a,b,c,!0)})}function c(b,c,e){var e=e||0,f=h("range"),g=[],i=-1,l="",n="",o,r;d.each(c,function(c){switch(c.type){case "tag":r=b.substring(c.start,c.end);/^<[\w\:\-]/.test(r)&&g.push(f.create({start:c.start+1,end:c.end}));break;case "attribute":i=c.start;l=b.substring(c.start,c.end);break;case "string":g.push(f.create(i,c.end-i)),o=f.create(c),n=o.substring(b),k(n.charAt(0))&&
o.start++,k(n.charAt(n.length-1))&&o.end--,g.push(o),l=="class"&&(g=g.concat(a(o.substring(b),o.start)))}});d.each(g,function(a){a.shift(e)});return d.chain(g).filter(function(a){return!!a.length()}).uniq(!1,function(a){return a.toString()}).value()}function a(a,b){var b=b||0,c=[],d=h("stringStream").create(a),e=h("range");d.eatSpace();d.start=d.pos;for(var f;f=d.next();)if(/[\s\u00a0]/.test(f))c.push(e.create(d.start+b,d.pos-d.start-1)),d.eatSpace(),d.start=d.pos;c.push(e.create(d.start+b,d.pos-
d.start));return c}function e(a,b,e,f){a=c(a,h("xmlParser").parse(a),b);f&&a.reverse();return(b=d.find(a,function(a){return a.equal(e)}))?(f=d.indexOf(a,b),f<a.length-1?a[f+1]:null):f?d.find(a,function(a){return a.start<e.start}):!b&&(f=d.filter(a,function(a){return a.inside(e.end)}),f.length>1)?f[1]:d.find(a,function(a){return a.end>e.end})}function i(a,b){var c;if(a.charAt(b)=="<"&&(c=a.substring(b,a.length).match(r)))return h("range").create(b,c[0])}function k(a){return a=='"'||a=="'"}function l(a){var b=
a.valueRange(!0),c=[a.range(!0),b],e=h("stringStream"),f=h("cssEditTree"),g=h("range"),i=a.value();d.each(a.valueParts(),function(a){var h=a.clone();c.push(h.shift(b.start));var m=e.create(a.substring(i));if(m.match(/^[\w\-]+\(/,!0)){m.start=m.pos;m.skipToPair("(",")");var k=m.current();c.push(g.create(h.start+m.start,k));d.each(f.findParts(k),function(a){c.push(g.create(h.start+m.start+a.start,a.substring(k)))})}});return d.chain(c).filter(function(a){return!!a.length()}).uniq(!1,function(a){return a.toString()}).value()}
function n(a,b,c){var e=null,f=null,g=a.list(),h,i;c?(g.reverse(),h=function(a){return a.range(!0).start<=b.start},i=function(a){return a.start<b.start}):(h=function(a){return a.range(!0).end>=b.end},i=function(a){return a.end>b.start});for(;e=d.find(g,h);){a=l(e);c&&a.reverse();if(f=d.find(a,function(a){return a.equal(b)})){if(f=d.indexOf(a,f),f!=a.length-1){f=a[f+1];break}}else{f=d.filter(a,function(a){return a.inside(b.end)});if(f.length>1){f=f[1];break}if(f=d.find(a,i))break}f=null;b.start=b.end=
c?e.range(!0).start-1:e.range(!0).end+1}return f}function o(a,b,c){a=h("cssEditTree").parse(a,{offset:b});b=a.nameRange(!0);return c.end<b.end?b:n(a,c,!1)}function q(a,b,c){b=h("cssEditTree").parse(a,{offset:b});a=n(b,c,!0);return!a&&(b=b.nameRange(!0),c.start>b.start)?b:a}var r=/^<([\w\:\-]+)((?:\s+[\w\-:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*)\s*(\/?)>/,t=h("actions");t.add("select_next_item",function(a){return a.getSyntax()=="css"?g(a,!1,h("cssEditTree").extractRule,o):f(a)});t.add("select_previous_item",
function(a){return a.getSyntax()=="css"?g(a,!0,h("cssEditTree").extractRule,q):b(a)})});
zen_coding.exec(function(h){function d(b,c,a){var c=String((c||"out").toLowerCase()),d=h("editorUtils").outputInfo(b,a),a=d.syntax,g=h("range"),k=g.create(b.getSelectionRange()),d=d.content,l=null,l=f.last_match.opening_tag,n=f.last_match.closing_tag;if(c=="in"&&l&&k.length())if(n)l.start==k.start?d.charAt(l.end)=="<"?(c=g.create(f.find(d,l.end+1,a)),l=c.start==l.end&&c.end==n.start?g.create(f(d,l.end+1,a)):g.create(l.end,n.start-l.end)):l=g.create(l.end,n.start-l.end):(c=d.substring(0,n.start).indexOf("<",
l.end),l=g.create(f(d,c!=-1?c+1:l.end,a)));else return!1;else l=g.create(f(d,k.end,a));return l&&l.start!=-1?(b.createSelection(l.start,l.end),!0):!1}var g=h("actions"),f=h("html_matcher");g.add("match_pair",d,{hidden:!0});g.add("match_pair_inward",function(b){return d(b,"in")},{label:"HTML/Match Pair Tag (inward)"});g.add("match_pair_outward",function(b){return d(b,"out")},{label:"HTML/Match Pair Tag (outward)"});g.add("matching_pair",function(b){var c=String(b.getContent()),a=b.getCaretPos();c.charAt(a)==
"<"&&a++;var d=f.getTags(c,a,String(b.getProfileName()));if(d&&d[0]&&(c=d[0],d=d[1]))if(c.start<=a&&c.end>=a)return b.setCaretPos(d.start),!0;else if(d.start<=a&&d.end>=a)return b.setCaretPos(c.start),!0;return!1},{label:"HTML/Go To Matching Tag Pair"})});
zen_coding.exec(function(h){h("actions").add("remove_tag",function(d){var g=h("utils"),f=h("editorUtils").outputInfo(d),b=h("html_matcher").getTags(f.content,d.getCaretPos(),f.profile);if(b&&b[0]){if(b[1]){var c=g.narrowToNonSpace(f.content,b[0].end,b[1].start-b[0].end),a=g.findNewlineBounds(f.content,c.start),a=g.getLinePadding(a.substring(f.content)),f=c.substring(f.content),f=g.unindentString(f,a);d.replaceContent(g.getCaretPlaceholder()+f,b[0].start,b[1].end)}else d.replaceContent(g.getCaretPlaceholder(),
b[0].start,b[0].end);return!0}return!1},{label:"HTML/Remove Tag"})});
zen_coding.exec(function(h){h("actions").add("split_join_tag",function(d,g){var f=h("html_matcher"),b=h("editorUtils").outputInfo(d,null,g),c=h("profile").get(b.profile);if((f=f.getTags(b.content,d.getCaretPos(),b.profile))&&f[0]){if(f[1]){b=h("utils");c=f[0].full_tag.replace(/\s*>$/,(c.self_closing_tag===!0?"/":" /")+">");if(c.length+f[0].start<d.getCaretPos())c+=b.getCaretPlaceholder();else var a=d.getCaretPos()-f[0].start,c=b.replaceSubstring(c,b.getCaretPlaceholder(),a);d.replaceContent(c,f[0].start,
f[1].end)}else{var e=h("utils"),b=e.getNewline(),a=h("resources").getVariable("indentation"),e=e.getCaretPlaceholder(),c=c.tag_nl===!0?b+a+e+b:e,c=f[0].full_tag.replace(/\s*\/>$/,">")+c+"</"+f[0].name+">";d.replaceContent(c,f[0].start,f[0].end)}f=!0}else f=!1;return f},{label:"HTML/Split\\Join Tag Declaration"})});
zen_coding.define("reflectCSSValue",function(h,d){function g(c){var a=h("cssEditTree"),d=h("editorUtils").outputInfo(c),c=c.getCaretPos();if(a=a.parseFromPosition(d.content,c))if(d=a.itemFromPosition(c,!0)){var f=a.source,g=a.options.offset,c=c-g-d.range().start;b.exec(!1,[d]);if(f!==a.source)return{data:a.source,start:g,end:g+f.length,caret:g+d.range().start+c}}}function f(b){var b=h("cssEditTree").baseName(b),a;if(b=="opacity"||b=="filter")return/^(?:\-\w+\-)?(?:opacity|filter)$/;else if(a=b.match(/^border-radius-(top|bottom)(left|right)/))return RegExp("^(?:\\-\\w+\\-)?(?:"+
b+"|border-"+a[1]+"-"+a[2]+"-radius)$");else if(a=b.match(/^border-(top|bottom)-(left|right)-radius/))return RegExp("^(?:\\-\\w+\\-)?(?:"+b+"|border-radius-"+a[1]+a[2]+")$");return RegExp("^(?:\\-\\w+\\-)?"+b+"$")}var b=h("handlerList").create();h("actions").add("reflect_css_value",function(b){return b.getSyntax()!="css"?!1:h("actionUtils").compoundUpdate(b,g(b))},{label:"CSS/Reflect Value"});b.add(function(b){var a=f(b.name());d.each(b.parent.list(),function(d){if(a.test(d.name())){var f;var g=b.name(),
l=b.value(),n=d.name();f=d.value();var o=h("cssEditTree"),q=h("utils"),g=o.baseName(g),n=o.baseName(n);f=g=="opacity"&&n=="filter"?f.replace(/opacity=[^)]*/i,"opacity="+Math.floor(parseFloat(l)*100)):g=="filter"&&n=="opacity"?(g=l.match(/opacity=([^)]*)/i))?q.prettifyNumber(parseInt(g[1])/100):f:l;d.value(f)}})},{order:-1});return{addHandler:function(c,a){b.add(c,a)},removeHandler:function(c){b.remove(c,options)}}});
zen_coding.exec(function(h){h("actions").add("evaluate_math_expression",function(d){var g=h("actionUtils"),f=h("utils"),b=String(d.getContent()),c=h("range").create(d.getSelectionRange());c.length()||(c=g.findExpressionBounds(d,function(a){return f.isNumeric(a)||".+-*/\\".indexOf(a)!=-1}));if(c&&c.length()){g=c.substring(b);g=g.replace(/([\d\.\-]+)\\([\d\.\-]+)/g,"Math.round($1/$2)");try{var a=f.prettifyNumber((new Function("return "+g))());d.replaceContent(a,c.start,c.end);d.setCaretPos(c.start+
a.length);return!0}catch(e){}}return!1},{label:"Numbers/Evaluate Math Expression"})});
zen_coding.exec(function(h,d){function g(b,a){var e=h("utils"),g=!1,k=!1,l=h("actionUtils").findExpressionBounds(b,function(a,b,c){return e.isNumeric(a)?!0:a=="."?!e.isNumeric(c.charAt(b+1))?!1:k?!1:k=!0:a=="-"?g?!1:g=!0:!1});if(l&&l.length()){var n=l.substring(String(b.getContent())),o=parseFloat(n);if(!d.isNaN(o)){o=e.prettifyNumber(o+a);if(/^(\-?)0+[1-9]/.test(n)){var q="";RegExp.$1&&(q="-",o=o.substring(1));o=o.split(".");o[0]=e.zeroPadString(o[0],f(n));o=q+o.join(".")}b.replaceContent(o,l.start,
l.end);b.createSelection(l.start,l.start+o.length);return!0}}return!1}function f(b){b=b.replace(/^\-/,"");return~b.indexOf(".")?b.split(".")[0].length:b.length}var b=h("actions");d.each([1,-1,10,-10,0.1,-0.1],function(c){var a=c>0?"increment":"decrement";b.add(a+"_number_by_"+String(Math.abs(c)).replace(".","").substring(0,2),function(a){return g(a,c)},{label:"Numbers/"+a.charAt(0).toUpperCase()+a.substring(1)+" number by "+Math.abs(c)})})});
zen_coding.exec(function(h){h("actions").add("merge_lines",function(d){var g=h("html_matcher"),f=h("utils"),b=h("editorUtils").outputInfo(d),c=h("range").create(d.getSelectionRange());if(!c.length()&&(g=g(b.content,d.getCaretPos(),b.profile)))c.start=g[0],c.end=g[1];if(c.length()){b=c.substring(b.content);f=f.splitByLines(b);for(b=1;b<f.length;b++)f[b]=f[b].replace(/^\s+/,"");b=f.join("").replace(/\s{2,}/," ");d.replaceContent(b,c.start,c.end);d.createSelection(c.start,c.start+b.length);return!0}return!1})});
zen_coding.exec(function(h){function d(d,f,b){b=b||0;return f.charAt(b)==d.charAt(0)&&f.substr(b,d.length)==d}h("actions").add("encode_decode_data_url",function(g){var f=String(g.getSelection()),b=g.getCaretPos();if(!f)for(var c=String(g.getContent());b-- >=0;)if(d("src=",c,b)){if(c=c.substr(b).match(/^(src=(["'])?)([^'"<>\s]+)\1?/))f=c[3],b+=c[1].length;break}else if(d("url(",c,b)){if(c=c.substr(b).match(/^(url\((['"])?)([^'"\)\s]+)\1?/))f=c[3],b+=c[1].length;break}if(f)if(d("data:",f))if(c=String(g.prompt("Enter path to file (absolute or relative)"))){var a=
h("file"),e=a.createPath(g.getFilePath(),c);if(!e)throw"Can't save file";a.save(e,h("base64").decode(f.replace(/^data\:.+?;.+?,/,"")));g.replaceContent("$0"+c,b,b+f.length);g=!0}else g=!1;else{c=h("file");a=h("actionUtils");e=g.getFilePath();if(e===null)throw"You should save your file before using this action";e=c.locateFile(e,f);if(e===null)throw"Can't find "+f+" file";var i=h("base64").encode(String(c.read(e)));if(!i)throw"Can't encode file content to base64";i="data:"+(a.mimeTypes[String(c.getExt(e))]||
"application/octet-stream")+";base64,"+i;g.replaceContent("$0"+i,b,b+f.length);g=!0}else g=!1;return g},{label:"Encode\\Decode data:URL image"})});
zen_coding.exec(function(h,d){function g(d,b){var c;if(b){if(/^data:/.test(b))c=h("base64").decode(b.replace(/^data\:.+?;.+?,/,""));else{c=h("file");var a=c.locateFile(d.getFilePath(),b);if(a===null)throw"Can't find "+b+" file";c=String(c.read(a))}return h("actionUtils").getImageSize(c)}}h("actions").add("update_image_size",function(f){var b;if(String(f.getSyntax())=="css")a:{b=f.getCaretPos();var c=h("editorUtils").outputInfo(f);if(c=h("cssEditTree").parseFromPosition(c.content,b,!0)){var a=c.itemFromPosition(b,
!0),e;if(a&&(e=/url\((["']?)(.+?)\1\)/i.exec(a.value()||"")))if(e=g(f,e[2])){a=c.range(!0);c.value("width",e.width+"px");c.value("height",e.height+"px",c.indexOf("width")+1);b=d.extend(a,{data:c.toString(),caret:b});break a}}b=null}else a:{b=f.getCaretPos();c=h("editorUtils").outputInfo(f);if((c=h("xmlEditTree").parseFromPosition(c.content,b,!0))&&c.name().toLowerCase()=="img")if(e=g(f,c.value("src"))){a=c.range(!0);c.value("width",e.width);c.value("height",e.height,c.indexOf("width")+1);b=d.extend(a,
{data:c.toString(),caret:b});break a}b=null}return h("actionUtils").compoundUpdate(f,b)})});
zen_coding.define("cssResolver",function(h,d){function g(a){var b=a&&a.charCodeAt(0);return a&&a=="."||b>47&&b<58}function f(a){a=h("utils").trim(a);if(~a.indexOf("/*"))return!1;a=h("tabStops").processText(a,{replaceCarets:!0,tabstop:function(){return"value"}});return a.split(":").length==2}function b(a){var b=h("utils"),a=b.trim(a);if(a.indexOf(":")==-1)return{name:a,value:o};a=a.split(":",2);return{name:b.trim(a[0]),value:b.trim(a[1])}}function c(a,b){var c=l[b];c||(c=d.find(l,function(a){return a.prefix==
b}));return c&&c.supports&&d.include(c.supports,a)}function a(a){var b=[];d.each(l,function(d,e){c(a,e)&&b.push(e)});b.length||d.each(l,function(a,c){a.obsolete||b.push(c)});return b}function e(a,b){d.isString(b)&&(b={prefix:b});l[a]=d.extend({},k,b)}function i(a,b){if(!d.isString(a))a=a.data;if(!f(a))return a;b&&(~a.indexOf(";")?a=a.split(";").join(" !important;"):a+=" !important");var c=a.indexOf(":");return a=a.substring(0,c).replace(/\s+$/,"")+q.get("css.valueSeparator")+h("utils").trim(a.substring(c+
1))}var k={prefix:"zen",obsolete:!1,transformName:function(a){return"-"+this.prefix+"-"+a},supports:null},l={},n={p:"%",e:"em",x:"ex"},o="${1};",q=h("preferences");q.define("css.valueSeparator",": ","Defines a symbol that should be placed between CSS property and value when expanding CSS abbreviations.");var r=d.template("A comma-separated list of CSS properties that may have <code><%= vendor %></code> vendor prefix. This list is used to generate a list of prefixed properties when expanding <code>-property</code> abbreviations. Empty list means that all possible CSS values may have <code><%= vendor %></code> prefix.");
d.each({webkit:"animation-delay, animation-direction, animation-duration, animation-fill-mode, animation-iteration-count, animation-name, animation-play-state, animation-timing-function, appearance, backface-visibility, background-clip, background-composite, background-origin, background-size, border-fit, border-horizontal-spacing, border-image, border-vertical-spacing, box-align, box-direction, box-flex, box-flex-group, box-lines, box-ordinal-group, box-orient, box-pack, box-reflect, box-shadow, color-correction, column-break-after, column-break-before, column-break-inside, column-count, column-gap, column-rule-color, column-rule-style, column-rule-width, column-span, column-width, dashboard-region, font-smoothing, highlight, hyphenate-character, hyphenate-limit-after, hyphenate-limit-before, hyphens, line-box-contain, line-break, line-clamp, locale, margin-before-collapse, margin-after-collapse, marquee-direction, marquee-increment, marquee-repetition, marquee-style, mask-attachment, mask-box-image, mask-box-image-outset, mask-box-image-repeat, mask-box-image-slice, mask-box-image-source, mask-box-image-width, mask-clip, mask-composite, mask-image, mask-origin, mask-position, mask-repeat, mask-size, nbsp-mode, perspective, perspective-origin, rtl-ordering, text-combine, text-decorations-in-effect, text-emphasis-color, text-emphasis-position, text-emphasis-style, text-fill-color, text-orientation, text-security, text-stroke-color, text-stroke-width, transform, transition, transform-origin, transform-style, transition-delay, transition-duration, transition-property, transition-timing-function, user-drag, user-modify, user-select, writing-mode, svg-shadow",
moz:"animation-delay, animation-direction, animation-duration, animation-fill-mode, animation-iteration-count, animation-name, animation-play-state, animation-timing-function, appearance, backface-visibility, background-inline-policy, binding, border-bottom-colors, border-image, border-left-colors, border-right-colors, border-top-colors, box-align, box-direction, box-flex, box-ordinal-group, box-orient, box-pack, box-shadow, box-sizing, column-count, column-gap, column-rule-color, column-rule-style, column-rule-width, column-width, float-edge, font-feature-settings, font-language-override, force-broken-image-icon, hyphens, image-region, orient, outline-radius-bottomleft, outline-radius-bottomright, outline-radius-topleft, outline-radius-topright, perspective, perspective-origin, stack-sizing, tab-size, text-blink, text-decoration-color, text-decoration-line, text-decoration-style, text-size-adjust, transform, transform-origin, transform-style, transition, transition-delay, transition-duration, transition-property, transition-timing-function, user-focus, user-input, user-modify, user-select, window-shadow",
ms:"accelerator, animation, animation-delay, animation-direction, animation-duration, animation-fill-mode, animation-iteration-count, animation-name, animation-play-state, animation-timing-function, backface-visibility, background-position-x, background-position-y, behavior, block-progression, box-align, box-direction, box-flex, box-line-progression, box-lines, box-ordinal-group, box-orient, box-pack, content-zoom-boundary, content-zoom-boundary-max, content-zoom-boundary-min, content-zoom-chaining, content-zoom-snap, content-zoom-snap-points, content-zoom-snap-type, content-zooming, filter, flow-from, flow-into, font-feature-settings, grid-column, grid-column-align, grid-column-span, grid-columns, grid-layer, grid-row, grid-row-align, grid-row-span, grid-rows, high-contrast-adjust, hyphenate-limit-chars, hyphenate-limit-lines, hyphenate-limit-zone, hyphens, ime-mode, interpolation-mode, layout-flow, layout-grid, layout-grid-char, layout-grid-line, layout-grid-mode, layout-grid-type, line-break, overflow-style, overflow-x, overflow-y, perspective, perspective-origin, perspective-origin-x, perspective-origin-y, scroll-boundary, scroll-boundary-bottom, scroll-boundary-left, scroll-boundary-right, scroll-boundary-top, scroll-chaining, scroll-rails, scroll-snap-points-x, scroll-snap-points-y, scroll-snap-type, scroll-snap-x, scroll-snap-y, scrollbar-arrow-color, scrollbar-base-color, scrollbar-darkshadow-color, scrollbar-face-color, scrollbar-highlight-color, scrollbar-shadow-color, scrollbar-track-color, text-align-last, text-autospace, text-justify, text-kashida-space, text-overflow, text-size-adjust, text-underline-position, touch-action, transform, transform-origin, transform-origin-x, transform-origin-y, transform-origin-z, transform-style, transition, transition-delay, transition-duration, transition-property, transition-timing-function, user-select, word-break, word-wrap, wrap-flow, wrap-margin, wrap-through, writing-mode, zoom",
o:"dashboard-region, animation, animation-delay, animation-direction, animation-duration, animation-fill-mode, animation-iteration-count, animation-name, animation-play-state, animation-timing-function, border-image, link, link-source, object-fit, object-position, tab-size, table-baseline, transform, transform-origin, transition, transition-delay, transition-duration, transition-property, transition-timing-function, accesskey, input-format, input-required, marquee-dir, marquee-loop, marquee-speed, marquee-style"},
function(a,b){q.define("css."+b+"Properties",a,r({vendor:b}))});q.define("css.unitlessProperties","z-index, line-height","The list of properties whose values \u200b\u200bmust not contain units.");e("w",{prefix:"webkit",supports:q.getArray("css.webkitProperties")});e("m",{prefix:"moz",supports:q.getArray("css.mozProperties")});e("s",{prefix:"ms",supports:q.getArray("css.msProperties")});e("o",{prefix:"o",supports:q.getArray("css.oProperties")});var t=q.getArray("css.unitlessProperties");h("resources").addResolver(function(a,
b){return b=="css"&&a.isElement()?h("cssResolver").expandToSnippet(a.abbreviation):null});return{addPrefix:e,supportsPrefix:c,prefixed:function(a,b){return c(a,b)?"-"+b+"-"+a:a},listPrefixes:function(){return d.map(l,function(a){return a.prefix})},getPrefix:function(a){return l[a]},removePrefix:function(a){a in l&&delete l[a]},addUnitAlias:function(a,b){n[a]=b},getUnitAlias:function(a){return n[a]},removeUnitAlias:function(a){a in n&&delete n[a]},extractPrefixes:function(a){if(a.charAt(0)!="-")return{property:a,
prefixes:null};for(var b=1,c=a.length,d,e=[];b<c;){d=a.charAt(b);if(d=="-"){b++;break}if(d in l)e.push(d);else{e.length=0;b=1;break}b++}if(b==c-1)b=1,e.length=1;return{property:a.substring(b),prefixes:e.length?e:"all"}},findValuesInAbbreviation:function(a){for(var b=0,c=a.length,d;b<c;){d=a.charAt(b);if(g(d)||d=="-"&&g(a.charAt(b+1)))return a.substring(b);b++}return""},parseValues:function(a){for(var b="",c=[],d=0,e=a.length,f,h;d<e;)if(f=a.charAt(d),f=="-"&&b)c.push(b),b="",d++;else if(b+=f,d++,
h=a.charAt(d),f!="-"&&!g(f)&&(g(h)||h=="-"))c.push(b),b="";b&&c.push(b);return c},extractValues:function(a){var b=this.findValuesInAbbreviation(a);return!b?{property:a,values:null}:{property:a.substring(0,a.length-b.length),values:this.parseValues(b)}},normalizeValue:function(a,b){b=(b||"").toLowerCase();return a.replace(/^(\-?[0-9\.]+)([a-z]*)$/,function(a,c,e){return!e&&(c=="0"||d.include(t,b))?c:!e?c+(~c.indexOf(".")?"em":"px"):c+(e in n?n[e]:e)})},expand:function(c,e){var g=h("resources"),k;if(k=
/^(.+)\!$/.test(c))c=RegExp.$1;var n=g.getSnippet("css",c);if(n)return i(n,k);var n=this.extractPrefixes(c),j=this.extractValues(n.property),j=d.extend(n,j);if(n=g.getSnippet("css",j.property)){if(!d.isString(n))n=n.data}else n=j.property+":"+o;if(!f(n))return n;var r=b(n),q=[];!e&&j.values&&(e=d.map(j.values,function(a){return this.normalizeValue(a,r.name)},this).join(" ")+";");r.value=e||r.value;j.prefixes&&(g=j.prefixes=="all"?a(r.name):j.prefixes,d.each(g,function(a){a in l&&q.push(i(l[a].transformName(r.name)+
":"+r.value,k))}));q.push(i(r.name+":"+r.value,k));return q},expandToSnippet:function(a,b){var c=this.expand(a,b);return d.isArray(c)?c.join("\n"):!d.isString(c)?c.data:String(c)}}});
zen_coding.define("cssGradient",function(h,d){function g(a){return h("utils").trim(a).replace(/\s+/g," ")}function f(a){var a=g(a),b=null,a=a.replace(/^(\w+\(.+?\))\s*/,function(a,c){b=c;return""});b||(a=a.split(" "),b=a[0],a=a[1]||"");var c={color:b};a&&a.replace(/^(\-?[\d\.]+)([a-z%]+)?$/,function(a,b,d){c.position=b;~b.indexOf(".")?d="":d||(d="%");if(d)c.unit=d});return c}function b(a){var b=0;d.each(a,function(c,e){if(!e)return c.position=c.position||0;if(e==a.length-1&&!("position"in c))c.position=
1;if("position"in c){var f=a[b].position||0,g=(c.position-f)/(e-b);d.each(a.slice(b,e),function(a,b){a.position=f+g*b});b=e}})}function c(a){var b=parseFloat(a);if(!d.isNaN(b))switch(b%360){case 0:return"left";case 90:return"bottom";case 180:return"right";case 240:return"top"}return a}function a(a){a=c(a);if(n.test(a))throw"The direction is an angle that can\u2019t be converted.";var b=function(b){return~a.indexOf(b)?"100%":"0"};return b("right")+" "+b("bottom")+", "+b("left")+" "+b("top")}function e(a){var b=
q.getArray("css.gradient.prefixes"),b=d.map(b,function(b){return"-"+b+"-"+a});b.push(a);return b}function i(a,b,c){var f=a.parent,g=h("utils"),i=h("cssResolver"),k=q.getArray("css.gradient.prefixes");d.each(f.getAll(e(a.name())),function(b){b!=a&&/gradient/i.test(b.value())&&f.remove(b)});var j=a.value();c||(c=h("range").create(0,a.value()));var l=function(a){return g.replaceSubstring(j,a,c)},n=h("cssGradient");a.value(l(n.toString(b)));var o=[];d.each(k,function(c){var d=i.prefixed(a.name(),c);if(c==
"webkit"&&q.get("css.gradient.oldWebkit"))try{o.push({name:d,value:l(n.oldWebkitLinearGradient(b))})}catch(e){}o.push({name:d,value:l(n.toString(b,c))})});o=o.sort(function(a,b){return b.name.length-a.name.length});d.each(o,function(b){f.add(b.name,b.value,f.indexOf(a))})}function k(a){var b=a.value(),c=h("cssGradient"),e=null;return(a=d.find(a.valueParts(),function(a){return e=c.parse(a.substring(b))}))&&e?{gradient:e,valueRange:a}:null}var l=["top","to bottom","0deg"],n=/\d+deg/i,o=/top|bottom|left|right/i,
q=h("preferences");q.define("css.gradient.prefixes","webkit, moz, ms, o","A comma-separated list of vendor-prefixes for which values should be generated.");q.define("css.gradient.oldWebkit",!0,"Generate gradient definition for old Webkit implementations");q.define("css.gradient.omitDefaultDirection",!0,"Do not output default direction definition in generated gradients.");h("expandAbbreviation").addHandler(function(a,b,c){b=h("editorUtils").outputInfo(a,b,c);if(b.syntax!="css")return!1;var e=a.getCaretPos();
if(b=h("cssEditTree").parseFromPosition(b.content,e,!0))if((c=b.itemFromPosition(e,!0))||(c=d.find(b.list(),function(a){return a.range(!0).end==e})),c){var f=k(c);if(f){c.end(";");var g=b.options.offset||0,l=g+b.toString().length;i(c,f.gradient,f.valueRange);a.replaceContent(b.toString(),g,l,!0);a.setCaretPos(c.valueRange(!0).end);return!0}}return!1});h("reflectCSSValue").addHandler(function(a){var b=h("cssGradient"),c=h("utils"),f=k(a);if(!f)return!1;var g=a.value(),i=function(a){return c.replaceSubstring(g,
a,f.valueRange)};d.each(a.parent.getAll(e(a.name())),function(c){if(c!==a){var d=c.value().match(/^\s*(\-([a-z]+)\-)?linear\-gradient/);d?c.value(i(b.toString(f.gradient,d[2]||""))):c.value().match(/\s*\-webkit\-gradient/)&&c.value(i(b.oldWebkitLinearGradient(f.gradient)))}});return!0});return{parse:function(a){var b=null,a=h("utils").trim(a).replace(/^([\w\-]+)\((.+?)\)$/,function(a,c,e){c=c.toLowerCase().replace(/^\-[a-z]+\-/,"");if(c=="linear-gradient"||c=="lg"){for(var a=l[0],e=h("stringStream").create(h("utils").trim(e)),
c=[],i;i=e.next();)e.peek()==","?(c.push(e.current()),e.next(),e.eatSpace(),e.start=e.pos):i=="("&&e.skipTo(")");c.push(e.current());c=d.compact(d.map(c,g));if(c.length){if(n.test(c[0])||o.test(c[0]))a=c.shift();b={type:"linear",direction:a,colorStops:d.map(c,f)}}else b=null;return""}return a});return b},oldWebkitLinearGradient:function(c){d.isString(c)&&(c=this.parse(c));if(!c)return null;var e=d.map(c.colorStops,d.clone);d.each(e,function(a){if("position"in a)if(~a.position.indexOf(".")||a.unit==
"%")a.position=parseFloat(a.position)/(a.unit=="%"?100:1);else throw"Can't convert color stop '"+(a.position+(a.unit||""))+"'";});b(e);e=d.map(e,function(a,b){return!a.position&&!b?"from("+a.color+")":a.position==1&&b==e.length-1?"to("+a.color+")":"color-stop("+a.position.toFixed(2).replace(/\.?0+$/,"")+", "+a.color+")"});return"-webkit-gradient(linear, "+a(c.direction)+", "+e.join(", ")+")"},toString:function(a,b){if(a.type=="linear"){var c=(b?"-"+b+"-":"")+"linear-gradient",e=d.map(a.colorStops,
function(a){return a.color+("position"in a?" "+a.position+(a.unit||""):"")});a.direction&&(!q.get("css.gradient.omitDefaultDirection")||!d.include(l,a.direction))&&e.unshift(a.direction);return c+"("+e.join(", ")+")"}}}});zen_coding.exec(function(h,d){var g=h("handlerList").create(),f=h("resources");d.extend(f,{addGenerator:function(b,c,a){d.isString(b)&&(b=RegExp(b));g.add(function(a,d){var f;return(f=b.exec(a.name()))?c(f,a,d):null},a)}});f.addResolver(function(b,c){return g.exec(null,d.toArray(arguments))})});
zen_coding.define("tagName",function(h,d){var g={empty:"area,base,basefont,br,col,frame,hr,img,input,isindex,link,meta,param,embed,keygen,command".split(","),blockLevel:"address,applet,blockquote,button,center,dd,del,dir,div,dl,dt,fieldset,form,frameset,hr,iframe,ins,isindex,li,link,map,menu,noframes,noscript,object,ol,p,pre,script,table,tbody,td,tfoot,th,thead,tr,ul,h1,h2,h3,h4,h5,h6".split(","),inlineLevel:"a,abbr,acronym,applet,b,basefont,bdo,big,br,button,cite,code,del,dfn,em,font,i,iframe,img,input,ins,kbd,label,map,object,q,s,samp,select,small,span,strike,strong,sub,sup,textarea,tt,u,var".split(",")},
f={ul:"li",ol:"li",table:"tr",tr:"td",tbody:"tr",thead:"tr",tfoot:"tr",colgroup:"col",select:"option",optgroup:"option",audio:"source",video:"source",object:"param",map:"area"};return{resolve:function(b){b=(b||"").toLowerCase();return b in f?this.getMapping(b):this.isInlineLevel(b)?"span":"div"},getMapping:function(b){return f[b.toLowerCase()]},isInlineLevel:function(b){return this.isTypeOf(b,"inlineLevel")},isBlockLevel:function(b){return this.isTypeOf(b,"blockLevel")},isEmptyElement:function(b){return this.isTypeOf(b,
"empty")},isTypeOf:function(b,c){return d.include(g[c],b)},addMapping:function(b,c){f[b]=c},removeMapping:function(b){b in f&&delete f[b]},addElementToCollection:function(b,c){g[c]||(g[c]=[]);var a=this.getCollection(c);d.include(a,b)||a.push(b)},removeElementFromCollection:function(b,c){c in g&&(g[c]=d.without(this.getCollection(c),b))},getCollection:function(b){return g[b]}}});
zen_coding.exec(function(h,d){function g(){return{element:e.get("bem.elementSeparator"),modifier:e.get("bem.modifierSeparator")}}function f(a){if(h("abbreviationUtils").isSnippet(a))return a;a.__bem={block:"",element:"",modifier:""};var e=b(a.attribute("class")).split(" "),f=/^[a-z]\-/i;a.__bem.block=d.find(e,function(a){return f.test(a)});if(!a.__bem.block)f=/^[a-z]/i,a.__bem.block=d.find(e,function(a){return f.test(a)})||"";e=d.chain(e).map(function(b){var d;d=c(b,a,"element");d=c(d,a,"modifier");
var e="",f="",h="",b=g();~d.indexOf(b.element)?(e=d.split(b.element),h=e[1].split(b.modifier),e=e[0],f=h.shift(),h=h.join(b.modifier)):~d.indexOf(b.modifier)&&(h=d.split(b.modifier),e=h.shift(),h=h.join(b.modifier));if(e||f||h){if(!e)e=a.__bem.block;d=e;var i=[];f&&(d+=b.element+f);i.push(d);h&&i.push(d+b.modifier+h);a.__bem.block=e;a.__bem.element=f;a.__bem.modifier=h;b=i}else b=d;return b}).flatten().uniq().value();a.attribute("class",e.join(" "));return a}function b(a){var b=h("utils"),a=(" "+
(a||"")+" ").replace(/\s+/g," "),c=e.get("bem.shortElementPrefix");c&&(c=RegExp("\\s("+b.escapeForRegexp(c)+"+)","g"),a=a.replace(c,function(a,c){return" "+b.repeatString(g().element,c.length)}));return b.trim(a)}function c(a,b,c){var d=g(),e=RegExp("^("+d[c]+")+","g");if(e.test(a)){for(var f=0,e=a.replace(e,function(a){f=a.length/d[c].length;return""}),h=b;h.parent&&f--;)h=h.parent;if(!h||!h.__bem)h=b;if(h&&h.__bem)return a=h.__bem.block,c=="modifier"&&h.__bem.element&&(a+=d.element+h.__bem.element),
a+d[c]+e}return a}function a(b,c){b.name&&f(b,c);var e=h("abbreviationUtils");d.each(b.children,function(b){a(b,c);!e.isSnippet(b)&&b.start&&(i=!0)});return b}var e=h("preferences");e.define("bem.elementSeparator","__","Class name\u2019s element separator.");e.define("bem.modifierSeparator","_","Class name\u2019s modifier separator.");e.define("bem.shortElementPrefix","-","Symbol for describing short \u201cblock-element\u201d notation. Class names prefixed with this symbol will be treated as element name for parent\u2018s block name. Each symbol instance traverses one level up in parsed tree for block name lookup. Empty value will disable short notation.");
var i=!1;h("filters").add("bem",function(b,c){i=!1;b=a(b,c);i&&(b=h("filters").apply(b,"html",c));return b})});
zen_coding.exec(function(h,d){function g(c,a,e){var f=h("utils"),g=b.get("filter.commentTrigger");if(g=="*"||d.find(g.split(","),function(a){return!!c.attribute(f.trim(a))}))g={node:c,name:c.name(),padding:c.parent?c.parent.padding:"",attr:function(a,b,d){return(a=c.attribute(a))?(b||"")+a+(d||""):""}},a=f.normalizeNewline(a?a(g):""),e=f.normalizeNewline(e?e(g):""),c.start=c.start.replace(/</,a+"<"),c.end=c.end.replace(/>/,">"+e)}function f(b,a,e){var i=h("abbreviationUtils");d.each(b.children,function(b){i.isBlock(b)&&
g(b,a,e);f(b,a,e)});return b}var b=h("preferences");b.define("filter.commentAfter",'\n<\!-- /<%= attr("id", "#") %><%= attr("class", ".") %> --\>',"A definition of comment that should be placed <i>after</i> matched element when <code>comment</code> filter is applied. This definition is an ERB-style template passed to <code>_.template()</code> function (see Underscore.js docs for details). In template context, the following properties and functions are availabe:\n<ul><li><code>attr(name, before, after)</code> \u2013 a function that outputsspecified attribute value concatenated with <code>before</code> and <code>after</code> strings. If attribute doesn't exists, the empty string will be returned.</li><li><code>node</code> \u2013 current node (instance of <code>ZenNode</code>)</li><li><code>name</code> \u2013 name of current tag</li><li><code>padding</code> \u2013 current string padding, can be used for formatting</li></ul>");
b.define("filter.commentBefore","","A definition of comment that should be placed <i>before</i> matched element when <code>comment</code> filter is applied. For more info, read description of <code>filter.commentAfter</code> property");b.define("filter.commentTrigger","id, class","A comma-separated list of attribute names that should exist in abbreviatoin where comment should be added. If you wish to add comment for every element, set this option to <code>*</code>");h("filters").add("c",function(c){var a=
d.template(b.get("filter.commentBefore")),e=d.template(b.get("filter.commentAfter"));return f(c,a,e)})});zen_coding.exec(function(h,d){function g(b){return b.replace(/([<>&])/g,function(b,a){return f[a]})}var f={"<":"&lt;",">":"&gt;","&":"&amp;"};h("filters").add("e",function c(a){d.each(a.children,function(a){a.start=g(a.start);a.end=g(a.end);a.content=g(a.content);c(a)});return a})});
zen_coding.exec(function(h,d){function g(){return h("resources").getVariable("indentation")||"\t"}function f(b){return b.parent&&!b.parent.parent&&!b.index()}function b(b,a){var e=h("abbreviationUtils");if(a.tag_nl===!0||e.isBlock(b))return!0;if(!b.parent||!a.inline_break)return!1;var f=0;return!!d.find(b.parent.children,function(b){b.isTextNode()||!e.isInline(b)?f=0:e.isInline(b)&&f++;if(f>=a.inline_break)return!0})}h("filters").add("_format",function a(e,i,k){var k=k||0,l=h("abbreviationUtils");
d.each(e.children,function(d){if(l.isSnippet(d)){if(!f(d))d.start=h("utils").getNewline()+d.start}else{d.start=d.end="%s";var e=h("utils"),q=h("abbreviationUtils"),r=q.isUnary(d),e=e.getNewline();if(i.tag_nl!==!1){var t=i.tag_nl===!0;if(!d.isTextNode()){if(b(d,i)){if(!f(d)&&(!q.isSnippet(d.parent)||d.index()))d.start=e+d.start;if(q.hasBlockChildren(d)||d.children.length&&b(d.children[0],i)||t&&!r)d.end=e+d.end;if(q.hasTagsInContent(d)||t&&!d.children.length&&!r)d.start+=e+g()}else if(q.isInline(d)&&
d.parent&&h("abbreviationUtils").hasBlockChildren(d.parent)&&!f(d))d.start=e+d.start;else if(q.isInline(d)&&q.hasBlockChildren(d))d.end=e+d.end;d.padding=g()}}}a(d,i,k+1)});return e})});
zen_coding.exec(function(h,d){function g(f,b){var c="",a=[],e=b.attributeQuote(),g=b.cursor();d.each(f.attributeList(),function(d){var f=b.attributeName(d.name);switch(f.toLowerCase()){case "id":c+="#"+(d.value||g);break;case "class":c+="."+h("utils").trim(d.value||g).replace(/\s+/g,".");break;default:a.push(":"+f+" => "+e+(d.value||g)+e)}});a.length&&(c+="{"+a.join(", ")+"}");return c}h("filters").add("haml",function b(c,a,e){var e=e||0,i=h("abbreviationUtils");e||(c=h("filters").apply(c,"_format",
a));d.each(c.children,function(c){if(!i.isSnippet(c)&&c.parent){var d=h("abbreviationUtils"),n=h("utils"),o=g(c,a),q=a.cursor(),d=d.isUnary(c),r=a.self_closing_tag&&d?"/":"",t="",t="%"+a.tagName(c.name());t.toLowerCase()=="%div"&&o&&o.indexOf("{")==-1&&(t="");c.end="";c.start=n.replaceSubstring(c.start,t+o+r+" ",c.start.indexOf("%s"),"%s");!c.children.length&&!d&&(c.start+=q)}b(c,a,e+1)});return c})});
zen_coding.exec(function(h,d){function g(f,b){var c=b.attributeQuote(),a=b.cursor();return d.map(f.attributeList(),function(d){return" "+b.attributeName(d.name)+"="+c+(d.value||a)+c}).join("")}h("filters").add("html",function b(c,a,e){var e=e||0,i=h("abbreviationUtils");e||(c=h("filters").apply(c,"_format",a));d.each(c.children,function(c){if(!i.isSnippet(c)&&c.parent){var d=h("abbreviationUtils"),n=h("utils"),o=g(c,a),q=a.cursor(),d=d.isUnary(c),r="",t="";if(!c.isTextNode()){var m=a.tagName(c.name());
d?(r="<"+m+o+a.selfClosing()+">",c.end=""):(r="<"+m+o+">",t="</"+m+">")}c.start=n.replaceSubstring(c.start,r,c.start.indexOf("%s"),"%s");c.end=n.replaceSubstring(c.end,t,c.end.indexOf("%s"),"%s");!c.children.length&&!d&&c.content.indexOf(q)==-1&&(c.start+=q)}b(c,a,e+1)});return c})});
zen_coding.exec(function(h,d){var g=/^\s+/,f=/[\n\r]/g;h("filters").add("s",function c(a){var e=h("abbreviationUtils");d.each(a.children,function(a){if(!e.isSnippet(a))a.start=a.start.replace(g,""),a.end=a.end.replace(g,"");a.start=a.start.replace(f,"");a.end=a.end.replace(f,"");a.content=a.content.replace(f,"");c(a)});return a})});
zen_coding.exec(function(h,d){function g(f,b){d.each(f.children,function(c){if(c.content)c.content=c.content.replace(b,"");g(c,b)});return f}h("preferences").define("filter.trimRegexp","[\\s|\\u00a0]*[\\d|#|\\-|*|\\u2022]+\\.?\\s*","Regular expression used to remove list markers (numbers, dashes, bullets, etc.) in <code>t</code> (trim) filter. The trim filter is useful for wrapping with abbreviation lists, pased from other documents (for example, Word documents).");h("filters").add("t",function(d){var b=
RegExp(h("preferences").get("filter.trimRegexp"));return g(d,b)})});zen_coding.exec(function(h,d){var g={"xsl:variable":1,"xsl:with-param":1};h("filters").add("xsl",function b(c){var a=h("abbreviationUtils");d.each(c.children,function(c){if(!a.isSnippet(c)&&(c.name()||"").toLowerCase()in g&&c.children.length)c.start=c.start.replace(/\s+select\s*=\s*(['"]).*?\1/,"");b(c)});return c})});
zen_coding.exec(function(h,d){function g(a,b){return Math.round(Math.random()*(b-a)+a)}function f(a,b){for(var c=a.length,e=Math.min(c,b),f=[];f.length<e;){var h=g(0,c-1);d.include(f,h)||f.push(h)}return d.map(f,function(b){return a[b]})}function b(a,b){a.length&&(a[0]=a[0].charAt(0).toUpperCase()+a[0].substring(1));return a.join(" ")+(b||(d.isString("?!...")?"?!...".charAt(g(0,4)):"?!..."[g(0,4)]))}function c(a){var b=a.length,c=0,c=b>3&&b<=6?g(0,1):b>6&&b<=12?g(0,2):g(1,4);d.each(f(d.range(c)),
function(b){a[b]+=","})}h("abbreviationParser").addPreprocessor(function(d){var h=/^(?:lorem|lipsum)(\d*)$/i,l;d.findAll(function(d){if(d._name&&(l=d._name.match(h))){var i=l[1]||30;d._name="";d.data("forceNameResolving",d.isRepeating()||d.attributeList().length);d.data("paste",function(d){var h=[],k=0;d||(d=a.slice(0,i+1),d.length>5&&(d[4]+=","),k+=d.length,h.push(b(d,".")));for(;k<i;)d=f(e,Math.min(g(3,12)*g(1,5),i-k)),k+=d.length,c(d),h.push(b(d));return h.join(" ")})}})});var a="lorem ipsum dolor sit amet consectetur adipisicing elit".split(" "),
e=["exercitationem","perferendis","perspiciatis","laborum","eveniet","sunt","iure","nam","nobis","eum","cum","officiis","excepturi","odio","consectetur","quasi","aut","quisquam","vel","eligendi","itaque","non","odit","tempore","quaerat","dignissimos","facilis","neque","nihil","expedita","vitae","vero","ipsum","nisi","animi","cumque","pariatur","velit","modi","natus","iusto","eaque","sequi","illo","sed","ex","et","voluptatibus","tempora","veritatis","ratione","assumenda","incidunt","nostrum","placeat",
"aliquid","fuga","provident","praesentium","rem","necessitatibus","suscipit","adipisci","quidem","possimus","voluptas","debitis","sint","accusantium","unde","sapiente","voluptate","qui","aspernatur","laudantium","soluta","amet","quo","aliquam","saepe","culpa","libero","ipsa","dicta","reiciendis","nesciunt","doloribus","autem","impedit","minima","maiores","repudiandae","ipsam","obcaecati","ullam","enim","totam","delectus","ducimus","quis","voluptates","dolores","molestiae","harum","dolorem","quia",
"voluptatem","molestias","magni","distinctio","omnis","illum","dolorum","voluptatum","ea","quas","quam","corporis","quae","blanditiis","atque","deserunt","laboriosam","earum","consequuntur","hic","cupiditate","quibusdam","accusamus","ut","rerum","error","minus","eius","ab","ad","nemo","fugit","officia","at","in","id","quos","reprehenderit","numquam","iste","fugiat","sit","inventore","beatae","repellendus","magnam","recusandae","quod","explicabo","doloremque","aperiam","consequatur","asperiores","commodi",
"optio","dolor","labore","temporibus","repellat","veniam","architecto","est","esse","mollitia","nulla","a","similique","eos","alias","dolore","tenetur","deleniti","porro","facere","maxime","corrupti"]});
zen_coding.exec(function(h){var d=h("actions"),g=h("preferences");g.define("css.closeBraceIndentation","\n","Indentation before closing brace of CSS rule. Some users prefereindented closing brace of CSS rule for better readability. This preference\u2019s value will be automatically inserted before closing brace when user adds newline in newly created CSS rule (e.g. when \u201cInsert formatted linebreak\u201d action will be performed in CSS file). If you\u2019re such user, you may want to write put a value like <code>\\n\\t</code> in this preference.");d.add("insert_formatted_line_break_only",
function(d){var b=h("utils"),c=h("resources"),a=h("editorUtils").outputInfo(d),e=d.getCaretPos(),i=b.getNewline();if(a.syntax=="html"){if(c=c.getVariable("indentation"),a=h("html_matcher").getTags(a.content,e,a.profile),a[0]&&a[1]&&a[0].type=="tag"&&a[0].end==e&&a[1].start==e)return d.replaceContent(i+c+b.getCaretPlaceholder()+i,e),!0}else if(a.syntax=="css"&&(a=a.content,e&&a.charAt(e-1)=="{")){var k=g.get("css.closeBraceIndentation"),c=c.getVariable("indentation"),l=a.charAt(e)=="}";if(!l)for(var n=
e,o=a.length,q;n<o;n++){q=a.charAt(n);if(q=="{")break;if(q=="}"){k="";l=!0;break}}l||(k+="}");b=i+c+b.getCaretPlaceholder()+k;d.replaceContent(b,e);return!0}return!1},{hidden:!0});d.add("insert_formatted_line_break",function(f){if(!d.run("insert_formatted_line_break_only",f)){for(var b=h("utils"),c=h("editorUtils").getCurrentLinePadding(f),a=String(f.getContent()),e=f.getCaretPos(),g=a.length,b=b.getNewline(),k="",l=f.getCurrentLineRange().end+1,n;l<g;l++)if(n=a.charAt(l),n==" "||n=="\t")k+=n;else break;
k.length>c.length?f.replaceContent(b+k,e,e,!0):f.replaceContent(b,e)}return!0},{hidden:!0})});zen_coding.require("actions").add("select_line",function(h){var d=h.getCurrentLineRange();h.createSelection(d.start,d.end);return!0});
zen_coding.require("resources").setVocabulary({variables:{lang:"ja",locale:"ja-JP",charset:"Shift_JIS",indentation:"\t",newline:"\n"},css:{filters:"html",snippets:{"@i":"@import url(|);","@m":"@media print {\n\t|\n}","@f":"@font-face {\n\tfont-family:|;\n\tsrc:url(|);\n}","!":"!important",pos:"position:|;","pos:s":"position:static;","pos:a":"position:absolute;","pos:r":"position:relative;","pos:f":"position:fixed;",t:"top:|;","t:a":"top:auto;",r:"right:|;","r:a":"right:auto;",b:"bottom:|;","b:a":"bottom:auto;",
brad:"-webkit-border-radius: ${1:radius};\n-moz-border-radius: $1;\nborder-radius: $1;",bsha:"-webkit-box-shadow: ${1:hoff} ${2:voff} ${3:blur} ${4:rgba(0,0,0,0.5)};\n-moz-box-shadow: $1 $2 $3 $4;\nbox-shadow: $1 $2 $3 $4;",l:"left:|;","l:a":"left:auto;",z:"z-index:|;","z:a":"z-index:auto;",fl:"float:|;","fl:n":"float:none;","fl:l":"float:left;","fl:r":"float:right;",cl:"clear:|;","cl:n":"clear:none;","cl:l":"clear:left;","cl:r":"clear:right;","cl:b":"clear:both;",d:"display:|;","d:n":"display:none;",
"d:b":"display:block;","d:i":"display:inline;","d:ib":"display:inline-block;","d:li":"display:list-item;","d:ri":"display:run-in;","d:cp":"display:compact;","d:tb":"display:table;","d:itb":"display:inline-table;","d:tbcp":"display:table-caption;","d:tbcl":"display:table-column;","d:tbclg":"display:table-column-group;","d:tbhg":"display:table-header-group;","d:tbfg":"display:table-footer-group;","d:tbr":"display:table-row;","d:tbrg":"display:table-row-group;","d:tbc":"display:table-cell;","d:rb":"display:ruby;",
"d:rbb":"display:ruby-base;","d:rbbg":"display:ruby-base-group;","d:rbt":"display:ruby-text;","d:rbtg":"display:ruby-text-group;",v:"visibility:|;","v:v":"visibility:visible;","v:h":"visibility:hidden;","v:c":"visibility:collapse;",ov:"overflow:|;","ov:v":"overflow:visible;","ov:h":"overflow:hidden;","ov:s":"overflow:scroll;","ov:a":"overflow:auto;",ovx:"overflow-x:|;","ovx:v":"overflow-x:visible;","ovx:h":"overflow-x:hidden;","ovx:s":"overflow-x:scroll;","ovx:a":"overflow-x:auto;",ovy:"overflow-y:|;",
"ovy:v":"overflow-y:visible;","ovy:h":"overflow-y:hidden;","ovy:s":"overflow-y:scroll;","ovy:a":"overflow-y:auto;",ovs:"overflow-style:|;","ovs:a":"overflow-style:auto;","ovs:s":"overflow-style:scrollbar;","ovs:p":"overflow-style:panner;","ovs:m":"overflow-style:move;","ovs:mq":"overflow-style:marquee;",zoo:"zoom:1;",cp:"clip:|;","cp:a":"clip:auto;","cp:r":"clip:rect(|);",bxz:"box-sizing:|;","bxz:cb":"box-sizing:content-box;","bxz:bb":"box-sizing:border-box;",bxsh:"box-shadow:|;","bxsh:n":"box-shadow:none;",
"bxsh:w":"-webkit-box-shadow:0 0 0 #000;","bxsh:m":"-moz-box-shadow:0 0 0 0 #000;",m:"margin:|;","m:a":"margin:auto;","m:0":"margin:0;","m:2":"margin:0 0;","m:3":"margin:0 0 0;","m:4":"margin:0 0 0 0;",mt:"margin-top:|;","mt:a":"margin-top:auto;",mr:"margin-right:|;","mr:a":"margin-right:auto;",mb:"margin-bottom:|;","mb:a":"margin-bottom:auto;",ml:"margin-left:|;","ml:a":"margin-left:auto;",p:"padding:|;","p:0":"padding:0;","p:2":"padding:0 0;","p:3":"padding:0 0 0;","p:4":"padding:0 0 0 0;",pt:"padding-top:|;",
pr:"padding-right:|;",pb:"padding-bottom:|;",pl:"padding-left:|;",w:"width:|;","w:a":"width:auto;",h:"height:|;","h:a":"height:auto;",maw:"max-width:|;","maw:n":"max-width:none;",mah:"max-height:|;","mah:n":"max-height:none;",miw:"min-width:|;",mih:"min-height:|;",o:"outline:|;","o:n":"outline:none;",oo:"outline-offset:|;",ow:"outline-width:|;",os:"outline-style:|;",oc:"outline-color:${0:#000};","oc:i":"outline-color:invert;",bd:"border:|;","bd+":"border:1px solid #000;","bd:n":"border:none;",bdbk:"border-break:|;",
"bdbk:c":"border-break:close;",bdcl:"border-collapse:|;","bdcl:c":"border-collapse:collapse;","bdcl:s":"border-collapse:separate;",bdc:"border-color:${0:#000};",bdi:"border-image:url(|);","bdi:n":"border-image:none;","bdi:w":"-webkit-border-image:url(|) 0 0 0 0 stretch stretch;","bdi:m":"-moz-border-image:url(|) 0 0 0 0 stretch stretch;",bdti:"border-top-image:url(|);","bdti:n":"border-top-image:none;",bdri:"border-right-image:url(|);","bdri:n":"border-right-image:none;",bdbi:"border-bottom-image:url(|);",
"bdbi:n":"border-bottom-image:none;",bdli:"border-left-image:url(|);","bdli:n":"border-left-image:none;",bdci:"border-corner-image:url(|);","bdci:n":"border-corner-image:none;","bdci:c":"border-corner-image:continue;",bdtli:"border-top-left-image:url(|);","bdtli:n":"border-top-left-image:none;","bdtli:c":"border-top-left-image:continue;",bdtri:"border-top-right-image:url(|);","bdtri:n":"border-top-right-image:none;","bdtri:c":"border-top-right-image:continue;",bdbri:"border-bottom-right-image:url(|);",
"bdbri:n":"border-bottom-right-image:none;","bdbri:c":"border-bottom-right-image:continue;",bdbli:"border-bottom-left-image:url(|);","bdbli:n":"border-bottom-left-image:none;","bdbli:c":"border-bottom-left-image:continue;",bdf:"border-fit:|;","bdf:c":"border-fit:clip;","bdf:r":"border-fit:repeat;","bdf:sc":"border-fit:scale;","bdf:st":"border-fit:stretch;","bdf:ow":"border-fit:overwrite;","bdf:of":"border-fit:overflow;","bdf:sp":"border-fit:space;",bdl:"border-length:|;","bdl:a":"border-length:auto;",
bdsp:"border-spacing:|;",bds:"border-style:|;","bds:n":"border-style:none;","bds:h":"border-style:hidden;","bds:dt":"border-style:dotted;","bds:ds":"border-style:dashed;","bds:s":"border-style:solid;","bds:db":"border-style:double;","bds:dtds":"border-style:dot-dash;","bds:dtdtds":"border-style:dot-dot-dash;","bds:w":"border-style:wave;","bds:g":"border-style:groove;","bds:r":"border-style:ridge;","bds:i":"border-style:inset;","bds:o":"border-style:outset;",bdw:"border-width:|;",bdt:"border-top:|;",
"bdt+":"border-top:1px solid #000;","bdt:n":"border-top:none;",bdtw:"border-top-width:|;",bdts:"border-top-style:|;","bdts:n":"border-top-style:none;",bdtc:"border-top-color:#000;",bdr:"border-right:|;","bdr+":"border-right:1px solid #000;","bdr:n":"border-right:none;",bdrw:"border-right-width:|;",bdrs:"border-right-style:|;","bdrs:n":"border-right-style:none;",bdrc:"border-right-color:#000;",bdb:"border-bottom:|;","bdb+":"border-bottom:1px solid #000;","bdb:n":"border-bottom:none;",bdbw:"border-bottom-width:|;",
bdbs:"border-bottom-style:|;","bdbs:n":"border-bottom-style:none;",bdbc:"border-bottom-color:#000;",bdl:"border-left:|;","bdl+":"border-left:1px solid #000;","bdl:n":"border-left:none;",bdlw:"border-left-width:|;",bdls:"border-left-style:|;","bdls:n":"border-left-style:none;",bdlc:"border-left-color:#000;",bdrs:"border-radius:|;",bdtrrs:"border-top-right-radius:|;",bdtlrs:"border-top-left-radius:|;",bdbrrs:"border-bottom-right-radius:|;",bdblrs:"border-bottom-left-radius:|;",bg:"background:|;","bg+":"background:#FFF url(|) 0 0 no-repeat;",
"bg:n":"background:none;","bg:ie":"filter:progid:DXImageTransform.Microsoft.AlphaImageLoader(src='${1:x}.png',sizingMethod='${2:crop}');",bgc:"background-color:${0:#fff};",bgi:"background-image:url(|);","bgi:n":"background-image:none;",bgr:"background-repeat:|;","bgr:n":"background-repeat:no-repeat;","bgr:x":"background-repeat:repeat-x;","bgr:y":"background-repeat:repeat-y;",bga:"background-attachment:|;","bga:f":"background-attachment:fixed;","bga:s":"background-attachment:scroll;",bgp:"background-position:0 0;",
bgpx:"background-position-x:|;",bgpy:"background-position-y:|;",bgbk:"background-break:|;","bgbk:bb":"background-break:bounding-box;","bgbk:eb":"background-break:each-box;","bgbk:c":"background-break:continuous;",bgcp:"background-clip:|;","bgcp:bb":"background-clip:border-box;","bgcp:pb":"background-clip:padding-box;","bgcp:cb":"background-clip:content-box;","bgcp:nc":"background-clip:no-clip;",bgo:"background-origin:|;","bgo:pb":"background-origin:padding-box;","bgo:bb":"background-origin:border-box;",
"bgo:cb":"background-origin:content-box;",bgz:"background-size:|;","bgz:a":"background-size:auto;","bgz:ct":"background-size:contain;","bgz:cv":"background-size:cover;",c:"color:${0:#000};",tbl:"table-layout:|;","tbl:a":"table-layout:auto;","tbl:f":"table-layout:fixed;",cps:"caption-side:|;","cps:t":"caption-side:top;","cps:b":"caption-side:bottom;",ec:"empty-cells:|;","ec:s":"empty-cells:show;","ec:h":"empty-cells:hide;",lis:"list-style:|;","lis:n":"list-style:none;",lisp:"list-style-position:|;",
"lisp:i":"list-style-position:inside;","lisp:o":"list-style-position:outside;",list:"list-style-type:|;","list:n":"list-style-type:none;","list:d":"list-style-type:disc;","list:c":"list-style-type:circle;","list:s":"list-style-type:square;","list:dc":"list-style-type:decimal;","list:dclz":"list-style-type:decimal-leading-zero;","list:lr":"list-style-type:lower-roman;","list:ur":"list-style-type:upper-roman;",lisi:"list-style-image:|;","lisi:n":"list-style-image:none;",q:"quotes:|;","q:n":"quotes:none;",
"q:ru":"quotes:'\\00AB' '\\00BB' '\\201E' '\\201C';","q:en":"quotes:'\\201C' '\\201D' '\\2018' '\\2019';",ct:"content:|;","ct:n":"content:normal;","ct:oq":"content:open-quote;","ct:noq":"content:no-open-quote;","ct:cq":"content:close-quote;","ct:ncq":"content:no-close-quote;","ct:a":"content:attr(|);","ct:c":"content:counter(|);","ct:cs":"content:counters(|);",coi:"counter-increment:|;",cor:"counter-reset:|;",va:"vertical-align:|;","va:sup":"vertical-align:super;","va:t":"vertical-align:top;","va:tt":"vertical-align:text-top;",
"va:m":"vertical-align:middle;","va:bl":"vertical-align:baseline;","va:b":"vertical-align:bottom;","va:tb":"vertical-align:text-bottom;","va:sub":"vertical-align:sub;",ta:"text-align:|;","ta:l":"text-align:left;","ta:c":"text-align:center;","ta:r":"text-align:right;",tal:"text-align-last:|;","tal:a":"text-align-last:auto;","tal:l":"text-align-last:left;","tal:c":"text-align-last:center;","tal:r":"text-align-last:right;",td:"text-decoration:|;","td:n":"text-decoration:none;","td:u":"text-decoration:underline;",
"td:o":"text-decoration:overline;","td:l":"text-decoration:line-through;",te:"text-emphasis:|;","te:n":"text-emphasis:none;","te:ac":"text-emphasis:accent;","te:dt":"text-emphasis:dot;","te:c":"text-emphasis:circle;","te:ds":"text-emphasis:disc;","te:b":"text-emphasis:before;","te:a":"text-emphasis:after;",th:"text-height:|;","th:a":"text-height:auto;","th:f":"text-height:font-size;","th:t":"text-height:text-size;","th:m":"text-height:max-size;",ti:"text-indent:|;","ti:-":"text-indent:-9999px;",tj:"text-justify:|;",
"tj:a":"text-justify:auto;","tj:iw":"text-justify:inter-word;","tj:ii":"text-justify:inter-ideograph;","tj:ic":"text-justify:inter-cluster;","tj:d":"text-justify:distribute;","tj:k":"text-justify:kashida;","tj:t":"text-justify:tibetan;",to:"text-outline:|;","to+":"text-outline:0 0 #000;","to:n":"text-outline:none;",tr:"text-replace:|;","tr:n":"text-replace:none;",tt:"text-transform:|;","tt:n":"text-transform:none;","tt:c":"text-transform:capitalize;","tt:u":"text-transform:uppercase;","tt:l":"text-transform:lowercase;",
tw:"text-wrap:|;","tw:n":"text-wrap:normal;","tw:no":"text-wrap:none;","tw:u":"text-wrap:unrestricted;","tw:s":"text-wrap:suppress;",tsh:"text-shadow:|;","tsh+":"text-shadow:0 0 0 #000;","tsh:n":"text-shadow:none;",lh:"line-height:|;",whs:"white-space:|;","whs:n":"white-space:normal;","whs:p":"white-space:pre;","whs:nw":"white-space:nowrap;","whs:pw":"white-space:pre-wrap;","whs:pl":"white-space:pre-line;",whsc:"white-space-collapse:|;","whsc:n":"white-space-collapse:normal;","whsc:k":"white-space-collapse:keep-all;",
"whsc:l":"white-space-collapse:loose;","whsc:bs":"white-space-collapse:break-strict;","whsc:ba":"white-space-collapse:break-all;",wob:"word-break:|;","wob:n":"word-break:normal;","wob:k":"word-break:keep-all;","wob:l":"word-break:loose;","wob:bs":"word-break:break-strict;","wob:ba":"word-break:break-all;",wos:"word-spacing:|;",wow:"word-wrap:|;","wow:nm":"word-wrap:normal;","wow:n":"word-wrap:none;","wow:u":"word-wrap:unrestricted;","wow:s":"word-wrap:suppress;",lts:"letter-spacing:|;",f:"font:|;",
"f+":"font:1em Arial,sans-serif;",fw:"font-weight:|;","fw:n":"font-weight:normal;","fw:b":"font-weight:bold;","fw:br":"font-weight:bolder;","fw:lr":"font-weight:lighter;",fs:"font-style:|;","fs:n":"font-style:normal;","fs:i":"font-style:italic;","fs:o":"font-style:oblique;",fv:"font-variant:|;","fv:n":"font-variant:normal;","fv:sc":"font-variant:small-caps;",fz:"font-size:|;",fza:"font-size-adjust:|;","fza:n":"font-size-adjust:none;",ff:"font-family:|;","ff:s":"font-family:serif;","ff:ss":"font-family:sans-serif;",
"ff:c":"font-family:cursive;","ff:f":"font-family:fantasy;","ff:m":"font-family:monospace;",fef:"font-effect:|;","fef:n":"font-effect:none;","fef:eg":"font-effect:engrave;","fef:eb":"font-effect:emboss;","fef:o":"font-effect:outline;",fem:"font-emphasize:|;",femp:"font-emphasize-position:|;","femp:b":"font-emphasize-position:before;","femp:a":"font-emphasize-position:after;",fems:"font-emphasize-style:|;","fems:n":"font-emphasize-style:none;","fems:ac":"font-emphasize-style:accent;","fems:dt":"font-emphasize-style:dot;",
"fems:c":"font-emphasize-style:circle;","fems:ds":"font-emphasize-style:disc;",fsm:"font-smooth:|;","fsm:a":"font-smooth:auto;","fsm:n":"font-smooth:never;","fsm:aw":"font-smooth:always;",fst:"font-stretch:|;","fst:n":"font-stretch:normal;","fst:uc":"font-stretch:ultra-condensed;","fst:ec":"font-stretch:extra-condensed;","fst:c":"font-stretch:condensed;","fst:sc":"font-stretch:semi-condensed;","fst:se":"font-stretch:semi-expanded;","fst:e":"font-stretch:expanded;","fst:ee":"font-stretch:extra-expanded;",
"fst:ue":"font-stretch:ultra-expanded;",op:"opacity:|;","op:ie":"filter:progid:DXImageTransform.Microsoft.Alpha(Opacity=100);","op:ms":"-ms-filter:'progid:DXImageTransform.Microsoft.Alpha(Opacity=100)';",rz:"resize:|;","rz:n":"resize:none;","rz:b":"resize:both;","rz:h":"resize:horizontal;","rz:v":"resize:vertical;",cur:"cursor:|;","cur:a":"cursor:auto;","cur:d":"cursor:default;","cur:c":"cursor:crosshair;","cur:ha":"cursor:hand;","cur:he":"cursor:help;","cur:m":"cursor:move;","cur:p":"cursor:pointer;",
"cur:t":"cursor:text;",pgbb:"page-break-before:|;","pgbb:au":"page-break-before:auto;","pgbb:al":"page-break-before:always;","pgbb:l":"page-break-before:left;","pgbb:r":"page-break-before:right;",pgbi:"page-break-inside:|;","pgbi:au":"page-break-inside:auto;","pgbi:av":"page-break-inside:avoid;",pgba:"page-break-after:|;","pgba:au":"page-break-after:auto;","pgba:al":"page-break-after:always;","pgba:l":"page-break-after:left;","pgba:r":"page-break-after:right;",orp:"orphans:|;",wid:"widows:|;"}},html:{filters:"html",
snippets:{"cc:ie6":"<\!--[if lte IE 6]>\n\t${child}|\n<![endif]--\>","cc:ie":"<\!--[if IE]>\n\t${child}|\n<![endif]--\>","cc:noie":"<\!--[if !IE]><\!--\>\n\t${child}|\n<\!--<![endif]--\>","html:4t":'<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">\n<html lang="${lang}">\n<head>\n\t<meta http-equiv="Content-Type" content="text/html;charset=${charset}">\n\t<title></title>\n</head>\n<body>\n\t${child}|\n</body>\n</html>',"html:4s":'<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">\n<html lang="${lang}">\n<head>\n\t<meta http-equiv="Content-Type" content="text/html;charset=${charset}">\n\t<title></title>\n</head>\n<body>\n\t${child}|\n</body>\n</html>',
"html:xt":'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">\n<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="${lang}">\n<head>\n\t<meta http-equiv="Content-Type" content="text/html;charset=${charset}" />\n\t<title></title>\n</head>\n<body>\n\t${child}|\n</body>\n</html>',"html:xs":'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">\n<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="${lang}">\n<head>\n\t<meta http-equiv="Content-Type" content="text/html;charset=${charset}" />\n\t<title></title>\n</head>\n<body>\n\t${child}|\n</body>\n</html>',
"html:xxs":'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="${lang}">\n<head>\n\t<meta http-equiv="Content-Type" content="text/html;charset=${charset}" />\n\t<title></title>\n</head>\n<body>\n\t${child}|\n</body>\n</html>',"html:5":'<!doctype html>\n<html lang="${lang}">\n<head>\n\t<meta charset="${charset}">\n\t<title></title>\n</head>\n<body>\n\t${child}|\n</body>\n</html>'},abbreviations:{a:'<a href="">',
"a:link":'<a href="http://|">',"a:mail":'<a href="mailto:|">',abbr:'<abbr title="">',acronym:'<acronym title="">',base:'<base href="" />',bdo:'<bdo dir="">',"bdo:r":'<bdo dir="rtl">',"bdo:l":'<bdo dir="ltr">',link:'<link rel="stylesheet" href="" />',"link:css":'<link rel="stylesheet" href="${0:style}.css" media="all" />',"link:print":'<link rel="stylesheet" href="${0:print}.css" media="print" />',"link:favicon":'<link rel="shortcut icon" type="image/x-icon" href="${0:favicon.ico}" />',"link:touch":'<link rel="apple-touch-icon" href="${0:favicon.png}" />',
"link:rss":'<link rel="alternate" type="application/rss+xml" title="RSS" href="${0:rss.xml}" />',"link:atom":'<link rel="alternate" type="application/atom+xml" title="Atom" href="${0:atom.xml}" />',"meta:utf":'<meta http-equiv="Content-Type" content="text/html;charset=UTF-8" />',"meta:win":'<meta http-equiv="Content-Type" content="text/html;charset=windows-1251" />',"meta:compat":'<meta http-equiv="X-UA-Compatible" content="${0:IE=7}" />',style:"<style>",script:"<script>","script:src":'<script src="">',
img:'<img src="" alt="" />',iframe:'<iframe src="" frameborder="0">',embed:'<embed src="" type="" />',object:'<object data="" type="">',param:'<param name="" value="" />',map:'<map name="">',area:'<area shape="" coords="" href="" alt="" />',"area:d":'<area shape="default" href="" alt="" />',"area:c":'<area shape="circle" coords="" href="" alt="" />',"area:r":'<area shape="rect" coords="" href="" alt="" />',"area:p":'<area shape="poly" coords="" href="" alt="" />',form:'<form action="">',"form:get":'<form action="" method="get">',
"form:post":'<form action="" method="post">',label:'<label for="">',input:'<input type="" />',"input:hidden":'<input type="hidden" name="" />',"input:h":'<input type="hidden" name="" />',"input:text":'<input type="text" name="" id="" />',"input:t":'<input type="text" name="" id="" />',"input:search":'<input type="search" name="" id="" />',"input:email":'<input type="email" name="" id="" />',"input:url":'<input type="url" name="" id="" />',"input:password":'<input type="password" name="" id="" />',
"input:p":'<input type="password" name="" id="" />',"input:datetime":'<input type="datetime" name="" id="" />',"input:date":'<input type="date" name="" id="" />',"input:datetime-local":'<input type="datetime-local" name="" id="" />',"input:month":'<input type="month" name="" id="" />',"input:week":'<input type="week" name="" id="" />',"input:time":'<input type="time" name="" id="" />',"input:number":'<input type="number" name="" id="" />',"input:color":'<input type="color" name="" id="" />',"input:checkbox":'<input type="checkbox" name="" id="" />',
"input:c":'<input type="checkbox" name="" id="" />',"input:radio":'<input type="radio" name="" id="" />',"input:r":'<input type="radio" name="" id="" />',"input:range":'<input type="range" name="" id="" />',"input:file":'<input type="file" name="" id="" />',"input:f":'<input type="file" name="" id="" />',"input:submit":'<input type="submit" value="" />',"input:s":'<input type="submit" value="" />',"input:image":'<input type="image" src="" alt="" />',"input:i":'<input type="image" src="" alt="" />',
"input:reset":'<input type="reset" value="" />',"input:button":'<input type="button" value="" />',"input:b":'<input type="button" value="" />',select:'<select name="" id=""></select>',option:'<option value=""></option>',textarea:'<textarea name="" id="" cols="${0:30}" rows="${1:10}">',"menu:context":'<menu type="context">',"menu:c":'<menu type="context">',"menu:toolbar":'<menu type="toolbar">',"menu:t":'<menu type="toolbar">',video:'<video src="">',audio:'<audio src="">',"html:xml":'<html xmlns="http://www.w3.org/1999/xhtml">',
bq:"<blockquote>",acr:"<acronym>",fig:"<figure>",figc:"<figcaption>",ifr:"<iframe>",emb:"<embed>",obj:"<object>",src:"<source>",cap:"<caption>",colg:"<colgroup>",fst:"<fieldset>",btn:"<button>",optg:"<optgroup>",opt:"<option>",tarea:"<textarea>",leg:"<legend>",sect:"<section>",art:"<article>",hdr:"<header>",ftr:"<footer>",adr:"<address>",dlg:"<dialog>",str:"<strong>",prog:"<progress>",fset:"<fieldset>",datag:"<datagrid>",datal:"<datalist>",kg:"<keygen>",out:"<output>",det:"<details>",cmd:"<command>",
"ol+":"ol>li","ul+":"ul>li","dl+":"dl>dt+dd","map+":"map>area","table+":"table>tr>td","colgroup+":"colgroup>col","colg+":"colgroup>col","tr+":"tr>td","select+":"select>option","optgroup+":"optgroup>option","optg+":"optgroup>option"}},xml:{"extends":"html",profile:"xml",filters:"html"},xsl:{"extends":"html",filters:"html, xsl",abbreviations:{tm:'<xsl:template match="" mode="">',tmatch:"tm",tn:'<xsl:template name="">',tname:"tn",call:'<xsl:call-template name=""/>',ap:'<xsl:apply-templates select="" mode=""/>',
api:"<xsl:apply-imports/>",imp:'<xsl:import href=""/>',inc:'<xsl:include href=""/>',ch:"<xsl:choose>","xsl:when":'<xsl:when test="">',wh:"xsl:when",ot:"<xsl:otherwise>","if":'<xsl:if test="">',par:'<xsl:param name="">',pare:'<xsl:param name="" select=""/>',"var":'<xsl:variable name="">',vare:'<xsl:variable name="" select=""/>',wp:'<xsl:with-param name="" select=""/>',key:'<xsl:key name="" match="" use=""/>',elem:'<xsl:element name="">',attr:'<xsl:attribute name="">',attrs:'<xsl:attribute-set name="">',
cp:'<xsl:copy select=""/>',co:'<xsl:copy-of select=""/>',val:'<xsl:value-of select=""/>',each:'<xsl:for-each select="">',"for":"each",tex:"<xsl:text></xsl:text>",com:"<xsl:comment>",msg:'<xsl:message terminate="no">',fall:"<xsl:fallback>",num:'<xsl:number value=""/>',nam:'<namespace-alias stylesheet-prefix="" result-prefix=""/>',pres:'<xsl:preserve-space elements=""/>',strip:'<xsl:strip-space elements=""/>',proc:'<xsl:processing-instruction name="">',sort:'<xsl:sort select="" order=""/>',"choose+":"xsl:choose>xsl:when+xsl:otherwise"}},
haml:{filters:"haml","extends":"html"},scss:{filters:"haml","extends":"css"},sass:{filters:"haml","extends":"css"},less:{filters:"haml","extends":"css"}},"system");
zen_coding.exec(function(h,d){var g={"text/html":"html","application/xml":"xml","text/xsl":"xsl","text/css":"css","text/x-less":"less"},f={context:null,getSelectionRange:function(){var b=this.getCaretPos();return{start:b,end:b+this.getSelection().length}},createSelection:function(b,c){b==c?this.context.setCursor(this.context.posFromIndex(b)):this.context.setSelection(this.context.posFromIndex(b),this.context.posFromIndex(c))},getCurrentLineRange:function(){var b=this.context.getCursor(!0);return{start:this.context.indexFromPos({line:b.line,
ch:0}),end:this.context.indexFromPos({line:b.line,ch:this.context.getLine(b.line).length})}},getCaretPos:function(){return this.context.indexFromPos(this.context.getCursor(!0))},setCaretPos:function(b){this.createSelection(b,b)},getCurrentLine:function(){return this.context.getLine(this.context.getCursor(!0).line)||""},replaceContent:function(b,c,a,e){d.isUndefined(a)&&(a=d.isUndefined(c)?content.length:c);d.isUndefined(c)&&(c=0);var f=h("utils");e||(b=f.padString(b,f.getLinePaddingFromPosition(this.getContent(),
c)));var e=h("tabStops").extract(b,{escape:function(a){return a}}),b=e.text,g=e.tabstops[0];g?(g.start+=c,g.end+=c):g={start:b.length+c,end:b.length+c};var l=this;this.context.compoundChange(function(){l.context.replaceRange(b,l.context.posFromIndex(c),l.context.posFromIndex(a));l.createSelection(g.start,g.end)})},getContent:function(){return this.context.getValue()},getSyntax:function(){var b=this.context.getOption("mode");b in g&&(b=g[b]);var c=this.getCaretPos();h("resources").hasSyntax(b)||(b=
"html");if(b=="html"){var a=h("html_matcher").getTags(this.getContent(),c);a&&a[0]&&a[0].type=="tag"&&a[0].name.toLowerCase()=="style"&&a[0].end<=c&&a[1].start>=c&&(b="css")}return b},getProfileName:function(){if(this.context.getOption("profile"))return this.context.getOption("profile");switch(this.getSyntax()){case "xml":case "xsl":return"xml";case "html":var b=h("resources").getVariable("profile");b||(b=this.getContent().search(/<!DOCTYPE[^>]+XHTML/i)!=-1?"xhtml":"html");return b}return"xhtml"},
prompt:function(b){return prompt(b)},getSelection:function(){return this.context.getSelection()||""},getFilePath:function(){return location.href}};if(!CodeMirror.defaults.extraKeys)CodeMirror.defaults.extraKeys={};d.each({"Cmd-E":"expand_abbreviation",Tab:"expand_abbreviation","Cmd-D":"match_pair_outward","Shift-Cmd-D":"match_pair_inward","Shift-Cmd-A":"wrap_with_abbreviation","Ctrl-Alt-Right":"next_edit_point","Ctrl-Alt-Left":"prev_edit_point","Cmd-L":"select_line","Cmd-Shift-M":"merge_lines","Cmd-/":"toggle_comment",
"Cmd-J":"split_join_tag","Cmd-K":"remove_tag","Shift-Cmd-Y":"evaluate_math_expression","Ctrl-Up":"increment_number_by_1","Ctrl-Down":"decrement_number_by_1","Alt-Up":"increment_number_by_01","Alt-Down":"decrement_number_by_01","Ctrl-Alt-Up":"increment_number_by_10","Ctrl-Alt-Down":"decrement_number_by_10","Cmd-.":"select_next_item","Cmd-,":"select_previous_item","Cmd-B":"reflect_css_value",Enter:"insert_formatted_line_break"},function(b,c){CodeMirror.defaults.extraKeys[c]=function(a){f.context=a;
var c="\t";a.getOption("indentWithTabs")||(c=h("utils").repeatString(" ",a.getOption("indentUnit")));h("resources").setVariable("indentation",c);try{if(h("actions").run(b,f))return!0}catch(d){}throw CodeMirror.Pass;}})});CodeMirror.defineMode("ecl", function(config) {

  function words(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }

  function metaHook(stream, state) {
    if (!state.startOfLine) return false;
    stream.skipToEnd();
    return "meta";
  }

  function tokenAtString(stream, state) {
    var next;
    while ((next = stream.next()) != null) {
      if (next == '"' && !stream.eat('"')) {
        state.tokenize = null;
        break;
      }
    }
    return "string";
  }

  var indentUnit = config.indentUnit;
  var keyword = words("abs acos allnodes ascii asin asstring atan atan2 ave case choose choosen choosesets clustersize combine correlation cos cosh count covariance cron dataset dedup define denormalize distribute distributed distribution ebcdic enth error evaluate event eventextra eventname exists exp failcode failmessage fetch fromunicode getisvalid global graph group hash hash32 hash64 hashcrc hashmd5 having if index intformat isvalid iterate join keyunicode length library limit ln local log loop map matched matchlength matchposition matchtext matchunicode max merge mergejoin min nolocal nonempty normalize parse pipe power preload process project pull random range rank ranked realformat recordof regexfind regexreplace regroup rejected rollup round roundup row rowdiff sample set sin sinh sizeof soapcall sort sorted sqrt stepped stored sum table tan tanh thisnode topn tounicode transfer trim truncate typeof ungroup unicodeorder variance which workunit xmldecode xmlencode xmltext xmlunicode");
  var variable = words("apply assert build buildindex evaluate fail keydiff keypatch loadxml nothor notify output parallel sequential soapcall wait");
  var variable_2 = words("__compressed__ all and any as atmost before beginc++ best between case const counter csv descend encrypt end endc++ endmacro except exclusive expire export extend false few first flat from full function group header heading hole ifblock import in interface joined keep keyed last left limit load local locale lookup macro many maxcount maxlength min skew module named nocase noroot noscan nosort not of only opt or outer overwrite packed partition penalty physicallength pipe quote record relationship repeat return right scan self separator service shared skew skip sql store terminator thor threshold token transform trim true type unicodeorder unsorted validate virtual whole wild within xml xpath");
  var variable_3 = words("ascii big_endian boolean data decimal ebcdic integer pattern qstring real record rule set of string token udecimal unicode unsigned varstring varunicode");
  var builtin = words("checkpoint deprecated failcode failmessage failure global independent onwarning persist priority recovery stored success wait when");
  var blockKeywords = words("catch class do else finally for if switch try while");
  var atoms = words("true false null");
  var hooks = {"#": metaHook};
  var multiLineStrings;
  var isOperatorChar = /[+\-*&%=<>!?|\/]/;

  var curPunc;

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (hooks[ch]) {
      var result = hooks[ch](stream, state);
      if (result !== false) return result;
    }
    if (ch == '"' || ch == "'") {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    if (/[\[\]{}\(\),;\:\.]/.test(ch)) {
      curPunc = ch;
      return null
    }
    if (/\d/.test(ch)) {
      stream.eatWhile(/[\w\.]/);
      return "number";
    }
    if (ch == "/") {
      if (stream.eat("*")) {
        state.tokenize = tokenComment;
        return tokenComment(stream, state);
      }
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
    }
    if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return "operator";
    }
    stream.eatWhile(/[\w\$_]/);
    var cur = stream.current().toLowerCase();
    if (keyword.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "keyword";
    } else if (variable.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "variable";
    } else if (variable_2.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "variable-2";
    } else if (variable_3.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "variable-3";
    } else if (builtin.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "builtin";
    } else { //Data types are of from KEYWORD## 
		var i = cur.length - 1;
		while(i >= 0 && (!isNaN(cur[i]) || cur[i] == '_'))
			--i;
		
		if (i > 0) {
			var cur2 = cur.substr(0, i + 1);
	    	if (variable_3.propertyIsEnumerable(cur2)) {
	      		if (blockKeywords.propertyIsEnumerable(cur2)) curPunc = "newstatement";
	      		return "variable-3";
	      	}
	    }
    }
    if (atoms.propertyIsEnumerable(cur)) return "atom";
    return "word";
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !(escaped || multiLineStrings))
        state.tokenize = tokenBase;
      return "string";
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  function Context(indented, column, type, align, prev) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.align = align;
    this.prev = prev;
  }
  function pushContext(state, col, type) {
    return state.context = new Context(state.indented, col, type, null, state.context);
  }
  function popContext(state) {
    var t = state.context.type;
    if (t == ")" || t == "]" || t == "}")
      state.indented = state.context.indented;
    return state.context = state.context.prev;
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: null,
        context: new Context((basecolumn || 0) - indentUnit, 0, "top", false),
        indented: 0,
        startOfLine: true
      };
    },

    token: function(stream, state) {
      var ctx = state.context;
      if (stream.sol()) {
        if (ctx.align == null) ctx.align = false;
        state.indented = stream.indentation();
        state.startOfLine = true;
      }
      if (stream.eatSpace()) return null;
      curPunc = null;
      var style = (state.tokenize || tokenBase)(stream, state);
      if (style == "comment" || style == "meta") return style;
      if (ctx.align == null) ctx.align = true;

      if ((curPunc == ";" || curPunc == ":") && ctx.type == "statement") popContext(state);
      else if (curPunc == "{") pushContext(state, stream.column(), "}");
      else if (curPunc == "[") pushContext(state, stream.column(), "]");
      else if (curPunc == "(") pushContext(state, stream.column(), ")");
      else if (curPunc == "}") {
        while (ctx.type == "statement") ctx = popContext(state);
        if (ctx.type == "}") ctx = popContext(state);
        while (ctx.type == "statement") ctx = popContext(state);
      }
      else if (curPunc == ctx.type) popContext(state);
      else if (ctx.type == "}" || ctx.type == "top" || (ctx.type == "statement" && curPunc == "newstatement"))
        pushContext(state, stream.column(), "statement");
      state.startOfLine = false;
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase && state.tokenize != null) return 0;
      var ctx = state.context, firstChar = textAfter && textAfter.charAt(0);
      if (ctx.type == "statement" && firstChar == "}") ctx = ctx.prev;
      var closing = firstChar == ctx.type;
      if (ctx.type == "statement") return ctx.indented + (firstChar == "{" ? 0 : indentUnit);
      else if (ctx.align) return ctx.column + (closing ? 0 : 1);
      else return ctx.indented + (closing ? 0 : indentUnit);
    },

    electricChars: "{}"
  };
});

CodeMirror.defineMIME("text/x-ecl", "ecl");
// erlang    -> CodeMirror tag
//
// atom      -> atom
// attribute -> attribute
// builtin   -> builtin
// comment   -> comment
// error     -> error
// fun       -> meta
// function  -> tag
// guard     -> property
// keyword   -> keyword
// macro     -> variable-2
// number    -> number
// operator  -> operator
// record    -> bracket
// string    -> string
// type      -> def
// variable  -> variable

CodeMirror.defineMIME("text/x-erlang", "erlang");

CodeMirror.defineMode("erlang", function(cmCfg, modeCfg) {

  var typeWords = [
    "-type", "-spec", "-export_type", "-opaque"];

  var keywordWords = [
    "after","begin","catch","case","cond","end","fun","if",
    "let","of","query","receive","try","when"];

  var operatorWords = [
    "and","andalso","band","bnot","bor","bsl","bsr","bxor",
    "div","not","or","orelse","rem","xor"];

  var operatorSymbols = [
    "+","-","*","/",">",">=","<","=<","=:=","==","=/=","/=","||","<-"];

  var guardWords = [
    "is_atom","is_binary","is_bitstring","is_boolean","is_float",
    "is_function","is_integer","is_list","is_number","is_pid",
    "is_port","is_record","is_reference","is_tuple",
    "atom","binary","bitstring","boolean","function","integer","list",
    "number","pid","port","record","reference","tuple"];

  var bifWords = [
    "abs","adler32","adler32_combine","alive","apply","atom_to_binary",
    "atom_to_list","binary_to_atom","binary_to_existing_atom",
    "binary_to_list","binary_to_term","bit_size","bitstring_to_list",
    "byte_size","check_process_code","contact_binary","crc32",
    "crc32_combine","date","decode_packet","delete_module",
    "disconnect_node","element","erase","exit","float","float_to_list",
    "garbage_collect","get","get_keys","group_leader","halt","hd",
    "integer_to_list","internal_bif","iolist_size","iolist_to_binary",
    "is_alive","is_atom","is_binary","is_bitstring","is_boolean",
    "is_float","is_function","is_integer","is_list","is_number","is_pid",
    "is_port","is_process_alive","is_record","is_reference","is_tuple",
    "length","link","list_to_atom","list_to_binary","list_to_bitstring",
    "list_to_existing_atom","list_to_float","list_to_integer",
    "list_to_pid","list_to_tuple","load_module","make_ref","module_loaded",
    "monitor_node","node","node_link","node_unlink","nodes","notalive",
    "now","open_port","pid_to_list","port_close","port_command",
    "port_connect","port_control","pre_loaded","process_flag",
    "process_info","processes","purge_module","put","register",
    "registered","round","self","setelement","size","spawn","spawn_link",
    "spawn_monitor","spawn_opt","split_binary","statistics",
    "term_to_binary","time","throw","tl","trunc","tuple_size",
    "tuple_to_list","unlink","unregister","whereis"];

  function isMember(element,list) {
    return (-1 < list.indexOf(element));
  }

  function isPrev(stream,string) {
    var start = stream.start;
    var len = string.length;
    if (len <= start) {
      var word = stream.string.slice(start-len,start);
      return word == string;
    }else{
      return false;
    }
  }

  var smallRE = /[a-z_]/;
  var largeRE = /[A-Z_]/;
  var digitRE = /[0-9]/;
  var octitRE = /[0-7]/;
  var idRE = /[a-z_A-Z0-9]/;

  function tokenize(stream, state) {
    if (stream.eatSpace()) {
      return null;
    }

    // attributes and type specs
    if (stream.sol() && stream.peek() == '-') {
      stream.next();
      if (stream.eat(smallRE) && stream.eatWhile(idRE)) {
        if (stream.peek() == "(") {
          return "attribute";
        }else if (isMember(stream.current(),typeWords)) {
          return "def";
        }else{
          return null;
        }
      }
      stream.backUp(1);
    }

    var ch = stream.next();

    // comment
    if (ch == '%') {
      stream.skipToEnd();
      return "comment";
    }

    // macro
    if (ch == '?') {
      stream.eatWhile(idRE);
      return "variable-2";
    }

    // record
    if ( ch == "#") {
      stream.eatWhile(idRE);
      return "bracket";
    }

    // char
    if ( ch == "$") {
      if (stream.next() == "\\") {
        if (!stream.eatWhile(octitRE)) {
          stream.next();
        }
      }
      return "string";
    }

    // quoted atom
    if (ch == '\'') {
      return singleQuote(stream);
    }

    // string
    if (ch == '"') {
      return doubleQuote(stream);
    }

    // variable
    if (largeRE.test(ch)) {
      stream.eatWhile(idRE);
      return "variable";
    }

    // atom/keyword/BIF/function
    if (smallRE.test(ch)) {
      stream.eatWhile(idRE);

      if (stream.peek() == "/") {
        stream.next();
        if (stream.eatWhile(digitRE)) {
          return "meta";      // f/0 style fun
        }else{
          stream.backUp(1);
          return "atom";
        }
      }

      var w = stream.current();

      if (isMember(w,keywordWords)) {
        return "keyword";           // keyword
      }
      if (stream.peek() == "(") {
        if (isMember(w,bifWords) &&
            (!isPrev(stream,":") || isPrev(stream,"erlang:"))) {
          return "builtin";         // BIF
        }else{
          return "tag";             // function
        }
      }
      if (isMember(w,guardWords)) {
        return "property";          // guard
      }
      if (isMember(w,operatorWords)) {
        return "operator";          // operator
      }


      if (stream.peek() == ":") {
        if (w == "erlang") {         // f:now() is highlighted incorrectly
          return "builtin";
        } else {
          return "tag";              // function application
        }
      }

      return "atom";
    }

    // number
    if (digitRE.test(ch)) {
      stream.eatWhile(digitRE);
      if (stream.eat('#')) {
        stream.eatWhile(digitRE);    // 16#10  style integer
      } else {
        if (stream.eat('.')) {       // float
          stream.eatWhile(digitRE);
        }
        if (stream.eat(/[eE]/)) {
          stream.eat(/[-+]/);        // float with exponent
          stream.eatWhile(digitRE);
        }
      }
      return "number";               // normal integer
    }

    return null;
  }

  function doubleQuote(stream) {
    return Quote(stream, '"', '\\', "string");
  }

  function singleQuote(stream) {
    return Quote(stream,'\'','\\',"atom");
  }

  function Quote(stream,quoteChar,escapeChar,tag) {
    while (!stream.eol()) {
      var ch = stream.next();
      if (ch == quoteChar) {
        return tag;
      }else if (ch == escapeChar) {
        stream.next();
      }
    }
    return "error";
  }

  return {
    startState: function() {
      return {};
    },

    token: function(stream, state) {
      return tokenize(stream, state);
    }
  };
});
CodeMirror.defineMode("gfm", function(config, parserConfig) {
  var mdMode = CodeMirror.getMode(config, "markdown");
  var aliases = {
    html: "htmlmixed",
    js: "javascript",
    json: "application/json",
    c: "text/x-csrc",
    "c++": "text/x-c++src",
    java: "text/x-java",
    csharp: "text/x-csharp",
    "c#": "text/x-csharp"
  };

  // make this lazy so that we don't need to load GFM last
  var getMode = (function () {
    var i, modes = {}, mimes = {}, mime;

    var list = CodeMirror.listModes();
    for (i = 0; i < list.length; i++) {
      modes[list[i]] = list[i];
    }
    var mimesList = CodeMirror.listMIMEs();
    for (i = 0; i < mimesList.length; i++) {
      mime = mimesList[i].mime;
      mimes[mime] = mimesList[i].mime;
    }

    for (var a in aliases) {
      if (aliases[a] in modes || aliases[a] in mimes)
        modes[a] = aliases[a];
    }
    
    return function (lang) {
      return modes[lang] ? CodeMirror.getMode(config, modes[lang]) : null;
    }
  }());

  function markdown(stream, state) {
    // intercept fenced code blocks
    if (stream.sol() && stream.match(/^```([\w+#]*)/)) {
      // try switching mode
      state.localMode = getMode(RegExp.$1)
      if (state.localMode)
        state.localState = state.localMode.startState();

      state.token = local;
      return 'code';
    }

    return mdMode.token(stream, state.mdState);
  }

  function local(stream, state) {
    if (stream.sol() && stream.match(/^```/)) {
      state.localMode = state.localState = null;
      state.token = markdown;
      return 'code';
    }
    else if (state.localMode) {
      return state.localMode.token(stream, state.localState);
    } else {
      stream.skipToEnd();
      return 'code';
    }
  }

  // custom handleText to prevent emphasis in the middle of a word
  // and add autolinking
  function handleText(stream, mdState) {
    var match;
    if (stream.match(/^\w+:\/\/\S+/)) {
      return 'link';
    }
    if (stream.match(/^[^\[*\\<>` _][^\[*\\<>` ]*[^\[*\\<>` _]/)) {
      return mdMode.getType(mdState);
    }
    if (match = stream.match(/^[^\[*\\<>` ]+/)) {
      var word = match[0];
      if (word[0] === '_' && word[word.length-1] === '_') {
        stream.backUp(word.length);
        return undefined;
      }
      return mdMode.getType(mdState);
    }
    if (stream.eatSpace()) {
      return null;
    }
  }

  return {
    startState: function() {
      var mdState = mdMode.startState();
      mdState.text = handleText;
      return {token: markdown, mode: "markdown", mdState: mdState,
              localMode: null, localState: null};
    },

    copyState: function(state) {
      return {token: state.token, mode: state.mode, mdState: CodeMirror.copyState(mdMode, state.mdState),
              localMode: state.localMode,
              localState: state.localMode ? CodeMirror.copyState(state.localMode, state.localState) : null};
    },

    token: function(stream, state) {
        /* Parse GFM double bracket links */
        if ((ch = stream.peek()) != undefined && ch == '[') {
            stream.next(); // Advance the stream

            /* Only handle double bracket links */
            if ((ch = stream.peek()) == undefined || ch != '[') {
                stream.backUp(1);
                return state.token(stream, state);
            } 

            while ((ch = stream.next()) != undefined && ch != ']') {}

            if (ch == ']' && (ch = stream.next()) != undefined && ch == ']') 
                return 'link';

            /* If we did not find the second ']' */
            stream.backUp(1);
        }

        /* Match GFM latex formulas, as well as latex formulas within '$' */
        if (stream.match(/^\$[^\$]+\$/)) {
            return "string";
        }

        if (stream.match(/^\\\((.*?)\\\)/)) {
            return "string";
        }

        if (stream.match(/^\$\$[^\$]+\$\$/)) {
            return "string";
        }
        
        if (stream.match(/^\\\[(.*?)\\\]/)) {
            return "string";
        }

        return state.token(stream, state);
    }
  }
}, "markdown");
CodeMirror.defineMode("go", function(config, parserConfig) {
  var indentUnit = config.indentUnit;

  var keywords = {
    "break":true, "case":true, "chan":true, "const":true, "continue":true,
    "default":true, "defer":true, "else":true, "fallthrough":true, "for":true,
    "func":true, "go":true, "goto":true, "if":true, "import":true,
    "interface":true, "map":true, "package":true, "range":true, "return":true,
    "select":true, "struct":true, "switch":true, "type":true, "var":true,
    "bool":true, "byte":true, "complex64":true, "complex128":true,
    "float32":true, "float64":true, "int8":true, "int16":true, "int32":true,
    "int64":true, "string":true, "uint8":true, "uint16":true, "uint32":true,
    "uint64":true, "int":true, "uint":true, "uintptr":true
  };

  var atoms = {
    "true":true, "false":true, "iota":true, "nil":true, "append":true,
    "cap":true, "close":true, "complex":true, "copy":true, "imag":true,
    "len":true, "make":true, "new":true, "panic":true, "print":true,
    "println":true, "real":true, "recover":true
  };

  var blockKeywords = {
    "else":true, "for":true, "func":true, "if":true, "interface":true,
    "select":true, "struct":true, "switch":true
  };

  var isOperatorChar = /[+\-*&^%:=<>!|\/]/;

  var curPunc;

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (ch == '"' || ch == "'" || ch == "`") {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    if (/[\d\.]/.test(ch)) {
      if (ch == ".") {
        stream.match(/^[0-9]+([eE][\-+]?[0-9]+)?/);
      } else if (ch == "0") {
        stream.match(/^[xX][0-9a-fA-F]+/) || stream.match(/^0[0-7]+/);
      } else {
        stream.match(/^[0-9]*\.?[0-9]*([eE][\-+]?[0-9]+)?/);
      }
      return "number";
    }
    if (/[\[\]{}\(\),;\:\.]/.test(ch)) {
      curPunc = ch;
      return null
    }
    if (ch == "/") {
      if (stream.eat("*")) {
        state.tokenize = tokenComment;
        return tokenComment(stream, state);
      }
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
    }
    if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return "operator";
    }
    stream.eatWhile(/[\w\$_]/);
    var cur = stream.current();
    if (keywords.propertyIsEnumerable(cur)) {
      if (cur == "case" || cur == "default") curPunc = "case";
      return "keyword";
    }
    if (atoms.propertyIsEnumerable(cur)) return "atom";
    return "word";
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !(escaped || quote == "`"))
        state.tokenize = tokenBase;
      return "string";
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  function Context(indented, column, type, align, prev) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.align = align;
    this.prev = prev;
  }
  function pushContext(state, col, type) {
    return state.context = new Context(state.indented, col, type, null, state.context);
  }
  function popContext(state) {
    var t = state.context.type;
    if (t == ")" || t == "]" || t == "}")
      state.indented = state.context.indented;
    return state.context = state.context.prev;
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: null,
        context: new Context((basecolumn || 0) - indentUnit, 0, "top", false),
        indented: 0,
        startOfLine: true
      };
    },

    token: function(stream, state) {
      var ctx = state.context;
      if (stream.sol()) {
        if (ctx.align == null) ctx.align = false;
        state.indented = stream.indentation();
        state.startOfLine = true;
        if (ctx.type == "case") ctx.type = "}";
      }
      if (stream.eatSpace()) return null;
      curPunc = null;
      var style = (state.tokenize || tokenBase)(stream, state);
      if (style == "comment") return style;
      if (ctx.align == null) ctx.align = true;

      if (curPunc == "{") pushContext(state, stream.column(), "}");
      else if (curPunc == "[") pushContext(state, stream.column(), "]");
      else if (curPunc == "(") pushContext(state, stream.column(), ")");
      else if (curPunc == "case") ctx.type = "case"
      else if (curPunc == "}" && ctx.type == "}") ctx = popContext(state);
      else if (curPunc == ctx.type) popContext(state);
      state.startOfLine = false;
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase && state.tokenize != null) return 0;
      var ctx = state.context, firstChar = textAfter && textAfter.charAt(0);
      if (ctx.type == "case" && /^(?:case|default)\b/.test(textAfter)) {
        state.context.type = "}";
        return ctx.indented;
      }
      var closing = firstChar == ctx.type;
      if (ctx.align) return ctx.column + (closing ? 0 : 1);
      else return ctx.indented + (closing ? 0 : indentUnit);
    },

    electricChars: "{}:"
  };
});

CodeMirror.defineMIME("text/x-go", "go");
CodeMirror.defineMode("groovy", function(config, parserConfig) {
  function words(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }
  var keywords = words(
    "abstract as assert boolean break byte case catch char class const continue def default " +
    "do double else enum extends final finally float for goto if implements import in " +
    "instanceof int interface long native new package private protected public return " +
    "short static strictfp super switch synchronized threadsafe throw throws transient " +
    "try void volatile while");
  var blockKeywords = words("catch class do else finally for if switch try while enum interface def");
  var atoms = words("null true false this");

  var curPunc;
  function tokenBase(stream, state) {
    var ch = stream.next();
    if (ch == '"' || ch == "'") {
      return startString(ch, stream, state);
    }
    if (/[\[\]{}\(\),;\:\.]/.test(ch)) {
      curPunc = ch;
      return null
    }
    if (/\d/.test(ch)) {
      stream.eatWhile(/[\w\.]/);
      if (stream.eat(/eE/)) { stream.eat(/\+\-/); stream.eatWhile(/\d/); }
      return "number";
    }
    if (ch == "/") {
      if (stream.eat("*")) {
        state.tokenize.push(tokenComment);
        return tokenComment(stream, state);
      }
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
      if (expectExpression(state.lastToken)) {
        return startString(ch, stream, state);
      }
    }
    if (ch == "-" && stream.eat(">")) {
      curPunc = "->";
      return null;
    }
    if (/[+\-*&%=<>!?|\/~]/.test(ch)) {
      stream.eatWhile(/[+\-*&%=<>|~]/);
      return "operator";
    }
    stream.eatWhile(/[\w\$_]/);
    if (ch == "@") { stream.eatWhile(/[\w\$_\.]/); return "meta"; }
    if (state.lastToken == ".") return "property";
    if (stream.eat(":")) { curPunc = "proplabel"; return "property"; }
    var cur = stream.current();
    if (atoms.propertyIsEnumerable(cur)) { return "atom"; }
    if (keywords.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "keyword";
    }
    return "word";
  }
  tokenBase.isBase = true;

  function startString(quote, stream, state) {
    var tripleQuoted = false;
    if (quote != "/" && stream.eat(quote)) {
      if (stream.eat(quote)) tripleQuoted = true;
      else return "string";
    }
    function t(stream, state) {
      var escaped = false, next, end = !tripleQuoted;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {
          if (!tripleQuoted) { break; }
          if (stream.match(quote + quote)) { end = true; break; }
        }
        if (quote == '"' && next == "$" && !escaped && stream.eat("{")) {
          state.tokenize.push(tokenBaseUntilBrace());
          return "string";
        }
        escaped = !escaped && next == "\\";
      }
      if (end) state.tokenize.pop();
      return "string";
    }
    state.tokenize.push(t);
    return t(stream, state);
  }

  function tokenBaseUntilBrace() {
    var depth = 1;
    function t(stream, state) {
      if (stream.peek() == "}") {
        depth--;
        if (depth == 0) {
          state.tokenize.pop();
          return state.tokenize[state.tokenize.length-1](stream, state);
        }
      } else if (stream.peek() == "{") {
        depth++;
      }
      return tokenBase(stream, state);
    }
    t.isBase = true;
    return t;
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize.pop();
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  function expectExpression(last) {
    return !last || last == "operator" || last == "->" || /[\.\[\{\(,;:]/.test(last) ||
      last == "newstatement" || last == "keyword" || last == "proplabel";
  }

  function Context(indented, column, type, align, prev) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.align = align;
    this.prev = prev;
  }
  function pushContext(state, col, type) {
    return state.context = new Context(state.indented, col, type, null, state.context);
  }
  function popContext(state) {
    var t = state.context.type;
    if (t == ")" || t == "]" || t == "}")
      state.indented = state.context.indented;
    return state.context = state.context.prev;
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: [tokenBase],
        context: new Context((basecolumn || 0) - config.indentUnit, 0, "top", false),
        indented: 0,
        startOfLine: true,
        lastToken: null
      };
    },

    token: function(stream, state) {
      var ctx = state.context;
      if (stream.sol()) {
        if (ctx.align == null) ctx.align = false;
        state.indented = stream.indentation();
        state.startOfLine = true;
        // Automatic semicolon insertion
        if (ctx.type == "statement" && !expectExpression(state.lastToken)) {
          popContext(state); ctx = state.context;
        }
      }
      if (stream.eatSpace()) return null;
      curPunc = null;
      var style = state.tokenize[state.tokenize.length-1](stream, state);
      if (style == "comment") return style;
      if (ctx.align == null) ctx.align = true;

      if ((curPunc == ";" || curPunc == ":") && ctx.type == "statement") popContext(state);
      // Handle indentation for {x -> \n ... }
      else if (curPunc == "->" && ctx.type == "statement" && ctx.prev.type == "}") {
        popContext(state);
        state.context.align = false;
      }
      else if (curPunc == "{") pushContext(state, stream.column(), "}");
      else if (curPunc == "[") pushContext(state, stream.column(), "]");
      else if (curPunc == "(") pushContext(state, stream.column(), ")");
      else if (curPunc == "}") {
        while (ctx.type == "statement") ctx = popContext(state);
        if (ctx.type == "}") ctx = popContext(state);
        while (ctx.type == "statement") ctx = popContext(state);
      }
      else if (curPunc == ctx.type) popContext(state);
      else if (ctx.type == "}" || ctx.type == "top" || (ctx.type == "statement" && curPunc == "newstatement"))
        pushContext(state, stream.column(), "statement");
      state.startOfLine = false;
      state.lastToken = curPunc || style;
      return style;
    },

    indent: function(state, textAfter) {
      if (!state.tokenize[state.tokenize.length-1].isBase) return 0;
      var firstChar = textAfter && textAfter.charAt(0), ctx = state.context;
      if (ctx.type == "statement" && !expectExpression(state.lastToken)) ctx = ctx.prev;
      var closing = firstChar == ctx.type;
      if (ctx.type == "statement") return ctx.indented + (firstChar == "{" ? 0 : config.indentUnit);
      else if (ctx.align) return ctx.column + (closing ? 0 : 1);
      else return ctx.indented + (closing ? 0 : config.indentUnit);
    },

    electricChars: "{}"
  };
});

CodeMirror.defineMIME("text/x-groovy", "groovy");
CodeMirror.defineMode("haskell", function(cmCfg, modeCfg) {

  function switchState(source, setState, f) {
    setState(f);
    return f(source, setState);
  }
  
  // These should all be Unicode extended, as per the Haskell 2010 report
  var smallRE = /[a-z_]/;
  var largeRE = /[A-Z]/;
  var digitRE = /[0-9]/;
  var hexitRE = /[0-9A-Fa-f]/;
  var octitRE = /[0-7]/;
  var idRE = /[a-z_A-Z0-9']/;
  var symbolRE = /[-!#$%&*+.\/<=>?@\\^|~:]/;
  var specialRE = /[(),;[\]`{}]/;
  var whiteCharRE = /[ \t\v\f]/; // newlines are handled in tokenizer
    
  function normal(source, setState) {
    if (source.eatWhile(whiteCharRE)) {
      return null;
    }
      
    var ch = source.next();
    if (specialRE.test(ch)) {
      if (ch == '{' && source.eat('-')) {
        var t = "comment";
        if (source.eat('#')) {
          t = "meta";
        }
        return switchState(source, setState, ncomment(t, 1));
      }
      return null;
    }
    
    if (ch == '\'') {
      if (source.eat('\\')) {
        source.next();  // should handle other escapes here
      }
      else {
        source.next();
      }
      if (source.eat('\'')) {
        return "string";
      }
      return "error";
    }
    
    if (ch == '"') {
      return switchState(source, setState, stringLiteral);
    }
      
    if (largeRE.test(ch)) {
      source.eatWhile(idRE);
      if (source.eat('.')) {
        return "qualifier";
      }
      return "variable-2";
    }
      
    if (smallRE.test(ch)) {
      source.eatWhile(idRE);
      return "variable";
    }
      
    if (digitRE.test(ch)) {
      if (ch == '0') {
        if (source.eat(/[xX]/)) {
          source.eatWhile(hexitRE); // should require at least 1
          return "integer";
        }
        if (source.eat(/[oO]/)) {
          source.eatWhile(octitRE); // should require at least 1
          return "number";
        }
      }
      source.eatWhile(digitRE);
      var t = "number";
      if (source.eat('.')) {
        t = "number";
        source.eatWhile(digitRE); // should require at least 1
      }
      if (source.eat(/[eE]/)) {
        t = "number";
        source.eat(/[-+]/);
        source.eatWhile(digitRE); // should require at least 1
      }
      return t;
    }
      
    if (symbolRE.test(ch)) {
      if (ch == '-' && source.eat(/-/)) {
        source.eatWhile(/-/);
        if (!source.eat(symbolRE)) {
          source.skipToEnd();
          return "comment";
        }
      }
      var t = "variable";
      if (ch == ':') {
        t = "variable-2";
      }
      source.eatWhile(symbolRE);
      return t;    
    }
      
    return "error";
  }
    
  function ncomment(type, nest) {
    if (nest == 0) {
      return normal;
    }
    return function(source, setState) {
      var currNest = nest;
      while (!source.eol()) {
        var ch = source.next();
        if (ch == '{' && source.eat('-')) {
          ++currNest;
        }
        else if (ch == '-' && source.eat('}')) {
          --currNest;
          if (currNest == 0) {
            setState(normal);
            return type;
          }
        }
      }
      setState(ncomment(type, currNest));
      return type;
    }
  }
    
  function stringLiteral(source, setState) {
    while (!source.eol()) {
      var ch = source.next();
      if (ch == '"') {
        setState(normal);
        return "string";
      }
      if (ch == '\\') {
        if (source.eol() || source.eat(whiteCharRE)) {
          setState(stringGap);
          return "string";
        }
        if (source.eat('&')) {
        }
        else {
          source.next(); // should handle other escapes here
        }
      }
    }
    setState(normal);
    return "error";
  }
  
  function stringGap(source, setState) {
    if (source.eat('\\')) {
      return switchState(source, setState, stringLiteral);
    }
    source.next();
    setState(normal);
    return "error";
  }
  
  
  var wellKnownWords = (function() {
    var wkw = {};
    function setType(t) {
      return function () {
        for (var i = 0; i < arguments.length; i++)
          wkw[arguments[i]] = t;
      }
    }
    
    setType("keyword")(
      "case", "class", "data", "default", "deriving", "do", "else", "foreign",
      "if", "import", "in", "infix", "infixl", "infixr", "instance", "let",
      "module", "newtype", "of", "then", "type", "where", "_");
      
    setType("keyword")(
      "\.\.", ":", "::", "=", "\\", "\"", "<-", "->", "@", "~", "=>");
      
    setType("builtin")(
      "!!", "$!", "$", "&&", "+", "++", "-", ".", "/", "/=", "<", "<=", "=<<",
      "==", ">", ">=", ">>", ">>=", "^", "^^", "||", "*", "**");
      
    setType("builtin")(
      "Bool", "Bounded", "Char", "Double", "EQ", "Either", "Enum", "Eq",
      "False", "FilePath", "Float", "Floating", "Fractional", "Functor", "GT",
      "IO", "IOError", "Int", "Integer", "Integral", "Just", "LT", "Left",
      "Maybe", "Monad", "Nothing", "Num", "Ord", "Ordering", "Rational", "Read",
      "ReadS", "Real", "RealFloat", "RealFrac", "Right", "Show", "ShowS",
      "String", "True");
      
    setType("builtin")(
      "abs", "acos", "acosh", "all", "and", "any", "appendFile", "asTypeOf",
      "asin", "asinh", "atan", "atan2", "atanh", "break", "catch", "ceiling",
      "compare", "concat", "concatMap", "const", "cos", "cosh", "curry",
      "cycle", "decodeFloat", "div", "divMod", "drop", "dropWhile", "either",
      "elem", "encodeFloat", "enumFrom", "enumFromThen", "enumFromThenTo",
      "enumFromTo", "error", "even", "exp", "exponent", "fail", "filter",
      "flip", "floatDigits", "floatRadix", "floatRange", "floor", "fmap",
      "foldl", "foldl1", "foldr", "foldr1", "fromEnum", "fromInteger",
      "fromIntegral", "fromRational", "fst", "gcd", "getChar", "getContents",
      "getLine", "head", "id", "init", "interact", "ioError", "isDenormalized",
      "isIEEE", "isInfinite", "isNaN", "isNegativeZero", "iterate", "last",
      "lcm", "length", "lex", "lines", "log", "logBase", "lookup", "map",
      "mapM", "mapM_", "max", "maxBound", "maximum", "maybe", "min", "minBound",
      "minimum", "mod", "negate", "not", "notElem", "null", "odd", "or",
      "otherwise", "pi", "pred", "print", "product", "properFraction",
      "putChar", "putStr", "putStrLn", "quot", "quotRem", "read", "readFile",
      "readIO", "readList", "readLn", "readParen", "reads", "readsPrec",
      "realToFrac", "recip", "rem", "repeat", "replicate", "return", "reverse",
      "round", "scaleFloat", "scanl", "scanl1", "scanr", "scanr1", "seq",
      "sequence", "sequence_", "show", "showChar", "showList", "showParen",
      "showString", "shows", "showsPrec", "significand", "signum", "sin",
      "sinh", "snd", "span", "splitAt", "sqrt", "subtract", "succ", "sum",
      "tail", "take", "takeWhile", "tan", "tanh", "toEnum", "toInteger",
      "toRational", "truncate", "uncurry", "undefined", "unlines", "until",
      "unwords", "unzip", "unzip3", "userError", "words", "writeFile", "zip",
      "zip3", "zipWith", "zipWith3");
      
    return wkw;
  })();
    
  
  
  return {
    startState: function ()  { return { f: normal }; },
    copyState:  function (s) { return { f: s.f }; },
    
    token: function(stream, state) {
      var t = state.f(stream, function(s) { state.f = s; });
      var w = stream.current();
      return (w in wellKnownWords) ? wellKnownWords[w] : t;
    }
  };

});

CodeMirror.defineMIME("text/x-haskell", "haskell");
CodeMirror.defineMode("xml", function(config, parserConfig) {
  var indentUnit = config.indentUnit;
  var Kludges = parserConfig.htmlMode ? {
    autoSelfClosers: {'area': true, 'base': true, 'br': true, 'col': true, 'command': true,
                      'embed': true, 'frame': true, 'hr': true, 'img': true, 'input': true,
                      'keygen': true, 'link': true, 'meta': true, 'param': true, 'source': true,
                      'track': true, 'wbr': true},
    implicitlyClosed: {'dd': true, 'li': true, 'optgroup': true, 'option': true, 'p': true,
                       'rp': true, 'rt': true, 'tbody': true, 'td': true, 'tfoot': true,
                       'th': true, 'tr': true},
    contextGrabbers: {
      'dd': {'dd': true, 'dt': true},
      'dt': {'dd': true, 'dt': true},
      'li': {'li': true},
      'option': {'option': true, 'optgroup': true},
      'optgroup': {'optgroup': true},
      'p': {'address': true, 'article': true, 'aside': true, 'blockquote': true, 'dir': true,
            'div': true, 'dl': true, 'fieldset': true, 'footer': true, 'form': true,
            'h1': true, 'h2': true, 'h3': true, 'h4': true, 'h5': true, 'h6': true,
            'header': true, 'hgroup': true, 'hr': true, 'menu': true, 'nav': true, 'ol': true,
            'p': true, 'pre': true, 'section': true, 'table': true, 'ul': true},
      'rp': {'rp': true, 'rt': true},
      'rt': {'rp': true, 'rt': true},
      'tbody': {'tbody': true, 'tfoot': true},
      'td': {'td': true, 'th': true},
      'tfoot': {'tbody': true},
      'th': {'td': true, 'th': true},
      'thead': {'tbody': true, 'tfoot': true},
      'tr': {'tr': true}
    },
    doNotIndent: {"pre": true},
    allowUnquoted: true,
    allowMissing: false
  } : {
    autoSelfClosers: {},
    implicitlyClosed: {},
    contextGrabbers: {},
    doNotIndent: {},
    allowUnquoted: false,
    allowMissing: false
  };
  var alignCDATA = parserConfig.alignCDATA;

  // Return variables for tokenizers
  var tagName, type;

  function inText(stream, state) {
    function chain(parser) {
      state.tokenize = parser;
      return parser(stream, state);
    }

    var ch = stream.next();
    if (ch == "<") {
      if (stream.eat("!")) {
        if (stream.eat("[")) {
          if (stream.match("CDATA[")) return chain(inBlock("atom", "]]>"));
          else return null;
        }
        else if (stream.match("--")) return chain(inBlock("comment", "-->"));
        else if (stream.match("DOCTYPE", true, true)) {
          stream.eatWhile(/[\w\._\-]/);
          return chain(doctype(1));
        }
        else return null;
      }
      else if (stream.eat("?")) {
        stream.eatWhile(/[\w\._\-]/);
        state.tokenize = inBlock("meta", "?>");
        return "meta";
      }
      else {
        type = stream.eat("/") ? "closeTag" : "openTag";
        stream.eatSpace();
        tagName = "";
        var c;
        while ((c = stream.eat(/[^\s\u00a0=<>\"\'\/?]/))) tagName += c;
        state.tokenize = inTag;
        return "tag";
      }
    }
    else if (ch == "&") {
      var ok;
      if (stream.eat("#")) {
        if (stream.eat("x")) {
          ok = stream.eatWhile(/[a-fA-F\d]/) && stream.eat(";");          
        } else {
          ok = stream.eatWhile(/[\d]/) && stream.eat(";");
        }
      } else {
        ok = stream.eatWhile(/[\w\.\-:]/) && stream.eat(";");
      }
      return ok ? "atom" : "error";
    }
    else {
      stream.eatWhile(/[^&<]/);
      return null;
    }
  }

  function inTag(stream, state) {
    var ch = stream.next();
    if (ch == ">" || (ch == "/" && stream.eat(">"))) {
      state.tokenize = inText;
      type = ch == ">" ? "endTag" : "selfcloseTag";
      return "tag";
    }
    else if (ch == "=") {
      type = "equals";
      return null;
    }
    else if (/[\'\"]/.test(ch)) {
      state.tokenize = inAttribute(ch);
      return state.tokenize(stream, state);
    }
    else {
      stream.eatWhile(/[^\s\u00a0=<>\"\'\/?]/);
      return "word";
    }
  }

  function inAttribute(quote) {
    return function(stream, state) {
      while (!stream.eol()) {
        if (stream.next() == quote) {
          state.tokenize = inTag;
          break;
        }
      }
      return "string";
    };
  }

  function inBlock(style, terminator) {
    return function(stream, state) {
      while (!stream.eol()) {
        if (stream.match(terminator)) {
          state.tokenize = inText;
          break;
        }
        stream.next();
      }
      return style;
    };
  }
  function doctype(depth) {
    return function(stream, state) {
      var ch;
      while ((ch = stream.next()) != null) {
        if (ch == "<") {
          state.tokenize = doctype(depth + 1);
          return state.tokenize(stream, state);
        } else if (ch == ">") {
          if (depth == 1) {
            state.tokenize = inText;
            break;
          } else {
            state.tokenize = doctype(depth - 1);
            return state.tokenize(stream, state);
          }
        }
      }
      return "meta";
    };
  }

  var curState, setStyle;
  function pass() {
    for (var i = arguments.length - 1; i >= 0; i--) curState.cc.push(arguments[i]);
  }
  function cont() {
    pass.apply(null, arguments);
    return true;
  }

  function pushContext(tagName, startOfLine) {
    var noIndent = Kludges.doNotIndent.hasOwnProperty(tagName) || (curState.context && curState.context.noIndent);
    curState.context = {
      prev: curState.context,
      tagName: tagName,
      indent: curState.indented,
      startOfLine: startOfLine,
      noIndent: noIndent
    };
  }
  function popContext() {
    if (curState.context) curState.context = curState.context.prev;
  }

  function element(type) {
    if (type == "openTag") {
      curState.tagName = tagName;
      return cont(attributes, endtag(curState.startOfLine));
    } else if (type == "closeTag") {
      var err = false;
      if (curState.context) {
        if (curState.context.tagName != tagName) {
          if (Kludges.implicitlyClosed.hasOwnProperty(curState.context.tagName.toLowerCase())) {
            popContext();
          }
          err = !curState.context || curState.context.tagName != tagName;
        }
      } else {
        err = true;
      }
      if (err) setStyle = "error";
      return cont(endclosetag(err));
    }
    return cont();
  }
  function endtag(startOfLine) {
    return function(type) {
      if (type == "selfcloseTag" ||
          (type == "endTag" && Kludges.autoSelfClosers.hasOwnProperty(curState.tagName.toLowerCase()))) {
        maybePopContext(curState.tagName.toLowerCase());
        return cont();
      }
      if (type == "endTag") {
        maybePopContext(curState.tagName.toLowerCase());
        pushContext(curState.tagName, startOfLine);
        return cont();
      }
      return cont();
    };
  }
  function endclosetag(err) {
    return function(type) {
      if (err) setStyle = "error";
      if (type == "endTag") { popContext(); return cont(); }
      setStyle = "error";
      return cont(arguments.callee);
    }
  }
  function maybePopContext(nextTagName) {
    var parentTagName;
    while (true) {
      if (!curState.context) {
        return;
      }
      parentTagName = curState.context.tagName.toLowerCase();
      if (!Kludges.contextGrabbers.hasOwnProperty(parentTagName) ||
          !Kludges.contextGrabbers[parentTagName].hasOwnProperty(nextTagName)) {
        return;
      }
      popContext();
    }
  }

  function attributes(type) {
    if (type == "word") {setStyle = "attribute"; return cont(attribute, attributes);}
    if (type == "endTag" || type == "selfcloseTag") return pass();
    setStyle = "error";
    return cont(attributes);
  }
  function attribute(type) {
    if (type == "equals") return cont(attvalue, attributes);
    if (!Kludges.allowMissing) setStyle = "error";
    return (type == "endTag" || type == "selfcloseTag") ? pass() : cont();
  }
  function attvalue(type) {
    if (type == "string") return cont(attvaluemaybe);
    if (type == "word" && Kludges.allowUnquoted) {setStyle = "string"; return cont();}
    setStyle = "error";
    return (type == "endTag" || type == "selfCloseTag") ? pass() : cont();
  }
  function attvaluemaybe(type) {
    if (type == "string") return cont(attvaluemaybe);
    else return pass();
  }

  return {
    startState: function() {
      return {tokenize: inText, cc: [], indented: 0, startOfLine: true, tagName: null, context: null};
    },

    token: function(stream, state) {
      if (stream.sol()) {
        state.startOfLine = true;
        state.indented = stream.indentation();
      }
      if (stream.eatSpace()) return null;

      setStyle = type = tagName = null;
      var style = state.tokenize(stream, state);
      state.type = type;
      if ((style || type) && style != "comment") {
        curState = state;
        while (true) {
          var comb = state.cc.pop() || element;
          if (comb(type || style)) break;
        }
      }
      state.startOfLine = false;
      return setStyle || style;
    },

    indent: function(state, textAfter, fullLine) {
      var context = state.context;
      if ((state.tokenize != inTag && state.tokenize != inText) ||
          context && context.noIndent)
        return fullLine ? fullLine.match(/^(\s*)/)[0].length : 0;
      if (alignCDATA && /<!\[CDATA\[/.test(textAfter)) return 0;
      if (context && /^<\//.test(textAfter))
        context = context.prev;
      while (context && !context.startOfLine)
        context = context.prev;
      if (context) return context.indent + indentUnit;
      else return 0;
    },

    compareStates: function(a, b) {
      if (a.indented != b.indented || a.tokenize != b.tokenize) return false;
      for (var ca = a.context, cb = b.context; ; ca = ca.prev, cb = cb.prev) {
        if (!ca || !cb) return ca == cb;
        if (ca.tagName != cb.tagName) return false;
      }
    },

    electricChars: "/"
  };
});

CodeMirror.defineMIME("application/xml", "xml");
if (!CodeMirror.mimeModes.hasOwnProperty("text/html"))
  CodeMirror.defineMIME("text/html", {name: "xml", htmlMode: true});
CodeMirror.defineMode("css", function(config) {
  var indentUnit = config.indentUnit, type;
  function ret(style, tp) {type = tp; return style;}

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (ch == "@") {stream.eatWhile(/[\w\\\-]/); return ret("meta", stream.current());}
    else if (ch == "/" && stream.eat("*")) {
      state.tokenize = tokenCComment;
      return tokenCComment(stream, state);
    }
    else if (ch == "<" && stream.eat("!")) {
      state.tokenize = tokenSGMLComment;
      return tokenSGMLComment(stream, state);
    }
    else if (ch == "=") ret(null, "compare");
    else if ((ch == "~" || ch == "|") && stream.eat("=")) return ret(null, "compare");
    else if (ch == "\"" || ch == "'") {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    else if (ch == "#") {
      stream.eatWhile(/[\w\\\-]/);
      return ret("atom", "hash");
    }
    else if (ch == "!") {
      stream.match(/^\s*\w*/);
      return ret("keyword", "important");
    }
    else if (/\d/.test(ch)) {
      stream.eatWhile(/[\w.%]/);
      return ret("number", "unit");
    }
    else if (/[,.+>*\/]/.test(ch)) {
      return ret(null, "select-op");
    }
    else if (/[;{}:\[\]]/.test(ch)) {
      return ret(null, ch);
    }
    else {
      stream.eatWhile(/[\w\\\-]/);
      return ret("variable", "variable");
    }
  }

  function tokenCComment(stream, state) {
    var maybeEnd = false, ch;
    while ((ch = stream.next()) != null) {
      if (maybeEnd && ch == "/") {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return ret("comment", "comment");
  }

  function tokenSGMLComment(stream, state) {
    var dashes = 0, ch;
    while ((ch = stream.next()) != null) {
      if (dashes >= 2 && ch == ">") {
        state.tokenize = tokenBase;
        break;
      }
      dashes = (ch == "-") ? dashes + 1 : 0;
    }
    return ret("comment", "comment");
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && !escaped)
          break;
        escaped = !escaped && ch == "\\";
      }
      if (!escaped) state.tokenize = tokenBase;
      return ret("string", "string");
    };
  }

  return {
    startState: function(base) {
      return {tokenize: tokenBase,
              baseIndent: base || 0,
              stack: []};
    },

    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);

      var context = state.stack[state.stack.length-1];
      if (type == "hash" && context != "rule") style = "string-2";
      else if (style == "variable") {
        if (context == "rule") style = "number";
        else if (!context || context == "@media{") style = "tag";
      }

      if (context == "rule" && /^[\{\};]$/.test(type))
        state.stack.pop();
      if (type == "{") {
        if (context == "@media") state.stack[state.stack.length-1] = "@media{";
        else state.stack.push("{");
      }
      else if (type == "}") state.stack.pop();
      else if (type == "@media") state.stack.push("@media");
      else if (context == "{" && type != "comment") state.stack.push("rule");
      return style;
    },

    indent: function(state, textAfter) {
      var n = state.stack.length;
      if (/^\}/.test(textAfter))
        n -= state.stack[state.stack.length-1] == "rule" ? 2 : 1;
      return state.baseIndent + n * indentUnit;
    },

    electricChars: "}"
  };
});

CodeMirror.defineMIME("text/css", "css");
CodeMirror.defineMode("htmlembedded", function(config, parserConfig) {
  
  //config settings
  var scriptStartRegex = parserConfig.scriptStartRegex || /^<%/i,
      scriptEndRegex = parserConfig.scriptEndRegex || /^%>/i;
  
  //inner modes
  var scriptingMode, htmlMixedMode;
  
  //tokenizer when in html mode
  function htmlDispatch(stream, state) {
      if (stream.match(scriptStartRegex, false)) {
          state.token=scriptingDispatch;
          return scriptingMode.token(stream, state.scriptState);
          }
      else
          return htmlMixedMode.token(stream, state.htmlState);
    }

  //tokenizer when in scripting mode
  function scriptingDispatch(stream, state) {
      if (stream.match(scriptEndRegex, false))  {
          state.token=htmlDispatch;
          return htmlMixedMode.token(stream, state.htmlState);
         }
      else
          return scriptingMode.token(stream, state.scriptState);
         }


  return {
    startState: function() {
      scriptingMode = scriptingMode || CodeMirror.getMode(config, parserConfig.scriptingModeSpec);
      htmlMixedMode = htmlMixedMode || CodeMirror.getMode(config, "htmlmixed");
      return { 
          token :  parserConfig.startOpen ? scriptingDispatch : htmlDispatch,
          htmlState : htmlMixedMode.startState(),
          scriptState : scriptingMode.startState()
          }
    },

    token: function(stream, state) {
      return state.token(stream, state);
    },

    indent: function(state, textAfter) {
      if (state.token == htmlDispatch)
        return htmlMixedMode.indent(state.htmlState, textAfter);
      else
        return scriptingMode.indent(state.scriptState, textAfter);
    },
    
    copyState: function(state) {
      return {
       token : state.token,
       htmlState : CodeMirror.copyState(htmlMixedMode, state.htmlState),
       scriptState : CodeMirror.copyState(scriptingMode, state.scriptState)
       }
    },
    

    electricChars: "/{}:"
  }
}, "htmlmixed");

CodeMirror.defineMIME("application/x-ejs", { name: "htmlembedded", scriptingModeSpec:"javascript"});
CodeMirror.defineMIME("application/x-aspx", { name: "htmlembedded", scriptingModeSpec:"text/x-csharp"});
CodeMirror.defineMIME("application/x-jsp", { name: "htmlembedded", scriptingModeSpec:"text/x-java"});
CodeMirror.defineMode("htmlmixed", function(config, parserConfig) {
  var htmlMode = CodeMirror.getMode(config, {name: "xml", htmlMode: true});
  var jsMode = CodeMirror.getMode(config, "javascript");
  var cssMode = CodeMirror.getMode(config, "css");

  function html(stream, state) {
    var style = htmlMode.token(stream, state.htmlState);
    if (style == "tag" && stream.current() == ">" && state.htmlState.context) {
      if (/^script$/i.test(state.htmlState.context.tagName)) {
        state.token = javascript;
        state.localState = jsMode.startState(htmlMode.indent(state.htmlState, ""));
        state.mode = "javascript";
      }
      else if (/^style$/i.test(state.htmlState.context.tagName)) {
        state.token = css;
        state.localState = cssMode.startState(htmlMode.indent(state.htmlState, ""));
        state.mode = "css";
      }
    }
    return style;
  }
  function maybeBackup(stream, pat, style) {
    var cur = stream.current();
    var close = cur.search(pat);
    if (close > -1) stream.backUp(cur.length - close);
    return style;
  }
  function javascript(stream, state) {
    if (stream.match(/^<\/\s*script\s*>/i, false)) {
      state.token = html;
      state.localState = null;
      state.mode = "html";
      return html(stream, state);
    }
    return maybeBackup(stream, /<\/\s*script\s*>/,
                       jsMode.token(stream, state.localState));
  }
  function css(stream, state) {
    if (stream.match(/^<\/\s*style\s*>/i, false)) {
      state.token = html;
      state.localState = null;
      state.mode = "html";
      return html(stream, state);
    }
    return maybeBackup(stream, /<\/\s*style\s*>/,
                       cssMode.token(stream, state.localState));
  }

  return {
    startState: function() {
      var state = htmlMode.startState();
      return {token: html, localState: null, mode: "html", htmlState: state};
    },

    copyState: function(state) {
      if (state.localState)
        var local = CodeMirror.copyState(state.token == css ? cssMode : jsMode, state.localState);
      return {token: state.token, localState: local, mode: state.mode,
              htmlState: CodeMirror.copyState(htmlMode, state.htmlState)};
    },

    token: function(stream, state) {
      return state.token(stream, state);
    },

    indent: function(state, textAfter) {
      if (state.token == html || /^\s*<\//.test(textAfter))
        return htmlMode.indent(state.htmlState, textAfter);
      else if (state.token == javascript)
        return jsMode.indent(state.localState, textAfter);
      else
        return cssMode.indent(state.localState, textAfter);
    },

    compareStates: function(a, b) {
      if (a.mode != b.mode) return false;
      if (a.localState) return CodeMirror.Pass;
      return htmlMode.compareStates(a.htmlState, b.htmlState);
    },

    electricChars: "/{}:"
  }
}, "xml", "javascript", "css");

CodeMirror.defineMIME("text/html", "htmlmixed");
CodeMirror.defineMode("javascript", function(config, parserConfig) {
  var indentUnit = config.indentUnit;
  var jsonMode = parserConfig.json;

  // Tokenizer

  var keywords = function(){
    function kw(type) {return {type: type, style: "keyword"};}
    var A = kw("keyword a"), B = kw("keyword b"), C = kw("keyword c");
    var operator = kw("operator"), atom = {type: "atom", style: "atom"};
    return {
      "if": A, "while": A, "with": A, "else": B, "do": B, "try": B, "finally": B,
      "return": C, "break": C, "continue": C, "new": C, "delete": C, "throw": C,
      "var": kw("var"), "const": kw("var"), "let": kw("var"),
      "function": kw("function"), "catch": kw("catch"),
      "for": kw("for"), "switch": kw("switch"), "case": kw("case"), "default": kw("default"),
      "in": operator, "typeof": operator, "instanceof": operator,
      "true": atom, "false": atom, "null": atom, "undefined": atom, "NaN": atom, "Infinity": atom
    };
  }();

  var isOperatorChar = /[+\-*&%=<>!?|]/;

  function chain(stream, state, f) {
    state.tokenize = f;
    return f(stream, state);
  }

  function nextUntilUnescaped(stream, end) {
    var escaped = false, next;
    while ((next = stream.next()) != null) {
      if (next == end && !escaped)
        return false;
      escaped = !escaped && next == "\\";
    }
    return escaped;
  }

  // Used as scratch variables to communicate multiple values without
  // consing up tons of objects.
  var type, content;
  function ret(tp, style, cont) {
    type = tp; content = cont;
    return style;
  }

  function jsTokenBase(stream, state) {
    var ch = stream.next();
    if (ch == '"' || ch == "'")
      return chain(stream, state, jsTokenString(ch));
    else if (/[\[\]{}\(\),;\:\.]/.test(ch))
      return ret(ch);
    else if (ch == "0" && stream.eat(/x/i)) {
      stream.eatWhile(/[\da-f]/i);
      return ret("number", "number");
    }      
    else if (/\d/.test(ch) || ch == "-" && stream.eat(/\d/)) {
      stream.match(/^\d*(?:\.\d*)?(?:[eE][+\-]?\d+)?/);
      return ret("number", "number");
    }
    else if (ch == "/") {
      if (stream.eat("*")) {
        return chain(stream, state, jsTokenComment);
      }
      else if (stream.eat("/")) {
        stream.skipToEnd();
        return ret("comment", "comment");
      }
      else if (state.reAllowed) {
        nextUntilUnescaped(stream, "/");
        stream.eatWhile(/[gimy]/); // 'y' is "sticky" option in Mozilla
        return ret("regexp", "string-2");
      }
      else {
        stream.eatWhile(isOperatorChar);
        return ret("operator", null, stream.current());
      }
    }
    else if (ch == "#") {
        stream.skipToEnd();
        return ret("error", "error");
    }
    else if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return ret("operator", null, stream.current());
    }
    else {
      stream.eatWhile(/[\w\$_]/);
      var word = stream.current(), known = keywords.propertyIsEnumerable(word) && keywords[word];
      return (known && state.kwAllowed) ? ret(known.type, known.style, word) :
                     ret("variable", "variable", word);
    }
  }

  function jsTokenString(quote) {
    return function(stream, state) {
      if (!nextUntilUnescaped(stream, quote))
        state.tokenize = jsTokenBase;
      return ret("string", "string");
    };
  }

  function jsTokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = jsTokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return ret("comment", "comment");
  }

  // Parser

  var atomicTypes = {"atom": true, "number": true, "variable": true, "string": true, "regexp": true};

  function JSLexical(indented, column, type, align, prev, info) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.prev = prev;
    this.info = info;
    if (align != null) this.align = align;
  }

  function inScope(state, varname) {
    for (var v = state.localVars; v; v = v.next)
      if (v.name == varname) return true;
  }

  function parseJS(state, style, type, content, stream) {
    var cc = state.cc;
    // Communicate our context to the combinators.
    // (Less wasteful than consing up a hundred closures on every call.)
    cx.state = state; cx.stream = stream; cx.marked = null, cx.cc = cc;
  
    if (!state.lexical.hasOwnProperty("align"))
      state.lexical.align = true;

    while(true) {
      var combinator = cc.length ? cc.pop() : jsonMode ? expression : statement;
      if (combinator(type, content)) {
        while(cc.length && cc[cc.length - 1].lex)
          cc.pop()();
        if (cx.marked) return cx.marked;
        if (type == "variable" && inScope(state, content)) return "variable-2";
        return style;
      }
    }
  }

  // Combinator utils

  var cx = {state: null, column: null, marked: null, cc: null};
  function pass() {
    for (var i = arguments.length - 1; i >= 0; i--) cx.cc.push(arguments[i]);
  }
  function cont() {
    pass.apply(null, arguments);
    return true;
  }
  function register(varname) {
    var state = cx.state;
    if (state.context) {
      cx.marked = "def";
      for (var v = state.localVars; v; v = v.next)
        if (v.name == varname) return;
      state.localVars = {name: varname, next: state.localVars};
    }
  }

  // Combinators

  var defaultVars = {name: "this", next: {name: "arguments"}};
  function pushcontext() {
    if (!cx.state.context) cx.state.localVars = defaultVars;
    cx.state.context = {prev: cx.state.context, vars: cx.state.localVars};
  }
  function popcontext() {
    cx.state.localVars = cx.state.context.vars;
    cx.state.context = cx.state.context.prev;
  }
  function pushlex(type, info) {
    var result = function() {
      var state = cx.state;
      state.lexical = new JSLexical(state.indented, cx.stream.column(), type, null, state.lexical, info)
    };
    result.lex = true;
    return result;
  }
  function poplex() {
    var state = cx.state;
    if (state.lexical.prev) {
      if (state.lexical.type == ")")
        state.indented = state.lexical.indented;
      state.lexical = state.lexical.prev;
    }
  }
  poplex.lex = true;

  function expect(wanted) {
    return function expecting(type) {
      if (type == wanted) return cont();
      else if (wanted == ";") return pass();
      else return cont(arguments.callee);
    };
  }

  function statement(type) {
    if (type == "var") return cont(pushlex("vardef"), vardef1, expect(";"), poplex);
    if (type == "keyword a") return cont(pushlex("form"), expression, statement, poplex);
    if (type == "keyword b") return cont(pushlex("form"), statement, poplex);
    if (type == "{") return cont(pushlex("}"), block, poplex);
    if (type == ";") return cont();
    if (type == "function") return cont(functiondef);
    if (type == "for") return cont(pushlex("form"), expect("("), pushlex(")"), forspec1, expect(")"),
                                      poplex, statement, poplex);
    if (type == "variable") return cont(pushlex("stat"), maybelabel);
    if (type == "switch") return cont(pushlex("form"), expression, pushlex("}", "switch"), expect("{"),
                                         block, poplex, poplex);
    if (type == "case") return cont(expression, expect(":"));
    if (type == "default") return cont(expect(":"));
    if (type == "catch") return cont(pushlex("form"), pushcontext, expect("("), funarg, expect(")"),
                                        statement, poplex, popcontext);
    return pass(pushlex("stat"), expression, expect(";"), poplex);
  }
  function expression(type) {
    if (atomicTypes.hasOwnProperty(type)) return cont(maybeoperator);
    if (type == "function") return cont(functiondef);
    if (type == "keyword c") return cont(maybeexpression);
    if (type == "(") return cont(pushlex(")"), maybeexpression, expect(")"), poplex, maybeoperator);
    if (type == "operator") return cont(expression);
    if (type == "[") return cont(pushlex("]"), commasep(expression, "]"), poplex, maybeoperator);
    if (type == "{") return cont(pushlex("}"), commasep(objprop, "}"), poplex, maybeoperator);
    return cont();
  }
  function maybeexpression(type) {
    if (type.match(/[;\}\)\],]/)) return pass();
    return pass(expression);
  }
    
  function maybeoperator(type, value) {
    if (type == "operator" && /\+\+|--/.test(value)) return cont(maybeoperator);
    if (type == "operator" || type == ":") return cont(expression);
    if (type == ";") return;
    if (type == "(") return cont(pushlex(")"), commasep(expression, ")"), poplex, maybeoperator);
    if (type == ".") return cont(property, maybeoperator);
    if (type == "[") return cont(pushlex("]"), expression, expect("]"), poplex, maybeoperator);
  }
  function maybelabel(type) {
    if (type == ":") return cont(poplex, statement);
    return pass(maybeoperator, expect(";"), poplex);
  }
  function property(type) {
    if (type == "variable") {cx.marked = "property"; return cont();}
  }
  function objprop(type) {
    if (type == "variable") cx.marked = "property";
    if (atomicTypes.hasOwnProperty(type)) return cont(expect(":"), expression);
  }
  function commasep(what, end) {
    function proceed(type) {
      if (type == ",") return cont(what, proceed);
      if (type == end) return cont();
      return cont(expect(end));
    }
    return function commaSeparated(type) {
      if (type == end) return cont();
      else return pass(what, proceed);
    };
  }
  function block(type) {
    if (type == "}") return cont();
    return pass(statement, block);
  }
  function vardef1(type, value) {
    if (type == "variable"){register(value); return cont(vardef2);}
    return cont();
  }
  function vardef2(type, value) {
    if (value == "=") return cont(expression, vardef2);
    if (type == ",") return cont(vardef1);
  }
  function forspec1(type) {
    if (type == "var") return cont(vardef1, forspec2);
    if (type == ";") return pass(forspec2);
    if (type == "variable") return cont(formaybein);
    return pass(forspec2);
  }
  function formaybein(type, value) {
    if (value == "in") return cont(expression);
    return cont(maybeoperator, forspec2);
  }
  function forspec2(type, value) {
    if (type == ";") return cont(forspec3);
    if (value == "in") return cont(expression);
    return cont(expression, expect(";"), forspec3);
  }
  function forspec3(type) {
    if (type != ")") cont(expression);
  }
  function functiondef(type, value) {
    if (type == "variable") {register(value); return cont(functiondef);}
    if (type == "(") return cont(pushlex(")"), pushcontext, commasep(funarg, ")"), poplex, statement, popcontext);
  }
  function funarg(type, value) {
    if (type == "variable") {register(value); return cont();}
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: jsTokenBase,
        reAllowed: true,
        kwAllowed: true,
        cc: [],
        lexical: new JSLexical((basecolumn || 0) - indentUnit, 0, "block", false),
        localVars: parserConfig.localVars,
        context: parserConfig.localVars && {vars: parserConfig.localVars},
        indented: 0
      };
    },

    token: function(stream, state) {
      if (stream.sol()) {
        if (!state.lexical.hasOwnProperty("align"))
          state.lexical.align = false;
        state.indented = stream.indentation();
      }
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);
      if (type == "comment") return style;
      state.reAllowed = !!(type == "operator" || type == "keyword c" || type.match(/^[\[{}\(,;:]$/));
      state.kwAllowed = type != '.';
      return parseJS(state, style, type, content, stream);
    },

    indent: function(state, textAfter) {
      if (state.tokenize != jsTokenBase) return 0;
      var firstChar = textAfter && textAfter.charAt(0), lexical = state.lexical;
      if (lexical.type == "stat" && firstChar == "}") lexical = lexical.prev;
      var type = lexical.type, closing = firstChar == type;
      if (type == "vardef") return lexical.indented + 4;
      else if (type == "form" && firstChar == "{") return lexical.indented;
      else if (type == "stat" || type == "form") return lexical.indented + indentUnit;
      else if (lexical.info == "switch" && !closing)
        return lexical.indented + (/^(?:case|default)\b/.test(textAfter) ? indentUnit : 2 * indentUnit);
      else if (lexical.align) return lexical.column + (closing ? 0 : 1);
      else return lexical.indented + (closing ? 0 : indentUnit);
    },

    electricChars: ":{}"
  };
});

CodeMirror.defineMIME("text/javascript", "javascript");
CodeMirror.defineMIME("application/json", {name: "javascript", json: true});
CodeMirror.defineMode("jinja2", function(config, parserConf) {
    var keywords = ["block", "endblock", "for", "endfor", "in", "true", "false", 
                    "loop", "none", "self", "super", "if", "as", "not", "and",
                    "else", "import", "with", "without", "context"];
    keywords = new RegExp("^((" + keywords.join(")|(") + "))\\b");

    function tokenBase (stream, state) {
        var ch = stream.next();
        if (ch == "{") {
            if (ch = stream.eat(/\{|%|#/)) {
                stream.eat("-");
                state.tokenize = inTag(ch);
                return "tag";
            }
        }
    }
    function inTag (close) {
        if (close == "{") {
            close = "}";
        }
        return function (stream, state) {
            var ch = stream.next();
            if ((ch == close || (ch == "-" && stream.eat(close)))
                && stream.eat("}")) {
                state.tokenize = tokenBase;
                return "tag";
            }
            if (stream.match(keywords)) {
                return "keyword";
            }
            return close == "#" ? "comment" : "string";
        };
    }
    return {
        startState: function () {
            return {tokenize: tokenBase};
        },
        token: function (stream, state) {
            return state.tokenize(stream, state);
        }
    }; 
});
/*
LESS mode - http://www.lesscss.org/
Ported to CodeMirror by Peter Kroon
*/

CodeMirror.defineMode("less", function(config) {
  var indentUnit = config.indentUnit, type;
  function ret(style, tp) {type = tp; return style;}
  //html5 tags
  var tags = ["a","abbr","acronym","address","applet","area","article","aside","audio","b","base","basefont","bdi","bdo","big","blockquote","body","br","button","canvas","caption","cite","code","col","colgroup","command","datalist","dd","del","details","dfn","dir","div","dl","dt","em","embed","fieldset","figcaption","figure","font","footer","form","frame","frameset","h1","h2","h3","h4","h5","h6","head","header","hgroup","hr","html","i","iframe","img","input","ins","keygen","kbd","label","legend","li","link","map","mark","menu","meta","meter","nav","noframes","noscript","object","ol","optgroup","option","output","p","param","pre","progress","q","rp","rt","ruby","s","samp","script","section","select","small","source","span","strike","strong","style","sub","summary","sup","table","tbody","td","textarea","tfoot","th","thead","time","title","tr","track","tt","u","ul","var","video","wbr"];
  
  function inTagsArray(val){
	  for(var i=0; i<tags.length; i++){
		  if(val === tags[i]){
			  return true;
		  }
	  }
  }

  function tokenBase(stream, state) {
    var ch = stream.next();

	if (ch == "@") {stream.eatWhile(/[\w\-]/); return ret("meta", stream.current());}
    else if (ch == "/" && stream.eat("*")) {
      state.tokenize = tokenCComment;
      return tokenCComment(stream, state);
    }
    else if (ch == "<" && stream.eat("!")) {
      state.tokenize = tokenSGMLComment;
      return tokenSGMLComment(stream, state);
    }
    else if (ch == "=") ret(null, "compare");
    else if ((ch == "~" || ch == "|") && stream.eat("=")) return ret(null, "compare");
    else if (ch == "\"" || ch == "'") {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
	else if (ch == "/") { // lesscss e.g.: .png will not be parsed as a class
	  if(stream.eat("/")){
		state.tokenize = tokenSComment
      	return tokenSComment(stream, state);
	  }else{
	    stream.eatWhile(/[\a-zA-Z0-9\-_.\s]/);
		if(/\/|\)|#/.test(stream.peek() || stream.eol() || (stream.eatSpace() && stream.peek() == ")")))return ret("string", "string");//let url(/images/logo.png) without quotes return as string
        return ret("number", "unit");
	  }
    }
    else if (ch == "!") {
      stream.match(/^\s*\w*/);
      return ret("keyword", "important");
    }
    else if (/\d/.test(ch)) {
      stream.eatWhile(/[\w.%]/);
      return ret("number", "unit");
    }
    else if (/[,+<>*\/]/.test(ch)) {//removed . dot character original was [,.+>*\/]
      return ret(null, "select-op");
    }
    else if (/[;{}:\[\]()]/.test(ch)) { //added () char for lesscss original was [;{}:\[\]]
      if(ch == ":"){
		stream.eatWhile(/[active|hover|link|visited]/);
		if( stream.current().match(/active|hover|link|visited/)){
		  return ret("tag", "tag");
		}else{
		  return ret(null, ch);	
		}
	  }else{
  	    return ret(null, ch);
	  }
    }
	else if (ch == ".") { // lesscss
	  stream.eatWhile(/[\a-zA-Z0-9\-_]/);
      return ret("tag", "tag");
    }
	else if (ch == "#") { // lesscss
	  //we don't eat white-space, we want the hex color and or id only
	  stream.eatWhile(/[A-Za-z0-9]/);
	  //check if there is a proper hex color length e.g. #eee || #eeeEEE
	  if(stream.current().length ===4 || stream.current().length ===7){
		  if(stream.current().match(/[A-Fa-f0-9]{6}|[A-Fa-f0-9]{3}/,false) != null){//is there a valid hex color value present in the current stream
		  	//when not a valid hex value, parse as id
			if(stream.current().substring(1) != stream.current().match(/[A-Fa-f0-9]{6}|[A-Fa-f0-9]{3}/,false))return ret("atom", "tag");
			//eat white-space
			stream.eatSpace();
			//when hex value declaration doesn't end with [;,] but is does with a slash/cc comment treat it as an id, just like the other hex values that don't end with[;,]
			if( /[\/<>.(){!$%^&*_\-\\?=+\|#'~`]/.test(stream.peek()) )return ret("atom", "tag");
			//#time { color: #aaa }
			else if(stream.peek() == "}" )return ret("number", "unit");
			//we have a valid hex color value, parse as id whenever an element/class is defined after the hex(id) value e.g. #eee aaa || #eee .aaa
			else if( /[a-zA-Z\\]/.test(stream.peek()) )return ret("atom", "tag");
			//when a hex value is on the end of a line, parse as id
			else if(stream.eol())return ret("atom", "tag");
			//default
			else return ret("number", "unit");
		  }else{//when not a valid hexvalue in the current stream e.g. #footer
			stream.eatWhile(/[\w\\\-]/);
			return ret("atom", "tag"); 
		  }
	  }else{
		stream.eatWhile(/[\w\\\-]/);		
		return ret("atom", "tag");
	  }
    }
	else if (ch == "&") {
	  stream.eatWhile(/[\w\-]/);
	  return ret(null, ch);
	}
    else {
      stream.eatWhile(/[\w\\\-_%.{]/);
	  if(stream.current().match(/http|https/) != null){
		stream.eatWhile(/[\w\\\-_%.{:\/]/);
		return ret("string", "string");
	  }else if(stream.peek() == "<" || stream.peek() == ">"){
		return ret("tag", "tag");
	  }else if( stream.peek().match(/\(/) != null ){// lessc
		return ret(null, ch);
	  }else if (stream.peek() == "/" && state.stack[state.stack.length-1] != undefined){ // url(dir/center/image.png)
	  	return ret("string", "string");
	  }else if( stream.current().match(/\-\d|\-.\d/) ){ // lesscss match e.g.: -5px -0.4 etc... only colorize the minus sign
		//stream.backUp(stream.current().length-1); //commment out these 2 comment if you want the minus sign to be parsed as null -500px
	  	//return ret(null, ch);
		return ret("number", "unit");
	  }else if( inTagsArray(stream.current()) ){ // lesscss match html tags
	  	return ret("tag", "tag");
	  }else if( /\/|[\s\)]/.test(stream.peek() || stream.eol() || (stream.eatSpace() && stream.peek() == "/")) && stream.current().indexOf(".") !== -1){
		if(stream.current().substring(stream.current().length-1,stream.current().length) == "{"){
			stream.backUp(1);
			return ret("tag", "tag");
		}//end if
		if( (stream.eatSpace() && stream.peek().match(/[{<>.a-zA-Z]/) != null)  || stream.eol() )return ret("tag", "tag");//e.g. button.icon-plus
		return ret("string", "string");//let url(/images/logo.png) without quotes return as string
	  }else if( stream.eol() ){
		  if(stream.current().substring(stream.current().length-1,stream.current().length) == "{")stream.backUp(1);
		  return ret("tag", "tag");
	  }else{
      	return ret("variable", "variable");
	  }
    }
    
  }

  function tokenSComment(stream, state) {// SComment = Slash comment
    stream.skipToEnd();
	state.tokenize = tokenBase;
    return ret("comment", "comment");
  }
    
  function tokenCComment(stream, state) {
    var maybeEnd = false, ch;
    while ((ch = stream.next()) != null) {
      if (maybeEnd && ch == "/") {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return ret("comment", "comment");
  }

  function tokenSGMLComment(stream, state) {
    var dashes = 0, ch;
    while ((ch = stream.next()) != null) {
      if (dashes >= 2 && ch == ">") {
        state.tokenize = tokenBase;
        break;
      }
      dashes = (ch == "-") ? dashes + 1 : 0;
    }
    return ret("comment", "comment");
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && !escaped)
          break;
        escaped = !escaped && ch == "\\";
      }
      if (!escaped) state.tokenize = tokenBase;
      return ret("string", "string");
    };
  }

  return {
    startState: function(base) { 
      return {tokenize: tokenBase,
              baseIndent: base || 0,
              stack: []};
    },

    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);

      var context = state.stack[state.stack.length-1];
      if (type == "hash" && context == "rule") style = "atom";
      else if (style == "variable") {
        if (context == "rule") style = null; //"tag"
        else if (!context || context == "@media{"){ 
			style = stream.current() 	== "when" 	? "variable" 	: 
			stream.string.match(/#/g) 	!= undefined 	? null 		: 
			/[\s,|\s\)]/.test(stream.peek()) 		? "tag" 	: null;
		}
      }

      if (context == "rule" && /^[\{\};]$/.test(type))
        state.stack.pop();
      if (type == "{") {
        if (context == "@media") state.stack[state.stack.length-1] = "@media{";
        else state.stack.push("{");
      }
      else if (type == "}") state.stack.pop();
      else if (type == "@media") state.stack.push("@media");
      else if (context == "{" && type != "comment") state.stack.push("rule");
      return style;
    },

    indent: function(state, textAfter) {
      var n = state.stack.length;
      if (/^\}/.test(textAfter))
        n -= state.stack[state.stack.length-1] == "rule" ? 2 : 1;
      return state.baseIndent + n * indentUnit;
    },

    electricChars: "}"
  };
});

CodeMirror.defineMIME("text/x-less", "less");
if (!CodeMirror.mimeModes.hasOwnProperty("text/css"))
  CodeMirror.defineMIME("text/css", "less");
// LUA mode. Ported to CodeMirror 2 from Franciszek Wawrzak's
// CodeMirror 1 mode.
// highlights keywords, strings, comments (no leveling supported! ("[==[")), tokens, basic indenting
 
CodeMirror.defineMode("lua", function(config, parserConfig) {
  var indentUnit = config.indentUnit;

  function prefixRE(words) {
    return new RegExp("^(?:" + words.join("|") + ")", "i");
  }
  function wordRE(words) {
    return new RegExp("^(?:" + words.join("|") + ")$", "i");
  }
  var specials = wordRE(parserConfig.specials || []);
 
  // long list of standard functions from lua manual
  var builtins = wordRE([
    "_G","_VERSION","assert","collectgarbage","dofile","error","getfenv","getmetatable","ipairs","load",
    "loadfile","loadstring","module","next","pairs","pcall","print","rawequal","rawget","rawset","require",
    "select","setfenv","setmetatable","tonumber","tostring","type","unpack","xpcall",

    "coroutine.create","coroutine.resume","coroutine.running","coroutine.status","coroutine.wrap","coroutine.yield",

    "debug.debug","debug.getfenv","debug.gethook","debug.getinfo","debug.getlocal","debug.getmetatable",
    "debug.getregistry","debug.getupvalue","debug.setfenv","debug.sethook","debug.setlocal","debug.setmetatable",
    "debug.setupvalue","debug.traceback",

    "close","flush","lines","read","seek","setvbuf","write",

    "io.close","io.flush","io.input","io.lines","io.open","io.output","io.popen","io.read","io.stderr","io.stdin",
    "io.stdout","io.tmpfile","io.type","io.write",

    "math.abs","math.acos","math.asin","math.atan","math.atan2","math.ceil","math.cos","math.cosh","math.deg",
    "math.exp","math.floor","math.fmod","math.frexp","math.huge","math.ldexp","math.log","math.log10","math.max",
    "math.min","math.modf","math.pi","math.pow","math.rad","math.random","math.randomseed","math.sin","math.sinh",
    "math.sqrt","math.tan","math.tanh",

    "os.clock","os.date","os.difftime","os.execute","os.exit","os.getenv","os.remove","os.rename","os.setlocale",
    "os.time","os.tmpname",

    "package.cpath","package.loaded","package.loaders","package.loadlib","package.path","package.preload",
    "package.seeall",

    "string.byte","string.char","string.dump","string.find","string.format","string.gmatch","string.gsub",
    "string.len","string.lower","string.match","string.rep","string.reverse","string.sub","string.upper",

    "table.concat","table.insert","table.maxn","table.remove","table.sort"
  ]);
  var keywords = wordRE(["and","break","elseif","false","nil","not","or","return",
			 "true","function", "end", "if", "then", "else", "do", 
			 "while", "repeat", "until", "for", "in", "local" ]);

  var indentTokens = wordRE(["function", "if","repeat","do", "\\(", "{"]);
  var dedentTokens = wordRE(["end", "until", "\\)", "}"]);
  var dedentPartial = prefixRE(["end", "until", "\\)", "}", "else", "elseif"]);

  function readBracket(stream) {
    var level = 0;
    while (stream.eat("=")) ++level;
    stream.eat("[");
    return level;
  }

  function normal(stream, state) {
    var ch = stream.next();
    if (ch == "-" && stream.eat("-")) {
      if (stream.eat("["))
        return (state.cur = bracketed(readBracket(stream), "comment"))(stream, state);
      stream.skipToEnd();
      return "comment";
    } 
    if (ch == "\"" || ch == "'")
      return (state.cur = string(ch))(stream, state);
    if (ch == "[" && /[\[=]/.test(stream.peek()))
      return (state.cur = bracketed(readBracket(stream), "string"))(stream, state);
    if (/\d/.test(ch)) {
      stream.eatWhile(/[\w.%]/);
      return "number";
    }
    if (/[\w_]/.test(ch)) {
      stream.eatWhile(/[\w\\\-_.]/);
      return "variable";
    }
    return null;
  }

  function bracketed(level, style) {
    return function(stream, state) {
      var curlev = null, ch;
      while ((ch = stream.next()) != null) {
        if (curlev == null) {if (ch == "]") curlev = 0;}
        else if (ch == "=") ++curlev;
        else if (ch == "]" && curlev == level) { state.cur = normal; break; }
        else curlev = null;
      }
      return style;
    };
  }

  function string(quote) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && !escaped) break;
        escaped = !escaped && ch == "\\";
      }
      if (!escaped) state.cur = normal;
      return "string";
    };
  }
    
  return {
    startState: function(basecol) {
      return {basecol: basecol || 0, indentDepth: 0, cur: normal};
    },

    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      var style = state.cur(stream, state);
      var word = stream.current();
      if (style == "variable") {
        if (keywords.test(word)) style = "keyword";
        else if (builtins.test(word)) style = "builtin";
	else if (specials.test(word)) style = "variable-2";
      }
      if ((style != "comment") && (style != "string")){
        if (indentTokens.test(word)) ++state.indentDepth;
        else if (dedentTokens.test(word)) --state.indentDepth;
      }
      return style;
    },

    indent: function(state, textAfter) {
      var closing = dedentPartial.test(textAfter);
      return state.basecol + indentUnit * (state.indentDepth - (closing ? 1 : 0));
    }
  };
});

CodeMirror.defineMIME("text/x-lua", "lua");
CodeMirror.defineMode("markdown", function(cmCfg, modeCfg) {

  var htmlMode = CodeMirror.getMode(cmCfg, { name: 'xml', htmlMode: true });

  var header   = 'header'
  ,   code     = 'comment'
  ,   quote    = 'quote'
  ,   list     = 'string'
  ,   hr       = 'hr'
  ,   linktext = 'link'
  ,   linkhref = 'string'
  ,   em       = 'em'
  ,   strong   = 'strong'
  ,   emstrong = 'emstrong';

  var hrRE = /^([*\-=_])(?:\s*\1){2,}\s*$/
  ,   ulRE = /^[*\-+]\s+/
  ,   olRE = /^[0-9]+\.\s+/
  ,   headerRE = /^(?:\={3,}|-{3,})$/
  ,   textRE = /^[^\[*_\\<>`]+/;

  function switchInline(stream, state, f) {
    state.f = state.inline = f;
    return f(stream, state);
  }

  function switchBlock(stream, state, f) {
    state.f = state.block = f;
    return f(stream, state);
  }


  // Blocks

  function blankLine(state) {
    // Reset EM state
    state.em = false;
    // Reset STRONG state
    state.strong = false;
    return null;
  }

  function blockNormal(stream, state) {
    var match;
    if (state.indentationDiff >= 4) {
      state.indentation -= state.indentationDiff;
      stream.skipToEnd();
      return code;
    } else if (stream.eatSpace()) {
      return null;
    } else if (stream.peek() === '#' || stream.match(headerRE)) {
      state.header = true;
    } else if (stream.eat('>')) {
      state.indentation++;
      state.quote = true;
    } else if (stream.peek() === '[') {
      return switchInline(stream, state, footnoteLink);
    } else if (stream.match(hrRE, true)) {
      return hr;
    } else if (match = stream.match(ulRE, true) || stream.match(olRE, true)) {
      state.indentation += match[0].length;
      return list;
    }
    
    return switchInline(stream, state, state.inline);
  }

  function htmlBlock(stream, state) {
    var style = htmlMode.token(stream, state.htmlState);
    if (style === 'tag' && state.htmlState.type !== 'openTag' && !state.htmlState.context) {
      state.f = inlineNormal;
      state.block = blockNormal;
    }
    return style;
  }


  // Inline
  function getType(state) {
    var styles = [];
    
    if (state.strong) { styles.push(state.em ? emstrong : strong); }
    else if (state.em) { styles.push(em); }
    
    if (state.header) { styles.push(header); }
    if (state.quote) { styles.push(quote); }

    return styles.length ? styles.join(' ') : null;
  }

  function handleText(stream, state) {
    if (stream.match(textRE, true)) {
      return getType(state);
    }
    return undefined;        
  }

  function inlineNormal(stream, state) {
    var style = state.text(stream, state)
    if (typeof style !== 'undefined')
      return style;
    
    var ch = stream.next();
    
    if (ch === '\\') {
      stream.next();
      return getType(state);
    }
    if (ch === '`') {
      return switchInline(stream, state, inlineElement(code, '`'));
    }
    if (ch === '[') {
      return switchInline(stream, state, linkText);
    }
    if (ch === '<' && stream.match(/^\w/, false)) {
      stream.backUp(1);
      return switchBlock(stream, state, htmlBlock);
    }

    var t = getType(state);
    if (ch === '*' || ch === '_') {
      if (stream.eat(ch)) {
        return (state.strong = !state.strong) ? getType(state) : t;
      }
      return (state.em = !state.em) ? getType(state) : t;
    }
    
    return getType(state);
  }

  function linkText(stream, state) {
    while (!stream.eol()) {
      var ch = stream.next();
      if (ch === '\\') stream.next();
      if (ch === ']') {
        state.inline = state.f = linkHref;
        return linktext;
      }
    }
    return linktext;
  }

  function linkHref(stream, state) {
    stream.eatSpace();
    var ch = stream.next();
    if (ch === '(' || ch === '[') {
      return switchInline(stream, state, inlineElement(linkhref, ch === '(' ? ')' : ']'));
    }
    return 'error';
  }

  function footnoteLink(stream, state) {
    if (stream.match(/^[^\]]*\]:/, true)) {
      state.f = footnoteUrl;
      return linktext;
    }
    return switchInline(stream, state, inlineNormal);
  }

  function footnoteUrl(stream, state) {
    stream.eatSpace();
    stream.match(/^[^\s]+/, true);
    state.f = state.inline = inlineNormal;
    return linkhref;
  }

  function inlineRE(endChar) {
    if (!inlineRE[endChar]) {
      // match any not-escaped-non-endChar and any escaped char
      // then match endChar or eol
      inlineRE[endChar] = new RegExp('^(?:[^\\\\\\' + endChar + ']|\\\\.)*(?:\\' + endChar + '|$)');
    }
    return inlineRE[endChar];
  }

  function inlineElement(type, endChar, next) {
    next = next || inlineNormal;
    return function(stream, state) {
      stream.match(inlineRE(endChar));
      state.inline = state.f = next;
      return type;
    };
  }

  return {
    startState: function() {
      return {
        f: blockNormal,
        
        block: blockNormal,
        htmlState: htmlMode.startState(),
        indentation: 0,
        
        inline: inlineNormal,
        text: handleText,
        em: false,
        strong: false,
        header: false,
        quote: false
      };
    },

    copyState: function(s) {
      return {
        f: s.f,
        
        block: s.block,
        htmlState: CodeMirror.copyState(htmlMode, s.htmlState),
        indentation: s.indentation,
        
        inline: s.inline,
        text: s.text,
        em: s.em,
        strong: s.strong,
        header: s.header,
        quote: s.quote
      };
    },

    token: function(stream, state) {
      if (stream.sol()) {
        if (stream.match(/^\s*$/, true)) { return blankLine(state); }

        // Reset state.header
        state.header = false;
        // Reset state.quote
        state.quote = false;

        state.f = state.block;
        var indentation = stream.match(/^\s*/, true)[0].replace(/\t/g, '    ').length;
        state.indentationDiff = indentation - state.indentation;
        state.indentation = indentation;
        if (indentation > 0) { return null; }
      }
      return state.f(stream, state);
    },

    blankLine: blankLine,

    getType: getType
  };

}, "xml");

CodeMirror.defineMIME("text/x-markdown", "markdown");
/*
 *	MySQL Mode for CodeMirror 2 by MySQL-Tools
 *	@author James Thorne (partydroid)
 *	@link 	http://github.com/partydroid/MySQL-Tools
 * 	@link 	http://mysqltools.org
 *	@version 02/Jan/2012
*/
CodeMirror.defineMode("mysql", function(config) {
  var indentUnit = config.indentUnit;
  var curPunc;

  function wordRegexp(words) {
    return new RegExp("^(?:" + words.join("|") + ")$", "i");
  }
  var ops = wordRegexp(["str", "lang", "langmatches", "datatype", "bound", "sameterm", "isiri", "isuri",
                        "isblank", "isliteral", "union", "a"]);
  var keywords = wordRegexp([
  	('ACCESSIBLE'),('ALTER'),('AS'),('BEFORE'),('BINARY'),('BY'),('CASE'),('CHARACTER'),('COLUMN'),('CONTINUE'),('CROSS'),('CURRENT_TIMESTAMP'),('DATABASE'),('DAY_MICROSECOND'),('DEC'),('DEFAULT'),
	('DESC'),('DISTINCT'),('DOUBLE'),('EACH'),('ENCLOSED'),('EXIT'),('FETCH'),('FLOAT8'),('FOREIGN'),('GRANT'),('HIGH_PRIORITY'),('HOUR_SECOND'),('IN'),('INNER'),('INSERT'),('INT2'),('INT8'),
	('INTO'),('JOIN'),('KILL'),('LEFT'),('LINEAR'),('LOCALTIME'),('LONG'),('LOOP'),('MATCH'),('MEDIUMTEXT'),('MINUTE_SECOND'),('NATURAL'),('NULL'),('OPTIMIZE'),('OR'),('OUTER'),('PRIMARY'),
	('RANGE'),('READ_WRITE'),('REGEXP'),('REPEAT'),('RESTRICT'),('RIGHT'),('SCHEMAS'),('SENSITIVE'),('SHOW'),('SPECIFIC'),('SQLSTATE'),('SQL_CALC_FOUND_ROWS'),('STARTING'),('TERMINATED'),
	('TINYINT'),('TRAILING'),('UNDO'),('UNLOCK'),('USAGE'),('UTC_DATE'),('VALUES'),('VARCHARACTER'),('WHERE'),('WRITE'),('ZEROFILL'),('ALL'),('AND'),('ASENSITIVE'),('BIGINT'),('BOTH'),('CASCADE'),
	('CHAR'),('COLLATE'),('CONSTRAINT'),('CREATE'),('CURRENT_TIME'),('CURSOR'),('DAY_HOUR'),('DAY_SECOND'),('DECLARE'),('DELETE'),('DETERMINISTIC'),('DIV'),('DUAL'),('ELSEIF'),('EXISTS'),('FALSE'),
	('FLOAT4'),('FORCE'),('FULLTEXT'),('HAVING'),('HOUR_MINUTE'),('IGNORE'),('INFILE'),('INSENSITIVE'),('INT1'),('INT4'),('INTERVAL'),('ITERATE'),('KEYS'),('LEAVE'),('LIMIT'),('LOAD'),('LOCK'),
	('LONGTEXT'),('MASTER_SSL_VERIFY_SERVER_CERT'),('MEDIUMINT'),('MINUTE_MICROSECOND'),('MODIFIES'),('NO_WRITE_TO_BINLOG'),('ON'),('OPTIONALLY'),('OUT'),('PRECISION'),('PURGE'),('READS'),
	('REFERENCES'),('RENAME'),('REQUIRE'),('REVOKE'),('SCHEMA'),('SELECT'),('SET'),('SPATIAL'),('SQLEXCEPTION'),('SQL_BIG_RESULT'),('SSL'),('TABLE'),('TINYBLOB'),('TO'),('TRUE'),('UNIQUE'),
	('UPDATE'),('USING'),('UTC_TIMESTAMP'),('VARCHAR'),('WHEN'),('WITH'),('YEAR_MONTH'),('ADD'),('ANALYZE'),('ASC'),('BETWEEN'),('BLOB'),('CALL'),('CHANGE'),('CHECK'),('CONDITION'),('CONVERT'),
	('CURRENT_DATE'),('CURRENT_USER'),('DATABASES'),('DAY_MINUTE'),('DECIMAL'),('DELAYED'),('DESCRIBE'),('DISTINCTROW'),('DROP'),('ELSE'),('ESCAPED'),('EXPLAIN'),('FLOAT'),('FOR'),('FROM'),
	('GROUP'),('HOUR_MICROSECOND'),('IF'),('INDEX'),('INOUT'),('INT'),('INT3'),('INTEGER'),('IS'),('KEY'),('LEADING'),('LIKE'),('LINES'),('LOCALTIMESTAMP'),('LONGBLOB'),('LOW_PRIORITY'),
	('MEDIUMBLOB'),('MIDDLEINT'),('MOD'),('NOT'),('NUMERIC'),('OPTION'),('ORDER'),('OUTFILE'),('PROCEDURE'),('READ'),('REAL'),('RELEASE'),('REPLACE'),('RETURN'),('RLIKE'),('SECOND_MICROSECOND'),
	('SEPARATOR'),('SMALLINT'),('SQL'),('SQLWARNING'),('SQL_SMALL_RESULT'),('STRAIGHT_JOIN'),('THEN'),('TINYTEXT'),('TRIGGER'),('UNION'),('UNSIGNED'),('USE'),('UTC_TIME'),('VARBINARY'),('VARYING'),
	('WHILE'),('XOR'),('FULL'),('COLUMNS'),('MIN'),('MAX'),('STDEV'),('COUNT')
  ]);
  var operatorChars = /[*+\-<>=&|]/;

  function tokenBase(stream, state) {
    var ch = stream.next();
    curPunc = null;
    if (ch == "$" || ch == "?") {
      stream.match(/^[\w\d]*/);
      return "variable-2";
    }
    else if (ch == "<" && !stream.match(/^[\s\u00a0=]/, false)) {
      stream.match(/^[^\s\u00a0>]*>?/);
      return "atom";
    }
    else if (ch == "\"" || ch == "'") {
      state.tokenize = tokenLiteral(ch);
      return state.tokenize(stream, state);
    }
    else if (ch == "`") {
      state.tokenize = tokenOpLiteral(ch);
      return state.tokenize(stream, state);
    }
    else if (/[{}\(\),\.;\[\]]/.test(ch)) {
      curPunc = ch;
      return null;
    }
    else if (ch == "-") {
		ch2 = stream.next();
		if(ch2=="-")
		{
			stream.skipToEnd();
			return "comment";
		}

    }
    else if (operatorChars.test(ch)) {
      stream.eatWhile(operatorChars);
      return null;
    }
    else if (ch == ":") {
      stream.eatWhile(/[\w\d\._\-]/);
      return "atom";
    }
    else {
      stream.eatWhile(/[_\w\d]/);
      if (stream.eat(":")) {
        stream.eatWhile(/[\w\d_\-]/);
        return "atom";
      }
      var word = stream.current(), type;
      if (ops.test(word))
        return null;
      else if (keywords.test(word))
        return "keyword";
      else
        return "variable";
    }
  }

  function tokenLiteral(quote) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && !escaped) {
          state.tokenize = tokenBase;
          break;
        }
        escaped = !escaped && ch == "\\";
      }
      return "string";
    };
  }

  function tokenOpLiteral(quote) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && !escaped) {
          state.tokenize = tokenBase;
          break;
        }
        escaped = !escaped && ch == "\\";
      }
      return "variable-2";
    };
  }


  function pushContext(state, type, col) {
    state.context = {prev: state.context, indent: state.indent, col: col, type: type};
  }
  function popContext(state) {
    state.indent = state.context.indent;
    state.context = state.context.prev;
  }

  return {
    startState: function(base) {
      return {tokenize: tokenBase,
              context: null,
              indent: 0,
              col: 0};
    },

    token: function(stream, state) {
      if (stream.sol()) {
        if (state.context && state.context.align == null) state.context.align = false;
        state.indent = stream.indentation();
      }
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);

      if (style != "comment" && state.context && state.context.align == null && state.context.type != "pattern") {
        state.context.align = true;
      }

      if (curPunc == "(") pushContext(state, ")", stream.column());
      else if (curPunc == "[") pushContext(state, "]", stream.column());
      else if (curPunc == "{") pushContext(state, "}", stream.column());
      else if (/[\]\}\)]/.test(curPunc)) {
        while (state.context && state.context.type == "pattern") popContext(state);
        if (state.context && curPunc == state.context.type) popContext(state);
      }
      else if (curPunc == "." && state.context && state.context.type == "pattern") popContext(state);
      else if (/atom|string|variable/.test(style) && state.context) {
        if (/[\}\]]/.test(state.context.type))
          pushContext(state, "pattern", stream.column());
        else if (state.context.type == "pattern" && !state.context.align) {
          state.context.align = true;
          state.context.col = stream.column();
        }
      }

      return style;
    },

    indent: function(state, textAfter) {
      var firstChar = textAfter && textAfter.charAt(0);
      var context = state.context;
      if (/[\]\}]/.test(firstChar))
        while (context && context.type == "pattern") context = context.prev;

      var closing = context && firstChar == context.type;
      if (!context)
        return 0;
      else if (context.type == "pattern")
        return context.col;
      else if (context.align)
        return context.col + (closing ? 0 : 1);
      else
        return context.indent + (closing ? 0 : indentUnit);
    }
  };
});

CodeMirror.defineMIME("text/x-mysql", "mysql");
/**********************************************************
* This script provides syntax highlighting support for 
* the Ntriples format.
* Ntriples format specification: 
*     http://www.w3.org/TR/rdf-testcases/#ntriples
***********************************************************/

/* 
    The following expression defines the defined ASF grammar transitions.

    pre_subject ->
        {
        ( writing_subject_uri | writing_bnode_uri )
            -> pre_predicate 
                -> writing_predicate_uri 
                    -> pre_object 
                        -> writing_object_uri | writing_object_bnode | 
                          ( 
                            writing_object_literal 
                                -> writing_literal_lang | writing_literal_type
                          )
                            -> post_object
                                -> BEGIN
         } otherwise {
             -> ERROR
         }
*/
CodeMirror.defineMode("ntriples", function() {  

  Location = {
    PRE_SUBJECT         : 0,
    WRITING_SUB_URI     : 1,
    WRITING_BNODE_URI   : 2,
    PRE_PRED            : 3,
    WRITING_PRED_URI    : 4,
    PRE_OBJ             : 5,
    WRITING_OBJ_URI     : 6,
    WRITING_OBJ_BNODE   : 7,
    WRITING_OBJ_LITERAL : 8,
    WRITING_LIT_LANG    : 9,
    WRITING_LIT_TYPE    : 10,
    POST_OBJ            : 11,
    ERROR               : 12
  };
  function transitState(currState, c) {
    var currLocation = currState.location;
    var ret;
    
    // Opening.
    if     (currLocation == Location.PRE_SUBJECT && c == '<') ret = Location.WRITING_SUB_URI;
    else if(currLocation == Location.PRE_SUBJECT && c == '_') ret = Location.WRITING_BNODE_URI;
    else if(currLocation == Location.PRE_PRED    && c == '<') ret = Location.WRITING_PRED_URI;
    else if(currLocation == Location.PRE_OBJ     && c == '<') ret = Location.WRITING_OBJ_URI;
    else if(currLocation == Location.PRE_OBJ     && c == '_') ret = Location.WRITING_OBJ_BNODE;
    else if(currLocation == Location.PRE_OBJ     && c == '"') ret = Location.WRITING_OBJ_LITERAL;
    
    // Closing.
    else if(currLocation == Location.WRITING_SUB_URI     && c == '>') ret = Location.PRE_PRED;
    else if(currLocation == Location.WRITING_BNODE_URI   && c == ' ') ret = Location.PRE_PRED;
    else if(currLocation == Location.WRITING_PRED_URI    && c == '>') ret = Location.PRE_OBJ;
    else if(currLocation == Location.WRITING_OBJ_URI     && c == '>') ret = Location.POST_OBJ;
    else if(currLocation == Location.WRITING_OBJ_BNODE   && c == ' ') ret = Location.POST_OBJ;
    else if(currLocation == Location.WRITING_OBJ_LITERAL && c == '"') ret = Location.POST_OBJ;
    else if(currLocation == Location.WRITING_LIT_LANG && c == ' ') ret = Location.POST_OBJ;
    else if(currLocation == Location.WRITING_LIT_TYPE && c == '>') ret = Location.POST_OBJ;
    
    // Closing typed and language literal.
    else if(currLocation == Location.WRITING_OBJ_LITERAL && c == '@') ret = Location.WRITING_LIT_LANG;
    else if(currLocation == Location.WRITING_OBJ_LITERAL && c == '^') ret = Location.WRITING_LIT_TYPE;

    // Spaces.
    else if( c == ' ' &&                             
             (
               currLocation == Location.PRE_SUBJECT || 
               currLocation == Location.PRE_PRED    || 
               currLocation == Location.PRE_OBJ     || 
               currLocation == Location.POST_OBJ
             )
           ) ret = currLocation;
    
    // Reset.
    else if(currLocation == Location.POST_OBJ && c == '.') ret = Location.PRE_SUBJECT;    
    
    // Error
    else ret = Location.ERROR;
    
    currState.location=ret;
  }

  untilSpace  = function(c) { return c != ' '; };
  untilEndURI = function(c) { return c != '>'; };
  return {
    startState: function() {
       return { 
           location : Location.PRE_SUBJECT,
           uris     : [],
           anchors  : [],
           bnodes   : [],
           langs    : [],
           types    : []
       };
    },
    token: function(stream, state) {
      var ch = stream.next();
      if(ch == '<') {
         transitState(state, ch);
         var parsedURI = '';
         stream.eatWhile( function(c) { if( c != '#' && c != '>' ) { parsedURI += c; return true; } return false;} );
         state.uris.push(parsedURI);
         if( stream.match('#', false) ) return 'variable';
         stream.next();
         transitState(state, '>');
         return 'variable';
      }
      if(ch == '#') {
        var parsedAnchor = '';
        stream.eatWhile(function(c) { if(c != '>' && c != ' ') { parsedAnchor+= c; return true; } return false});
        state.anchors.push(parsedAnchor);
        return 'variable-2';
      }
      if(ch == '>') {
          transitState(state, '>');
          return 'variable';
      }
      if(ch == '_') {
          transitState(state, ch);
          var parsedBNode = '';
          stream.eatWhile(function(c) { if( c != ' ' ) { parsedBNode += c; return true; } return false;});
          state.bnodes.push(parsedBNode);
          stream.next();
          transitState(state, ' ');
          return 'builtin';
      }
      if(ch == '"') {
          transitState(state, ch);
          stream.eatWhile( function(c) { return c != '"'; } );
          stream.next();
          if( stream.peek() != '@' && stream.peek() != '^' ) {
              transitState(state, '"');
          }
          return 'string';
      }
      if( ch == '@' ) {
          transitState(state, '@');
          var parsedLang = '';
          stream.eatWhile(function(c) { if( c != ' ' ) { parsedLang += c; return true; } return false;});
          state.langs.push(parsedLang);
          stream.next();
          transitState(state, ' ');
          return 'string-2';
      }
      if( ch == '^' ) {
          stream.next();
          transitState(state, '^');
          var parsedType = '';
          stream.eatWhile(function(c) { if( c != '>' ) { parsedType += c; return true; } return false;} );
          state.types.push(parsedType);
          stream.next();
          transitState(state, '>');
          return 'variable';
      }
      if( ch == ' ' ) {
          transitState(state, ch);
      }
      if( ch == '.' ) {
          transitState(state, ch);
      }
    }
  };
});

CodeMirror.defineMIME("text/n-triples", "ntriples");
CodeMirror.defineMode("pascal", function(config) {
  function words(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }
  var keywords = words("and array begin case const div do downto else end file for forward integer " +
                       "boolean char function goto if in label mod nil not of or packed procedure " +
                       "program record repeat set string then to type until var while with");
  var atoms = {"null": true};

  var isOperatorChar = /[+\-*&%=<>!?|\/]/;

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (ch == "#" && state.startOfLine) {
      stream.skipToEnd();
      return "meta";
    }
    if (ch == '"' || ch == "'") {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    if (ch == "(" && stream.eat("*")) {
      state.tokenize = tokenComment;
      return tokenComment(stream, state);
    }
    if (/[\[\]{}\(\),;\:\.]/.test(ch)) {
      return null
    }
    if (/\d/.test(ch)) {
      stream.eatWhile(/[\w\.]/);
      return "number";
    }
    if (ch == "/") {
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
    }
    if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return "operator";
    }
    stream.eatWhile(/[\w\$_]/);
    var cur = stream.current();
    if (keywords.propertyIsEnumerable(cur)) return "keyword";
    if (atoms.propertyIsEnumerable(cur)) return "atom";
    return "word";
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !escaped) state.tokenize = null;
      return "string";
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == ")" && maybeEnd) {
        state.tokenize = null;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {tokenize: null};
    },

    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      var style = (state.tokenize || tokenBase)(stream, state);
      if (style == "comment" || style == "meta") return style;
      return style;
    },

    electricChars: "{}"
  };
});

CodeMirror.defineMIME("text/x-pascal", "pascal");
// CodeMirror2 mode/perl/perl.js (text/x-perl) beta 0.10 (2011-11-08)
// This is a part of CodeMirror from https://github.com/sabaca/CodeMirror_mode_perl (mail@sabaca.com)
CodeMirror.defineMode("perl",function(config,parserConfig){
	// http://perldoc.perl.org
	var PERL={				    	//   null - magic touch
							//   1 - keyword
							//   2 - def
							//   3 - atom
							//   4 - operator
							//   5 - variable-2 (predefined)
							//   [x,y] - x=1,2,3; y=must be defined if x{...}
						//	PERL operators
		'->'				:   4,
		'++'				:   4,
		'--'				:   4,
		'**'				:   4,
							//   ! ~ \ and unary + and -
		'=~'				:   4,
		'!~'				:   4,
		'*'				:   4,
		'/'				:   4,
		'%'				:   4,
		'x'				:   4,
		'+'				:   4,
		'-'				:   4,
		'.'				:   4,
		'<<'				:   4,
		'>>'				:   4,
							//   named unary operators
		'<'				:   4,
		'>'				:   4,
		'<='				:   4,
		'>='				:   4,
		'lt'				:   4,
		'gt'				:   4,
		'le'				:   4,
		'ge'				:   4,
		'=='				:   4,
		'!='				:   4,
		'<=>'				:   4,
		'eq'				:   4,
		'ne'				:   4,
		'cmp'				:   4,
		'~~'				:   4,
		'&'				:   4,
		'|'				:   4,
		'^'				:   4,
		'&&'				:   4,
		'||'				:   4,
		'//'				:   4,
		'..'				:   4,
		'...'				:   4,
		'?'				:   4,
		':'				:   4,
		'='				:   4,
		'+='				:   4,
		'-='				:   4,
		'*='				:   4,	//   etc. ???
		','				:   4,
		'=>'				:   4,
		'::'				:   4,
				   			//   list operators (rightward)
		'not'				:   4,
		'and'				:   4,
		'or'				:   4,
		'xor'				:   4,
						//	PERL predefined variables (I know, what this is a paranoid idea, but may be needed for people, who learn PERL, and for me as well, ...and may be for you?;)
		'BEGIN'				:   [5,1],
		'END'				:   [5,1],
		'PRINT'				:   [5,1],
		'PRINTF'			:   [5,1],
		'GETC'				:   [5,1],
		'READ'				:   [5,1],
		'READLINE'			:   [5,1],
		'DESTROY'			:   [5,1],
		'TIE'				:   [5,1],
		'TIEHANDLE'			:   [5,1],
		'UNTIE'				:   [5,1],
		'STDIN'				:    5,
		'STDIN_TOP'			:    5,
		'STDOUT'			:    5,
		'STDOUT_TOP'			:    5,
		'STDERR'			:    5,
		'STDERR_TOP'			:    5,
		'$ARG'				:    5,
		'$_'				:    5,
		'@ARG'				:    5,
		'@_'				:    5,
		'$LIST_SEPARATOR'		:    5,
		'$"'				:    5,
		'$PROCESS_ID'			:    5,
		'$PID'				:    5,
		'$$'				:    5,
		'$REAL_GROUP_ID'		:    5,
		'$GID'				:    5,
		'$('				:    5,
		'$EFFECTIVE_GROUP_ID'		:    5,
		'$EGID'				:    5,
		'$)'				:    5,
		'$PROGRAM_NAME'			:    5,
		'$0'				:    5,
		'$SUBSCRIPT_SEPARATOR'		:    5,
		'$SUBSEP'			:    5,
		'$;'				:    5,
		'$REAL_USER_ID'			:    5,
		'$UID'				:    5,
		'$<'				:    5,
		'$EFFECTIVE_USER_ID'		:    5,
		'$EUID'				:    5,
		'$>'				:    5,
		'$a'				:    5,
		'$b'				:    5,
		'$COMPILING'			:    5,
		'$^C'				:    5,
		'$DEBUGGING'			:    5,
		'$^D'				:    5,
		'${^ENCODING}'			:    5,
		'$ENV'				:    5,
		'%ENV'				:    5,
		'$SYSTEM_FD_MAX'		:    5,
		'$^F'				:    5,
		'@F'				:    5,
		'${^GLOBAL_PHASE}'		:    5,
		'$^H'				:    5,
		'%^H'				:    5,
		'@INC'				:    5,
		'%INC'				:    5,
		'$INPLACE_EDIT'			:    5,
		'$^I'				:    5,
		'$^M'				:    5,
		'$OSNAME'			:    5,
		'$^O'				:    5,
		'${^OPEN}'			:    5,
		'$PERLDB'			:    5,
		'$^P'				:    5,
		'$SIG'				:    5,
		'%SIG'				:    5,
		'$BASETIME'			:    5,
		'$^T'				:    5,
		'${^TAINT}'			:    5,
		'${^UNICODE}'			:    5,
		'${^UTF8CACHE}'			:    5,
		'${^UTF8LOCALE}'		:    5,
		'$PERL_VERSION'			:    5,
		'$^V'				:    5,
		'${^WIN32_SLOPPY_STAT}'		:    5,
		'$EXECUTABLE_NAME'		:    5,
		'$^X'				:    5,
		'$1'				:    5,	// - regexp $1, $2...
		'$MATCH'			:    5,
		'$&'				:    5,
		'${^MATCH}'			:    5,
		'$PREMATCH'			:    5,
		'$`'				:    5,
		'${^PREMATCH}'			:    5,
		'$POSTMATCH'			:    5,
		"$'"				:    5,
		'${^POSTMATCH}'			:    5,
		'$LAST_PAREN_MATCH'		:    5,
		'$+'				:    5,
		'$LAST_SUBMATCH_RESULT'		:    5,
		'$^N'				:    5,
		'@LAST_MATCH_END'		:    5,
		'@+'				:    5,
		'%LAST_PAREN_MATCH'		:    5,
		'%+'				:    5,
		'@LAST_MATCH_START'		:    5,
		'@-'				:    5,
		'%LAST_MATCH_START'		:    5,
		'%-'				:    5,
		'$LAST_REGEXP_CODE_RESULT'	:    5,
		'$^R'				:    5,
		'${^RE_DEBUG_FLAGS}'		:    5,
		'${^RE_TRIE_MAXBUF}'		:    5,
		'$ARGV'				:    5,
		'@ARGV'				:    5,
		'ARGV'				:    5,
		'ARGVOUT'			:    5,
		'$OUTPUT_FIELD_SEPARATOR'	:    5,
		'$OFS'				:    5,
		'$,'				:    5,
		'$INPUT_LINE_NUMBER'		:    5,
		'$NR'				:    5,
		'$.'				:    5,
		'$INPUT_RECORD_SEPARATOR'	:    5,
		'$RS'				:    5,
		'$/'				:    5,
		'$OUTPUT_RECORD_SEPARATOR'	:    5,
		'$ORS'				:    5,
		'$\\'				:    5,
		'$OUTPUT_AUTOFLUSH'		:    5,
		'$|'				:    5,
		'$ACCUMULATOR'			:    5,
		'$^A'				:    5,
		'$FORMAT_FORMFEED'		:    5,
		'$^L'				:    5,
		'$FORMAT_PAGE_NUMBER'		:    5,
		'$%'				:    5,
		'$FORMAT_LINES_LEFT'		:    5,
		'$-'				:    5,
		'$FORMAT_LINE_BREAK_CHARACTERS'	:    5,
		'$:'				:    5,
		'$FORMAT_LINES_PER_PAGE'	:    5,
		'$='				:    5,
		'$FORMAT_TOP_NAME'		:    5,
		'$^'				:    5,
		'$FORMAT_NAME'			:    5,
		'$~'				:    5,
		'${^CHILD_ERROR_NATIVE}'	:    5,
		'$EXTENDED_OS_ERROR'		:    5,
		'$^E'				:    5,
		'$EXCEPTIONS_BEING_CAUGHT'	:    5,
		'$^S'				:    5,
		'$WARNING'			:    5,
		'$^W'				:    5,
		'${^WARNING_BITS}'		:    5,
		'$OS_ERROR'			:    5,
		'$ERRNO'			:    5,
		'$!'				:    5,
		'%OS_ERROR'			:    5,
		'%ERRNO'			:    5,
		'%!'				:    5,
		'$CHILD_ERROR'			:    5,
		'$?'				:    5,
		'$EVAL_ERROR'			:    5,
		'$@'				:    5,
		'$OFMT'				:    5,
		'$#'				:    5,
		'$*'				:    5,
		'$ARRAY_BASE'			:    5,
		'$['				:    5,
		'$OLD_PERL_VERSION'		:    5,
		'$]'				:    5,
						//	PERL blocks
		'if'				:[1,1],
		elsif				:[1,1],
		'else'				:[1,1],
		'while'				:[1,1],
		unless				:[1,1],
		'for'				:[1,1],
		foreach				:[1,1],
						//	PERL functions
		'abs'				:1,	// - absolute value function
		accept				:1,	// - accept an incoming socket connect
		alarm				:1,	// - schedule a SIGALRM
		'atan2'				:1,	// - arctangent of Y/X in the range -PI to PI
		bind				:1,	// - binds an address to a socket
		binmode				:1,	// - prepare binary files for I/O
		bless				:1,	// - create an object
		bootstrap			:1,	//
		'break'				:1,	// - break out of a "given" block
		caller				:1,	// - get context of the current subroutine call
		chdir				:1,	// - change your current working directory
		chmod				:1,	// - changes the permissions on a list of files
		chomp				:1,	// - remove a trailing record separator from a string
		chop				:1,	// - remove the last character from a string
		chown				:1,	// - change the owership on a list of files
		chr				:1,	// - get character this number represents
		chroot				:1,	// - make directory new root for path lookups
		close				:1,	// - close file (or pipe or socket) handle
		closedir			:1,	// - close directory handle
		connect				:1,	// - connect to a remote socket
		'continue'			:[1,1],	// - optional trailing block in a while or foreach
		'cos'				:1,	// - cosine function
		crypt				:1,	// - one-way passwd-style encryption
		dbmclose			:1,	// - breaks binding on a tied dbm file
		dbmopen				:1,	// - create binding on a tied dbm file
		'default'			:1,	//
		defined				:1,	// - test whether a value, variable, or function is defined
		'delete'			:1,	// - deletes a value from a hash
		die				:1,	// - raise an exception or bail out
		'do'				:1,	// - turn a BLOCK into a TERM
		dump				:1,	// - create an immediate core dump
		each				:1,	// - retrieve the next key/value pair from a hash
		endgrent			:1,	// - be done using group file
		endhostent			:1,	// - be done using hosts file
		endnetent			:1,	// - be done using networks file
		endprotoent			:1,	// - be done using protocols file
		endpwent			:1,	// - be done using passwd file
		endservent			:1,	// - be done using services file
		eof				:1,	// - test a filehandle for its end
		'eval'				:1,	// - catch exceptions or compile and run code
		'exec'				:1,	// - abandon this program to run another
		exists				:1,	// - test whether a hash key is present
		exit				:1,	// - terminate this program
		'exp'				:1,	// - raise I to a power
		fcntl				:1,	// - file control system call
		fileno				:1,	// - return file descriptor from filehandle
		flock				:1,	// - lock an entire file with an advisory lock
		fork				:1,	// - create a new process just like this one
		format				:1,	// - declare a picture format with use by the write() function
		formline			:1,	// - internal function used for formats
		getc				:1,	// - get the next character from the filehandle
		getgrent			:1,	// - get next group record
		getgrgid			:1,	// - get group record given group user ID
		getgrnam			:1,	// - get group record given group name
		gethostbyaddr			:1,	// - get host record given its address
		gethostbyname			:1,	// - get host record given name
		gethostent			:1,	// - get next hosts record
		getlogin			:1,	// - return who logged in at this tty
		getnetbyaddr			:1,	// - get network record given its address
		getnetbyname			:1,	// - get networks record given name
		getnetent			:1,	// - get next networks record
		getpeername			:1,	// - find the other end of a socket connection
		getpgrp				:1,	// - get process group
		getppid				:1,	// - get parent process ID
		getpriority			:1,	// - get current nice value
		getprotobyname			:1,	// - get protocol record given name
		getprotobynumber		:1,	// - get protocol record numeric protocol
		getprotoent			:1,	// - get next protocols record
		getpwent			:1,	// - get next passwd record
		getpwnam			:1,	// - get passwd record given user login name
		getpwuid			:1,	// - get passwd record given user ID
		getservbyname			:1,	// - get services record given its name
		getservbyport			:1,	// - get services record given numeric port
		getservent			:1,	// - get next services record
		getsockname			:1,	// - retrieve the sockaddr for a given socket
		getsockopt			:1,	// - get socket options on a given socket
		given				:1,	//
		glob				:1,	// - expand filenames using wildcards
		gmtime				:1,	// - convert UNIX time into record or string using Greenwich time
		'goto'				:1,	// - create spaghetti code
		grep				:1,	// - locate elements in a list test true against a given criterion
		hex				:1,	// - convert a string to a hexadecimal number
		'import'			:1,	// - patch a module's namespace into your own
		index				:1,	// - find a substring within a string
		'int'				:1,	// - get the integer portion of a number
		ioctl				:1,	// - system-dependent device control system call
		'join'				:1,	// - join a list into a string using a separator
		keys				:1,	// - retrieve list of indices from a hash
		kill				:1,	// - send a signal to a process or process group
		last				:1,	// - exit a block prematurely
		lc				:1,	// - return lower-case version of a string
		lcfirst				:1,	// - return a string with just the next letter in lower case
		length				:1,	// - return the number of bytes in a string
		'link'				:1,	// - create a hard link in the filesytem
		listen				:1,	// - register your socket as a server
		local				: 2,	// - create a temporary value for a global variable (dynamic scoping)
		localtime			:1,	// - convert UNIX time into record or string using local time
		lock				:1,	// - get a thread lock on a variable, subroutine, or method
		'log'				:1,	// - retrieve the natural logarithm for a number
		lstat				:1,	// - stat a symbolic link
		m				:null,	// - match a string with a regular expression pattern
		map				:1,	// - apply a change to a list to get back a new list with the changes
		mkdir				:1,	// - create a directory
		msgctl				:1,	// - SysV IPC message control operations
		msgget				:1,	// - get SysV IPC message queue
		msgrcv				:1,	// - receive a SysV IPC message from a message queue
		msgsnd				:1,	// - send a SysV IPC message to a message queue
		my				: 2,	// - declare and assign a local variable (lexical scoping)
		'new'				:1,	//
		next				:1,	// - iterate a block prematurely
		no				:1,	// - unimport some module symbols or semantics at compile time
		oct				:1,	// - convert a string to an octal number
		open				:1,	// - open a file, pipe, or descriptor
		opendir				:1,	// - open a directory
		ord				:1,	// - find a character's numeric representation
		our				: 2,	// - declare and assign a package variable (lexical scoping)
		pack				:1,	// - convert a list into a binary representation
		'package'			:1,	// - declare a separate global namespace
		pipe				:1,	// - open a pair of connected filehandles
		pop				:1,	// - remove the last element from an array and return it
		pos				:1,	// - find or set the offset for the last/next m//g search
		print				:1,	// - output a list to a filehandle
		printf				:1,	// - output a formatted list to a filehandle
		prototype			:1,	// - get the prototype (if any) of a subroutine
		push				:1,	// - append one or more elements to an array
		q				:null,	// - singly quote a string
		qq				:null,	// - doubly quote a string
		qr				:null,	// - Compile pattern
		quotemeta			:null,	// - quote regular expression magic characters
		qw				:null,	// - quote a list of words
		qx				:null,	// - backquote quote a string
		rand				:1,	// - retrieve the next pseudorandom number
		read				:1,	// - fixed-length buffered input from a filehandle
		readdir				:1,	// - get a directory from a directory handle
		readline			:1,	// - fetch a record from a file
		readlink			:1,	// - determine where a symbolic link is pointing
		readpipe			:1,	// - execute a system command and collect standard output
		recv				:1,	// - receive a message over a Socket
		redo				:1,	// - start this loop iteration over again
		ref				:1,	// - find out the type of thing being referenced
		rename				:1,	// - change a filename
		require				:1,	// - load in external functions from a library at runtime
		reset				:1,	// - clear all variables of a given name
		'return'			:1,	// - get out of a function early
		reverse				:1,	// - flip a string or a list
		rewinddir			:1,	// - reset directory handle
		rindex				:1,	// - right-to-left substring search
		rmdir				:1,	// - remove a directory
		s				:null,	// - replace a pattern with a string
		say				:1,	// - print with newline
		scalar				:1,	// - force a scalar context
		seek				:1,	// - reposition file pointer for random-access I/O
		seekdir				:1,	// - reposition directory pointer
		select				:1,	// - reset default output or do I/O multiplexing
		semctl				:1,	// - SysV semaphore control operations
		semget				:1,	// - get set of SysV semaphores
		semop				:1,	// - SysV semaphore operations
		send				:1,	// - send a message over a socket
		setgrent			:1,	// - prepare group file for use
		sethostent			:1,	// - prepare hosts file for use
		setnetent			:1,	// - prepare networks file for use
		setpgrp				:1,	// - set the process group of a process
		setpriority			:1,	// - set a process's nice value
		setprotoent			:1,	// - prepare protocols file for use
		setpwent			:1,	// - prepare passwd file for use
		setservent			:1,	// - prepare services file for use
		setsockopt			:1,	// - set some socket options
		shift				:1,	// - remove the first element of an array, and return it
		shmctl				:1,	// - SysV shared memory operations
		shmget				:1,	// - get SysV shared memory segment identifier
		shmread				:1,	// - read SysV shared memory
		shmwrite			:1,	// - write SysV shared memory
		shutdown			:1,	// - close down just half of a socket connection
		'sin'				:1,	// - return the sine of a number
		sleep				:1,	// - block for some number of seconds
		socket				:1,	// - create a socket
		socketpair			:1,	// - create a pair of sockets
		'sort'				:1,	// - sort a list of values
		splice				:1,	// - add or remove elements anywhere in an array
		'split'				:1,	// - split up a string using a regexp delimiter
		sprintf				:1,	// - formatted print into a string
		'sqrt'				:1,	// - square root function
		srand				:1,	// - seed the random number generator
		stat				:1,	// - get a file's status information
		state				:1,	// - declare and assign a state variable (persistent lexical scoping)
		study				:1,	// - optimize input data for repeated searches
		'sub'				:1,	// - declare a subroutine, possibly anonymously
		'substr'			:1,	// - get or alter a portion of a stirng
		symlink				:1,	// - create a symbolic link to a file
		syscall				:1,	// - execute an arbitrary system call
		sysopen				:1,	// - open a file, pipe, or descriptor
		sysread				:1,	// - fixed-length unbuffered input from a filehandle
		sysseek				:1,	// - position I/O pointer on handle used with sysread and syswrite
		system				:1,	// - run a separate program
		syswrite			:1,	// - fixed-length unbuffered output to a filehandle
		tell				:1,	// - get current seekpointer on a filehandle
		telldir				:1,	// - get current seekpointer on a directory handle
		tie				:1,	// - bind a variable to an object class
		tied				:1,	// - get a reference to the object underlying a tied variable
		time				:1,	// - return number of seconds since 1970
		times				:1,	// - return elapsed time for self and child processes
		tr				:null,	// - transliterate a string
		truncate			:1,	// - shorten a file
		uc				:1,	// - return upper-case version of a string
		ucfirst				:1,	// - return a string with just the next letter in upper case
		umask				:1,	// - set file creation mode mask
		undef				:1,	// - remove a variable or function definition
		unlink				:1,	// - remove one link to a file
		unpack				:1,	// - convert binary structure into normal perl variables
		unshift				:1,	// - prepend more elements to the beginning of a list
		untie				:1,	// - break a tie binding to a variable
		use				:1,	// - load in a module at compile time
		utime				:1,	// - set a file's last access and modify times
		values				:1,	// - return a list of the values in a hash
		vec				:1,	// - test or set particular bits in a string
		wait				:1,	// - wait for any child process to die
		waitpid				:1,	// - wait for a particular child process to die
		wantarray			:1,	// - get void vs scalar vs list context of current subroutine call
		warn				:1,	// - print debugging info
		when				:1,	//
		write				:1,	// - print a picture record
		y				:null};	// - transliterate a string

	var RXstyle="string-2";
	var RXmodifiers=/[goseximacplud]/;		// NOTE: "m", "s", "y" and "tr" need to correct real modifiers for each regexp type

	function tokenChain(stream,state,chain,style,tail){	// NOTE: chain.length > 2 is not working now (it's for s[...][...]geos;)
		state.chain=null;                               //                                                          12   3tail
		state.style=null;
		state.tail=null;
		state.tokenize=function(stream,state){
			var e=false,c,i=0;
			while(c=stream.next()){
				if(c===chain[i]&&!e){
					if(chain[++i]!==undefined){
						state.chain=chain[i];
						state.style=style;
						state.tail=tail}
					else if(tail)
						stream.eatWhile(tail);
					state.tokenize=tokenPerl;
					return style}
				e=!e&&c=="\\"}
			return style};
		return state.tokenize(stream,state)}

	function tokenSOMETHING(stream,state,string){
		state.tokenize=function(stream,state){
			if(stream.string==string)
				state.tokenize=tokenPerl;
			stream.skipToEnd();
			return "string"};
		return state.tokenize(stream,state)}

	function tokenPerl(stream,state){
		if(stream.eatSpace())
			return null;
		if(state.chain)
			return tokenChain(stream,state,state.chain,state.style,state.tail);
		if(stream.match(/^\-?[\d\.]/,false))
			if(stream.match(/^(\-?(\d*\.\d+(e[+-]?\d+)?|\d+\.\d*)|0x[\da-fA-F]+|0b[01]+|\d+(e[+-]?\d+)?)/))
				return 'number';
		if(stream.match(/^<<(?=\w)/)){			// NOTE: <<SOMETHING\n...\nSOMETHING\n
			stream.eatWhile(/\w/);
			return tokenSOMETHING(stream,state,stream.current().substr(2))}
		if(stream.sol()&&stream.match(/^\=item(?!\w)/)){// NOTE: \n=item...\n=cut\n
			return tokenSOMETHING(stream,state,'=cut')}
		var ch=stream.next();
		if(ch=='"'||ch=="'"){				// NOTE: ' or " or <<'SOMETHING'\n...\nSOMETHING\n or <<"SOMETHING"\n...\nSOMETHING\n
			if(stream.prefix(3)=="<<"+ch){
				var p=stream.pos;
				stream.eatWhile(/\w/);
				var n=stream.current().substr(1);
				if(n&&stream.eat(ch))
					return tokenSOMETHING(stream,state,n);
				stream.pos=p}
			return tokenChain(stream,state,[ch],"string")}
		if(ch=="q"){
			var c=stream.look(-2);
			if(!(c&&/\w/.test(c))){
				c=stream.look(0);
				if(c=="x"){
					c=stream.look(1);
					if(c=="("){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[")"],RXstyle,RXmodifiers)}
					if(c=="["){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["]"],RXstyle,RXmodifiers)}
					if(c=="{"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["}"],RXstyle,RXmodifiers)}
					if(c=="<"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[">"],RXstyle,RXmodifiers)}
					if(/[\^'"!~\/]/.test(c)){
						stream.eatSuffix(1);
						return tokenChain(stream,state,[stream.eat(c)],RXstyle,RXmodifiers)}}
				else if(c=="q"){
					c=stream.look(1);
					if(c=="("){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[")"],"string")}
					if(c=="["){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["]"],"string")}
					if(c=="{"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["}"],"string")}
					if(c=="<"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[">"],"string")}
					if(/[\^'"!~\/]/.test(c)){
						stream.eatSuffix(1);
						return tokenChain(stream,state,[stream.eat(c)],"string")}}
				else if(c=="w"){
					c=stream.look(1);
					if(c=="("){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[")"],"bracket")}
					if(c=="["){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["]"],"bracket")}
					if(c=="{"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["}"],"bracket")}
					if(c=="<"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[">"],"bracket")}
					if(/[\^'"!~\/]/.test(c)){
						stream.eatSuffix(1);
						return tokenChain(stream,state,[stream.eat(c)],"bracket")}}
				else if(c=="r"){
					c=stream.look(1);
					if(c=="("){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[")"],RXstyle,RXmodifiers)}
					if(c=="["){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["]"],RXstyle,RXmodifiers)}
					if(c=="{"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,["}"],RXstyle,RXmodifiers)}
					if(c=="<"){
						stream.eatSuffix(2);
						return tokenChain(stream,state,[">"],RXstyle,RXmodifiers)}
					if(/[\^'"!~\/]/.test(c)){
						stream.eatSuffix(1);
						return tokenChain(stream,state,[stream.eat(c)],RXstyle,RXmodifiers)}}
				else if(/[\^'"!~\/(\[{<]/.test(c)){
					if(c=="("){
						stream.eatSuffix(1);
						return tokenChain(stream,state,[")"],"string")}
					if(c=="["){
						stream.eatSuffix(1);
						return tokenChain(stream,state,["]"],"string")}
					if(c=="{"){
						stream.eatSuffix(1);
						return tokenChain(stream,state,["}"],"string")}
					if(c=="<"){
						stream.eatSuffix(1);
						return tokenChain(stream,state,[">"],"string")}
					if(/[\^'"!~\/]/.test(c)){
						return tokenChain(stream,state,[stream.eat(c)],"string")}}}}
		if(ch=="m"){
			var c=stream.look(-2);
			if(!(c&&/\w/.test(c))){
				c=stream.eat(/[(\[{<\^'"!~\/]/);
				if(c){
					if(/[\^'"!~\/]/.test(c)){
						return tokenChain(stream,state,[c],RXstyle,RXmodifiers)}
					if(c=="("){
						return tokenChain(stream,state,[")"],RXstyle,RXmodifiers)}
					if(c=="["){
						return tokenChain(stream,state,["]"],RXstyle,RXmodifiers)}
					if(c=="{"){
						return tokenChain(stream,state,["}"],RXstyle,RXmodifiers)}
					if(c=="<"){
						return tokenChain(stream,state,[">"],RXstyle,RXmodifiers)}}}}
		if(ch=="s"){
			var c=/[\/>\]})\w]/.test(stream.look(-2));
			if(!c){
				c=stream.eat(/[(\[{<\^'"!~\/]/);
				if(c){
					if(c=="[")
						return tokenChain(stream,state,["]","]"],RXstyle,RXmodifiers);
					if(c=="{")
						return tokenChain(stream,state,["}","}"],RXstyle,RXmodifiers);
					if(c=="<")
						return tokenChain(stream,state,[">",">"],RXstyle,RXmodifiers);
					if(c=="(")
						return tokenChain(stream,state,[")",")"],RXstyle,RXmodifiers);
					return tokenChain(stream,state,[c,c],RXstyle,RXmodifiers)}}}
		if(ch=="y"){
			var c=/[\/>\]})\w]/.test(stream.look(-2));
			if(!c){
				c=stream.eat(/[(\[{<\^'"!~\/]/);
				if(c){
					if(c=="[")
						return tokenChain(stream,state,["]","]"],RXstyle,RXmodifiers);
					if(c=="{")
						return tokenChain(stream,state,["}","}"],RXstyle,RXmodifiers);
					if(c=="<")
						return tokenChain(stream,state,[">",">"],RXstyle,RXmodifiers);
					if(c=="(")
						return tokenChain(stream,state,[")",")"],RXstyle,RXmodifiers);
					return tokenChain(stream,state,[c,c],RXstyle,RXmodifiers)}}}
		if(ch=="t"){
			var c=/[\/>\]})\w]/.test(stream.look(-2));
			if(!c){
				c=stream.eat("r");if(c){
				c=stream.eat(/[(\[{<\^'"!~\/]/);
				if(c){
					if(c=="[")
						return tokenChain(stream,state,["]","]"],RXstyle,RXmodifiers);
					if(c=="{")
						return tokenChain(stream,state,["}","}"],RXstyle,RXmodifiers);
					if(c=="<")
						return tokenChain(stream,state,[">",">"],RXstyle,RXmodifiers);
					if(c=="(")
						return tokenChain(stream,state,[")",")"],RXstyle,RXmodifiers);
					return tokenChain(stream,state,[c,c],RXstyle,RXmodifiers)}}}}
		if(ch=="`"){
			return tokenChain(stream,state,[ch],"variable-2")}
		if(ch=="/"){
			if(!/~\s*$/.test(stream.prefix()))
				return "operator";
			else
				return tokenChain(stream,state,[ch],RXstyle,RXmodifiers)}
		if(ch=="$"){
			var p=stream.pos;
			if(stream.eatWhile(/\d/)||stream.eat("{")&&stream.eatWhile(/\d/)&&stream.eat("}"))
				return "variable-2";
			else
				stream.pos=p}
		if(/[$@%]/.test(ch)){
			var p=stream.pos;
			if(stream.eat("^")&&stream.eat(/[A-Z]/)||!/[@$%&]/.test(stream.look(-2))&&stream.eat(/[=|\\\-#?@;:&`~\^!\[\]*'"$+.,\/<>()]/)){
				var c=stream.current();
				if(PERL[c])
					return "variable-2"}
			stream.pos=p}
		if(/[$@%&]/.test(ch)){
			if(stream.eatWhile(/[\w$\[\]]/)||stream.eat("{")&&stream.eatWhile(/[\w$\[\]]/)&&stream.eat("}")){
				var c=stream.current();
				if(PERL[c])
					return "variable-2";
				else
					return "variable"}}
		if(ch=="#"){
			if(stream.look(-2)!="$"){
				stream.skipToEnd();
				return "comment"}}
		if(/[:+\-\^*$&%@=<>!?|\/~\.]/.test(ch)){
			var p=stream.pos;
			stream.eatWhile(/[:+\-\^*$&%@=<>!?|\/~\.]/);
			if(PERL[stream.current()])
				return "operator";
			else
				stream.pos=p}
		if(ch=="_"){
			if(stream.pos==1){
				if(stream.suffix(6)=="_END__"){
					return tokenChain(stream,state,['\0'],"comment")}
				else if(stream.suffix(7)=="_DATA__"){
					return tokenChain(stream,state,['\0'],"variable-2")}
				else if(stream.suffix(7)=="_C__"){
					return tokenChain(stream,state,['\0'],"string")}}}
		if(/\w/.test(ch)){
			var p=stream.pos;
			if(stream.look(-2)=="{"&&(stream.look(0)=="}"||stream.eatWhile(/\w/)&&stream.look(0)=="}"))
				return "string";
			else
				stream.pos=p}
		if(/[A-Z]/.test(ch)){
			var l=stream.look(-2);
			var p=stream.pos;
			stream.eatWhile(/[A-Z_]/);
			if(/[\da-z]/.test(stream.look(0))){
				stream.pos=p}
			else{
				var c=PERL[stream.current()];
				if(!c)
					return "meta";
				if(c[1])
					c=c[0];
				if(l!=":"){
					if(c==1)
						return "keyword";
					else if(c==2)
						return "def";
					else if(c==3)
						return "atom";
					else if(c==4)
						return "operator";
					else if(c==5)
						return "variable-2";
					else
						return "meta"}
				else
					return "meta"}}
		if(/[a-zA-Z_]/.test(ch)){
			var l=stream.look(-2);
			stream.eatWhile(/\w/);
			var c=PERL[stream.current()];
			if(!c)
				return "meta";
			if(c[1])
				c=c[0];
			if(l!=":"){
				if(c==1)
					return "keyword";
				else if(c==2)
					return "def";
				else if(c==3)
					return "atom";
				else if(c==4)
					return "operator";
				else if(c==5)
					return "variable-2";
				else
					return "meta"}
			else
				return "meta"}
		return null}

	return{
		startState:function(){
			return{
				tokenize:tokenPerl,
				chain:null,
				style:null,
				tail:null}},
		token:function(stream,state){
			return (state.tokenize||tokenPerl)(stream,state)},
		electricChars:"{}"}});

CodeMirror.defineMIME("text/x-perl", "perl");

// it's like "peek", but need for look-ahead or look-behind if index < 0
CodeMirror.StringStream.prototype.look=function(c){
	return this.string.charAt(this.pos+(c||0))};

// return a part of prefix of current stream from current position
CodeMirror.StringStream.prototype.prefix=function(c){
	if(c){
		var x=this.pos-c;
		return this.string.substr((x>=0?x:0),c)}
	else{
		return this.string.substr(0,this.pos-1)}};

// return a part of suffix of current stream from current position
CodeMirror.StringStream.prototype.suffix=function(c){
	var y=this.string.length;
	var x=y-this.pos+1;
	return this.string.substr(this.pos,(c&&c<y?c:x))};

// return a part of suffix of current stream from current position and change current position
CodeMirror.StringStream.prototype.nsuffix=function(c){
	var p=this.pos;
	var l=c||(this.string.length-this.pos+1);
	this.pos+=l;
	return this.string.substr(p,l)};

// eating and vomiting a part of stream from current position
CodeMirror.StringStream.prototype.eatSuffix=function(c){
	var x=this.pos+c;
	var y;
	if(x<=0)
		this.pos=0;
	else if(x>=(y=this.string.length-1))
		this.pos=y;
	else
		this.pos=x};
(function() {
  function keywords(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }
  function heredoc(delim) {
    return function(stream, state) {
      if (stream.match(delim)) state.tokenize = null;
      else stream.skipToEnd();
      return "string";
    }
  }
  var phpConfig = {
    name: "clike",
    keywords: keywords("abstract and array as break case catch class clone const continue declare default " +
                       "do else elseif enddeclare endfor endforeach endif endswitch endwhile extends final " +
                       "for foreach function global goto if implements interface instanceof namespace " +
                       "new or private protected public static switch throw trait try use var while xor " +
                       "die echo empty exit eval include include_once isset list require require_once return " +
                       "print unset __halt_compiler self static parent"),
    blockKeywords: keywords("catch do else elseif for foreach if switch try while"),
    atoms: keywords("true false null TRUE FALSE NULL"),
    multiLineStrings: true,
    hooks: {
      "$": function(stream, state) {
        stream.eatWhile(/[\w\$_]/);
        return "variable-2";
      },
      "<": function(stream, state) {
        if (stream.match(/<</)) {
          stream.eatWhile(/[\w\.]/);
          state.tokenize = heredoc(stream.current().slice(3));
          return state.tokenize(stream, state);
        }
        return false;
      },
      "#": function(stream, state) {
        while (!stream.eol() && !stream.match("?>", false)) stream.next();
        return "comment";
      },
      "/": function(stream, state) {
        if (stream.eat("/")) {
          while (!stream.eol() && !stream.match("?>", false)) stream.next();
          return "comment";
        }
        return false;
      }
    }
  };

  CodeMirror.defineMode("php", function(config, parserConfig) {
    var htmlMode = CodeMirror.getMode(config, {name: "xml", htmlMode: true});
    var jsMode = CodeMirror.getMode(config, "javascript");
    var cssMode = CodeMirror.getMode(config, "css");
    var phpMode = CodeMirror.getMode(config, phpConfig);

    function dispatch(stream, state) { // TODO open PHP inside text/css
      var isPHP = state.mode == "php";
      if (stream.sol() && state.pending != '"') state.pending = null;
      if (state.curMode == htmlMode) {
        if (stream.match(/^<\?\w*/)) {
          state.curMode = phpMode;
          state.curState = state.php;
          state.curClose = "?>";
	  state.mode = "php";
          return "meta";
        }
        if (state.pending == '"') {
          while (!stream.eol() && stream.next() != '"') {}
          var style = "string";
        } else if (state.pending && stream.pos < state.pending.end) {
          stream.pos = state.pending.end;
          var style = state.pending.style;
        } else {
          var style = htmlMode.token(stream, state.curState);
        }
        state.pending = null;
        var cur = stream.current(), openPHP = cur.search(/<\?/);
        if (openPHP != -1) {
          if (style == "string" && /\"$/.test(cur) && !/\?>/.test(cur)) state.pending = '"';
          else state.pending = {end: stream.pos, style: style};
          stream.backUp(cur.length - openPHP);
        } else if (style == "tag" && stream.current() == ">" && state.curState.context) {
          if (/^script$/i.test(state.curState.context.tagName)) {
            state.curMode = jsMode;
            state.curState = jsMode.startState(htmlMode.indent(state.curState, ""));
            state.curClose = /^<\/\s*script\s*>/i;
	    state.mode = "javascript";
          }
          else if (/^style$/i.test(state.curState.context.tagName)) {
            state.curMode = cssMode;
            state.curState = cssMode.startState(htmlMode.indent(state.curState, ""));
            state.curClose = /^<\/\s*style\s*>/i;
            state.mode = "css";
          }
        }
        return style;
      } else if ((!isPHP || state.php.tokenize == null) &&
                 stream.match(state.curClose, isPHP)) {
        state.curMode = htmlMode;
        state.curState = state.html;
        state.curClose = null;
	state.mode = "html";
        if (isPHP) return "meta";
        else return dispatch(stream, state);
      } else {
        return state.curMode.token(stream, state.curState);
      }
    }

    return {
      startState: function() {
        var html = htmlMode.startState();
        return {html: html,
                php: phpMode.startState(),
                curMode: parserConfig.startOpen ? phpMode : htmlMode,
                curState: parserConfig.startOpen ? phpMode.startState() : html,
                curClose: parserConfig.startOpen ? /^\?>/ : null,
		mode: parserConfig.startOpen ? "php" : "html",
                pending: null}
      },

      copyState: function(state) {
        var html = state.html, htmlNew = CodeMirror.copyState(htmlMode, html),
            php = state.php, phpNew = CodeMirror.copyState(phpMode, php), cur;
        if (state.curState == html) cur = htmlNew;
        else if (state.curState == php) cur = phpNew;
        else cur = CodeMirror.copyState(state.curMode, state.curState);
        return {html: htmlNew, php: phpNew, curMode: state.curMode, curState: cur,
                curClose: state.curClose, mode: state.mode,
                pending: state.pending};
      },

      token: dispatch,

      indent: function(state, textAfter) {
        if ((state.curMode != phpMode && /^\s*<\//.test(textAfter)) ||
            (state.curMode == phpMode && /^\?>/.test(textAfter)))
          return htmlMode.indent(state.html, textAfter);
        return state.curMode.indent(state.curState, textAfter);
      },

      electricChars: "/{}:"
    }
  }, "xml", "clike", "javascript", "css");
  CodeMirror.defineMIME("application/x-httpd-php", "php");
  CodeMirror.defineMIME("application/x-httpd-php-open", {name: "php", startOpen: true});
  CodeMirror.defineMIME("text/x-php", phpConfig);
})();
/*
 *	Pig Latin Mode for CodeMirror 2 
 *	@author Prasanth Jayachandran
 *	@link 	https://github.com/prasanthj/pig-codemirror-2
 *  This implementation is adapted from PL/SQL mode in CodeMirror 2.
*/
CodeMirror.defineMode("pig", function(config, parserConfig) {
	var indentUnit = config.indentUnit,
		keywords = parserConfig.keywords,
		builtins = parserConfig.builtins,
		types = parserConfig.types,
		multiLineStrings = parserConfig.multiLineStrings;
	
	var isOperatorChar = /[*+\-%<>=&?:\/!|]/;
	
	function chain(stream, state, f) {
		state.tokenize = f;
		return f(stream, state);
	}
	
	var type;
	function ret(tp, style) {
		type = tp;
		return style;
	}
	
	function tokenComment(stream, state) {
		var isEnd = false;
		var ch;
		while(ch = stream.next()) {
			if(ch == "/" && isEnd) {
				state.tokenize = tokenBase;
				break;
			}
			isEnd = (ch == "*");
		}
		return ret("comment", "comment");
	}
	
	function tokenString(quote) {
		return function(stream, state) {
			var escaped = false, next, end = false;
			while((next = stream.next()) != null) {
				if (next == quote && !escaped) {
					end = true; break;
				}
				escaped = !escaped && next == "\\";
			}
			if (end || !(escaped || multiLineStrings))
				state.tokenize = tokenBase;
			return ret("string", "error");
		};
	}
	
	function tokenBase(stream, state) {
		var ch = stream.next();
		
		// is a start of string?
		if (ch == '"' || ch == "'")
			return chain(stream, state, tokenString(ch));
		// is it one of the special chars
		else if(/[\[\]{}\(\),;\.]/.test(ch))
			return ret(ch);
		// is it a number?
		else if(/\d/.test(ch)) {
			stream.eatWhile(/[\w\.]/);
			return ret("number", "number");
		}
		// multi line comment or operator
		else if (ch == "/") {
			if (stream.eat("*")) {
				return chain(stream, state, tokenComment);
			}
			else {
				stream.eatWhile(isOperatorChar);
				return ret("operator", "operator");
			}
		}
		// single line comment or operator
		else if (ch=="-") {
			if(stream.eat("-")){
				stream.skipToEnd();
				return ret("comment", "comment");
			}
			else {
				stream.eatWhile(isOperatorChar);
				return ret("operator", "operator");
			}
		}
		// is it an operator
		else if (isOperatorChar.test(ch)) {
			stream.eatWhile(isOperatorChar);
			return ret("operator", "operator");
		}
		else {
			// get the while word
			stream.eatWhile(/[\w\$_]/);
			// is it one of the listed keywords?
			if (keywords && keywords.propertyIsEnumerable(stream.current().toUpperCase())) {
				if (stream.eat(")") || stream.eat(".")) {
					//keywords can be used as variables like flatten(group), group.$0 etc..
				}
				else {
					return ("keyword", "keyword");
				}
			}
			// is it one of the builtin functions?
			if (builtins && builtins.propertyIsEnumerable(stream.current().toUpperCase()))
			{
				return ("keyword", "variable-2")
			}
			// is it one of the listed types?
			if (types && types.propertyIsEnumerable(stream.current().toUpperCase()))
				return ("keyword", "variable-3")
			// default is a 'word'
			return ret("word", "pig-word");
		}
	}
	
	// Interface
	return {
		startState: function(basecolumn) {
			return {
				tokenize: tokenBase,
				startOfLine: true
			};
		},
		
		token: function(stream, state) {
			if(stream.eatSpace()) return null;
			var style = state.tokenize(stream, state);
			return style;
		}
	};
});

(function() {
	function keywords(str) {
		var obj = {}, words = str.split(" ");
		for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
 		return obj;
 	}

	// builtin funcs taken from trunk revision 1303237
	var pBuiltins = "ABS ACOS ARITY ASIN ATAN AVG BAGSIZE BINSTORAGE BLOOM BUILDBLOOM CBRT CEIL " 
	+ "CONCAT COR COS COSH COUNT COUNT_STAR COV CONSTANTSIZE CUBEDIMENSIONS DIFF DISTINCT DOUBLEABS "
	+ "DOUBLEAVG DOUBLEBASE DOUBLEMAX DOUBLEMIN DOUBLEROUND DOUBLESUM EXP FLOOR FLOATABS FLOATAVG "
	+ "FLOATMAX FLOATMIN FLOATROUND FLOATSUM GENERICINVOKER INDEXOF INTABS INTAVG INTMAX INTMIN "
	+ "INTSUM INVOKEFORDOUBLE INVOKEFORFLOAT INVOKEFORINT INVOKEFORLONG INVOKEFORSTRING INVOKER "
	+ "ISEMPTY JSONLOADER JSONMETADATA JSONSTORAGE LAST_INDEX_OF LCFIRST LOG LOG10 LOWER LONGABS "
	+ "LONGAVG LONGMAX LONGMIN LONGSUM MAX MIN MAPSIZE MONITOREDUDF NONDETERMINISTIC OUTPUTSCHEMA  "
	+ "PIGSTORAGE PIGSTREAMING RANDOM REGEX_EXTRACT REGEX_EXTRACT_ALL REPLACE ROUND SIN SINH SIZE "
	+ "SQRT STRSPLIT SUBSTRING SUM STRINGCONCAT STRINGMAX STRINGMIN STRINGSIZE TAN TANH TOBAG "
	+ "TOKENIZE TOMAP TOP TOTUPLE TRIM TEXTLOADER TUPLESIZE UCFIRST UPPER UTF8STORAGECONVERTER "; 
	
	// taken from QueryLexer.g
	var pKeywords = "VOID IMPORT RETURNS DEFINE LOAD FILTER FOREACH ORDER CUBE DISTINCT COGROUP "
	+ "JOIN CROSS UNION SPLIT INTO IF OTHERWISE ALL AS BY USING INNER OUTER ONSCHEMA PARALLEL "
	+ "PARTITION GROUP AND OR NOT GENERATE FLATTEN ASC DESC IS STREAM THROUGH STORE MAPREDUCE "
	+ "SHIP CACHE INPUT OUTPUT STDERROR STDIN STDOUT LIMIT SAMPLE LEFT RIGHT FULL EQ GT LT GTE LTE " 
	+ "NEQ MATCHES TRUE FALSE "; 
	
	// data types
	var pTypes = "BOOLEAN INT LONG FLOAT DOUBLE CHARARRAY BYTEARRAY BAG TUPLE MAP "
	
	CodeMirror.defineMIME("text/x-pig", {
	 name: "pig",
	 builtins: keywords(pBuiltins),
	 keywords: keywords(pKeywords),
	 types: keywords(pTypes)
	 });
}());
CodeMirror.defineMode("plsql", function(config, parserConfig) {
  var indentUnit       = config.indentUnit,
      keywords         = parserConfig.keywords,
      functions        = parserConfig.functions,
      types            = parserConfig.types,
      sqlplus          = parserConfig.sqlplus,
      multiLineStrings = parserConfig.multiLineStrings;
  var isOperatorChar   = /[+\-*&%=<>!?:\/|]/;
  function chain(stream, state, f) {
    state.tokenize = f;
    return f(stream, state);
  }

  var type;
  function ret(tp, style) {
    type = tp;
    return style;
  }

  function tokenBase(stream, state) {
    var ch = stream.next();
    // start of string?
    if (ch == '"' || ch == "'")
      return chain(stream, state, tokenString(ch));
    // is it one of the special signs []{}().,;? Seperator?
    else if (/[\[\]{}\(\),;\.]/.test(ch))
      return ret(ch);
    // start of a number value?
    else if (/\d/.test(ch)) {
      stream.eatWhile(/[\w\.]/);
      return ret("number", "number");
    }
    // multi line comment or simple operator?
    else if (ch == "/") {
      if (stream.eat("*")) {
        return chain(stream, state, tokenComment);
      }
      else {
        stream.eatWhile(isOperatorChar);
        return ret("operator", "operator");
      }
    }
    // single line comment or simple operator?
    else if (ch == "-") {
      if (stream.eat("-")) {
        stream.skipToEnd();
        return ret("comment", "comment");
      }
      else {
        stream.eatWhile(isOperatorChar);
        return ret("operator", "operator");
      }
    }
    // pl/sql variable?
    else if (ch == "@" || ch == "$") {
      stream.eatWhile(/[\w\d\$_]/);
      return ret("word", "variable");
    }
    // is it a operator?
    else if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return ret("operator", "operator");
    }
    else {
      // get the whole word
      stream.eatWhile(/[\w\$_]/);
      // is it one of the listed keywords?
      if (keywords && keywords.propertyIsEnumerable(stream.current().toLowerCase())) return ret("keyword", "keyword");
      // is it one of the listed functions?
      if (functions && functions.propertyIsEnumerable(stream.current().toLowerCase())) return ret("keyword", "builtin");
      // is it one of the listed types?
      if (types && types.propertyIsEnumerable(stream.current().toLowerCase())) return ret("keyword", "variable-2");
      // is it one of the listed sqlplus keywords?
      if (sqlplus && sqlplus.propertyIsEnumerable(stream.current().toLowerCase())) return ret("keyword", "variable-3");
      // default: just a "word"
      return ret("word", "plsql-word");
    }
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !(escaped || multiLineStrings))
        state.tokenize = tokenBase;
      return ret("string", "plsql-string");
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return ret("comment", "plsql-comment");
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: tokenBase,
        startOfLine: true
      };
    },

    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);
      return style;
    }
  };
});

(function() {
  function keywords(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }
  var cKeywords = "abort accept access add all alter and any array arraylen as asc assert assign at attributes audit " +
        "authorization avg " +
        "base_table begin between binary_integer body boolean by " +
        "case cast char char_base check close cluster clusters colauth column comment commit compress connect " +
        "connected constant constraint crash create current currval cursor " +
        "data_base database date dba deallocate debugoff debugon decimal declare default definition delay delete " +
        "desc digits dispose distinct do drop " +
        "else elsif enable end entry escape exception exception_init exchange exclusive exists exit external " +
        "fast fetch file for force form from function " +
        "generic goto grant group " +
        "having " +
        "identified if immediate in increment index indexes indicator initial initrans insert interface intersect " +
        "into is " +
        "key " +
        "level library like limited local lock log logging long loop " +
        "master maxextents maxtrans member minextents minus mislabel mode modify multiset " +
        "new next no noaudit nocompress nologging noparallel not nowait number_base " +
        "object of off offline on online only open option or order out " +
        "package parallel partition pctfree pctincrease pctused pls_integer positive positiven pragma primary prior " +
        "private privileges procedure public " +
        "raise range raw read rebuild record ref references refresh release rename replace resource restrict return " +
        "returning reverse revoke rollback row rowid rowlabel rownum rows run " +
        "savepoint schema segment select separate session set share snapshot some space split sql start statement " +
        "storage subtype successful synonym " +
        "tabauth table tables tablespace task terminate then to trigger truncate type " +
        "union unique unlimited unrecoverable unusable update use using " +
        "validate value values variable view views " +
        "when whenever where while with work";

  var cFunctions = "abs acos add_months ascii asin atan atan2 average " +
        "bfilename " +
        "ceil chartorowid chr concat convert cos cosh count " +
        "decode deref dual dump dup_val_on_index " +
        "empty error exp " +
        "false floor found " +
        "glb greatest " +
        "hextoraw " +
        "initcap instr instrb isopen " +
        "last_day least lenght lenghtb ln lower lpad ltrim lub " +
        "make_ref max min mod months_between " +
        "new_time next_day nextval nls_charset_decl_len nls_charset_id nls_charset_name nls_initcap nls_lower " +
        "nls_sort nls_upper nlssort no_data_found notfound null nvl " +
        "others " +
        "power " +
        "rawtohex reftohex round rowcount rowidtochar rpad rtrim " +
        "sign sin sinh soundex sqlcode sqlerrm sqrt stddev substr substrb sum sysdate " +
        "tan tanh to_char to_date to_label to_multi_byte to_number to_single_byte translate true trunc " +
        "uid upper user userenv " +
        "variance vsize";

  var cTypes = "bfile blob " +
        "character clob " +
        "dec " +
        "float " +
        "int integer " +
        "mlslabel " +
        "natural naturaln nchar nclob number numeric nvarchar2 " +
        "real rowtype " +
        "signtype smallint string " +
        "varchar varchar2";

  var cSqlplus = "appinfo arraysize autocommit autoprint autorecovery autotrace " +
        "blockterminator break btitle " +
        "cmdsep colsep compatibility compute concat copycommit copytypecheck " +
        "define describe " +
        "echo editfile embedded escape exec execute " +
        "feedback flagger flush " +
        "heading headsep " +
        "instance " +
        "linesize lno loboffset logsource long longchunksize " +
        "markup " +
        "native newpage numformat numwidth " +
        "pagesize pause pno " +
        "recsep recsepchar release repfooter repheader " +
        "serveroutput shiftinout show showmode size spool sqlblanklines sqlcase sqlcode sqlcontinue sqlnumber " +
        "sqlpluscompatibility sqlprefix sqlprompt sqlterminator suffix " +
        "tab term termout time timing trimout trimspool ttitle " +
        "underline " +
        "verify version " +
        "wrap";

  CodeMirror.defineMIME("text/x-plsql", {
    name: "plsql",
    keywords: keywords(cKeywords),
    functions: keywords(cFunctions),
    types: keywords(cTypes),
    sqlplus: keywords(cSqlplus)
  });
}());
CodeMirror.defineMode("properties", function() {
  return {
    token: function(stream, state) {
      var sol = stream.sol() || state.afterSection;
      var eol = stream.eol();

      state.afterSection = false;

      if (sol) {
        if (state.nextMultiline) {
          state.inMultiline = true;
          state.nextMultiline = false;
        } else {
          state.position = "def";
        }
      }

      if (eol && ! state.nextMultiline) {
        state.inMultiline = false;
        state.position = "def";
      }

      if (sol) {
        while(stream.eatSpace());
      }

      var ch = stream.next();

      if (sol && (ch === "#" || ch === "!" || ch === ";")) {
        state.position = "comment";
        stream.skipToEnd();
        return "comment";
      } else if (sol && ch === "[") {
        state.afterSection = true;
        stream.skipTo("]"); stream.eat("]");
        return "header";
      } else if (ch === "=" || ch === ":") {
        state.position = "quote";
        return null;
      } else if (ch === "\\" && state.position === "quote") {
        if (stream.next() !== "u") {    // u = Unicode sequence \u1234
          // Multiline value
          state.nextMultiline = true;
        }
      }

      return state.position;
    },

    startState: function() {
      return {
        position : "def",       // Current position, "def", "quote" or "comment"
        nextMultiline : false,  // Is the next line multiline value
        inMultiline : false,    // Is the current line a multiline value
        afterSection : false    // Did we just open a section
      };
    }

  };
});

CodeMirror.defineMIME("text/x-properties", "properties");
CodeMirror.defineMIME("text/x-ini", "properties");
CodeMirror.defineMode("python", function(conf, parserConf) {
    var ERRORCLASS = 'error';
    
    function wordRegexp(words) {
        return new RegExp("^((" + words.join(")|(") + "))\\b");
    }
    
    var singleOperators = new RegExp("^[\\+\\-\\*/%&|\\^~<>!]");
    var singleDelimiters = new RegExp('^[\\(\\)\\[\\]\\{\\}@,:`=;\\.]');
    var doubleOperators = new RegExp("^((==)|(!=)|(<=)|(>=)|(<>)|(<<)|(>>)|(//)|(\\*\\*))");
    var doubleDelimiters = new RegExp("^((\\+=)|(\\-=)|(\\*=)|(%=)|(/=)|(&=)|(\\|=)|(\\^=))");
    var tripleDelimiters = new RegExp("^((//=)|(>>=)|(<<=)|(\\*\\*=))");
    var identifiers = new RegExp("^[_A-Za-z][_A-Za-z0-9]*");

    var wordOperators = wordRegexp(['and', 'or', 'not', 'is', 'in']);
    var commonkeywords = ['as', 'assert', 'break', 'class', 'continue',
                          'def', 'del', 'elif', 'else', 'except', 'finally',
                          'for', 'from', 'global', 'if', 'import',
                          'lambda', 'pass', 'raise', 'return',
                          'try', 'while', 'with', 'yield'];
    var commonBuiltins = ['abs', 'all', 'any', 'bin', 'bool', 'bytearray', 'callable', 'chr',
                          'classmethod', 'compile', 'complex', 'delattr', 'dict', 'dir', 'divmod',
                          'enumerate', 'eval', 'filter', 'float', 'format', 'frozenset',
                          'getattr', 'globals', 'hasattr', 'hash', 'help', 'hex', 'id',
                          'input', 'int', 'isinstance', 'issubclass', 'iter', 'len',
                          'list', 'locals', 'map', 'max', 'memoryview', 'min', 'next',
                          'object', 'oct', 'open', 'ord', 'pow', 'property', 'range',
                          'repr', 'reversed', 'round', 'set', 'setattr', 'slice',
                          'sorted', 'staticmethod', 'str', 'sum', 'super', 'tuple',
                          'type', 'vars', 'zip', '__import__', 'NotImplemented',
                          'Ellipsis', '__debug__'];
    var py2 = {'builtins': ['apply', 'basestring', 'buffer', 'cmp', 'coerce', 'execfile',
                            'file', 'intern', 'long', 'raw_input', 'reduce', 'reload',
                            'unichr', 'unicode', 'xrange', 'False', 'True', 'None'],
               'keywords': ['exec', 'print']};
    var py3 = {'builtins': ['ascii', 'bytes', 'exec', 'print'],
               'keywords': ['nonlocal', 'False', 'True', 'None']};

    if (!!parserConf.version && parseInt(parserConf.version, 10) === 3) {
        commonkeywords = commonkeywords.concat(py3.keywords);
        commonBuiltins = commonBuiltins.concat(py3.builtins);
        var stringPrefixes = new RegExp("^(([rb]|(br))?('{3}|\"{3}|['\"]))", "i");
    } else {
        commonkeywords = commonkeywords.concat(py2.keywords);
        commonBuiltins = commonBuiltins.concat(py2.builtins);
        var stringPrefixes = new RegExp("^(([rub]|(ur)|(br))?('{3}|\"{3}|['\"]))", "i");
    }
    var keywords = wordRegexp(commonkeywords);
    var builtins = wordRegexp(commonBuiltins);

    var indentInfo = null;

    // tokenizers
    function tokenBase(stream, state) {
        // Handle scope changes
        if (stream.sol()) {
            var scopeOffset = state.scopes[0].offset;
            if (stream.eatSpace()) {
                var lineOffset = stream.indentation();
                if (lineOffset > scopeOffset) {
                    indentInfo = 'indent';
                } else if (lineOffset < scopeOffset) {
                    indentInfo = 'dedent';
                }
                return null;
            } else {
                if (scopeOffset > 0) {
                    dedent(stream, state);
                }
            }
        }
        if (stream.eatSpace()) {
            return null;
        }
        
        var ch = stream.peek();
        
        // Handle Comments
        if (ch === '#') {
            stream.skipToEnd();
            return 'comment';
        }
        
        // Handle Number Literals
        if (stream.match(/^[0-9\.]/, false)) {
            var floatLiteral = false;
            // Floats
            if (stream.match(/^\d*\.\d+(e[\+\-]?\d+)?/i)) { floatLiteral = true; }
            if (stream.match(/^\d+\.\d*/)) { floatLiteral = true; }
            if (stream.match(/^\.\d+/)) { floatLiteral = true; }
            if (floatLiteral) {
                // Float literals may be "imaginary"
                stream.eat(/J/i);
                return 'number';
            }
            // Integers
            var intLiteral = false;
            // Hex
            if (stream.match(/^0x[0-9a-f]+/i)) { intLiteral = true; }
            // Binary
            if (stream.match(/^0b[01]+/i)) { intLiteral = true; }
            // Octal
            if (stream.match(/^0o[0-7]+/i)) { intLiteral = true; }
            // Decimal
            if (stream.match(/^[1-9]\d*(e[\+\-]?\d+)?/)) {
                // Decimal literals may be "imaginary"
                stream.eat(/J/i);
                // TODO - Can you have imaginary longs?
                intLiteral = true;
            }
            // Zero by itself with no other piece of number.
            if (stream.match(/^0(?![\dx])/i)) { intLiteral = true; }
            if (intLiteral) {
                // Integer literals may be "long"
                stream.eat(/L/i);
                return 'number';
            }
        }
        
        // Handle Strings
        if (stream.match(stringPrefixes)) {
            state.tokenize = tokenStringFactory(stream.current());
            return state.tokenize(stream, state);
        }
        
        // Handle operators and Delimiters
        if (stream.match(tripleDelimiters) || stream.match(doubleDelimiters)) {
            return null;
        }
        if (stream.match(doubleOperators)
            || stream.match(singleOperators)
            || stream.match(wordOperators)) {
            return 'operator';
        }
        if (stream.match(singleDelimiters)) {
            return null;
        }
        
        if (stream.match(keywords)) {
            return 'keyword';
        }
        
        if (stream.match(builtins)) {
            return 'builtin';
        }
        
        if (stream.match(identifiers)) {
            return 'variable';
        }
        
        // Handle non-detected items
        stream.next();
        return ERRORCLASS;
    }
    
    function tokenStringFactory(delimiter) {
        while ('rub'.indexOf(delimiter.charAt(0).toLowerCase()) >= 0) {
            delimiter = delimiter.substr(1);
        }
        var singleline = delimiter.length == 1;
        var OUTCLASS = 'string';
        
        return function tokenString(stream, state) {
            while (!stream.eol()) {
                stream.eatWhile(/[^'"\\]/);
                if (stream.eat('\\')) {
                    stream.next();
                    if (singleline && stream.eol()) {
                        return OUTCLASS;
                    }
                } else if (stream.match(delimiter)) {
                    state.tokenize = tokenBase;
                    return OUTCLASS;
                } else {
                    stream.eat(/['"]/);
                }
            }
            if (singleline) {
                if (parserConf.singleLineStringErrors) {
                    return ERRORCLASS;
                } else {
                    state.tokenize = tokenBase;
                }
            }
            return OUTCLASS;
        };
    }
    
    function indent(stream, state, type) {
        type = type || 'py';
        var indentUnit = 0;
        if (type === 'py') {
            if (state.scopes[0].type !== 'py') {
                state.scopes[0].offset = stream.indentation();
                return;
            }
            for (var i = 0; i < state.scopes.length; ++i) {
                if (state.scopes[i].type === 'py') {
                    indentUnit = state.scopes[i].offset + conf.indentUnit;
                    break;
                }
            }
        } else {
            indentUnit = stream.column() + stream.current().length;
        }
        state.scopes.unshift({
            offset: indentUnit,
            type: type
        });
    }
    
    function dedent(stream, state, type) {
        type = type || 'py';
        if (state.scopes.length == 1) return;
        if (state.scopes[0].type === 'py') {
            var _indent = stream.indentation();
            var _indent_index = -1;
            for (var i = 0; i < state.scopes.length; ++i) {
                if (_indent === state.scopes[i].offset) {
                    _indent_index = i;
                    break;
                }
            }
            if (_indent_index === -1) {
                return true;
            }
            while (state.scopes[0].offset !== _indent) {
                state.scopes.shift();
            }
            return false
        } else {
            if (type === 'py') {
                state.scopes[0].offset = stream.indentation();
                return false;
            } else {
                if (state.scopes[0].type != type) {
                    return true;
                }
                state.scopes.shift();
                return false;
            }
        }
    }

    function tokenLexer(stream, state) {
        indentInfo = null;
        var style = state.tokenize(stream, state);
        var current = stream.current();

        // Handle '.' connected identifiers
        if (current === '.') {
            style = stream.match(identifiers, false) ? null : ERRORCLASS;
            if (style === null && state.lastToken === 'meta') {
                // Apply 'meta' style to '.' connected identifiers when
                // appropriate.
                style = 'meta';
            }
            return style;
        }
        
        // Handle decorators
        if (current === '@') {
            return stream.match(identifiers, false) ? 'meta' : ERRORCLASS;
        }

        if ((style === 'variable' || style === 'builtin')
            && state.lastToken === 'meta') {
            style = 'meta';
        }
        
        // Handle scope changes.
        if (current === 'pass' || current === 'return') {
            state.dedent += 1;
        }
        if (current === 'lambda') state.lambda = true;
        if ((current === ':' && !state.lambda && state.scopes[0].type == 'py')
            || indentInfo === 'indent') {
            indent(stream, state);
        }
        var delimiter_index = '[({'.indexOf(current);
        if (delimiter_index !== -1) {
            indent(stream, state, '])}'.slice(delimiter_index, delimiter_index+1));
        }
        if (indentInfo === 'dedent') {
            if (dedent(stream, state)) {
                return ERRORCLASS;
            }
        }
        delimiter_index = '])}'.indexOf(current);
        if (delimiter_index !== -1) {
            if (dedent(stream, state, current)) {
                return ERRORCLASS;
            }
        }
        if (state.dedent > 0 && stream.eol() && state.scopes[0].type == 'py') {
            if (state.scopes.length > 1) state.scopes.shift();
            state.dedent -= 1;
        }
        
        return style;
    }

    var external = {
        startState: function(basecolumn) {
            return {
              tokenize: tokenBase,
              scopes: [{offset:basecolumn || 0, type:'py'}],
              lastToken: null,
              lambda: false,
              dedent: 0
          };
        },
        
        token: function(stream, state) {
            var style = tokenLexer(stream, state);
            
            state.lastToken = style;
            
            if (stream.eol() && stream.lambda) {
                state.lambda = false;
            }
            
            return style;
        },
        
        indent: function(state, textAfter) {
            if (state.tokenize != tokenBase) {
                return 0;
            }
            
            return state.scopes[0].offset;
        }
        
    };
    return external;
});

CodeMirror.defineMIME("text/x-python", "python");
CodeMirror.defineMode("r", function(config) {
  function wordObj(str) {
    var words = str.split(" "), res = {};
    for (var i = 0; i < words.length; ++i) res[words[i]] = true;
    return res;
  }
  var atoms = wordObj("NULL NA Inf NaN NA_integer_ NA_real_ NA_complex_ NA_character_");
  var builtins = wordObj("list quote bquote eval return call parse deparse");
  var keywords = wordObj("if else repeat while function for in next break");
  var blockkeywords = wordObj("if else repeat while function for");
  var opChars = /[+\-*\/^<>=!&|~$:]/;
  var curPunc;

  function tokenBase(stream, state) {
    curPunc = null;
    var ch = stream.next();
    if (ch == "#") {
      stream.skipToEnd();
      return "comment";
    } else if (ch == "0" && stream.eat("x")) {
      stream.eatWhile(/[\da-f]/i);
      return "number";
    } else if (ch == "." && stream.eat(/\d/)) {
      stream.match(/\d*(?:e[+\-]?\d+)?/);
      return "number";
    } else if (/\d/.test(ch)) {
      stream.match(/\d*(?:\.\d+)?(?:e[+\-]\d+)?L?/);
      return "number";
    } else if (ch == "'" || ch == '"') {
      state.tokenize = tokenString(ch);
      return "string";
    } else if (ch == "." && stream.match(/.[.\d]+/)) {
      return "keyword";
    } else if (/[\w\.]/.test(ch) && ch != "_") {
      stream.eatWhile(/[\w\.]/);
      var word = stream.current();
      if (atoms.propertyIsEnumerable(word)) return "atom";
      if (keywords.propertyIsEnumerable(word)) {
        if (blockkeywords.propertyIsEnumerable(word)) curPunc = "block";
        return "keyword";
      }
      if (builtins.propertyIsEnumerable(word)) return "builtin";
      return "variable";
    } else if (ch == "%") {
      if (stream.skipTo("%")) stream.next();
      return "variable-2";
    } else if (ch == "<" && stream.eat("-")) {
      return "arrow";
    } else if (ch == "=" && state.ctx.argList) {
      return "arg-is";
    } else if (opChars.test(ch)) {
      if (ch == "$") return "dollar";
      stream.eatWhile(opChars);
      return "operator";
    } else if (/[\(\){}\[\];]/.test(ch)) {
      curPunc = ch;
      if (ch == ";") return "semi";
      return null;
    } else {
      return null;
    }
  }

  function tokenString(quote) {
    return function(stream, state) {
      if (stream.eat("\\")) {
        var ch = stream.next();
        if (ch == "x") stream.match(/^[a-f0-9]{2}/i);
        else if ((ch == "u" || ch == "U") && stream.eat("{") && stream.skipTo("}")) stream.next();
        else if (ch == "u") stream.match(/^[a-f0-9]{4}/i);
        else if (ch == "U") stream.match(/^[a-f0-9]{8}/i);
        else if (/[0-7]/.test(ch)) stream.match(/^[0-7]{1,2}/);
        return "string-2";
      } else {
        var next;
        while ((next = stream.next()) != null) {
          if (next == quote) { state.tokenize = tokenBase; break; }
          if (next == "\\") { stream.backUp(1); break; }
        }
        return "string";
      }
    };
  }

  function push(state, type, stream) {
    state.ctx = {type: type,
                 indent: state.indent,
                 align: null,
                 column: stream.column(),
                 prev: state.ctx};
  }
  function pop(state) {
    state.indent = state.ctx.indent;
    state.ctx = state.ctx.prev;
  }

  return {
    startState: function(base) {
      return {tokenize: tokenBase,
              ctx: {type: "top",
                    indent: -config.indentUnit,
                    align: false},
              indent: 0,
              afterIdent: false};
    },

    token: function(stream, state) {
      if (stream.sol()) {
        if (state.ctx.align == null) state.ctx.align = false;
        state.indent = stream.indentation();
      }
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);
      if (style != "comment" && state.ctx.align == null) state.ctx.align = true;

      var ctype = state.ctx.type;
      if ((curPunc == ";" || curPunc == "{" || curPunc == "}") && ctype == "block") pop(state);
      if (curPunc == "{") push(state, "}", stream);
      else if (curPunc == "(") {
        push(state, ")", stream);
        if (state.afterIdent) state.ctx.argList = true;
      }
      else if (curPunc == "[") push(state, "]", stream);
      else if (curPunc == "block") push(state, "block", stream);
      else if (curPunc == ctype) pop(state);
      state.afterIdent = style == "variable" || style == "keyword";
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase) return 0;
      var firstChar = textAfter && textAfter.charAt(0), ctx = state.ctx,
          closing = firstChar == ctx.type;
      if (ctx.type == "block") return ctx.indent + (firstChar == "{" ? 0 : config.indentUnit);
      else if (ctx.align) return ctx.column + (closing ? 0 : 1);
      else return ctx.indent + (closing ? 0 : config.indentUnit);
    }
  };
});

CodeMirror.defineMIME("text/x-rsrc", "r");
CodeMirror.defineMode('rst', function(config, options) {
    function setState(state, fn, ctx) {
        state.fn = fn;
        setCtx(state, ctx);
    }

    function setCtx(state, ctx) {
        state.ctx = ctx || {};
    }

    function setNormal(state, ch) {
        if (ch && (typeof ch !== 'string')) {
            var str = ch.current();
            ch = str[str.length-1];
        }

        setState(state, normal, {back: ch});
    }

    function hasMode(mode) {
        if (mode) {
            var modes = CodeMirror.listModes();

            for (var i in modes) {
                if (modes[i] == mode) {
                    return true;
                }
            }
        }

        return false;
    }

    function getMode(mode) {
        if (hasMode(mode)) {
            return CodeMirror.getMode(config, mode);
        } else {
            return null;
        }
    }

    var verbatimMode = getMode(options.verbatim);
    var pythonMode = getMode('python');

    var reSection = /^[!"#$%&'()*+,-./:;<=>?@[\\\]^_`{|}~]/;
    var reDirective = /^\s*\w([-:.\w]*\w)?::(\s|$)/;
    var reHyperlink = /^\s*_[\w-]+:(\s|$)/;
    var reFootnote = /^\s*\[(\d+|#)\](\s|$)/;
    var reCitation = /^\s*\[[A-Za-z][\w-]*\](\s|$)/;
    var reFootnoteRef = /^\[(\d+|#)\]_/;
    var reCitationRef = /^\[[A-Za-z][\w-]*\]_/;
    var reDirectiveMarker = /^\.\.(\s|$)/;
    var reVerbatimMarker = /^::\s*$/;
    var rePreInline = /^[-\s"([{</:]/;
    var rePostInline = /^[-\s`'")\]}>/:.,;!?\\_]/;
    var reEnumeratedList = /^\s*((\d+|[A-Za-z#])[.)]|\((\d+|[A-Z-a-z#])\))\s/;
    var reBulletedList = /^\s*[-\+\*]\s/;
    var reExamples = /^\s+(>>>|In \[\d+\]:)\s/;

    function normal(stream, state) {
        var ch, sol, i;

        if (stream.eat(/\\/)) {
            ch = stream.next();
            setNormal(state, ch);
            return null;
        }

        sol = stream.sol();

        if (sol && (ch = stream.eat(reSection))) {
            for (i = 0; stream.eat(ch); i++);

            if (i >= 3 && stream.match(/^\s*$/)) {
                setNormal(state, null);
                return 'header';
            } else {
                stream.backUp(i + 1);
            }
        }

        if (sol && stream.match(reDirectiveMarker)) {
            if (!stream.eol()) {
                setState(state, directive);
            }
            return 'meta';
        }

        if (stream.match(reVerbatimMarker)) {
            if (!verbatimMode) {
                setState(state, verbatim);
            } else {
                var mode = verbatimMode;

                setState(state, verbatim, {
                    mode: mode,
                    local: mode.startState()
                });
            }
            return 'meta';
        }

        if (sol && stream.match(reExamples, false)) {
            if (!pythonMode) {
                setState(state, verbatim);
                return 'meta';
            } else {
                var mode = pythonMode;

                setState(state, verbatim, {
                    mode: mode,
                    local: mode.startState()
                });

                return null;
            }
        }

        function testBackward(re) {
            return sol || !state.ctx.back || re.test(state.ctx.back);
        }

        function testForward(re) {
            return stream.eol() || stream.match(re, false);
        }

        function testInline(re) {
            return stream.match(re) && testBackward(/\W/) && testForward(/\W/);
        }

        if (testInline(reFootnoteRef)) {
            setNormal(state, stream);
            return 'footnote';
        }

        if (testInline(reCitationRef)) {
            setNormal(state, stream);
            return 'citation';
        }

        ch = stream.next();

        if (testBackward(rePreInline)) {
            if ((ch === ':' || ch === '|') && stream.eat(/\S/)) {
                var token;

                if (ch === ':') {
                    token = 'builtin';
                } else {
                    token = 'atom';
                }

                setState(state, inline, {
                    ch: ch,
                    wide: false,
                    prev: null,
                    token: token
                });

                return token;
            }

            if (ch === '*' || ch === '`') {
                var orig = ch,
                    wide = false;

                ch = stream.next();

                if (ch == orig) {
                    wide = true;
                    ch = stream.next();
                }

                if (ch && !/\s/.test(ch)) {
                    var token;

                    if (orig === '*') {
                        token = wide ? 'strong' : 'em';
                    } else {
                        token = wide ? 'string' : 'string-2';
                    }

                    setState(state, inline, {
                        ch: orig,               // inline() has to know what to search for
                        wide: wide,             // are we looking for `ch` or `chch`
                        prev: null,             // terminator must not be preceeded with whitespace
                        token: token            // I don't want to recompute this all the time
                    });

                    return token;
                }
            }
        }

        setNormal(state, ch);
        return null;
    }

    function inline(stream, state) {
        var ch = stream.next(),
            token = state.ctx.token;

        function finish(ch) {
            state.ctx.prev = ch;
            return token;
        }

        if (ch != state.ctx.ch) {
            return finish(ch);
        }

        if (/\s/.test(state.ctx.prev)) {
            return finish(ch);
        }

        if (state.ctx.wide) {
            ch = stream.next();

            if (ch != state.ctx.ch) {
                return finish(ch);
            }
        }

        if (!stream.eol() && !rePostInline.test(stream.peek())) {
            if (state.ctx.wide) {
                stream.backUp(1);
            }

            return finish(ch);
        }

        setState(state, normal);
        setNormal(state, ch);

        return token;
    }

    function directive(stream, state) {
        var token = null;

        if (stream.match(reDirective)) {
            token = 'attribute';
        } else if (stream.match(reHyperlink)) {
            token = 'link';
        } else if (stream.match(reFootnote)) {
            token = 'quote';
        } else if (stream.match(reCitation)) {
            token = 'quote';
        } else {
            stream.eatSpace();

            if (stream.eol()) {
                setNormal(state, stream);
                return null;
            } else {
                stream.skipToEnd();
                setState(state, comment);
                return 'comment';
            }
        }

        // FIXME this is unreachable
        setState(state, body, {start: true});
        return token;
    }

    function body(stream, state) {
        var token = 'body';

        if (!state.ctx.start || stream.sol()) {
            return block(stream, state, token);
        }

        stream.skipToEnd();
        setCtx(state);

        return token;
    }

    function comment(stream, state) {
        return block(stream, state, 'comment');
    }

    function verbatim(stream, state) {
        if (!verbatimMode) {
            return block(stream, state, 'meta');
        } else {
            if (stream.sol()) {
                if (!stream.eatSpace()) {
                    setNormal(state, stream);
                }

                return null;
            }

            return verbatimMode.token(stream, state.ctx.local);
        }
    }

    function block(stream, state, token) {
        if (stream.eol() || stream.eatSpace()) {
            stream.skipToEnd();
            return token;
        } else {
            setNormal(state, stream);
            return null;
        }
    }

    return {
        startState: function() {
            return {fn: normal, ctx: {}};
        },

        copyState: function(state) {
            return {fn: state.fn, ctx: state.ctx};
        },

        token: function(stream, state) {
            var token = state.fn(stream, state);
            return token;
        }
    };
}, "python");

CodeMirror.defineMIME("text/x-rst", "rst");
CodeMirror.defineMode("ruby", function(config, parserConfig) {
  function wordObj(words) {
    var o = {};
    for (var i = 0, e = words.length; i < e; ++i) o[words[i]] = true;
    return o;
  }
  var keywords = wordObj([
    "alias", "and", "BEGIN", "begin", "break", "case", "class", "def", "defined?", "do", "else",
    "elsif", "END", "end", "ensure", "false", "for", "if", "in", "module", "next", "not", "or",
    "redo", "rescue", "retry", "return", "self", "super", "then", "true", "undef", "unless",
    "until", "when", "while", "yield", "nil", "raise", "throw", "catch", "fail", "loop", "callcc",
    "caller", "lambda", "proc", "public", "protected", "private", "require", "load",
    "require_relative", "extend", "autoload"
  ]);
  var indentWords = wordObj(["def", "class", "case", "for", "while", "do", "module", "then",
                             "catch", "loop", "proc", "begin"]);
  var dedentWords = wordObj(["end", "until"]);
  var matching = {"[": "]", "{": "}", "(": ")"};
  var curPunc;

  function chain(newtok, stream, state) {
    state.tokenize.push(newtok);
    return newtok(stream, state);
  }

  function tokenBase(stream, state) {
    curPunc = null;
    if (stream.sol() && stream.match("=begin") && stream.eol()) {
      state.tokenize.push(readBlockComment);
      return "comment";
    }
    if (stream.eatSpace()) return null;
    var ch = stream.next(), m;
    if (ch == "`" || ch == "'" || ch == '"' ||
        (ch == "/" && !stream.eol() && stream.peek() != " ")) {
      return chain(readQuoted(ch, "string", ch == '"' || ch == "`"), stream, state);
    } else if (ch == "%") {
      var style, embed = false;
      if (stream.eat("s")) style = "atom";
      else if (stream.eat(/[WQ]/)) { style = "string"; embed = true; }
      else if (stream.eat(/[wxqr]/)) style = "string";
      var delim = stream.eat(/[^\w\s]/);
      if (!delim) return "operator";
      if (matching.propertyIsEnumerable(delim)) delim = matching[delim];
      return chain(readQuoted(delim, style, embed, true), stream, state);
    } else if (ch == "#") {
      stream.skipToEnd();
      return "comment";
    } else if (ch == "<" && (m = stream.match(/^<-?[\`\"\']?([a-zA-Z_?]\w*)[\`\"\']?(?:;|$)/))) {
      return chain(readHereDoc(m[1]), stream, state);
    } else if (ch == "0") {
      if (stream.eat("x")) stream.eatWhile(/[\da-fA-F]/);
      else if (stream.eat("b")) stream.eatWhile(/[01]/);
      else stream.eatWhile(/[0-7]/);
      return "number";
    } else if (/\d/.test(ch)) {
      stream.match(/^[\d_]*(?:\.[\d_]+)?(?:[eE][+\-]?[\d_]+)?/);
      return "number";
    } else if (ch == "?") {
      while (stream.match(/^\\[CM]-/)) {}
      if (stream.eat("\\")) stream.eatWhile(/\w/);
      else stream.next();
      return "string";
    } else if (ch == ":") {
      if (stream.eat("'")) return chain(readQuoted("'", "atom", false), stream, state);
      if (stream.eat('"')) return chain(readQuoted('"', "atom", true), stream, state);
      stream.eatWhile(/[\w\?]/);
      return "atom";
    } else if (ch == "@") {
      stream.eat("@");
      stream.eatWhile(/[\w\?]/);
      return "variable-2";
    } else if (ch == "$") {
      stream.next();
      stream.eatWhile(/[\w\?]/);
      return "variable-3";
    } else if (/\w/.test(ch)) {
      stream.eatWhile(/[\w\?]/);
      if (stream.eat(":")) return "atom";
      return "ident";
    } else if (ch == "|" && (state.varList || state.lastTok == "{" || state.lastTok == "do")) {
      curPunc = "|";
      return null;
    } else if (/[\(\)\[\]{}\\;]/.test(ch)) {
      curPunc = ch;
      return null;
    } else if (ch == "-" && stream.eat(">")) {
      return "arrow";
    } else if (/[=+\-\/*:\.^%<>~|]/.test(ch)) {
      stream.eatWhile(/[=+\-\/*:\.^%<>~|]/);
      return "operator";
    } else {
      return null;
    }
  }

  function tokenBaseUntilBrace() {
    var depth = 1;
    return function(stream, state) {
      if (stream.peek() == "}") {
        depth--;
        if (depth == 0) {
          state.tokenize.pop();
          return state.tokenize[state.tokenize.length-1](stream, state);
        }
      } else if (stream.peek() == "{") {
        depth++;
      }
      return tokenBase(stream, state);
    };
  }
  function readQuoted(quote, style, embed, unescaped) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && (unescaped || !escaped)) {
          state.tokenize.pop();
          break;
        }
        if (embed && ch == "#" && !escaped && stream.eat("{")) {
          state.tokenize.push(tokenBaseUntilBrace(arguments.callee));
          break;
        }
        escaped = !escaped && ch == "\\";
      }
      return style;
    };
  }
  function readHereDoc(phrase) {
    return function(stream, state) {
      if (stream.match(phrase)) state.tokenize.pop();
      else stream.skipToEnd();
      return "string";
    };
  }
  function readBlockComment(stream, state) {
    if (stream.sol() && stream.match("=end") && stream.eol())
      state.tokenize.pop();
    stream.skipToEnd();
    return "comment";
  }

  return {
    startState: function() {
      return {tokenize: [tokenBase],
              indented: 0,
              context: {type: "top", indented: -config.indentUnit},
              continuedLine: false,
              lastTok: null,
              varList: false};
    },

    token: function(stream, state) {
      if (stream.sol()) state.indented = stream.indentation();
      var style = state.tokenize[state.tokenize.length-1](stream, state), kwtype;
      if (style == "ident") {
        var word = stream.current();
        style = keywords.propertyIsEnumerable(stream.current()) ? "keyword"
          : /^[A-Z]/.test(word) ? "tag"
          : (state.lastTok == "def" || state.lastTok == "class" || state.varList) ? "def"
          : "variable";
        if (indentWords.propertyIsEnumerable(word)) kwtype = "indent";
        else if (dedentWords.propertyIsEnumerable(word)) kwtype = "dedent";
        else if ((word == "if" || word == "unless") && stream.column() == stream.indentation())
          kwtype = "indent";
      }
      if (curPunc || (style && style != "comment")) state.lastTok = word || curPunc || style;
      if (curPunc == "|") state.varList = !state.varList;

      if (kwtype == "indent" || /[\(\[\{]/.test(curPunc))
        state.context = {prev: state.context, type: curPunc || style, indented: state.indented};
      else if ((kwtype == "dedent" || /[\)\]\}]/.test(curPunc)) && state.context.prev)
        state.context = state.context.prev;

      if (stream.eol())
        state.continuedLine = (curPunc == "\\" || style == "operator");
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize[state.tokenize.length-1] != tokenBase) return 0;
      var firstChar = textAfter && textAfter.charAt(0);
      var ct = state.context;
      var closing = ct.type == matching[firstChar] ||
        ct.type == "keyword" && /^(?:end|until|else|elsif|when|rescue)\b/.test(textAfter);
      return ct.indented + (closing ? 0 : config.indentUnit) +
        (state.continuedLine ? config.indentUnit : 0);
    },
     electricChars: "}de" // enD and rescuE

  };
});

CodeMirror.defineMIME("text/x-ruby", "ruby");

CodeMirror.defineMode("rust", function() {
  var indentUnit = 4, altIndentUnit = 2;
  var valKeywords = {
    "if": "if-style", "while": "if-style", "else": "else-style",
    "do": "else-style", "ret": "else-style", "fail": "else-style",
    "break": "atom", "cont": "atom", "const": "let", "resource": "fn",
    "let": "let", "fn": "fn", "for": "for", "alt": "alt", "iface": "iface",
    "impl": "impl", "type": "type", "enum": "enum", "mod": "mod",
    "as": "op", "true": "atom", "false": "atom", "assert": "op", "check": "op",
    "claim": "op", "native": "ignore", "unsafe": "ignore", "import": "else-style",
    "export": "else-style", "copy": "op", "log": "op", "log_err": "op",
    "use": "op", "bind": "op", "self": "atom"
  };
  var typeKeywords = function() {
    var keywords = {"fn": "fn", "block": "fn", "obj": "obj"};
    var atoms = "bool uint int i8 i16 i32 i64 u8 u16 u32 u64 float f32 f64 str char".split(" ");
    for (var i = 0, e = atoms.length; i < e; ++i) keywords[atoms[i]] = "atom";
    return keywords;
  }();
  var operatorChar = /[+\-*&%=<>!?|\.@]/;

  // Tokenizer

  // Used as scratch variable to communicate multiple values without
  // consing up tons of objects.
  var tcat, content;
  function r(tc, style) {
    tcat = tc;
    return style;
  }

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (ch == '"') {
      state.tokenize = tokenString;
      return state.tokenize(stream, state);
    }
    if (ch == "'") {
      tcat = "atom";
      if (stream.eat("\\")) {
        if (stream.skipTo("'")) { stream.next(); return "string"; }
        else { return "error"; }
      } else {
        stream.next();
        return stream.eat("'") ? "string" : "error";
      }
    }
    if (ch == "/") {
      if (stream.eat("/")) { stream.skipToEnd(); return "comment"; }
      if (stream.eat("*")) {
        state.tokenize = tokenComment(1);
        return state.tokenize(stream, state);
      }
    }
    if (ch == "#") {
      if (stream.eat("[")) { tcat = "open-attr"; return null; }
      stream.eatWhile(/\w/);
      return r("macro", "meta");
    }
    if (ch == ":" && stream.match(":<")) {
      return r("op", null);
    }
    if (ch.match(/\d/) || (ch == "." && stream.eat(/\d/))) {
      var flp = false;
      if (!stream.match(/^x[\da-f]+/i) && !stream.match(/^b[01]+/)) {
        stream.eatWhile(/\d/);
        if (stream.eat(".")) { flp = true; stream.eatWhile(/\d/); }
        if (stream.match(/^e[+\-]?\d+/i)) { flp = true; }
      }
      if (flp) stream.match(/^f(?:32|64)/);
      else stream.match(/^[ui](?:8|16|32|64)/);
      return r("atom", "number");
    }
    if (ch.match(/[()\[\]{}:;,]/)) return r(ch, null);
    if (ch == "-" && stream.eat(">")) return r("->", null);
    if (ch.match(operatorChar)) {
      stream.eatWhile(operatorChar);
      return r("op", null);
    }
    stream.eatWhile(/\w/);
    content = stream.current();
    if (stream.match(/^::\w/)) {
      stream.backUp(1);
      return r("prefix", "variable-2");
    }
    if (state.keywords.propertyIsEnumerable(content))
      return r(state.keywords[content], content.match(/true|false/) ? "atom" : "keyword");
    return r("name", "variable");
  }

  function tokenString(stream, state) {
    var ch, escaped = false;
    while (ch = stream.next()) {
      if (ch == '"' && !escaped) {
        state.tokenize = tokenBase;
        return r("atom", "string");
      }
      escaped = !escaped && ch == "\\";
    }
    // Hack to not confuse the parser when a string is split in
    // pieces.
    return r("op", "string");
  }

  function tokenComment(depth) {
    return function(stream, state) {
      var lastCh = null, ch;
      while (ch = stream.next()) {
        if (ch == "/" && lastCh == "*") {
          if (depth == 1) {
            state.tokenize = tokenBase;
            break;
          } else {
            state.tokenize = tokenComment(depth - 1);
            return state.tokenize(stream, state);
          }
        }
        if (ch == "*" && lastCh == "/") {
          state.tokenize = tokenComment(depth + 1);
          return state.tokenize(stream, state);
        }
        lastCh = ch;
      }
      return "comment";
    };
  }

  // Parser

  var cx = {state: null, stream: null, marked: null, cc: null};
  function pass() {
    for (var i = arguments.length - 1; i >= 0; i--) cx.cc.push(arguments[i]);
  }
  function cont() {
    pass.apply(null, arguments);
    return true;
  }

  function pushlex(type, info) {
    var result = function() {
      var state = cx.state;
      state.lexical = {indented: state.indented, column: cx.stream.column(),
                       type: type, prev: state.lexical, info: info};
    };
    result.lex = true;
    return result;
  }
  function poplex() {
    var state = cx.state;
    if (state.lexical.prev) {
      if (state.lexical.type == ")")
        state.indented = state.lexical.indented;
      state.lexical = state.lexical.prev;
    }
  }
  function typecx() { cx.state.keywords = typeKeywords; }
  function valcx() { cx.state.keywords = valKeywords; }
  poplex.lex = typecx.lex = valcx.lex = true;

  function commasep(comb, end) {
    function more(type) {
      if (type == ",") return cont(comb, more);
      if (type == end) return cont();
      return cont(more);
    }
    return function(type) {
      if (type == end) return cont();
      return pass(comb, more);
    };
  }

  function stat_of(comb, tag) {
    return cont(pushlex("stat", tag), comb, poplex, block);
  }
  function block(type) {
    if (type == "}") return cont();
    if (type == "let") return stat_of(letdef1, "let");
    if (type == "fn") return stat_of(fndef);
    if (type == "type") return cont(pushlex("stat"), tydef, endstatement, poplex, block);
    if (type == "enum") return stat_of(enumdef);
    if (type == "mod") return stat_of(mod);
    if (type == "iface") return stat_of(iface);
    if (type == "impl") return stat_of(impl);
    if (type == "open-attr") return cont(pushlex("]"), commasep(expression, "]"), poplex);
    if (type == "ignore" || type.match(/[\]\);,]/)) return cont(block);
    return pass(pushlex("stat"), expression, poplex, endstatement, block);
  }
  function endstatement(type) {
    if (type == ";") return cont();
    return pass();
  }
  function expression(type) {
    if (type == "atom" || type == "name") return cont(maybeop);
    if (type == "{") return cont(pushlex("}"), exprbrace, poplex);
    if (type.match(/[\[\(]/)) return matchBrackets(type, expression);
    if (type.match(/[\]\)\};,]/)) return pass();
    if (type == "if-style") return cont(expression, expression);
    if (type == "else-style" || type == "op") return cont(expression);
    if (type == "for") return cont(pattern, maybetype, inop, expression, expression);
    if (type == "alt") return cont(expression, altbody);
    if (type == "fn") return cont(fndef);
    if (type == "macro") return cont(macro);
    return cont();
  }
  function maybeop(type) {
    if (content == ".") return cont(maybeprop);
    if (content == "::<"){return cont(typarams, maybeop);}
    if (type == "op" || content == ":") return cont(expression);
    if (type == "(" || type == "[") return matchBrackets(type, expression);
    return pass();
  }
  function maybeprop(type) {
    if (content.match(/^\w+$/)) {cx.marked = "variable"; return cont(maybeop);}
    return pass(expression);
  }
  function exprbrace(type) {
    if (type == "op") {
      if (content == "|") return cont(blockvars, poplex, pushlex("}", "block"), block);
      if (content == "||") return cont(poplex, pushlex("}", "block"), block);
    }
    if (content == "mutable" || (content.match(/^\w+$/) && cx.stream.peek() == ":"
                                 && !cx.stream.match("::", false)))
      return pass(record_of(expression));
    return pass(block);
  }
  function record_of(comb) {
    function ro(type) {
      if (content == "mutable" || content == "with") {cx.marked = "keyword"; return cont(ro);}
      if (content.match(/^\w*$/)) {cx.marked = "variable"; return cont(ro);}
      if (type == ":") return cont(comb, ro);
      if (type == "}") return cont();
      return cont(ro);
    }
    return ro;
  }
  function blockvars(type) {
    if (type == "name") {cx.marked = "def"; return cont(blockvars);}
    if (type == "op" && content == "|") return cont();
    return cont(blockvars);
  }

  function letdef1(type) {
    if (type.match(/[\]\)\};]/)) return cont();
    if (content == "=") return cont(expression, letdef2);
    if (type == ",") return cont(letdef1);
    return pass(pattern, maybetype, letdef1);
  }
  function letdef2(type) {
    if (type.match(/[\]\)\};,]/)) return pass(letdef1);
    else return pass(expression, letdef2);
  }
  function maybetype(type) {
    if (type == ":") return cont(typecx, rtype, valcx);
    return pass();
  }
  function inop(type) {
    if (type == "name" && content == "in") {cx.marked = "keyword"; return cont();}
    return pass();
  }
  function fndef(type) {
    if (content == "@" || content == "~") {cx.marked = "keyword"; return cont(fndef);}
    if (type == "name") {cx.marked = "def"; return cont(fndef);}
    if (content == "<") return cont(typarams, fndef);
    if (type == "{") return pass(expression);
    if (type == "(") return cont(pushlex(")"), commasep(argdef, ")"), poplex, fndef);
    if (type == "->") return cont(typecx, rtype, valcx, fndef);
    if (type == ";") return cont();
    return cont(fndef);
  }
  function tydef(type) {
    if (type == "name") {cx.marked = "def"; return cont(tydef);}
    if (content == "<") return cont(typarams, tydef);
    if (content == "=") return cont(typecx, rtype, valcx);
    return cont(tydef);
  }
  function enumdef(type) {
    if (type == "name") {cx.marked = "def"; return cont(enumdef);}
    if (content == "<") return cont(typarams, enumdef);
    if (content == "=") return cont(typecx, rtype, valcx, endstatement);
    if (type == "{") return cont(pushlex("}"), typecx, enumblock, valcx, poplex);
    return cont(enumdef);
  }
  function enumblock(type) {
    if (type == "}") return cont();
    if (type == "(") return cont(pushlex(")"), commasep(rtype, ")"), poplex, enumblock);
    if (content.match(/^\w+$/)) cx.marked = "def";
    return cont(enumblock);
  }
  function mod(type) {
    if (type == "name") {cx.marked = "def"; return cont(mod);}
    if (type == "{") return cont(pushlex("}"), block, poplex);
    return pass();
  }
  function iface(type) {
    if (type == "name") {cx.marked = "def"; return cont(iface);}
    if (content == "<") return cont(typarams, iface);
    if (type == "{") return cont(pushlex("}"), block, poplex);
    return pass();
  }
  function impl(type) {
    if (content == "<") return cont(typarams, impl);
    if (content == "of" || content == "for") {cx.marked = "keyword"; return cont(rtype, impl);}
    if (type == "name") {cx.marked = "def"; return cont(impl);}
    if (type == "{") return cont(pushlex("}"), block, poplex);
    return pass();
  }
  function typarams(type) {
    if (content == ">") return cont();
    if (content == ",") return cont(typarams);
    if (content == ":") return cont(rtype, typarams);
    return pass(rtype, typarams);
  }
  function argdef(type) {
    if (type == "name") {cx.marked = "def"; return cont(argdef);}
    if (type == ":") return cont(typecx, rtype, valcx);
    return pass();
  }
  function rtype(type) {
    if (type == "name") {cx.marked = "variable-3"; return cont(rtypemaybeparam); }
    if (content == "mutable") {cx.marked = "keyword"; return cont(rtype);}
    if (type == "atom") return cont(rtypemaybeparam);
    if (type == "op" || type == "obj") return cont(rtype);
    if (type == "fn") return cont(fntype);
    if (type == "{") return cont(pushlex("{"), record_of(rtype), poplex);
    return matchBrackets(type, rtype);
  }
  function rtypemaybeparam(type) {
    if (content == "<") return cont(typarams);
    return pass();
  }
  function fntype(type) {
    if (type == "(") return cont(pushlex("("), commasep(rtype, ")"), poplex, fntype);
    if (type == "->") return cont(rtype);
    return pass();
  }
  function pattern(type) {
    if (type == "name") {cx.marked = "def"; return cont(patternmaybeop);}
    if (type == "atom") return cont(patternmaybeop);
    if (type == "op") return cont(pattern);
    if (type.match(/[\]\)\};,]/)) return pass();
    return matchBrackets(type, pattern);
  }
  function patternmaybeop(type) {
    if (type == "op" && content == ".") return cont();
    if (content == "to") {cx.marked = "keyword"; return cont(pattern);}
    else return pass();
  }
  function altbody(type) {
    if (type == "{") return cont(pushlex("}", "alt"), altblock1, poplex);
    return pass();
  }
  function altblock1(type) {
    if (type == "}") return cont();
    if (type == "|") return cont(altblock1);
    if (content == "when") {cx.marked = "keyword"; return cont(expression, altblock2);}
    if (type.match(/[\]\);,]/)) return cont(altblock1);
    return pass(pattern, altblock2);
  }
  function altblock2(type) {
    if (type == "{") return cont(pushlex("}", "alt"), block, poplex, altblock1);
    else return pass(altblock1);
  }

  function macro(type) {
    if (type.match(/[\[\(\{]/)) return matchBrackets(type, expression);
    return pass();
  }
  function matchBrackets(type, comb) {
    if (type == "[") return cont(pushlex("]"), commasep(comb, "]"), poplex);
    if (type == "(") return cont(pushlex(")"), commasep(comb, ")"), poplex);
    if (type == "{") return cont(pushlex("}"), commasep(comb, "}"), poplex);
    return cont();
  }

  function parse(state, stream, style) {
    var cc = state.cc;
    // Communicate our context to the combinators.
    // (Less wasteful than consing up a hundred closures on every call.)
    cx.state = state; cx.stream = stream; cx.marked = null, cx.cc = cc;

    while (true) {
      var combinator = cc.length ? cc.pop() : block;
      if (combinator(tcat)) {
        while(cc.length && cc[cc.length - 1].lex)
          cc.pop()();
        return cx.marked || style;
      }
    }
  }

  return {
    startState: function() {
      return {
        tokenize: tokenBase,
        cc: [],
        lexical: {indented: -indentUnit, column: 0, type: "top", align: false},
        keywords: valKeywords,
        indented: 0
      };
    },

    token: function(stream, state) {
      if (stream.sol()) {
        if (!state.lexical.hasOwnProperty("align"))
          state.lexical.align = false;
        state.indented = stream.indentation();
      }
      if (stream.eatSpace()) return null;
      tcat = content = null;
      var style = state.tokenize(stream, state);
      if (style == "comment") return style;
      if (!state.lexical.hasOwnProperty("align"))
        state.lexical.align = true;
      if (tcat == "prefix") return style;
      if (!content) content = stream.current();
      return parse(state, stream, style);
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase) return 0;
      var firstChar = textAfter && textAfter.charAt(0), lexical = state.lexical,
          type = lexical.type, closing = firstChar == type;
      if (type == "stat") return lexical.indented + indentUnit;
      if (lexical.align) return lexical.column + (closing ? 0 : 1);
      return lexical.indented + (closing ? 0 : (lexical.info == "alt" ? altIndentUnit : indentUnit));
    },

    electricChars: "{}"
  };
});

CodeMirror.defineMIME("text/x-rustsrc", "rust");
/**
 * Author: Koh Zi Han, based on implementation by Koh Zi Chun
 */
CodeMirror.defineMode("scheme", function (config, mode) {
    var BUILTIN = "builtin", COMMENT = "comment", STRING = "string",
        ATOM = "atom", NUMBER = "number", BRACKET = "bracket", KEYWORD="keyword";
    var INDENT_WORD_SKIP = 2, KEYWORDS_SKIP = 1;

    function makeKeywords(str) {
        var obj = {}, words = str.split(" ");
        for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
        return obj;
    }

    var keywords = makeKeywords("λ case-lambda call/cc class define-class exit-handler field import inherit init-field interface let*-values let-values let/ec mixin opt-lambda override protect provide public rename require require-for-syntax syntax syntax-case syntax-error unit/sig unless when with-syntax and begin call-with-current-continuation call-with-input-file call-with-output-file case cond define define-syntax delay do dynamic-wind else for-each if lambda let let* let-syntax letrec letrec-syntax map or syntax-rules abs acos angle append apply asin assoc assq assv atan boolean? caar cadr call-with-input-file call-with-output-file call-with-values car cdddar cddddr cdr ceiling char->integer char-alphabetic? char-ci<=? char-ci<? char-ci=? char-ci>=? char-ci>? char-downcase char-lower-case? char-numeric? char-ready? char-upcase char-upper-case? char-whitespace? char<=? char<? char=? char>=? char>? char? close-input-port close-output-port complex? cons cos current-input-port current-output-port denominator display eof-object? eq? equal? eqv? eval even? exact->inexact exact? exp expt #f floor force gcd imag-part inexact->exact inexact? input-port? integer->char integer? interaction-environment lcm length list list->string list->vector list-ref list-tail list? load log magnitude make-polar make-rectangular make-string make-vector max member memq memv min modulo negative? newline not null-environment null? number->string number? numerator odd? open-input-file open-output-file output-port? pair? peek-char port? positive? procedure? quasiquote quote quotient rational? rationalize read read-char real-part real? remainder reverse round scheme-report-environment set! set-car! set-cdr! sin sqrt string string->list string->number string->symbol string-append string-ci<=? string-ci<? string-ci=? string-ci>=? string-ci>? string-copy string-fill! string-length string-ref string-set! string<=? string<? string=? string>=? string>? string? substring symbol->string symbol? #t tan transcript-off transcript-on truncate values vector vector->list vector-fill! vector-length vector-ref vector-set! with-input-from-file with-output-to-file write write-char zero?");
    var indentKeys = makeKeywords("define let letrec let* lambda");

    function stateStack(indent, type, prev) { // represents a state stack object
        this.indent = indent;
        this.type = type;
        this.prev = prev;
    }

    function pushStack(state, indent, type) {
        state.indentStack = new stateStack(indent, type, state.indentStack);
    }

    function popStack(state) {
        state.indentStack = state.indentStack.prev;
    }

    var binaryMatcher = new RegExp(/^(?:[-+]i|[-+][01]+#*(?:\/[01]+#*)?i|[-+]?[01]+#*(?:\/[01]+#*)?@[-+]?[01]+#*(?:\/[01]+#*)?|[-+]?[01]+#*(?:\/[01]+#*)?[-+](?:[01]+#*(?:\/[01]+#*)?)?i|[-+]?[01]+#*(?:\/[01]+#*)?)(?=[()\s;"]|$)/i);
    var octalMatcher = new RegExp(/^(?:[-+]i|[-+][0-7]+#*(?:\/[0-7]+#*)?i|[-+]?[0-7]+#*(?:\/[0-7]+#*)?@[-+]?[0-7]+#*(?:\/[0-7]+#*)?|[-+]?[0-7]+#*(?:\/[0-7]+#*)?[-+](?:[0-7]+#*(?:\/[0-7]+#*)?)?i|[-+]?[0-7]+#*(?:\/[0-7]+#*)?)(?=[()\s;"]|$)/i);
    var hexMatcher = new RegExp(/^(?:[-+]i|[-+][\da-f]+#*(?:\/[\da-f]+#*)?i|[-+]?[\da-f]+#*(?:\/[\da-f]+#*)?@[-+]?[\da-f]+#*(?:\/[\da-f]+#*)?|[-+]?[\da-f]+#*(?:\/[\da-f]+#*)?[-+](?:[\da-f]+#*(?:\/[\da-f]+#*)?)?i|[-+]?[\da-f]+#*(?:\/[\da-f]+#*)?)(?=[()\s;"]|$)/i);
    var decimalMatcher = new RegExp(/^(?:[-+]i|[-+](?:(?:(?:\d+#+\.?#*|\d+\.\d*#*|\.\d+#*|\d+)(?:[esfdl][-+]?\d+)?)|\d+#*\/\d+#*)i|[-+]?(?:(?:(?:\d+#+\.?#*|\d+\.\d*#*|\.\d+#*|\d+)(?:[esfdl][-+]?\d+)?)|\d+#*\/\d+#*)@[-+]?(?:(?:(?:\d+#+\.?#*|\d+\.\d*#*|\.\d+#*|\d+)(?:[esfdl][-+]?\d+)?)|\d+#*\/\d+#*)|[-+]?(?:(?:(?:\d+#+\.?#*|\d+\.\d*#*|\.\d+#*|\d+)(?:[esfdl][-+]?\d+)?)|\d+#*\/\d+#*)[-+](?:(?:(?:\d+#+\.?#*|\d+\.\d*#*|\.\d+#*|\d+)(?:[esfdl][-+]?\d+)?)|\d+#*\/\d+#*)?i|(?:(?:(?:\d+#+\.?#*|\d+\.\d*#*|\.\d+#*|\d+)(?:[esfdl][-+]?\d+)?)|\d+#*\/\d+#*))(?=[()\s;"]|$)/i);

    function isBinaryNumber (stream) {
        return stream.match(binaryMatcher);
    }

    function isOctalNumber (stream) {
        return stream.match(octalMatcher);
    }

    function isDecimalNumber (stream, backup) {
        if (backup === true) {
            stream.backUp(1);
        }
        return stream.match(decimalMatcher);
    }

    function isHexNumber (stream) {
        return stream.match(hexMatcher);
    }

    return {
        startState: function () {
            return {
                indentStack: null,
                indentation: 0,
                mode: false,
                sExprComment: false
            };
        },

        token: function (stream, state) {
            if (state.indentStack == null && stream.sol()) {
                // update indentation, but only if indentStack is empty
                state.indentation = stream.indentation();
            }

            // skip spaces
            if (stream.eatSpace()) {
                return null;
            }
            var returnType = null;

            switch(state.mode){
                case "string": // multi-line string parsing mode
                    var next, escaped = false;
                    while ((next = stream.next()) != null) {
                        if (next == "\"" && !escaped) {

                            state.mode = false;
                            break;
                        }
                        escaped = !escaped && next == "\\";
                    }
                    returnType = STRING; // continue on in scheme-string mode
                    break;
                case "comment": // comment parsing mode
                    var next, maybeEnd = false;
                    while ((next = stream.next()) != null) {
                        if (next == "#" && maybeEnd) {

                            state.mode = false;
                            break;
                        }
                        maybeEnd = (next == "|");
                    }
                    returnType = COMMENT;
                    break;
                case "s-expr-comment": // s-expr commenting mode
                    state.mode = false;
                    if(stream.peek() == "(" || stream.peek() == "["){
                        // actually start scheme s-expr commenting mode
                        state.sExprComment = 0;
                    }else{
                        // if not we just comment the entire of the next token
                        stream.eatWhile(/[^/s]/); // eat non spaces
                        returnType = COMMENT;
                        break;
                    }
                default: // default parsing mode
                    var ch = stream.next();

                    if (ch == "\"") {
                        state.mode = "string";
                        returnType = STRING;

                    } else if (ch == "'") {
                        returnType = ATOM;
                    } else if (ch == '#') {
                        if (stream.eat("|")) {                    // Multi-line comment
                            state.mode = "comment"; // toggle to comment mode
                            returnType = COMMENT;
                        } else if (stream.eat(/[tf]/i)) {            // #t/#f (atom)
                            returnType = ATOM;
                        } else if (stream.eat(';')) {                // S-Expr comment
                            state.mode = "s-expr-comment";
                            returnType = COMMENT;
                        } else {
                            var numTest = null, hasExactness = false, hasRadix = true;
                            if (stream.eat(/[ei]/i)) {
                                hasExactness = true;
                            } else {
                                stream.backUp(1);       // must be radix specifier
                            }
                            if (stream.match(/^#b/i)) {
                                numTest = isBinaryNumber;
                            } else if (stream.match(/^#o/i)) {
                                numTest = isOctalNumber;
                            } else if (stream.match(/^#x/i)) {
                                numTest = isHexNumber;
                            } else if (stream.match(/^#d/i)) {
                                numTest = isDecimalNumber;
                            } else if (stream.match(/^[-+0-9.]/, false)) {
                                hasRadix = false;
                                numTest = isDecimalNumber;
                            // re-consume the intial # if all matches failed
                            } else if (!hasExactness) {
                                stream.eat('#');
                            }
                            if (numTest != null) {
                                if (hasRadix && !hasExactness) {
                                    // consume optional exactness after radix
                                    stream.match(/^#[ei]/i);
                                }
                                if (numTest(stream))
                                    returnType = NUMBER;
                            }
                        }
                    } else if (/^[-+0-9.]/.test(ch) && isDecimalNumber(stream, true)) { // match non-prefixed number, must be decimal
                        returnType = NUMBER;
                    } else if (ch == ";") { // comment
                        stream.skipToEnd(); // rest of the line is a comment
                        returnType = COMMENT;
                    } else if (ch == "(" || ch == "[") {
                        var keyWord = ''; var indentTemp = stream.column();
                        /**
                        Either
                        (indent-word ..
                        (non-indent-word ..
                        (;something else, bracket, etc.
                        */

                        while ((letter = stream.eat(/[^\s\(\[\;\)\]]/)) != null) {
                            keyWord += letter;
                        }

                        if (keyWord.length > 0 && indentKeys.propertyIsEnumerable(keyWord)) { // indent-word

                            pushStack(state, indentTemp + INDENT_WORD_SKIP, ch);
                        } else { // non-indent word
                            // we continue eating the spaces
                            stream.eatSpace();
                            if (stream.eol() || stream.peek() == ";") {
                                // nothing significant after
                                // we restart indentation 1 space after
                                pushStack(state, indentTemp + 1, ch);
                            } else {
                                pushStack(state, indentTemp + stream.current().length, ch); // else we match
                            }
                        }
                        stream.backUp(stream.current().length - 1); // undo all the eating

                        if(typeof state.sExprComment == "number") state.sExprComment++;

                        returnType = BRACKET;
                    } else if (ch == ")" || ch == "]") {
                        returnType = BRACKET;
                        if (state.indentStack != null && state.indentStack.type == (ch == ")" ? "(" : "[")) {
                            popStack(state);

                            if(typeof state.sExprComment == "number"){
                                if(--state.sExprComment == 0){
                                    returnType = COMMENT; // final closing bracket
                                    state.sExprComment = false; // turn off s-expr commenting mode
                                }
                            }
                        }
                    } else {
                        stream.eatWhile(/[\w\$_\-!$%&*+\.\/:<=>?@\^~]/);

                        if (keywords && keywords.propertyIsEnumerable(stream.current())) {
                            returnType = BUILTIN;
                        } else returnType = "variable";
                    }
            }
            return (typeof state.sExprComment == "number") ? COMMENT : returnType;
        },

        indent: function (state, textAfter) {
            if (state.indentStack == null) return state.indentation;
            return state.indentStack.indent;
        }
    };
});

CodeMirror.defineMIME("text/x-scheme", "scheme");
CodeMirror.defineMode('shell', function(config) {

  var atoms = ['true','false'],
      keywords = ['if','then','do','else','elif','while','until','for','in','esac','fi','fin','fil','done','exit','set','unset','export','function'],
      commands = ['ab','awk','bash','beep','cat','cc','cd','chown','chmod','chroot','clear','cp','curl','cut','diff','echo','find','gawk','gcc','get','git','grep','kill','killall','ls','make','mkdir','openssl','mv','nc','node','npm','ping','ps','restart','rm','rmdir','sed','service','sh','shopt','shred','source','sort','sleep','ssh','start','stop','su','sudo','tee','telnet','top','touch','vi','vim','wall','wc','wget','who','write','yes','zsh'];

  function tokenBase(stream, state) {

    var sol = stream.sol();
    var ch = stream.next();

    if (ch === '\'' || ch === '"' || ch === '`') {
      state.tokens.unshift(tokenString(ch));
      return tokenize(stream, state);
    }
    if (ch === '#') {
      if (sol && stream.eat('!')) {
        stream.skipToEnd();
        return 'meta'; // 'comment'?
      }
      stream.skipToEnd();
      return 'comment';
    }
    if (ch === '$') {
      state.tokens.unshift(tokenDollar);
      return tokenize(stream, state);
    }
    if (ch === '+' || ch === '=') {
      return 'operator';
    }
    if (ch === '-') {
      stream.eat('-');
      stream.eatWhile(/\w/);
      return 'attribute';
    }
    if (/\d/.test(ch)) {
      stream.eatWhile(/\d/);
      if(!/\w/.test(stream.peek())) {
        return 'number';
      }
    }
    stream.eatWhile(/\w/);
    var cur = stream.current();
    if (stream.peek() === '=' && /\w+/.test(cur)) return 'def';
    if (atoms.indexOf(cur) !== -1) return 'atom';
    if (commands.indexOf(cur) !== -1) return 'builtin';
    if (keywords.indexOf(cur) !== -1) return 'keyword';
    return 'word';
  }

  function tokenString(quote) {
    return function(stream, state) {
      var next, end = false, escaped = false;
      while ((next = stream.next()) != null) {
        if (next === quote && !escaped) {
          end = true;
          break;
        }
        if (next === '$' && !escaped && quote !== '\'') {
          escaped = true;
          stream.backUp(1);
          state.tokens.unshift(tokenDollar);
          break;
        }
        escaped = !escaped && next === '\\';
      }
      if (end || !escaped) {
        state.tokens.shift();
      }
      return (quote === '`' || quote === ')' ? 'quote' : 'string');
    };
  };

  var tokenDollar = function(stream, state) {
    if (state.tokens.length > 1) stream.eat('$');
    var ch = stream.next(), hungry = /\w/;
    if (ch === '{') hungry = /[^}]/;
    if (ch === '(') {
      state.tokens[0] = tokenString(')');
      return tokenize(stream, state);
    }
    if (!/\d/.test(ch)) {
      stream.eatWhile(hungry);
      stream.eat('}');
    }
    state.tokens.shift();
    return 'def';
  };

  function tokenize(stream, state) {
    return (state.tokens[0] || tokenBase) (stream, state);
  };

  return {
    startState: function() {return {tokens:[]}},
    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      return tokenize(stream, state);
    }
  };
});
  
CodeMirror.defineMIME('text/x-sh', 'shell');
CodeMirror.defineMode('smalltalk', function(config, modeConfig) {

	var specialChars = /[+\-/\\*~<>=@%|&?!.:;^]/;
	var keywords = /true|false|nil|self|super|thisContext/;

	var Context = function(tokenizer, parent) {
		this.next = tokenizer;
		this.parent = parent;
	};

	var Token = function(name, context, eos) {
		this.name = name;
		this.context = context;
		this.eos = eos;
	};

	var State = function() {
		this.context = new Context(next, null);
		this.expectVariable = true;
		this.indentation = 0;
		this.userIndentationDelta = 0;
	};

	State.prototype.userIndent = function(indentation) {
		this.userIndentationDelta = indentation > 0 ? (indentation / config.indentUnit - this.indentation) : 0;
	};

	var next = function(stream, context, state) {
		var token = new Token(null, context, false);
		var aChar = stream.next();

		if (aChar === '"') {
			token = nextComment(stream, new Context(nextComment, context));

		} else if (aChar === '\'') {
			token = nextString(stream, new Context(nextString, context));

		} else if (aChar === '#') {
			stream.eatWhile(/[^ .]/);
			token.name = 'string-2';

		} else if (aChar === '$') {
			stream.eatWhile(/[^ ]/);
			token.name = 'string-2';

		} else if (aChar === '|' && state.expectVariable) {
			token.context = new Context(nextTemporaries, context);

		} else if (/[\[\]{}()]/.test(aChar)) {
			token.name = 'bracket';
			token.eos = /[\[{(]/.test(aChar);

			if (aChar === '[') {
				state.indentation++;
			} else if (aChar === ']') {
				state.indentation = Math.max(0, state.indentation - 1);
			}

		} else if (specialChars.test(aChar)) {
			stream.eatWhile(specialChars);
			token.name = 'operator';
			token.eos = aChar !== ';'; // ; cascaded message expression

		} else if (/\d/.test(aChar)) {
			stream.eatWhile(/[\w\d]/);
			token.name = 'number'

		} else if (/[\w_]/.test(aChar)) {
			stream.eatWhile(/[\w\d_]/);
			token.name = state.expectVariable ? (keywords.test(stream.current()) ? 'keyword' : 'variable') : null;

		} else {
			token.eos = state.expectVariable;
		}

		return token;
	};

	var nextComment = function(stream, context) {
		stream.eatWhile(/[^"]/);
		return new Token('comment', stream.eat('"') ? context.parent : context, true);
	};

	var nextString = function(stream, context) {
		stream.eatWhile(/[^']/);
		return new Token('string', stream.eat('\'') ? context.parent : context, false);
	};

	var nextTemporaries = function(stream, context, state) {
		var token = new Token(null, context, false);
		var aChar = stream.next();

		if (aChar === '|') {
			token.context = context.parent;
			token.eos = true;

		} else {
			stream.eatWhile(/[^|]/);
			token.name = 'variable';
		}

		return token;
	}

	return {
		startState: function() {
			return new State;
		},

		token: function(stream, state) {
			state.userIndent(stream.indentation());

			if (stream.eatSpace()) {
				return null;
			}

			var token = state.context.next(stream, state.context, state);
			state.context = token.context;
			state.expectVariable = token.eos;

			state.lastToken = token;
			return token.name;
		},

		blankLine: function(state) {
			state.userIndent(0);
		},

		indent: function(state, textAfter) {
			var i = state.context.next === next && textAfter && textAfter.charAt(0) === ']' ? -1 : state.userIndentationDelta;
			return (state.indentation + i) * config.indentUnit;
		},

		electricChars: ']'
	};

});

CodeMirror.defineMIME('text/x-stsrc', {name: 'smalltalk'});CodeMirror.defineMode("smarty", function(config, parserConfig) {
  var keyFuncs = ["debug", "extends", "function", "include", "literal"];
  var last;
  var regs = {
    operatorChars: /[+\-*&%=<>!?]/,
    validIdentifier: /[a-zA-Z0-9\_]/,
    stringChar: /[\'\"]/
  }
  var leftDelim = (typeof config.mode.leftDelimiter != 'undefined') ? config.mode.leftDelimiter : "{";
  var rightDelim = (typeof config.mode.rightDelimiter != 'undefined') ? config.mode.rightDelimiter : "}";
  function ret(style, lst) { last = lst; return style; }


  function tokenizer(stream, state) {
    function chain(parser) {
      state.tokenize = parser;
      return parser(stream, state);
    }

    if (stream.match(leftDelim, true)) {
      if (stream.eat("*")) {
        return chain(inBlock("comment", "*" + rightDelim));
      }
      else {
        state.tokenize = inSmarty;
        return "tag";
      }
    }
    else {
      // I'd like to do an eatWhile() here, but I can't get it to eat only up to the rightDelim string/char
      stream.next();
      return null;
    }
  }

  function inSmarty(stream, state) {
    if (stream.match(rightDelim, true)) {
      state.tokenize = tokenizer;
      return ret("tag", null);
    }

    var ch = stream.next();
    if (ch == "$") {
      stream.eatWhile(regs.validIdentifier);
      return ret("variable-2", "variable");
    }
    else if (ch == ".") {
      return ret("operator", "property");
    }
    else if (regs.stringChar.test(ch)) {
      state.tokenize = inAttribute(ch);
      return ret("string", "string");
    }
    else if (regs.operatorChars.test(ch)) {
      stream.eatWhile(regs.operatorChars);
      return ret("operator", "operator");
    }
    else if (ch == "[" || ch == "]") {
      return ret("bracket", "bracket");
    }
    else if (/\d/.test(ch)) {
      stream.eatWhile(/\d/);
      return ret("number", "number");
    }
    else {
      if (state.last == "variable") {
        if (ch == "@") {
          stream.eatWhile(regs.validIdentifier);
          return ret("property", "property");
        }
        else if (ch == "|") {
          stream.eatWhile(regs.validIdentifier);
          return ret("qualifier", "modifier");
        }
      }
      else if (state.last == "whitespace") {
        stream.eatWhile(regs.validIdentifier);
        return ret("attribute", "modifier");
      }
      else if (state.last == "property") {
        stream.eatWhile(regs.validIdentifier);
        return ret("property", null);
      }
      else if (/\s/.test(ch)) {
        last = "whitespace";
        return null;
      }

      var str = "";
      if (ch != "/") {
    	str += ch;
      }
      var c = "";
      while ((c = stream.eat(regs.validIdentifier))) {
        str += c;
      }
      var i, j;
      for (i=0, j=keyFuncs.length; i<j; i++) {
        if (keyFuncs[i] == str) {
          return ret("keyword", "keyword");
        }
      }
      if (/\s/.test(ch)) {
    	return null;
      }
      return ret("tag", "tag");
    }
  }

  function inAttribute(quote) {
    return function(stream, state) {
      while (!stream.eol()) {
        if (stream.next() == quote) {
          state.tokenize = inSmarty;
          break;
        }
      }
      return "string";
    };
  }

  function inBlock(style, terminator) {
    return function(stream, state) {
      while (!stream.eol()) {
        if (stream.match(terminator)) {
          state.tokenize = tokenizer;
          break;
        }
        stream.next();
      }
      return style;
    };
  }

  return {
    startState: function() {
      return { tokenize: tokenizer, mode: "smarty", last: null };
    },
    token: function(stream, state) {
      var style = state.tokenize(stream, state);
      state.last = last;
      return style;
    },
    electricChars: ""
  }
});

CodeMirror.defineMIME("text/x-smarty", "smarty");CodeMirror.defineMode("sparql", function(config) {
  var indentUnit = config.indentUnit;
  var curPunc;

  function wordRegexp(words) {
    return new RegExp("^(?:" + words.join("|") + ")$", "i");
  }
  var ops = wordRegexp(["str", "lang", "langmatches", "datatype", "bound", "sameterm", "isiri", "isuri",
                        "isblank", "isliteral", "union", "a"]);
  var keywords = wordRegexp(["base", "prefix", "select", "distinct", "reduced", "construct", "describe",
                             "ask", "from", "named", "where", "order", "limit", "offset", "filter", "optional",
                             "graph", "by", "asc", "desc"]);
  var operatorChars = /[*+\-<>=&|]/;

  function tokenBase(stream, state) {
    var ch = stream.next();
    curPunc = null;
    if (ch == "$" || ch == "?") {
      stream.match(/^[\w\d]*/);
      return "variable-2";
    }
    else if (ch == "<" && !stream.match(/^[\s\u00a0=]/, false)) {
      stream.match(/^[^\s\u00a0>]*>?/);
      return "atom";
    }
    else if (ch == "\"" || ch == "'") {
      state.tokenize = tokenLiteral(ch);
      return state.tokenize(stream, state);
    }
    else if (/[{}\(\),\.;\[\]]/.test(ch)) {
      curPunc = ch;
      return null;
    }
    else if (ch == "#") {
      stream.skipToEnd();
      return "comment";
    }
    else if (operatorChars.test(ch)) {
      stream.eatWhile(operatorChars);
      return null;
    }
    else if (ch == ":") {
      stream.eatWhile(/[\w\d\._\-]/);
      return "atom";
    }
    else {
      stream.eatWhile(/[_\w\d]/);
      if (stream.eat(":")) {
        stream.eatWhile(/[\w\d_\-]/);
        return "atom";
      }
      var word = stream.current(), type;
      if (ops.test(word))
        return null;
      else if (keywords.test(word))
        return "keyword";
      else
        return "variable";
    }
  }

  function tokenLiteral(quote) {
    return function(stream, state) {
      var escaped = false, ch;
      while ((ch = stream.next()) != null) {
        if (ch == quote && !escaped) {
          state.tokenize = tokenBase;
          break;
        }
        escaped = !escaped && ch == "\\";
      }
      return "string";
    };
  }

  function pushContext(state, type, col) {
    state.context = {prev: state.context, indent: state.indent, col: col, type: type};
  }
  function popContext(state) {
    state.indent = state.context.indent;
    state.context = state.context.prev;
  }

  return {
    startState: function(base) {
      return {tokenize: tokenBase,
              context: null,
              indent: 0,
              col: 0};
    },

    token: function(stream, state) {
      if (stream.sol()) {
        if (state.context && state.context.align == null) state.context.align = false;
        state.indent = stream.indentation();
      }
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);

      if (style != "comment" && state.context && state.context.align == null && state.context.type != "pattern") {
        state.context.align = true;
      }

      if (curPunc == "(") pushContext(state, ")", stream.column());
      else if (curPunc == "[") pushContext(state, "]", stream.column());
      else if (curPunc == "{") pushContext(state, "}", stream.column());
      else if (/[\]\}\)]/.test(curPunc)) {
        while (state.context && state.context.type == "pattern") popContext(state);
        if (state.context && curPunc == state.context.type) popContext(state);
      }
      else if (curPunc == "." && state.context && state.context.type == "pattern") popContext(state);
      else if (/atom|string|variable/.test(style) && state.context) {
        if (/[\}\]]/.test(state.context.type))
          pushContext(state, "pattern", stream.column());
        else if (state.context.type == "pattern" && !state.context.align) {
          state.context.align = true;
          state.context.col = stream.column();
        }
      }
      
      return style;
    },

    indent: function(state, textAfter) {
      var firstChar = textAfter && textAfter.charAt(0);
      var context = state.context;
      if (/[\]\}]/.test(firstChar))
        while (context && context.type == "pattern") context = context.prev;

      var closing = context && firstChar == context.type;
      if (!context)
        return 0;
      else if (context.type == "pattern")
        return context.col;
      else if (context.align)
        return context.col + (closing ? 0 : 1);
      else
        return context.indent + (closing ? 0 : indentUnit);
    }
  };
});

CodeMirror.defineMIME("application/x-sparql-query", "sparql");
/*
 * Author: Constantin Jucovschi (c.jucovschi@jacobs-university.de)
 * Licence: MIT
 */

CodeMirror.defineMode("stex", function(cmCfg, modeCfg) 
{    
    function pushCommand(state, command) {
	state.cmdState.push(command);
    }

    function peekCommand(state) { 
	if (state.cmdState.length>0)
	    return state.cmdState[state.cmdState.length-1];
	else
	    return null;
    }

    function popCommand(state) {
	if (state.cmdState.length>0) {
	    var plug = state.cmdState.pop();
	    plug.closeBracket();
	}	    
    }

    function applyMostPowerful(state) {
      var context = state.cmdState;
      for (var i = context.length - 1; i >= 0; i--) {
	  var plug = context[i];
	  if (plug.name=="DEFAULT")
	      continue;
	  return plug.styleIdentifier();
      }
      return null;
    }

    function addPluginPattern(pluginName, cmdStyle, brackets, styles) {
	return function () {
	    this.name=pluginName;
	    this.bracketNo = 0;
	    this.style=cmdStyle;
	    this.styles = styles;
	    this.brackets = brackets;

	    this.styleIdentifier = function(content) {
		if (this.bracketNo<=this.styles.length)
		    return this.styles[this.bracketNo-1];
		else
		    return null;
	    };
	    this.openBracket = function(content) {
		this.bracketNo++;
		return "bracket";
	    };
	    this.closeBracket = function(content) {
	    };
	}
    }

    var plugins = new Array();
   
    plugins["importmodule"] = addPluginPattern("importmodule", "tag", "{[", ["string", "builtin"]);
    plugins["documentclass"] = addPluginPattern("documentclass", "tag", "{[", ["", "atom"]);
    plugins["usepackage"] = addPluginPattern("documentclass", "tag", "[", ["atom"]);
    plugins["begin"] = addPluginPattern("documentclass", "tag", "[", ["atom"]);
    plugins["end"] = addPluginPattern("documentclass", "tag", "[", ["atom"]);

    plugins["DEFAULT"] = function () {
	this.name="DEFAULT";
	this.style="tag";

	this.styleIdentifier = function(content) {
	};
	this.openBracket = function(content) {
	};
	this.closeBracket = function(content) {
	};
    };

    function setState(state, f) {
	state.f = f;
    }

    function normal(source, state) {
	if (source.match(/^\\[a-zA-Z@]+/)) {
	    var cmdName = source.current();
	    cmdName = cmdName.substr(1, cmdName.length-1);
            var plug;
            if (plugins.hasOwnProperty(cmdName)) {
	      plug = plugins[cmdName];
            } else {
              plug = plugins["DEFAULT"];
            }
	    plug = new plug();
	    pushCommand(state, plug);
	    setState(state, beginParams);
	    return plug.style;
	}

        // escape characters 
        if (source.match(/^\\[$&%#{}_]/)) {
          return "tag";
        }

        // white space control characters
        if (source.match(/^\\[,;!\/]/)) {
          return "tag";
        }

	var ch = source.next();
	if (ch == "%") {
            // special case: % at end of its own line; stay in same state
            if (!source.eol()) {
              setState(state, inCComment);
            }
	    return "comment";
	} 
	else if (ch=='}' || ch==']') {
	    plug = peekCommand(state);
	    if (plug) {
		plug.closeBracket(ch);
		setState(state, beginParams);
	    } else
		return "error";
	    return "bracket";
	} else if (ch=='{' || ch=='[') {
	    plug = plugins["DEFAULT"];	    
	    plug = new plug();
	    pushCommand(state, plug);
	    return "bracket";	    
	}
	else if (/\d/.test(ch)) {
	    source.eatWhile(/[\w.%]/);
	    return "atom";
	}
	else {
	    source.eatWhile(/[\w-_]/);
	    return applyMostPowerful(state);
	}
    }

    function inCComment(source, state) {
	source.skipToEnd();
	setState(state, normal);
	return "comment";
    }

    function beginParams(source, state) {
	var ch = source.peek();
	if (ch == '{' || ch == '[') {
	   var lastPlug = peekCommand(state);
	   var style = lastPlug.openBracket(ch);
	   source.eat(ch);
	   setState(state, normal);
	   return "bracket";
	}
	if (/[ \t\r]/.test(ch)) {
	    source.eat(ch);
	    return null;
	}
	setState(state, normal);
	lastPlug = peekCommand(state);
	if (lastPlug) {
	    popCommand(state);
	}
        return normal(source, state);
    }

    return {
     startState: function() { return { f:normal, cmdState:[] }; },
	 copyState: function(s) { return { f: s.f, cmdState: s.cmdState.slice(0, s.cmdState.length) }; },
	 
	 token: function(stream, state) {
	 var t = state.f(stream, state);
	 var w = stream.current();
	 return t;
     }
 };
});


CodeMirror.defineMIME("text/x-stex", "stex");
/***
|''Name''|tiddlywiki.js|
|''Description''|Enables TiddlyWikiy syntax highlighting using CodeMirror|
|''Author''|PMario|
|''Version''|0.1.7|
|''Status''|''stable''|
|''Source''|[[GitHub|https://github.com/pmario/CodeMirror2/blob/tw-syntax/mode/tiddlywiki]]|
|''Documentation''|http://codemirror.tiddlyspace.com/|
|''License''|[[MIT License|http://www.opensource.org/licenses/mit-license.php]]|
|''CoreVersion''|2.5.0|
|''Requires''|codemirror.js|
|''Keywords''|syntax highlighting color code mirror codemirror|
! Info
CoreVersion parameter is needed for TiddlyWiki only!
***/
//{{{
CodeMirror.defineMode("tiddlywiki", function (config, parserConfig) {
	var indentUnit = config.indentUnit;

	// Tokenizer
	var textwords = function () {
		function kw(type) {
			return {
				type: type,
				style: "text"
			};
		}
		return {};
	}();

	var keywords = function () {
		function kw(type) {
			return { type: type, style: "macro"};
		}
		return {
			"allTags": kw('allTags'), "closeAll": kw('closeAll'), "list": kw('list'),
			"newJournal": kw('newJournal'), "newTiddler": kw('newTiddler'),
			"permaview": kw('permaview'), "saveChanges": kw('saveChanges'),
			"search": kw('search'), "slider": kw('slider'),	"tabs": kw('tabs'),
			"tag": kw('tag'), "tagging": kw('tagging'),	"tags": kw('tags'),
			"tiddler": kw('tiddler'), "timeline": kw('timeline'),
			"today": kw('today'), "version": kw('version'),	"option": kw('option'),

			"with": kw('with'),
			"filter": kw('filter')
		};
	}();

	var isSpaceName = /[\w_\-]/i,
		reHR = /^\-\-\-\-+$/,					// <hr>
		reWikiCommentStart = /^\/\*\*\*$/,		// /***
		reWikiCommentStop = /^\*\*\*\/$/,		// ***/
		reBlockQuote = /^<<<$/,

		reJsCodeStart = /^\/\/\{\{\{$/,			// //{{{ js block start
		reJsCodeStop = /^\/\/\}\}\}$/,			// //}}} js stop
		reXmlCodeStart = /^<!--\{\{\{-->$/,		// xml block start
		reXmlCodeStop = /^<!--\}\}\}-->$/,		// xml stop

		reCodeBlockStart = /^\{\{\{$/,			// {{{ TW text div block start
		reCodeBlockStop = /^\}\}\}$/,			// }}} TW text stop

		reCodeStart = /\{\{\{/,					// {{{ code span start
		reUntilCodeStop = /.*?\}\}\}/;

	function chain(stream, state, f) {
		state.tokenize = f;
		return f(stream, state);
	}

	// used for strings
	function nextUntilUnescaped(stream, end) {
		var escaped = false,
			next;
		while ((next = stream.next()) != null) {
			if (next == end && !escaped) return false;
			escaped = !escaped && next == "\\";
		}
		return escaped;
	}

	// Used as scratch variables to communicate multiple values without
	// consing up tons of objects.
	var type, content;

	function ret(tp, style, cont) {
		type = tp;
		content = cont;
		return style;
	}

	function jsTokenBase(stream, state) {
		var sol = stream.sol(), 
			ch, tch;
			
		state.block = false;	// indicates the start of a code block.

		ch = stream.peek(); 	// don't eat, to make matching simpler
		
		// check start of  blocks
		if (sol && /[<\/\*{}\-]/.test(ch)) {
			if (stream.match(reCodeBlockStart)) {
				state.block = true;
				return chain(stream, state, twTokenCode);
			}
			if (stream.match(reBlockQuote)) {
				return ret('quote', 'quote');
			}
			if (stream.match(reWikiCommentStart) || stream.match(reWikiCommentStop)) {
				return ret('code', 'comment');
			}
			if (stream.match(reJsCodeStart) || stream.match(reJsCodeStop) || stream.match(reXmlCodeStart) || stream.match(reXmlCodeStop)) {
				return ret('code', 'comment');
			}
			if (stream.match(reHR)) {
				return ret('hr', 'hr');
			}
		} // sol
		ch = stream.next();

		if (sol && /[\/\*!#;:>|]/.test(ch)) {
			if (ch == "!") { // tw header
				stream.skipToEnd();
				return ret("header", "header");
			}
			if (ch == "*") { // tw list
				stream.eatWhile('*');
				return ret("list", "comment");
			}
			if (ch == "#") { // tw numbered list
				stream.eatWhile('#');
				return ret("list", "comment");
			}
			if (ch == ";") { // definition list, term
				stream.eatWhile(';');
				return ret("list", "comment");
			}
			if (ch == ":") { // definition list, description
				stream.eatWhile(':');
				return ret("list", "comment");
			}
			if (ch == ">") { // single line quote
				stream.eatWhile(">");
				return ret("quote", "quote");
			}
			if (ch == '|') {
				return ret('table', 'header');
			}
		}

		if (ch == '{' && stream.match(/\{\{/)) {
			return chain(stream, state, twTokenCode);
		}

		// rudimentary html:// file:// link matching. TW knows much more ...
		if (/[hf]/i.test(ch)) {
			if (/[ti]/i.test(stream.peek()) && stream.match(/\b(ttps?|tp|ile):\/\/[\-A-Z0-9+&@#\/%?=~_|$!:,.;]*[A-Z0-9+&@#\/%=~_|$]/i)) {
				return ret("link", "link");
			}
		}
		// just a little string indicator, don't want to have the whole string covered
		if (ch == '"') {
			return ret('string', 'string');
		}
		if (ch == '~') {	// _no_ CamelCase indicator should be bold
			return ret('text', 'brace');
		}
		if (/[\[\]]/.test(ch)) { // check for [[..]]
			if (stream.peek() == ch) {
				stream.next();
				return ret('brace', 'brace');
			}
		}
		if (ch == "@") {	// check for space link. TODO fix @@...@@ highlighting
			stream.eatWhile(isSpaceName);
			return ret("link", "link");
		}
		if (/\d/.test(ch)) {	// numbers
			stream.eatWhile(/\d/);
			return ret("number", "number");
		}
		if (ch == "/") { // tw invisible comment
			if (stream.eat("%")) {
				return chain(stream, state, twTokenComment);
			}
			else if (stream.eat("/")) { // 
				return chain(stream, state, twTokenEm);
			}
		}
		if (ch == "_") { // tw underline
			if (stream.eat("_")) {
				return chain(stream, state, twTokenUnderline);
			}
		}
		// strikethrough and mdash handling
		if (ch == "-") {
			if (stream.eat("-")) {
				// if strikethrough looks ugly, change CSS.
				if (stream.peek() != ' ')
					return chain(stream, state, twTokenStrike);
				// mdash
				if (stream.peek() == ' ')
					return ret('text', 'brace');
			}
		}
		if (ch == "'") { // tw bold
			if (stream.eat("'")) {
				return chain(stream, state, twTokenStrong);
			}
		}
		if (ch == "<") { // tw macro
			if (stream.eat("<")) {
				return chain(stream, state, twTokenMacro);
			}
		}
		else {
			return ret(ch);
		}

		// core macro handling
		stream.eatWhile(/[\w\$_]/);
		var word = stream.current(),
			known = textwords.propertyIsEnumerable(word) && textwords[word];

		return known ? ret(known.type, known.style, word) : ret("text", null, word);

	} // jsTokenBase()

	function twTokenString(quote) {
		return function (stream, state) {
			if (!nextUntilUnescaped(stream, quote)) state.tokenize = jsTokenBase;
			return ret("string", "string");
		};
	}

	// tw invisible comment
	function twTokenComment(stream, state) {
		var maybeEnd = false,
			ch;
		while (ch = stream.next()) {
			if (ch == "/" && maybeEnd) {
				state.tokenize = jsTokenBase;
				break;
			}
			maybeEnd = (ch == "%");
		}
		return ret("comment", "comment");
	}

	// tw strong / bold
	function twTokenStrong(stream, state) {
		var maybeEnd = false,
			ch;
		while (ch = stream.next()) {
			if (ch == "'" && maybeEnd) {
				state.tokenize = jsTokenBase;
				break;
			}
			maybeEnd = (ch == "'");
		}
		return ret("text", "strong");
	}

	// tw code
	function twTokenCode(stream, state) {
		var ch, sb = state.block;
		
		if (sb && stream.current()) {
			return ret("code", "comment");
		}

		if (!sb && stream.match(reUntilCodeStop)) {
			state.tokenize = jsTokenBase;
			return ret("code", "comment");
		}

		if (sb && stream.sol() && stream.match(reCodeBlockStop)) {
			state.tokenize = jsTokenBase;
			return ret("code", "comment");
		}

		ch = stream.next();
		return (sb) ? ret("code", "comment") : ret("code", "comment");
	}

	// tw em / italic
	function twTokenEm(stream, state) {
		var maybeEnd = false,
			ch;
		while (ch = stream.next()) {
			if (ch == "/" && maybeEnd) {
				state.tokenize = jsTokenBase;
				break;
			}
			maybeEnd = (ch == "/");
		}
		return ret("text", "em");
	}

	// tw underlined text
	function twTokenUnderline(stream, state) {
		var maybeEnd = false,
			ch;
		while (ch = stream.next()) {
			if (ch == "_" && maybeEnd) {
				state.tokenize = jsTokenBase;
				break;
			}
			maybeEnd = (ch == "_");
		}
		return ret("text", "underlined");
	}

	// tw strike through text looks ugly
	// change CSS if needed
	function twTokenStrike(stream, state) {
		var maybeEnd = false,
			ch, nr;
			
		while (ch = stream.next()) {
			if (ch == "-" && maybeEnd) {
				state.tokenize = jsTokenBase;
				break;
			}
			maybeEnd = (ch == "-");
		}
		return ret("text", "strikethrough");
	}

	// macro
	function twTokenMacro(stream, state) {
		var ch, tmp, word, known;

		if (stream.current() == '<<') {
			return ret('brace', 'macro');
		}

		ch = stream.next();
		if (!ch) {
			state.tokenize = jsTokenBase;
			return ret(ch);
		}
		if (ch == ">") {
			if (stream.peek() == '>') {
				stream.next();
				state.tokenize = jsTokenBase;
				return ret("brace", "macro");
			}
		}

		stream.eatWhile(/[\w\$_]/);
		word = stream.current();
		known = keywords.propertyIsEnumerable(word) && keywords[word];

		if (known) {
			return ret(known.type, known.style, word);
		}
		else {
			return ret("macro", null, word);
		}
	}

	// Interface
	return {
		startState: function (basecolumn) {
			return {
				tokenize: jsTokenBase,
				indented: 0,
				level: 0
			};
		},

		token: function (stream, state) {
			if (stream.eatSpace()) return null;
			var style = state.tokenize(stream, state);
			return style;
		},

		electricChars: ""
	};
});

CodeMirror.defineMIME("text/x-tiddlywiki", "tiddlywiki");
//}}}
CodeMirror.defineMode('tiki', function(config, parserConfig) {
	function inBlock(style, terminator, returnTokenizer) {
		return function(stream, state) {
			while (!stream.eol()) {
				if (stream.match(terminator)) {
					state.tokenize = inText;
					break;
				}
				stream.next();
			}
			
			if (returnTokenizer) state.tokenize = returnTokenizer;
			
			return style;
		};
	}
	
	function inLine(style, terminator) {
		return function(stream, state) {
			while(!stream.eol()) {
				stream.next()
			}
			state.tokenize = inText;
			return style;
		};
	}
	
	function inText(stream, state) {
		function chain(parser) {
			state.tokenize = parser;
			return parser(stream, state);
		}
		
		var sol = stream.sol();
		var ch = stream.next();
		
		//non start of line
		switch (ch) { //switch is generally much faster than if, so it is used here
			case "{": //plugin
				type = stream.eat("/") ? "closeTag" : "openTag";
				stream.eatSpace();
				tagName = "";
				var c;
				while ((c = stream.eat(/[^\s\u00a0=\"\'\/?(}]/))) tagName += c;
				state.tokenize = inPlugin;
				return "tag";
				break;
			case "_": //bold
				if (stream.eat("_")) {
					return chain(inBlock("strong", "__", inText));
				}
				break;
			case "'": //italics
				if (stream.eat("'")) {
					// Italic text
					return chain(inBlock("em", "''", inText));
				}
				break;
			case "(":// Wiki Link
				if (stream.eat("(")) {
					return chain(inBlock("variable-2", "))", inText));
				}
				break;
			case "[":// Weblink
				return chain(inBlock("variable-3", "]", inText));
				break;
			case "|": //table
				if (stream.eat("|")) {
					return chain(inBlock("comment", "||"));
				}
				break;
			case "-": 
				if (stream.eat("=")) {//titleBar
					return chain(inBlock("header string", "=-", inText));
				} else if (stream.eat("-")) {//deleted
					return chain(inBlock("error tw-deleted", "--", inText));
				}
				break;
			case "=": //underline
				if (stream.match("==")) {
					return chain(inBlock("tw-underline", "===", inText));
				}
				break;
			case ":":
				if (stream.eat(":")) {
					return chain(inBlock("comment", "::"));
				}
				break;
			case "^": //box
				return chain(inBlock("tw-box", "^"));
				break;
			case "~": //np
				if (stream.match("np~")) {
					return chain(inBlock("meta", "~/np~"));
				}
				break;
		}
		
		//start of line types
		if (sol) {
			switch (ch) {
				case "!": //header at start of line
					if (stream.match('!!!!!')) {
						return chain(inLine("header string"));
					} else if (stream.match('!!!!')) {
						return chain(inLine("header string"));
					} else if (stream.match('!!!')) {
						return chain(inLine("header string"));
					} else if (stream.match('!!')) {
						return chain(inLine("header string"));
					} else {
						return chain(inLine("header string"));
					}
					break;
				case "*": //unordered list line item, or <li /> at start of line
				case "#": //ordered list line item, or <li /> at start of line
				case "+": //ordered list line item, or <li /> at start of line
					return chain(inLine("tw-listitem bracket"));
					break;
			}
		}
		
		//stream.eatWhile(/[&{]/); was eating up plugins, turned off to act less like html and more like tiki
		return null;
	}
	
	var indentUnit = config.indentUnit;

	// Return variables for tokenizers
	var pluginName, type;
	function inPlugin(stream, state) {
		var ch = stream.next();
		var peek = stream.peek();
		
		if (ch == "}") {
			state.tokenize = inText;
			//type = ch == ")" ? "endPlugin" : "selfclosePlugin"; inPlugin
			return "tag";
		} else if (ch == "(" || ch == ")") {
			return "bracket";
		} else if (ch == "=") {
			type = "equals";
			
			if (peek == ">") {
				ch = stream.next();
				peek = stream.peek();
			}
			
			//here we detect values directly after equal character with no quotes
			if (!/[\'\"]/.test(peek)) {
				state.tokenize = inAttributeNoQuote();
			}
			//end detect values
			
			return "operator";
		} else if (/[\'\"]/.test(ch)) {
			state.tokenize = inAttribute(ch);
			return state.tokenize(stream, state);
		} else {
			stream.eatWhile(/[^\s\u00a0=\"\'\/?]/);
			return "keyword";
		}
	}

	function inAttribute(quote) {
		return function(stream, state) {
			while (!stream.eol()) {
				if (stream.next() == quote) {
					state.tokenize = inPlugin;
					break;
				}
			}
			return "string";
		};
	}
	
	function inAttributeNoQuote() {
		return function(stream, state) {
			while (!stream.eol()) {
				var ch = stream.next();
				var peek = stream.peek();
				if (ch == " " || ch == "," || /[ )}]/.test(peek)) {
					state.tokenize = inPlugin;
					break;
				}
			}
			return "string";
		};
	}

	var curState, setStyle;
	function pass() {
		for (var i = arguments.length - 1; i >= 0; i--) curState.cc.push(arguments[i]);
	}
	
	function cont() {
		pass.apply(null, arguments);
		return true;
	}

	function pushContext(pluginName, startOfLine) {
		var noIndent = curState.context && curState.context.noIndent;
		curState.context = {
			prev: curState.context,
			pluginName: pluginName,
			indent: curState.indented,
			startOfLine: startOfLine,
			noIndent: noIndent
		};
	}
	
	function popContext() {
		if (curState.context) curState.context = curState.context.prev;
	}

	function element(type) {
		if (type == "openPlugin") {curState.pluginName = pluginName; return cont(attributes, endplugin(curState.startOfLine));}
		else if (type == "closePlugin") {
			var err = false;
			if (curState.context) {
				err = curState.context.pluginName != pluginName;
				popContext();
			} else {
				err = true;
			}
			if (err) setStyle = "error";
			return cont(endcloseplugin(err));
		}
		else if (type == "string") {
			if (!curState.context || curState.context.name != "!cdata") pushContext("!cdata");
			if (curState.tokenize == inText) popContext();
			return cont();
		}
		else return cont();
	}
	
	function endplugin(startOfLine) {
		return function(type) {
			if (
			type == "selfclosePlugin" ||
			type == "endPlugin"
		)
				return cont();
			if (type == "endPlugin") {pushContext(curState.pluginName, startOfLine); return cont();}
			return cont();
		};
	}
	
	function endcloseplugin(err) {
		return function(type) {
			if (err) setStyle = "error";
			if (type == "endPlugin") return cont();
			return pass();
		}
	}

	function attributes(type) {
		if (type == "keyword") {setStyle = "attribute"; return cont(attributes);}
		if (type == "equals") return cont(attvalue, attributes);
		return pass();
	}
	function attvalue(type) {
		if (type == "keyword") {setStyle = "string"; return cont();}
		if (type == "string") return cont(attvaluemaybe);
			return pass();
	}
	function attvaluemaybe(type) {
		if (type == "string") return cont(attvaluemaybe);
		else return pass();
	}
	return {
		startState: function() {
			return {tokenize: inText, cc: [], indented: 0, startOfLine: true, pluginName: null, context: null};
		},
		token: function(stream, state) {
			if (stream.sol()) {
				state.startOfLine = true;
				state.indented = stream.indentation();
			}
			if (stream.eatSpace()) return null;

			setStyle = type = pluginName = null;
			var style = state.tokenize(stream, state);
				if ((style || type) && style != "comment") {
				curState = state;
				while (true) {
					var comb = state.cc.pop() || element;
					if (comb(type || style)) break;
				}
			}
			state.startOfLine = false;
			return setStyle || style;
				},
		indent: function(state, textAfter) {
			var context = state.context;
			if (context && context.noIndent) return 0;
			if (context && /^{\//.test(textAfter))
				context = context.prev;
			while (context && !context.startOfLine)
				context = context.prev;
			if (context) return context.indent + indentUnit;
			else return 0;
		},
		compareStates: function(a, b) {
			if (a.indented != b.indented || a.pluginName != b.pluginName) return false;
			for (var ca = a.context, cb = b.context; ; ca = ca.prev, cb = cb.prev) {
				if (!ca || !cb) return ca == cb;
				if (ca.pluginName != cb.pluginName) return false;
			}
		},
		electricChars: "/"
	};
});

//I figure, why not
CodeMirror.defineMIME("text/tiki", "tiki");
CodeMirror.defineMode("vbscript", function() {
  var regexVBScriptKeyword = /^(?:Call|Case|CDate|Clear|CInt|CLng|Const|CStr|Description|Dim|Do|Each|Else|ElseIf|End|Err|Error|Exit|False|For|Function|If|LCase|Loop|LTrim|Next|Nothing|Now|Number|On|Preserve|Quit|ReDim|Resume|RTrim|Select|Set|Sub|Then|To|Trim|True|UBound|UCase|Until|VbCr|VbCrLf|VbLf|VbTab)$/im;

  return {
    token: function(stream) {
      if (stream.eatSpace()) return null;
      var ch = stream.next();
      if (ch == "'") {
      	stream.skipToEnd();
      	return "comment";
      }
      if (ch == '"') {
      	stream.skipTo('"');
      	return "string";
      }

      if (/\w/.test(ch)) {
        stream.eatWhile(/\w/);
        if (regexVBScriptKeyword.test(stream.current())) return "keyword";
      }
      return null;
    }
  };
});

CodeMirror.defineMIME("text/vbscript", "vbscript");
CodeMirror.defineMode("velocity", function(config) {
    function parseWords(str) {
        var obj = {}, words = str.split(" ");
        for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
        return obj;
    }

    var indentUnit = config.indentUnit
    var keywords = parseWords("#end #else #break #stop #[[ #]] " +
                              "#{end} #{else} #{break} #{stop}");
    var functions = parseWords("#if #elseif #foreach #set #include #parse #macro #define #evaluate " +
                               "#{if} #{elseif} #{foreach} #{set} #{include} #{parse} #{macro} #{define} #{evaluate}");
    var specials = parseWords("$foreach.count $foreach.hasNext $foreach.first $foreach.last $foreach.topmost $foreach.parent $velocityCount");
    var isOperatorChar = /[+\-*&%=<>!?:\/|]/;
    var multiLineStrings =true;

    function chain(stream, state, f) {
        state.tokenize = f;
        return f(stream, state);
    }
    function tokenBase(stream, state) {
        var beforeParams = state.beforeParams;
        state.beforeParams = false;
        var ch = stream.next();
        // start of string?
        if ((ch == '"' || ch == "'") && state.inParams)
            return chain(stream, state, tokenString(ch));
        // is it one of the special signs []{}().,;? Seperator?
        else if (/[\[\]{}\(\),;\.]/.test(ch)) {
            if (ch == "(" && beforeParams) state.inParams = true;
            else if (ch == ")") state.inParams = false;
            return null;
        }
        // start of a number value?
        else if (/\d/.test(ch)) {
            stream.eatWhile(/[\w\.]/);
            return "number";
        }
        // multi line comment?
        else if (ch == "#" && stream.eat("*")) {
            return chain(stream, state, tokenComment);
        }
        // unparsed content?
        else if (ch == "#" && stream.match(/ *\[ *\[/)) {
            return chain(stream, state, tokenUnparsed);
        }
        // single line comment?
        else if (ch == "#" && stream.eat("#")) {
            stream.skipToEnd();
            return "comment";
        }
        // variable?
        else if (ch == "$") {
            stream.eatWhile(/[\w\d\$_\.{}]/);
            // is it one of the specials?
            if (specials && specials.propertyIsEnumerable(stream.current().toLowerCase())) {
                return "keyword";
            }
            else {
                state.beforeParams = true;
                return "builtin";
            }
        }
        // is it a operator?
        else if (isOperatorChar.test(ch)) {
            stream.eatWhile(isOperatorChar);
            return "operator";
        }
        else {
            // get the whole word
            stream.eatWhile(/[\w\$_{}]/);
            var word = stream.current().toLowerCase();
            // is it one of the listed keywords?
            if (keywords && keywords.propertyIsEnumerable(word))
                return "keyword";
            // is it one of the listed functions?
            if (functions && functions.propertyIsEnumerable(word) ||
                stream.current().match(/^#[a-z0-9_]+ *$/i) && stream.peek()=="(") {
                state.beforeParams = true;
                return "keyword";
            }
            // default: just a "word"
            return null;
        }
    }

    function tokenString(quote) {
        return function(stream, state) {
            var escaped = false, next, end = false;
            while ((next = stream.next()) != null) {
                if (next == quote && !escaped) {
                    end = true;
                    break;
                }
                escaped = !escaped && next == "\\";
            }
            if (end) state.tokenize = tokenBase;
            return "string";
        };
    }

    function tokenComment(stream, state) {
        var maybeEnd = false, ch;
        while (ch = stream.next()) {
            if (ch == "#" && maybeEnd) {
                state.tokenize = tokenBase;
                break;
            }
            maybeEnd = (ch == "*");
        }
        return "comment";
    }

    function tokenUnparsed(stream, state) {
        var maybeEnd = 0, ch;
        while (ch = stream.next()) {
            if (ch == "#" && maybeEnd == 2) {
                state.tokenize = tokenBase;
                break;
            }
            if (ch == "]")
                maybeEnd++;
            else if (ch != " ")
                maybeEnd = 0;
        }
        return "meta";
    }
    // Interface

    return {
        startState: function(basecolumn) {
            return {
                tokenize: tokenBase,
                beforeParams: false,
                inParams: false
            };
        },

        token: function(stream, state) {
            if (stream.eatSpace()) return null;
            return state.tokenize(stream, state);
        }
    };
});

CodeMirror.defineMIME("text/velocity", "velocity");
CodeMirror.defineMode("verilog", function(config, parserConfig) {
  var indentUnit = config.indentUnit,
      keywords = parserConfig.keywords || {},
      blockKeywords = parserConfig.blockKeywords || {},
      atoms = parserConfig.atoms || {},
      hooks = parserConfig.hooks || {},
      multiLineStrings = parserConfig.multiLineStrings;
  var isOperatorChar = /[&|~><!\)\(*#%@+\/=?\:;}{,\.\^\-\[\]]/;

  var curPunc;

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (hooks[ch]) {
      var result = hooks[ch](stream, state);
      if (result !== false) return result;
    }
    if (ch == '"') {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    if (/[\[\]{}\(\),;\:\.]/.test(ch)) {
      curPunc = ch;
      return null
    }
    if (/[\d']/.test(ch)) {
      stream.eatWhile(/[\w\.']/);
      return "number";
    }
    if (ch == "/") {
      if (stream.eat("*")) {
        state.tokenize = tokenComment;
        return tokenComment(stream, state);
      }
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
    }
    if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return "operator";
    }
    stream.eatWhile(/[\w\$_]/);
    var cur = stream.current();
    if (keywords.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "keyword";
    }
    if (atoms.propertyIsEnumerable(cur)) return "atom";
    return "word";
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !(escaped || multiLineStrings))
        state.tokenize = tokenBase;
      return "string";
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = tokenBase;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  function Context(indented, column, type, align, prev) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.align = align;
    this.prev = prev;
  }
  function pushContext(state, col, type) {
    return state.context = new Context(state.indented, col, type, null, state.context);
  }
  function popContext(state) {
    var t = state.context.type;
    if (t == ")" || t == "]" || t == "}")
      state.indented = state.context.indented;
    return state.context = state.context.prev;
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: null,
        context: new Context((basecolumn || 0) - indentUnit, 0, "top", false),
        indented: 0,
        startOfLine: true
      };
    },

    token: function(stream, state) {
      var ctx = state.context;
      if (stream.sol()) {
        if (ctx.align == null) ctx.align = false;
        state.indented = stream.indentation();
        state.startOfLine = true;
      }
      if (stream.eatSpace()) return null;
      curPunc = null;
      var style = (state.tokenize || tokenBase)(stream, state);
      if (style == "comment" || style == "meta") return style;
      if (ctx.align == null) ctx.align = true;

      if ((curPunc == ";" || curPunc == ":") && ctx.type == "statement") popContext(state);
      else if (curPunc == "{") pushContext(state, stream.column(), "}");
      else if (curPunc == "[") pushContext(state, stream.column(), "]");
      else if (curPunc == "(") pushContext(state, stream.column(), ")");
      else if (curPunc == "}") {
        while (ctx.type == "statement") ctx = popContext(state);
        if (ctx.type == "}") ctx = popContext(state);
        while (ctx.type == "statement") ctx = popContext(state);
      }
      else if (curPunc == ctx.type) popContext(state);
      else if (ctx.type == "}" || ctx.type == "top" || (ctx.type == "statement" && curPunc == "newstatement"))
        pushContext(state, stream.column(), "statement");
      state.startOfLine = false;
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase && state.tokenize != null) return 0;
      var firstChar = textAfter && textAfter.charAt(0), ctx = state.context, closing = firstChar == ctx.type;
      if (ctx.type == "statement") return ctx.indented + (firstChar == "{" ? 0 : indentUnit);
      else if (ctx.align) return ctx.column + (closing ? 0 : 1);
      else return ctx.indented + (closing ? 0 : indentUnit);
    },

    electricChars: "{}"
  };
});

(function() {
  function words(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }

  var verilogKeywords = "always and assign automatic begin buf bufif0 bufif1 case casex casez cell cmos config " +
    "deassign default defparam design disable edge else end endcase endconfig endfunction endgenerate endmodule " +
    "endprimitive endspecify endtable endtask event for force forever fork function generate genvar highz0 " +
    "highz1 if ifnone incdir include initial inout input instance integer join large liblist library localparam " +
    "macromodule medium module nand negedge nmos nor noshowcancelled not notif0 notif1 or output parameter pmos " +
    "posedge primitive pull0 pull1 pulldown pullup pulsestyle_onevent pulsestyle_ondetect rcmos real realtime " +
    "reg release repeat rnmos rpmos rtran rtranif0 rtranif1 scalared showcancelled signed small specify specparam " +
    "strong0 strong1 supply0 supply1 table task time tran tranif0 tranif1 tri tri0 tri1 triand trior trireg " +
    "unsigned use vectored wait wand weak0 weak1 while wire wor xnor xor";

  var verilogBlockKeywords = "begin bufif0 bufif1 case casex casez config else end endcase endconfig endfunction " +
    "endgenerate endmodule endprimitive endspecify endtable endtask for forever function generate if ifnone " +
    "macromodule module primitive repeat specify table task while";

  function metaHook(stream, state) {
    stream.eatWhile(/[\w\$_]/);
    return "meta";
  }

  // C#-style strings where "" escapes a quote.
  function tokenAtString(stream, state) {
    var next;
    while ((next = stream.next()) != null) {
      if (next == '"' && !stream.eat('"')) {
        state.tokenize = null;
        break;
      }
    }
    return "string";
  }

  CodeMirror.defineMIME("text/x-verilog", {
    name: "verilog",
    keywords: words(verilogKeywords),
    blockKeywords: words(verilogBlockKeywords),
    atoms: words("null"),
    hooks: {"`": metaHook, "$": metaHook}
  });
}());
/*
Copyright (C) 2011 by MarkLogic Corporation
Author: Mike Brevoort <mike@brevoort.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
CodeMirror.defineMode("xquery", function(config, parserConfig) {

  // The keywords object is set to the result of this self executing
  // function. Each keyword is a property of the keywords object whose
  // value is {type: atype, style: astyle}
  var keywords = function(){
    // conveinence functions used to build keywords object
    function kw(type) {return {type: type, style: "keyword"};}
    var A = kw("keyword a")
      , B = kw("keyword b")
      , C = kw("keyword c")
      , operator = kw("operator")
      , atom = {type: "atom", style: "atom"}
      , punctuation = {type: "punctuation", style: ""}
      , qualifier = {type: "axis_specifier", style: "qualifier"};
    
    // kwObj is what is return from this function at the end
    var kwObj = {
      'if': A, 'switch': A, 'while': A, 'for': A,
      'else': B, 'then': B, 'try': B, 'finally': B, 'catch': B,
      'element': C, 'attribute': C, 'let': C, 'implements': C, 'import': C, 'module': C, 'namespace': C, 
      'return': C, 'super': C, 'this': C, 'throws': C, 'where': C, 'private': C,      
      ',': punctuation,
      'null': atom, 'fn:false()': atom, 'fn:true()': atom
    };
    
    // a list of 'basic' keywords. For each add a property to kwObj with the value of 
    // {type: basic[i], style: "keyword"} e.g. 'after' --> {type: "after", style: "keyword"}
    var basic = ['after','ancestor','ancestor-or-self','and','as','ascending','assert','attribute','before',
    'by','case','cast','child','comment','declare','default','define','descendant','descendant-or-self',
    'descending','document','document-node','element','else','eq','every','except','external','following',
    'following-sibling','follows','for','function','if','import','in','instance','intersect','item',
    'let','module','namespace','node','node','of','only','or','order','parent','precedes','preceding',
    'preceding-sibling','processing-instruction','ref','return','returns','satisfies','schema','schema-element',
    'self','some','sortby','stable','text','then','to','treat','typeswitch','union','variable','version','where',
    'xquery', 'empty-sequence'];
    for(var i=0, l=basic.length; i < l; i++) { kwObj[basic[i]] = kw(basic[i])};
    
    // a list of types. For each add a property to kwObj with the value of 
    // {type: "atom", style: "atom"}
    var types = ['xs:string', 'xs:float', 'xs:decimal', 'xs:double', 'xs:integer', 'xs:boolean', 'xs:date', 'xs:dateTime', 
    'xs:time', 'xs:duration', 'xs:dayTimeDuration', 'xs:time', 'xs:yearMonthDuration', 'numeric', 'xs:hexBinary', 
    'xs:base64Binary', 'xs:anyURI', 'xs:QName', 'xs:byte','xs:boolean','xs:anyURI','xf:yearMonthDuration'];
    for(var i=0, l=types.length; i < l; i++) { kwObj[types[i]] = atom;};
    
    // each operator will add a property to kwObj with value of {type: "operator", style: "keyword"}
    var operators = ['eq', 'ne', 'lt', 'le', 'gt', 'ge', ':=', '=', '>', '>=', '<', '<=', '.', '|', '?', 'and', 'or', 'div', 'idiv', 'mod', '*', '/', '+', '-'];
    for(var i=0, l=operators.length; i < l; i++) { kwObj[operators[i]] = operator;};
    
    // each axis_specifiers will add a property to kwObj with value of {type: "axis_specifier", style: "qualifier"}
    var axis_specifiers = ["self::", "attribute::", "child::", "descendant::", "descendant-or-self::", "parent::", 
    "ancestor::", "ancestor-or-self::", "following::", "preceding::", "following-sibling::", "preceding-sibling::"];
    for(var i=0, l=axis_specifiers.length; i < l; i++) { kwObj[axis_specifiers[i]] = qualifier; };

    return kwObj;
  }();

  // Used as scratch variables to communicate multiple values without
  // consing up tons of objects.
  var type, content;
  
  function ret(tp, style, cont) {
    type = tp; content = cont;
    return style;
  }
  
  function chain(stream, state, f) {
    state.tokenize = f;
    return f(stream, state);
  }
  
  // the primary mode tokenizer
  function tokenBase(stream, state) {
    var ch = stream.next(), 
        mightBeFunction = false,
        isEQName = isEQNameAhead(stream);
    
    // an XML tag (if not in some sub, chained tokenizer)
    if (ch == "<") {
      if(stream.match("!--", true))
        return chain(stream, state, tokenXMLComment);
        
      if(stream.match("![CDATA", false)) {
        state.tokenize = tokenCDATA;
        return ret("tag", "tag");
      }
      
      if(stream.match("?", false)) {
        return chain(stream, state, tokenPreProcessing);
      }
      
      var isclose = stream.eat("/");
      stream.eatSpace();
      var tagName = "", c;
      while ((c = stream.eat(/[^\s\u00a0=<>\"\'\/?]/))) tagName += c;
      
      return chain(stream, state, tokenTag(tagName, isclose));
    }
    // start code block
    else if(ch == "{") {
      pushStateStack(state,{ type: "codeblock"});
      return ret("", "");
    }
    // end code block
    else if(ch == "}") {
      popStateStack(state);
      return ret("", "");
    }
    // if we're in an XML block
    else if(isInXmlBlock(state)) {
      if(ch == ">")
        return ret("tag", "tag");
      else if(ch == "/" && stream.eat(">")) {
        popStateStack(state);
        return ret("tag", "tag");
      }
      else  
        return ret("word", "word");
    }
    // if a number
    else if (/\d/.test(ch)) {
      stream.match(/^\d*(?:\.\d*)?(?:E[+\-]?\d+)?/);
      return ret("number", "atom");
    }
    // comment start
    else if (ch === "(" && stream.eat(":")) {
      pushStateStack(state, { type: "comment"});
      return chain(stream, state, tokenComment);
    }
    // quoted string
    else if (  !isEQName && (ch === '"' || ch === "'"))
      return chain(stream, state, tokenString(ch));
    // variable
    else if(ch === "$") {
      return chain(stream, state, tokenVariable);
    }
    // assignment
    else if(ch ===":" && stream.eat("=")) {
      return ret("operator", "keyword");
    }
    // open paren
    else if(ch === "(") {
      pushStateStack(state, { type: "paren"});
      return ret("", "");
    }
    // close paren
    else if(ch === ")") {
      popStateStack(state);
      return ret("", "");
    }
    // open paren
    else if(ch === "[") {
      pushStateStack(state, { type: "bracket"});
      return ret("", "");
    }
    // close paren
    else if(ch === "]") {
      popStateStack(state);
      return ret("", "");
    }
    else {
      var known = keywords.propertyIsEnumerable(ch) && keywords[ch];

      // if there's a EQName ahead, consume the rest of the string portion, it's likely a function
      if(isEQName && ch === '\"') while(stream.next() !== '"'){}
      if(isEQName && ch === '\'') while(stream.next() !== '\''){}
      
      // gobble up a word if the character is not known
      if(!known) stream.eatWhile(/[\w\$_-]/);
      
      // gobble a colon in the case that is a lib func type call fn:doc
      var foundColon = stream.eat(":")
      
      // if there's not a second colon, gobble another word. Otherwise, it's probably an axis specifier
      // which should get matched as a keyword
      if(!stream.eat(":") && foundColon) {
        stream.eatWhile(/[\w\$_-]/);
      }
      // if the next non whitespace character is an open paren, this is probably a function (if not a keyword of other sort)
      if(stream.match(/^[ \t]*\(/, false)) {
        mightBeFunction = true;
      }
      // is the word a keyword?
      var word = stream.current();
      known = keywords.propertyIsEnumerable(word) && keywords[word];
      
      // if we think it's a function call but not yet known, 
      // set style to variable for now for lack of something better
      if(mightBeFunction && !known) known = {type: "function_call", style: "variable def"};
      
      // if the previous word was element, attribute, axis specifier, this word should be the name of that
      if(isInXmlConstructor(state)) {
        popStateStack(state);
        return ret("word", "word", word);
      }
      // as previously checked, if the word is element,attribute, axis specifier, call it an "xmlconstructor" and 
      // push the stack so we know to look for it on the next word
      if(word == "element" || word == "attribute" || known.type == "axis_specifier") pushStateStack(state, {type: "xmlconstructor"});
      
      // if the word is known, return the details of that else just call this a generic 'word'
      return known ? ret(known.type, known.style, word) :
                     ret("word", "word", word);
    }
  }

  // handle comments, including nested 
  function tokenComment(stream, state) {
    var maybeEnd = false, maybeNested = false, nestedCount = 0, ch;
    while (ch = stream.next()) {
      if (ch == ")" && maybeEnd) {
        if(nestedCount > 0)
          nestedCount--;
        else {
          popStateStack(state);
          break;
        }
      }
      else if(ch == ":" && maybeNested) {
        nestedCount++;
      }
      maybeEnd = (ch == ":");
      maybeNested = (ch == "(");
    }
    
    return ret("comment", "comment");
  }

  // tokenizer for string literals
  // optionally pass a tokenizer function to set state.tokenize back to when finished
  function tokenString(quote, f) {
    return function(stream, state) {
      var ch;

      if(isInString(state) && stream.current() == quote) {
        popStateStack(state);
        if(f) state.tokenize = f;
        return ret("string", "string");
      }

      pushStateStack(state, { type: "string", name: quote, tokenize: tokenString(quote, f) });

      // if we're in a string and in an XML block, allow an embedded code block
      if(stream.match("{", false) && isInXmlAttributeBlock(state)) {
        state.tokenize = tokenBase;
        return ret("string", "string"); 
      }

      
      while (ch = stream.next()) {
        if (ch ==  quote) {
          popStateStack(state);
          if(f) state.tokenize = f;
          break;
        }
        else {
          // if we're in a string and in an XML block, allow an embedded code block in an attribute
          if(stream.match("{", false) && isInXmlAttributeBlock(state)) {
            state.tokenize = tokenBase;
            return ret("string", "string"); 
          }

        }
      }
      
      return ret("string", "string");
    };
  }
  
  // tokenizer for variables
  function tokenVariable(stream, state) {
    var isVariableChar = /[\w\$_-]/;

    // a variable may start with a quoted EQName so if the next character is quote, consume to the next quote
    if(stream.eat("\"")) {
      while(stream.next() !== '\"'){};
      stream.eat(":");
    } else {
      stream.eatWhile(isVariableChar);
      if(!stream.match(":=", false)) stream.eat(":");
    }
    stream.eatWhile(isVariableChar);
    state.tokenize = tokenBase;
    return ret("variable", "variable");
  }
  
  // tokenizer for XML tags
  function tokenTag(name, isclose) {
    return function(stream, state) {
      stream.eatSpace();
      if(isclose && stream.eat(">")) {
        popStateStack(state);
        state.tokenize = tokenBase;
        return ret("tag", "tag");
      }
      // self closing tag without attributes?
      if(!stream.eat("/"))
        pushStateStack(state, { type: "tag", name: name, tokenize: tokenBase});
      if(!stream.eat(">")) {
        state.tokenize = tokenAttribute;
        return ret("tag", "tag");
      }
      else {
        state.tokenize = tokenBase;        
      }
      return ret("tag", "tag");
    }
  }

  // tokenizer for XML attributes
  function tokenAttribute(stream, state) {
    var ch = stream.next();
    
    if(ch == "/" && stream.eat(">")) {
      if(isInXmlAttributeBlock(state)) popStateStack(state);
      if(isInXmlBlock(state)) popStateStack(state);
      return ret("tag", "tag");
    }
    if(ch == ">") {
      if(isInXmlAttributeBlock(state)) popStateStack(state);
      return ret("tag", "tag");
    }
    if(ch == "=")
      return ret("", "");
    // quoted string
    if (ch == '"' || ch == "'")
      return chain(stream, state, tokenString(ch, tokenAttribute));

    if(!isInXmlAttributeBlock(state)) 
      pushStateStack(state, { type: "attribute", name: name, tokenize: tokenAttribute});

    stream.eat(/[a-zA-Z_:]/);
    stream.eatWhile(/[-a-zA-Z0-9_:.]/);
    stream.eatSpace();

    // the case where the attribute has not value and the tag was closed
    if(stream.match(">", false) || stream.match("/", false)) {
      popStateStack(state);
      state.tokenize = tokenBase;      
    }

    return ret("attribute", "attribute");
  }
  
  // handle comments, including nested 
  function tokenXMLComment(stream, state) {
    while (ch = stream.next()) {
      if (ch == "-" && stream.match("->", true)) {
        state.tokenize = tokenBase;        
        return ret("comment", "comment");
      }
    }
  }


  // handle CDATA
  function tokenCDATA(stream, state) {
    while (ch = stream.next()) {
      if (ch == "]" && stream.match("]", true)) {
        state.tokenize = tokenBase;        
        return ret("comment", "comment");
      }
    }
  }

  // handle preprocessing instructions
  function tokenPreProcessing(stream, state) {
    while (ch = stream.next()) {
      if (ch == "?" && stream.match(">", true)) {
        state.tokenize = tokenBase;        
        return ret("comment", "comment meta");
      }
    }
  }
  
  
  // functions to test the current context of the state
  function isInXmlBlock(state) { return isIn(state, "tag"); }
  function isInXmlAttributeBlock(state) { return isIn(state, "attribute"); }
  function isInCodeBlock(state) { return isIn(state, "codeblock"); }
  function isInXmlConstructor(state) { return isIn(state, "xmlconstructor"); }
  function isInString(state) { return isIn(state, "string"); }

  function isEQNameAhead(stream) { 
    // assume we've already eaten a quote (")
    if(stream.current() === '"')
      return stream.match(/^[^\"]+\"\:/, false);
    else if(stream.current() === '\'')
      return stream.match(/^[^\"]+\'\:/, false);
    else
      return false;
  }
  
  function isIn(state, type) {
    return (state.stack.length && state.stack[state.stack.length - 1].type == type);    
  }
  
  function pushStateStack(state, newState) {
    state.stack.push(newState);
  }
  
  function popStateStack(state) {
    var popped = state.stack.pop();
    var reinstateTokenize = state.stack.length && state.stack[state.stack.length-1].tokenize
    state.tokenize = reinstateTokenize || tokenBase;
  }
  
  // the interface for the mode API
  return {
    startState: function(basecolumn) {
      return {
        tokenize: tokenBase,
        cc: [],
        stack: []
      };
    },

    token: function(stream, state) {
      if (stream.eatSpace()) return null;
      var style = state.tokenize(stream, state);
      return style;
    }
  };

});

CodeMirror.defineMIME("application/xquery", "xquery");
CodeMirror.defineMode("yaml", function() {
	
	var cons = ['true', 'false', 'on', 'off', 'yes', 'no'];
	var keywordRegex = new RegExp("\\b(("+cons.join(")|(")+"))$", 'i');
	
	return {
		token: function(stream, state) {
			var ch = stream.peek();
			var esc = state.escaped;
			state.escaped = false;
			/* comments */
			if (ch == "#") { stream.skipToEnd(); return "comment"; }
			if (state.literal && stream.indentation() > state.keyCol) {
				stream.skipToEnd(); return "string";
			} else if (state.literal) { state.literal = false; }
			if (stream.sol()) {
				state.keyCol = 0;
				state.pair = false;
				state.pairStart = false;
				/* document start */
				if(stream.match(/---/)) { return "def"; }
				/* document end */
				if (stream.match(/\.\.\./)) { return "def"; }
				/* array list item */
				if (stream.match(/\s*-\s+/)) { return 'meta'; }
			}
			/* pairs (associative arrays) -> key */
			if (!state.pair && stream.match(/^\s*([a-z0-9\._-])+(?=\s*:)/i)) {
				state.pair = true;
				state.keyCol = stream.indentation();
				return "atom";
			}
			if (state.pair && stream.match(/^:\s*/)) { state.pairStart = true; return 'meta'; }
			
			/* inline pairs/lists */
			if (stream.match(/^(\{|\}|\[|\])/)) {
				if (ch == '{')
					state.inlinePairs++;
				else if (ch == '}')
					state.inlinePairs--;
				else if (ch == '[')
					state.inlineList++;
				else
					state.inlineList--;
				return 'meta';
			}
			
			/* list seperator */
			if (state.inlineList > 0 && !esc && ch == ',') {
				stream.next();
				return 'meta';
			}
			/* pairs seperator */
			if (state.inlinePairs > 0 && !esc && ch == ',') {
				state.keyCol = 0;
				state.pair = false;
				state.pairStart = false;
				stream.next();
				return 'meta';
			}
			
			/* start of value of a pair */
			if (state.pairStart) {
				/* block literals */
				if (stream.match(/^\s*(\||\>)\s*/)) { state.literal = true; return 'meta'; };
				/* references */
				if (stream.match(/^\s*(\&|\*)[a-z0-9\._-]+\b/i)) { return 'variable-2'; }
				/* numbers */
				if (state.inlinePairs == 0 && stream.match(/^\s*-?[0-9\.\,]+\s?$/)) { return 'number'; }
				if (state.inlinePairs > 0 && stream.match(/^\s*-?[0-9\.\,]+\s?(?=(,|}))/)) { return 'number'; }
				/* keywords */
				if (stream.match(keywordRegex)) { return 'keyword'; }
			}

			/* nothing found, continue */
			state.pairStart = false;
			state.escaped = (ch == '\\');
			stream.next();
			return null;
		},
		startState: function() {
			return {
				pair: false,
				pairStart: false,
				keyCol: 0,
				inlinePairs: 0,
				inlineList: 0,
				literal: false,
				escaped: false
			};
		}
	};
});

CodeMirror.defineMIME("text/x-yaml", "yaml");
CodeMirror.defineMode("clike", function(config, parserConfig) {
  var indentUnit = config.indentUnit,
      keywords = parserConfig.keywords || {},
      blockKeywords = parserConfig.blockKeywords || {},
      atoms = parserConfig.atoms || {},
      hooks = parserConfig.hooks || {},
      multiLineStrings = parserConfig.multiLineStrings;
  var isOperatorChar = /[+\-*&%=<>!?|\/]/;

  var curPunc;

  function tokenBase(stream, state) {
    var ch = stream.next();
    if (hooks[ch]) {
      var result = hooks[ch](stream, state);
      if (result !== false) return result;
    }
    if (ch == '"' || ch == "'") {
      state.tokenize = tokenString(ch);
      return state.tokenize(stream, state);
    }
    if (/[\[\]{}\(\),;\:\.]/.test(ch)) {
      curPunc = ch;
      return null;
    }
    if (/\d/.test(ch)) {
      stream.eatWhile(/[\w\.]/);
      return "number";
    }
    if (ch == "/") {
      if (stream.eat("*")) {
        state.tokenize = tokenComment;
        return tokenComment(stream, state);
      }
      if (stream.eat("/")) {
        stream.skipToEnd();
        return "comment";
      }
    }
    if (isOperatorChar.test(ch)) {
      stream.eatWhile(isOperatorChar);
      return "operator";
    }
    stream.eatWhile(/[\w\$_]/);
    var cur = stream.current();
    if (keywords.propertyIsEnumerable(cur)) {
      if (blockKeywords.propertyIsEnumerable(cur)) curPunc = "newstatement";
      return "keyword";
    }
    if (atoms.propertyIsEnumerable(cur)) return "atom";
    return "word";
  }

  function tokenString(quote) {
    return function(stream, state) {
      var escaped = false, next, end = false;
      while ((next = stream.next()) != null) {
        if (next == quote && !escaped) {end = true; break;}
        escaped = !escaped && next == "\\";
      }
      if (end || !(escaped || multiLineStrings))
        state.tokenize = null;
      return "string";
    };
  }

  function tokenComment(stream, state) {
    var maybeEnd = false, ch;
    while (ch = stream.next()) {
      if (ch == "/" && maybeEnd) {
        state.tokenize = null;
        break;
      }
      maybeEnd = (ch == "*");
    }
    return "comment";
  }

  function Context(indented, column, type, align, prev) {
    this.indented = indented;
    this.column = column;
    this.type = type;
    this.align = align;
    this.prev = prev;
  }
  function pushContext(state, col, type) {
    return state.context = new Context(state.indented, col, type, null, state.context);
  }
  function popContext(state) {
    var t = state.context.type;
    if (t == ")" || t == "]" || t == "}")
      state.indented = state.context.indented;
    return state.context = state.context.prev;
  }

  // Interface

  return {
    startState: function(basecolumn) {
      return {
        tokenize: null,
        context: new Context((basecolumn || 0) - indentUnit, 0, "top", false),
        indented: 0,
        startOfLine: true
      };
    },

    token: function(stream, state) {
      var ctx = state.context;
      if (stream.sol()) {
        if (ctx.align == null) ctx.align = false;
        state.indented = stream.indentation();
        state.startOfLine = true;
      }
      if (stream.eatSpace()) return null;
      curPunc = null;
      var style = (state.tokenize || tokenBase)(stream, state);
      if (style == "comment" || style == "meta") return style;
      if (ctx.align == null) ctx.align = true;

      if ((curPunc == ";" || curPunc == ":") && ctx.type == "statement") popContext(state);
      else if (curPunc == "{") pushContext(state, stream.column(), "}");
      else if (curPunc == "[") pushContext(state, stream.column(), "]");
      else if (curPunc == "(") pushContext(state, stream.column(), ")");
      else if (curPunc == "}") {
        while (ctx.type == "statement") ctx = popContext(state);
        if (ctx.type == "}") ctx = popContext(state);
        while (ctx.type == "statement") ctx = popContext(state);
      }
      else if (curPunc == ctx.type) popContext(state);
      else if (ctx.type == "}" || ctx.type == "top" || (ctx.type == "statement" && curPunc == "newstatement"))
        pushContext(state, stream.column(), "statement");
      state.startOfLine = false;
      return style;
    },

    indent: function(state, textAfter) {
      if (state.tokenize != tokenBase && state.tokenize != null) return 0;
      var ctx = state.context, firstChar = textAfter && textAfter.charAt(0);
      if (ctx.type == "statement" && firstChar == "}") ctx = ctx.prev;
      var closing = firstChar == ctx.type;
      if (ctx.type == "statement") return ctx.indented + (firstChar == "{" ? 0 : indentUnit);
      else if (ctx.align) return ctx.column + (closing ? 0 : 1);
      else return ctx.indented + (closing ? 0 : indentUnit);
    },

    electricChars: "{}"
  };
});

(function() {
  function words(str) {
    var obj = {}, words = str.split(" ");
    for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
    return obj;
  }
  var cKeywords = "auto if break int case long char register continue return default short do sizeof " +
    "double static else struct entry switch extern typedef float union for unsigned " +
    "goto while enum void const signed volatile";

  function cppHook(stream, state) {
    if (!state.startOfLine) return false;
    stream.skipToEnd();
    return "meta";
  }

  // C#-style strings where "" escapes a quote.
  function tokenAtString(stream, state) {
    var next;
    while ((next = stream.next()) != null) {
      if (next == '"' && !stream.eat('"')) {
        state.tokenize = null;
        break;
      }
    }
    return "string";
  }

  CodeMirror.defineMIME("text/x-csrc", {
    name: "clike",
    keywords: words(cKeywords),
    blockKeywords: words("case do else for if switch while struct"),
    atoms: words("null"),
    hooks: {"#": cppHook}
  });
  CodeMirror.defineMIME("text/x-c++src", {
    name: "clike",
    keywords: words(cKeywords + " asm dynamic_cast namespace reinterpret_cast try bool explicit new " +
                    "static_cast typeid catch operator template typename class friend private " +
                    "this using const_cast inline public throw virtual delete mutable protected " +
                    "wchar_t"),
    blockKeywords: words("catch class do else finally for if struct switch try while"),
    atoms: words("true false null"),
    hooks: {"#": cppHook}
  });
  CodeMirror.defineMIME("text/x-java", {
    name: "clike",
    keywords: words("abstract assert boolean break byte case catch char class const continue default " + 
                    "do double else enum extends final finally float for goto if implements import " +
                    "instanceof int interface long native new package private protected public " +
                    "return short static strictfp super switch synchronized this throw throws transient " +
                    "try void volatile while"),
    blockKeywords: words("catch class do else finally for if switch try while"),
    atoms: words("true false null"),
    hooks: {
      "@": function(stream, state) {
        stream.eatWhile(/[\w\$_]/);
        return "meta";
      }
    }
  });
  CodeMirror.defineMIME("text/x-csharp", {
    name: "clike",
    keywords: words("abstract as base bool break byte case catch char checked class const continue decimal" + 
                    " default delegate do double else enum event explicit extern finally fixed float for" + 
                    " foreach goto if implicit in int interface internal is lock long namespace new object" + 
                    " operator out override params private protected public readonly ref return sbyte sealed short" + 
                    " sizeof stackalloc static string struct switch this throw try typeof uint ulong unchecked" + 
                    " unsafe ushort using virtual void volatile while add alias ascending descending dynamic from get" + 
                    " global group into join let orderby partial remove select set value var yield"),
    blockKeywords: words("catch class do else finally for foreach if struct switch try while"),
    atoms: words("true false null"),
    hooks: {
      "@": function(stream, state) {
        if (stream.eat('"')) {
          state.tokenize = tokenAtString;
          return tokenAtString(stream, state);
        }
        stream.eatWhile(/[\w\$_]/);
        return "meta";
      }
    }
  });
  CodeMirror.defineMIME("text/x-scala", {
    name: "clike",
    keywords: words(
      
      /* scala */
      "abstract case catch class def do else extends false final finally for forSome if " +
      "implicit import lazy match new null object override package private protected return " +
      "sealed super this throw trait try trye type val var while with yield _ : = => <- <: " +
      "<% >: # @ " +
                    
      /* package scala */
      "assert assume require print println printf readLine readBoolean readByte readShort " +
      "readChar readInt readLong readFloat readDouble " +
      
      "AnyVal App Application Array BufferedIterator BigDecimal BigInt Char Console Either " +
      "Enumeration Equiv Error Exception Fractional Function IndexedSeq Integral Iterable " +
      "Iterator List Map Numeric Nil NotNull Option Ordered Ordering PartialFunction PartialOrdering " +
      "Product Proxy Range Responder Seq Serializable Set Specializable Stream StringBuilder " +
      "StringContext Symbol Throwable Traversable TraversableOnce Tuple Unit Vector :: #:: " +
      
      /* package java.lang */            
      "Boolean Byte Character CharSequence Class ClassLoader Cloneable Comparable " +
      "Compiler Double Exception Float Integer Long Math Number Object Package Pair Process " +
      "Runtime Runnable SecurityManager Short StackTraceElement StrictMath String " +
      "StringBuffer System Thread ThreadGroup ThreadLocal Throwable Triple Void"
      
      
    ),
    blockKeywords: words("catch class do else finally for forSome if match switch try while"),
    atoms: words("true false null"),
    hooks: {
      "@": function(stream, state) {
        stream.eatWhile(/[\w\$_]/);
        return "meta";
      }
    }
  });
}());
/**
 * Author: Hans Engel
 * Branched from CodeMirror's Scheme mode (by Koh Zi Han, based on implementation by Koh Zi Chun)
 */
CodeMirror.defineMode("clojure", function (config, mode) {
    var BUILTIN = "builtin", COMMENT = "comment", STRING = "string", TAG = "tag",
        ATOM = "atom", NUMBER = "number", BRACKET = "bracket", KEYWORD = "keyword";
    var INDENT_WORD_SKIP = 2, KEYWORDS_SKIP = 1;

    function makeKeywords(str) {
        var obj = {}, words = str.split(" ");
        for (var i = 0; i < words.length; ++i) obj[words[i]] = true;
        return obj;
    }

    var atoms = makeKeywords("true false nil");
    
    var keywords = makeKeywords(
      "defn defn- def def- defonce defmulti defmethod defmacro defstruct deftype defprotocol defrecord defproject deftest slice defalias defhinted defmacro- defn-memo defnk defnk defonce- defunbound defunbound- defvar defvar- let letfn do case cond condp for loop recur when when-not when-let when-first if if-let if-not . .. -> ->> doto and or dosync doseq dotimes dorun doall load import unimport ns in-ns refer try catch finally throw with-open with-local-vars binding gen-class gen-and-load-class gen-and-save-class handler-case handle");

    var builtins = makeKeywords(
        "* *1 *2 *3 *agent* *allow-unresolved-vars* *assert *clojure-version* *command-line-args* *compile-files* *compile-path* *e *err* *file* *flush-on-newline* *in* *macro-meta* *math-context* *ns* *out* *print-dup* *print-length* *print-level* *print-meta* *print-readably* *read-eval* *source-path* *use-context-classloader* *warn-on-reflection* + - / < <= = == > >= accessor aclone agent agent-errors aget alength alias all-ns alter alter-meta! alter-var-root amap ancestors and apply areduce array-map aset aset-boolean aset-byte aset-char aset-double aset-float aset-int aset-long aset-short assert assoc assoc! assoc-in associative? atom await await-for await1 bases bean bigdec bigint binding bit-and bit-and-not bit-clear bit-flip bit-not bit-or bit-set bit-shift-left bit-shift-right bit-test bit-xor boolean boolean-array booleans bound-fn bound-fn* butlast byte byte-array bytes case cast char char-array char-escape-string char-name-string char? chars chunk chunk-append chunk-buffer chunk-cons chunk-first chunk-next chunk-rest chunked-seq? class class? clear-agent-errors clojure-version coll? comment commute comp comparator compare compare-and-set! compile complement concat cond condp conj conj! cons constantly construct-proxy contains? count counted? create-ns create-struct cycle dec decimal? declare definline defmacro defmethod defmulti defn defn- defonce defstruct delay delay? deliver deref derive descendants destructure disj disj! dissoc dissoc! distinct distinct? doall doc dorun doseq dosync dotimes doto double double-array doubles drop drop-last drop-while empty empty? ensure enumeration-seq eval even? every? extend extend-protocol extend-type extends? extenders false? ffirst file-seq filter find find-doc find-ns find-var first float float-array float? floats flush fn fn? fnext for force format future future-call future-cancel future-cancelled? future-done? future? gen-class gen-interface gensym get get-in get-method get-proxy-class get-thread-bindings get-validator hash hash-map hash-set identical? identity if-let if-not ifn? import in-ns inc init-proxy instance? int int-array integer? interleave intern interpose into into-array ints io! isa? iterate iterator-seq juxt key keys keyword keyword? last lazy-cat lazy-seq let letfn line-seq list list* list? load load-file load-reader load-string loaded-libs locking long long-array longs loop macroexpand macroexpand-1 make-array make-hierarchy map map? mapcat max max-key memfn memoize merge merge-with meta method-sig methods min min-key mod name namespace neg? newline next nfirst nil? nnext not not-any? not-empty not-every? not= ns ns-aliases ns-imports ns-interns ns-map ns-name ns-publics ns-refers ns-resolve ns-unalias ns-unmap nth nthnext num number? odd? or parents partial partition pcalls peek persistent! pmap pop pop! pop-thread-bindings pos? pr pr-str prefer-method prefers primitives-classnames print print-ctor print-doc print-dup print-method print-namespace-doc print-simple print-special-doc print-str printf println println-str prn prn-str promise proxy proxy-call-with-super proxy-mappings proxy-name proxy-super push-thread-bindings pvalues quot rand rand-int range ratio? rational? rationalize re-find re-groups re-matcher re-matches re-pattern re-seq read read-line read-string reify reduce ref ref-history-count ref-max-history ref-min-history ref-set refer refer-clojure release-pending-sends rem remove remove-method remove-ns repeat repeatedly replace replicate require reset! reset-meta! resolve rest resultset-seq reverse reversible? rseq rsubseq satisfies? second select-keys send send-off seq seq? seque sequence sequential? set set-validator! set? short short-array shorts shutdown-agents slurp some sort sort-by sorted-map sorted-map-by sorted-set sorted-set-by sorted? special-form-anchor special-symbol? split-at split-with str stream? string? struct struct-map subs subseq subvec supers swap! symbol symbol? sync syntax-symbol-anchor take take-last take-nth take-while test the-ns time to-array to-array-2d trampoline transient tree-seq true? type unchecked-add unchecked-dec unchecked-divide unchecked-inc unchecked-multiply unchecked-negate unchecked-remainder unchecked-subtract underive unquote unquote-splicing update-in update-proxy use val vals var-get var-set var? vary-meta vec vector vector? when when-first when-let when-not while with-bindings with-bindings* with-in-str with-loading-context with-local-vars with-meta with-open with-out-str with-precision xml-seq");

    var indentKeys = makeKeywords(
        // Built-ins
        "ns fn def defn defmethod bound-fn if if-not case condp when while when-not when-first do future comment doto locking proxy with-open with-precision reify deftype defrecord defprotocol extend extend-protocol extend-type try catch " +

        // Binding forms
        "let letfn binding loop for doseq dotimes when-let if-let " +

        // Data structures
        "defstruct struct-map assoc " +

        // clojure.test
        "testing deftest " +

        // contrib
        "handler-case handle dotrace deftrace");

    var tests = {
        digit: /\d/,
        digit_or_colon: /[\d:]/,
        hex: /[0-9a-fA-F]/,
        sign: /[+-]/,
        exponent: /[eE]/,
        keyword_char: /[^\s\(\[\;\)\]]/,
        basic: /[\w\$_\-]/,
        lang_keyword: /[\w*+!\-_?:\/]/
    };

    function stateStack(indent, type, prev) { // represents a state stack object
        this.indent = indent;
        this.type = type;
        this.prev = prev;
    }

    function pushStack(state, indent, type) {
        state.indentStack = new stateStack(indent, type, state.indentStack);
    }

    function popStack(state) {
        state.indentStack = state.indentStack.prev;
    }

    function isNumber(ch, stream){
        // hex
        if ( ch === '0' && 'x' == stream.peek().toLowerCase() ) {
            stream.eat('x');
            stream.eatWhile(tests.hex);
            return true;
        }

        // leading sign
        if ( ch == '+' || ch == '-' ) {
          stream.eat(tests.sign);
          ch = stream.next();
        }

        if ( tests.digit.test(ch) ) {
            stream.eat(ch);
            stream.eatWhile(tests.digit);

            if ( '.' == stream.peek() ) {
                stream.eat('.');
                stream.eatWhile(tests.digit);
            }

            if ( 'e' == stream.peek().toLowerCase() ) {
                stream.eat(tests.exponent);
                stream.eat(tests.sign);
                stream.eatWhile(tests.digit);
            }

            return true;
        }

        return false;
    }

    return {
        startState: function () {
            return {
                indentStack: null,
                indentation: 0,
                mode: false
            };
        },

        token: function (stream, state) {
            if (state.indentStack == null && stream.sol()) {
                // update indentation, but only if indentStack is empty
                state.indentation = stream.indentation();
            }

            // skip spaces
            if (stream.eatSpace()) {
                return null;
            }
            var returnType = null;

            switch(state.mode){
                case "string": // multi-line string parsing mode
                    var next, escaped = false;
                    while ((next = stream.next()) != null) {
                        if (next == "\"" && !escaped) {

                            state.mode = false;
                            break;
                        }
                        escaped = !escaped && next == "\\";
                    }
                    returnType = STRING; // continue on in string mode
                    break;
                default: // default parsing mode
                    var ch = stream.next();

                    if (ch == "\"") {
                        state.mode = "string";
                        returnType = STRING;
                    } else if (ch == "'" && !( tests.digit_or_colon.test(stream.peek()) )) {
                        returnType = ATOM;
                    } else if (ch == ";") { // comment
                        stream.skipToEnd(); // rest of the line is a comment
                        returnType = COMMENT;
                    } else if (isNumber(ch,stream)){
                        returnType = NUMBER;
                    } else if (ch == "(" || ch == "[") {
                        var keyWord = ''; var indentTemp = stream.column();
                        /**
                        Either
                        (indent-word ..
                        (non-indent-word ..
                        (;something else, bracket, etc.
                        */

                        if (ch == "(") while ((letter = stream.eat(tests.keyword_char)) != null) {
                            keyWord += letter;
                        }

                        if (keyWord.length > 0 && indentKeys.propertyIsEnumerable(keyWord)) { // indent-word
                            pushStack(state, indentTemp + INDENT_WORD_SKIP, ch);
                        } else { // non-indent word
                            // we continue eating the spaces
                            stream.eatSpace();
                            if (stream.eol() || stream.peek() == ";") {
                                // nothing significant after
                                // we restart indentation 1 space after
                                pushStack(state, indentTemp + 1, ch);
                            } else {
                                pushStack(state, indentTemp + stream.current().length, ch); // else we match
                            }
                        }
                        stream.backUp(stream.current().length - 1); // undo all the eating

                        returnType = BRACKET;
                    } else if (ch == ")" || ch == "]") {
                        returnType = BRACKET;
                        if (state.indentStack != null && state.indentStack.type == (ch == ")" ? "(" : "[")) {
                            popStack(state);
                        }
                    } else if ( ch == ":" ) {
                        stream.eatWhile(tests.lang_keyword);
                        return ATOM;
                    } else {
                        stream.eatWhile(tests.basic);

                        if (keywords && keywords.propertyIsEnumerable(stream.current())) {
                            returnType = KEYWORD;
                        } else if (builtins && builtins.propertyIsEnumerable(stream.current())) {
                            returnType = BUILTIN;
                        } else if (atoms && atoms.propertyIsEnumerable(stream.current())) {
                            returnType = ATOM;
                        } else returnType = null;
                    }
            }

            return returnType;
        },

        indent: function (state, textAfter) {
            if (state.indentStack == null) return state.indentation;
            return state.indentStack.indent;
        }
    };
});

CodeMirror.defineMIME("text/x-clojure", "clojure");
/**
 * Link to the project's GitHub page:
 * https://github.com/pickhardt/coffeescript-codemirror-mode
 */
CodeMirror.defineMode('coffeescript', function(conf) {
    var ERRORCLASS = 'error';

    function wordRegexp(words) {
        return new RegExp("^((" + words.join(")|(") + "))\\b");
    }

    var singleOperators = new RegExp("^[\\+\\-\\*/%&|\\^~<>!\?]");
    var singleDelimiters = new RegExp('^[\\(\\)\\[\\]\\{\\}@,:`=;\\.]');
    var doubleOperators = new RegExp("^((\->)|(\=>)|(\\+\\+)|(\\+\\=)|(\\-\\-)|(\\-\\=)|(\\*\\*)|(\\*\\=)|(\\/\\/)|(\\/\\=)|(==)|(!=)|(<=)|(>=)|(<>)|(<<)|(>>)|(//))");
    var doubleDelimiters = new RegExp("^((\\.\\.)|(\\+=)|(\\-=)|(\\*=)|(%=)|(/=)|(&=)|(\\|=)|(\\^=))");
    var tripleDelimiters = new RegExp("^((\\.\\.\\.)|(//=)|(>>=)|(<<=)|(\\*\\*=))");
    var identifiers = new RegExp("^[_A-Za-z$][_A-Za-z$0-9]*");

    var wordOperators = wordRegexp(['and', 'or', 'not',
                                    'is', 'isnt', 'in',
                                    'instanceof', 'typeof']);
    var indentKeywords = ['for', 'while', 'loop', 'if', 'unless', 'else',
                          'switch', 'try', 'catch', 'finally', 'class'];
    var commonKeywords = ['break', 'by', 'continue', 'debugger', 'delete',
                          'do', 'in', 'of', 'new', 'return', 'then',
                          'this', 'throw', 'when', 'until'];

    var keywords = wordRegexp(indentKeywords.concat(commonKeywords));

    indentKeywords = wordRegexp(indentKeywords);


    var stringPrefixes = new RegExp("^('{3}|\"{3}|['\"])");
    var regexPrefixes = new RegExp("^(/{3}|/)");
    var commonConstants = ['Infinity', 'NaN', 'undefined', 'null', 'true', 'false', 'on', 'off', 'yes', 'no'];
    var constants = wordRegexp(commonConstants);

    // Tokenizers
    function tokenBase(stream, state) {
        // Handle scope changes
        if (stream.sol()) {
            var scopeOffset = state.scopes[0].offset;
            if (stream.eatSpace()) {
                var lineOffset = stream.indentation();
                if (lineOffset > scopeOffset) {
                    return 'indent';
                } else if (lineOffset < scopeOffset) {
                    return 'dedent';
                }
                return null;
            } else {
                if (scopeOffset > 0) {
                    dedent(stream, state);
                }
            }
        }
        if (stream.eatSpace()) {
            return null;
        }

        var ch = stream.peek();

        // Handle docco title comment (single line)
        if (stream.match("####")) {
            stream.skipToEnd();
            return 'comment';
        }

        // Handle multi line comments
        if (stream.match("###")) {
            state.tokenize = longComment;
            return state.tokenize(stream, state);
        }

        // Single line comment
        if (ch === '#') {
            stream.skipToEnd();
            return 'comment';
        }

        // Handle number literals
        if (stream.match(/^-?[0-9\.]/, false)) {
            var floatLiteral = false;
            // Floats
            if (stream.match(/^-?\d*\.\d+(e[\+\-]?\d+)?/i)) {
              floatLiteral = true;
            }
            if (stream.match(/^-?\d+\.\d*/)) {
              floatLiteral = true;
            }
            if (stream.match(/^-?\.\d+/)) {
              floatLiteral = true;
            }

            if (floatLiteral) {
                // prevent from getting extra . on 1..
                if (stream.peek() == "."){
                    stream.backUp(1);
                }
                return 'number';
            }
            // Integers
            var intLiteral = false;
            // Hex
            if (stream.match(/^-?0x[0-9a-f]+/i)) {
              intLiteral = true;
            }
            // Decimal
            if (stream.match(/^-?[1-9]\d*(e[\+\-]?\d+)?/)) {
                intLiteral = true;
            }
            // Zero by itself with no other piece of number.
            if (stream.match(/^-?0(?![\dx])/i)) {
              intLiteral = true;
            }
            if (intLiteral) {
                return 'number';
            }
        }

        // Handle strings
        if (stream.match(stringPrefixes)) {
            state.tokenize = tokenFactory(stream.current(), 'string');
            return state.tokenize(stream, state);
        }
        // Handle regex literals
        if (stream.match(regexPrefixes)) {
            if (stream.current() != '/' || stream.match(/^.*\//, false)) { // prevent highlight of division
                state.tokenize = tokenFactory(stream.current(), 'string-2');
                return state.tokenize(stream, state);
            } else {
                stream.backUp(1);
            }
        }

        // Handle operators and delimiters
        if (stream.match(tripleDelimiters) || stream.match(doubleDelimiters)) {
            return 'punctuation';
        }
        if (stream.match(doubleOperators)
            || stream.match(singleOperators)
            || stream.match(wordOperators)) {
            return 'operator';
        }
        if (stream.match(singleDelimiters)) {
            return 'punctuation';
        }

        if (stream.match(constants)) {
            return 'atom';
        }

        if (stream.match(keywords)) {
            return 'keyword';
        }

        if (stream.match(identifiers)) {
            return 'variable';
        }

        // Handle non-detected items
        stream.next();
        return ERRORCLASS;
    }

    function tokenFactory(delimiter, outclass) {
        var singleline = delimiter.length == 1;
        return function tokenString(stream, state) {
            while (!stream.eol()) {
                stream.eatWhile(/[^'"\/\\]/);
                if (stream.eat('\\')) {
                    stream.next();
                    if (singleline && stream.eol()) {
                        return outclass;
                    }
                } else if (stream.match(delimiter)) {
                    state.tokenize = tokenBase;
                    return outclass;
                } else {
                    stream.eat(/['"\/]/);
                }
            }
            if (singleline) {
                if (conf.mode.singleLineStringErrors) {
                    outclass = ERRORCLASS
                } else {
                    state.tokenize = tokenBase;
                }
            }
            return outclass;
        };
    }

    function longComment(stream, state) {
        while (!stream.eol()) {
            stream.eatWhile(/[^#]/);
            if (stream.match("###")) {
                state.tokenize = tokenBase;
                break;
            }
            stream.eatWhile("#");
        }
        return "comment"
    }

    function indent(stream, state, type) {
        type = type || 'coffee';
        var indentUnit = 0;
        if (type === 'coffee') {
            for (var i = 0; i < state.scopes.length; i++) {
                if (state.scopes[i].type === 'coffee') {
                    indentUnit = state.scopes[i].offset + conf.indentUnit;
                    break;
                }
            }
        } else {
            indentUnit = stream.column() + stream.current().length;
        }
        state.scopes.unshift({
            offset: indentUnit,
            type: type
        });
    }

    function dedent(stream, state) {
        if (state.scopes.length == 1) return;
        if (state.scopes[0].type === 'coffee') {
            var _indent = stream.indentation();
            var _indent_index = -1;
            for (var i = 0; i < state.scopes.length; ++i) {
                if (_indent === state.scopes[i].offset) {
                    _indent_index = i;
                    break;
                }
            }
            if (_indent_index === -1) {
                return true;
            }
            while (state.scopes[0].offset !== _indent) {
                state.scopes.shift();
            }
            return false
        } else {
            state.scopes.shift();
            return false;
        }
    }

    function tokenLexer(stream, state) {
        var style = state.tokenize(stream, state);
        var current = stream.current();

        // Handle '.' connected identifiers
        if (current === '.') {
            style = state.tokenize(stream, state);
            current = stream.current();
            if (style === 'variable') {
                return 'variable';
            } else {
                return ERRORCLASS;
            }
        }

        // Handle properties
        if (current === '@') {
            stream.eat('@');
            return 'keyword';
        }

        // Handle scope changes.
        if (current === 'return') {
            state.dedent += 1;
        }
        if (((current === '->' || current === '=>') &&
                  !state.lambda &&
                  state.scopes[0].type == 'coffee' &&
                  stream.peek() === '')
               || style === 'indent') {
            indent(stream, state);
        }
        var delimiter_index = '[({'.indexOf(current);
        if (delimiter_index !== -1) {
            indent(stream, state, '])}'.slice(delimiter_index, delimiter_index+1));
        }
        if (indentKeywords.exec(current)){
            indent(stream, state);
        }
        if (current == 'then'){
            dedent(stream, state);
        }


        if (style === 'dedent') {
            if (dedent(stream, state)) {
                return ERRORCLASS;
            }
        }
        delimiter_index = '])}'.indexOf(current);
        if (delimiter_index !== -1) {
            if (dedent(stream, state)) {
                return ERRORCLASS;
            }
        }
        if (state.dedent > 0 && stream.eol() && state.scopes[0].type == 'coffee') {
            if (state.scopes.length > 1) state.scopes.shift();
            state.dedent -= 1;
        }

        return style;
    }

    var external = {
        startState: function(basecolumn) {
            return {
              tokenize: tokenBase,
              scopes: [{offset:basecolumn || 0, type:'coffee'}],
              lastToken: null,
              lambda: false,
              dedent: 0
          };
        },

        token: function(stream, state) {
            var style = tokenLexer(stream, state);

            state.lastToken = {style:style, content: stream.current()};

            if (stream.eol() && stream.lambda) {
                state.lambda = false;
            }

            return style;
        },

        indent: function(state, textAfter) {
            if (state.tokenize != tokenBase) {
                return 0;
            }

            return state.scopes[0].offset;
        }

    };
    return external;
});

CodeMirror.defineMIME('text/x-coffeescript', 'coffeescript');
CodeMirror.defineMode("diff", function() {

  var TOKEN_NAMES = {
    '+': 'tag',
    '-': 'string',
    '@': 'meta'
  };

  return {
    token: function(stream) {
      var tw_pos = stream.string.search(/[\t ]+?$/);

      if (!stream.sol() || tw_pos === 0) {
        stream.skipToEnd();
        return ("error " + (
          TOKEN_NAMES[stream.string.charAt(0)] || '')).replace(/ $/, '');
      }

      var token_name = TOKEN_NAMES[stream.peek()] || stream.skipToEnd();

      if (tw_pos === -1) {
        stream.skipToEnd();
      } else {
        stream.pos = tw_pos;
      }

      return token_name;
    }
  };
});

CodeMirror.defineMIME("text/x-diff", "diff");
/*{id:"uuStorage.js",ver:0.11,license:"MIT",author:"uupaa.js@gmail.com"}*/
var uu;uu||(function(aE,U){uu=bb(w,{ver:aD(0.7),plugin:n,require:W,like:a4,type:bb(aR,{HASH:1,NODE:2,FAKEARRAY:4,DATE:8,NULL:16,VOID:32,UNDEFINED:32,BOOLEAN:64,FUNCTION:128,NUMBER:256,STRING:512,ARRAY:1024,REGEXP:2048,CSS:4096}),isNumber:e,isString:al,isFunction:bl,arg:bf,mix:bb,each:z,keys:ax,values:i,hash:bb(aI,{has:aa,nth:x,size:av,clone:aO,indexOf:T}),array:bb(v,{has:aa,nth:x,size:av,sort:az,clean:o,clone:aO,toHash:ac,unique:bo}),attr:bb(aK,{get:k,set:aL,getAll:an}),css:bb(aB,{get:N,set:bc,opacity:bb(I,{get:u,set:aT})}),style:bb(bg,{quick:r}),id:a0,tag:m,query:a6,klass:bb(aU,{has:G,add:j,sub:aS,toggle:bd}),Class:bb(L,{guid:J,singleton:K}),event:bb(a2,{has:O,fire:l,stop:ai,unbind:f,attach:H,detach:aq}),node:bb(a3,{add:q,has:P,root:U.documentElement||U.html,bulk:aw,swap:aQ,wrap:bq,clear:c,clone:aw,remove:ab}),nodeid:bb(a7,{toNode:bm,remove:au}),fix:t,trim:bb(h,{tag:aN,url:ba,inner:ak,quote:aW,bracket:aZ}),split:bb(aA,{comma:be,toHash:ar}),format:bk,json:bb(aV,{decode:C}),date:ae,puff:X,trace:bb(a9,{clear:bp}),ready:bb(aH,{gone:{dom:0,win:0,audio:0,video:0,canvas:0,storage:0,blackout:0}}),lazy:bb(ap,{fire:V}),js:at,win:{size:y},nop:function(){},dmz:{},guid:aG});E.prototype={send:bh,post:aC,register:R,unregister:ay};uu.msg=new E();Array.isArray||(Array.isArray=Y);bb(Array.prototype,{map:a1,some:bi,every:D,filter:d,forEach:b,indexOf:S,lastIndexOf:aX},0,0);bb(Array.prototype,{reduce:aY,reduceRight:aj},0,0);bb(Boolean.prototype,{toJSON:aF},0,0);bb(Date.prototype,{toISOString:a,toJSON:a},0,0);bb(Number.prototype,{toJSON:aF},0,0);bb(String.prototype,{trim:B,toJSON:ah},0,0);uu.ver.gecko&&aE.HTMLElement&&!aE.HTMLElement.prototype.innerText&&(function(br){br.__defineGetter__("innerText",F);br.__defineSetter__("innerText",s);br.__defineGetter__("outerHTML",am);br.__defineSetter__("outerHTML",ad)})(aE.HTMLElement.prototype);uu.ie=uu.ver.ie;uu.opera=uu.ver.opera;uu.gecko=uu.ver.gecko;uu.webkit=uu.ver.webkit;uu.config=bf(aE.xconfig,{aria:false,debug:false,right:false,altcss:0,storage:0,visited:false,baseDir:m("script").pop().src.replace(/[^\/]+$/,"")});U.html||(U.html=m("html")[0]);U.head||(U.head=m("head")[0]);aE.xver&&aE.xver(uu.ver);function w(bv,bu,bt,bs,br){if(typeof bv==="string"&&L[bv]){return new L[bv](bu,bt,bs,br)}return new uu.jam(bv,bu,bt,bs,br)}function n(){return ax(n)}function W(bs){var bv,br=400,bt=false;try{bv=aE.ActiveXObject?new ActiveXObject("Microsoft.XMLHTTP"):aE.XMLHttpRequest?new XMLHttpRequest():0;bv.open("GET",bs,false);bv.send(null);if(!bv.status||(bv.status>=200&&bv.status<300)){bt=true;br=bv.status||200}}catch(bu){bv=bv||{responseText:"",responseXML:"",status:400}}return{ok:bt,url:bs,xhr:bv,guid:aG(),status:br,isXMLContent:function(){return/xml/i.test(this.xhr.getResponseHeader("Content-Type")||"")}}}function a4(br,bt){var bs=aR(br);if(bs!==aR(bt)){return false}switch(bs){case 128:return false;case 8:return ae(br).ISO()===ae(bt).ISO();case 1:return(av(br)===av(bt)&&aa(br,bt));case 4:case 1024:return v(br)+""==v(bt)}return br===bt}function aR(bs,br){var bt=aR._TYPEOF[typeof bs]||aR._TYPEOF[Object.prototype.toString.call(bs)]||(!bs?16:bs.nodeType?2:"length" in bs?4:1);return br?!!(br&bt):bt}aR._TYPEOF={"undefined":aR.VOID,"[object Boolean]":64,"boolean":64,"[object Number]":256,number:256,"[object String]":512,string:512,"[object Function]":128,"function":128,"[object RegExp]":2048,"[object Array]":1024,"[object Date]":8,"[object CSSStyleDeclaration]":4096,"[object ComputedCSSStyleDeclaration]":4096};function e(br){return typeof br==="number"}function al(br){return typeof br==="string"}function Y(br){return Object.prototype.toString.call(br)==="[object Array]"}function aJ(br){return Object.prototype.toString.call(br)==="[object Date]"}function bl(br){return Object.prototype.toString.call(br)==="[object Function]"}function bf(bt,bs,br){return bb(bb({},bt||{}),bs,br,0)}function bb(bv,br,bu,bt){var bs;if(bt===void 0||bt){for(bs in br){bv[bs]=br[bs]}}else{for(bs in br){bs in bv||(bv[bs]=br[bs])}}return bu?bb(bv,bu,0,bt):bv}function z(bs,bt){if(Array.isArray(bs)){bs.forEach(bt)}else{for(var br in bs){bt(bs[br],br)}}}function ax(bu,bt){var bw=[],br=-1,bs,bv;if(Array.isArray(bu)){for(bs=0,bv=bu.length;bs<bv;++bs){bs in bu&&(bw[++br]=bt?bu[bs]:bs)}}else{if(!bt&&Object.keys){return Object.keys(bu)}for(bs in bu){if(bu.hasOwnProperty(bs)){bw[++br]=bt?bu[bs]:bs}}}return bw}function i(br){return ax(br,true)}function aI(br,bs){if(arguments.length===1){return br}var bt={};bt[br]=bs;return bt}a8("0123456789",aI._dd2num={},aI._num2dd={});a8("0123456789abcdef",aI._hh2num={},aI._num2hh={});function v(bt){var bs=aR(bt),bv,br,bu;if(bs===4){for(bv=[],br=0,bu=bt.length;br<bu;++br){bv[br]=bt[br]}return bv}return(bs===1024)?bt:[bt]}function aa(bt,bs){if(bt&&bs){var br,bu;if(Array.isArray(bt)){bs=v(bs);for(br=0,bu=bs.length;br<bu;++br){if(br in bs&&bt.indexOf(bs[br])<0){return false}}return true}for(br in bs){if(!(br in bt)){return false}if(bt[br]!==bs[br]&&ao(bt[br])!==ao(bs[br])){return false}}return true}return false}function x(bu,bs){var bt,br=0;if(Array.isArray(bu)){if(bt in bu){return[bt,bu[bt]]}}else{for(bt in bu){if(br++===bs){return[bt,bu[bt]]}}}return[void 0,void 0]}function av(br){return Array.isArray(br)?br.length:ax(br).length}function aO(br){return Array.isArray(br)?br.concat():bb({},br)}function T(bt,bs){for(var br in bt){if(bt.hasOwnProperty(br)&&bt[br]===bs){return br}}return void 0}function az(bs,bt){function br(bv,bu){return bv-bu}if(Array.isArray(bs)){switch(bt||"0-9"){case"0-9":bs.sort(br);break;case"9-0":bs.sort(br).reverse();break;case"A-Z":bs.sort();break;case"Z-A":bs.sort().reverse();break;default:bs.sort(bt)}}return bs}function o(bs){if(Array.isArray(bs)){var bu=[],br=0,bt=bs.length;for(;br<bt;++br){if(br in bs){if(bs[br]!=null){bu.push(bs[br])}}}return bu}return bs}function bo(br,bx){bx=bx?true:false;if(Array.isArray(br)){var bt=[],by=-1,bz,bw=0,bv,bs=br.length,bA,bu={};for(;bw<bs;++bw){bz=br[bw];if(bz!=null){if(bx){bu[bz]||(bu[bz]=1,bt[++by]=bz)}else{for(bA=0,bv=bw-1;!bA&&bv>=0;--bv){bA=(bz===br[bv])}!bA&&(bt[++by]=bz)}}}return bt}return br}function ac(bt,bu,br){var bx={},bs=0,bw=bt.length,bv;if(Array.isArray(bu)){for(;bs<bw;++bs){bx[bt[bs]]=br?+(bu[bs]):bu[bs]}}else{bv=br?+(bu):bu;for(;bs<bw;++bs){bx[bt[bs]]=bv}}return bx}function aK(bs,br,bt){return(bt===void 0&&al(br)?k:aL)(bs,br,bt)}aK._HASH=ar(uu.ver.ie67?"for,htmlFor,className,class":("class,className,for,htmlFor,colspan,colSpan,accesskey,accessKey,rowspan,rowSpan,tabindex,tabIndex"));function k(bu,by){var bt={},bs=by.split(","),bz,bx,bw=0,br=bs.length,bv=aK._HASH;for(;bw<br;++bw){bz=bs[bw];bx=bv[bz]||bz;if(uu.ie){if(uu.ver.ie89||bz==="href"||bz==="src"){bt[bz]=bu.getAttribute(bz,2)||""}else{bt[bz]=bu[bx]||""}}else{bt[bz]=bu.getAttribute(bx)||""}}return(bs.length===1)?bt[bs[0]]:bt}function aL(bu,bt,bv){var bw,bs,br=aK._HASH;al(bt)?(bw={},bw[bt]=bv):(bw=bt);for(bs in bw){bu.setAttribute(br[bs]||bs,bw[bs])}return bu}function an(bw,bv){bv=bv?true:false;var bx={},bu=bw.attributes,bs,br,bt=-1;while((bs=bu[++bt])){br=bs.name;if(!bv){bx[br]=bs.value}else{if(bs.specified&&br!=="style"&&br.indexOf("uu")){bx[br]=bs.value}}}return bx}function aB(bs,br,bt){return(bt===void 0&&al(br)?N:bc)(bs,br,bt)}function N(bx,bw){var by={},bt=bw.split(","),br,bs=-1,bu=bg(bx),bv=t._db;while((br=bt[++bs])){by[br]=bu[bv[br]||br]||""}return(bt.length===1)?by[bt[0]]:by}function bc(bt,bA,bz){var bu=aI(bA,bz),bx=bt.style,br,bB,bv,bs,by=t._db,bw=bc._hook;for(bv in bu){bB=bu[bv];br=by[bv]||bv;if(typeof bB==="string"){bx[br]=bB}else{bs=bw[br];if(bs===2){aT(bt,bB)}else{bx[br]=bs?bB:(bB+"px")}}}return bt}bc._hook={opacity:2,lineHeight:1,fontWeight:1,fontSizeAdjust:1,zIndex:1,zoom:1};function I(bt,bs,br){return(bs===void 0?u:aT)(bt,bs,br)}function u(bs){if(uu.ie){var br=bs.uucssopacity;return br===void 0?1:(br-1)}return parseFloat(bs.style.opacity||aE.getComputedStyle(bs,null).opacity)}function aT(bu,bs,br){var bt;if(uu.ver.ie678){bt=bu.style;if(bu.uucssopacity===void 0){if(uu.ver.ie67){if((bu.currentStyle||{}).width==="auto"){bt.zoom=1}}}}br&&(bs+=u(bu));bs=(bs>0.999)?1:(bs<0.001)?0:bs;bu.style.opacity=bs;if(uu.ver.ie678){bu.uucssopacity=bs+1;bt.visibility=bs?"":"hidden";bt.filter=((bs>0&&bs<1)?"alpha(opacity="+(bs*100)+") ":"")+bt.filter.replace(aT._alpha,"")}return bu}aT._alpha=/^alpha\([^\x29]+\) ?/;function bg(bG,bF){if(uu.ver.ie678){if(bF===4){return bG.currentStyle}if(!bG.currentStyle){return{}}var by={},bL=bG.style,bw=bG.currentStyle,bz=bG.runtimeStyle,bC=bg._HASH.box,bx=bg._UNITS,bH=bg._RECTANGLE,bu,bv,bA,bE,bD,bB,bJ=-1,bI=-1,bt,br,bK=!bF?bg._HASH.full:(bF===1)?bg._HASH.more:0,bs={"0px":"0px","1px":"1px","2px":"2px","5px":"5px",thin:"1px",medium:"3px",thick:bg._THICK_FIX};if(bK){while((bD=bK[++bI])){by[bD]=bw[bD]}}bu=parseFloat(bw.fontSize)*(bg._UNIT_PT.test(bw.fontSize)?4/3:1);bv=bG.getBoundingClientRect();while((bD=bC[++bJ])){bE=bw[bD];if(!(bE in bs)){bB=bE;switch(bA=bx[bE.slice(-1)]){case 1:bB=parseFloat(bE)*bu;break;case 2:bB=parseFloat(bE)*4/3;break;case 3:bt=bL.left,br=bz.left;bz.left=bw.left,bL.left=bE;bB=bL.pixelLeft,bL.left=bt,bz.left=br}bs[bE]=bA?bB+"px":bB}by[bD]=bs[bE]}for(bD in bH){bE=bw[bD];switch(bA=bx[bE.slice(-1)]){case 1:bE=parseFloat(bE)*bu;break;case 2:bE=parseFloat(bE)*4/3;break;case 3:switch(bH[bD]){case 1:bE=bG.offsetTop;break;case 2:bE=bG.offsetLeft;break;case 3:bE=(bG.offsetWidth||bv.right-bv.left)-parseInt(by.borderLeftWidth)-parseInt(by.borderRightWidth)-parseInt(by.paddingLeft)-parseInt(by.paddingRight);bE=bE>0?bE:0;break;case 4:bE=(bG.offsetHeight||bv.bottom-bv.top)-parseInt(by.borderTopWidth)-parseInt(by.borderBottomWidth)-parseInt(by.paddingTop)-parseInt(by.paddingBottom);bE=bE>0?bE:0}}by[bD]=bA?bE+"px":bE}by.fontSize=bu+"px";by.cssFloat=bw.styleFloat;return by}return aE.getComputedStyle(bG,null)}bg._HASH=uu.ver.ie678?Z():{};bg._UNITS={m:1,t:2,"%":3,o:3};bg._UNIT_PT=/pt$/;bg._THICK_FIX=uu.ver.ie89?"5px":"6px";bg._RECTANGLE={top:1,left:2,width:3,height:4};function r(br){return bg(br,4)}function Z(){var bw={full:[],more:[],box:[]},bu=[" "],bt,bs,br=/^\s+|\s+$/g,bv=U.html.currentStyle;for(bt in bv){bu.push(bt)}bu.sort();bs=bu.join(" ").replace(/ (?:accelerator|behavior|hasLayout|zoom)/g,"");bw.full=bs.replace(br,"").split(" ");bw.more=bs.replace(/ (?:lay\w+|rub\w+|text\w+|pageB\w+|ms\w+|scr\w+)/g,"").replace(/ (?:blockDirection|orphans|quotes|widows|filter|styleFloat)/g,"").replace(/ (?:imeMode|writingMode|unicodeBidi|emptyCells|tableLayout)/g,"").replace(/ (?:border(?:Color|Style|Width)|margin|padding|outline) /g," ").replace(/ (border\w+Width|margin\w+|padding\w+)/g,function(by,bx){return bw.box.push(bx),by}).replace(br,"").concat(" textAlign textOverflow textIndent").split(" ").sort();return bw}function G(bt,bu){var br,bs,bv=bt.className;if(!bu||!bv){return false}if(bu.indexOf(" ")<0){return(" "+bv+" ").indexOf(" "+bu+" ")>=0}bs=aA(bu);br=bv.match(bj(bs));return br&&br.length>=bs.length}function j(br,bs){br.className+=" "+bs;return br}function aS(br,bs){br.className=ak(br.className.replace(bj(aA(bs)),""));return br}function bd(br,bs){(G(br,bs)?aS:j)(br,bs);return br}function L(bz,bx){var bs=bz.split(/\s*[\x3a-\x40]\s*/),bv,bw,br=bs[0],by=bs[1]||"";L[br]=function bu(){var bA=this,bB=bA.superClass||0,bC=bB?bB.superClass:0;J(bA);bA.msgbox||(bA.msgbox=uu.nop);uu.msg.register(bA);bC&&bC.init&&bC.init.apply(bA,arguments);bB&&bB.init&&bB.init.apply(bA,arguments);bA.init&&bA.init.apply(bA,arguments);bA["~fin"]=bA.fin||uu.nop;if(bA.fin){uu.event.attach(aE,"unload",function(){bA.fin&&bA.fin()})}bA.fin=function bD(){bA["~fin"]();bB&&bB.fin&&bB.fin.call(bA);bC&&bC.fin&&bC.fin.call(bA);for(var bE in bA){bA[bE]=null}}};L[br].prototype=bx||{};if(by){bv=function(){};bv.prototype=uu.Class[by].prototype;L[br].prototype=new bv;for(bw in bx){L[br].prototype[bw]=bx[bw]}L[br].prototype.constructor=L[br];L[br].prototype.superClass=uu.Class[by].prototype;L[br].prototype.superMethod=bt}function bt(bC,bB){var bA=this.superClass;if(bC===bA[bB]||bt.caller===bA[bB]){bA=bA.superClass}return bA[bB].apply(this,uu.array(arguments).slice(2))}}function J(br){return br.uuguid||(br.uuguid=uu.guid())}function K(br,bs){L[br]=function(){var bv=this,bt=arguments,bu=bt.callee;if(bu.instance){bv.stable&&bv.stable.apply(bv,bt)}else{J(bv);bv.init&&bv.init.apply(bv,bt);bv.fin&&uu.event.attach(aE,"unload",function(){bv.fin()});bv.msgbox||(bv.msgbox=uu.nop);uu.msg.register(bv)}return bu.instance||(bu.instance=bv)};L[br].prototype=bs||{}}function E(){this._addressMap={};this._broadcast=[]}function bh(by,bz,bt){var bs=[],bw=-1,bx,bu,bv=-1,br=by?uu.array(by):this._broadcast;while((bx=br[++bv])){bu=this._addressMap[bx.uuguid||bx||0];bu&&(bs[++bw]=bu.msgbox(bz,bt))}return bs}function aC(br,bt,bu){var bs=this;setTimeout(function(){bs.send(br?uu.array(br):bs._broadcast,bt,bu)},0)}function R(br){this._addressMap[J(br)]=br;this._broadcast=uu.keys(this._addressMap)}function ay(br){delete this.db[J(br)];this._broadcast=uu.keys(this._addressMap)}function a2(bt,bE,bx,bC){function by(bH,bI){bH=bH||aE.event;if(!bI&&!bH.code){var bK=bH.srcElement||bH.target,bJ;bK=(uu.webkit&&bK.nodeType===3)?bK.parentNode:bK;bH.code=(bF[bH.type]||0)&255;bH.node=bt;bH.src=bK;if(uu.ie){bJ=uu.quirks?U.body:uu.node.root;bH.px=bH.clientX+bJ.scrollLeft;bH.py=bH.clientY+bJ.scrollTop}else{bH.px=bH.pageX;bH.py=bH.pageY}bH.ox=bH.offsetX||bH.layerX||0;bH.oy=bH.offsetY||bH.layerY||0}bu?bG.call(bx,bH,bt,bK):bx(bH,bt,bK)}var bz=bt.uueventtypes||(bt.uueventfn={},bt.uueventtypes=","),bw=bE.split(","),bD,bv=-1,bs,bB,br,bA,bG,bu=false,bF=a2._CODE;if(bC){bA=bx.uueventclosure||bx}else{bG=bl(bx)?bx:(bu=true,bx.handleEvent);bA=bx.uueventclosure=by}while((bD=bw[++bv])){bs=a2._PARSE.exec(bD);if(bs){bB=bs[2];br=bs[3];if(uu.ie){if(br&&bB==="mousemove"){a2(bt,"losecapture",bA,bC)}}if(bz.indexOf(","+bD+",")>=0){if(bC){if(uu.ie){if(bB==="losecapture"&&bt.releaseCapture){bt.releaseCapture()}}bt.uueventtypes=bt.uueventtypes.replace(","+bD+",",",");bt.uueventfn[bD]=void 0;aq(bt,bB,bA,br)}}else{if(!bC){uu.ie&&bB==="losecapture"&&bt.setCapture&&bt.setCapture();bt.uueventtypes+=bD+",";bt.uueventfn[bD]=bA;H(bt,bB,bA,br)}}}}return bt}a2._PARSE=/^(?:(\w+)\.)?(\w+)(\+)?$/;a2._LIST=("mousedown,mouseup,mousemove,mousewheel,click,dblclick,keydown,keypress,keyup,change,submit,focus,blur,contextmenu").split(",");a2._CODE={mousedown:1,mouseup:2,mousemove:3,mousewheel:4,click:5,dblclick:6,keydown:7,keypress:8,keyup:9,mouseenter:10,mouseleave:11,mouseover:12,mouseout:13,contextmenu:14,focus:15,blur:16,resize:17,losecapture:258,DOMMouseScroll:260};function O(bs,br){return(bs.uueventtypes||"").indexOf(","+br+",")>=0}function l(bs,br,bt){if(uu.event.has(bs,br)){bs.uueventfn[br].call(bs,{stopPropagation:uu.nop,preventDefault:uu.nop,node:bs,name:br,code:0,src:bs,rel:bs,px:0,py:0,ox:0,oy:0,type:br,param:bt},true)}return bs}function ai(br){uu.ie?(br.cancelBubble=true):br.stopPropagation();uu.ie?(br.returnValue=false):br.preventDefault();return br}function f(by,bs){function br(bz){!bz.indexOf(bx)&&a2(by,bz,by.uueventfn[bz],true)}var bw=by.uueventtypes,bv,bt,bu=-1,bx;if(bw&&bw.length>1){if(bs){bv=be(bs);while((bt=bv[++bu])){if(bt.lastIndexOf(".*")>1){bx=bt.slice(0,-1);be(bw).forEach(br)}else{(bw.indexOf(","+bt+",")>=0)&&a2(by,bt,by.uueventfn[bt],true)}}}else{bv=be(bw);while((bt=bv[++bu])){a2(by,bt,by.uueventfn[bt],true)}}}return by}function H(bt,bs,bu,br){bs=H._FIX[bs]||bs;if(bt.addEventListener){bt.addEventListener(bs,bu,!!(br||0))}else{bt.attachEvent("on"+bs,bu)}}H._FIX=uu.gecko?{mousewheel:"DOMMouseScroll"}:uu.opera?{contextmenu:"mousedown"}:{};function aq(bt,bs,bu,br){bs=H._FIX[bs]||bs;if(bt.removeEventListener){bt.removeEventListener(bs,bu,!!(br||0))}else{bt.detachEvent("on"+bs,bu)}}function aH(bs,br){if(bs!==void 0&&!aH.gone.blackout){aH.gone.dom?bs(uu):ap("boot",bs,br||0)}}function a3(bs,br){var bt=U.createElement(bs||"div");return br===void 0?bt:bb(bt,br)}function q(bv,bt,bs){bt=bt||U.body;var bu=!bv?U.createElement("div"):bv.nodeType?bv:!bv.indexOf("<")?aw(bv):U.createElement(bv),br=bt.parentNode,bw=(bu.nodeType===11)?bu.firstChild:bu;switch(bs||6){case 1:br.insertBefore(bu,br.firstChild);break;case 2:br.insertBefore(bu,bt);break;case 3:br.insertBefore(bu,bt.nextSibling);break;case 4:br.appendChild(bu);break;case 5:bt.insertBefore(bu,bt.firstChild);break;case 6:bt.appendChild(bu)}return bw}function a7(br){if(!br.uuguid){a7._db[br.uuguid=++a7._num]=br}return br.uuguid}a7._num=0;a7._db={};function bm(br){return a7._db[br]}function au(br){br.uuguid&&(a7._db[br.uuguid]=null,br.uuguid=null);return br}function P(bs,br){for(var bt=bs;bt&&bt!==br;){bt=bt.parentNode}return bs!==br&&bt===br}function aw(br){var bt=U.createDocumentFragment(),bs=a3();bs.innerHTML=al(br)?br:br.outerHTML;while(bs.firstChild){bt.appendChild(bs.removeChild(bs.firstChild))}return bt}function aQ(br,bs){return bs.parentNode.replaceChild(br,bs)}function bq(bs,br){return br.appendChild(aQ(br,bs))}function c(bt){var bu=uu.tag("*",bt),br,bs=-1;while((br=bu[++bs])){au(br);f(br)}bu=[];while(bt.firstChild){bt.removeChild(bt.firstChild)}return bt}function ab(br){if(br&&br.parentNode){au(br);return br.parentNode.removeChild(br)}return br}function a6(bv,bt){if(bt&&U.querySelectorAll&&bt.nodeType&&!a6._NGWORD.test(bv)){try{var bx=[],br=(bt||U).querySelectorAll(bv),bs=0,bw=br.length;for(;bs<bw;++bs){bx[bs]=br[bs]}return bx}catch(bu){}}return a6.selectorAll(bv,bt||U)}a6._NGWORD=/(:(a|b|co|dig|first-l|li|mom|ne|p|sc|t|v))|!=|\/=|<=|>=|&=|x7b/;function a0(bs,br){return(br||U).getElementById(bs)}function m(bw,bv){var bt=(bv||U).getElementsByTagName(bw),by=[],br=-1,bs,bu=0,bx=bt.length;if(uu.ie&&bw==="*"){for(;bu<bx;++bu){(bs=bt[bu]).nodeType===1&&(by[++br]=bs)}}else{for(;bu<bx;++bu){by[bu]=bt[bu]}}return by}m.HTML4=("a,b,br,dd,div,dl,dt,h1,h2,h3,h4,h5,h6,i,img,iframe,input,li,ol,option,p,pre,select,span,table,tbody,tr,td,th,tfoot,textarea,u,ul").split(",");m.HTML5=("abbr,article,aside,audio,bb,canvas,datagrid,datalist,details,dialog,eventsource,figure,footer,header,hgroup,mark,menu,meter,nav,output,progress,section,time,video").split(",");function aU(bA,bs){bs=bs||U;var bu=[],bC=-1,by=0,bt,bD,bv,bz,bB,bx,bw,br;if(bs.getElementsByClassName){bv=bs.getElementsByClassName(bA);for(bt=bv.length;by<bt;++by){bu[by]=bv[by]}}else{bv=bs.getElementsByTagName("*");br=aA(bA);br.length>1&&(br=bo(br,1));bw=bj(br);for(bx=br.length,bt=bv.length;by<bt;++by){bD=bv[by];bB=bD.className;if(bB){bz=bB.match(bw);(bz&&bz.length>=bx)&&(bu[++bC]=bD)}}}return bu}function t(br){return t._db[br]||br}t._db={};function h(br){return br.replace(h._TRIM,"")}h._TAG=/<\/?[^>]+>/g;h._TRIM=/^\s+|\s+$/g;h._QUOTE=/^\s*["']?|["']?\s*$/g;h._SPACES=/\s\s+/g;h._BRACKET=/^\s*[\(\[\{<]?|[>\}\]\)]?\s*$/g;function aN(br){return br.replace(h._TRIM,"").replace(h._TAG,"").replace(h._SPACES," ")}function ba(br){return(!br.indexOf("url(")&&br.indexOf(")")===br.length-1)?br.slice(4,-1).replace(h._QUOTE,""):br}function ak(br){return br.replace(h._TRIM,"").replace(h._SPACES," ")}function aW(br){return br.replace(h._QUOTE,"")}function aZ(br){return br.replace(h._BRACKET,"")}function aA(br){return br.replace(h._SPACES," ").replace(h._TRIM,"").split(" ")}aA._MANY_COMMAS=/,,+/g;aA._TRIM_SPACE_AND_COMMAS=/^[ ,]+|[ ,]+$/g;function be(br){return br.replace(aA._MANY_COMMAS,",").replace(aA._TRIM_SPACE_AND_COMMAS,"").split(",")}function ar(bw,bv,br){var by={},bu=bw.split(bv||","),bt=0,bx=bu.length,bs=br?true:false;for(;bt<bx;bt+=2){by[bu[bt]]=bs?+(bu[bt+1]):bu[bt+1]}return by}function bk(bt){var bs=0,br=arguments;return bt.replace(bk._PLACEHOLDER,function(){return br[++bs]})}bk._PLACEHOLDER=/\?/g;function X(br){alert(ao(br))}function a9(br,bu){var bs=a0("trace"),bt,bv="";if(bs){if(bu!==void 0){bv=br;bt=ao(bu)}else{bt=ao(br)}if(bs.tagName.toLowerCase()==="textarea"){bs.value+=bv+bt}else{bs.innerHTML+="<p>"+bv+bt+"</p>"}}}function bp(){var br=a0("trace");if(br){if(br.tagName.toLowerCase()==="textarea"){br.value=""}else{br.innerHTML=""}}}function aV(bs,br,bt){return br&&aE.JSON?aE.JSON.stringify(bs)||"":ao(bs,bt)}function C(bs,br){var bt=h(bs);if(br&&aE.JSON){return aE.JSON.parse(bt)}return C._NGWORD.test(bt.replace(C._UNESCAPE,""))?false:at("return "+bt+";")}C._NGWORD=/[^,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t]/;C._UNESCAPE=/"(\\.|[^"\\])*"/g;function ag(bu,bs){function bt(bw){return ag._SWAP[bw]}function br(bw,bx){bx=bw.charCodeAt(0);return"\\u"+aI._num2hh[(bx>>8)&255]+aI._num2hh[bx&255]}var bv=bu.replace(ag._ESCAPE,bt).replace(ag._ENCODE,br);return bs?'"'+bv+'"':bv}ag._SWAP=ar('",\\",\b,\\b,\f,\\f,\n,\\n,\r,\\r,\t,\\t,\\,\\\\');ag._ESCAPE=/(?:\"|\\[bfnrt\\])/g;ag._ENCODE=/[\x00-\x1F\u0080-\uFFEE]/g;function ao(bx,bw){var bu,bv=aR(bx),bs,br=-1,bt,by;if(bx===aE){return'"window"'}switch(bv){case 4096:case 1:bu=[];break;case 2:return'"uuguid":'+a7(bx);case 16:return"null";case aR.VOID:return"undefined";case 8:return ae(bx).ISO();case 64:case 128:case 256:return bx.toString();case 512:return ag(bx,1);case 1024:case 4:for(bu=[],bt=0,by=bx.length;bt<by;++bt){bu[++br]=ao(bx[bt],bw)}return"["+bu+"]";default:return bw?(bw(bx)||""):""}if(bv===4096){bs=uu.webkit;for(bt in bx){if(typeof bx[bt]==="string"&&(bs||bt!=(+bt+""))){bs&&(bt=bx.item(bt));bu[++br]='"'+bt+'":'+ag(bx[bt],1)}}}else{for(bt in bx){bu[++br]=ag(bt,1)+":"+ao(bx[bt],bw)}}return"{"+bu+"}"}function ae(br){return br===void 0?af(new Date()):aJ(br)?af(br):e(br)?af(new Date(br)):br.GMT?aO(br):af(p(br)||new Date(br))}function af(br){return{Y:br.getUTCFullYear(),M:br.getUTCMonth()+1,D:br.getUTCDate(),h:br.getUTCHours(),m:br.getUTCMinutes(),s:br.getUTCSeconds(),ms:br.getUTCMilliseconds(),time:br.getTime(),ISO:Q,RFC:A,GMT:A}}function p(bt){function bs(bw,bv,bu,bx){return bv+" "+bx+" "+bu}var br=p._PARSE.exec(bt);if(br){return new Date(Date.UTC(+br[1],+br[2]-1,+br[3],+br[4],+br[5],+br[6],+br[7]))}if(uu.ie&&bt.indexOf("GMT")>0){bt=bt.replace(/GMT/,"UTC")}return new Date(bt.replace(",","").replace(p._DATE,bs))}p._PARSE=/^(\d{4})-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)(?:\.(\d*))?Z$/;p._DATE=/^([\w]+) (\w+) (\w+)/;function Q(){var bs=(this.ms<10)?"00":(this.ms<100)?"0":"",br=aI._num2dd;return bk("?-?-?T?:?:?.?Z",this.Y,br[this.M],br[this.D],br[this.h],br[this.m],br[this.s],bs+this.ms)}function A(){var br=(new Date(this.time)).toUTCString();if(uu.ie&&br.indexOf("UTC")>0){br=br.replace(/UTC/,"GMT");(br.length<29)&&(br=br.replace(/, /,", 0"))}return br}function at(br){return(new Function(br))()}function y(){if(uu.ie){var br=uu.quirks?U.body:uu.node.root;return{innerWidth:br.clientWidth,innerHeight:br.clientHeight,pageXOffset:br.scrollLeft,pageYOffset:br.scrollTop}}return{innerWidth:aE.innerWidth,innerHeight:aE.innerHeight,pageXOffset:aE.pageXOffset,pageYOffset:aE.pageYOffset}}function aG(){return ++aG._num}aG._num=0;function ap(bt,bs,br){ap._db[bt]||(ap._db[bt]=[[],[],[]]);ap._db[bt][br||0].push(bs)}ap._db={};function V(bv){if(ap._db[bv]){var bu,bt=-1,br=ap._db[bv],bs=br[2].concat(br[1],br[0]);ap._db[bv]=null;while((bu=bs[++bt])){bu(uu)}}}function S(bs,bt){var bu=this.length,br=bt||0;br=(br<0)?br+bu:br;for(;br<bu;++br){if(br in this&&this[br]===bs){return br}}return -1}function aX(bs,bt){var bu=this.length,br=bt;br=(br<0)?br+bu+1:bu;while(--br>=0){if(br in this&&this[br]===bs){return br}}return -1}function D(bu,bs){for(var br=0,bt=this.length;br<bt;++br){if(br in this&&!bu.call(bs,this[br],br,this)){return false}}return true}function bi(bu,bs){for(var br=0,bt=this.length;br<bt;++br){if(br in this&&bu.call(bs,this[br],br,this)){return true}}return false}function b(bu,bs){var br=0,bt=this.length;if(bs){for(;br<bt;++br){br in this&&bu.call(bs,this[br],br,this)}}else{for(;br<bt;++br){br in this&&bu(this[br],br,this)}}}function a1(bv,bs){for(var bu=this.length,bt=Array(bu),br=0;br<bu;++br){br in this&&(bt[br]=bv.call(bs,this[br],br,this))}return bt}function d(bx,bu){for(var bw=[],br=-1,bs,bt=0,bv=this.length;bt<bv;++bt){bt in this&&bx.call(bu,bs=this[bt],bt,this)&&(bw[++br]=bs)}return bw}function aY(bx,br){var bu,bt=0,bw=br===bu?bu:(++bt,br),bs=0,bv=this.length;for(;bs<bv;++bs){bs in this&&(bw=bt?bx(bw,this[bs],bs,this):(++bt,this[bs]))}if(!bt){throw new Error(aY._MSG)}return bw}aY._MSG="reduce of empty array with no initial value";function aj(bw,br){var bu,bt=0,bv=br===bu?bu:(++bt,br),bs=this.length;while(--bs>=0){bs in this&&(bv=bt?bw(bv,this[bs],bs,this):(++bt,this[bs]))}if(!bt){throw new Error(aY._MSG)}return bv}function a(){return ae(this).ISO()}function aF(){return this.toString()}function B(){return this.replace(h._TRIM,"")}function ah(){return ag(this)}function F(){return this.textContent}function s(br){while(this.hasChildNodes()){this.removeChild(this.lastChild)}this.appendChild(U.createTextNode(br))}function am(){var bv,bs=this,bt=bs.parentNode,br=U.createRange(),bu=U.createElement("div");bt||U.body.appendChild(bs);br.selectNode(bs);bu.appendChild(br.cloneContents());bv=bu.innerHTML;bt||bs.parentNode.removeChild(bs);return bv}function ad(br){var bs=U.createRange();bs.setStartBefore(this);this.parentNode.replaceChild(bs.createContextualFragment(br),this)}a2._LIST.forEach(function(bs){uu[bs]=function bt(bv,bu){return a2(bv,bs,bu)};uu["un"+bs]=function br(bu){return a2(bu,bs,0,true)}});try{uu.ver.ie6&&U.execCommand("BackgroundImageCache",false,true)}catch(aP){}function g(){if(!aH.gone.blackout&&!aH.gone.dom++){V("boot");bl(aE.xboot||0)&&aE.xboot(uu)}}function a5(){aH.gone.win=1;g();bl(aE.xwin||0)&&aE.xwin(uu);V("canvas");V("audio");V("video")}function aM(){try{U.firstChild.doScroll("up"),g()}catch(br){setTimeout(aM,64)}}H(aE,"load",a5);uu.ie?aM():H(U,"DOMContentLoaded",g);function bn(){var bw,bv,bt,bs,br;for(bw in a7._db){try{bv=a7._db[bw];bt=bv.attributes,bs=-1;while((br=bt[++bs])){!br.name.indexOf("uu")&&(bv[br.name]=null)}}catch(bu){}}U.html=null;U.head=null;aE.detachEvent("onload",a5);aE.detachEvent("onunload",bn)}uu.ie&&aE.attachEvent("onunload",bn);aH(function(){var bs=U.html.getElementsByTagName("*"),br,bt=-1,bu=ar((uu.ie?"float,styleFloat,cssFloat,styleFloat":"float,cssFloat,styleFloat,cssFloat")+",pos,position,w,width,h,height,x,left,y,top,o,opacity,bg,background,bgcolor,backgroundColor,bgimg,backgroundImage,bgrpt,backgroundRepeat,bgpos,backgroundPosition");bb(M(t._db,uu.webkit?bg(U.html):U.html.style),bu,aK._HASH);a7(U.html);while((br=bs[++bt])){br.nodeType===1&&a7(br)}},2);function M(by,bv){function bu(bz,bA){return bA.toUpperCase()}function br(bz,bB,bA){return bB+"-"+bA.toLowerCase()}var bt,bs,bx=/-([a-z])/g,bw=/([a-z])([A-Z])/g;for(bt in bv){if(typeof bv[bt]==="string"){if(uu.webkit){bs=bt=bv.item(bt);bt.indexOf("-")>=0&&(bs=bt.replace(bx,bu));(bt!==bs)&&(by[bt]=bs)}else{bs=((uu.gecko&&!bt.indexOf("Moz"))?"-moz"+bt.slice(3):(uu.ie&&!bt.indexOf("ms"))?"-ms"+bt.slice(2):(uu.opera&&!bt.indexOf("O"))?"-o"+bt.slice(1):bt).replace(bw,br);(bt!==bs)&&(by[bs]=bt)}}}return by}function bj(br){return RegExp("(?:^| )("+br.join("|")+")(?:$|(?= ))","g")}function a8(bw,by,bx){var bv=0,bu,bt=-1,bz,bs=bw.split(""),br=bs.length;for(;bv<br;++bv){for(bu=0;bu<br;++bu){bz=bs[bv]+bs[bu];by[bz]=++bt;bx[bt]=bz}}}function aD(bv){function bt(bA){var bz=((/(?:rv\:|ari\/|sto\/)(\d+\.\d+(\.\d+)?)/.exec(bA)||[,0])[1]).toString();return parseFloat(bz.replace(/[^\d\.]/g,"").replace(/^(\d+\.\d+)(\.(\d+))?$/,"$1$3"))}function bw(bA){var bz=window.opera||false;return bz?+(bz.version().replace(/\d$/,"")):parseFloat((/(?:IE |fox\/|ome\/|ion\/)(\d+\.\d)/.exec(bA)||[,0])[1])}function br(bD){var bE=0,bC,bA,bz;try{bC=bD?new ActiveXObject("ShockwaveFlash.ShockwaveFlash"):navigator.plugins["Shockwave Flash"];bA=bD?bC.GetVariable("$version").replace(/,/g,"."):bC.description;bz=/\d+\.\d+/.exec(bA);bE=bz?parseFloat(bz[0]):0}catch(bB){}return bE<7?0:bE}function bs(bC){var bD=0,bB,bz=3;try{if(bC){bB=new ActiveXObject("AgControl.AgControl");while(bB.IsVersionSupported(bz+".0")){bD=bz++}}else{bB=navigator.plugins["Silverlight Plug-In"];bD=parseInt(/\d+\.\d+/.exec(bB.description)[0])}}catch(bA){}return bD<3?0:bD}var by={library:bv},bx=!!U.uniqueID,bu=navigator.userAgent;by.render=bt(bu);by.browser=bw(bu);by.flash=br(bx);by.silverlight=bs(bx);by.ie=bx;by.ie6=bx&&by.browser===6;by.ie7=bx&&by.browser===7;by.ie8=bx&&(U.documentMode||0)===8;by.ie9=bx&&(U.documentMode||0)===9;by.ie67=by.ie6||by.ie7;by.ie678=by.ie6||by.ie7||by.ie8;by.ie89=by.ie8||by.ie9;by.opera=!!(window.opera||false);by.gecko=bu.indexOf("Gecko/")>0;by.webkit=bu.indexOf("WebKit")>0;by.chrome=bu.indexOf("Chrome")>0;by.safari=!by.chrome&&bu.indexOf("Safari")>0;by.iphone=by.webkit&&/iPad|iPod|iPhone/.test(bu);by.quirks=(U.compatMode||"")!=="CSS1Compat";by.xml=a3("div").tagName===a3("DIV").tagName;by.win=bu.indexOf("Win")>0;by.mac=bu.indexOf("Mac")>0;by.unix=/X11|Linux/.test(bu);by.as3=by.flash>=9;by.advanced=(bx&&by.browser>=9)||(by.gecko&&by.render>1.9)||(by.webkit&&by.render>=528)||(by.opera&&by.browser>=10.5);by.major=(bx&&by.browser>=6)||(by.opera&&by.browser>=9.5)||(by.gecko&&by.render>=1.9)||(by.webkit&&by.render>524);by.jit=by.advanced;return by}})(window,document);uu.ajax||(function(k,n,c){c.mix(c,{ajax:c.mix(h,{get:e,post:p,sync:c.require,ifmod:o,queue:d,create:l,expire:f}),jsonp:c.mix(a,{queue:m})});function h(r,s,u,t){return q(r,s||{},u,t)}function q(w,E,y,z,K){function A(){if(B.readyState!==4){return}var L=B.status||0,M,v;if((L>=200&&L<300)||(!L&&!w.indexOf("file:"))){if(y&&!F++){v=t(true,L||200);y(v);K&&K(v)}if(E.ifmod){M=B.getResponseHeader("Last-Modified");if(M){q._cache[w]=c.date(Date.parse(M)).GMT()}}}else{u(L||((c.opera&&E.ifmod)?304:400))}r()}function t(L,v){return{id:E.id,ok:L,url:w,xhr:B,guid:G,status:v,isXMLContent:function(){return/xml/i.test(this.xhr.getResponseHeader("Content-Type")||"")}}}function u(v){z&&!F++&&z(t(false,v))}function r(){C&&c.event.detach(k,"beforeunload",C);B&&(B.onreadystatechange=c.nop,B=null)}function s(){H();u(408);r()}function H(){try{B&&B.abort()}catch(v){}}var B=E.xhr||l(),G=c.guid(),I=E.header||[],F=0,D,J=-1,C;w=i(w);if(E.nocache){w+=(w.indexOf("?")<0?"?":w.indexOf("&")<0?";":"&")+"uuguid="+ +(new Date)}if(B){try{c.gecko&&c.event.attach(k,"beforeunload",C=H);B.open(E.method||(E.data?"POST":"GET"),w,true);B.onreadystatechange=A;I.push("X-Requested-With","XMLHttpRequest");if(E.ifmod&&q._cache[w]){I.push("If-Modified-Since",q._cache[w])}if(E.data){I.push("Content-Type","application/x-www-form-urlencoded")}while((D=I[++J])){B.setRequestHeader(D,I[++J])}E.beforeCallback&&E.beforeCallback(B);B.send(E.data||null);setTimeout(s,(E.timeout||10)*1000);return G}catch(x){B=B||{responseText:"",responseXML:"",status:400}}}setTimeout(function(){u(400);r()},0);return G}q._cache={};function e(r,s,u,t){return q(r,c.arg(s,{data:null}),u,t)}function p(r,t,s,v,u){return q(r,c.arg(s,{data:t}),v,u)}function o(r,s,u,t){return q(r,c.arg(s,{ifmod:1}),u,t)}function d(u,s,t,w,r,v){j(1,u,s,t,w,r||c.nop,v||c.nop,[])}function j(B,v,r,x,C,y,z,t){function w(D){j(B,v,r,x,C,y,z,t.concat(D))}if(!v){y(t);return}var u,A=v.split(""),s=[];v="";while((u=A.shift())){s.push(r.shift(),c.mix(x.shift()||{},{id:u}),C.shift());if(A.shift()===">"){v=A.join("");break}}g(B,s,w,z)}function g(A,s,x,z){function C(E){var D=y.indexOf(E.guid);u[D]=E;++t>=r/3&&!v++&&x(u)}function B(D){!v++&&z(D)}var u=[],y=[],w=0,r=s.length,t=0,v=0;for(;w<r;w+=3){y.push((A?q:b)(s[w],s[w+1],s[w+2],B,C))}}function l(){return k.ActiveXObject?new ActiveXObject("Microsoft.XMLHTTP"):k.XMLHttpRequest?new XMLHttpRequest():null}function f(){q._cache={}}function a(r,s,u,t){return b(r,s||{},u,t)}a._jobdb={};function b(t,u,A,v,C){function B(){a._jobdb[x]("",408)}function w(){n.head.removeChild(y);k[r]=null;delete a._jobdb[x]}var x=c.guid(),z=u.timeout||10,s=t,y=c.node("script",{type:"text/javascript",charset:"utf-8",uujsonprun:0}),r=u.method||"callback";if(/\=\?$/.test(s)){s=c.format(s,"?",r)}a._jobdb[x]=r;k[r]=function(E,F){if(!y.uujsonprun++){if(E){var D={id:u.id,ok:true,url:s,json:E,guid:x,status:200};A&&A(D);C&&C(D)}else{v&&v({id:u.id,ok:false,url:s,json:void 0,guid:x,status:F||404})}setTimeout(w,(z+10)*1000)}};n.head.appendChild(y);y.setAttribute("src",s);setTimeout(B,z*1000);return x}function m(u,s,t,w,r,v){j(0,u,s,t,w,r||c.nop,v||c.nop,[])}function i(r){if(!i._ABSURL.test(r)){var s=c.node();s.innerHTML='<a href="'+r+'" />';r=s.firstChild?s.firstChild.href:/href\="([^"]+)"/.exec(s.innerHTML)[1]}return r}i._ABSURL=/^(?:file|https?):/})(window,document,uu);uu.flash||(function(e,d,b){b.mix(b,{flash:b.mix(g,{send:a,post:c,alert:f})});function g(j,m){m=b.arg(m,{id:"external"+b.guid(),width:"100%",height:"100%",marker:null,param:[]});if(m.param.indexOf("allowScriptAccess")<=0){m.param.push("allowScriptAccess","always")}b.ie&&m.param.push("movie",j);if(!m.marker){m.marker=b.node.add()}var h=[],l=0,n=m.param.length,k;for(;l<n;l+=2){h.push(b.format('<param name="?" value="?" />',m.param[l],m.param[l+1]))}k=b.format('<object id="?" data="?" width="?" height="?" ?>?</object>',m.id,j,m.width,m.height,b.ie?'classid="clsid:d27cdb6e-ae6d-11cf-96b8-444553540000"':'type="application/x-shockwave-flash"',h.join(""));b.node.swap(b.node.bulk(k),m.marker);return b.id(m.id)}function c(h,j,i){b.msg.post(h,j,i)}function a(h,j,i){return b.msg.send(h,j,i)}function f(i,h){if(h){b.config.debug&&b.puff(i)}else{b.puff(i)}}})(window,document,uu);uu.Class.Storage||(function(k,n,d){var l="LocalStorage,FlashStorage,IEStorage,CookieStorage,MemStorage";d.Class.singleton("Storage",{init:o,nth:a,get:e,set:m,size:p,pairs:b,clear:j,remove:g,getAll:h,saveToServer:f,loadFromServer:i,toString:c});d.Class.Storage.saveToServer=function(s,q,t,u){var r=d.json(s.getAll());d.ajax.post(q,r,t,function(v){u&&u(v)},function(v){u&&u(v)})};d.Class.Storage.loadFromServer=function(r,q,s,t){d.jsonp(q,s,function(w){if(w.ok){var v,u=w.json;for(v in u){r.set(v,u[v])}}t&&t(w)},function(u){t&&t(u)})};function o(){var r=this,q=d.config.storage;l.split(",").some(function(u){var s=d.Class[u];if(s&&s.isReady()){try{d(u,function(w){var v=w.size();if(q&&q>v.max){r.backend=d("MemStorage")}else{r.backend=w}d.ready.gone.storage=1;if(d.isFunction(k.xstorage)){setTimeout(function(){k.xstorage(d,r)},0)}})}catch(t){return false}return true}return false})}function a(q){return this.backend.nth(q)||""}function e(q){return this.backend.get(q)||""}function m(q,r){return this.backend.set(q,r)}function h(){return this.backend.getAll()}function p(){return this.backend.size()}function b(){return this.backend.pairs()}function j(){this.backend.clear()}function g(q){this.backend.remove(q)}function f(q,r,s){d.Class.Storage.saveToServer(this.backend,q,r,s)}function i(q,r,s){d.Class.Storage.loadFromServer(this.backend,q,r,s)}function c(){return this.backend.toString()}d.ready(function(){d.Class.Storage&&d("Storage")})})(window,document,uu);uu.Class.LocalStorage||(function(l,n,d){var g=d.gecko?5*1024*1024-260:d.ver.iphone?2.5*1024*1024-260:d.ver.safari?8*1024*1024:d.ver.chrome?2.5*1024*1024-260:d.webkit?2.5*1024*1024-260:d.opera?1.875*1024*1024-128:d.ie?5*1000*1000:0;d.Class.singleton("LocalStorage",{init:o,nth:a,get:e,set:m,size:p,pairs:b,clear:k,remove:h,getAll:i,saveToServer:f,loadFromServer:j,toString:c});d.Class.LocalStorage.isReady=function(){return !!l.localStorage};function o(q){this.storage=l.localStorage;q&&q(this)}function a(q){return this.storage.key(q)||""}function e(q){return this.storage[q]||""}function i(){var t={},r,q=0,s=this.storage.length;for(;q<s;++q){r=this.storage.key(q);t[r]=this.storage.getItem(r)}return t}function m(q,s){try{this.storage[q]="";this.storage[q]=s}catch(r){return false}return this.storage[q]===s}function p(){var r=0,q=0,t,s;if(d.ie&&"remainingSpace" in this.storage){s=this.storage.remainingSpace;if(g<s){g=5*1000*1000}return{used:g-s,max:g}}for(t=this.storage.length;q<t;++q){r+=this.storage.getItem(this.storage.key(q)).length}return{used:r,max:g}}function b(){return this.storage.length}function k(){this.storage.clear()}function h(q){this.storage.removeItem(q)}function f(q,r,s){d.Class.Storage.saveToServer(this,q,r,s)}function j(q,r,s){d.Class.Storage.loadFromServer(this,q,r,s)}function c(){return"LocalStorage"}})(window,document,uu);uu.Class.FlashStorage||(function(m,o,e){var l=e.config.baseDir+"uu.storage.swf",b="externalflashstorage";e.Class.singleton("FlashStorage",{init:p,nth:a,get:f,set:n,size:q,pairs:c,clear:k,remove:h,getAll:i,saveToServer:g,loadFromServer:j,toString:d});e.Class.FlashStorage.isReady=function(){return e.ver.as3&&e.require(l).ok};function p(t){var s=this;function r(){setTimeout(function(){e.dmz[b]=null;t&&t(s)},0)}e.dmz[b]=r;this.storage=e.flash(l,{id:b,width:1,height:1,param:[]})}function a(r){return this.storage.ex_nth(r)}function f(r){return this.storage.ex_get(r)}function i(){return this.storage.ex_getAll()}function n(r,s){return this.storage.ex_set(r,s)}function q(){return this.storage.ex_size()}function c(){return this.storage.ex_pairs()}function k(){this.storage.ex_clear()}function h(r){this.storage.ex_remove(r)}function g(r,s,t){e.Class.Storage.saveToServer(this,r,s,t)}function j(r,s,t){e.Class.Storage.loadFromServer(this,r,s,t)}function d(){return"FlashStorage"}})(window,document,uu);uu.Class.IEStorage||(function(d,r,f){var a="uustorage",g="uuindex",o=(new Date(2032,1,1)).toUTCString(),p=63*1024;f.Class.singleton("IEStorage",{init:m,nth:b,get:n,set:h,size:i,pairs:s,clear:k,remove:q,getAll:j,saveToServer:c,loadFromServer:e,toString:l});f.Class.IEStorage.isReady=function(){return f.ie};function m(t){this.storage=f.node.add("script",r.head);this.storage.addBehavior("#default#userData");this.storage.expires=o;t&&t(this)}function b(t){this.storage.load(a);return(this.storage.getAttribute(g)||"").split("\t")[t]||""}function n(t){this.storage.load(a);return this.storage.getAttribute(t)||""}function j(){this.storage.load(a);var w={},v,u=-1,t=(this.storage.getAttribute(g)||"").split("\t");while((v=t[++u])){w[v]=this.storage.getAttribute(v)||""}return w}function h(u,w){this.storage.load(a);var t=this.storage.getAttribute(g);try{if(!t){this.storage.setAttribute(g,u)}else{if(("\t"+t+"\t").indexOf("\t"+u+"\t")<0){this.storage.setAttribute(g,t+"\t"+u)}}this.storage.setAttribute(u,w);this.storage.save(a)}catch(v){return false}return this.storage.getAttribute(u)===w}function i(){this.storage.load(a);var u=(this.storage.getAttribute(g)||""),x=u.length,t=u.split("\t"),w,v=-1;while((w=t[++v])){x+=(this.storage.getAttribute(w)||"").length}return{used:x,max:p}}function s(){this.storage.load(a);return(this.storage.getAttribute(g)||"").split("\t").length}function k(){this.storage.load(a);var t=(this.storage.getAttribute(g)||"").split("\t"),v,u=-1;while((v=t[++u])){this.storage.removeAttribute(v)}this.storage.setAttribute(g,"");this.storage.save(a)}function q(v){this.storage.load(a);var u=(this.storage.getAttribute(g)||""),t="\t"+u+"\t",w="\t"+v+"\t";if(t.indexOf(w)>=0){u=f.trim(t.replace(new RegExp(w),""));this.storage.setAttribute(g,u);this.storage.removeAttribute(v);this.storage.save(a)}}function c(t,u,v){f.Class.Storage.saveToServer(this,t,u,v)}function e(t,u,v){f.Class.Storage.loadFromServer(this,t,u,v)}function l(){return"IEStorage"}})(window,document,uu);uu.Class.CookieStorage||(function(d,u,g){var a="uustorage",r=(new Date(2032,1,1)).toUTCString(),m=(new Date(0)).toUTCString(),s=3800,h=location.protocol==="https:"?"; secure":"";g.Class.singleton("CookieStorage",{init:p,nth:b,get:q,set:i,size:k,pairs:v,clear:n,remove:t,getAll:l,saveToServer:c,loadFromServer:f,toString:o,store:e,retrieve:j});g.Class.CookieStorage.isReady=function(){return !!navigator.cookieEnabled};function p(w){this._shadowCookie=this.retrieve();w&&w(this)}function b(w){return g.hash.nth(this._shadowCookie,w)[0]||""}function q(w){return this._shadowCookie[w]||""}function l(){return this._shadowCookie}function i(w,y){var x=u.cookie.length;if(x>s){return false}if(x){x+=2}x+=this.store(g.hash(w,y),r);if(x!==u.cookie.length){return false}this._shadowCookie[w]=y;return true}function k(){return{used:u.cookie.length,max:s}}function v(){return g.hash.size(this._shadowCookie)}function n(){this.store(this._shadowCookie,m);this._shadowCookie={}}function t(w){this.store(g.hash(w,""),m);delete this._shadowCookie[w]}function j(){var B={},w=-1,z,A,y,x=a.length;if(u.cookie){z=u.cookie.split("; ");while((A=z[++w])){y=A.split("=");if(!y[0].indexOf(a)){B[y[0].slice(x)]=decodeURIComponent(y[1]||"")}}}return B}function e(y,w){var z="",x;for(x in y){z=a+x+"="+encodeURIComponent(y[x]);u.cookie=z+"; expires="+w+h}return z.length}function c(w,x,y){g.Class.Storage.saveToServer(this,w,x,y)}function f(w,x,y){g.Class.Storage.loadFromServer(this,w,x,y)}function o(){return"CookieStorage"}})(window,document,uu);uu.Class.MemStorage||(function(k,m,d){d.Class.singleton("MemStorage",{init:n,nth:a,get:e,set:l,size:o,pairs:b,clear:j,remove:g,getAll:h,saveToServer:f,loadFromServer:i,toString:c});d.Class.MemStorage.isReady=function(){return true};function n(p){this.storage={};p&&p(this)}function a(p){return d.hash.nth(this.storage,p)[0]||""}function e(p){return this.storage[p]||""}function l(p,q){this.storage[p]=q;return true}function h(){return this.storage}function o(){return{used:0,max:Number.MAX_VALUE}}function b(){return d.hash.size(this.storage)}function j(){this.storage={}}function g(p){delete this.storage[p]}function f(p,q,r){d.Class.Storage.saveToServer(this,p,q,r)}function i(p,q,r){d.Class.Storage.loadFromServer(this,p,q,r)}function c(){return"MemStorage"}})(window,document,uu);uu.lazy.fire("init");//
//	base.js
//
//		1. This source code is MIT licence.
//		2. Copyright(C) All rights 2012 blueskis
//		3. If you find bugs that I have to fix , Please send mail blueskis382 @ gmail.com
//
//		1. このソースコードはMIT licenceとして配布します。
//		3. もしバグを見つけたら、 blueskis382 @ gmail.com にメールを送っていただけたらうれしいです。
//

var Invariant = new Array() ;	//定数
Invariant["APP_NAME"] = "BIEditor" ;
Invariant["APP_VERSION"] = "0.31" ;
Invariant["EditorHeightMines"] = 20 ;

Invariant["UnSupportedEncodeErrorMessage"] = "このWebブラウザでサポートされていないエンコードの指定です" ;
Invariant["UnSupportedFileReadingMessage"] = "このWebブラウザではファイルの読み込みができません" 
Invariant["UnSupportedCopyToClipBoardMessage"] = "このWebブラウザではコピーが出来ません。" ;
Invariant["ErrorMessage"] = "エラー :\n"
Invariant["UserScriptErrorWhenStart"] = "起動時スクリプトエラー :\n"
//タイトル関連
Invariant["File_Open"] = "ファイルを開く" ;
Invariant["Search"] = "検索" ;
Invariant["Replace"] = "置換" ;
Invariant["SettingEnvironment"] = "環境設定" ;
Invariant["Close_Message"] = "保存していないデータは失われますが、よろしいですか?" ;

var util = {	//ユーティリティ関数
	SupportedChecker : function(){	//サポートされているかのチェック
		var str = "" ;
		if(!window.File) str += "・ファイルを開く/保存する機能、画像を表示する機能\n" ;
		if(!str == "") alert("このWebブラウザでは以下の機能が利用できません。\n" + str) ;
	} ,
	TransmissionMIMETypeForCodeMirror : function(filePath){ 	//拡張子からCodeMirror用のMIME Typeに設定
		var querys = [
			[".bash" , "text/x-sh"] ,
			[".bat" , "text/plain"] ,
			[".c" , "text/x-csrc"] ,
			[".cfg" , "text/plain"] ,
			[".clj" , "text/x-clojure"] ,
			[".coffee" , "text/x-coffeescript"] ,
			[".conf" , "text/plain"] ,
			[".cpp" , "text/x-c++src"] ,
			[".cshape" , "text/x-csharp"] ,
			[".css" , "text/css"] ,
			[".csv" , "text/plain"] ,
			[".diff" , "text/x-diff"] ,
			[".docx" , "text/xml"] ,
			[".ks" , "text/plain"] , //KAG3(ノベルゲームスクリプトエンジン)
			[".go" , "text/x-go"] ,
			[".h" , "text/x-csrc"] ,
			[".hta" , "text/html"] ,
			[".htaccess" , "text/plain"] ,
			[".hs" , "text/x-haskell"] ,
			[".htm" , "text/html"] ,
			[".html" , "text/html"] ,
			[".hs" , "text/x-haskell"] ,
			[".hx" , "text/x-haxe"] ,
			[".ini" , "text/x-ini"] ,
			[".java" ,  "text/x-java"] ,
			[".js" , "text/javascript"] ,
			[".jse" , "text/javascript"] ,
			[".json" , "application/json"] ,
			[".l" , "text/x-ecl"] ,
			[".latex" , "text/x-stex"] ,
			[".less" , "text/x-less"] ,
			[".lisp" , "text/x-ecl"] ,
			[".log" , "text/plain"] ,
			[".lua" , "text/x-lua"] ,
			[".mac" , "text/plain"] , //秀丸エディタマクロ
			[".mark" , "text/markdown"] ,
			[".markdown" , "text/markdown"] ,
			[".md" , "text/x-markdown"] ,
			[".mdown" , "text/markdown"] ,
			[".mkd" , "text/markdown"] ,
			[".mkdn" , "text/markdown"] ,
			[".mo" , "text/plain"] ,
			[".ps1" , "text/plain"] ,
			[".ps1xml" , "application/xml"] ,
			[".psc1" , "text/plain"] ,
			[".psd1" , "text/plain"] ,
			[".psm1" , "text/plain"] ,
			[".py" , "text/x-python"] ,
			[".php" , "application/x-httpd-php"] ,
			[".pl" , "text/x-perl"] ,
			[".r" , "text/x-rsrc"] ,
			[".rb" , "text/x-ruby"] ,
			[".rs" , "text/x-rustsrc"] ,
			[".rss" , "application/xml"] ,
			[".s" , "text/x-rsrc"] ,
			[".scm" , "text/x-scheme"] ,
			[".scp" , "text/plain"] ,
			[".sct" , "text/plain"] ,
			[".sh" , "text/x-sh"] ,
			[".shtml" , "text/html"] ,
			[".sql" , "text/x-mysql"] ,
			[".sql" , "text/x-mysql"] ,
			[".svg" , "application/xml"] ,
			[".svn-base" , "text/plain"] ,
			[".sys" , "text/plain"] ,
			[".st" , "text/x-stsrc"] ,
			[".srx" , "application/x-sparql-query"] ,
			[".tex" , "text/x-stex"] ,
			[".txt" , "text/plain"] ,
			[".tpl" , "text/x-smarty"] ,
			[".v" , "text/x-verilog"] ,
			[".vb" , "text/vb"] ,
			[".vbe" , "text/vbscript"] ,
			[".vbs" , "text/vbscript"] ,
			[".vm" , "text/velocity"] ,
			[".wsc" , "text/plain"] ,
			[".wsf" , "text/plain"] ,
			[".wtx" , "text/plain"] ,
			[".xaml" , "application/xml"] ,
			[".xbap" , "text/plain"] ,
			[".xml" , "application/xml"] ,
			[".xquery" , "application/xquery"] ,
			[".xul" , "application/xml"]
		] ;
		for(var i=0;i<querys.length;i++){
			if(filePath.substring(
				filePath.length - querys[i][0].length  ,
				filePath.length 
			) == querys[i][0])
				return querys[i][1] ;
		}
		return "text/plain" ;
	} ,
	getFileFullPath : function(element){	//ファイルのフルパスを取得(※Chromeでは<input type="file">のvalueがC:\fakepath\みたいになるから
		if(window.File){
			return element.files[0].name
		}else{
			return element.value ; //For IE
		}
	} ,
	File : function(elem , encode){	//クロスブラウザ対応ファイルオブジェクト

		//IEのFileSystemObjectって、UTF-16かShift-JISしか対応していない
		if(window.ActiveXObject && encode != "utf-16" && encode != "shift-jis"){
			alert(Invariant["UnSupportedEncodeErrorMessage"]) ;
		}
		
		this.elem = elem ;
		this.encode = encode ;	
		this.getFileName = function(){	//ファイル名を取得
			var fileURL ;
			if(BrowsingMode.SupportedFileAPI){
				fileURL = this.elem.files[0].name ;
			}else{
				fileURL = this.elem.value ;
			}
			return url.match(/.*\/(.*)$/)[1] ;			
		}
		
		this.read = function(loadedFunc){	//ファイル読み込み	(loadedFunc : ファイル読み込み後に実行する関数)
			if(window.File){	//File API (Firefox , Google Chrome , IE10)
				var reader = new FileReader() ;
				reader.onload = function(event){
					loadedFunc(event.target.result) ;
				} ;
				reader.readAsText(this.elem.files[0] , this.encode) ;

			}else if(window.ActiveXObject){	//IE系
				try{
					var isUnicode = true ;
					if(this.encode == "shift-jis") isUnicode = false ;
					
					var stream = new ActiveXObject("FileSystemObject").OpenTextFile(this.elem.value , 2 , isUnicode) ;
					loadedFunc(stream.ReadAll()) ;
					
				}catch(e){
					alert(Invariant["ErrorMessage"] + e) ;
				}
			}else{
				alert(Invariant["UnSupportedFileReadingMessage"]) ;
			}
		}
		this.write = function(str){
			if(BrowsingMode.SupportedBlobBuilder){	//BlobBuider対応
				var blobBuilder;
				var blobURL ;
				if ("MozBlobBuilder" in window) {
					blobBuilder = new MozBlobBuilder();
					blobURL = URL ;
				} else if ("WebKitBlobBuilder" in window) {
					blobBuilder = new WebKitBlobBuilder();
					blobURL = webkitURL ;
				} else if ("BlobBuider" in window){
					blobBuilder = new window.BlobBuider() ;
					blobURL = URL ;
				}

				blobBuilder.append(str);
				var url = blobURL.createObjectURL(blobBuilder.getBlob()) ;
				
				var downloader = $("<a></a>") ;
				downloader[0].download = this.getFileName() ;
				downloader.attr("href" , url) ;
				
				downloader.click() ;
			}else if(BrowserMode.SuppotedActiveX){
				//ActiveX...
			}
		}
	} ,
	CopyToClipboard : function(text){
		var input, success;
		if ("console" in window && "notifyFirebug" in console) {
			console.notifyFirebug([text], "copy", "firebugExecuteCommand");
			success = true;
		} else {
			input = document.createElement("input");
			input.style.display = "none" ;
			input.value = text;
			document.body.appendChild(input);
			input.select();
			try {
				success = document.execCommand("copy", false, null);
			} catch (ex) {
				success = false;
			} finally {
				document.body.removeChild(input);
			}
		}
		if (!success) {
			alert(Invariant["UnSupportedCopyToClipBoardMessage"]) ;
		}
	} ,
	getFonts : function(){
		//対応していないブラウザ用(※プログラミングに適しているといわれているフォントたち)
		var default_fonts = ["Consolas" , "Courier" , "Courier New" , 
			"Andale Mono" ,"Monaco" , "Profont" ,"Monofur" ,
			"Proggy" , "Droid Sans Mono" , "Deja Vu Sans Mono" ,
			"Inconsolata" , "Ricty" , "Terminal" , "FixedSys" ,
			"M+2VM+IPAG circle" ,"VL ゴシック" , "VL Pゴシック" ,
			"KonatuTohaba" , "Osaka" , "ゆたぽん（コーディング)" ,
			"Mig 1M" , "M+ BITMAP FONTS" , "Bitstream Vera Sans Mono" ,
			"Anonymous" ,"Arial Monospaced" , "Aurulent Sans Mono" ,
			"BPMono" , "Century Schoolbook Mono BT" , "Crystal" ,
			"Dina" , "DPCustomMono2" , "Envy Code R" , "Lucida Console" ,
			"Lucida Sans Typewriter" , "Monospace 821 BT" ,
			"Nu Sans Mono" , "Onuava" , "Pragmata" , "ProFontWindows" ,
			"Proggy Clean with Slashed Zero" , "saxMono" , "Selectric" ,
			"Terminus" , "Ti92Pluspc" , "Verily Serif Mono" ,
			"Osaka 等幅" , "メイリオ" , "Arial" , "Dina" , "Lucida Sans" ,
			"MS Sans Serif" , "Menlo" ,
			"ＭＳ Ｐゴシック" , "ＭＳ　ゴシック"].sort() ;
		if(window.ActiveXObject){
			$("body").prepend('<OBJECT id="dlgHelper" CLASSID="clsid:3050f819-98b5-11cf-bb82-00aa00bdce0b" width="0px" height="0px"></OBJECT>') ;
			var fonts = [] ;
			try {
				for (var i = 1; i < dlgHelper.fonts.count; i++) {
					fonts.push(dlgHelper.fonts(i)) ;
				}
				fonts.sort() ;
				$("#dlgHelper").remove() ;
				return fonts ;
			}catch(e) { 
				return default_fonts ;
			} 
		}else
			return default_fonts ;
	} ,
	setSettingItem : function(){
		//テーマの設定
		Sys.storage.set("TextEditor.ThemeSelector" , $("#theme-selector option:selected").val()) ;
		//キーマップの設定
		Sys.storage.set("TextEditor.keyMap" , $("#keyMap option:selected").val()) ;
		//行番号の表示の有無
		Sys.storage.set("TextEditor.LineNumberShow" , $("#lineNumbers").is(":checked")) ;
		//行の表示の有無
		Sys.storage.set("TextEditor.LineVisible" , $("#line_symbol").is(":checked")) ;
		//タブの表示の有無
		Sys.storage.set("TextEditor.TabVisible" , $("#tab").is(":checked")) ;
		//下線の表示の有無
		Sys.storage.set("TextEditor.CurrentLineUnderLine" , $("#underline").is(":checked")) ;
		//EOF記号の表示の有無
		Sys.storage.set("TextEditor.EOFShow" , $("#eof_symbol").is(":checked")) ;
		//背景画像の設定
		Sys.storage.set("TextEditor.BackGroundImageSource" , $("#background_image_src").val()) ;
		//背景画像透過率の設定
		Sys.storage.set("TextEditor.BackGroundImageOpacity" , $("#background_opacity").val()) ;
		//タブサイズの設定
		Sys.storage.set("TextEditor.TabSize" , $("#TabSize").val()) ;
		//起動時スクリプト
		Sys.storage.set("TextEditor.BeginScript" , Sys.beginEditor.getValue()) ;
		//フォントファミリ
		Sys.storage.set("TextEditor.FontFamily" , $("#fonts").val()) ;
		//フォントサイズ
		Sys.storage.set("TextEditor.FontSize" , $("#fontSize").val()) ;
		//画像の幅指定
		Sys.storage.set("TextEditor.BackGroundImageStyle" , $("#background-style option:selected").val()) ;
		//エディタの背景色の指定
		Sys.storage.set("TextEditor.ThemeBackGroundApply" , $("#editor_background").is(":checked")) ;
		//初期化
		var value = Sys.commonEditor.getValue() ;
		var currentmode = Sys.commonEditor.getOption("mode") ;
		$(".CodeMirror").remove() ;
		Sys.init() ;
		Sys.commonEditor.setValue(value) ;
		Sys.commonEditor.setOption("mode" , currentmode) ;
		
		
		Sys.onSettingEnd() ;
	} ,
	setBackGroundImageSize : function(mode , width , height){
		$(".CodeMirror-scroll").css("height" , height)
		var left = $(".CodeMirror-gutter").width() ;
		if(Sys.storage.get("TextEditor.LineNumberShow") == "false") left = 0 ;
		switch(mode){
				case "fit_width_height" :
					$("#background_image")
						.css("text-align" , "left")
						.css("width" , width) 
						.css("height" , height)
						.css("left" , left) ;
					break ;
				case "fit_width" :
					$("#background_image")
						.css("text-align" , "left")
						.css("width" , width)
						.css("height" , "auto")
						.css("left" , left) ;
						break ;
				case "fit_height" :
					$("#background_image")
						.css("text-align" , "left")
						.css("width" , "auto")
						.css("height" , height) 
						.css("left" , left) ;
						break ;
				case "right_top" :
					$("#background_image")
						.css("text-align" , "left")
						.css("width" , "auto")
						.css("height" , "auto") 
						.css("left" , left) ;
						break ;
				case "right_center" : 
					$("#background_image")
						.css("text-align" , "center")
						.css("width" , "auto")
						.css("height" , "auto")
						.css("left" , left) ;
						break ;
			}	
	}
}
var Exception = {	//例外用オブジェクト
	File : {	//ファイル系エラー
		UnavaiableFileEncodingException : function(){	//ファイルのエンコードが利用不可
			this.message = Invariant["UnSupportedEncodeErrorMessage"] ;
			this.className = "UnavaiableFileEncodingException" ;
		} ,
		UnSupportedFileReading : function(){	//ファイル読み込みがサポートされていない
			this.message = Invariant["UnSupportedFileReadingMessage"] ;
			this.className = "UnSupportedFileReading" ;
		}
	}
}
var ui = {
	Window : function(str){	//JavaのJFrameっぽいかんじで作ってみた
		this.mainElement = $("<div id=\"temp_" + new Date().getTime() + "\"><div class=\"wrapper\"></div></div>") ;
		this.mainElement.dialog({
			title : str ,
			autoOpen : false ,
			draggable : true ,
			resizable : true ,
			modal : false
		}) ;
		this.remove = function(){
			this.mainElement.dialog("destroy") ;
		}
		this.setWidth = function(width){
			this.mainElement.dialog("option" , "width" , width) ;
		}
		this.setContent = function(s){
			this.mainElement.children(".wrapper").html(s)
		}
		this.setVisible = function(isvisible){
			if(isvisible) this.mainElement.dialog("open")
			else this.mainElement.dialog("close") ;
		}
		this.setHeight = function(height){
			this.mainElement.dialog("option" , "height" , height) ;
		}
		this.setButton = function(obj){
			this.mainElement.dialog("option" , "buttons" , obj) ;
		}
		this.setModal = function(ismodal){
			if(ismodal) this.mainElement.dialog("option" , "modal" , ismodal)
			else this.mainElement.dialog("option" , "modal" , ismodal) ;
		}
		this.setTitle = function(str){
			this.mainElement.dialog("option" , "title" , str) ;
		}
		this.setResizable = function(isresizable){
			this.mainElement.dialog("option" , isresizable) ;
		}
		this.setLocation = function(x , y){
			this.mainElement.dialog("position" , [x , y]) ;
		}
		this.setPosition = function(x , y){
			this.mainElement.dialog("position" , [x , y]) ;
		}
		this.addEventListener = function(str , func){
			this.mainElement.bind(str , func) ;
		}
		this.getTitle = function(str){
			return this.mainElement.dialog("option" , "title") ;
		}
		this.getWidth = function(){
			return this.mainElement.dialog("option" , "width") ;
		}
		this.getHeight = function(){
			return this.mainElement.dialog("option" , "height") ;
		}	
		this.isModal = function(){
			return this.mainElement.dialog("option" , "modal") ;
		}
		this.isResizable = function(){
			return this.mainElement.dialog("option" , "resizable") ;
		}
		this.getWindowElement = function(){
			return this.mainElement ;
		}
	}
}
var BrowsingMode = {
	InternetExplorer7 : navigator.userAgent.match(/MSIE 7/) ? true : false ,
	InternetExplorer8 : navigator.userAgent.match(/MSIE 8/) ? true : false ,
	InternetExplorer9 : navigator.userAgent.match(/MSIE 9/) ? true : false ,
	InternetExplorer10 : navigator.userAgent.match(/MSIE 10/) ? true : false ,

	Firefox : navigator.userAgent.match(/Firefox/) ? true : false ,
	SeaMonkey : navigator.userAgent.match(/SeaMonkey/) ? true : false ,

	Opera : window.opera ? true : false ,
	OperaExtension : window.opera && window.opera.extension ? true : false ,

	Chrome : window.chrome ? true : false ,
	ChromeExtension : window.chrome && window.chrome.extension ? true : false ,

	Safari : window.safari ? true : false ,
	
	SupportedBlobBuilder : window.MozBlobBuilder || window.WebKitBlobBuilder || window.BlobBuilder ? true : false ,
	SupportedFileAPI : window.opera || navigator.userAgent.match(/[Firefox|SeaMonkey]/) || window.chrome ? true : false ,
	SuppotedActiveX : navigator.userAgent.match(/MSIE [7-9]/) ? true : false,
	
	UnSupportedBrowser : navigator.userAgent.match(/MSIE 6/) ? true : false
}
var Sys = {									//テキストエディタのシステムに関わる関数
	initSettingItem : function(){			//設定項目の初期化
		if(Sys.storage.get("TextEditor.ThemeSelector")){
			$("#theme-selector").val(Sys.storage.get("TextEditor.ThemeSelector")) ;
			$("#keyMap").val(Sys.storage.get("TextEditor.keyMap")) ;
			$("#background_image_src").val(Sys.storage.get("TextEditor.BackGroundImageSource")) ;
			$("#background_opacity").val(Sys.storage.get("TextEditor.BackGroundImageOpacity")) ;
			$("#TabSize").val(Sys.storage.get("TextEditor.TabSize")) ;
			$("#fonts").val( Sys.storage.get("TextEditor.FontFamily")) 	;
			$("#fontSize").val( Sys.storage.get("TextEditor.FontSize")) ;
			$("#background-style").val( Sys.storage.get("TextEditor.BackGroundImageStyle")) ;
			
			if(Sys.storage.get("TextEditor.LineNumberShow") === "false") $("#lineNumbers")[0].checked = false ;
			if(Sys.storage.get("TextEditor.LineVisible") === "false") $("#line_symbol")[0].checked = false ;
			if(Sys.storage.get("TextEditor.TabShow") === "false") $("#tab")[0].checked = false ;
			if(Sys.storage.get("TextEditor.CurrentLineUnderLine") === "false") $("#underline")[0].checked = false ;
			if(Sys.storage.get("TextEditor.EOFShow") == "false") $("#eof_symbol")[0].checked = false ;
			
			$("#background_image").attr("src" , Sys.storage.get("TextEditor.BackGroundImageSource")) 
								.css("opacity" , Sys.storage.get("TextEditor.BackGroundImageOpacity")) ;
		}
		Sys.outputStorageData() ;
	} ,
	outputStorageData : function(){			//設定項目のストレージデータの読み込み
		var str = "" ;
		var import_text = "{" ;
		var cnt = 0 ;
		for(var s in Sys.storage.getAll()){
			if(cnt != 0) import_text += "," ;
			if(s.indexOf("TextEditor") != -1){
				str +=  "<tr><td>" + s + "</td><td><input type=\"text\" value=\"" + Sys.storage.get(s) + "\"></td></tr>" ;
				import_text += "\n\t\"" + s + "\" : \"" + Sys.storage.get(s).replace(/\t/g).replace(/\\t/g).replace(/\n/g , "\\n").replace(/\"/g , "\\\"") + "\"" 
			}
			cnt++ ;
		}
		$("#storage_result").html(str) ;
		$("#import_text").val(import_text + "\n}") ;
	} ,
	initCommonEditor : function(){			//メインのエディタの初期化
		var hlLine = null　, hlLine2 = null ;
		Sys.commonEditor = CodeMirror.fromTextArea(document.getElementById("editor"), {
			lineNumbers: (Sys.storage.get("TextEditor.LineNumberShow") == "false" ? false : true) ,
			tabSize: (Sys.storage.get("TextEditor.TabSize") ? Sys.storage.get("TextEditor.TabSize") : 4) ,
			indentUnit: 4,
			indentWithTabs: true ,
			theme : (Sys.storage.get("TextEditor.ThemeSelector") ? Sys.storage.get("TextEditor.ThemeSelector") : "eclipse") ,
			onCursorActivity: function() {
				Sys.commonEditor.setLineClass(hlLine, null, null);
				hlLine = Sys.commonEditor.setLineClass(Sys.commonEditor.getCursor().line, null, "activeline");
			} ,
			onGutterClick: function(cm, n) {
				var info = cm.lineInfo(n);
				if (info.markerText)
					cm.clearMarker(n);
				else
					cm.setMarker(n, "<span style=\"color: #900\">●</span> %N%");
			} ,
			mode: "text/plain" ,
			profile : "html" ,
			keyMap : (Sys.storage.get("TextEditor.KeyMap") ? Sys.storage.get("TextEditor.KeyMap") : "default")
		});
		Sys.commonEditor.setValue("") ;


		hlLine = Sys.commonEditor.setLineClass(0, "activeline");
		
		Sys.beginEditor = CodeMirror.fromTextArea(document.getElementById("stating-js-code"), {
			lineNumbers: (Sys.storage.get("TextEditor.LineNumberShow") == "false" ? false : true) ,
			tabSize: (Sys.storage.get("TextEditor.TabSize") ? Sys.storage.get("TextEditor.TabSize") : 4) ,
			indentUnit: 4,
			indentWithTabs: true ,
			theme : (Sys.storage.get("TextEditor.ThemeSelector") ? Sys.storage.get("TextEditor.ThemeSelector") : "eclipse") ,
			onCursorActivity: function() {
				Sys.commonEditor.setLineClass(hlLine2, null, null);
				hlLine2 = Sys.commonEditor.setLineClass(Sys.commonEditor.getCursor().line, null, "activeline");
			} ,
			onGutterClick: function(cm, n) {
				var info = cm.lineInfo(n);
				if (info.markerText)
					cm.clearMarker(n);
				else
					cm.setMarker(n, "<span style=\"color: #900\">●</span> %N%");
			} ,
			mode: "text/javascript" ,
			keyMap : (Sys.storage.get("TextEditor.KeyMap") ? Sys.storage.get("TextEditor.KeyMap") : "default")
		});
		hlLine2 = Sys.beginEditor.setLineClass(0 , "activeline") ;
		$(".CodeMirror")
							.css("font-family" , Sys.storage.get("TextEditor.FontFamily") ? Sys.storage.get("TextEditor.FontFamily") : "Consolas") 
							.css("font-size" , Sys.storage.get("TextEditor.FontSize") ? Sys.storage.get("TextEditor.FontSize") + "px" : "15px") 
							.addClass("TabVisible")
							.addClass("LineVisible")
							.addClass("CurrentLineUnderLine") 
							.addClass("EOFShow")
		
		if(Sys.storage.get("TextEditor.ThemeBackGroundApply") == "true") $(".CodeMirror-scroll").css("background-color" , "transparent") ;
		if(Sys.storage.get("TextEditor.TabShow") == "false") $(".CodeMirror").removeClass("TabVisible") ;
		if(Sys.storage.get("TextEditor.LineVisible") == "false") $(".CodeMirror").removeClass("LineVisible") ;
		if(Sys.storage.get("TextEditor.CurrentLineUnderLine") == "false") $(".CodeMirror").removeClass("CurrentLineUnderLine") ;
		if(Sys.storage.get("TextEditor.EOFShow") == "false") $(".CodeMirror").removeClass("EOFShow") ;
	} ,
	initClickedWork : function(){			//クリックされた時の動作の初期化
			$('[data-create="Open"]').click(function(){
				Sys.commonEditor.setValue("") ;
				Sys.currentFile = null ;
			}) ;
			$('[data-action="Open"]').click(function(){
				$("#Open").dialog("open") ;
			}) ;
			$('[data-action="ReOpen"]').click(function(){
				var file = new File(currentFile) ;
				file.read(function(s){
					Sys.commonEditor.setValue(s) ;
				}) ;
			}) ;
			$('[data-action="About"]').click(function(){
				$("#About").dialog("open") ;
			}) ;
			$('[data-action="Save"]').click(function(){
				Sys.onFileSave() ;
				var file = new util.File() ;
				file.write(Sys.commonEditor.getValue()) ;

			}) ;
			$('[data-action="SaveAs"]').click(function(){
			}) ;
			$('[data-action="Print"]').click(function(){
				window.scroll(0 , 0) ;
				window.print() ;
				
			}) ;
			$('[data-action="Exit"]').click(function(){
				window.close() ;
			}) ;
			$('[data-action="Undo"]').click(Sys.commonEditor.undo)
			$('[data-action="Redo"]').click(Sys.commonEditor.redo)
			$('[data-action="Copy"]').click(function(){
				util.CopyToClipboard(editor.getSelection) ;
			}) ;
			$('[data-action="Paste"]').click(function(){
				document.execCommand("paste") ;
			}) ;
			$('[data-action="Delete"]').click(function(){
				document.execCommand("delete") ;
			}) ;
			$('[data-action="SelectAll"]').click(function(){
				document.execCommand("selectAll") ;
			}) ;
			$('[data-action="SettingEnvironment"]').click(function(){
				$("#SettingEnvironment").dialog("open") ;
			}) ;
			$('[data-action="Search"]').click(function(){
				$("#searchbox").dialog("open") ;
			}) ;
			$('[data-action="Replace"]').click(function(){
				$("#replacebox").dialog("open") ;
			}) ;
			$('[data-action="reload_Sys.storage_data"]').click(function(){
				Sys.outputStorageData() ;
			}) ;
			$('[data-action="NewDocument"]').click(function(){
				document.title = Invariant["APP_NAME"] ;
				Sys.commonEditor.setValue("") ;
			}) ;
			$('[data-action="New_Tab"]').click(function(){
				window.open(location.href) ;
			}) ;
			$('[data-action="SourceForge"]').click(function(){
				window.open("http://sourceforge.jp/projects/bieditor/simple/") ;
			}) ;
			$('[data-action="Export"]').click(function(){
				if(window.JSON){
					try{
						var obj = JSON.parse($("#export_text").val()) ;
					}catch(e){
						alert(Invariant["ErrorMessage"] + e) ;
					}
				}else{
					try{
						var obj = eval($("#export_text").val()) ;
					}catch(e){
						alert(Invariant["ErrorMessage"] + e) ;
					}
				}
				for(var n in obj){
					eval("Sys.storage.set(\"" + n + "\" , \"" + obj[n].replace(/\t/g).replace(/\\t/g).replace(/\n/g , "\\n").replace(/\"/g , "\\\"")  + "\")") ;
				}
				Sys.init() ;
				$("#SettingEnvironment").dialog("close") ;
				
			}) ;
		} ,
	initDialog : function(){					//ダイアログボックスの初期化
		$("#Open").dialog({
			modal : true ,
			draggable : true ,
			title : Invariant["File_Open"] ,
			height : 240 ,
			width : 320 ,
			autoOpen : false ,
			buttons : {
				"OK" : function(event){
					Sys.onFileOpen() ;
					var file = new util.File( document.getElementById("Open-file-chooser") , $("#Open-file-encode option:selected").val()) ;
					try{
						file.read(function(str){
							var mime = util.TransmissionMIMETypeForCodeMirror($("#Open-file-chooser")[0].value) ;
							Sys.commonEditor.setOption("mode" , mime) ;
							Sys.commonEditor.setValue(str) ;
							document.title = Invariant["APP_NAME"] + " -" + $("#Open-file-chooser")[0].value ;
							Sys.setWindowTitle(util.getFileFullPath($("#Open-file-chooser")[0]))
						}) ;
					}catch(e){
						alert(Invariant["ErrorMessage"] + e) ;
					}
					$("#Open").dialog("close") ;
				} ,
				"キャンセル" : function(event){
					$("#Open").dialog("close") ;
				}
			}
		}) ;
		$("#searchbox").dialog({
			modal : false ,
			draggable : true ,
			title : Invariant["Search"] ,
			height : 200 ,
			width : 300 ,
			autoOpen : false ,
			buttons : {
				"検索" : function(){
					CodeMirror.commands.find($("#search_query").val() , Sys.commonEditor) ;
				} ,
				"閉じる" : function(){
					$("#searchbox").dialog("close") ;
				}
			}
		}) ;
		$("#replacebox").dialog({
			modal : false ,
			draggable : true ,
			width : 530 ,
			height : 190 ,
			title : Invariant["Replace"] ,
			autoOpen : false ,
			buttons : {
				"検索" : function(){
					CodeMirror.commands.find($("#search_query_for_replace").val() , Sys.commonEditor) ;
				} ,
				"すべて置換" : function(){
					CodeMirror.commands.replace(Sys.commonEditor , true) ;
				}
			}
		}) ;
		
		var s = "" ;
		var fonts = util.getFonts() ;
		$("#fonts").html("") ;
		for(var i = 0 ; i < fonts.length;i++)
			s += "<option value=\"" + fonts[i] + "\" style=\"font-family:'" + fonts[i] + "'\">" + fonts[i] + "</option>\n" ;
		$("#fonts").append(s) ;
		$("#fonts").val( Sys.storage.get("TextEditor.FontFamily")) 	;
		$("#SettingEnvironment").dialog({
			modal : true ,
			draggable : true ,
			title : Invariant["SettingEnvironment"] ,
			height : 400 ,
			width : 500 ,
			autoOpen : false ,
			buttons : {
				"OK" : function(){
					util.setSettingItem() ;
					$("#SettingEnvironment").dialog("close") ;
				} ,
				"適用" : function(){
					util.setSettingItem() ;
				} ,
				"キャンセル" : function(){
					$("#SettingEnvironment").dialog("close") ;
				}
			}
		}) ;
		$("#FontTesterResult")
				.css("font-family" , $("#fonts option:selected").val())
				.css("font-size" , $("#fontSize").val() + "px") ;
		$("#fonts , #fontSize").change(function(){
			$("#FontTesterResult")
				.css("font-family" , $("#fonts option:selected").val())
				.css("font-size" , $("#fontSize").val() + "px") ;
		}) ;
		$("#About").dialog({
			modal : false ,
			draggable : true ,
			title : Invariant["APP_NAME"]  ,
			height : 400 ,
			width : 320 ,
			buttons : {
				"OK" : function(e){
					$("#About").dialog("close") ;
				}
			} ,
			autoOpen : false
		}) ;
		$("#setenv-tabs").tabs() ;
	} ,
	init : function(){							//全体の初期化
		
		
		Sys.outputStorageData() ;
		Sys.initCommonEditor() ;
		Sys.initClickedWork() ;
		Sys.initSettingItem() ;
		Sys.initDialog() ;
		var imgStyle = (Sys.storage.get("TextEditor.BackGroundImageStyle") ? Sys.storage.get("TextEditor.BackGroundImageStyle") : "fit_width_height") ;
		if(window.ActiveXObject){
			util.setBackGroundImageSize(imgStyle , document.documentElement.clientWidth + "px", (document.documentElement.clientHeight - Invariant["EditorHeightMines"] ) + "px") ;
		}else{
			util.setBackGroundImageSize(imgStyle , window.innerWidth + "px" , (window.innerHeight- Invariant["EditorHeightMines"] ) + "px" );
		}
		$(window).resize(function(){
			if(window.ActiveXObject){
				util.setBackGroundImageSize(imgStyle , document.documentElement.clientWidth + "px", (document.documentElement.clientHeight - Invariant["EditorHeightMines"] )+ "px" );
			}else{
				util.setBackGroundImageSize(imgStyle , window.innerWidth + "px" , (window.innerHeight - Invariant["EditorHeightMines"] )+ "px") ;
			}
		}) ;
		Sys.commonEditor.setValue("\n") ;	//行番号の幅がおかしいからこうしてやる。
		

		
		Sys.beginEditor.setValue(
			(Sys.storage.get("TextEditor.BeginScript") ? Sys.storage.get("TextEditor.BeginScript") : "//起動時に実行されるスクリプト(JavaScript)を書いてください\n")
		) ;
		
		Sys.onInitLoaded() ;

	} ,
	setWindowTitle : function(filePath){		//ウィンドウタイトルの設定
		document.title = Invariant["APP_NAME"] + " -" + filePath ;
	} ,
	commonEditor : null ,						//メインエディタのCodeMirrorオブジェクト
	storage : null　,							//ストレージデータ
	beginEditor : null ,						//開始時実行のスクリプトのエディタ
	currentFile : null ,						//現在開いているファイル
	addEventListener : function(name , func){	//イベントリスナの追加
		var events = [
			["mainloaded" , "onMainLoaded"] ,
			["fileopen" , "onFileOpen"] ,
			["filesave" , "onFileSave"] ,
			["windowclose" , "onWindowClose"] ,
			["settingend" , "onSettingEnd"] ,
			["exit" , "onExit"] ,
			["keyevent" , "onkeyEvent"] ,
			["initloaded" , "onInitLoaded"]
		]
		name = name.toLowerCase() ;
		for(var i=0;i<events.length;i++){
			if(name == events[i][0]){
				var frontfunc = eval("Sys." + events[i][1]) ;
				eval("Sys." + events[i][1] + "= function(){frontfunc();func();} " ) ;
			}
		}
	} ,
	onMainLoaded : function(){} ,
	onFileOpen : function(){} ,
	onFileSave : function(){} ,
	onWindowClose : function(){} ,
	onSettingEnd : function(){} ,
	onExit : function(){} ,
	onkeyEvent : function(){} ,
	onInitLoaded : function(){}
}

window.xstorage = function(uu , storage){	//ストレージ利用可能後(window.onloadより後)
	Sys.storage = storage ;
	if(Sys.storage.get("TextEditor.BeginScript")){
		try{
			eval(Sys.storage.get("TextEditor.BeginScript")) ;
		}catch(e){
			alert(Invariant["UserScriptErrorWhenStart"] + e) ;
		}
	}
	Sys.init(storage) ;	
	Sys.onMainLoaded() ;
}

$(function(){	
	document.title = Invariant["APP_NAME"] + " -無題" ;

	//jQueryで実装するメニューバー
	$("ul.sub").hide();
	$("ul.menubar li").hover(
		function(){
			$(">ul:not(:animated)",this).show() ;
		},
		function(){
			$(">ul",this).hide();
		}
	)
	//IE7とか8はvimスタイルに対応していない。
	if(!navigator.userAgent.match(/MSIE [7-8]/)){
		$("head").append('<script type="text/javascript" src="js/codemirror-util.vim.js"></scr' + 'ipt>')
		$("#keyMap").append('<option value="vim">vim</option>') ;
	}
	$(window).bind('beforeunload', function(){
		return Invariant["Close_Message"] ;
	}) ;

}) ;

/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.core.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){function c(b,c){var e=b.nodeName.toLowerCase();if("area"===e){var f=b.parentNode,g=f.name,h;return!b.href||!g||f.nodeName.toLowerCase()!=="map"?!1:(h=a("img[usemap=#"+g+"]")[0],!!h&&d(h))}return(/input|select|textarea|button|object/.test(e)?!b.disabled:"a"==e?b.href||c:c)&&d(b)}function d(b){return!a(b).parents().andSelf().filter(function(){return a.curCSS(this,"visibility")==="hidden"||a.expr.filters.hidden(this)}).length}a.ui=a.ui||{};if(a.ui.version)return;a.extend(a.ui,{version:"1.8.23",keyCode:{ALT:18,BACKSPACE:8,CAPS_LOCK:20,COMMA:188,COMMAND:91,COMMAND_LEFT:91,COMMAND_RIGHT:93,CONTROL:17,DELETE:46,DOWN:40,END:35,ENTER:13,ESCAPE:27,HOME:36,INSERT:45,LEFT:37,MENU:93,NUMPAD_ADD:107,NUMPAD_DECIMAL:110,NUMPAD_DIVIDE:111,NUMPAD_ENTER:108,NUMPAD_MULTIPLY:106,NUMPAD_SUBTRACT:109,PAGE_DOWN:34,PAGE_UP:33,PERIOD:190,RIGHT:39,SHIFT:16,SPACE:32,TAB:9,UP:38,WINDOWS:91}}),a.fn.extend({propAttr:a.fn.prop||a.fn.attr,_focus:a.fn.focus,focus:function(b,c){return typeof b=="number"?this.each(function(){var d=this;setTimeout(function(){a(d).focus(),c&&c.call(d)},b)}):this._focus.apply(this,arguments)},scrollParent:function(){var b;return a.browser.msie&&/(static|relative)/.test(this.css("position"))||/absolute/.test(this.css("position"))?b=this.parents().filter(function(){return/(relative|absolute|fixed)/.test(a.curCSS(this,"position",1))&&/(auto|scroll)/.test(a.curCSS(this,"overflow",1)+a.curCSS(this,"overflow-y",1)+a.curCSS(this,"overflow-x",1))}).eq(0):b=this.parents().filter(function(){return/(auto|scroll)/.test(a.curCSS(this,"overflow",1)+a.curCSS(this,"overflow-y",1)+a.curCSS(this,"overflow-x",1))}).eq(0),/fixed/.test(this.css("position"))||!b.length?a(document):b},zIndex:function(c){if(c!==b)return this.css("zIndex",c);if(this.length){var d=a(this[0]),e,f;while(d.length&&d[0]!==document){e=d.css("position");if(e==="absolute"||e==="relative"||e==="fixed"){f=parseInt(d.css("zIndex"),10);if(!isNaN(f)&&f!==0)return f}d=d.parent()}}return 0},disableSelection:function(){return this.bind((a.support.selectstart?"selectstart":"mousedown")+".ui-disableSelection",function(a){a.preventDefault()})},enableSelection:function(){return this.unbind(".ui-disableSelection")}}),a("<a>").outerWidth(1).jquery||a.each(["Width","Height"],function(c,d){function h(b,c,d,f){return a.each(e,function(){c-=parseFloat(a.curCSS(b,"padding"+this,!0))||0,d&&(c-=parseFloat(a.curCSS(b,"border"+this+"Width",!0))||0),f&&(c-=parseFloat(a.curCSS(b,"margin"+this,!0))||0)}),c}var e=d==="Width"?["Left","Right"]:["Top","Bottom"],f=d.toLowerCase(),g={innerWidth:a.fn.innerWidth,innerHeight:a.fn.innerHeight,outerWidth:a.fn.outerWidth,outerHeight:a.fn.outerHeight};a.fn["inner"+d]=function(c){return c===b?g["inner"+d].call(this):this.each(function(){a(this).css(f,h(this,c)+"px")})},a.fn["outer"+d]=function(b,c){return typeof b!="number"?g["outer"+d].call(this,b):this.each(function(){a(this).css(f,h(this,b,!0,c)+"px")})}}),a.extend(a.expr[":"],{data:a.expr.createPseudo?a.expr.createPseudo(function(b){return function(c){return!!a.data(c,b)}}):function(b,c,d){return!!a.data(b,d[3])},focusable:function(b){return c(b,!isNaN(a.attr(b,"tabindex")))},tabbable:function(b){var d=a.attr(b,"tabindex"),e=isNaN(d);return(e||d>=0)&&c(b,!e)}}),a(function(){var b=document.body,c=b.appendChild(c=document.createElement("div"));c.offsetHeight,a.extend(c.style,{minHeight:"100px",height:"auto",padding:0,borderWidth:0}),a.support.minHeight=c.offsetHeight===100,a.support.selectstart="onselectstart"in c,b.removeChild(c).style.display="none"}),a.curCSS||(a.curCSS=a.css),a.extend(a.ui,{plugin:{add:function(b,c,d){var e=a.ui[b].prototype;for(var f in d)e.plugins[f]=e.plugins[f]||[],e.plugins[f].push([c,d[f]])},call:function(a,b,c){var d=a.plugins[b];if(!d||!a.element[0].parentNode)return;for(var e=0;e<d.length;e++)a.options[d[e][0]]&&d[e][1].apply(a.element,c)}},contains:function(a,b){return document.compareDocumentPosition?a.compareDocumentPosition(b)&16:a!==b&&a.contains(b)},hasScroll:function(b,c){if(a(b).css("overflow")==="hidden")return!1;var d=c&&c==="left"?"scrollLeft":"scrollTop",e=!1;return b[d]>0?!0:(b[d]=1,e=b[d]>0,b[d]=0,e)},isOverAxis:function(a,b,c){return a>b&&a<b+c},isOver:function(b,c,d,e,f,g){return a.ui.isOverAxis(b,d,f)&&a.ui.isOverAxis(c,e,g)}})})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.widget.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){if(a.cleanData){var c=a.cleanData;a.cleanData=function(b){for(var d=0,e;(e=b[d])!=null;d++)try{a(e).triggerHandler("remove")}catch(f){}c(b)}}else{var d=a.fn.remove;a.fn.remove=function(b,c){return this.each(function(){return c||(!b||a.filter(b,[this]).length)&&a("*",this).add([this]).each(function(){try{a(this).triggerHandler("remove")}catch(b){}}),d.call(a(this),b,c)})}}a.widget=function(b,c,d){var e=b.split(".")[0],f;b=b.split(".")[1],f=e+"-"+b,d||(d=c,c=a.Widget),a.expr[":"][f]=function(c){return!!a.data(c,b)},a[e]=a[e]||{},a[e][b]=function(a,b){arguments.length&&this._createWidget(a,b)};var g=new c;g.options=a.extend(!0,{},g.options),a[e][b].prototype=a.extend(!0,g,{namespace:e,widgetName:b,widgetEventPrefix:a[e][b].prototype.widgetEventPrefix||b,widgetBaseClass:f},d),a.widget.bridge(b,a[e][b])},a.widget.bridge=function(c,d){a.fn[c]=function(e){var f=typeof e=="string",g=Array.prototype.slice.call(arguments,1),h=this;return e=!f&&g.length?a.extend.apply(null,[!0,e].concat(g)):e,f&&e.charAt(0)==="_"?h:(f?this.each(function(){var d=a.data(this,c),f=d&&a.isFunction(d[e])?d[e].apply(d,g):d;if(f!==d&&f!==b)return h=f,!1}):this.each(function(){var b=a.data(this,c);b?b.option(e||{})._init():a.data(this,c,new d(e,this))}),h)}},a.Widget=function(a,b){arguments.length&&this._createWidget(a,b)},a.Widget.prototype={widgetName:"widget",widgetEventPrefix:"",options:{disabled:!1},_createWidget:function(b,c){a.data(c,this.widgetName,this),this.element=a(c),this.options=a.extend(!0,{},this.options,this._getCreateOptions(),b);var d=this;this.element.bind("remove."+this.widgetName,function(){d.destroy()}),this._create(),this._trigger("create"),this._init()},_getCreateOptions:function(){return a.metadata&&a.metadata.get(this.element[0])[this.widgetName]},_create:function(){},_init:function(){},destroy:function(){this.element.unbind("."+this.widgetName).removeData(this.widgetName),this.widget().unbind("."+this.widgetName).removeAttr("aria-disabled").removeClass(this.widgetBaseClass+"-disabled "+"ui-state-disabled")},widget:function(){return this.element},option:function(c,d){var e=c;if(arguments.length===0)return a.extend({},this.options);if(typeof c=="string"){if(d===b)return this.options[c];e={},e[c]=d}return this._setOptions(e),this},_setOptions:function(b){var c=this;return a.each(b,function(a,b){c._setOption(a,b)}),this},_setOption:function(a,b){return this.options[a]=b,a==="disabled"&&this.widget()[b?"addClass":"removeClass"](this.widgetBaseClass+"-disabled"+" "+"ui-state-disabled").attr("aria-disabled",b),this},enable:function(){return this._setOption("disabled",!1)},disable:function(){return this._setOption("disabled",!0)},_trigger:function(b,c,d){var e,f,g=this.options[b];d=d||{},c=a.Event(c),c.type=(b===this.widgetEventPrefix?b:this.widgetEventPrefix+b).toLowerCase(),c.target=this.element[0],f=c.originalEvent;if(f)for(e in f)e in c||(c[e]=f[e]);return this.element.trigger(c,d),!(a.isFunction(g)&&g.call(this.element[0],c,d)===!1||c.isDefaultPrevented())}}})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.mouse.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){var c=!1;a(document).mouseup(function(a){c=!1}),a.widget("ui.mouse",{options:{cancel:":input,option",distance:1,delay:0},_mouseInit:function(){var b=this;this.element.bind("mousedown."+this.widgetName,function(a){return b._mouseDown(a)}).bind("click."+this.widgetName,function(c){if(!0===a.data(c.target,b.widgetName+".preventClickEvent"))return a.removeData(c.target,b.widgetName+".preventClickEvent"),c.stopImmediatePropagation(),!1}),this.started=!1},_mouseDestroy:function(){this.element.unbind("."+this.widgetName),this._mouseMoveDelegate&&a(document).unbind("mousemove."+this.widgetName,this._mouseMoveDelegate).unbind("mouseup."+this.widgetName,this._mouseUpDelegate)},_mouseDown:function(b){if(c)return;this._mouseStarted&&this._mouseUp(b),this._mouseDownEvent=b;var d=this,e=b.which==1,f=typeof this.options.cancel=="string"&&b.target.nodeName?a(b.target).closest(this.options.cancel).length:!1;if(!e||f||!this._mouseCapture(b))return!0;this.mouseDelayMet=!this.options.delay,this.mouseDelayMet||(this._mouseDelayTimer=setTimeout(function(){d.mouseDelayMet=!0},this.options.delay));if(this._mouseDistanceMet(b)&&this._mouseDelayMet(b)){this._mouseStarted=this._mouseStart(b)!==!1;if(!this._mouseStarted)return b.preventDefault(),!0}return!0===a.data(b.target,this.widgetName+".preventClickEvent")&&a.removeData(b.target,this.widgetName+".preventClickEvent"),this._mouseMoveDelegate=function(a){return d._mouseMove(a)},this._mouseUpDelegate=function(a){return d._mouseUp(a)},a(document).bind("mousemove."+this.widgetName,this._mouseMoveDelegate).bind("mouseup."+this.widgetName,this._mouseUpDelegate),b.preventDefault(),c=!0,!0},_mouseMove:function(b){return!a.browser.msie||document.documentMode>=9||!!b.button?this._mouseStarted?(this._mouseDrag(b),b.preventDefault()):(this._mouseDistanceMet(b)&&this._mouseDelayMet(b)&&(this._mouseStarted=this._mouseStart(this._mouseDownEvent,b)!==!1,this._mouseStarted?this._mouseDrag(b):this._mouseUp(b)),!this._mouseStarted):this._mouseUp(b)},_mouseUp:function(b){return a(document).unbind("mousemove."+this.widgetName,this._mouseMoveDelegate).unbind("mouseup."+this.widgetName,this._mouseUpDelegate),this._mouseStarted&&(this._mouseStarted=!1,b.target==this._mouseDownEvent.target&&a.data(b.target,this.widgetName+".preventClickEvent",!0),this._mouseStop(b)),!1},_mouseDistanceMet:function(a){return Math.max(Math.abs(this._mouseDownEvent.pageX-a.pageX),Math.abs(this._mouseDownEvent.pageY-a.pageY))>=this.options.distance},_mouseDelayMet:function(a){return this.mouseDelayMet},_mouseStart:function(a){},_mouseDrag:function(a){},_mouseStop:function(a){},_mouseCapture:function(a){return!0}})})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.position.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){a.ui=a.ui||{};var c=/left|center|right/,d=/top|center|bottom/,e="center",f={},g=a.fn.position,h=a.fn.offset;a.fn.position=function(b){if(!b||!b.of)return g.apply(this,arguments);b=a.extend({},b);var h=a(b.of),i=h[0],j=(b.collision||"flip").split(" "),k=b.offset?b.offset.split(" "):[0,0],l,m,n;return i.nodeType===9?(l=h.width(),m=h.height(),n={top:0,left:0}):i.setTimeout?(l=h.width(),m=h.height(),n={top:h.scrollTop(),left:h.scrollLeft()}):i.preventDefault?(b.at="left top",l=m=0,n={top:b.of.pageY,left:b.of.pageX}):(l=h.outerWidth(),m=h.outerHeight(),n=h.offset()),a.each(["my","at"],function(){var a=(b[this]||"").split(" ");a.length===1&&(a=c.test(a[0])?a.concat([e]):d.test(a[0])?[e].concat(a):[e,e]),a[0]=c.test(a[0])?a[0]:e,a[1]=d.test(a[1])?a[1]:e,b[this]=a}),j.length===1&&(j[1]=j[0]),k[0]=parseInt(k[0],10)||0,k.length===1&&(k[1]=k[0]),k[1]=parseInt(k[1],10)||0,b.at[0]==="right"?n.left+=l:b.at[0]===e&&(n.left+=l/2),b.at[1]==="bottom"?n.top+=m:b.at[1]===e&&(n.top+=m/2),n.left+=k[0],n.top+=k[1],this.each(function(){var c=a(this),d=c.outerWidth(),g=c.outerHeight(),h=parseInt(a.curCSS(this,"marginLeft",!0))||0,i=parseInt(a.curCSS(this,"marginTop",!0))||0,o=d+h+(parseInt(a.curCSS(this,"marginRight",!0))||0),p=g+i+(parseInt(a.curCSS(this,"marginBottom",!0))||0),q=a.extend({},n),r;b.my[0]==="right"?q.left-=d:b.my[0]===e&&(q.left-=d/2),b.my[1]==="bottom"?q.top-=g:b.my[1]===e&&(q.top-=g/2),f.fractions||(q.left=Math.round(q.left),q.top=Math.round(q.top)),r={left:q.left-h,top:q.top-i},a.each(["left","top"],function(c,e){a.ui.position[j[c]]&&a.ui.position[j[c]][e](q,{targetWidth:l,targetHeight:m,elemWidth:d,elemHeight:g,collisionPosition:r,collisionWidth:o,collisionHeight:p,offset:k,my:b.my,at:b.at})}),a.fn.bgiframe&&c.bgiframe(),c.offset(a.extend(q,{using:b.using}))})},a.ui.position={fit:{left:function(b,c){var d=a(window),e=c.collisionPosition.left+c.collisionWidth-d.width()-d.scrollLeft();b.left=e>0?b.left-e:Math.max(b.left-c.collisionPosition.left,b.left)},top:function(b,c){var d=a(window),e=c.collisionPosition.top+c.collisionHeight-d.height()-d.scrollTop();b.top=e>0?b.top-e:Math.max(b.top-c.collisionPosition.top,b.top)}},flip:{left:function(b,c){if(c.at[0]===e)return;var d=a(window),f=c.collisionPosition.left+c.collisionWidth-d.width()-d.scrollLeft(),g=c.my[0]==="left"?-c.elemWidth:c.my[0]==="right"?c.elemWidth:0,h=c.at[0]==="left"?c.targetWidth:-c.targetWidth,i=-2*c.offset[0];b.left+=c.collisionPosition.left<0?g+h+i:f>0?g+h+i:0},top:function(b,c){if(c.at[1]===e)return;var d=a(window),f=c.collisionPosition.top+c.collisionHeight-d.height()-d.scrollTop(),g=c.my[1]==="top"?-c.elemHeight:c.my[1]==="bottom"?c.elemHeight:0,h=c.at[1]==="top"?c.targetHeight:-c.targetHeight,i=-2*c.offset[1];b.top+=c.collisionPosition.top<0?g+h+i:f>0?g+h+i:0}}},a.offset.setOffset||(a.offset.setOffset=function(b,c){/static/.test(a.curCSS(b,"position"))&&(b.style.position="relative");var d=a(b),e=d.offset(),f=parseInt(a.curCSS(b,"top",!0),10)||0,g=parseInt(a.curCSS(b,"left",!0),10)||0,h={top:c.top-e.top+f,left:c.left-e.left+g};"using"in c?c.using.call(b,h):d.css(h)},a.fn.offset=function(b){var c=this[0];return!c||!c.ownerDocument?null:b?a.isFunction(b)?this.each(function(c){a(this).offset(b.call(this,c,a(this).offset()))}):this.each(function(){a.offset.setOffset(this,b)}):h.call(this)}),a.curCSS||(a.curCSS=a.css),function(){var b=document.getElementsByTagName("body")[0],c=document.createElement("div"),d,e,g,h,i;d=document.createElement(b?"div":"body"),g={visibility:"hidden",width:0,height:0,border:0,margin:0,background:"none"},b&&a.extend(g,{position:"absolute",left:"-1000px",top:"-1000px"});for(var j in g)d.style[j]=g[j];d.appendChild(c),e=b||document.documentElement,e.insertBefore(d,e.firstChild),c.style.cssText="position: absolute; left: 10.7432222px; top: 10.432325px; height: 30px; width: 201px;",h=a(c).offset(function(a,b){return b}).offset(),d.innerHTML="",e.removeChild(d),i=h.top+h.left+(b?2e3:0),f.fractions=i>21&&i<22}()})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.resizable.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){a.widget("ui.resizable",a.ui.mouse,{widgetEventPrefix:"resize",options:{alsoResize:!1,animate:!1,animateDuration:"slow",animateEasing:"swing",aspectRatio:!1,autoHide:!1,containment:!1,ghost:!1,grid:!1,handles:"e,s,se",helper:!1,maxHeight:null,maxWidth:null,minHeight:10,minWidth:10,zIndex:1e3},_create:function(){var b=this,c=this.options;this.element.addClass("ui-resizable"),a.extend(this,{_aspectRatio:!!c.aspectRatio,aspectRatio:c.aspectRatio,originalElement:this.element,_proportionallyResizeElements:[],_helper:c.helper||c.ghost||c.animate?c.helper||"ui-resizable-helper":null}),this.element[0].nodeName.match(/canvas|textarea|input|select|button|img/i)&&(this.element.wrap(a('<div class="ui-wrapper" style="overflow: hidden;"></div>').css({position:this.element.css("position"),width:this.element.outerWidth(),height:this.element.outerHeight(),top:this.element.css("top"),left:this.element.css("left")})),this.element=this.element.parent().data("resizable",this.element.data("resizable")),this.elementIsWrapper=!0,this.element.css({marginLeft:this.originalElement.css("marginLeft"),marginTop:this.originalElement.css("marginTop"),marginRight:this.originalElement.css("marginRight"),marginBottom:this.originalElement.css("marginBottom")}),this.originalElement.css({marginLeft:0,marginTop:0,marginRight:0,marginBottom:0}),this.originalResizeStyle=this.originalElement.css("resize"),this.originalElement.css("resize","none"),this._proportionallyResizeElements.push(this.originalElement.css({position:"static",zoom:1,display:"block"})),this.originalElement.css({margin:this.originalElement.css("margin")}),this._proportionallyResize()),this.handles=c.handles||(a(".ui-resizable-handle",this.element).length?{n:".ui-resizable-n",e:".ui-resizable-e",s:".ui-resizable-s",w:".ui-resizable-w",se:".ui-resizable-se",sw:".ui-resizable-sw",ne:".ui-resizable-ne",nw:".ui-resizable-nw"}:"e,s,se");if(this.handles.constructor==String){this.handles=="all"&&(this.handles="n,e,s,w,se,sw,ne,nw");var d=this.handles.split(",");this.handles={};for(var e=0;e<d.length;e++){var f=a.trim(d[e]),g="ui-resizable-"+f,h=a('<div class="ui-resizable-handle '+g+'"></div>');h.css({zIndex:c.zIndex}),"se"==f&&h.addClass("ui-icon ui-icon-gripsmall-diagonal-se"),this.handles[f]=".ui-resizable-"+f,this.element.append(h)}}this._renderAxis=function(b){b=b||this.element;for(var c in this.handles){this.handles[c].constructor==String&&(this.handles[c]=a(this.handles[c],this.element).show());if(this.elementIsWrapper&&this.originalElement[0].nodeName.match(/textarea|input|select|button/i)){var d=a(this.handles[c],this.element),e=0;e=/sw|ne|nw|se|n|s/.test(c)?d.outerHeight():d.outerWidth();var f=["padding",/ne|nw|n/.test(c)?"Top":/se|sw|s/.test(c)?"Bottom":/^e$/.test(c)?"Right":"Left"].join("");b.css(f,e),this._proportionallyResize()}if(!a(this.handles[c]).length)continue}},this._renderAxis(this.element),this._handles=a(".ui-resizable-handle",this.element).disableSelection(),this._handles.mouseover(function(){if(!b.resizing){if(this.className)var a=this.className.match(/ui-resizable-(se|sw|ne|nw|n|e|s|w)/i);b.axis=a&&a[1]?a[1]:"se"}}),c.autoHide&&(this._handles.hide(),a(this.element).addClass("ui-resizable-autohide").hover(function(){if(c.disabled)return;a(this).removeClass("ui-resizable-autohide"),b._handles.show()},function(){if(c.disabled)return;b.resizing||(a(this).addClass("ui-resizable-autohide"),b._handles.hide())})),this._mouseInit()},destroy:function(){this._mouseDestroy();var b=function(b){a(b).removeClass("ui-resizable ui-resizable-disabled ui-resizable-resizing").removeData("resizable").unbind(".resizable").find(".ui-resizable-handle").remove()};if(this.elementIsWrapper){b(this.element);var c=this.element;c.after(this.originalElement.css({position:c.css("position"),width:c.outerWidth(),height:c.outerHeight(),top:c.css("top"),left:c.css("left")})).remove()}return this.originalElement.css("resize",this.originalResizeStyle),b(this.originalElement),this},_mouseCapture:function(b){var c=!1;for(var d in this.handles)a(this.handles[d])[0]==b.target&&(c=!0);return!this.options.disabled&&c},_mouseStart:function(b){var d=this.options,e=this.element.position(),f=this.element;this.resizing=!0,this.documentScroll={top:a(document).scrollTop(),left:a(document).scrollLeft()},(f.is(".ui-draggable")||/absolute/.test(f.css("position")))&&f.css({position:"absolute",top:e.top,left:e.left}),this._renderProxy();var g=c(this.helper.css("left")),h=c(this.helper.css("top"));d.containment&&(g+=a(d.containment).scrollLeft()||0,h+=a(d.containment).scrollTop()||0),this.offset=this.helper.offset(),this.position={left:g,top:h},this.size=this._helper?{width:f.outerWidth(),height:f.outerHeight()}:{width:f.width(),height:f.height()},this.originalSize=this._helper?{width:f.outerWidth(),height:f.outerHeight()}:{width:f.width(),height:f.height()},this.originalPosition={left:g,top:h},this.sizeDiff={width:f.outerWidth()-f.width(),height:f.outerHeight()-f.height()},this.originalMousePosition={left:b.pageX,top:b.pageY},this.aspectRatio=typeof d.aspectRatio=="number"?d.aspectRatio:this.originalSize.width/this.originalSize.height||1;var i=a(".ui-resizable-"+this.axis).css("cursor");return a("body").css("cursor",i=="auto"?this.axis+"-resize":i),f.addClass("ui-resizable-resizing"),this._propagate("start",b),!0},_mouseDrag:function(b){var c=this.helper,d=this.options,e={},f=this,g=this.originalMousePosition,h=this.axis,i=b.pageX-g.left||0,j=b.pageY-g.top||0,k=this._change[h];if(!k)return!1;var l=k.apply(this,[b,i,j]),m=a.browser.msie&&a.browser.version<7,n=this.sizeDiff;this._updateVirtualBoundaries(b.shiftKey);if(this._aspectRatio||b.shiftKey)l=this._updateRatio(l,b);return l=this._respectSize(l,b),this._propagate("resize",b),c.css({top:this.position.top+"px",left:this.position.left+"px",width:this.size.width+"px",height:this.size.height+"px"}),!this._helper&&this._proportionallyResizeElements.length&&this._proportionallyResize(),this._updateCache(l),this._trigger("resize",b,this.ui()),!1},_mouseStop:function(b){this.resizing=!1;var c=this.options,d=this;if(this._helper){var e=this._proportionallyResizeElements,f=e.length&&/textarea/i.test(e[0].nodeName),g=f&&a.ui.hasScroll(e[0],"left")?0:d.sizeDiff.height,h=f?0:d.sizeDiff.width,i={width:d.helper.width()-h,height:d.helper.height()-g},j=parseInt(d.element.css("left"),10)+(d.position.left-d.originalPosition.left)||null,k=parseInt(d.element.css("top"),10)+(d.position.top-d.originalPosition.top)||null;c.animate||this.element.css(a.extend(i,{top:k,left:j})),d.helper.height(d.size.height),d.helper.width(d.size.width),this._helper&&!c.animate&&this._proportionallyResize()}return a("body").css("cursor","auto"),this.element.removeClass("ui-resizable-resizing"),this._propagate("stop",b),this._helper&&this.helper.remove(),!1},_updateVirtualBoundaries:function(a){var b=this.options,c,e,f,g,h;h={minWidth:d(b.minWidth)?b.minWidth:0,maxWidth:d(b.maxWidth)?b.maxWidth:Infinity,minHeight:d(b.minHeight)?b.minHeight:0,maxHeight:d(b.maxHeight)?b.maxHeight:Infinity};if(this._aspectRatio||a)c=h.minHeight*this.aspectRatio,f=h.minWidth/this.aspectRatio,e=h.maxHeight*this.aspectRatio,g=h.maxWidth/this.aspectRatio,c>h.minWidth&&(h.minWidth=c),f>h.minHeight&&(h.minHeight=f),e<h.maxWidth&&(h.maxWidth=e),g<h.maxHeight&&(h.maxHeight=g);this._vBoundaries=h},_updateCache:function(a){var b=this.options;this.offset=this.helper.offset(),d(a.left)&&(this.position.left=a.left),d(a.top)&&(this.position.top=a.top),d(a.height)&&(this.size.height=a.height),d(a.width)&&(this.size.width=a.width)},_updateRatio:function(a,b){var c=this.options,e=this.position,f=this.size,g=this.axis;return d(a.height)?a.width=a.height*this.aspectRatio:d(a.width)&&(a.height=a.width/this.aspectRatio),g=="sw"&&(a.left=e.left+(f.width-a.width),a.top=null),g=="nw"&&(a.top=e.top+(f.height-a.height),a.left=e.left+(f.width-a.width)),a},_respectSize:function(a,b){var c=this.helper,e=this._vBoundaries,f=this._aspectRatio||b.shiftKey,g=this.axis,h=d(a.width)&&e.maxWidth&&e.maxWidth<a.width,i=d(a.height)&&e.maxHeight&&e.maxHeight<a.height,j=d(a.width)&&e.minWidth&&e.minWidth>a.width,k=d(a.height)&&e.minHeight&&e.minHeight>a.height;j&&(a.width=e.minWidth),k&&(a.height=e.minHeight),h&&(a.width=e.maxWidth),i&&(a.height=e.maxHeight);var l=this.originalPosition.left+this.originalSize.width,m=this.position.top+this.size.height,n=/sw|nw|w/.test(g),o=/nw|ne|n/.test(g);j&&n&&(a.left=l-e.minWidth),h&&n&&(a.left=l-e.maxWidth),k&&o&&(a.top=m-e.minHeight),i&&o&&(a.top=m-e.maxHeight);var p=!a.width&&!a.height;return p&&!a.left&&a.top?a.top=null:p&&!a.top&&a.left&&(a.left=null),a},_proportionallyResize:function(){var b=this.options;if(!this._proportionallyResizeElements.length)return;var c=this.helper||this.element;for(var d=0;d<this._proportionallyResizeElements.length;d++){var e=this._proportionallyResizeElements[d];if(!this.borderDif){var f=[e.css("borderTopWidth"),e.css("borderRightWidth"),e.css("borderBottomWidth"),e.css("borderLeftWidth")],g=[e.css("paddingTop"),e.css("paddingRight"),e.css("paddingBottom"),e.css("paddingLeft")];this.borderDif=a.map(f,function(a,b){var c=parseInt(a,10)||0,d=parseInt(g[b],10)||0;return c+d})}if(!a.browser.msie||!a(c).is(":hidden")&&!a(c).parents(":hidden").length)e.css({height:c.height()-this.borderDif[0]-this.borderDif[2]||0,width:c.width()-this.borderDif[1]-this.borderDif[3]||0});else continue}},_renderProxy:function(){var b=this.element,c=this.options;this.elementOffset=b.offset();if(this._helper){this.helper=this.helper||a('<div style="overflow:hidden;"></div>');var d=a.browser.msie&&a.browser.version<7,e=d?1:0,f=d?2:-1;this.helper.addClass(this._helper).css({width:this.element.outerWidth()+f,height:this.element.outerHeight()+f,position:"absolute",left:this.elementOffset.left-e+"px",top:this.elementOffset.top-e+"px",zIndex:++c.zIndex}),this.helper.appendTo("body").disableSelection()}else this.helper=this.element},_change:{e:function(a,b,c){return{width:this.originalSize.width+b}},w:function(a,b,c){var d=this.options,e=this.originalSize,f=this.originalPosition;return{left:f.left+b,width:e.width-b}},n:function(a,b,c){var d=this.options,e=this.originalSize,f=this.originalPosition;return{top:f.top+c,height:e.height-c}},s:function(a,b,c){return{height:this.originalSize.height+c}},se:function(b,c,d){return a.extend(this._change.s.apply(this,arguments),this._change.e.apply(this,[b,c,d]))},sw:function(b,c,d){return a.extend(this._change.s.apply(this,arguments),this._change.w.apply(this,[b,c,d]))},ne:function(b,c,d){return a.extend(this._change.n.apply(this,arguments),this._change.e.apply(this,[b,c,d]))},nw:function(b,c,d){return a.extend(this._change.n.apply(this,arguments),this._change.w.apply(this,[b,c,d]))}},_propagate:function(b,c){a.ui.plugin.call(this,b,[c,this.ui()]),b!="resize"&&this._trigger(b,c,this.ui())},plugins:{},ui:function(){return{originalElement:this.originalElement,element:this.element,helper:this.helper,position:this.position,size:this.size,originalSize:this.originalSize,originalPosition:this.originalPosition}}}),a.extend(a.ui.resizable,{version:"1.8.23"}),a.ui.plugin.add("resizable","alsoResize",{start:function(b,c){var d=a(this).data("resizable"),e=d.options,f=function(b){a(b).each(function(){var b=a(this);b.data("resizable-alsoresize",{width:parseInt(b.width(),10),height:parseInt(b.height(),10),left:parseInt(b.css("left"),10),top:parseInt(b.css("top"),10)})})};typeof e.alsoResize=="object"&&!e.alsoResize.parentNode?e.alsoResize.length?(e.alsoResize=e.alsoResize[0],f(e.alsoResize)):a.each(e.alsoResize,function(a){f(a)}):f(e.alsoResize)},resize:function(b,c){var d=a(this).data("resizable"),e=d.options,f=d.originalSize,g=d.originalPosition,h={height:d.size.height-f.height||0,width:d.size.width-f.width||0,top:d.position.top-g.top||0,left:d.position.left-g.left||0},i=function(b,d){a(b).each(function(){var b=a(this),e=a(this).data("resizable-alsoresize"),f={},g=d&&d.length?d:b.parents(c.originalElement[0]).length?["width","height"]:["width","height","top","left"];a.each(g,function(a,b){var c=(e[b]||0)+(h[b]||0);c&&c>=0&&(f[b]=c||null)}),b.css(f)})};typeof e.alsoResize=="object"&&!e.alsoResize.nodeType?a.each(e.alsoResize,function(a,b){i(a,b)}):i(e.alsoResize)},stop:function(b,c){a(this).removeData("resizable-alsoresize")}}),a.ui.plugin.add("resizable","animate",{stop:function(b,c){var d=a(this).data("resizable"),e=d.options,f=d._proportionallyResizeElements,g=f.length&&/textarea/i.test(f[0].nodeName),h=g&&a.ui.hasScroll(f[0],"left")?0:d.sizeDiff.height,i=g?0:d.sizeDiff.width,j={width:d.size.width-i,height:d.size.height-h},k=parseInt(d.element.css("left"),10)+(d.position.left-d.originalPosition.left)||null,l=parseInt(d.element.css("top"),10)+(d.position.top-d.originalPosition.top)||null;d.element.animate(a.extend(j,l&&k?{top:l,left:k}:{}),{duration:e.animateDuration,easing:e.animateEasing,step:function(){var c={width:parseInt(d.element.css("width"),10),height:parseInt(d.element.css("height"),10),top:parseInt(d.element.css("top"),10),left:parseInt(d.element.css("left"),10)};f&&f.length&&a(f[0]).css({width:c.width,height:c.height}),d._updateCache(c),d._propagate("resize",b)}})}}),a.ui.plugin.add("resizable","containment",{start:function(b,d){var e=a(this).data("resizable"),f=e.options,g=e.element,h=f.containment,i=h instanceof a?h.get(0):/parent/.test(h)?g.parent().get(0):h;if(!i)return;e.containerElement=a(i);if(/document/.test(h)||h==document)e.containerOffset={left:0,top:0},e.containerPosition={left:0,top:0},e.parentData={element:a(document),left:0,top:0,width:a(document).width(),height:a(document).height()||document.body.parentNode.scrollHeight};else{var j=a(i),k=[];a(["Top","Right","Left","Bottom"]).each(function(a,b){k[a]=c(j.css("padding"+b))}),e.containerOffset=j.offset(),e.containerPosition=j.position(),e.containerSize={height:j.innerHeight()-k[3],width:j.innerWidth()-k[1]};var l=e.containerOffset,m=e.containerSize.height,n=e.containerSize.width,o=a.ui.hasScroll(i,"left")?i.scrollWidth:n,p=a.ui.hasScroll(i)?i.scrollHeight:m;e.parentData={element:i,left:l.left,top:l.top,width:o,height:p}}},resize:function(b,c){var d=a(this).data("resizable"),e=d.options,f=d.containerSize,g=d.containerOffset,h=d.size,i=d.position,j=d._aspectRatio||b.shiftKey,k={top:0,left:0},l=d.containerElement;l[0]!=document&&/static/.test(l.css("position"))&&(k=g),i.left<(d._helper?g.left:0)&&(d.size.width=d.size.width+(d._helper?d.position.left-g.left:d.position.left-k.left),j&&(d.size.height=d.size.width/d.aspectRatio),d.position.left=e.helper?g.left:0),i.top<(d._helper?g.top:0)&&(d.size.height=d.size.height+(d._helper?d.position.top-g.top:d.position.top),j&&(d.size.width=d.size.height*d.aspectRatio),d.position.top=d._helper?g.top:0),d.offset.left=d.parentData.left+d.position.left,d.offset.top=d.parentData.top+d.position.top;var m=Math.abs((d._helper?d.offset.left-k.left:d.offset.left-k.left)+d.sizeDiff.width),n=Math.abs((d._helper?d.offset.top-k.top:d.offset.top-g.top)+d.sizeDiff.height),o=d.containerElement.get(0)==d.element.parent().get(0),p=/relative|absolute/.test(d.containerElement.css("position"));o&&p&&(m-=d.parentData.left),m+d.size.width>=d.parentData.width&&(d.size.width=d.parentData.width-m,j&&(d.size.height=d.size.width/d.aspectRatio)),n+d.size.height>=d.parentData.height&&(d.size.height=d.parentData.height-n,j&&(d.size.width=d.size.height*d.aspectRatio))},stop:function(b,c){var d=a(this).data("resizable"),e=d.options,f=d.position,g=d.containerOffset,h=d.containerPosition,i=d.containerElement,j=a(d.helper),k=j.offset(),l=j.outerWidth()-d.sizeDiff.width,m=j.outerHeight()-d.sizeDiff.height;d._helper&&!e.animate&&/relative/.test(i.css("position"))&&a(this).css({left:k.left-h.left-g.left,width:l,height:m}),d._helper&&!e.animate&&/static/.test(i.css("position"))&&a(this).css({left:k.left-h.left-g.left,width:l,height:m})}}),a.ui.plugin.add("resizable","ghost",{start:function(b,c){var d=a(this).data("resizable"),e=d.options,f=d.size;d.ghost=d.originalElement.clone(),d.ghost.css({opacity:.25,display:"block",position:"relative",height:f.height,width:f.width,margin:0,left:0,top:0}).addClass("ui-resizable-ghost").addClass(typeof e.ghost=="string"?e.ghost:""),d.ghost.appendTo(d.helper)},resize:function(b,c){var d=a(this).data("resizable"),e=d.options;d.ghost&&d.ghost.css({position:"relative",height:d.size.height,width:d.size.width})},stop:function(b,c){var d=a(this).data("resizable"),e=d.options;d.ghost&&d.helper&&d.helper.get(0).removeChild(d.ghost.get(0))}}),a.ui.plugin.add("resizable","grid",{resize:function(b,c){var d=a(this).data("resizable"),e=d.options,f=d.size,g=d.originalSize,h=d.originalPosition,i=d.axis,j=e._aspectRatio||b.shiftKey;e.grid=typeof e.grid=="number"?[e.grid,e.grid]:e.grid;var k=Math.round((f.width-g.width)/(e.grid[0]||1))*(e.grid[0]||1),l=Math.round((f.height-g.height)/(e.grid[1]||1))*(e.grid[1]||1);/^(se|s|e)$/.test(i)?(d.size.width=g.width+k,d.size.height=g.height+l):/^(ne)$/.test(i)?(d.size.width=g.width+k,d.size.height=g.height+l,d.position.top=h.top-l):/^(sw)$/.test(i)?(d.size.width=g.width+k,d.size.height=g.height+l,d.position.left=h.left-k):(d.size.width=g.width+k,d.size.height=g.height+l,d.position.top=h.top-l,d.position.left=h.left-k)}});var c=function(a){return parseInt(a,10)||0},d=function(a){return!isNaN(parseInt(a,10))}})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.button.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){var c,d,e,f,g="ui-button ui-widget ui-state-default ui-corner-all",h="ui-state-hover ui-state-active ",i="ui-button-icons-only ui-button-icon-only ui-button-text-icons ui-button-text-icon-primary ui-button-text-icon-secondary ui-button-text-only",j=function(){var b=a(this).find(":ui-button");setTimeout(function(){b.button("refresh")},1)},k=function(b){var c=b.name,d=b.form,e=a([]);return c&&(d?e=a(d).find("[name='"+c+"']"):e=a("[name='"+c+"']",b.ownerDocument).filter(function(){return!this.form})),e};a.widget("ui.button",{options:{disabled:null,text:!0,label:null,icons:{primary:null,secondary:null}},_create:function(){this.element.closest("form").unbind("reset.button").bind("reset.button",j),typeof this.options.disabled!="boolean"?this.options.disabled=!!this.element.propAttr("disabled"):this.element.propAttr("disabled",this.options.disabled),this._determineButtonType(),this.hasTitle=!!this.buttonElement.attr("title");var b=this,h=this.options,i=this.type==="checkbox"||this.type==="radio",l="ui-state-hover"+(i?"":" ui-state-active"),m="ui-state-focus";h.label===null&&(h.label=this.buttonElement.html()),this.buttonElement.addClass(g).attr("role","button").bind("mouseenter.button",function(){if(h.disabled)return;a(this).addClass("ui-state-hover"),this===c&&a(this).addClass("ui-state-active")}).bind("mouseleave.button",function(){if(h.disabled)return;a(this).removeClass(l)}).bind("click.button",function(a){h.disabled&&(a.preventDefault(),a.stopImmediatePropagation())}),this.element.bind("focus.button",function(){b.buttonElement.addClass(m)}).bind("blur.button",function(){b.buttonElement.removeClass(m)}),i&&(this.element.bind("change.button",function(){if(f)return;b.refresh()}),this.buttonElement.bind("mousedown.button",function(a){if(h.disabled)return;f=!1,d=a.pageX,e=a.pageY}).bind("mouseup.button",function(a){if(h.disabled)return;if(d!==a.pageX||e!==a.pageY)f=!0})),this.type==="checkbox"?this.buttonElement.bind("click.button",function(){if(h.disabled||f)return!1;a(this).toggleClass("ui-state-active"),b.buttonElement.attr("aria-pressed",b.element[0].checked)}):this.type==="radio"?this.buttonElement.bind("click.button",function(){if(h.disabled||f)return!1;a(this).addClass("ui-state-active"),b.buttonElement.attr("aria-pressed","true");var c=b.element[0];k(c).not(c).map(function(){return a(this).button("widget")[0]}).removeClass("ui-state-active").attr("aria-pressed","false")}):(this.buttonElement.bind("mousedown.button",function(){if(h.disabled)return!1;a(this).addClass("ui-state-active"),c=this,a(document).one("mouseup",function(){c=null})}).bind("mouseup.button",function(){if(h.disabled)return!1;a(this).removeClass("ui-state-active")}).bind("keydown.button",function(b){if(h.disabled)return!1;(b.keyCode==a.ui.keyCode.SPACE||b.keyCode==a.ui.keyCode.ENTER)&&a(this).addClass("ui-state-active")}).bind("keyup.button",function(){a(this).removeClass("ui-state-active")}),this.buttonElement.is("a")&&this.buttonElement.keyup(function(b){b.keyCode===a.ui.keyCode.SPACE&&a(this).click()})),this._setOption("disabled",h.disabled),this._resetButton()},_determineButtonType:function(){this.element.is(":checkbox")?this.type="checkbox":this.element.is(":radio")?this.type="radio":this.element.is("input")?this.type="input":this.type="button";if(this.type==="checkbox"||this.type==="radio"){var a=this.element.parents().filter(":last"),b="label[for='"+this.element.attr("id")+"']";this.buttonElement=a.find(b),this.buttonElement.length||(a=a.length?a.siblings():this.element.siblings(),this.buttonElement=a.filter(b),this.buttonElement.length||(this.buttonElement=a.find(b))),this.element.addClass("ui-helper-hidden-accessible");var c=this.element.is(":checked");c&&this.buttonElement.addClass("ui-state-active"),this.buttonElement.attr("aria-pressed",c)}else this.buttonElement=this.element},widget:function(){return this.buttonElement},destroy:function(){this.element.removeClass("ui-helper-hidden-accessible"),this.buttonElement.removeClass(g+" "+h+" "+i).removeAttr("role").removeAttr("aria-pressed").html(this.buttonElement.find(".ui-button-text").html()),this.hasTitle||this.buttonElement.removeAttr("title"),a.Widget.prototype.destroy.call(this)},_setOption:function(b,c){a.Widget.prototype._setOption.apply(this,arguments);if(b==="disabled"){c?this.element.propAttr("disabled",!0):this.element.propAttr("disabled",!1);return}this._resetButton()},refresh:function(){var b=this.element.is(":disabled");b!==this.options.disabled&&this._setOption("disabled",b),this.type==="radio"?k(this.element[0]).each(function(){a(this).is(":checked")?a(this).button("widget").addClass("ui-state-active").attr("aria-pressed","true"):a(this).button("widget").removeClass("ui-state-active").attr("aria-pressed","false")}):this.type==="checkbox"&&(this.element.is(":checked")?this.buttonElement.addClass("ui-state-active").attr("aria-pressed","true"):this.buttonElement.removeClass("ui-state-active").attr("aria-pressed","false"))},_resetButton:function(){if(this.type==="input"){this.options.label&&this.element.val(this.options.label);return}var b=this.buttonElement.removeClass(i),c=a("<span></span>",this.element[0].ownerDocument).addClass("ui-button-text").html(this.options.label).appendTo(b.empty()).text(),d=this.options.icons,e=d.primary&&d.secondary,f=[];d.primary||d.secondary?(this.options.text&&f.push("ui-button-text-icon"+(e?"s":d.primary?"-primary":"-secondary")),d.primary&&b.prepend("<span class='ui-button-icon-primary ui-icon "+d.primary+"'></span>"),d.secondary&&b.append("<span class='ui-button-icon-secondary ui-icon "+d.secondary+"'></span>"),this.options.text||(f.push(e?"ui-button-icons-only":"ui-button-icon-only"),this.hasTitle||b.attr("title",c))):f.push("ui-button-text-only"),b.addClass(f.join(" "))}}),a.widget("ui.buttonset",{options:{items:":button, :submit, :reset, :checkbox, :radio, a, :data(button)"},_create:function(){this.element.addClass("ui-buttonset")},_init:function(){this.refresh()},_setOption:function(b,c){b==="disabled"&&this.buttons.button("option",b,c),a.Widget.prototype._setOption.apply(this,arguments)},refresh:function(){var b=this.element.css("direction")==="rtl";this.buttons=this.element.find(this.options.items).filter(":ui-button").button("refresh").end().not(":ui-button").button().end().map(function(){return a(this).button("widget")[0]}).removeClass("ui-corner-all ui-corner-left ui-corner-right").filter(":first").addClass(b?"ui-corner-right":"ui-corner-left").end().filter(":last").addClass(b?"ui-corner-left":"ui-corner-right").end().end()},destroy:function(){this.element.removeClass("ui-buttonset"),this.buttons.map(function(){return a(this).button("widget")[0]}).removeClass("ui-corner-left ui-corner-right").end().button("destroy"),a.Widget.prototype.destroy.call(this)}})})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.dialog.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){var c="ui-dialog ui-widget ui-widget-content ui-corner-all ",d={buttons:!0,height:!0,maxHeight:!0,maxWidth:!0,minHeight:!0,minWidth:!0,width:!0},e={maxHeight:!0,maxWidth:!0,minHeight:!0,minWidth:!0};a.widget("ui.dialog",{options:{autoOpen:!0,buttons:{},closeOnEscape:!0,closeText:"close",dialogClass:"",draggable:!0,hide:null,height:"auto",maxHeight:!1,maxWidth:!1,minHeight:150,minWidth:150,modal:!1,position:{my:"center",at:"center",collision:"fit",using:function(b){var c=a(this).css(b).offset().top;c<0&&a(this).css("top",b.top-c)}},resizable:!0,show:null,stack:!0,title:"",width:300,zIndex:1e3},_create:function(){this.originalTitle=this.element.attr("title"),typeof this.originalTitle!="string"&&(this.originalTitle=""),this.options.title=this.options.title||this.originalTitle;var b=this,d=b.options,e=d.title||"&#160;",f=a.ui.dialog.getTitleId(b.element),g=(b.uiDialog=a("<div></div>")).appendTo(document.body).hide().addClass(c+d.dialogClass).css({zIndex:d.zIndex}).attr("tabIndex",-1).css("outline",0).keydown(function(c){d.closeOnEscape&&!c.isDefaultPrevented()&&c.keyCode&&c.keyCode===a.ui.keyCode.ESCAPE&&(b.close(c),c.preventDefault())}).attr({role:"dialog","aria-labelledby":f}).mousedown(function(a){b.moveToTop(!1,a)}),h=b.element.show().removeAttr("title").addClass("ui-dialog-content ui-widget-content").appendTo(g),i=(b.uiDialogTitlebar=a("<div></div>")).addClass("ui-dialog-titlebar ui-widget-header ui-corner-all ui-helper-clearfix").prependTo(g),j=a('<a href="#"></a>').addClass("ui-dialog-titlebar-close ui-corner-all").attr("role","button").hover(function(){j.addClass("ui-state-hover")},function(){j.removeClass("ui-state-hover")}).focus(function(){j.addClass("ui-state-focus")}).blur(function(){j.removeClass("ui-state-focus")}).click(function(a){return b.close(a),!1}).appendTo(i),k=(b.uiDialogTitlebarCloseText=a("<span></span>")).addClass("ui-icon ui-icon-closethick").text(d.closeText).appendTo(j),l=a("<span></span>").addClass("ui-dialog-title").attr("id",f).html(e).prependTo(i);a.isFunction(d.beforeclose)&&!a.isFunction(d.beforeClose)&&(d.beforeClose=d.beforeclose),i.find("*").add(i).disableSelection(),d.draggable&&a.fn.draggable&&b._makeDraggable(),d.resizable&&a.fn.resizable&&b._makeResizable(),b._createButtons(d.buttons),b._isOpen=!1,a.fn.bgiframe&&g.bgiframe()},_init:function(){this.options.autoOpen&&this.open()},destroy:function(){var a=this;return a.overlay&&a.overlay.destroy(),a.uiDialog.hide(),a.element.unbind(".dialog").removeData("dialog").removeClass("ui-dialog-content ui-widget-content").hide().appendTo("body"),a.uiDialog.remove(),a.originalTitle&&a.element.attr("title",a.originalTitle),a},widget:function(){return this.uiDialog},close:function(b){var c=this,d,e;if(!1===c._trigger("beforeClose",b))return;return c.overlay&&c.overlay.destroy(),c.uiDialog.unbind("keypress.ui-dialog"),c._isOpen=!1,c.options.hide?c.uiDialog.hide(c.options.hide,function(){c._trigger("close",b)}):(c.uiDialog.hide(),c._trigger("close",b)),a.ui.dialog.overlay.resize(),c.options.modal&&(d=0,a(".ui-dialog").each(function(){this!==c.uiDialog[0]&&(e=a(this).css("z-index"),isNaN(e)||(d=Math.max(d,e)))}),a.ui.dialog.maxZ=d),c},isOpen:function(){return this._isOpen},moveToTop:function(b,c){var d=this,e=d.options,f;return e.modal&&!b||!e.stack&&!e.modal?d._trigger("focus",c):(e.zIndex>a.ui.dialog.maxZ&&(a.ui.dialog.maxZ=e.zIndex),d.overlay&&(a.ui.dialog.maxZ+=1,d.overlay.$el.css("z-index",a.ui.dialog.overlay.maxZ=a.ui.dialog.maxZ)),f={scrollTop:d.element.scrollTop(),scrollLeft:d.element.scrollLeft()},a.ui.dialog.maxZ+=1,d.uiDialog.css("z-index",a.ui.dialog.maxZ),d.element.attr(f),d._trigger("focus",c),d)},open:function(){if(this._isOpen)return;var b=this,c=b.options,d=b.uiDialog;return b.overlay=c.modal?new a.ui.dialog.overlay(b):null,b._size(),b._position(c.position),d.show(c.show),b.moveToTop(!0),c.modal&&d.bind("keydown.ui-dialog",function(b){if(b.keyCode!==a.ui.keyCode.TAB)return;var c=a(":tabbable",this),d=c.filter(":first"),e=c.filter(":last");if(b.target===e[0]&&!b.shiftKey)return d.focus(1),!1;if(b.target===d[0]&&b.shiftKey)return e.focus(1),!1}),a(b.element.find(":tabbable").get().concat(d.find(".ui-dialog-buttonpane :tabbable").get().concat(d.get()))).eq(0).focus(),b._isOpen=!0,b._trigger("open"),b},_createButtons:function(b){var c=this,d=!1,e=a("<div></div>").addClass("ui-dialog-buttonpane ui-widget-content ui-helper-clearfix"),f=a("<div></div>").addClass("ui-dialog-buttonset").appendTo(e);c.uiDialog.find(".ui-dialog-buttonpane").remove(),typeof b=="object"&&b!==null&&a.each(b,function(){return!(d=!0)}),d&&(a.each(b,function(b,d){d=a.isFunction(d)?{click:d,text:b}:d;var e=a('<button type="button"></button>').click(function(){d.click.apply(c.element[0],arguments)}).appendTo(f);a.each(d,function(a,b){if(a==="click")return;a in e?e[a](b):e.attr(a,b)}),a.fn.button&&e.button()}),e.appendTo(c.uiDialog))},_makeDraggable:function(){function f(a){return{position:a.position,offset:a.offset}}var b=this,c=b.options,d=a(document),e;b.uiDialog.draggable({cancel:".ui-dialog-content, .ui-dialog-titlebar-close",handle:".ui-dialog-titlebar",containment:"document",start:function(d,g){e=c.height==="auto"?"auto":a(this).height(),a(this).height(a(this).height()).addClass("ui-dialog-dragging"),b._trigger("dragStart",d,f(g))},drag:function(a,c){b._trigger("drag",a,f(c))},stop:function(g,h){c.position=[h.position.left-d.scrollLeft(),h.position.top-d.scrollTop()],a(this).removeClass("ui-dialog-dragging").height(e),b._trigger("dragStop",g,f(h)),a.ui.dialog.overlay.resize()}})},_makeResizable:function(c){function h(a){return{originalPosition:a.originalPosition,originalSize:a.originalSize,position:a.position,size:a.size}}c=c===b?this.options.resizable:c;var d=this,e=d.options,f=d.uiDialog.css("position"),g=typeof c=="string"?c:"n,e,s,w,se,sw,ne,nw";d.uiDialog.resizable({cancel:".ui-dialog-content",containment:"document",alsoResize:d.element,maxWidth:e.maxWidth,maxHeight:e.maxHeight,minWidth:e.minWidth,minHeight:d._minHeight(),handles:g,start:function(b,c){a(this).addClass("ui-dialog-resizing"),d._trigger("resizeStart",b,h(c))},resize:function(a,b){d._trigger("resize",a,h(b))},stop:function(b,c){a(this).removeClass("ui-dialog-resizing"),e.height=a(this).height(),e.width=a(this).width(),d._trigger("resizeStop",b,h(c)),a.ui.dialog.overlay.resize()}}).css("position",f).find(".ui-resizable-se").addClass("ui-icon ui-icon-grip-diagonal-se")},_minHeight:function(){var a=this.options;return a.height==="auto"?a.minHeight:Math.min(a.minHeight,a.height)},_position:function(b){var c=[],d=[0,0],e;if(b){if(typeof b=="string"||typeof b=="object"&&"0"in b)c=b.split?b.split(" "):[b[0],b[1]],c.length===1&&(c[1]=c[0]),a.each(["left","top"],function(a,b){+c[a]===c[a]&&(d[a]=c[a],c[a]=b)}),b={my:c.join(" "),at:c.join(" "),offset:d.join(" ")};b=a.extend({},a.ui.dialog.prototype.options.position,b)}else b=a.ui.dialog.prototype.options.position;e=this.uiDialog.is(":visible"),e||this.uiDialog.show(),this.uiDialog.css({top:0,left:0}).position(a.extend({of:window},b)),e||this.uiDialog.hide()},_setOptions:function(b){var c=this,f={},g=!1;a.each(b,function(a,b){c._setOption(a,b),a in d&&(g=!0),a in e&&(f[a]=b)}),g&&this._size(),this.uiDialog.is(":data(resizable)")&&this.uiDialog.resizable("option",f)},_setOption:function(b,d){var e=this,f=e.uiDialog;switch(b){case"beforeclose":b="beforeClose";break;case"buttons":e._createButtons(d);break;case"closeText":e.uiDialogTitlebarCloseText.text(""+d);break;case"dialogClass":f.removeClass(e.options.dialogClass).addClass(c+d);break;case"disabled":d?f.addClass("ui-dialog-disabled"):f.removeClass("ui-dialog-disabled");break;case"draggable":var g=f.is(":data(draggable)");g&&!d&&f.draggable("destroy"),!g&&d&&e._makeDraggable();break;case"position":e._position(d);break;case"resizable":var h=f.is(":data(resizable)");h&&!d&&f.resizable("destroy"),h&&typeof d=="string"&&f.resizable("option","handles",d),!h&&d!==!1&&e._makeResizable(d);break;case"title":a(".ui-dialog-title",e.uiDialogTitlebar).html(""+(d||"&#160;"))}a.Widget.prototype._setOption.apply(e,arguments)},_size:function(){var b=this.options,c,d,e=this.uiDialog.is(":visible");this.element.show().css({width:"auto",minHeight:0,height:0}),b.minWidth>b.width&&(b.width=b.minWidth),c=this.uiDialog.css({height:"auto",width:b.width}).height(),d=Math.max(0,b.minHeight-c);if(b.height==="auto")if(a.support.minHeight)this.element.css({minHeight:d,height:"auto"});else{this.uiDialog.show();var f=this.element.css("height","auto").height();e||this.uiDialog.hide(),this.element.height(Math.max(f,d))}else this.element.height(Math.max(b.height-c,0));this.uiDialog.is(":data(resizable)")&&this.uiDialog.resizable("option","minHeight",this._minHeight())}}),a.extend(a.ui.dialog,{version:"1.8.23",uuid:0,maxZ:0,getTitleId:function(a){var b=a.attr("id");return b||(this.uuid+=1,b=this.uuid),"ui-dialog-title-"+b},overlay:function(b){this.$el=a.ui.dialog.overlay.create(b)}}),a.extend(a.ui.dialog.overlay,{instances:[],oldInstances:[],maxZ:0,events:a.map("focus,mousedown,mouseup,keydown,keypress,click".split(","),function(a){return a+".dialog-overlay"}).join(" "),create:function(b){this.instances.length===0&&(setTimeout(function(){a.ui.dialog.overlay.instances.length&&a(document).bind(a.ui.dialog.overlay.events,function(b){if(a(b.target).zIndex()<a.ui.dialog.overlay.maxZ)return!1})},1),a(document).bind("keydown.dialog-overlay",function(c){b.options.closeOnEscape&&!c.isDefaultPrevented()&&c.keyCode&&c.keyCode===a.ui.keyCode.ESCAPE&&(b.close(c),c.preventDefault())}),a(window).bind("resize.dialog-overlay",a.ui.dialog.overlay.resize));var c=(this.oldInstances.pop()||a("<div></div>").addClass("ui-widget-overlay")).appendTo(document.body).css({width:this.width(),height:this.height()});return a.fn.bgiframe&&c.bgiframe(),this.instances.push(c),c},destroy:function(b){var c=a.inArray(b,this.instances);c!=-1&&this.oldInstances.push(this.instances.splice(c,1)[0]),this.instances.length===0&&a([document,window]).unbind(".dialog-overlay"),b.remove();var d=0;a.each(this.instances,function(){d=Math.max(d,this.css("z-index"))}),this.maxZ=d},height:function(){var b,c;return a.browser.msie&&a.browser.version<7?(b=Math.max(document.documentElement.scrollHeight,document.body.scrollHeight),c=Math.max(document.documentElement.offsetHeight,document.body.offsetHeight),b<c?a(window).height()+"px":b+"px"):a(document).height()+"px"},width:function(){var b,c;return a.browser.msie?(b=Math.max(document.documentElement.scrollWidth,document.body.scrollWidth),c=Math.max(document.documentElement.offsetWidth,document.body.offsetWidth),b<c?a(window).width()+"px":b+"px"):a(document).width()+"px"},resize:function(){var b=a([]);a.each(a.ui.dialog.overlay.instances,function(){b=b.add(this)}),b.css({width:0,height:0}).css({width:a.ui.dialog.overlay.width(),height:a.ui.dialog.overlay.height()})}}),a.extend(a.ui.dialog.overlay.prototype,{destroy:function(){a.ui.dialog.overlay.destroy(this.$el)}})})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.slider.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){var c=5;a.widget("ui.slider",a.ui.mouse,{widgetEventPrefix:"slide",options:{animate:!1,distance:0,max:100,min:0,orientation:"horizontal",range:!1,step:1,value:0,values:null},_create:function(){var b=this,d=this.options,e=this.element.find(".ui-slider-handle").addClass("ui-state-default ui-corner-all"),f="<a class='ui-slider-handle ui-state-default ui-corner-all' href='#'></a>",g=d.values&&d.values.length||1,h=[];this._keySliding=!1,this._mouseSliding=!1,this._animateOff=!0,this._handleIndex=null,this._detectOrientation(),this._mouseInit(),this.element.addClass("ui-slider ui-slider-"+this.orientation+" ui-widget"+" ui-widget-content"+" ui-corner-all"+(d.disabled?" ui-slider-disabled ui-disabled":"")),this.range=a([]),d.range&&(d.range===!0&&(d.values||(d.values=[this._valueMin(),this._valueMin()]),d.values.length&&d.values.length!==2&&(d.values=[d.values[0],d.values[0]])),this.range=a("<div></div>").appendTo(this.element).addClass("ui-slider-range ui-widget-header"+(d.range==="min"||d.range==="max"?" ui-slider-range-"+d.range:"")));for(var i=e.length;i<g;i+=1)h.push(f);this.handles=e.add(a(h.join("")).appendTo(b.element)),this.handle=this.handles.eq(0),this.handles.add(this.range).filter("a").click(function(a){a.preventDefault()}).hover(function(){d.disabled||a(this).addClass("ui-state-hover")},function(){a(this).removeClass("ui-state-hover")}).focus(function(){d.disabled?a(this).blur():(a(".ui-slider .ui-state-focus").removeClass("ui-state-focus"),a(this).addClass("ui-state-focus"))}).blur(function(){a(this).removeClass("ui-state-focus")}),this.handles.each(function(b){a(this).data("index.ui-slider-handle",b)}),this.handles.keydown(function(d){var e=a(this).data("index.ui-slider-handle"),f,g,h,i;if(b.options.disabled)return;switch(d.keyCode){case a.ui.keyCode.HOME:case a.ui.keyCode.END:case a.ui.keyCode.PAGE_UP:case a.ui.keyCode.PAGE_DOWN:case a.ui.keyCode.UP:case a.ui.keyCode.RIGHT:case a.ui.keyCode.DOWN:case a.ui.keyCode.LEFT:d.preventDefault();if(!b._keySliding){b._keySliding=!0,a(this).addClass("ui-state-active"),f=b._start(d,e);if(f===!1)return}}i=b.options.step,b.options.values&&b.options.values.length?g=h=b.values(e):g=h=b.value();switch(d.keyCode){case a.ui.keyCode.HOME:h=b._valueMin();break;case a.ui.keyCode.END:h=b._valueMax();break;case a.ui.keyCode.PAGE_UP:h=b._trimAlignValue(g+(b._valueMax()-b._valueMin())/c);break;case a.ui.keyCode.PAGE_DOWN:h=b._trimAlignValue(g-(b._valueMax()-b._valueMin())/c);break;case a.ui.keyCode.UP:case a.ui.keyCode.RIGHT:if(g===b._valueMax())return;h=b._trimAlignValue(g+i);break;case a.ui.keyCode.DOWN:case a.ui.keyCode.LEFT:if(g===b._valueMin())return;h=b._trimAlignValue(g-i)}b._slide(d,e,h)}).keyup(function(c){var d=a(this).data("index.ui-slider-handle");b._keySliding&&(b._keySliding=!1,b._stop(c,d),b._change(c,d),a(this).removeClass("ui-state-active"))}),this._refreshValue(),this._animateOff=!1},destroy:function(){return this.handles.remove(),this.range.remove(),this.element.removeClass("ui-slider ui-slider-horizontal ui-slider-vertical ui-slider-disabled ui-widget ui-widget-content ui-corner-all").removeData("slider").unbind(".slider"),this._mouseDestroy(),this},_mouseCapture:function(b){var c=this.options,d,e,f,g,h,i,j,k,l;return c.disabled?!1:(this.elementSize={width:this.element.outerWidth(),height:this.element.outerHeight()},this.elementOffset=this.element.offset(),d={x:b.pageX,y:b.pageY},e=this._normValueFromMouse(d),f=this._valueMax()-this._valueMin()+1,h=this,this.handles.each(function(b){var c=Math.abs(e-h.values(b));f>c&&(f=c,g=a(this),i=b)}),c.range===!0&&this.values(1)===c.min&&(i+=1,g=a(this.handles[i])),j=this._start(b,i),j===!1?!1:(this._mouseSliding=!0,h._handleIndex=i,g.addClass("ui-state-active").focus(),k=g.offset(),l=!a(b.target).parents().andSelf().is(".ui-slider-handle"),this._clickOffset=l?{left:0,top:0}:{left:b.pageX-k.left-g.width()/2,top:b.pageY-k.top-g.height()/2-(parseInt(g.css("borderTopWidth"),10)||0)-(parseInt(g.css("borderBottomWidth"),10)||0)+(parseInt(g.css("marginTop"),10)||0)},this.handles.hasClass("ui-state-hover")||this._slide(b,i,e),this._animateOff=!0,!0))},_mouseStart:function(a){return!0},_mouseDrag:function(a){var b={x:a.pageX,y:a.pageY},c=this._normValueFromMouse(b);return this._slide(a,this._handleIndex,c),!1},_mouseStop:function(a){return this.handles.removeClass("ui-state-active"),this._mouseSliding=!1,this._stop(a,this._handleIndex),this._change(a,this._handleIndex),this._handleIndex=null,this._clickOffset=null,this._animateOff=!1,!1},_detectOrientation:function(){this.orientation=this.options.orientation==="vertical"?"vertical":"horizontal"},_normValueFromMouse:function(a){var b,c,d,e,f;return this.orientation==="horizontal"?(b=this.elementSize.width,c=a.x-this.elementOffset.left-(this._clickOffset?this._clickOffset.left:0)):(b=this.elementSize.height,c=a.y-this.elementOffset.top-(this._clickOffset?this._clickOffset.top:0)),d=c/b,d>1&&(d=1),d<0&&(d=0),this.orientation==="vertical"&&(d=1-d),e=this._valueMax()-this._valueMin(),f=this._valueMin()+d*e,this._trimAlignValue(f)},_start:function(a,b){var c={handle:this.handles[b],value:this.value()};return this.options.values&&this.options.values.length&&(c.value=this.values(b),c.values=this.values()),this._trigger("start",a,c)},_slide:function(a,b,c){var d,e,f;this.options.values&&this.options.values.length?(d=this.values(b?0:1),this.options.values.length===2&&this.options.range===!0&&(b===0&&c>d||b===1&&c<d)&&(c=d),c!==this.values(b)&&(e=this.values(),e[b]=c,f=this._trigger("slide",a,{handle:this.handles[b],value:c,values:e}),d=this.values(b?0:1),f!==!1&&this.values(b,c,!0))):c!==this.value()&&(f=this._trigger("slide",a,{handle:this.handles[b],value:c}),f!==!1&&this.value(c))},_stop:function(a,b){var c={handle:this.handles[b],value:this.value()};this.options.values&&this.options.values.length&&(c.value=this.values(b),c.values=this.values()),this._trigger("stop",a,c)},_change:function(a,b){if(!this._keySliding&&!this._mouseSliding){var c={handle:this.handles[b],value:this.value()};this.options.values&&this.options.values.length&&(c.value=this.values(b),c.values=this.values()),this._trigger("change",a,c)}},value:function(a){if(arguments.length){this.options.value=this._trimAlignValue(a),this._refreshValue(),this._change(null,0);return}return this._value()},values:function(b,c){var d,e,f;if(arguments.length>1){this.options.values[b]=this._trimAlignValue(c),this._refreshValue(),this._change(null,b);return}if(!arguments.length)return this._values();if(!a.isArray(arguments[0]))return this.options.values&&this.options.values.length?this._values(b):this.value();d=this.options.values,e=arguments[0];for(f=0;f<d.length;f+=1)d[f]=this._trimAlignValue(e[f]),this._change(null,f);this._refreshValue()},_setOption:function(b,c){var d,e=0;a.isArray(this.options.values)&&(e=this.options.values.length),a.Widget.prototype._setOption.apply(this,arguments);switch(b){case"disabled":c?(this.handles.filter(".ui-state-focus").blur(),this.handles.removeClass("ui-state-hover"),this.handles.propAttr("disabled",!0),this.element.addClass("ui-disabled")):(this.handles.propAttr("disabled",!1),this.element.removeClass("ui-disabled"));break;case"orientation":this._detectOrientation(),this.element.removeClass("ui-slider-horizontal ui-slider-vertical").addClass("ui-slider-"+this.orientation),this._refreshValue();break;case"value":this._animateOff=!0,this._refreshValue(),this._change(null,0),this._animateOff=!1;break;case"values":this._animateOff=!0,this._refreshValue();for(d=0;d<e;d+=1)this._change(null,d);this._animateOff=!1}},_value:function(){var a=this.options.value;return a=this._trimAlignValue(a),a},_values:function(a){var b,c,d;if(arguments.length)return b=this.options.values[a],b=this._trimAlignValue(b),b;c=this.options.values.slice();for(d=0;d<c.length;d+=1)c[d]=this._trimAlignValue(c[d]);return c},_trimAlignValue:function(a){if(a<=this._valueMin())return this._valueMin();if(a>=this._valueMax())return this._valueMax();var b=this.options.step>0?this.options.step:1,c=(a-this._valueMin())%b,d=a-c;return Math.abs(c)*2>=b&&(d+=c>0?b:-b),parseFloat(d.toFixed(5))},_valueMin:function(){return this.options.min},_valueMax:function(){return this.options.max},_refreshValue:function(){var b=this.options.range,c=this.options,d=this,e=this._animateOff?!1:c.animate,f,g={},h,i,j,k;this.options.values&&this.options.values.length?this.handles.each(function(b,i){f=(d.values(b)-d._valueMin())/(d._valueMax()-d._valueMin())*100,g[d.orientation==="horizontal"?"left":"bottom"]=f+"%",a(this).stop(1,1)[e?"animate":"css"](g,c.animate),d.options.range===!0&&(d.orientation==="horizontal"?(b===0&&d.range.stop(1,1)[e?"animate":"css"]({left:f+"%"},c.animate),b===1&&d.range[e?"animate":"css"]({width:f-h+"%"},{queue:!1,duration:c.animate})):(b===0&&d.range.stop(1,1)[e?"animate":"css"]({bottom:f+"%"},c.animate),b===1&&d.range[e?"animate":"css"]({height:f-h+"%"},{queue:!1,duration:c.animate}))),h=f}):(i=this.value(),j=this._valueMin(),k=this._valueMax(),f=k!==j?(i-j)/(k-j)*100:0,g[d.orientation==="horizontal"?"left":"bottom"]=f+"%",this.handle.stop(1,1)[e?"animate":"css"](g,c.animate),b==="min"&&this.orientation==="horizontal"&&this.range.stop(1,1)[e?"animate":"css"]({width:f+"%"},c.animate),b==="max"&&this.orientation==="horizontal"&&this.range[e?"animate":"css"]({width:100-f+"%"},{queue:!1,duration:c.animate}),b==="min"&&this.orientation==="vertical"&&this.range.stop(1,1)[e?"animate":"css"]({height:f+"%"},c.animate),b==="max"&&this.orientation==="vertical"&&this.range[e?"animate":"css"]({height:100-f+"%"},{queue:!1,duration:c.animate}))}}),a.extend(a.ui.slider,{version:"1.8.23"})})(jQuery);;/*! jQuery UI - v1.8.23 - 2012-08-15
* https://github.com/jquery/jquery-ui
* Includes: jquery.ui.tabs.js
* Copyright (c) 2012 AUTHORS.txt; Licensed MIT, GPL */
(function(a,b){function e(){return++c}function f(){return++d}var c=0,d=0;a.widget("ui.tabs",{options:{add:null,ajaxOptions:null,cache:!1,cookie:null,collapsible:!1,disable:null,disabled:[],enable:null,event:"click",fx:null,idPrefix:"ui-tabs-",load:null,panelTemplate:"<div></div>",remove:null,select:null,show:null,spinner:"<em>Loading&#8230;</em>",tabTemplate:"<li><a href='#{href}'><span>#{label}</span></a></li>"},_create:function(){this._tabify(!0)},_setOption:function(a,b){if(a=="selected"){if(this.options.collapsible&&b==this.options.selected)return;this.select(b)}else this.options[a]=b,this._tabify()},_tabId:function(a){return a.title&&a.title.replace(/\s/g,"_").replace(/[^\w\u00c0-\uFFFF-]/g,"")||this.options.idPrefix+e()},_sanitizeSelector:function(a){return a.replace(/:/g,"\\:")},_cookie:function(){var b=this.cookie||(this.cookie=this.options.cookie.name||"ui-tabs-"+f());return a.cookie.apply(null,[b].concat(a.makeArray(arguments)))},_ui:function(a,b){return{tab:a,panel:b,index:this.anchors.index(a)}},_cleanup:function(){this.lis.filter(".ui-state-processing").removeClass("ui-state-processing").find("span:data(label.tabs)").each(function(){var b=a(this);b.html(b.data("label.tabs")).removeData("label.tabs")})},_tabify:function(c){function m(b,c){b.css("display",""),!a.support.opacity&&c.opacity&&b[0].style.removeAttribute("filter")}var d=this,e=this.options,f=/^#.+/;this.list=this.element.find("ol,ul").eq(0),this.lis=a(" > li:has(a[href])",this.list),this.anchors=this.lis.map(function(){return a("a",this)[0]}),this.panels=a([]),this.anchors.each(function(b,c){var g=a(c).attr("href"),h=g.split("#")[0],i;h&&(h===location.toString().split("#")[0]||(i=a("base")[0])&&h===i.href)&&(g=c.hash,c.href=g);if(f.test(g))d.panels=d.panels.add(d.element.find(d._sanitizeSelector(g)));else if(g&&g!=="#"){a.data(c,"href.tabs",g),a.data(c,"load.tabs",g.replace(/#.*$/,""));var j=d._tabId(c);c.href="#"+j;var k=d.element.find("#"+j);k.length||(k=a(e.panelTemplate).attr("id",j).addClass("ui-tabs-panel ui-widget-content ui-corner-bottom").insertAfter(d.panels[b-1]||d.list),k.data("destroy.tabs",!0)),d.panels=d.panels.add(k)}else e.disabled.push(b)}),c?(this.element.addClass("ui-tabs ui-widget ui-widget-content ui-corner-all"),this.list.addClass("ui-tabs-nav ui-helper-reset ui-helper-clearfix ui-widget-header ui-corner-all"),this.lis.addClass("ui-state-default ui-corner-top"),this.panels.addClass("ui-tabs-panel ui-widget-content ui-corner-bottom"),e.selected===b?(location.hash&&this.anchors.each(function(a,b){if(b.hash==location.hash)return e.selected=a,!1}),typeof e.selected!="number"&&e.cookie&&(e.selected=parseInt(d._cookie(),10)),typeof e.selected!="number"&&this.lis.filter(".ui-tabs-selected").length&&(e.selected=this.lis.index(this.lis.filter(".ui-tabs-selected"))),e.selected=e.selected||(this.lis.length?0:-1)):e.selected===null&&(e.selected=-1),e.selected=e.selected>=0&&this.anchors[e.selected]||e.selected<0?e.selected:0,e.disabled=a.unique(e.disabled.concat(a.map(this.lis.filter(".ui-state-disabled"),function(a,b){return d.lis.index(a)}))).sort(),a.inArray(e.selected,e.disabled)!=-1&&e.disabled.splice(a.inArray(e.selected,e.disabled),1),this.panels.addClass("ui-tabs-hide"),this.lis.removeClass("ui-tabs-selected ui-state-active"),e.selected>=0&&this.anchors.length&&(d.element.find(d._sanitizeSelector(d.anchors[e.selected].hash)).removeClass("ui-tabs-hide"),this.lis.eq(e.selected).addClass("ui-tabs-selected ui-state-active"),d.element.queue("tabs",function(){d._trigger("show",null,d._ui(d.anchors[e.selected],d.element.find(d._sanitizeSelector(d.anchors[e.selected].hash))[0]))}),this.load(e.selected)),a(window).bind("unload",function(){d.lis.add(d.anchors).unbind(".tabs"),d.lis=d.anchors=d.panels=null})):e.selected=this.lis.index(this.lis.filter(".ui-tabs-selected")),this.element[e.collapsible?"addClass":"removeClass"]("ui-tabs-collapsible"),e.cookie&&this._cookie(e.selected,e.cookie);for(var g=0,h;h=this.lis[g];g++)a(h)[a.inArray(g,e.disabled)!=-1&&!a(h).hasClass("ui-tabs-selected")?"addClass":"removeClass"]("ui-state-disabled");e.cache===!1&&this.anchors.removeData("cache.tabs"),this.lis.add(this.anchors).unbind(".tabs");if(e.event!=="mouseover"){var i=function(a,b){b.is(":not(.ui-state-disabled)")&&b.addClass("ui-state-"+a)},j=function(a,b){b.removeClass("ui-state-"+a)};this.lis.bind("mouseover.tabs",function(){i("hover",a(this))}),this.lis.bind("mouseout.tabs",function(){j("hover",a(this))}),this.anchors.bind("focus.tabs",function(){i("focus",a(this).closest("li"))}),this.anchors.bind("blur.tabs",function(){j("focus",a(this).closest("li"))})}var k,l;e.fx&&(a.isArray(e.fx)?(k=e.fx[0],l=e.fx[1]):k=l=e.fx);var n=l?function(b,c){a(b).closest("li").addClass("ui-tabs-selected ui-state-active"),c.hide().removeClass("ui-tabs-hide").animate(l,l.duration||"normal",function(){m(c,l),d._trigger("show",null,d._ui(b,c[0]))})}:function(b,c){a(b).closest("li").addClass("ui-tabs-selected ui-state-active"),c.removeClass("ui-tabs-hide"),d._trigger("show",null,d._ui(b,c[0]))},o=k?function(a,b){b.animate(k,k.duration||"normal",function(){d.lis.removeClass("ui-tabs-selected ui-state-active"),b.addClass("ui-tabs-hide"),m(b,k),d.element.dequeue("tabs")})}:function(a,b,c){d.lis.removeClass("ui-tabs-selected ui-state-active"),b.addClass("ui-tabs-hide"),d.element.dequeue("tabs")};this.anchors.bind(e.event+".tabs",function(){var b=this,c=a(b).closest("li"),f=d.panels.filter(":not(.ui-tabs-hide)"),g=d.element.find(d._sanitizeSelector(b.hash));if(c.hasClass("ui-tabs-selected")&&!e.collapsible||c.hasClass("ui-state-disabled")||c.hasClass("ui-state-processing")||d.panels.filter(":animated").length||d._trigger("select",null,d._ui(this,g[0]))===!1)return this.blur(),!1;e.selected=d.anchors.index(this),d.abort();if(e.collapsible){if(c.hasClass("ui-tabs-selected"))return e.selected=-1,e.cookie&&d._cookie(e.selected,e.cookie),d.element.queue("tabs",function(){o(b,f)}).dequeue("tabs"),this.blur(),!1;if(!f.length)return e.cookie&&d._cookie(e.selected,e.cookie),d.element.queue("tabs",function(){n(b,g)}),d.load(d.anchors.index(this)),this.blur(),!1}e.cookie&&d._cookie(e.selected,e.cookie);if(g.length)f.length&&d.element.queue("tabs",function(){o(b,f)}),d.element.queue("tabs",function(){n(b,g)}),d.load(d.anchors.index(this));else throw"jQuery UI Tabs: Mismatching fragment identifier.";a.browser.msie&&this.blur()}),this.anchors.bind("click.tabs",function(){return!1})},_getIndex:function(a){return typeof a=="string"&&(a=this.anchors.index(this.anchors.filter("[href$='"+a+"']"))),a},destroy:function(){var b=this.options;return this.abort(),this.element.unbind(".tabs").removeClass("ui-tabs ui-widget ui-widget-content ui-corner-all ui-tabs-collapsible").removeData("tabs"),this.list.removeClass("ui-tabs-nav ui-helper-reset ui-helper-clearfix ui-widget-header ui-corner-all"),this.anchors.each(function(){var b=a.data(this,"href.tabs");b&&(this.href=b);var c=a(this).unbind(".tabs");a.each(["href","load","cache"],function(a,b){c.removeData(b+".tabs")})}),this.lis.unbind(".tabs").add(this.panels).each(function(){a.data(this,"destroy.tabs")?a(this).remove():a(this).removeClass(["ui-state-default","ui-corner-top","ui-tabs-selected","ui-state-active","ui-state-hover","ui-state-focus","ui-state-disabled","ui-tabs-panel","ui-widget-content","ui-corner-bottom","ui-tabs-hide"].join(" "))}),b.cookie&&this._cookie(null,b.cookie),this},add:function(c,d,e){e===b&&(e=this.anchors.length);var f=this,g=this.options,h=a(g.tabTemplate.replace(/#\{href\}/g,c).replace(/#\{label\}/g,d)),i=c.indexOf("#")?this._tabId(a("a",h)[0]):c.replace("#","");h.addClass("ui-state-default ui-corner-top").data("destroy.tabs",!0);var j=f.element.find("#"+i);return j.length||(j=a(g.panelTemplate).attr("id",i).data("destroy.tabs",!0)),j.addClass("ui-tabs-panel ui-widget-content ui-corner-bottom ui-tabs-hide"),e>=this.lis.length?(h.appendTo(this.list),j.appendTo(this.list[0].parentNode)):(h.insertBefore(this.lis[e]),j.insertBefore(this.panels[e])),g.disabled=a.map(g.disabled,function(a,b){return a>=e?++a:a}),this._tabify(),this.anchors.length==1&&(g.selected=0,h.addClass("ui-tabs-selected ui-state-active"),j.removeClass("ui-tabs-hide"),this.element.queue("tabs",function(){f._trigger("show",null,f._ui(f.anchors[0],f.panels[0]))}),this.load(0)),this._trigger("add",null,this._ui(this.anchors[e],this.panels[e])),this},remove:function(b){b=this._getIndex(b);var c=this.options,d=this.lis.eq(b).remove(),e=this.panels.eq(b).remove();return d.hasClass("ui-tabs-selected")&&this.anchors.length>1&&this.select(b+(b+1<this.anchors.length?1:-1)),c.disabled=a.map(a.grep(c.disabled,function(a,c){return a!=b}),function(a,c){return a>=b?--a:a}),this._tabify(),this._trigger("remove",null,this._ui(d.find("a")[0],e[0])),this},enable:function(b){b=this._getIndex(b);var c=this.options;if(a.inArray(b,c.disabled)==-1)return;return this.lis.eq(b).removeClass("ui-state-disabled"),c.disabled=a.grep(c.disabled,function(a,c){return a!=b}),this._trigger("enable",null,this._ui(this.anchors[b],this.panels[b])),this},disable:function(a){a=this._getIndex(a);var b=this,c=this.options;return a!=c.selected&&(this.lis.eq(a).addClass("ui-state-disabled"),c.disabled.push(a),c.disabled.sort(),this._trigger("disable",null,this._ui(this.anchors[a],this.panels[a]))),this},select:function(a){a=this._getIndex(a);if(a==-1)if(this.options.collapsible&&this.options.selected!=-1)a=this.options.selected;else return this;return this.anchors.eq(a).trigger(this.options.event+".tabs"),this},load:function(b){b=this._getIndex(b);var c=this,d=this.options,e=this.anchors.eq(b)[0],f=a.data(e,"load.tabs");this.abort();if(!f||this.element.queue("tabs").length!==0&&a.data(e,"cache.tabs")){this.element.dequeue("tabs");return}this.lis.eq(b).addClass("ui-state-processing");if(d.spinner){var g=a("span",e);g.data("label.tabs",g.html()).html(d.spinner)}return this.xhr=a.ajax(a.extend({},d.ajaxOptions,{url:f,success:function(f,g){c.element.find(c._sanitizeSelector(e.hash)).html(f),c._cleanup(),d.cache&&a.data(e,"cache.tabs",!0),c._trigger("load",null,c._ui(c.anchors[b],c.panels[b]));try{d.ajaxOptions.success(f,g)}catch(h){}},error:function(a,f,g){c._cleanup(),c._trigger("load",null,c._ui(c.anchors[b],c.panels[b]));try{d.ajaxOptions.error(a,f,b,e)}catch(g){}}})),c.element.dequeue("tabs"),this},abort:function(){return this.element.queue([]),this.panels.stop(!1,!0),this.element.queue("tabs",this.element.queue("tabs").splice(-2,2)),this.xhr&&(this.xhr.abort(),delete this.xhr),this._cleanup(),this},url:function(a,b){return this.anchors.eq(a).removeData("cache.tabs").data("load.tabs",b),this},length:function(){return this.anchors.length}}),a.extend(a.ui.tabs,{version:"1.8.23"}),a.extend(a.ui.tabs.prototype,{rotation:null,rotate:function(a,b){var c=this,d=this.options,e=c._rotate||(c._rotate=function(b){clearTimeout(c.rotation),c.rotation=setTimeout(function(){var a=d.selected;c.select(++a<c.anchors.length?a:0)},a),b&&b.stopPropagation()}),f=c._unrotate||(c._unrotate=b?function(a){e()}:function(a){a.clientX&&c.rotate(null)});return a?(this.element.bind("tabsshow",e),this.anchors.bind(d.event+".tabs",f),e()):(clearTimeout(c.rotation),this.element.unbind("tabsshow",e),this.anchors.unbind(d.event+".tabs",f),delete this._rotate,delete this._unrotate),this}})})(jQuery);;