/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.filter.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.filter.core.valueobject.BentenImportHtmlProcessInput;

/**
 * Apache Antタスク [BentenImportHtml]のクラス。
 *
 * HTML インポートのための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.filter.core]にBentenImportHtmlBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenimporthtml&quot; classname=&quot;benten.twa.filter.core.BentenImportHtmlTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenImportHtmlTask extends Task {
    /**
     * HTML インポートのための Ant タスク。
     */
    protected BentenImportHtmlProcessInput fInput = new BentenImportHtmlProcessInput();

    /**
     * フィールド [sourcedir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcedirProcessed = false;

    /**
     * フィールド [targetdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetdirProcessed = false;

    /**
     * フィールド [normalizehtml] に値がセットされたかどうか。
     */
    protected boolean fIsFieldNormalizehtmlProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * インポート元となる HTML が格納されたディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcedir(final String arg) {
        fInput.setSourcedir(arg);
        fIsFieldSourcedirProcessed = true;
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * インポート元となる HTML が格納されたディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcedir() {
        return fInput.getSourcedir();
    }

    /**
     * Antタスクの[targetdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * インポート先となる HTML を格納するディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetdir(final String arg) {
        fInput.setTargetdir(arg);
        fIsFieldTargetdirProcessed = true;
    }

    /**
     * Antタスクの[targetdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * インポート先となる HTML を格納するディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetdir() {
        return fInput.getTargetdir();
    }

    /**
     * Antタスクの[normalizehtml]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * インポートの際に html を正常化するかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setNormalizehtml(final boolean arg) {
        fInput.setNormalizehtml(arg);
        fIsFieldNormalizehtmlProcessed = true;
    }

    /**
     * Antタスクの[normalizehtml]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * インポートの際に html を正常化するかどうか。<br>
     * デフォルト値[true]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getNormalizehtml() {
        return fInput.getNormalizehtml();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenImportHtmlTask begin.");

        // 項目番号[1]、アトリビュート[sourcedir]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcedirProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcedir]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[targetdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetdirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetdir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- sourcedir:[" + getSourcedir() + "]");
            System.out.println("- targetdir:[" + getTargetdir() + "]");
            System.out.println("- normalizehtml:[" + getNormalizehtml() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenImportHtmlProcessインタフェースを実装して benten.twa.filter.coreパッケージに BentenImportHtmlProcessImplクラスを作成することにより解決できる場合があります。
            final BentenImportHtmlProcess proc = new BentenImportHtmlProcessImpl();
            if (proc.execute(fInput) != BentenImportHtmlBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
