/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.properties;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Properties;

import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterEngine;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * Java プロパティファイルを処理します。
 * 
 * @author IGA Tosiki
 */
public class BentenTwaFilterJavaPropertyResourceBundleEngine implements BentenTwaFilterEngine {
	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".properties")) { //$NON-NLS-1$
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliff(final File sourceFile, final BlancoXliff xliff, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffFile xliffFile = xliff.getFileList().get(0);

		// Java プロパティファイルを表現します。
		xliffFile.setDatatype("javapropertyresourcebundle"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		final BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outStream));

		try {
			new AbstractJavaPropertiesFileProcessor() {
				@Override
				boolean replaceValue(final Properties props, final String key, final Writer writer) throws IOException {
					final String value = (String) props.get(key);
					resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);
					createTransUnit(xliffFile.getBody(), id, key, value, resultInfo);

					return false;
				}
			}.process(bytesFile, writer);
		} finally {
			writer.close();
		}
	}

	/**
	 * trans-unit の作成。
	 * 
	 * <UL>
	 * <LI>このメソッドは プロパティファイル -> XLIFF で利用されます。
	 * </UL>
	 * 
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	private void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String key,
			final String source, BentenProcessResultInfo resultInfo) {
		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toStringWithoutSeq() + "?key=" + key); //$NON-NLS-1$
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliff(final File sourceFile, final BlancoXliff xliff, final BentenProcessResultInfo resultInfo)
			throws IOException {

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		final BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outStream));

		try {
			new AbstractJavaPropertiesFileProcessor() {
				@Override
				boolean replaceValue(final Properties props, final String key, final Writer writer) throws IOException {
					resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

					for (BlancoXliffTransUnit transunit : xliff.getFileList().get(0).getBody().getTransUnitList()) {
						final int index = transunit.getId().indexOf("?key="); //$NON-NLS-1$
						if (index > 0) {
							final String propKey = transunit.getId().substring(index + "?key=".length()); //$NON-NLS-1$
							if (propKey.equals(key)) {
								if (transunit.getTranslate() == false) {
									return false;
								}
								if (transunit.getTarget() == null) {
									return false;
								}
								if (transunit.getTarget().getTarget() == null) {
									return false;
								}

								writer.write(getEscapedPropertiesValue(transunit.getTarget().getTarget()));
								return true;
							}
						}
					}
					return false;
				}
			}.process(bytesFile, writer);
		} finally {
			writer.close();
		}

		return outStream.toByteArray();
	}
}
