/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;

import benten.cat.tm.core.BentenTmDriverManager;
import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.ui.CatTmUiPlugin;
import benten.core.BentenConstants;
import benten.core.text.Strings;
import benten.twa.cat.core.valueobject.BentenApplyExistentTranslationProcessInput;
import benten.twa.cat.messages.BentenApplyExistentTranslationMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.tmx.BlancoTmxSerializer;
import blanco.tmx.valueobject.BlancoTmx;
import blanco.tmx.valueobject.BlancoTmxTu;
import blanco.tmx.valueobject.BlancoTmxTuv;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.BlancoXliffSerializer;
import blanco.xliff.BlancoXliffUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 過去訳の適用
 *
 * <pre>
 * このウィザードは、過去訳を XLIFF に適用します。
 *    1.  過去の翻訳で利用した XLIFF を利用して過去訳を適用します。
 *    2.  翻訳後の XLIFF にある過去訳を、XLIFF に適用します。
 *    3.  この処理では TMX を利用しません。
 * </pre>
 * 
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式機械支援翻訳機能: 過去訳適用機能」に対応します。
 * 
 * @author IGA Tosiki
 */
public class BentenApplyExistentTranslationProcessImpl extends AbstractTraverseDir implements
		BentenApplyExistentTranslationProcess {
	/**
	 * 過去訳適用機能のためのメッセージ。
	 */
	protected static final BentenApplyExistentTranslationMessages fMsg = new BentenApplyExistentTranslationMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenApplyExistentTranslationProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * この翻訳で利用する翻訳メモリー。
	 */
	protected BentenTmEngine fTm;

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenApplyExistentTranslationProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 * 
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenApplyExistentTranslationProcessInput input) throws IOException,
			IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException(
					"BentenApplyExistentTranslationProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirSource = new File(fInput.getSourcedir());
		final File dirTarget = new File(fInput.getTargetdir());
		final File dirTmp = new File(fInput.getTmpdir());
		if (dirSource.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getSourcedir()));
		}
		if (dirSource.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getSourcedir()));
		}
		if (dirTarget.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getTargetdir()));
		}
		if (dirTarget.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getTargetdir()));
		}
		if (dirTmp.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE008(fInput.getTargetdir()));
		}
		if (dirTmp.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE009(fInput.getTargetdir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenApplyExistentTranslationProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) {
				fCounter += 2;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(fInput);
		inner.processDir(new File(input.getSourcedir()));
		beginTask(inner.getCounter() + 5);

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		if (fInput.getTmdriverclassname() == null || fInput.getTmdriverclassname().trim().length() == 0) {
			// プラグインの場合のコース。
			fTm = CatTmUiPlugin.getDefault().newTmEnginePluginInstance().getDriver().getEngineInstance();
		} else {
			// Ant タスクの場合のコース。
			try {
				Class.forName(fInput.getTmdriverclassname());
			} catch (ClassNotFoundException e) {
				throw new IOException(fMsg.getCoreE001(fInput.getTmdriverclassname(), e.toString()));
			}
			if (BentenTmDriverManager.getDrivers().length == 0) {
				throw new IllegalArgumentException(fMsg.getCoreE002());
			}
			fTm = BentenTmDriverManager.getDrivers()[0].getEngineInstance();
		}
		if (fTm == null) {
			throw new IOException(fMsg.getCoreE003());
		}

		if (progress(fMsg.getCoreP003())) {
			return 6;
		}

		// 翻訳元、翻訳先の言語を設定。
		// TODO 現時点では過去訳適用のみにこのコードを実装しています。
		// TODO 本来は、類似訳適用、完全一致訳適用においても必要である可能性が高いです。
		// TODO 影響は Ant タスク実行時にあらわれるものと考えられます。GUI には影響ありません。
		fTm.setLang(fInput.getTranssourcelang(), fInput.getTranstargetlang());

		final BlancoTmx tmx = buildTmx(new File(fInput.getSourcedir()));
		loadTmTmx(tmx);

		if (progress(fMsg.getCoreP004())) {
			return 6;
		}

		processDir(new File(input.getTargetdir()));

		if (progress(fMsg.getCoreP005(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * 処理の内部処理でアイテムが処理されるたびに進捗報告としてコールバック。
	 *
	 * @param argProgressMessage 現在処理しているアイテムに関するメッセージ。
	 * @return 処理をそのまま継続する場合は false。処理中断をリクエストしたい場合は true。
	 */
	public boolean progressInner(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	protected void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenApplyExistentTranslationProcessImpl#processFile: 'fInput' is null."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		BlancoXliff xliffTarget = null;
		try {
			xliffTarget = new BlancoXliffParser().parse(file);
		} catch (IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE018(file.getName(), e.toString()));
		}

		int transUnitTotalCount = 0;
		for (final BlancoXliffFile xliffFile : xliffTarget.getFileList()) {
			transUnitTotalCount += xliffFile.getBody().getTransUnitList().size();
		}

		int transUnitCount = 0;
		for (BlancoXliffFile targetFile : xliffTarget.getFileList()) {
			for (BlancoXliffTransUnit transUnit : targetFile.getBody().getTransUnitList()) {
				if (progressInner(fMsg.getCoreP101(file.getName(), BigDecimal.valueOf(++transUnitCount), BigDecimal
						.valueOf(transUnitTotalCount)))) {
					break;
				}

				if (transUnit.getTranslate() == false) {
					continue;
				}
				if (transUnit.getSource() == null || transUnit.getSource().length() == 0) {
					continue;
				}
				if (transUnit.getTarget() != null && transUnit.getTarget().getTarget() != null) {
					// target の文字列長さは 0 でも良しとします。 
					// これは、"何か文字" → "" と変換したいニーズが存在するであろうと事を想定しています。

					// ここに入ってきた場合には、target は存在しています。

					if (transUnit.getTarget().getState() != null
							&& transUnit.getTarget().getState().trim().length() > 0) {
						// state が設定されています。これは既に翻訳確定済みと判断します。
					} else {
						if (transUnit.getSource().equals(transUnit.getTarget().getTarget())) {
							// source と target とが一致しています。

							// state が無指定または空で、しかも source と target とが一致している場合にのみ、訳を上書きします。
							processTransUnit(transUnit);
							continue;
						}
					}

					// target がすでに存在する場合、訳を上書きしてしまうことを避けるために、
					// ここで中断します。state のレベルによる詳細な判断は実施しません。
					continue;
				}

				// trans-unit を処理します。
				processTransUnit(transUnit);
				continue;
			}
		}

		// 書き出し。
		try {
			final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			new BlancoXliffSerializer().serialize(xliffTarget, outStream);
			outStream.flush();

			if (BlancoFileUtil.bytes2FileIfNecessary(outStream.toByteArray(), file) == 0) {
				// 変更無し。
				getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
			} else {
				// 更新のあったファイル数をカウントします。
				getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			}
		} catch (IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE019(file.getName(), e.toString()));
		}

		if (progress(fMsg.getCoreP013(file.getName()))) {
			return;
		}
	}

	/**
	 * XLIFF ディレクトリーを入力してTMX を構築。
	 * @return TMX オブジェクト。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	private BlancoTmx buildTmx(final File dirInput) throws IOException {
		final BlancoTmx tmx = new BlancoTmx();
		class MyDir extends AbstractTraverseDir {
			public void execute(final File dirInput) throws IOException {
				processDir(dirInput);
			}

			@Override
			protected boolean canProcess(final File file) {
				return file.getName().toLowerCase().endsWith(".xlf"); //$NON-NLS-1$
			}

			@Override
			protected void processFile(final File file, final String baseDir) throws IOException {
				BlancoXliff xliff = null;
				try {
					xliff = new BlancoXliffParser().parse(file);
				} catch (IllegalArgumentException e) {
					throw new IOException(fMsg.getCoreE011(file.getName(), e.toString()));
				}

				for (BlancoXliffFile xliffFile : xliff.getFileList()) {
					for (BlancoXliffTransUnit transUnit : xliffFile.getBody().getTransUnitList()) {
						if (transUnit.getTranslate() == false) {
							continue;
						}
						if (transUnit.getTarget() == null || transUnit.getTarget().getTarget() == null) {
							continue;
						}
						if (transUnit.getSource() == null || transUnit.getSource().length() == 0) {
							continue;
						}

						// TMX反映除外のチェック。
						if (BlancoXliffUtil.isXTmOmit(transUnit)) {
							continue;
						}

						// state のチェックは実施しません。

						final BlancoTmxTu tu = new BlancoTmxTu();
						tmx.getBody().getTuList().add(tu);
						{
							final BlancoTmxTuv tuv = new BlancoTmxTuv();
							tu.getTuvList().add(tuv);
							tuv.setLang(fInput.getTranssourcelang());
							tuv.setSeg(transUnit.getSource());
						}
						{
							final BlancoTmxTuv tuv = new BlancoTmxTuv();
							tu.getTuvList().add(tuv);
							tuv.setLang(fInput.getTranstargetlang());
							tuv.setSeg(transUnit.getTarget().getTarget());
						}
					}
				}
			}
		}
		new MyDir().execute(dirInput);
		return tmx;
	}

	/**
	 * TMX を一旦ファイル化したうえで TM にロード。
	 * @throws IOException
	 */
	private void loadTmTmx(final BlancoTmx tmx) throws IOException {
		boolean isTmpSubDirCreated = false;

		final File subTmpDir = new File(fInput.getTmpdir() + "/tmx"); //$NON-NLS-1$
		if (subTmpDir.exists() == false) {
			if (subTmpDir.mkdirs() == false) {
				throw new IOException(fMsg.getCoreE012(fInput.getTmpdir(), subTmpDir.getName()));
			}
			isTmpSubDirCreated = true;
		} else {
			if (subTmpDir.isDirectory() == false) {
				throw new IOException(fMsg.getCoreE013(fInput.getTmpdir(), subTmpDir.getName()));
			}
		}

		final File tmpFile = new File(subTmpDir, "work.tmx"); //$NON-NLS-1$

		try {
			new BlancoTmxSerializer().serialize(tmx, tmpFile);
		} catch (IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE014(tmpFile.getName(), e.toString()));
		}

		try {
			fTm.loadTmx(subTmpDir);
		} catch (IOException e) {
			throw new IOException(fMsg.getCoreE015(e.toString()));
		}

		if (tmpFile.delete() == false) {
			throw new IOException(fMsg.getCoreE016(tmpFile.getName()));
		}

		if (isTmpSubDirCreated) {
			if (subTmpDir.delete() == false) {
				throw new IOException(fMsg.getCoreE017(subTmpDir.getAbsolutePath()));
			}
		}
	}

	/**
	 * trans-unit の完全一致訳を処理。
	 *
	 * <UL>
	 * <LI>設定にかかわらず、最初に指定の文字で完全一致訳の検索を実施します。
	 * <LI>検索で見つからなかった場合に、ホワイトスペース無視の指定がある場合にはホワイトスペースを無視した完全一致訳の検索を実施します。
	 * <LI>ホワイトスペース無視の場合の 2 パス検索は、完全一致訳導出のための特殊な動きです。類似訳の場合には考慮しません。
	 * </UL>
	 *
	 * 前提条件
	 * <UL>
	 * <LI>execute または setInput が事前に呼び出されていること。
	 * </UL>
	 *
	 * @param transUnit 翻訳単位。
	 */
	protected void processTransUnit(final BlancoXliffTransUnit transUnit) {
		if (processTransUnitInternal(transUnit, false, false)) {
			return;
		}

		if (fInput.getIgnorewhitespacetmreference()) {
			// ホワイトスペース無視の設定がある場合には、ホワイトスペース無視で再度検索を実施します。
			processTransUnitInternal(transUnit, true, false);
		}

		if (fInput.getIgnoremnemonickeytmreference()) {
			// ニーモニック・キー無視の設定がある場合には、ニーモニック・キー無視で再度検索を実施します。
			processTransUnitInternal(transUnit, false, true);
		}
	}

	/**
	 * trans-unit の完全一致訳の内部処理。
	 *
	 * <UL>
	 * <LI>fuzzySearch に与える文字列をプリファランスの設定に従って加工するかどうかの判断は引数によって与えられます。
	 * </UL>
	 *
	 * @param transUnit 翻訳単位。
	 * @param isIgnorewhitespacetmreference ホワイトスペースを無視するかどうか。
	 * @param isIgnoreMnemonicKeyTmReference ニーモニック・キーを無視するかどうか。
	 * @return 処理したかどうか。
	 */
	protected boolean processTransUnitInternal(final BlancoXliffTransUnit transUnit,
			final boolean isIgnorewhitespacetmreference, final boolean isIgnoreMnemonicKeyTmReference) {
		String source = transUnit.getSource();
		if (isIgnorewhitespacetmreference) {
			source = Strings.removeRedundantWhitespace(source);
		}

		if (isIgnoreMnemonicKeyTmReference) {
			source = Strings.stripMnemonicKey(source);
		}

		final String targetString = BentenTmUtil.searchExactMatch(fTm, source);
		if (targetString == null) {
			// 完全一致訳はありませんでした。
			return false;
		}

		// 完全一致訳が見つかりました。

		fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

		final BlancoXliffTarget target = new BlancoXliffTarget();
		transUnit.setTarget(target);

		// 完全一致訳をセットします。
		target.setTarget(targetString);

		if (isIgnoreMnemonicKeyTmReference) {
			// ニーモニック・キー部分を追加。
			final String strMnemonicKey = transUnit.getSource().substring(source.length());
			target.setTarget(target.getTarget() + strMnemonicKey);
		}

		// 完全一致を target にセットする場合には、state に translated をセットとします。
		target.setState("translated"); //$NON-NLS-1$
		return true;
	}
}
