/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.tmx.handler;

import java.io.IOException;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.xml.sax.SAXException;

import benten.twa.tmx.core.BentenExtractTmxFragmentBatchProcess;
import benten.twa.tmx.core.BentenExtractTmxFragmentProcessImpl;
import benten.twa.tmx.core.valueobject.BentenExtractTmxFragmentProcessInput;
import benten.twa.tmx.dialogs.ExtractTmxFragmentDialog;
import benten.twa.tmx.messages.BentenExtractTmxFragmentMessages;
import benten.twa.tmx.status.BentenExtractTmxFragmentStatus;
import benten.twa.ui.jobs.AbstractWorkflowHandler;
import benten.ui.preference.BentenPreference;
import benten.ui.preference.BentenPreference.Preference;

/**
 * TMX フラグメント抽出ハンドラーです。
 *
 * TMX フラグメントの抽出
 *
 * <pre>
 * XLIFF フラグメントから、TMX フラグメントを抽出します。
 *   1.  XLIFF から翻訳単位を抽出して TMX を作成します。
 * </pre>
 * 
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳メモリー断片抽出・マージ機能: TMXフラグメント抽出機能」に対応します。
 * 
 * @author IGA Tosiki
 */
public class ExtractTmxFragmentHandler extends AbstractWorkflowHandler {
	/**
	 * TMXフラグメント抽出機能のためのメッセージ。
	 */
	protected static final BentenExtractTmxFragmentMessages fMsg = new BentenExtractTmxFragmentMessages();

	@Override
	protected StatusDialog createDialog(final Shell shell, final IStructuredSelection selection) {
		setJobMessage(fMsg.getJobName());
		return new ExtractTmxFragmentDialog(shell, selection);
	}

	@Override
	protected void processFile(final IStatus iStatus) throws IOException, SAXException {
		final BentenExtractTmxFragmentStatus status = (BentenExtractTmxFragmentStatus) iStatus;

		final BentenExtractTmxFragmentProcessInput input = new BentenExtractTmxFragmentProcessInput();
		input.setXliffdir(status.getTargetDir().getAbsolutePath());
		input.setTmxdir(status.getToDir().getAbsolutePath());
		input.setExtractbystate(status.getState());
		input.setIncludedate(status.getIncludeDate());
		input.setTranstargetid(status.getTransTargetId());
		input.setTranssourcelang(status.getTransSourceLang());

		final IPreferenceStore store = BentenPreference.getStore();
		// ホワイトスペースを無視した内容で TMX を抽出するかどうかのフラグ。
		input.setIgnorewhitespacetmxextract(store.getBoolean(Preference.IGNORE_WHITESPACE_TMX_EXTRACT.name()));
		// ニーモニック・キーを無視した内容で TMX を抽出するかどうかのフラグ。
		input.setIgnoremnemonickeytmxextract(store.getBoolean(Preference.IGNORE_MNEMONIC_KEY_TMX_EXTRACT.name()));

		new BentenExtractTmxFragmentBatchProcess().validateInput(input);
		final BentenExtractTmxFragmentProcessImpl process = new BentenExtractTmxFragmentProcessImpl() {
			@Override
			protected void beginTask(final int totalWork) {
				monitor.beginTask(null, totalWork);
			}

			@Override
			public boolean progress(final String argProgressMessage) {
				monitor.subTask(argProgressMessage);
				monitor.worked(1);
				Thread.yield();
				return monitor.isCanceled();
			}
		};
		process.execute(input);
		resultInfo = process.getResultInfo();
	}
}
