/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.ui.wizards;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;

import benten.twa.ui.wizards.messages.NewProjectCreationWizardMessages;
import benten.ui.views.UiToolkit;

/**
 * 翻訳プロジェクト作成ウィザード・ページ。
 * 
 * @author IGA Tosiki
 */
public class NewProjectCreationWizardPage extends WizardNewProjectCreationPage implements IWizardPage {
	/**
	 * 翻訳プロジェクト作成ウィザードのためのメッセージ。
	 */
	protected static final NewProjectCreationWizardMessages fMsg = new NewProjectCreationWizardMessages();

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	private final UiToolkit toolkit = new UiToolkit();

	/** 翻訳元の言語・コンボボックス */
	private Combo transSourceLangCombo;

	/** 翻訳先の言語・コンボボックス */
	private Combo transTargetLangCombo;

	/**
	 * コンストラクター。
	 */
	protected NewProjectCreationWizardPage() {
		super(NewProjectCreationWizardPage.class.getName());
		setTitle(fMsg.getWizardPageTitle());
		setDescription(fMsg.getWizardPageDescription());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void setControl(final Control newControl) {
		super.setControl(newControl);

		final Composite composite = (Composite) newControl;

		final Group transGroup = new Group(composite, SWT.BOLD);
		transGroup.setLayout(toolkit.createGridLayout(3, 5));
		transGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		transGroup.setText(fMsg.getLabelGroupTransLang());

		{
			toolkit.createLabel(transGroup, fMsg.getLabelSourceLang());

			transSourceLangCombo = new Combo(transGroup, SWT.READ_ONLY);
			for (final String locale : getLocaleList()) {
				transSourceLangCombo.add(locale);
			}
			transSourceLangCombo.setText(fMsg.getDefaultTransSourceLang());
			transSourceLangCombo.addModifyListener(new ModifyListener() {
				public void modifyText(final ModifyEvent e) {
					validateInput();
				}
			});

			final Button button = toolkit.createButton(transGroup, fMsg.getButtonReverse(), SWT.DEFAULT);
			button.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(final SelectionEvent e) {
					final String origSource = transSourceLangCombo.getText();
					transSourceLangCombo.setText(transTargetLangCombo.getText());
					transTargetLangCombo.setText(origSource);
				}
			});
		}

		{
			toolkit.createLabel(transGroup, fMsg.getLabelTargetLang());

			transTargetLangCombo = new Combo(transGroup, SWT.READ_ONLY);
			for (final String locale : getLocaleList()) {
				transTargetLangCombo.add(locale);
			}
			transTargetLangCombo.setText(fMsg.getDefaultTransTargetLang());
			transTargetLangCombo.addModifyListener(new ModifyListener() {
				public void modifyText(final ModifyEvent e) {
					validateInput();
				}
			});
		}
	}

	/**
	 * 入力内容の検証。
	 * @return 正常かどうか。
	 */
	boolean validateInput() {
		if (transSourceLangCombo.getText().equals(transTargetLangCombo.getText())) {
			setErrorMessage(fMsg.getDlgError0());
			return false;
		}
		setErrorMessage(null);
		return true;
	}

	/**
	 * ロケール一覧を取得します。
	 * 
	 * @return ロケールの一覧。
	 */
	List<String> getLocaleList() {
		final List<String> localeList = new LinkedList<String>();

		// 一覧を取得。
		final Locale[] availableLocales = Locale.getAvailableLocales();
		for (final Locale loc : availableLocales) {
			final String language = loc.getLanguage();
			final String country = loc.getCountry();
			if (country == null || country.trim().length() == 0) {
				// 国が指定されていないものはパス。
				continue;
			}

			final String localeString = language + "-" + country; //$NON-NLS-1$

			boolean isAlreadyExist = false;
			for (final String look : localeList) {
				if (localeString.equals(look)) {
					isAlreadyExist = true;
					break;
				}
			}
			if (isAlreadyExist) {
				// すでに登録されているものはパス。
				continue;
			}

			localeList.add(localeString);
		}

		// 名称順でソート。
		Collections.sort(localeList);

		return localeList;
	}

	/**
	 * 翻訳元の言語を取得
	 * @return 翻訳元の言語。
	 */
	String getTransSourceLang() {
		if (transSourceLangCombo == null) {
			return fMsg.getDefaultTransSourceLang();
		}
		return transSourceLangCombo.getText();
	}

	/**
	 * 翻訳先の言語を取得
	 * @return 翻訳先の言語。
	 */
	String getTransTargetLang() {
		if (transTargetLangCombo == null) {
			return fMsg.getDefaultTransTargetLang();
		}
		return transTargetLangCombo.getText();
	}
}
