/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.text;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 文字列ユーティリティー。
 *
 * @author KASHIHARA Shinji
 */
public class Strings {

	/**
	 * 日本風ニーモニック・キーをあらわす正規表現。
	 * <UL>
	 * <LI>対応するキー表現は、& _ ~ の 3 種類。
	 * <LI>(F) のように、無いものも処理対象。
	 * </UL>
	 */
	private static final String MNEMONIC_KEY_PATTERN = "\\((&|_|~)?.\\)\\s?$"; //$NON-NLS-1$

	/**
	 * 隠されたコンストラクター。
	 *
	 * <UL>
	 * <LI>このクラスはユーティリティ・クラスです。
	 * <LI>インスタンス生成を抑制するために、コンストラクターは private で修飾されています。
	 * </UL>
	 */
	private Strings() {
	}

	/**
	 * 連続する余分なホワイトスペースを除去。
	 *
	 * <UL>
	 * <LI>連続する空白(ホワイトスペース)を 1 つに集約します。
	 * <LI>HTLM 4.01 の「White space」と関連のある処理です。
	 * <LI>ただし、「&amp;nbsp;」 は処理しません。
	 * </UL>
	 *
	 * @param s 文字列
	 * @return 処理後の文字列
	 */
	public static String removeRedundantWhitespace(String s) {
		if (s == null) {
			return s;
		}
		s = s.trim();
		s = s.replaceAll("\\s+", " "); //$NON-NLS-1$ //$NON-NLS-2$
		return s;
	}

	/**
	 * ホワイトスペースで分割。
	 *
	 * <UL>
	 * <LI>「&amp;nbsp;」 は処理しません。
	 * </UL>
	 *
	 * @param s 文字列
	 * @return 分割後の文字列配列
	 */
	public static String[] splitByWhitespace(final String s) {
		if (s == null) {
			return new String[] {};
		}
		return s.trim().split("\\s+"); //$NON-NLS-1$
	}

	/**
	 * 与えられた文字列からニーモニック・キーを除去します。
	 *
	 * <UL>
	 * <LI>日本風ニーモニック・キーにのみ対応します。
	 * </UL>
	 *
	 * @param input ニーモニック付きの文字列。
	 * @return ニーモニックを除去した文字列。
	 */
	public static String stripMnemonicKey(final String input) {
		// 日本風ニーモニック・キーを検索するためのパターン。
		final Pattern mnemonicKeyPattern = Pattern.compile(MNEMONIC_KEY_PATTERN);

		final Matcher matcher = mnemonicKeyPattern.matcher(input);
		if (matcher.find()) {
			final String mnemonic = matcher.group(0);
			return input.substring(0, input.length() - mnemonic.length());
		} else {
			return input;
		}
	}
}
