/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 日本語、英語の翻訳文字列を表すクラスです。
 * 冗長な翻訳エントリーに対応するために、トリム・復元、
 * 句点解析によるエントリーの分割を行うことができます。
 *
 * @author KASHIHARA Shinji
 */
public class JapaneseTranslationString {

	/** トリムする空白文字の配列 */
	private static final char[] spaceChars = { ' ', '\t', '\r', '\n' };

	// クラス初期化
	static {
		// バイナリー・サーチのためにトリムする空白文字の配列をソート
		Arrays.sort(spaceChars);
	}

	/** 元の文字列 */
	private final String original;

	/** 先頭部 */
	protected StringBuilder starts;

	/** 末尾部 */
	protected StringBuilder ends;

	/** 前後空白を除いた文字列 */
	private String body;

	/**
	 * 翻訳項目を構築します。
	 * @param value 文字列
	 */
	public JapaneseTranslationString(final String value) {

		original = value;
		init();
	}

	/**
	 * 初期化します。
	 * trim 時に退避された文字列はクリアされます。
	 */
	private void init() {

		starts = new StringBuilder();
		ends = new StringBuilder();
		body = null;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 内容は元の文字列であることが保証されています。
	 * <p>
	 * @return このオブジェクトの文字列表現
	 */
	@Override
	public String toString() {
		return original;
	}

	/**
	 * 翻訳用にトリムします。
	 * <p>
	 * 翻訳用トリムとは翻訳に影響を与えない前後文字列を除去することを指します。
	 * 除去された文字列は revert メソッドで復元することができます。
	 * トリムの対象を以下に示します。
	 * <pre>
	 * ・前後の \n、\r、\t、半角空白、&nbsp;
	 * <pre>
	 * <p>
	 * @return 文字列
	 */
	public String trim() {

		if (body != null) {
			return body;
		}

		// 前後空白除去
		trimSpace(original);

		// 特定文字の除去
		trimSpecific("&nbsp;"); //$NON-NLS-1$

		return body;
	}

	/**
	 * 空白を除去します。
	 * <p>
	 * @param str 対象文字列
	 */
	private void trimSpace(final String str) {

		// パフォーマンスを考慮し可能な限り正規表現は使用しない

		final char[] cs = str.toCharArray();

		// 先頭部トリム
		for (int i = 0; i < cs.length; i++) {
			final char c = cs[i];
			if (Arrays.binarySearch(spaceChars, c) >= 0) {
				starts.append(c);
			} else {
				break;
			}
		}

		// 後方部トリム
		for (int i = cs.length - 1; i > 0; i--) {
			final char c = cs[i];
			if (Arrays.binarySearch(spaceChars, c) >= 0) {
				ends.insert(0, c);
			} else {
				break;
			}
		}

		// トリム後の文字列
		if (starts.length() == original.length() || ends.length() == original.length()) {
			body = ""; //$NON-NLS-1$
			return;
		}
		body = original.substring(starts.length(), original.length() - ends.length());
	}

	/**
	 * 囲み文字列を除去します。<br>
	 * trimSurround(s, s); と同じです。
	 * <p>
	 * @param str 囲み文字列
	 */
	private void trimSurround(final String str) {
		trimSurround(str, str);
	}

	/**
	 * 指定された囲み文字列を除去します。
	 * 囲み文字列の外側に句読点 。や . がある場合はそれも対象となり、
	 * 除去された文字列は revert のためにこのオブジェクト内に保持されます。
	 * その時、. は 。に変換されます。
	 * <p>
	 * @param begin 開始文字列
	 * @param end 終了文字列
	 */
	private void trimSurround(final String begin, String end) {

		if (!body.startsWith(begin)) {
			return;
		}
		if (countMatches(body, begin) != (begin.equals(end) ? 2 : 1)) {
			return;
		}
		if (body.endsWith(".") || body.endsWith("。")) { //$NON-NLS-1$ //$NON-NLS-2$
			end += body.substring(body.length() - 1);
		}
		if (!body.endsWith(end)) {
			return;
		}

		final int sLen = begin.length();
		final int eLen = end.length();
		final int bLen = body.length();

		starts.append(body.substring(0, sLen));
		ends.insert(0, body.substring(bLen - eLen).replace('.', '。'));
		body = body.substring(sLen, bLen - eLen);

		trimSpace(body);
	}

	/**
	 * 前後の連続する特定の文字をトリムします。
	 * @param strArray トリムする文字列の配列
	 */
	private void trimSpecific(final String... strArray) {

		trimStarts(strArray);
		trimEnds(strArray);
	}

	/**
	 * 先頭の連続する特定の文字をトリムします。
	 * @param strArray 文字列の配列
	 */
	private void trimStarts(final String... strArray) {

		while (true) {
			final int len = body.length();
			for (final String s : strArray) {
				if (body.startsWith(s)) {
					starts.append(s);
					body = body.substring(s.length());
					trimSpace(body);
				}
			}
			if (body.length() == len || body.length() == 0) {
				break;
			}
		}
	}

	/**
	 * 末尾の連続する特定の文字をトリムします。
	 * @param strArray 文字列の配列
	 */
	private void trimEnds(final String... strArray) {

		while (true) {
			final int len = body.length();
			for (final String s : strArray) {
				if (body.endsWith(s)) {
					ends.insert(0, s);
					body = body.substring(0, body.length() - s.length());
					trimSpace(body);
				}
			}
			if (body.length() == len || body.length() == 0) {
				break;
			}
		}
	}

	/**
	 * 前後の連続する文字 + 空白を除去します。
	 */
	protected void trimHyphen() {

		if (trimStartsWithSpace("-")) { //$NON-NLS-1$
			trimEndsWithSpace("-"); //$NON-NLS-1$
		}
	}

	/**
	 * 先頭の連続する文字 + 空白を除去します。
	 * @param str 文字
	 * @return 除去した場合は true
	 */
	protected boolean trimStartsWithSpace(final String str) {

		if (body.startsWith(str)) {
			final Pattern pat = Pattern.compile("(?s)^(" + str + "+\\s+)(.+)$"); //$NON-NLS-1$ //$NON-NLS-2$
			final Matcher mat = pat.matcher(body);
			if (mat.find()) {
				starts.append(mat.group(1));
				body = mat.group(2);
				return true;
			}
		}
		return false;
	}

	/**
	 * 末尾の空白 + 連続する文字を除去します。
	 * @param str 文字
	 * @return 除去した場合は true
	 */
	protected boolean trimEndsWithSpace(final String str) {

		if (body.endsWith(str)) {
			final Pattern pat = Pattern.compile("(?s)^(.+?)(\\s+" + str + "+)$"); //$NON-NLS-1$ //$NON-NLS-2$
			final Matcher mat = pat.matcher(body);
			if (mat.find()) {
				body = mat.group(1);
				ends.insert(0, mat.group(2));
				return true;
			}
		}
		return false;
	}

	/**
	 * 強制トリムします。
	 * <p>
	 * このメソッドの結果は翻訳に影響を与える可能性があるため、
	 * 辞書プロパティーの生成では使用されません。
	 * 用途としては辞書参照時に訳が見つからない場合に、このメソッドで
	 * 処理した後、再検索するために使用することが想定されています。
	 * 除去された文字列は revert のためにこのオブジェクト内に保持されます。
	 * 除去の対象を以下に示します。
	 * <pre>
	 * ・前後の囲み文字 ()、[]、<>、""、''、!! など (前後の組み合わせが一致する場合)
	 * ・先頭の IWAB0001E のような Eclipse 固有メッセージ・コード
	 * ・複数形を示す (s) (秒を示すものではない場合)
	 * ・前後の ...、.、: など
	 * <pre>
	 * 末尾の . が 1 つの場合は 。に変換され、revert 用に保持されます。
	 * <p>
	 * @return 文字列
	 */
	public String trimForce() {

		trim();

		// --- trim から移動 - BEGIN
		// 前後囲み文字の除去
		trimSurround("(", ")"); //$NON-NLS-1$ //$NON-NLS-2$
		trimSurround("[", "]"); //$NON-NLS-1$ //$NON-NLS-2$
		trimSurround("<!--", "-->"); //$NON-NLS-1$ //$NON-NLS-2$
		trimSurround("<", ">"); //$NON-NLS-1$ //$NON-NLS-2$
		trimSurround("「", "」"); //$NON-NLS-1$ //$NON-NLS-2$
		trimSurround("\""); //$NON-NLS-1$
		trimSurround("'"); //$NON-NLS-1$
		trimSurround("!"); //$NON-NLS-1$

		// 分割後の補正 例) (App Engine - 1.2.0)
		// (App Engine -
		// 1.2.0)
		if (body.startsWith("(") && !body.startsWith("( ") && !body.contains(")")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			trimStarts("("); //$NON-NLS-1$

		} else if (body.endsWith(")") && !body.endsWith(" )") && //「Missing right parenthesis )=右括弧 ) がありません」除外 //$NON-NLS-1$ //$NON-NLS-2$
				!body.contains("(") && !body.endsWith(":)")) { // 顔文字除外 //$NON-NLS-1$ //$NON-NLS-2$

			trimEnds(")"); //$NON-NLS-1$
		}

		// 前後の連続するハイフン + 空白  例) --- hoge ---
		trimHyphen();

		// 先頭部の Eclipse 固有エラーコード  例) IWAB0001E
		if (body.startsWith("I") || body.startsWith("C")) { //$NON-NLS-1$ //$NON-NLS-2$
			final Pattern pat = Pattern.compile("(?s)^([A-Z]{3,4}\\d{4}[A-Z]{0,1}[\\s:]+)(.+)$"); //$NON-NLS-1$
			final Matcher mat = pat.matcher(body);
			if (mat.find()) {
				starts.append(mat.group(1));
				body = mat.group(2);
			}
		}
		// --- trim から移動 - END

		trimSpecific(".", "。", ":", "?", "!"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$

		// 末尾の . が 1 つの場合は 。に変換
		final String e = ends.toString();
		if (e.startsWith(".") && !e.startsWith("..")) { //$NON-NLS-1$ //$NON-NLS-2$
			ends.replace(0, 1, "。"); //$NON-NLS-1$
		}
		return body;
	}

	/**
	 * trim により除去された文字列を復元します。
	 * このメソッドでは toString() と異なり trim された末尾の . は 。として復元されます。
	 * <p>
	 * @return 連結後の文字列
	 */
	public String revert() {
		trim();
		return starts + body + ends;
	}

	/**
	 * trim により除去された文字列を指定された文字列の前後に復元します。
	 * trimForce された末尾の . は 。として復元されます。
	 * <p>
	 * @param jaBody 新しい文字列 (日本語)
	 * @return 連結後の文字列
	 */
	public String revert(final String jaBody) {
		trim();
		return starts + jaBody + ends;
	}

	/**
	 * 分割されたときの各要素となる翻訳文字列クラス。
	 * 最後の要素以外の特殊処理を行います。
	 */
	private static class SplitTranslationString extends JapaneseTranslationString {

		/**
		 * コンストラクター。
		 * @param value 値。
		 */
		public SplitTranslationString(final String value) {
			super(value);
		}

		@Override
		protected void trimHyphen() {

			// 末尾の - が句点分割で残っているため。
			// 親クラスの動作では先頭に - がないと末尾 - は除去されない。
			trimEndsWithSpace("-"); //$NON-NLS-1$
		}

		@Override
		public String revert(final String jaBody) {

			trim();

			// 分割された翻訳文字列の特殊句点処理。英語ピリオド後の空白除去
			// 英語「. 」		-> 日本語「。」
			// 英語「.&nbsp;」	-> 日本語「。」
			if (jaBody.endsWith("。") && ends.length() != 0) { //$NON-NLS-1$
				if (ends.charAt(0) == ' ') {
					ends.deleteCharAt(0);
				} else if (ends.toString().startsWith("&nbsp;")) { //$NON-NLS-1$
					ends.delete(0, 6);
				}
			}
			// 強制トリムされていた場合
			else if (ends.toString().startsWith("。 ")) { //$NON-NLS-1$
				ends.deleteCharAt(1);
			}
			return starts + jaBody + ends;
		}
	}

	/**
	 * 文字列を「。」や「. 」などの句点で分割した翻訳文字列のリストを取得します。
	 * 分割された翻訳文字列には末尾の句点が含まれます。
	 * 分割できない場合、意図しない二重 trim や二重 revert を避けるため null を返します。
	 * つまり、戻り値は必ずサイズ 2 以上のリストまたは null になります。
	 * <p>
	 * リストの先頭要素の先頭と、末尾要素の末尾はトリムされます。
	 * 復元するには、リストをすべて連結し、revert することで復元できます。
	 * <p>
	 * @return 句点分割した翻訳文字列リスト (分割できない場合は null)
	 */
	public List<JapaneseTranslationString> split() {

		// trimForce が呼ばれていたときのために元に戻してから trim
		init();
		trim();

		// フォーマット済みテキスト (改行の後に \\s がある) の場合は分割しない
		//if (isFormatedText(original)) { // 現在、改行の除去復元に未対応

		// 改行がある場合は分割しない
		if (original.contains("\n")) { //$NON-NLS-1$
			return null;
		}

		final List<String> list = new LinkedList<String>();

		// 先頭、末尾の () や [] を分割
		for (final String s : splitParenthesis(body)) {

			// 句点分割
			list.addAll(splitPunct(s));
		}

		// 分割 1 の場合は二重トリムや二重 revert を避けるため null を返す
		if (list.size() == 1) {
			return null;
		}

		// TranslationString のリストに変換
		final List<JapaneseTranslationString> tsList = new LinkedList<JapaneseTranslationString>();
		for (int i = 0; i < list.size(); i++) {
			if (i < list.size() - 1) {
				// 最後の要素以外は特殊処理を行うため拡張クラス
				tsList.add(new SplitTranslationString(list.get(i)));
			} else {
				// 最後の要素
				tsList.add(new JapaneseTranslationString(list.get(i)));
			}
		}
		return tsList;
	}

	/**
	 * 先頭、末尾の括弧 (.+) [.+] を分割します。
	 * ただし、下記の場合を除きます。
	 * <pre>
	 * ・[{0}] のように { が含まれる (訳文の一部になる可能性があるため)
	 * </pre>
	 * @param value 文字列
	 * @return 分割後のリスト
	 */
	private List<String> splitParenthesis(final String value) {

		final List<String> list = new LinkedList<String>();
		if (!value.contains("(") && !value.contains("[")) { //$NON-NLS-1$ //$NON-NLS-2$
			list.add(value);
			return list;
		}

		// 先頭パターン
		Pattern pat = Pattern.compile("(?s)^" + "([\\(\\[][^\\{]+[\\)\\]]\\s)" + "(.+)$"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		Matcher mat = pat.matcher(value);
		boolean isFound = false;
		if (mat.find()) {
			isFound = true;

			// 末尾パターン
		} else {
			pat = Pattern.compile("(?s)^" + "(.+\\s)" + "([\\(\\[][^\\{]+[\\)\\]](|[\\.。:])\\s*)$"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			mat = pat.matcher(value);
			if (mat.find()) {

				// "(xxx)yyy)" みたいになった場合、g1 を最短一致に切り替え
				if (invalidParenthesis(mat.group(2))) {

					pat = Pattern.compile("(?s)^" + "(.+?\\s)" + "([\\(\\[][^\\{]+[\\)\\]](|[\\.。:])\\s*)$"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					mat = pat.matcher(value);
					isFound = mat.find();
				} else {
					isFound = true;
				}
			}
		}

		if (isFound) {

			final String g1 = mat.group(1);
			final String g2 = mat.group(2);

			if (invalidParenthesis(g1) || invalidParenthesis(g2)) {
				// 括弧の対応が不正
				list.add(value);
			} else {
				// 再帰
				list.addAll(splitParenthesis(g1));
				list.addAll(splitParenthesis(g2));
			}
		} else {
			// 分割なし
			list.add(value);
		}
		return list;
	}

	/**
	 * 括弧の対応が不正か判定します。
	 * @param value 文字列
	 * @return 不正な場合は true
	 */
	private boolean invalidParenthesis(final String value) {
		return (countMatches(value, "(") != countMatches(value, ")") || countMatches(value, "[") != countMatches(value, //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				"]")); //$NON-NLS-1$
	}

	/**
	 * 句点分割します。
	 * <p>
	 * @param value 文字列
	 * @return 分割後のリスト
	 */
	private List<String> splitPunct(final String value) {

		// 分割パターン
		final Pattern pat = Pattern.compile("\\s*(" + "[。：？！]+|" + // 日本語の句点 //$NON-NLS-1$ //$NON-NLS-2$
				"[\\.:;\\?!]+(\\s|&nbsp;)+|" + // 英語の句点 (次の文字が空白) //$NON-NLS-1$
				"\\s-\\s" + // ハイフン (前後空白) //$NON-NLS-1$
				")\\s*"); //$NON-NLS-1$
		final Matcher mat = pat.matcher(value);
		final List<String> list = new LinkedList<String>();

		final StringBuffer sb = new StringBuffer();
		while (mat.find()) {

			final String sep = mat.group();
			final int sPos = mat.start();
			final int ePos = mat.end();
			if (sPos > 0 && ePos < value.length()) {

				final String s = value.substring(0, sPos);
				final String e = value.substring(ePos);

				// [:-] の後に {n} があり、その後に単語、つまり文中で意味を成す場合は分割しない
				// 例) The\ non\ default\ location\:\ {0}\ for\ {1}\ is\ in\ use.
				//	   No\ data\ source\ available\ for\ data\ set\ -\ {0},\ please\ select\ a\ data\ source
				if (sep.contains(":") || sep.contains("-")) { //$NON-NLS-1$ //$NON-NLS-2$
					if (e.contains("{") && e.matches("(?s).*?[^\\p{Punct}\\d].*")) { //$NON-NLS-1$ //$NON-NLS-2$
						continue;
					}
				}
				// : が () で囲まれている場合は分割しない
				// 例) (EJB 2.0\: 22.2、22.5)
				if (sep.contains(":")) { //$NON-NLS-1$
					if (s.contains("(") && e.contains(")")) { //$NON-NLS-1$ //$NON-NLS-2$
						continue;
					}
				}
				// 前が _ の場合 (文字そのものを表す場合)
				// 例) A-Z a-z 0-9 . _ -
				// ハイフンの前後が数値の場合は分割しない
				// 例) 1 - 2
				else if (sep.contains("-")) { //$NON-NLS-1$
					if (s.endsWith("_")) { //$NON-NLS-1$
						continue;
					}
					if (s.matches("(?s).*\\d") && e.matches("(?s)\\d.*")) { //$NON-NLS-1$ //$NON-NLS-2$
						continue;
					}
				}
				// ... の前後に空白
				// 例) INSERT ... VALUES (...)、 (...)、 ...、
				// 前が空白の場合 (文字そのものを表す場合)
				// 例) A-Z a-z 0-9 . _ -
				// . が省略を意味する場合は分割しない
				// 例) hoge Inc. xxx
				else if (sep.contains(".")) { //$NON-NLS-1$
					if (sep.equals(" ... ")) { //$NON-NLS-1$
						continue;
					}
					if (sep.contains(" .")) { //$NON-NLS-1$
						continue;
					}
					if (endsWithAbbreviations(s + sep.charAt(0))) {
						continue;
					}
				}
				// ; が HTML 参照文字の末尾の場合は分割しない
				// 例) &lt;name&gt; is a file
				else if (sep.startsWith(";")) { //$NON-NLS-1$
					final String ref = s.replaceFirst("(?s).*(&.+)", "$1") + sep; //$NON-NLS-1$ //$NON-NLS-2$
					if (containsHtmlReference(ref)) {
						continue;
					}
				}
				// 。の次が ) の場合は分割しない
				// 例) 。)
				else if (sep.endsWith("。")) { //$NON-NLS-1$
					if (e.startsWith(")")) { //$NON-NLS-1$
						continue;
					}
				}
				// ? の次が = の場合は分割しない
				// 例) (* = any string, ? = any character)
				else if (sep.contains("?")) { //$NON-NLS-1$
					if (e.startsWith("=")) { //$NON-NLS-1$
						continue;
					}
				}
			}
			mat.appendReplacement(sb, Matcher.quoteReplacement(sep));
			list.add(sb.toString());
			sb.delete(0, sb.length());
		}

		mat.appendTail(sb);
		if (sb.length() > 0) {
			list.add(sb.toString());
		}

		return list;
	}

	//-------------------------------------------------------------------------

	/**
	 * new TranslationString(value).trim() のショートカットです。
	 * @param value 値
	 * @return トリム後の文字列
	 */
	public static String trim(final String value) {
		return new JapaneseTranslationString(value).trim();
	}

	/**
	 * 複数形を示す可能性が高い (s) を削除します。
	 * @param en 英語文字列
	 * @return 除去後の文字列
	 */
	public static String removeS(final String en) {

		if (en.contains("(s)") && !en.contains(" (s)")) { //$NON-NLS-1$ //$NON-NLS-2$
			return en.replace("(s)", ""); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return en;
	}

	/**
	 * フォーマット済みテキスト (改行の後に \\s がある) か判定します。
	 * 現在、このメソッドは未使用。将来、改行自動最適化を実装した場合に使用するかも。
	 * @param en 英語
	 * @return フォーマット済みの場合は true
	 */
	public static boolean isFormatedText(final String en) {

		return en.contains("\n") && en.matches("(?s).*?\\n\\s.*"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * 末尾が省略後か判定します。
	 * @param en 英語
	 * @return 末尾が省略後の場合は true
	 */
	public static boolean endsWithAbbreviations(final String en) {

		return en.matches("(?s)(?i).*?[\\s\\p{Punct}](Inc|Ltd|etc|Reg|Exp|e\\.g)\\."); //$NON-NLS-1$
	}

	/**
	 * 指定した文字列がいくつ含まれるかカウントします。
	 * <pre>
	 * countMatches(null, *)       = 0
	 * countMatches("", *)         = 0
	 * countMatches("abba", null)  = 0
	 * countMatches("abba", "")    = 0
	 * countMatches("abba", "a")   = 2
	 * countMatches("abba", "ab")  = 1
	 * countMatches("abba", "xxx") = 0
	 * </pre>
	 * @param str チェックする文字列。null 可。
	 * @param sub カウントする文字列。null 可。
	 * @return カウント。str または sub が null の場合は 0
	 */
	public static int countMatches(final String str, final String sub) {
		if (isEmpty(str) || isEmpty(sub)) {
			return 0;
		}
		int count = 0;
		int idx = 0;
		while ((idx = str.indexOf(sub, idx)) != -1) {
			count++;
			idx += sub.length();
		}
		return count;
	}

	/**
	 * 文字列が空か null かチェックします。
	 * <pre>
	 * StringUtils.isEmpty(null)      = true
	 * StringUtils.isEmpty("")        = true
	 * StringUtils.isEmpty(" ")       = false
	 * StringUtils.isEmpty("bob")     = false
	 * StringUtils.isEmpty("  bob  ") = false
	 * </pre>
	 * @param str チェックする文字列
	 * @return 空か null の場合は true
	 */
	public static boolean isEmpty(final String str) {
		return str == null || str.length() == 0;
	}

	/**
	 * 特定の html 実体参照文字が含まれているか判定します。
	 * <p>
	 * @param value 値
	 * @return 含まれている場合は true
	 */
	public static boolean containsHtmlReference(final String value) {

		return value.contains("&lt;") || value.contains("&gt;") || value.contains("&amp;") || value.contains("&quot;") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				|| value.contains("&yen;") || value.contains("&nbsp;"); //$NON-NLS-1$ //$NON-NLS-2$
	}
}
