/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.List;

import benten.cat.tm.core.BentenTmDriverManager;
import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;
import benten.cat.tm.ui.CatTmUiPlugin;
import benten.core.BentenConstants;
import benten.core.text.Strings;
import benten.twa.cat.core.valueobject.BentenApplyFuzzyMatchProcessInput;
import benten.twa.cat.messages.BentenApplyFuzzyMatchMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.BlancoXliffSerializer;
import blanco.xliff.BlancoXliffUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffAltTrans;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTool;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 類似訳の適用
 *
 * <pre>
 * このウィザードは、TMX の類似訳を XLIFF に適用します。
 *   1.  翻訳単位と類似する TMX を探し、翻訳単位の代替翻訳に適用します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式機械支援翻訳機能: 類似訳適用機能」に対応します。
 *
 * @author IGA Tosiki
 */
public class BentenApplyFuzzyMatchProcessImpl extends AbstractTraverseDir implements BentenApplyFuzzyMatchProcess {

	/**
	 * 類似訳適用機能のためのメッセージ。
	 */
	protected static final BentenApplyFuzzyMatchMessages fMsg = new BentenApplyFuzzyMatchMessages();

	/**
	 * TM エンジン。
	 */
	protected BentenTmEngine fTm;

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenApplyFuzzyMatchProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenApplyFuzzyMatchProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenApplyFuzzyMatchProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenApplyFuzzyMatchProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		File dirTmx = null;
		if (fInput.getTmxdir() != null && fInput.getTmxdir().length() > 0) {
			dirTmx = new File(fInput.getTmxdir());
			if (dirTmx.exists() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getTmxdir()));
			}
			if (dirTmx.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getTmxdir()));
			}
		}

		final File dirXliff = new File(fInput.getXliffdir());
		if (dirXliff.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getXliffdir()));
		}
		if (dirXliff.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getXliffdir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenApplyFuzzyMatchProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) {
				fCounter += 2;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(fInput);
		inner.processDir(new File(input.getXliffdir()));
		beginTask(inner.getCounter() + 4);

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		if (fInput.getTmdriverclassname() == null || fInput.getTmdriverclassname().trim().length() == 0) {
			// プラグインの場合のコース。
			fTm = CatTmUiPlugin.getDefault().getTmEngine();
		} else {
			// Ant タスクの場合のコース。
			try {
				Class.forName(fInput.getTmdriverclassname());
			} catch (final ClassNotFoundException e) {
				throw new IOException(fMsg.getCoreE001(fInput.getTmdriverclassname(), e.toString()));
			}
			if (BentenTmDriverManager.getDrivers().length == 0) {
				throw new IllegalArgumentException(fMsg.getCoreE002());
			}
			fTm = BentenTmDriverManager.getDrivers()[0].getEngineInstance();
		}
		if (fTm == null) {
			throw new IOException(fMsg.getCoreE003());
		}

		if (dirTmx != null) {
			if (progress(fMsg.getCoreP003())) {
				return 6;
			}

			fTm.loadTmx(dirTmx);
		}

		if (progress(fMsg.getCoreP004())) {
			return 6;
		}

		processDir(dirXliff);

		if (progress(fMsg.getCoreP005(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * 処理の内部処理でアイテムが処理されるたびに進捗報告としてコールバック。
	 *
	 * @param argProgressMessage 現在処理しているアイテムに関するメッセージ。
	 * @return 処理をそのまま継続する場合は false。処理中断をリクエストしたい場合は true。
	 */
	public boolean progressInner(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	protected void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException("BentenApplyFuzzyMatchProcessImpl#processFile: 'fInput' is null."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		final BlancoXliffParser parser = new BlancoXliffParser();

		BlancoXliff xliff = null;
		try {
			xliff = parser.parse(file);
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE011(file.getName(), e.toString()));
		}

		int transUnitTotalCount = 0;
		for (final BlancoXliffFile xliffFile : xliff.getFileList()) {
			transUnitTotalCount += xliffFile.getBody().getTransUnitList().size();
		}

		int transUnitCount = 0;
		for (final BlancoXliffFile xliffFile : xliff.getFileList()) {
			// 有無を言わさず tool-id を設定します。
			setToolId(xliffFile);

			for (final BlancoXliffTransUnit transUnit : xliffFile.getBody().getTransUnitList()) {
				if (progressInner(fMsg.getCoreP101(file.getName(), BigDecimal.valueOf(++transUnitCount), BigDecimal
						.valueOf(transUnitTotalCount)))) {
					break;
				}

				if (transUnit.getTranslate() == false) {
					continue;
				}
				if (transUnit.getSource() == null || transUnit.getSource().length() == 0) {
					continue;
				}

				// 自分が設定した alt-trans については、一旦クリアーします。
				clearSelfAltTrans(transUnit);

				// trans-unit を処理します。
				processTransUnit(transUnit, fTm);
			}
		}

		try {
			final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			new BlancoXliffSerializer().serialize(xliff, outStream);
			outStream.flush();

			if (BlancoFileUtil.bytes2FileIfNecessary(outStream.toByteArray(), file) == 0) {
				// 変更無し。
				getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
			} else {
				// 更新のあったファイル数をカウントします。
				getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			}
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE012(file.getName(), e.toString()));
		}

		if (progress(fMsg.getCoreP013(file.getName()))) {
			return;
		}
	}

	/**
	 * trans-unit を処理します。
	 *
	 * <UL>
	 * <LI>fuzzySearch に与える文字列をプリファランスの設定に従って加工するかどうかの判断は UI または Ant 引数によって与えられます。
	 * </UL>
	 *
	 * @param transUnit 翻訳単位
	 * @param tm TM エンジン
	 */
	protected void processTransUnit(final BlancoXliffTransUnit transUnit, final BentenTmEngine tm) {

		// ホワイトスペースの除去
		String source = transUnit.getSource();
		if (fInput.getIgnorewhitespacetmreference()) {
			source = Strings.removeRedundantWhitespace(source);
		}

		if (fInput.getIgnoremnemonickeytmreference()) {
			source = Strings.stripMnemonicKey(source);
		}

		// あいまい翻訳を検索
		final List<BentenTmSearchResult> matchList = tm.fuzzySearch(source);
		if (matchList.size() > 0) {
			fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);
		}
		for (final BentenTmSearchResult entry : matchList) {
			final BlancoXliffAltTrans altTrans = new BlancoXliffAltTrans();
			transUnit.getAltTransList().add(altTrans);

			altTrans.setToolId(BentenConstants.PROJECT_NAME_LOWER);
			altTrans.setOrigin(entry.getOrigin());
			altTrans.setMatchQuality(entry.getMatchQuality().trim());
			altTrans.setSource(entry.getSource());
			final BlancoXliffTarget target = new BlancoXliffTarget();
			altTrans.setTarget(target);
			altTrans.getTarget().setTarget(entry.getTarget());

			if ("100%".equals(altTrans.getMatchQuality())) { //$NON-NLS-1$
				// 100% 一致の場合のみ、完全一致にまつわる特別な処理をおこないます。
				final String targetString = BentenTmUtil.searchExactMatch(tm, source);
				if (targetString == null) {
					// 完全一致訳は無いものと判断されました。一致率を 99% で上書きします。
					altTrans.setMatchQuality("99%"); //$NON-NLS-1$
				}
			}
		}
	}

	/**
	 * 自分で登録した alt-trans をクリアーします。
	 *
	 * @param transUnit 翻訳単位
	 */
	protected void clearSelfAltTrans(final BlancoXliffTransUnit transUnit) {
		for (int index = transUnit.getAltTransList().size() - 1; index >= 0; index--) {
			final BlancoXliffAltTrans altTrans = transUnit.getAltTransList().get(index);
			if (altTrans.getToolId().equals(BentenConstants.PROJECT_NAME_LOWER)) {
				// これは自分の訳です。
				// 除去してしまいます。
				transUnit.getAltTransList().remove(index);
			}
		}
	}

	/**
	 * tool-id を設定します。
	 *
	 * @param file ファイル
	 */
	protected void setToolId(final BlancoXliffFile file) {
		final BlancoXliffTool tool = new BlancoXliffTool();
		tool.setToolId(BentenConstants.PROJECT_NAME_LOWER);
		tool.setToolName(BentenConstants.PROJECT_NAME);
		BlancoXliffUtil.setTool(file.getHeader(), tool);
	}
}
