/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui.dialogs;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import benten.cat.ui.dialogs.messages.NoteDialogMessages;
import benten.cat.ui.viewers.TransSourceViewer;
import benten.ui.UiStatus;
import benten.ui.views.UiToolkit;

/**
 * ノートを編集するためのダイアログ。
 *
 * @author KASHIHARA Shinji
 */
public class NoteDialog extends StatusDialog implements ModifyListener {

	/**
	 * ノートを編集するためのダイアログのためのメッセージ。
	 */
	protected static final NoteDialogMessages fMsg = new NoteDialogMessages();

	/** ノート・エントリー・クラス */
	public static class NoteDialogEntry {

		/** 担当 */
		public String from;

		/** 内容 */
		public String content;

		/** 修正の場合は true */
		public boolean modify;

		/**
		* コンストラクター。
		 * @param from 担当
		 * @param content 内容
		 */
		public NoteDialogEntry(final String from, final String content) {
			this.from = from;
			this.content = content;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((content == null) ? 0 : content.hashCode());
			result = prime * result + ((from == null) ? 0 : from.hashCode());
			return result;
		}

		@Override
		public boolean equals(final Object obj) {
			if (obj != null && obj instanceof NoteDialogEntry) {
				final NoteDialogEntry note = (NoteDialogEntry) obj;
				return note.from.equals(from) && note.content.equals(content);
			}
			return false;
		}
	}

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	private final UiToolkit toolkit = new UiToolkit();

	/** ノート・エントリー */
	private final NoteDialogEntry noteEntry;

	/** 担当テキスト */
	private Text fromText;

	/** 内容テキスト・ビューアー */
	private TextViewer textTextViewer;

	/**
	 * コンストラクター。
	 * @param parent 親シェル
	 * @param title タイトル
	 * @param noteEntry 編集対象となるノート・エントリー
	 */
	public NoteDialog(final Shell parent, final String title, final NoteDialogEntry noteEntry) {

		super(parent);
		this.noteEntry = noteEntry;
		setTitle(title);
		setHelpAvailable(false);
	}

	@Override
	protected Control createDialogArea(final Composite parent) {

		final Composite composite = (Composite) super.createDialogArea(parent);
		final Composite c = toolkit.createComposite(composite, 2);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		toolkit.createLabel(c, fMsg.getLabel0());
		fromText = new Text(c, SWT.SINGLE | SWT.BORDER);
		fromText.setText(noteEntry.from);
		fromText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		fromText.addModifyListener(this);

		toolkit.createLabel(c, fMsg.getLabel1());
		final GridData gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 60;
		gd.widthHint = 400;
		textTextViewer = new TransSourceViewer(c, true);
		textTextViewer.getControl().setLayoutData(gd);
		textTextViewer.getDocument().set(noteEntry.content);
		textTextViewer.getTextWidget().addModifyListener(this);

		return composite;
	}

	@Override
	protected void okPressed() {
		final NoteDialogEntry editedNoteEntry = getEditedNoteDialogEntry();

		// 変更がない場合はキャンセル扱いにする
		if (noteEntry.modify && editedNoteEntry.equals(noteEntry)) {
			cancelPressed();
		} else {
			noteEntry.from = editedNoteEntry.from;
			noteEntry.content = editedNoteEntry.content;
			super.okPressed();
		}
	}

	@Override
	protected boolean isResizable() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	public void modifyText(final ModifyEvent e) {

		Status status = UiStatus.OK_STATUS;
		final NoteDialogEntry editedNoteEntry = getEditedNoteDialogEntry();
		if (editedNoteEntry.from.equals("") && editedNoteEntry.content.equals("")) { //$NON-NLS-1$ //$NON-NLS-2$
			status = new UiStatus(IStatus.ERROR, fMsg.getDlgError0());
		}
		updateStatus(status);
	}

	/**
	 * ノート・ダイアログのエントリーを取得。
	 * @return ノート・ダイアログのエントリー
	 */
	private NoteDialogEntry getEditedNoteDialogEntry() {
		return new NoteDialogEntry(fromText.getText(), textTextViewer.getDocument().get());
	}
}
