/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.ui.preference;

import java.io.File;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.ui.IWorkbench;

import benten.cat.glossary.ui.CatGlossaryUiPlugin;
import benten.cat.glossary.ui.preference.messages.CatGlossaryUiPreferencePageMessages;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;
import benten.ui.fields.AbstractBrowseTextField;
import benten.ui.fields.FolderTextField;
import benten.ui.preference.AbstractPreferencePage;
import benten.ui.preference.BentenPreference;
import benten.ui.preference.BentenPreference.Preference;
import benten.ui.views.UiToolkit;

/**
 * 用語集設定ページ。
 *
 * @author IGA Tosiki
 */
public class CatGlossaryUiPreferencePage extends AbstractPreferencePage {
	/**
	 * 用語集設定ページのためのメッセージ。
	 */
	protected static final CatGlossaryUiPreferencePageMessages fMsg = new CatGlossaryUiPreferencePageMessages();

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	protected final UiToolkit toolkit = new UiToolkit();

	/** 用語集名コンボ */
	protected Combo glossaryNameCombo;

	/** TSV デフォルト使用チェックボックス */
	protected Button useTsvDefaultCheck;

	/** TSV ロケーション使用チェックボックス */
	protected Button useTsvLocationCheck;

	/** TSV ロケーション・テキスト */
	protected AbstractBrowseTextField tsvLocationText;

	/**
	 * コンストラクター。
	 */
	public CatGlossaryUiPreferencePage() {
	}

	/**
	 * コンストラクター。
	 * @param title タイトル
	 */
	public CatGlossaryUiPreferencePage(final String title) {
		super(title);
	}

	/**
	 * コンストラクター。
	 * @param title タイトル
	 * @param image イメージ
	 */
	public CatGlossaryUiPreferencePage(final String title, final ImageDescriptor image) {
		super(title, image);
	}

	@Override
	protected Control createContents(final Composite parent) {
		final Composite composite = toolkit.createComposite(parent, 1);
		final IPreferenceStore store = BentenPreference.getStore();

		final Composite c = toolkit.createComposite(composite, 2);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final Set<String> glossaryNameSet = CatGlossaryUiPlugin.getDefault().getGlossaryEnginePluginNameSet();
		toolkit.createLabel(c, fMsg.getLabel0());
		glossaryNameCombo = new Combo(c, SWT.READ_ONLY);
		glossaryNameCombo.setItems(glossaryNameSet.toArray(new String[glossaryNameSet.size()]));
		glossaryNameCombo.setText(store.getString(Preference.GLOSSARY_NAME.name()));

		createTsvGroup(composite);

		return composite;
	}

	/**
	 * TSV に関するグループの作成。
	 * @param parent 親コンポジット
	 */
	protected void createTsvGroup(final Composite parent) {
		final IPreferenceStore store = BentenPreference.getStore();

		final Group g = new Group(parent, SWT.BOLD);
		g.setLayout(toolkit.createGridLayout(4, 5));
		g.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		g.setText(fMsg.getLabel1());

		final GridData gd = new GridData();
		gd.horizontalSpan = 4;
		useTsvDefaultCheck = new Button(g, SWT.RADIO);
		useTsvDefaultCheck.setText(fMsg.getLabel2());
		useTsvDefaultCheck.setSelection(store.getBoolean(Preference.USE_TSV_DEFAULT.name()));
		useTsvDefaultCheck.setLayoutData(gd);
		useTsvDefaultCheck.addSelectionListener(new TsvSelectionAdapter());

		useTsvLocationCheck = new Button(g, SWT.RADIO);
		useTsvLocationCheck.setText(fMsg.getLabel3());
		useTsvLocationCheck.setSelection(store.getBoolean(Preference.USE_TSV_LOCATION.name()));
		useTsvLocationCheck.addSelectionListener(new TsvSelectionAdapter());

		tsvLocationText = new FolderTextField(g);
		tsvLocationText.setWidthHint(SWT.DEFAULT);
		tsvLocationText.setText(store.getString(Preference.TSV_LOCATION.name()));

		if (!useTsvDefaultCheck.getSelection() && !useTsvLocationCheck.getSelection()) {
			useTsvDefaultCheck.setSelection(true);
		}
		updateTsvCheck();
	}

	/**
	 * TSV 選択アダプター。
	 */
	protected class TsvSelectionAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(final SelectionEvent e) {
			updateTsvCheck();
		}
	}

	/**
	 * TSV チェックボックス変更による更新。
	 */
	protected void updateTsvCheck() {
		tsvLocationText.setEnabled(useTsvLocationCheck.getSelection());
	}

	/**
	 * {@inheritDoc}
	 */
	public void init(final IWorkbench workbench) {
	}

	@Override
	public boolean performOk() {
		final IPreferenceStore store = BentenPreference.getStore();

		final String newGlossaryName = glossaryNameCombo.getText();
		final boolean newUseTsvLocation = useTsvLocationCheck.getSelection();

		// TSV ロケーションのチェック
		if (newUseTsvLocation) {
			if (tsvLocationText.getText().equals("")) { //$NON-NLS-1$
				final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError0());
				UiPlugin.openDialog(fMsg.getDlgError1(), s);
				return false;
			}
			final File tsvDir = tsvLocationText.getFile();
			if (!tsvDir.exists() || tsvDir.isFile()) {
				final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError2());
				UiPlugin.openDialog(fMsg.getDlgError3(), s);
				return false;
			}
		}

		// 設定が load メソッド内で参照されるため、先に保管
		store.setValue(Preference.GLOSSARY_NAME.name(), newGlossaryName);
		store.setValue(Preference.USE_TSV_DEFAULT.name(), useTsvDefaultCheck.getSelection());
		store.setValue(Preference.USE_TSV_LOCATION.name(), useTsvLocationCheck.getSelection());
		store.setValue(Preference.TSV_LOCATION.name(), tsvLocationText.getText());

		try {
			CatGlossaryUiPlugin.getDefault().loadExtensions();
		} catch (final IllegalArgumentException e) {
			final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError4(e.getMessage()));
			UiPlugin.openDialog(fMsg.getDlgError5(), s);
			return false;
		}

		return true;
	}

	@Override
	protected void performDefaults() {
		final IPreferenceStore store = BentenPreference.getStore();
		glossaryNameCombo.setText(store.getDefaultString(Preference.GLOSSARY_NAME.name()));

		useTsvDefaultCheck.setSelection(store.getDefaultBoolean(Preference.USE_TSV_DEFAULT.name()));
		useTsvLocationCheck.setSelection(store.getDefaultBoolean(Preference.USE_TSV_LOCATION.name()));
		tsvLocationText.setText(store.getDefaultString(Preference.TSV_LOCATION.name()));
		updateTsvCheck();
	}

	@Override
	protected void setEnabled(final boolean enabled) {
		glossaryNameCombo.setEnabled(enabled);
		useTsvDefaultCheck.setEnabled(enabled);
		useTsvLocationCheck.setEnabled(enabled);
		tsvLocationText.setEnabled(enabled);
	}
}
