/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.tmx.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.tmx.core.valueobject.BentenMergeTmxProcessInput;

/**
 * Apache Antタスク [BentenMergeTmx]のクラス。
 *
 * TMX フラグメント・マージのための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.tmx.core]にBentenMergeTmxBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenmergetmx&quot; classname=&quot;benten.twa.tmx.core.BentenMergeTmxTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenMergeTmxTask extends Task {
    /**
     * TMX フラグメント・マージのための Ant タスク。
     */
    protected BentenMergeTmxProcessInput fInput = new BentenMergeTmxProcessInput();

    /**
     * フィールド [sourcefile] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcefileProcessed = false;

    /**
     * フィールド [targetfile] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetfileProcessed = false;

    /**
     * フィールド [transsourcelang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranssourcelangProcessed = false;

    /**
     * フィールド [transtargetlang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranstargetlangProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[sourcefile]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * マージ元となる TMX フラグメント・ファイル。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcefile(final String arg) {
        fInput.setSourcefile(arg);
        fIsFieldSourcefileProcessed = true;
    }

    /**
     * Antタスクの[sourcefile]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * マージ元となる TMX フラグメント・ファイル。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcefile() {
        return fInput.getSourcefile();
    }

    /**
     * Antタスクの[targetfile]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * マージ先となる TMX ファイル。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetfile(final String arg) {
        fInput.setTargetfile(arg);
        fIsFieldTargetfileProcessed = true;
    }

    /**
     * Antタスクの[targetfile]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * マージ先となる TMX ファイル。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetfile() {
        return fInput.getTargetfile();
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳元言語<br>
     *
     * @param arg セットしたい値
     */
    public void setTranssourcelang(final String arg) {
        fInput.setTranssourcelang(arg);
        fIsFieldTranssourcelangProcessed = true;
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳元言語<br>
     * デフォルト値[en-US]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranssourcelang() {
        return fInput.getTranssourcelang();
    }

    /**
     * Antタスクの[transtargetlang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * 翻訳先言語<br>
     *
     * @param arg セットしたい値
     */
    public void setTranstargetlang(final String arg) {
        fInput.setTranstargetlang(arg);
        fIsFieldTranstargetlangProcessed = true;
    }

    /**
     * Antタスクの[transtargetlang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * 翻訳先言語<br>
     * デフォルト値[ja-JP]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranstargetlang() {
        return fInput.getTranstargetlang();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenMergeTmxTask begin.");

        // 項目番号[1]、アトリビュート[sourcefile]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcefileProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcefile]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[targetfile]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetfileProcessed == false) {
            throw new BuildException("必須アトリビュート[targetfile]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- sourcefile:[" + getSourcefile() + "]");
            System.out.println("- targetfile:[" + getTargetfile() + "]");
            System.out.println("- transsourcelang:[" + getTranssourcelang() + "]");
            System.out.println("- transtargetlang:[" + getTranstargetlang() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenMergeTmxProcessインタフェースを実装して benten.twa.tmx.coreパッケージに BentenMergeTmxProcessImplクラスを作成することにより解決できる場合があります。
            final BentenMergeTmxProcess proc = new BentenMergeTmxProcessImpl();
            if (proc.execute(fInput) != BentenMergeTmxBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
