/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.stat.handler;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import org.xml.sax.SAXException;

import benten.cat.stat.model.TranslationRateModel;
import benten.core.model.BentenXliff;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 翻訳率取得ハンドラー。
 * <p>
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳統計取得機能: 翻訳率ビュー」に対応します。
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings("serial")
public class ShowTranslationRateHandler extends AbstractStatHandler<TranslationRateModel> {

	/** 状態 - 未翻訳 */
	private static final List<String> nottranslated = new LinkedList<String>() {
		{
			add(null);
			add(""); //$NON-NLS-1$
			add("new"); //$NON-NLS-1$
			add("needs-translation"); //$NON-NLS-1$
		}
	};

	/** 状態 - 翻訳済 */
	private static final List<String> translated = new LinkedList<String>() {
		{
			add("x-reviewed"); //$NON-NLS-1$
			add("final"); //$NON-NLS-1$
			add("signed-off"); //$NON-NLS-1$
		}
	};

	@Override
	protected TranslationRateModel processFile(final File file) throws IOException, SAXException {
		final BentenXliff xliff = BentenXliff.loadInstance(file);
		final List<BlancoXliffTransUnit> unitList = xliff.getAllTransUnitList();
		final TranslationRateModel model = new TranslationRateModel();
		model.setFile(file);

		setModel(unitList, model);

		return model;
	}

	/**
	 * モデルの値をセット。
	 * @param unitList 翻訳単位リスト
	 * @param model モデル
	 */
	protected void setModel(final List<BlancoXliffTransUnit> unitList, final TranslationRateModel model) {

		for (final BlancoXliffTransUnit unit : unitList) {
			final BlancoXliffTarget target = unit.getTarget();

			if (unit.getTranslate()) {

				final String state = target != null ? target.getState() : null;
				if (nottranslated.contains(state)) {
					// 未翻訳
					model.setNottranslated(model.getNottranslated() + 1);
				} else if (translated.contains(state)) {
					// 翻訳済
					model.setTranslated(model.getTranslated() + 1);
				} else {
					// 翻訳中
					model.setWorking(model.getWorking() + 1);
				}
			} else {
				// 翻訳対象外
				model.setUntranslate(model.getUntranslate() + 1);
			}
		}
		model.setTransUnit(unitList.size());
		model.setUntranslateRate(getRate(model.getUntranslate(), unitList.size()));
		model.setNottranslatedRate(getRate(model.getNottranslated(), unitList.size()));
		model.setWorkingRate(getRate(model.getWorking(), unitList.size()));
		model.setTranslatedRate(getRate(model.getTranslated(), unitList.size()));
	}

	/**
	 * 翻訳率の取得。
	 * @param size サイズ
	 * @param totalSize 分母となるトータル・サイズ
	 * @return 翻訳率。
	 */
	private int getRate(final int size, final int totalSize) {
		if (totalSize == 0) {
			return 0;
		}
		return (int) Math.ceil(100d * size / totalSize);
	}
}
