/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.dom;

import java.lang.reflect.Constructor;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import benten.core.CorePlugin;

/**
 * DOM 要素のデリゲート基底クラス。
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings("restriction")
public abstract class ElementDelegate {

	/**
	 * オリジナルの DOM 要素の取得。
	 * @return DOM 要素
	 */
	public Element getElement() {
		return element;
	}

	/**
	 * 要素が持つテキスト・ノードの値の取得。
	 * @return テキスト・ノードの値
	 */
	public String getValue() {
		final Node child = element.getFirstChild();
		if (child == null) {
			return ""; //$NON-NLS-1$
		}
		return child.getNodeValue();
	}

	/**
	 * テキスト・ノードの値を設定。
	 * @param value テキスト・ノードの値
	 */
	public void setValue(final String value) {
		Node child = element.getFirstChild();
		if (child == null) {
			child = domDocument.createTextNode(""); //$NON-NLS-1$
			element.appendChild(child);
		}
		child.setNodeValue(value);
	}

	/**
	 * DOM 要素デリゲートのリストを DOM ノード・リストに変換。
	 * @param list DOM 要素デリゲートのリスト
	 * @return DOM ノード・リスト
	 */
	public static List<Element> toElementList(final List<? extends ElementDelegate> list) {
		final List<Element> elementList = new LinkedList<Element>();
		for (final ElementDelegate facade : list) {
			elementList.add(facade.getElement());
		}
		return elementList;
	}

	//-------------------------------------------------------------------------------
	// protected

	/** 要素 */
	protected final Element element;

	/** DOM ドキュメント */
	protected final IDOMDocument domDocument;

	/**
	 * コンストラクター。
	 * @param domDocument dom ドキュメント
	 * @param element オリジナル DOM 要素
	 */
	protected ElementDelegate(final IDOMDocument domDocument, final Element element) {
		this.domDocument = domDocument;
		this.element = element;
	}

	/**
	 * 属性値または子要素のテキスト・ノードの値の取得。
	 * このメソッドは getValue(xpath, "") と同じです。
	 * @param xpath XPATH
	 * @return 属性値または子要素のテキスト・ノードの値
	 */
	protected String getValue(final String xpath) {
		return getValue(xpath, ""); //$NON-NLS-1$
	}

	/**
	 * 属性値または子要素のテキスト・ノードの値の取得。
	 * @param xpath XPATH
	 * @param defaultValue 値を取得するできなかった場合のデフォルト値
	 * @return 属性値または子要素のテキスト・ノードの値
	 */
	protected String getValue(final String xpath, final String defaultValue) {
		Element ele = element;
		for (final String name : xpath.split("/")) { //$NON-NLS-1$
			if (name.startsWith("@")) { //$NON-NLS-1$
				final String value = ele.getAttribute(name.substring(1));
				return value == null ? defaultValue : value;
			}
			final Element e = getChildElement(ele, name);
			if (e == null) {
				return defaultValue;
			}
			ele = e;
		}
		final Node child = ele.getFirstChild();
		if (child == null) {
			return defaultValue;
		}
		final String value = child.getNodeValue();
		return value == null ? defaultValue : value;
	}

	/**
	 * 属性値または子要素のテキスト・ノードの値を設定。
	 * @param xpath XPATH
	 * @param value テキスト・ノードの値
	 * @return 設定できた場合は true
	 */
	protected boolean setValue(final String xpath, final String value) {
		final LinkedList<String> pathList = new LinkedList<String>(Arrays.asList(xpath.split("/"))); //$NON-NLS-1$
		String attributeName = null;
		if (pathList.getLast().startsWith("@")) { //$NON-NLS-1$
			attributeName = pathList.removeLast().substring(1);
		}
		Element ele = element;
		for (final String name : pathList) {
			ele = getChildElement(ele, name);
			if (ele == null) {
				throw new IllegalArgumentException("ElementDelegate#setValue: Invalid xpath: " + xpath); //$NON-NLS-1$
			}
		}
		if (attributeName != null) {
			ele.setAttribute(attributeName, value);
		} else {
			Node child = ele.getFirstChild();
			if (child == null) {
				child = domDocument.createTextNode(""); //$NON-NLS-1$
				ele.appendChild(child);
			}
			child.setNodeValue(value);
		}
		return true;
	}

	/**
	 * タグ名を指定して、この要素直下の DOM 要素デリゲートのリストの取得。
	 * @param clazz 要素クラス
	 * @param <T> 型パラメーター
	 * @param tagName タグ名
	 * @return DOM 要素デリゲートのリスト
	 */
	protected <T extends ElementDelegate> List<T> getList(final Class<T> clazz, final String tagName) {
		final NodeList nodeList = element.getChildNodes();
		final List<T> list = new LinkedList<T>();

		final int nodeLength = nodeList.getLength();
		for (int i = 0; i < nodeLength; i++) {
			final Node node = nodeList.item(i);
			if (node instanceof Element) {
				final Element element = (Element) node;
				if (element.getTagName().equals(tagName)) {
					try {
						final Constructor<T> constructor = clazz.getConstructor(IDOMDocument.class, Element.class);
						final T facade = constructor.newInstance(domDocument, element);
						list.add(facade);
					} catch (final Exception e) {
						CorePlugin.getDefault().log(e);
					}
				}
			}
		}
		return list;
	}
	/**
	 * 指定したタグを持つノードの取得。
	 * @param tagName タグ名
	 * @return ノード。存在しない場合は null。
	 */
	protected Node getChildElement(final String tagName) {
		return getChildElement(element, tagName);
	}

	//-------------------------------------------------------------------------------
	// private

	/**
	 * 指定したタグを持つノードの取得。
	 * @param ele エレメント
	 * @param tagName タグ名
	 * @return ノード。存在しない場合は null。
	 */
	private Element getChildElement(final Element ele, final String tagName) {
		Element e = null;
		final NodeList nodeList = ele.getChildNodes();
		final int nodeLength = nodeList.getLength();
		for (int i = 0; i < nodeLength; i++) {
			final Node node = nodeList.item(i);
			if (node instanceof Element) {
				final Element element = (Element) node;
				if (element.getTagName().equals(tagName)) {
					e = (Element) node;
					break;
				}
			}
		}
		return e;
	}
}
