/*****************************************************************************
 * Copyright (c) 2004, 2007 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and
 * is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: IBM Corporation - initial API and implementation
 ****************************************************************************/
/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
/**
 * コンテキスト・メニューから用語集を追加する機能（定数 ADD_NEW_GLOSSARY_ENTRY 
 * の利用箇所参照）は、現在実装されておらず、メニュー上にも表示されません。
 */
package benten.cat.ui.viewers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.NotEnabledException;
import org.eclipse.core.commands.NotHandledException;
import org.eclipse.core.commands.common.NotDefinedException;
import org.eclipse.core.expressions.IEvaluationContext;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.jface.fieldassist.FieldDecoration;
import org.eclipse.jface.fieldassist.FieldDecorationRegistry;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IPainter;
import org.eclipse.jface.text.ITextListener;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.jface.text.TextEvent;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.jface.text.WhitespaceCharacterPainter;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.VerifyKeyListener;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IEditorActionDelegate;
import org.eclipse.ui.ISources;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.IHandlerActivation;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.navigator.ICommonMenuConstants;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;
import org.eclipse.ui.texteditor.IUpdate;

import benten.cat.glossary.ui.actions.ShowGlossaryAction;
import benten.cat.tm.ui.actions.ShowTranslationAction;
import benten.cat.ui.CatUiPlugin;
import benten.cat.ui.viewers.messages.TranslationSourceViewerMessages;

/**
 * 翻訳コンテキスト・メニューを持つソース・ビューアー。
 *
 * <UL>
 * <LI>※コンテキスト・メニューの切り取り、コピー、貼り付けの実装には
 * org.eclipse.jdt.internal.ui.preferences.EditTemplateDialog 由来のソースコードが含まれます。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class TranslationSourceViewer extends SourceViewer implements IPreferenceChangeListener {

	private static final String ADD_NEW_GLOSSARY_COMMAND_ID = "benten.cat.glossary.ui.optional.command1";

	/**
	 * 翻訳コンテキスト・メニューを持つソース・ビューアーのためのメッセージ。
	 */
	protected static final TranslationSourceViewerMessages fMsg = new TranslationSourceViewerMessages();

	/**
	 * 改行ペインター。
	 */
	private final IPainter lineBreakPainter;

	/**
	 * 翻訳アシスト装飾。
	 */
	private ControlDecoration translationAssistDecoration;

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param editable 編集可能
	 */
	public TranslationSourceViewer(final Composite parent, final boolean editable) {
		this(parent, SWT.MULTI | SWT.WRAP | SWT.BORDER | SWT.V_SCROLL);
		setEditable(editable);
		final GridData gd = new GridData(GridData.FILL_BOTH);
		gd.minimumHeight = 19;
		final Control control = getControl();
		control.setLayoutData(gd);
	}

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param styles スタイル
	 */
	protected TranslationSourceViewer(final Composite parent, final int styles) {
		super(parent, null, styles);

		setDocument(new Document());

		final Font mono = new Font(parent.getDisplay(), "Monospaced", 10, SWT.NONE); //$NON-NLS-1$
		getControl().setFont(mono);
		lineBreakPainter = new WhitespaceCharacterPainter(this);
		preferenceChange(null);
		CatUiPlugin.getDefault().addPreferenceChangeListner(this);

		initializeActions(this);
	}

	/**
	 * {@inheritDoc}
	 */
	public void preferenceChange(final PreferenceChangeEvent event) {
		changePainter(CatUiPlugin.CatUiPreference.SHOW_LINE_BREAK, lineBreakPainter);
	}

	/**
	 * ペインターの変更。
	 * @param key キー
	 * @param painter ペインター
	 */
	private void changePainter(final CatUiPlugin.CatUiPreference key, final IPainter painter) {
		final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();
		final boolean show = store.getBoolean(key.name());
		if (show) {
			addPainter(painter);
		} else {
			removePainter(painter);
		}
	}

	//-------------------------------------------------------------------------
	// SourceViewer オーバーライド

	private static final int SHOW_GLOSSARY = 201;
	private static final int SHOW_TRANSLATION = 202;
	private static final int ADD_NEW_GLOSSARY_ENTRY = 203;

	@Override
	public void configure(final SourceViewerConfiguration configuration) {

		super.configure(configuration);

		if (translationAssistDecoration != null) {
			return;
		}
		translationAssistDecoration = new ControlDecoration(getControl(), SWT.RIGHT | SWT.BOTTOM);
		translationAssistDecoration.setDescriptionText(fMsg.getLabel1());
		final FieldDecoration fieldDecoration = FieldDecorationRegistry.getDefault().getFieldDecoration(
				FieldDecorationRegistry.DEC_CONTENT_PROPOSAL);
		translationAssistDecoration.setImage(fieldDecoration.getImage());
		translationAssistDecoration.show();
	}

	@Override
	public void setEditable(final boolean editable) {

		super.setEditable(editable);

		final int color = editable ? SWT.COLOR_WHITE : SWT.COLOR_WIDGET_BACKGROUND;
		final Control control = getControl();
		control.setBackground(control.getDisplay().getSystemColor(color));

		if (translationAssistDecoration != null) {
			if (editable) {
				translationAssistDecoration.show();
			} else {
				translationAssistDecoration.hide();
			}
		}
		updateSelectionDependentActions();
	}

	@Override
	public boolean canDoOperation(final int operation) {

		if (getTextWidget() == null || (!redraws() && operation != FORMAT)) {
			return false;
		}
		if (operation == SHOW_GLOSSARY) {
			return getTextWidget().getSelectionCount() > 0;
		} else if (operation == SHOW_TRANSLATION) {
			return getTextWidget().getSelectionCount() > 0;
		} else if (operation == ADD_NEW_GLOSSARY_ENTRY) {
			return getTextWidget().getSelectionCount() > 0;
		} else if (operation == UNDO) {
			final IUndoManager undoManager = getUndoManager();
			if (undoManager != null) {
				return true;
			}
		}
		return super.canDoOperation(operation);
	}

	@Override
	public void doOperation(final int operation) {

		if (getTextWidget() == null || (!redraws() && operation != FORMAT)) {
			return;
		}
		switch (operation) {
		case SHOW_TRANSLATION:
			runAction(new ShowTranslationAction());
			break;
		case SHOW_GLOSSARY:
			runAction(new ShowGlossaryAction());
			break;
		case ADD_NEW_GLOSSARY_ENTRY:
			runCommand(ADD_NEW_GLOSSARY_COMMAND_ID);
			break;
		default:
			super.doOperation(operation);
		}
	}

	private void runCommand(String commandId) {
		IHandlerService handlerService = (IHandlerService) CatUiPlugin.getDefault().getActiveEditorPart().getSite()
				.getService(IHandlerService.class);
		IEvaluationContext evaluationContext = handlerService.getCurrentState();
		evaluationContext.addVariable(ISources.ACTIVE_EDITOR_NAME, CatUiPlugin.getDefault().getActiveEditorPart());
		evaluationContext.addVariable(ISources.ACTIVE_CURRENT_SELECTION_NAME, getSelection());
		try {
			handlerService.executeCommand(commandId, null);
		} catch (ExecutionException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (NotDefinedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (NotEnabledException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (NotHandledException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	private void runAction(final IEditorActionDelegate action) {
		action.setActiveEditor(null, CatUiPlugin.getDefault().getActiveEditorPart());
		action.selectionChanged(null, getSelection());
		action.run(null);
	}

	@Override
	protected void handleDispose() {
		super.handleDispose();
		CatUiPlugin.getDefault().removePreferenceChangeListner(this);
	}

	//-------------------------------------------------------------------------
	// この場所以降は、Benten に翻訳のためのコンテキスト・メニューを追加するためのコードです。
	// org.eclipse.jdt.internal.ui.preferences.EditTemplateDialog を由来とするものです。

	private final Map<String, TextViewerAction> fGlobalActions = new HashMap<String, TextViewerAction>(10);
	private final List<String> fSelectionActions = new ArrayList<String>(3);

	private void initializeActions(final TextViewer textViewer) {

		addTextListener(new ITextListener() {
			public void textChanged(final TextEvent event) {
				if (event.getDocumentEvent() != null)
					doSourceChanged(event.getDocumentEvent().getDocument());
			}
		});
		addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(final SelectionChangedEvent event) {
				updateSelectionDependentActions();
			}
		});
		appendVerifyKeyListener(new VerifyKeyListener() {
			public void verifyKey(final VerifyEvent event) {
				if (event.stateMask == SWT.CTRL) {
					if (event.character == ' ') {
						if (canDoOperation(CONTENTASSIST_PROPOSALS)) {
							doOperation(CONTENTASSIST_PROPOSALS);
						}
						event.doit = false;
					} else if (event.keyCode == 't') {
						if (canDoOperation(SHOW_TRANSLATION)) {
							doOperation(SHOW_TRANSLATION);
						}
						event.doit = false;
					} else if (event.keyCode == 'g') {
						if (canDoOperation(SHOW_GLOSSARY)) {
							doOperation(SHOW_GLOSSARY);
						}
						event.doit = false;
					} else if (event.keyCode == 'a') {
						if (canDoOperation(ADD_NEW_GLOSSARY_ENTRY)) {
							doOperation(ADD_NEW_GLOSSARY_ENTRY);
						}
						event.doit = false;
					}
				}
			}
		});

		final List<IHandlerActivation> handlerActivations = new ArrayList<IHandlerActivation>();
		final IHandlerService handlerService = (IHandlerService) PlatformUI.getWorkbench().getAdapter(
				IHandlerService.class);
		getControl().getShell().addDisposeListener(new DisposeListener() {
			public void widgetDisposed(final DisposeEvent e) {
				handlerService.deactivateHandlers(handlerActivations);
			}
		});
		TextViewerAction action = null;

		action = new TextViewerAction(textViewer, ITextOperationTarget.CUT);
		action.setText(fMsg.getLabel11());
		fGlobalActions.put(ITextEditorActionConstants.CUT, action);
		fSelectionActions.add(ITextEditorActionConstants.CUT);

		action = new TextViewerAction(textViewer, ITextOperationTarget.COPY);
		action.setText(fMsg.getLabel12());
		fGlobalActions.put(ITextEditorActionConstants.COPY, action);
		fSelectionActions.add(ITextEditorActionConstants.COPY);

		action = new TextViewerAction(textViewer, ITextOperationTarget.PASTE);
		action.setText(fMsg.getLabel13());
		fGlobalActions.put(ITextEditorActionConstants.PASTE, action);
		fSelectionActions.add(ITextEditorActionConstants.PASTE);

		action = new TextViewerAction(textViewer, ITextOperationTarget.SELECT_ALL);
		action.setText(fMsg.getLabel14());
		fGlobalActions.put(ITextEditorActionConstants.SELECT_ALL, action);

		action = new TextViewerAction(textViewer, CONTENTASSIST_PROPOSALS);
		action.setText(fMsg.getLabel15());
		fGlobalActions.put(String.valueOf(CONTENTASSIST_PROPOSALS), action);
		fSelectionActions.add(String.valueOf(CONTENTASSIST_PROPOSALS));

		action = new TextViewerAction(textViewer, SHOW_TRANSLATION);
		action.setText(fMsg.getLabel16());
		fGlobalActions.put(String.valueOf(SHOW_TRANSLATION), action);
		fSelectionActions.add(String.valueOf(SHOW_TRANSLATION));

		action = new TextViewerAction(textViewer, SHOW_GLOSSARY);
		action.setText(fMsg.getLabel17());
		fGlobalActions.put(String.valueOf(SHOW_GLOSSARY), action);
		fSelectionActions.add(String.valueOf(SHOW_GLOSSARY));

		action = new TextViewerAction(textViewer, ADD_NEW_GLOSSARY_ENTRY);
		action.setText("Add New Glossary Entry...");
		fGlobalActions.put(String.valueOf(ADD_NEW_GLOSSARY_ENTRY), action);
		fSelectionActions.add(String.valueOf(ADD_NEW_GLOSSARY_ENTRY));

		final MenuManager manager = new MenuManager(null, null);
		manager.setRemoveAllWhenShown(true);
		manager.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(final IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});

		final StyledText text = textViewer.getTextWidget();
		final Menu menu = manager.createContextMenu(text);
		text.setMenu(menu);
	}

	private void fillContextMenu(final IMenuManager menu) {
		menu.add(new Separator(ITextEditorActionConstants.GROUP_EDIT));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, fGlobalActions.get(ITextEditorActionConstants.CUT));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, fGlobalActions.get(ITextEditorActionConstants.COPY));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, fGlobalActions.get(ITextEditorActionConstants.PASTE));
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, fGlobalActions
				.get(ITextEditorActionConstants.SELECT_ALL));

		menu.add(new Separator(ICommonMenuConstants.GROUP_GENERATE));
		menu.appendToGroup(ICommonMenuConstants.GROUP_GENERATE, fGlobalActions.get(String
				.valueOf(CONTENTASSIST_PROPOSALS)));

		menu.add(new Separator(ICommonMenuConstants.GROUP_ADDITIONS));
		menu.appendToGroup(ICommonMenuConstants.GROUP_ADDITIONS, fGlobalActions.get(String.valueOf(SHOW_TRANSLATION)));
		menu.appendToGroup(ICommonMenuConstants.GROUP_ADDITIONS, fGlobalActions.get(String.valueOf(SHOW_GLOSSARY)));
		if (isCommandExist(ADD_NEW_GLOSSARY_COMMAND_ID)) {
			menu.appendToGroup(ICommonMenuConstants.GROUP_ADDITIONS, fGlobalActions.get(String
					.valueOf(ADD_NEW_GLOSSARY_ENTRY)));
		}
	}

	private void doSourceChanged(final IDocument document) {
		final IUndoManager undoManager = getUndoManager();
		if (undoManager != null) {
			undoManager.endCompoundChange();
		}
		updateAction(ITextEditorActionConstants.UNDO);
	}

	private void updateSelectionDependentActions() {
		final Iterator<String> iterator = fSelectionActions.iterator();
		while (iterator.hasNext())
			updateAction(iterator.next());
	}

	private void updateAction(final String actionId) {
		final IAction action = fGlobalActions.get(actionId);
		if (action instanceof IUpdate)
			((IUpdate) action).update();
	}

	private static class TextViewerAction extends Action implements IUpdate {

		private int fOperationCode = -1;
		private final ITextOperationTarget fOperationTarget;

		public TextViewerAction(final ITextViewer viewer, final int operationCode) {
			fOperationCode = operationCode;
			fOperationTarget = viewer.getTextOperationTarget();
			update();
		}

		public void update() {
			if (fOperationCode == ITextOperationTarget.REDO)
				return;
			final boolean wasEnabled = isEnabled();
			final boolean isEnabled = (fOperationTarget != null && fOperationTarget.canDoOperation(fOperationCode));
			setEnabled(isEnabled);

			if (wasEnabled != isEnabled) {
				firePropertyChange(ENABLED, wasEnabled ? Boolean.TRUE : Boolean.FALSE, isEnabled ? Boolean.TRUE
						: Boolean.FALSE);
			}
		}

		@Override
		public void run() {
			if (fOperationCode != -1 && fOperationTarget != null) {
				fOperationTarget.doOperation(fOperationCode);
			}
		}
	}

	private boolean isCommandExist(String commandId) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint("org.eclipse.ui.commands"); //$NON-NLS-1$
		for (final IExtension extension : point.getExtensions()) {
			for (final IConfigurationElement cfgElem : extension.getConfigurationElements()) {
				final String name = cfgElem.getAttribute("id"); //$NON-NLS-1$
				if (ADD_NEW_GLOSSARY_COMMAND_ID.equals(name)) {
					return true;
				}
			}
		}

		return false;
	}
}
