/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui.viewers;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.ITextListener;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.TextEvent;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.wst.sse.core.StructuredModelManager;
import org.eclipse.wst.sse.core.internal.provisional.IStructuredModel;
import org.eclipse.wst.sse.ui.internal.contentoutline.ConfigurableContentOutlinePage;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel;
import org.eclipse.wst.xml.ui.internal.tabletree.IDesignViewer;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import benten.cat.glossary.ui.editor.GlossaryConfiguration;
import benten.cat.ui.CatUiPlugin;
import benten.cat.ui.dialogs.NoteDialog;
import benten.cat.ui.dialogs.NoteDialog.NoteDialogEntry;
import benten.cat.ui.internal.tabletree.XLIFFSelectionProvider;
import benten.cat.ui.viewers.messages.TransUnitViewerMessages;
import benten.core.dom.AltTransDelegate;
import benten.core.dom.ElementDelegate;
import benten.core.dom.NoteDelegate;
import benten.core.dom.TransUnitDelegate;
import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.ui.views.AsyncSelectionChangeAdapter;
import benten.ui.views.AutoColumnWidthTableViewer;
import benten.ui.views.TableLabelProvider;
import benten.ui.views.UiToolkit;

/**
 * XLIFF エディターの翻訳単位タブに表示するビューアー。
 *
 * <UL>
 * <LI>XLIFF エディターの翻訳単位タブを実現します。
 * <LI>※ソースコード可読性の維持のために、いくつかの private メソッド、private フィールドおよびインナークラスの public メソッドの JavaDoc について記載していません。
 * </UL>
 * <p>
 * ★基本設計「翻訳支援機能: 翻訳中間形式エディター機能: 翻訳単位編集」に対応します。<br>
 * ★基本設計「翻訳支援機能: 翻訳中間形式エディター機能: 翻訳単位選択」に対応します。<br>
 * ※選択、編集が、両方ともここに該当します。<br>
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings( { "restriction" })
public class TransUnitViewer extends ContentViewer implements IDesignViewer {
	/**
	 * XLIFF エディターの翻訳単位タブに表示するビューアーのためのメッセージ。
	 */
	protected static final TransUnitViewerMessages fMsg = new TransUnitViewerMessages();

	/**
	 * 保管キー。
	 * <UL>
	 * <LI>この定義は XLIFF エディター固有のものです。
	 * </UL>
	 */
	protected static enum StoreKey {

		/** デフォルトのノート担当者 */
		DEFAULT_NOTE_FROM,

		/** state フィルター・チェックボックス展開コンポジットの展開状態 */
		EXPAND_FILTER_STATE_CHECKS,

		/** alt-trans 展開コンポジットの展開状態 */
		EXPAND_ALT_TRANS_TABLE,

		/** note リスト展開コンポジットの展開状態 */
		EXPAND_NOTE_LIST_TABLE
	}

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	protected final UiToolkit toolkit = new UiToolkit();

	/** エディター・パート。 */
	protected final IEditorPart editorPart;
	/** サッシュ。 */
	protected final SashForm sashForm;

	// 画面の構成部品。

	protected AutoColumnWidthTableViewer transUnitTableViewer;
	protected Label transUnitCountLabel;
	protected Button joinTransUnitButton;
	protected Button prevTransUnitButton;
	protected Button nextTransUnitButton;

	protected Text idText;
	protected Combo stateCombo;
	protected Button notTranslateCheck;
	protected Button updateTmCheck;

	protected SourceViewer sourceTextViewer;
	protected SourceViewer targetTextViewer;
	protected SourceViewer altSourceTextViewer;
	protected SourceViewer altTargetTextViewer;

	protected Button copyFromSourceButton;
	protected Button copyFromAltTransButton;

	protected AutoColumnWidthTableViewer altTransTableViewer;
	protected Text originLabel;
	protected ProgressBar altTransQualityBar;
	protected Label altTransCountLabel;
	protected Button prevAltTransButton;
	protected Button nextAltTransButton;

	protected AutoColumnWidthTableViewer noteTableViewer;
	protected Button newNoteButton;
	protected Button editNoteButton;
	protected Button removeNoteButton;

	// XML を操作するためのオブジェクト。

	protected TransUnitDelegate selectedTransUnit;
	protected NoteDelegate selectedNote;

	// 状態を記憶するためのフィールド。

	protected final Set<String> checkedStateItems = new HashSet<String>();
	protected boolean internalSelectionChange;

	/** 選択プロバイダー。 */
	protected final ISelectionProvider selectionProvider = new XLIFFSelectionProvider(this);

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param editorPart エディター・パーツ
	 */
	public TransUnitViewer(final Composite parent, final IEditorPart editorPart) {

		this.editorPart = editorPart;

		sashForm = new SashForm(parent, SWT.BORDER | SWT.VERTICAL | SWT.V_SCROLL | SWT.H_SCROLL);
		createTransUnitListGroup(sashForm);
		createTransUnitGroup(sashForm);
		sashForm.setWeights(new int[] { 35, 65 });

		setContentProvider(new IContentProvider() {
			public void dispose() {
			}

			public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
			}
		});
	}

	/**
	 * 「翻訳単位」グループの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitListGroup(final Composite parent) {

		final Composite c = toolkit.createComposite(parent, 1, 10);

		final Group g = new Group(c, SWT.BOLD);
		g.setLayout(toolkit.createGridLayout(1, 5));
		g.setLayoutData(new GridData(GridData.FILL_BOTH));
		g.setText(fMsg.getLabelTransUnitList());

		createStateItemGroup(g);
		createTransUnitTableViewer(g);
		createTransUnitButtons(g);
	}

	/**
	 * 「状態」グループの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createStateItemGroup(final Composite parent) {

		final PersistenceExpandableComposite expandable = new PersistenceExpandableComposite(parent, fMsg
				.getLabelStateOfFilter());
		final Composite outer = new Composite(expandable, SWT.BORDER);
		outer.setLayout(new FillLayout());

		final Composite c = toolkit.createComposite(outer, 6, 5);

		for (final String stateItem : BentenXliff.getStateItems()) {

			checkedStateItems.add(stateItem);

			final Button checkButton = new Button(c, SWT.CHECK);
			checkButton.setData(stateItem);
			checkButton.setText(stateItem.equals("") ? fMsg.getLabelFilterNone() : stateItem); //$NON-NLS-1$
			checkButton.setSelection(true);

			checkButton.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(final SelectionEvent event) {
					final Button b = (Button) event.widget;
					final String selectedStateItem = (String) b.getData();
					if (b.getSelection()) {
						checkedStateItems.add(selectedStateItem);
					} else {
						checkedStateItems.remove(selectedStateItem);
					}
					try {
						refresh();
						changeSelectionTransUnit();
						scrollTransUnitSelection(0);
					} catch (final RuntimeException e) {
						CatUiPlugin.getDefault().log(e);
					}
				}
			});
		}

		expandable.loadExpandedSetting(StoreKey.EXPAND_FILTER_STATE_CHECKS.name());
		expandable.setClient(outer);
		expandable.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				packTable(parent, e.getState(), transUnitTableViewer);
			}
		});
	}

	/**
	 * 「翻訳単位」テーブル・ビューアーの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitTableViewer(final Composite parent) {

		transUnitTableViewer = new AutoColumnWidthTableViewer(parent);
		transUnitTableViewer.addColumn(fMsg.getLabelListTitleState(), 16);
		transUnitTableViewer.addColumn(fMsg.getLabelListTitleSource(), 52);
		transUnitTableViewer.addColumn(fMsg.getLabelListTitleTarget(), 52);

		transUnitTableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(final Object element, final int columnIndex) {

				final TransUnitDelegate transUnit = new TransUnitDelegate((IDOMDocument) getInput(), (Element) element);
				switch (columnIndex) {
				case 0:
					return transUnit.getTargetState();
				case 1:
					return transUnit.getSource();
				case 2:
					return transUnit.getTarget();
				}
				return null;
			}
		});
		transUnitTableViewer.addPostSelectionChangedListener(new AsyncSelectionChangeAdapter() {
			@Override
			public void selectionChanged(final IStructuredSelection selection) {
				if (selection.isEmpty()) {
					return;
				}
				selectedTransUnit = new TransUnitDelegate((IDOMDocument) getInput(), (Element) selection
						.getFirstElement());
				changeSelectionTransUnit();
			}
		});
	}

	/**
	 * 選択翻訳単位の変更。
	 */
	protected void changeSelectionTransUnit() {
		final List<NoteDelegate> noteList = selectedTransUnit.getNoteList();
		final List<AltTransDelegate> altTransList = selectedTransUnit.getAltTransList();

		noteTableViewer.setInput(ElementDelegate.toElementList(noteList));
		idText.setText(selectedTransUnit.getId());
		stateCombo.setText(selectedTransUnit.getTargetState());
		notTranslateCheck.setSelection(!selectedTransUnit.isTranslate());
		updateTmCheck.setSelection(!selectedTransUnit.isContextGourpTmOmit());
		sourceTextViewer.getDocument().set(selectedTransUnit.getSource());
		targetTextViewer.getDocument().set(selectedTransUnit.getTarget());

		setEnabledTransUnit(true);
		updateTranslateCheck(notTranslateCheck.getSelection());

		if (altTransList.size() == 0) {
			clearSelectedAltTrans();
		} else {
			altTransTableViewer.setInput(ElementDelegate.toElementList(altTransList));
			altTransTableViewer.getTable().setSelection(0);
			setSelectedAltTrans(altTransList.get(0));
		}
		updateTransUnitButtons();
		updateAltTransButtons();
		updateNoteButtons();
		updateOutlineView(new StructuredSelection(selectedTransUnit.getElement()));
	}

	/**
	 * 「翻訳単位」グループの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitGroup(final Composite parent) {

		final Composite c = new Composite(parent, SWT.NONE);
		c.setLayout(toolkit.createGridLayout(1, 10));

		final Group g = new Group(c, SWT.NONE);
		g.setLayout(toolkit.createGridLayout(1, 5));
		g.setLayoutData(new GridData(GridData.FILL_BOTH));
		g.setText(fMsg.getLabelGroupTransUnit());

		createTransUnitAttributes(g);
		createTransUnitText(g);
		createAltTransButtons(g);
		createAltTransTableViewer(g);
		createNoteTableViewer(g);
	}

	/**
	 * 「翻訳単位」の属性の作成。翻訳対象外のチェックボックスなどが含まれる。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitAttributes(final Composite parent) {

		Composite c = toolkit.createComposite(parent, 2);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		toolkit.createLabel(c, fMsg.getLabelId(), 50);
		idText = new Text(c, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
		idText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		c = toolkit.createComposite(parent, 4);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		toolkit.createLabel(c, fMsg.getLabelState(), 50);
		stateCombo = new Combo(c, SWT.READ_ONLY);
		stateCombo.setItems(BentenXliff.getStateItems());
		stateCombo.addModifyListener(new ModifyListener() {
			public void modifyText(final ModifyEvent e) {
				if (selectedTransUnit == null || !targetTextViewer.isEditable()) {
					return;
				}
				final String state = ((Combo) e.getSource()).getText();
				if (selectedTransUnit.setTargetState(state)) {
					transUnitTableViewer.update(selectedTransUnit.getElement(), null);
				}
			}
		});

		GridData gd = new GridData();
		gd.horizontalAlignment = SWT.RIGHT;
		gd.grabExcessHorizontalSpace = true;
		notTranslateCheck = new Button(c, SWT.CHECK);
		notTranslateCheck.setLayoutData(gd);
		notTranslateCheck.setText(fMsg.getLabelOutOfTranslation());
		notTranslateCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				selectedTransUnit.setTranslate(!checked);
				updateTranslateCheck(checked);
			}
		});

		gd = new GridData();
		gd.horizontalAlignment = SWT.RIGHT;
		gd.horizontalIndent = 30;
		updateTmCheck = new Button(c, SWT.CHECK);
		updateTmCheck.setLayoutData(gd);
		updateTmCheck.setText(fMsg.getLabelApplyToTM());
		updateTmCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				selectedTransUnit.setContextGroupTmOmit(!checked);
			}
		});
	}

	/**
	 * 「翻訳単位」のテキスト部の作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitText(final Composite parent) {
		final Composite c = toolkit.createComposite(parent, 3);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		sourceTextViewer = createTextViewer(c, fMsg.getLabelSource());
		new Label(c, SWT.NONE);
		altSourceTextViewer = createTextViewer(c, fMsg.getLabelSelectedAltTransSource());

		targetTextViewer = createTargetTextViewer(c, fMsg.getLabelTarget());

		final Composite copyButtonOuter = toolkit.createComposite(c, 1);
		copyButtonOuter.setLayoutData(new GridData(GridData.FILL_VERTICAL));

		final Label label = toolkit.createLabel(copyButtonOuter, ""); //$NON-NLS-1$
		final GridData gd = new GridData();
		gd.heightHint = 16;
		label.setLayoutData(gd);

		altTargetTextViewer = createTextViewer(c, fMsg.getLabelSelectedAltTransTarget());
		copyFromAltTransButton = createCopyButton(copyButtonOuter, SWT.LEFT, altTargetTextViewer);
	}

	/**
	 * 「翻訳単位」の「コピー」ボタンの作成。
	 * @param parent 親オブジェクト。
	 * @param direction 方向。
	 * @param from コピー元。
	 * @return ボタン・オブジェクト。
	 */
	protected Button createCopyButton(final Composite parent, final int direction, final ITextViewer from) {
		final Button button = new Button(parent, SWT.ARROW | direction);
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final String copyFrom = from.getDocument().get();
				targetTextViewer.getDocument().set(copyFrom);
			}
		});
		return button;
	}

	/**
	 * 「翻訳コンテキスト・メニューを持つソース・ビューアー」をソースとして作成。
	 * @param parent 親オブジェクト。
	 * @param text ラベル・テキスト。
	 * @return ソース・ビューアー。
	 */
	protected SourceViewer createTextViewer(final Composite parent, final String text) {
		final Composite c = toolkit.createComposite(parent, 1, 1);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Label label = toolkit.createLabel(c, text, 200);
		final GridData gd = (GridData) label.getLayoutData();
		gd.heightHint = 16;
		gd.verticalAlignment = SWT.BOTTOM;

		final SourceViewer viewer = new TranslationSourceViewer(c, false);
		return viewer;
	}

	/**
	 * 「翻訳コンテキスト・メニューを持つソース・ビューアー」をターゲットとして作成。
	 * @param parent 親オブジェクト。
	 * @param text ラベル・テキスト。
	 * @return ソース・ビューアー。
	 */
	protected SourceViewer createTargetTextViewer(final Composite parent, final String text) {
		final Composite c = toolkit.createComposite(parent, 2, 1);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Label label = toolkit.createLabel(c, text, 200);
		GridData gd = (GridData) label.getLayoutData();
		gd.heightHint = 16;
		gd.verticalAlignment = SWT.BOTTOM;

		copyFromSourceButton = createCopyButton(c, SWT.DOWN, sourceTextViewer);
		copyFromSourceButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));

		final SourceViewer viewer = new TranslationSourceViewer(c, true);
		gd = (GridData) viewer.getControl().getLayoutData();
		gd.horizontalSpan = 2;
		viewer.getControl().setLayoutData(gd);

		viewer.configure(new GlossaryConfiguration());
		viewer.addTextListener(new ITextListener() {
			public void textChanged(final TextEvent event) {
				if (selectedTransUnit == null || !viewer.isEditable()) {
					return;
				}
				Display.getCurrent().asyncExec(new Runnable() {
					public void run() {
						final String target = event.getDocumentEvent().getDocument().get();
						if (selectedTransUnit.setTarget(target)) {
							transUnitTableViewer.update(selectedTransUnit.getElement(), null);
						}
					}
				});
			}
		});
		return viewer;
	}

	/**
	 * 「代替翻訳」テーブルの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createAltTransTableViewer(final Composite parent) {
		final PersistenceExpandableComposite expandable = new PersistenceExpandableComposite(parent, fMsg
				.getLabelAltTransList());

		altTransTableViewer = new AutoColumnWidthTableViewer(expandable);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleOrigin(), 12);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleQuality(), 6);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleSource(), 41);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleTarget(), 41);

		expandable.loadExpandedSetting(StoreKey.EXPAND_ALT_TRANS_TABLE.name());
		expandable.setClient(altTransTableViewer.getTable().getParent());
		expandable.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				packTable(parent, !e.getState(), altTransTableViewer);
			}
		});

		altTransTableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(final Object element, final int columnIndex) {
				if (!(element instanceof Element)) {
					return null;
				}
				final AltTransDelegate altTrans = new AltTransDelegate((IDOMDocument) getInput(), (Element) element);
				switch (columnIndex) {
				case 0:
					return altTrans.getOrigin();
				case 1:
					return altTrans.getMatchQuality();
				case 2:
					return altTrans.getSource();
				case 3:
					return altTrans.getTarget();
				}
				return null;
			}
		});

		altTransTableViewer.addPostSelectionChangedListener(new AsyncSelectionChangeAdapter() {
			@Override
			public void selectionChanged(final IStructuredSelection selection) {
				if (selection.isEmpty()) {
					return;
				}
				final AltTransDelegate altTrans = new AltTransDelegate((IDOMDocument) getInput(), (Element) selection
						.toArray()[0]);
				setSelectedAltTrans(altTrans);
				updateAltTransButtons();
				updateOutlineView(selection);
			}
		});

		// 展開したときに中身がないと高さが 0 に近いものになってしまうのを抑止
		final String[] dummyForTableHeight = { "", "", "", "", "" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		altTransTableViewer.setInput(Arrays.asList(dummyForTableHeight));
	}

	/**
	 * 「ノート」テーブル・ビューアの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createNoteTableViewer(final Composite parent) {
		final PersistenceExpandableComposite expandable = new PersistenceExpandableComposite(parent, fMsg
				.getLabelNoteList());
		final Composite c = toolkit.createComposite(expandable, 2);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		noteTableViewer = new AutoColumnWidthTableViewer(c);
		noteTableViewer.addColumn(fMsg.getLabelNoteTitleTranslator(), 20);
		noteTableViewer.addColumn(fMsg.getLabelNoteTitleContents(), 80);

		expandable.loadExpandedSetting(StoreKey.EXPAND_NOTE_LIST_TABLE.name());
		expandable.setClient(c);
		expandable.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				packTable(parent, !e.getState(), noteTableViewer);
			}
		});

		noteTableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(final Object element, final int columnIndex) {
				final NoteDelegate note = new NoteDelegate((IDOMDocument) getInput(), (Element) element);
				switch (columnIndex) {
				case 0:
					return note.getFrom();
				case 1:
					return note.getValue();
				}
				return null;
			}
		});
		noteTableViewer.addPostSelectionChangedListener(new AsyncSelectionChangeAdapter() {
			@Override
			public void selectionChanged(final IStructuredSelection selection) {
				if (!selection.isEmpty()) {
					selectedNote = new NoteDelegate((IDOMDocument) getInput(), (Element) selection.getFirstElement());
				}
				updateNoteButtons();
				updateOutlineView(selection);
			}
		});
		noteTableViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(final DoubleClickEvent event) {
				editNoteDialog();
			}
		});

		final Composite buttonOuter = toolkit.createComposite(c, 1);

		newNoteButton = toolkit.createButton(buttonOuter, fMsg.getLabelNoteNew(), 60);
		newNoteButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();
				final String from = store.getString(StoreKey.DEFAULT_NOTE_FROM.name());

				final NoteDialogEntry note = new NoteDialogEntry(from, ""); //$NON-NLS-1$
				final NoteDialog dialog = new NoteDialog(sashForm.getShell(), fMsg.getLabelNoteNewText(), note);

				if (dialog.open() == Window.OK) {
					final Node noteElement = selectedTransUnit.appendNote(note.from, note.content);
					noteTableViewer.add(noteElement);
					store.setValue(StoreKey.DEFAULT_NOTE_FROM.name(), note.from);
				}
			}
		});
		editNoteButton = toolkit.createButton(buttonOuter, fMsg.getLabelNoteEdit(), 60);
		editNoteButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				editNoteDialog();
			}
		});
		removeNoteButton = toolkit.createButton(buttonOuter, fMsg.getLabelNoteRemove(), 60);
		removeNoteButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				selectedTransUnit.removeNote(selectedNote);
				final Table table = noteTableViewer.getTable();
				final int index = table.getSelectionIndex();

				table.remove(index);
				if (index > 0) {
					noteTableViewer
							.setSelection(new StructuredSelection(noteTableViewer.getElementAt(index - 1)), true);
				} else {
					noteTableViewer.setSelection(new StructuredSelection(), true);
				}
			}
		});
	}

	/**
	 * ノートをノート・ダイアログによって編集。
	 * 
	 * <UL>
	 * <LI>選択されたノートをノートダイアログで編集します。
	 * <LI>万が一「selectedNote」が null のままこのメソッドが呼び出された場合には、何もせずに処理を戻します。(環境由来のバグ回避策)
	 * </UL>
	 */
	protected void editNoteDialog() {
		if (selectedNote == null) {
			// 環境によって selectedNote が null のまま このメソッドが呼び出される場合があります。
			// その場合には、何もせずに処理を終了します。
			return;
		}

		final NoteDialogEntry note = new NoteDialogEntry(selectedNote.getFrom(), selectedNote.getValue());
		note.modify = true;
		final NoteDialog dialog = new NoteDialog(sashForm.getShell(), fMsg.getLabelEditNote(), note);

		if (dialog.open() == Window.OK) {
			selectedNote.setFrom(note.from);
			selectedNote.setValue(note.content);
			noteTableViewer.update(selectedNote.getElement(), null);
		}
	}

	/**
	 * 翻訳単位のためのボタンの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitButtons(final Composite parent) {
		final Composite c = toolkit.createComposite(parent, 5);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		joinTransUnitButton = toolkit.createButton(c, fMsg.getLabelJoinToNextTransUnit(), 170);

		toolkit.createHorizontalSpacer(c);

		transUnitCountLabel = toolkit.createLabel(c, "", 120); //$NON-NLS-1$
		transUnitCountLabel.setAlignment(SWT.RIGHT);

		prevTransUnitButton = toolkit.createButton(c, fMsg.getLabelPrevTransUnit(), 140);
		nextTransUnitButton = toolkit.createButton(c, fMsg.getLabelNextTransUnit(), 140);

		joinTransUnitButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				selectedTransUnit.joinNextTransUnit();
				refresh();
				scrollTransUnitSelection(0);
			}
		});
		prevTransUnitButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				scrollTransUnitSelection(-1);
			}
		});
		nextTransUnitButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				scrollTransUnitSelection(1);
			}
		});
	}

	/**
	 * 翻訳単位の選択を移動。
	 * @param move 移動距離。
	 */
	protected void scrollTransUnitSelection(final int move) {
		final Table table = transUnitTableViewer.getTable();
		final int index = table.getSelectionIndex();
		final Object dest = transUnitTableViewer.getElementAt(index + move);
		if (dest != null) {
			transUnitTableViewer.setSelection(new StructuredSelection(dest), true);
		} else {
			clearSelectedTransUnit();
		}
	}

	/**
	 * 「代替翻訳」のボタンの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createAltTransButtons(final Composite parent) {

		final Composite c = toolkit.createComposite(parent, 5);
		final GridData gd0 = new GridData(GridData.FILL_HORIZONTAL);
		c.setLayoutData(gd0);

		originLabel = new Text(c, SWT.SINGLE | SWT.READ_ONLY | SWT.RIGHT);
		originLabel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final GridData gd = new GridData();
		gd.widthHint = 80;
		altTransQualityBar = new ProgressBar(c, SWT.SMOOTH);
		altTransQualityBar.setLayoutData(gd);
		altTransQualityBar.setMinimum(0);
		altTransQualityBar.setMaximum(100);

		altTransCountLabel = toolkit.createLabel(c, "", 60); //$NON-NLS-1$
		altTransCountLabel.setAlignment(SWT.RIGHT);

		prevAltTransButton = toolkit.createButton(c, fMsg.getLabelPrevAltTrans(), 140);
		nextAltTransButton = toolkit.createButton(c, fMsg.getLabelNextAltTrans(), 140);

		prevAltTransButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Table table = altTransTableViewer.getTable();
				final int index = table.getSelectionIndex();
				if (index > 0) {
					altTransTableViewer.setSelection(new StructuredSelection(altTransTableViewer
							.getElementAt(index - 1)), true);
				}
				updateAltTransButtons();
			}
		});
		nextAltTransButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Table table = altTransTableViewer.getTable();
				final int index = table.getSelectionIndex();
				if (index < table.getItemCount() - 1) {
					altTransTableViewer.setSelection(new StructuredSelection(altTransTableViewer
							.getElementAt(index + 1)), true);
				}
				updateAltTransButtons();
			}
		});
	}

	//-------------------------------------------------------------------------
	// 可視制御

	/**
	 * テーブルをパック。
	 * @param parent 親コンポジット
	 * @param addSashTopWeight サッシュ上部重み
	 * @param tableViewer テーブル・ビューアー
	 */
	protected void packTable(final Composite parent, final boolean addSashTopWeight, final TableViewer tableViewer) {

		// 翻訳ソース・ビューの横幅が崩れるの回避 (横幅の平均値をセット)
		final Point sourcePoint = sourceTextViewer.getControl().computeSize(SWT.DEFAULT, SWT.DEFAULT);
		final Point altSourcePoint = altSourceTextViewer.getControl().computeSize(SWT.DEFAULT, SWT.DEFAULT);
		final int width = (sourcePoint.x + altSourcePoint.x) / 2;
		final GridData gs = (GridData) sourceTextViewer.getTextWidget().getLayoutData();
		final GridData gt = (GridData) altSourceTextViewer.getTextWidget().getLayoutData();
		gs.widthHint = width;
		gt.widthHint = width;
		sourceTextViewer.getTextWidget().setLayoutData(gs);
		altSourceTextViewer.getTextWidget().setLayoutData(gt);

		// pack 開始
		sashForm.setVisible(false);
		parent.pack();

		// SashForm を少しだけサイズ変更して SWT サイズ自動調整に任せる
		final int[] weights = sashForm.getWeights();
		final int sum = weights[0] + weights[1];
		weights[0] += addSashTopWeight ? 10 : -10;
		weights[1] = sum - weights[0];
		sashForm.setWeights(weights);

		// 第一カラムを再セットすることで、リサイズ時に横スクロールバーが表示されるのを抑止
		// → TableColumnLayout のバグ扱いとし、対応しない 2009.07.10
		//final TableColumn column = tableViewer.getTable().getColumn(0);
		//column.setWidth(column.getWidth());

		sashForm.setVisible(true);
	}

	/**
	 * 「翻訳単位」の可視制御。
	 * @param enabled 有効化するかどうか。
	 */
	protected void setEnabledTransUnit(final boolean enabled) {
		stateCombo.setEnabled(enabled);
		notTranslateCheck.setEnabled(enabled);
		updateTmCheck.setEnabled(enabled);
		targetTextViewer.setEditable(enabled);
	}

	/**
	 * 「翻訳単位」のボタンを更新。
	 */
	protected void updateTransUnitButtons() {
		final boolean hasContinue = HelpTransUnitId.hasContinue(selectedTransUnit.getId());
		joinTransUnitButton.setEnabled(hasContinue);

		final Table table = transUnitTableViewer.getTable();
		final int index = table.getSelectionIndex();
		prevTransUnitButton.setEnabled(index > 0);
		nextTransUnitButton.setEnabled(index < table.getItemCount() - 1);

		final int row = table.getSelectionIndex() + 1;
		final String s = row + " / " + table.getItemCount(); //$NON-NLS-1$
		transUnitCountLabel.setText(s);
	}

	/**
	 * 選択した「翻訳単位」をクリアー。
	 */
	protected void clearSelectedTransUnit() {
		setEnabledTransUnit(false);

		noteTableViewer.setInput(Collections.EMPTY_LIST);
		idText.setText(""); //$NON-NLS-1$
		stateCombo.setText(""); //$NON-NLS-1$
		sourceTextViewer.getDocument().set(""); //$NON-NLS-1$
		targetTextViewer.getDocument().set(""); //$NON-NLS-1$

		clearSelectedAltTrans();
		updateAltTransButtons();
	}

	/**
	 * 選択した「代替翻訳」をクリアー。
	 */
	protected void clearSelectedAltTrans() {
		altSourceTextViewer.getDocument().set(""); //$NON-NLS-1$
		altTargetTextViewer.getDocument().set(""); //$NON-NLS-1$
		altTransTableViewer.setInput(Collections.EMPTY_LIST);
		originLabel.setText(""); //$NON-NLS-1$
		altTransQualityBar.setVisible(false);
		altTransCountLabel.setText(""); //$NON-NLS-1$
	}

	/**
	 * 選択した「代替翻訳」をセット。
	 * @param altTrans 代替翻訳。
	 */
	protected void setSelectedAltTrans(final AltTransDelegate altTrans) {
		altSourceTextViewer.getDocument().set(altTrans.getSource());
		altTargetTextViewer.getDocument().set(altTrans.getTarget());

		final String quality = altTrans.getMatchQuality();
		try {
			altTransQualityBar.setSelection(Integer.parseInt(quality.replace("%", ""))); //$NON-NLS-1$ //$NON-NLS-2$
			altTransQualityBar.setVisible(true);
		} catch (final RuntimeException e) {
			altTransQualityBar.setSelection(0);
			altTransQualityBar.setVisible(false);
		}

		String text = altTrans.getOrigin();
		if (quality != null && !quality.equals("")) { //$NON-NLS-1$
			text += ": " + quality; //$NON-NLS-1$
		}
		originLabel.setText(text);

		final Table table = altTransTableViewer.getTable();
		final int row = table.getSelectionIndex() + 1;
		final String s = row + " / " + table.getItemCount(); //$NON-NLS-1$
		altTransCountLabel.setText(s);
	}

	/**
	 * 「代替翻訳」のボタンを更新。
	 */
	protected void updateAltTransButtons() {
		final Table table = altTransTableViewer.getTable();
		final int index = table.getSelectionIndex();
		prevAltTransButton.setEnabled(index > 0);
		nextAltTransButton.setEnabled(index < table.getItemCount() - 1);
	}

	/**
	 * 「ノート」のボタンを更新。
	 */
	protected void updateNoteButtons() {
		final ISelection selection = noteTableViewer.getSelection();
		editNoteButton.setEnabled(!selection.isEmpty());
		removeNoteButton.setEnabled(!selection.isEmpty());
	}

	/**
	 * 「翻訳」関連ボタンの更新。
	 * @param selection 選択
	 */
	protected void updateTranslateCheck(final boolean selection) {
		stateCombo.setEnabled(!selection);
		targetTextViewer.setEditable(!selection);
		copyFromSourceButton.setEnabled(!selection);
		copyFromAltTransButton.setEnabled(!selection);
	}

	/**
	 * 「アウトライン」ビューの更新。
	 * @param selection 選択。
	 */
	protected void updateOutlineView(final ISelection selection) {
		if (internalSelectionChange) {
			return;
		}
		final ConfigurableContentOutlinePage outline = (ConfigurableContentOutlinePage) editorPart
				.getAdapter(IContentOutlinePage.class);
		if (outline.getConfiguration().isLinkedWithEditor(null)) {
			outline.setSelection(selection);
		}
	}

	//-------------------------------------------------------------------------
	// org.eclipse.wst.xml.ui.internal.tabletree.IDesignViewer の実装

	/**
	 * CTabItem 上に表示するコントロールを取得。
	 * @return コントロール
	 */
	@Override
	public Control getControl() {
		return sashForm;
	}

	/**
	 * 選択プロバイダーを取得。
	 * @return 選択プロバイダー
	 */
	public ISelectionProvider getSelectionProvider() {
		return selectionProvider;
	}

	/**
	 * ビューアーのタブ・タイトルを取得。
	 * @return タブ・タイトル
	 */
	public String getTitle() {
		return fMsg.getViewerTitle();
	}

	/**
	 * ドキュメントを設定。
	 * @param document ドキュメント
	 */
	public void setDocument(final IDocument document) {

		IStructuredModel model = null;
		try {
			model = StructuredModelManager.getModelManager().getExistingModelForRead(document);
			if (model != null) {
				final IDOMDocument domDoc = ((IDOMModel) model).getDocument();

				// ファイルを開いたときにリフレッシュが必要な状態
				if (domDoc.getFirstChild() == null) {
					final IFile file = ((IFileEditorInput) editorPart.getEditorInput()).getFile();
					file.refreshLocal(0, null);
					model = StructuredModelManager.getModelManager().getModelForRead(file);
				}

				setInput(domDoc);
				refresh();

				// UNDO マネージャーの開始
				model.getUndoManager().disableUndoManagement();
				model.getUndoManager().enableUndoManagement();

				// UNDO は XML エディター側の機能を利用するため、ドキュメントの変更が発生した場合は
				// 翻訳単位タブに反映するようにリスナーを設定
				// → 入力したとときにカーソルが移動する、コンテンツ・アシストで 1 文字ずれるなどの
				//    現象が発生するため、廃止。
				//document.removeDocumentListener(documentListener);
				//document.addDocumentListener(documentListener);
			}

		} catch (final IOException e) {
			CatUiPlugin.getDefault().log(e);

		} catch (final CoreException e) {
			CatUiPlugin.getDefault().log(e);

		} finally {
			if (model != null) {
				model.releaseFromRead();
			}
		}
	}

	/**
	 * ドキュメント・リスナー (UNDO 反映)。
	 */
	protected final IDocumentListener documentListener = new IDocumentListener() {

		public void documentAboutToBeChanged(final DocumentEvent event) {
		}

		public void documentChanged(final DocumentEvent event) {
			final int caretOffset = targetTextViewer.getTextWidget().getCaretOffset();
			refresh();
			changeSelectionTransUnit();
			targetTextViewer.getTextWidget().setCaretOffset(caretOffset);
		}
	};

	//-------------------------------------------------------------------------
	// org.eclipse.jface.viewers.Viewer オーバーライド

	/**
	 * このビューアーをリフレッシュ。
	 *
	 * <UL>
	 * <LI>LabelProvider や ContentProvider が設定されたときなどに呼び出されます。
	 * </UL>
	 */
	@Override
	public void refresh() {

		final IDOMDocument domDoc = (IDOMDocument) getInput();
		if (domDoc == null) {
			return;
		}

		final List<TransUnitDelegate> allTransUnitList = TransUnitDelegate.listOf(domDoc);
		final List<TransUnitDelegate> transUnitList = new LinkedList<TransUnitDelegate>();

		for (final TransUnitDelegate transUnit : allTransUnitList) {
			final String state = transUnit.getTargetState();
			if (checkedStateItems.contains(state)) {
				transUnitList.add(transUnit);
			}
		}
		transUnitTableViewer.setInput(ElementDelegate.toElementList(transUnitList));

		if (transUnitList.size() == 0) {

			clearSelectedTransUnit();

		} else if (transUnitTableViewer.getSelection().isEmpty()) {

			final Object firstElement = transUnitTableViewer.getElementAt(0);
			final StructuredSelection sel = new StructuredSelection(firstElement);
			transUnitTableViewer.setSelection(sel, true);
			final IContentOutlinePage outline = (IContentOutlinePage) editorPart.getAdapter(IContentOutlinePage.class);
			if (outline != null) {
				outline.setSelection(sel);
			}

			selectedTransUnit = new TransUnitDelegate((IDOMDocument) getInput(), (Element) firstElement);
			final List<NoteDelegate> noteList = selectedTransUnit.getNoteList();
			if (noteList.size() == 0) {
				selectedNote = null;
			} else {
				selectedNote = new NoteDelegate((IDOMDocument) getInput(), noteList.get(0).getElement());
			}
		}
	}

	/**
	 * このビューアーの選択を取得。
	 *
	 * <UL>
	 * <LI>このビューアーがアクティブになった時に呼び出されます。
	 * <LI>この選択はアウトライン・ビューなどに反映されます。
	 * </UL>
	 */
	@Override
	public ISelection getSelection() {

		refresh();

		CatUiPlugin.getDefault().setActiveEditorPart(editorPart);

		ISelection selection = transUnitTableViewer.getSelection();
//		ISelection selection = altTransTableViewer.getSelection();
//		if (selection.isEmpty()) {
//			selection = transUnitTableViewer.getSelection();
//		}
		return selection;
	}

	/**
	 * このビューアーの選択を設定。
	 *
	 * <UL>
	 * <LI>ソース・タブやアウトライン・ビュー選択時に呼び出されます。
	 * </UL>
	 */
	@Override
	public void setSelection(final ISelection selection, final boolean reveal) {

		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {

			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			final Node firstElement = (Node) structuredSelection.getFirstElement();
			final Node node = firstElement;
			Node transUnitNode = null;
			Node altTransNode = null;
			Node noteNode = null;

			for (Node n = node; n != null; n = n.getParentNode()) {
				final String nodeName = n.getNodeName();
				if (noteNode == null && nodeName.equals("note")) { //$NON-NLS-1$
					noteNode = n;
				} else if (altTransNode == null && nodeName.equals("alt-trans")) { //$NON-NLS-1$
					altTransNode = n;
				} else if (transUnitNode == null && nodeName.equals("trans-unit")) { //$NON-NLS-1$
					transUnitNode = n;
					break;
				}
			}
			try {
				internalSelectionChange = true;
				if (transUnitNode != null) {
					final ISelection sel = new StructuredSelection(transUnitNode);
					transUnitTableViewer.setSelection(sel, true);
				}
				if (altTransNode != null) {
					final ISelection sel = new StructuredSelection(altTransNode);
					altTransTableViewer.setSelection(sel, true);
				}
				if (noteNode != null) {
					final ISelection sel = new StructuredSelection(noteNode);
					noteTableViewer.setSelection(sel, true);
				}
			} finally {
				internalSelectionChange = false;
			}
		}
	}
	
	public String getAltSourceTextViewerText(){
		return altSourceTextViewer.getTextWidget().getText();
	}

	public String getAltTargetTextViewerText(){
		return altTargetTextViewer.getTextWidget().getText();
	}
}
