/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.ui.jobs;

import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.swt.widgets.Display;

import benten.twa.TwaPlugin;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.ui.jobs.messages.WorkflowJobMessages;
import benten.ui.UiMultiStatus;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;
import benten.ui.jobs.BentenSchedulingRule;

/**
 * 翻訳ワークフロー・ジョブ。
 *
 * @author KASHIHARA Shinji
 */
public class WorkflowJob extends Job {
	/**
	 * 翻訳ワークフロー・ジョブのためのメッセージ。
	 */
	protected static final WorkflowJobMessages fMsg = new WorkflowJobMessages();

	private AbstractWorkflowHandler handler;
	private IStatus status;
	private String completDialogTitle;
	private String completeMessage;

	/**
	 * コンストラクター。
	 * @param name ジョブ名
	 */
	public WorkflowJob(final String name) {
		super(name);
		setUser(true);

		// ワークフロー・ジョブの同時実行はすべてブロック
		setRule(BentenSchedulingRule.INSTANCE);
	}

	/**
	 * ステータスのセット。
	 * @param status ステータス
	 */
	public void setStatus(final IStatus status) {
		this.status = status;
	}

	/**
	 * ハンドラーのセット。
	 * @param handler ハンドラー
	 */
	public void setHandler(final AbstractWorkflowHandler handler) {
		this.handler = handler;
	}

	@Override
	protected IStatus run(final IProgressMonitor monitor) {
		handler.monitor = monitor;

		try {
			// ハンドラー実行
			try {
				handler.run(status);
			} catch (final Exception e) {
				// 最初にロギングをおこなう。
				TwaPlugin.getDefault().log(e);

				// ロギング後、メッセージを表示したいが、これは外側のコードで実施する。
				throw new IllegalArgumentException(e);
			}

			// ユーザーによるキャンセル
			if (monitor.isCanceled()) {
				return Status.CANCEL_STATUS;
			}

		} finally {

			// プロジェクトのリフレッシュ。
			// 異常終了時、キャンセル時でもリフレッシュは行う。
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					if (status instanceof UiStatus) {
						final UiStatus uiStatus = (UiStatus) status;
						if (uiStatus.getProject() != null) {
							try {
								uiStatus.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
							} catch (final CoreException e) {
								TwaPlugin.getDefault().log(e);
							}
						}
					}
				}
			});
		}

		// 完了メッセージ表示
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				showCompleteMessage();
			}
		});

		// ハンドラー固有の完了処理
		handler.complete(status);

		return Status.OK_STATUS;
	}

	/**
	 * この処理の実行結果情報を取得。
	 * @return 処理結果情報
	 */
	public BentenProcessResultInfo getResultInfo() {
		return handler.getResultInfo();
	}

	/**
	 * 完了メッセージの表示。
	 */
	protected void showCompleteMessage() {

		// ダイアログに表示するメッセージの作成
		final StringBuffer sb = new StringBuffer();
		sb.append(getCompleteMessage());
		sb.append("\n"); //$NON-NLS-1$
		sb.append(BentenTwaProcessUtil.getResultMessage(getResultInfo()));

		// 詳細ボタン押下時のメッセージを作成
		IStatus s = new UiStatus(IStatus.INFO, sb.toString());
		final List<String> messageList = getResultInfo().getMessageList();
		if (messageList.size() > 0) {
			final UiMultiStatus multiStatus = new UiMultiStatus(sb.toString());
			for (final String message : getResultInfo().getMessageList()) {
				multiStatus.add(new UiStatus(IStatus.INFO, message));
			}
			s = multiStatus;
		}

		// ダイアログ・オープン
		UiPlugin.openDialog(getCompleteDialogTitle(), s);
	}

	/**
	 * 完了メッセージのセット。
	 * @param dialogTitle ダイアログ・タイトル
	 * @param message メッセージ
	 */
	public void setCompleteMessage(final String dialogTitle, final String message) {
		this.completDialogTitle = dialogTitle;
		this.completeMessage = message;
	}

	/**
	 * 完了ダイアログのタイトルを取得。<br>
	 * セットされていない場合はデフォルトのタイトルを返します。
	 * 現状、ウィザードでは個別、ダイアログではデフォルトのタイトルが使用されています。
	 *
	 * @return 完了ダイアログのタイトル
	 */
	protected String getCompleteDialogTitle() {
		return completDialogTitle == null ? fMsg.getMsgDefaultCompleteMessage() : completDialogTitle;
	}

	/**
	 * 完了メッセージの取得。<br>
	 * セットされていない場合はデフォルトのメッセージを返します。
	 * 現状、ウィザードでは個別、ダイアログではデフォルトのメッセージが使用されています。
	 *
	 * @return 完了メッセージ
	 */
	protected String getCompleteMessage() {
		return completeMessage == null ? fMsg.getMsgDefaultCompleteMessage() : completeMessage;
	}
}
