/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.ui.preference;

import java.io.File;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.ui.IWorkbench;

import benten.cat.tm.ui.CatTmUiPlugin;
import benten.cat.tm.ui.preference.messages.CatTmUiPreferencePageMessages;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;
import benten.ui.fields.AbstractBrowseTextField;
import benten.ui.fields.FolderTextField;
import benten.ui.preference.AbstractPreferencePage;
import benten.ui.preference.BentenPreference.Preference;
import benten.ui.views.UiToolkit;

/**
 * TM 設定ページ。
 *
 * <UL>
 * <LI>TM に関する設定を行う設定ページ・クラスです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class CatTmUiPreferencePage extends AbstractPreferencePage {

	/**
	 * TM 設定ページのためのメッセージ。
	 */
	protected static final CatTmUiPreferencePageMessages fMsg = new CatTmUiPreferencePageMessages();

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	protected final UiToolkit toolkit = new UiToolkit();

	/** TM 名コンボ */
	protected Combo tmNameCombo;

	/** TMX デフォルト使用チェックボックス */
	protected Button useTmxDefaultCheck;

	/** TMX ロケーション使用チェックボックス */
	protected Button useTmxLocationCheck;

	/** TMX ロケーション・テキスト */
	protected AbstractBrowseTextField tmxLocationText;

	/** TM 参照時にホワイトスペースを無視するチェックボックス */
	protected Button ignoreWhitespaceTmReferenceCheck;

	/** TMX 抽出時にホワイトスペースを無視するチェックボックス */
	protected Button ignoreWhitespaceTmxExtractCheck;

	/** TM 参照時にニーモニック・キーを無視するチェックボックス */
	protected Button ignoreMnemonicKeyTmReferenceCheck;

	/** TMX 抽出時にニーモニック・キーを無視するチェックボックス */
	protected Button ignoreMnemonicKeyTmxExtractCheck;

	/**
	 * コンストラクター。
	 */
	public CatTmUiPreferencePage() {
	}

	/**
	 * コンストラクター。
	 * @param title タイトル
	 */
	public CatTmUiPreferencePage(final String title) {
		super(title);
	}

	/**
	 * コンストラクター。
	 * @param title タイトル
	 * @param image イメージ
	 */
	public CatTmUiPreferencePage(final String title, final ImageDescriptor image) {
		super(title, image);
	}

	@Override
	protected Control createContents(final Composite parent) {
		final Composite composite = toolkit.createComposite(parent, 1);
		final IPreferenceStore store = getPreferenceStore();

		final Composite c = toolkit.createComposite(composite, 2);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final Set<String> tmNameSet = CatTmUiPlugin.getDefault().getTmEnginePluginNameSet();
		toolkit.createLabel(c, fMsg.getLabel0());
		tmNameCombo = new Combo(c, SWT.READ_ONLY);
		tmNameCombo.setItems(tmNameSet.toArray(new String[tmNameSet.size()]));
		tmNameCombo.setText(store.getString(Preference.TM_NAME.name()));

		createTmxGroup(composite);
		createWhitespaceGroup(composite);
		createMnemonicKeyGroup(composite);

		return composite;
	}

	/**
	 * TMX 関連グループの作成。
	 * @param parent 親コンポジット
	 */
	protected void createTmxGroup(final Composite parent) {
		final IPreferenceStore store = getPreferenceStore();

		final Group g = new Group(parent, SWT.BOLD);
		g.setLayout(toolkit.createGridLayout(4, 5));
		g.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		g.setText(fMsg.getLabel1());

		final GridData gd = new GridData();
		gd.horizontalSpan = 4;
		useTmxDefaultCheck = new Button(g, SWT.RADIO);
		useTmxDefaultCheck.setText(fMsg.getLabel2());
		useTmxDefaultCheck.setSelection(store.getBoolean(Preference.USE_TMX_DEFAULT.name()));
		useTmxDefaultCheck.setLayoutData(gd);
		useTmxDefaultCheck.addSelectionListener(new TmxSelectionAdapter());

		useTmxLocationCheck = new Button(g, SWT.RADIO);
		useTmxLocationCheck.setText(fMsg.getLabel3());
		useTmxLocationCheck.setSelection(store.getBoolean(Preference.USE_TMX_LOCATION.name()));
		useTmxLocationCheck.addSelectionListener(new TmxSelectionAdapter());

		tmxLocationText = new FolderTextField(g);
		tmxLocationText.setWidthHint(SWT.DEFAULT);
		tmxLocationText.setText(store.getString(Preference.TMX_LOCATION.name()));

		if (!useTmxDefaultCheck.getSelection() && !useTmxLocationCheck.getSelection()) {
			useTmxDefaultCheck.setSelection(true);
		}
		updateTmxCheck();
	}

	/**
	 * TMX 選択アダプター。
	 */
	protected class TmxSelectionAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(final SelectionEvent e) {
			updateTmxCheck();
		}
	}

	/**
	 * TMX チェックボックス変更による更新。
	 */
	protected void updateTmxCheck() {
		tmxLocationText.setEnabled(useTmxLocationCheck.getSelection());
	}

	/**
	 * ホワイトスペース関連グループの作成。
	 * @param parent 親コンポジット
	 */
	protected void createWhitespaceGroup(final Composite parent) {
		final IPreferenceStore store = getPreferenceStore();

		final Group g = new Group(parent, SWT.BOLD);
		g.setLayout(toolkit.createGridLayout(1, 5));
		g.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		g.setText(fMsg.getLabel11());

		ignoreWhitespaceTmReferenceCheck = new Button(g, SWT.CHECK);
		ignoreWhitespaceTmReferenceCheck.setText(fMsg.getLabel12());
		ignoreWhitespaceTmReferenceCheck.setSelection(store
				.getBoolean(Preference.IGNORE_WHITESPACE_TM_REFERENCE.name()));

		ignoreWhitespaceTmxExtractCheck = new Button(g, SWT.CHECK);
		ignoreWhitespaceTmxExtractCheck.setText(fMsg.getLabel13());
		ignoreWhitespaceTmxExtractCheck.setSelection(store.getBoolean(Preference.IGNORE_WHITESPACE_TMX_EXTRACT.name()));
	}

	/**
	 * ニーモニック・キー関連グループの作成。
	 * @param parent 親コンポジット
	 */
	protected void createMnemonicKeyGroup(final Composite parent) {
		final IPreferenceStore store = getPreferenceStore();

		final Group g = new Group(parent, SWT.BOLD);
		g.setLayout(toolkit.createGridLayout(1, 5));
		g.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		g.setText(fMsg.getLabelMnemonicKeyGroup());

		ignoreMnemonicKeyTmReferenceCheck = new Button(g, SWT.CHECK);
		ignoreMnemonicKeyTmReferenceCheck.setText(fMsg.getLabelMnemonicKeyIgnoreTmReference());
		ignoreMnemonicKeyTmReferenceCheck.setSelection(store.getBoolean(Preference.IGNORE_MNEMONIC_KEY_TM_REFERENCE
				.name()));

		ignoreMnemonicKeyTmxExtractCheck = new Button(g, SWT.CHECK);
		ignoreMnemonicKeyTmxExtractCheck.setText(fMsg.getLabelMnemonicKeyIgnoreTmxExtract());
		ignoreMnemonicKeyTmxExtractCheck.setSelection(store.getBoolean(Preference.IGNORE_MNEMONIC_KEY_TMX_EXTRACT
				.name()));
	}

	/**
	 * {@inheritDoc}
	 */
	public void init(final IWorkbench workbench) {
	}

	@Override
	public boolean performOk() {
		final IPreferenceStore store = getPreferenceStore();
		final String newTmName = tmNameCombo.getText();
		final boolean newUseTmxLocation = useTmxLocationCheck.getSelection();

		// TMX ロケーションのチェック
		if (newUseTmxLocation) {
			if (tmxLocationText.getText().equals("")) { //$NON-NLS-1$
				final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError0());
				UiPlugin.openDialog(fMsg.getDlgError1(), s);
				return false;
			}
			final File tmxDir = tmxLocationText.getFile();
			if (!tmxDir.exists() || tmxDir.isFile()) {
				final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError2());
				UiPlugin.openDialog(fMsg.getDlgError3(), s);
				return false;
			}
		}

		// 設定が load メソッド内で参照されるため、先に保管
		store.setValue(Preference.TM_NAME.name(), newTmName);
		store.setValue(Preference.USE_TMX_DEFAULT.name(), useTmxDefaultCheck.getSelection());
		store.setValue(Preference.USE_TMX_LOCATION.name(), useTmxLocationCheck.getSelection());
		store.setValue(Preference.TMX_LOCATION.name(), tmxLocationText.getText());
		store.setValue(Preference.IGNORE_WHITESPACE_TM_REFERENCE.name(), ignoreWhitespaceTmReferenceCheck
				.getSelection());
		store.setValue(Preference.IGNORE_WHITESPACE_TMX_EXTRACT.name(), ignoreWhitespaceTmxExtractCheck.getSelection());
		store.setValue(Preference.IGNORE_MNEMONIC_KEY_TM_REFERENCE.name(), ignoreMnemonicKeyTmReferenceCheck
				.getSelection());
		store.setValue(Preference.IGNORE_MNEMONIC_KEY_TMX_EXTRACT.name(), ignoreMnemonicKeyTmxExtractCheck
				.getSelection());

		try {
			CatTmUiPlugin.getDefault().loadExtensions();
		} catch (final IllegalArgumentException e) {
			final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError4(e.getMessage()));
			UiPlugin.openDialog(fMsg.getDlgError5(), s);
			return false;
		}
		return true;
	}

	@Override
	protected void performDefaults() {
		final IPreferenceStore store = getPreferenceStore();
		tmNameCombo.setText(store.getDefaultString(Preference.TM_NAME.name()));

		useTmxDefaultCheck.setSelection(store.getDefaultBoolean(Preference.USE_TMX_DEFAULT.name()));
		useTmxLocationCheck.setSelection(store.getDefaultBoolean(Preference.USE_TMX_LOCATION.name()));
		tmxLocationText.setText(store.getDefaultString(Preference.TMX_LOCATION.name()));

		ignoreWhitespaceTmReferenceCheck.setSelection(store.getDefaultBoolean(Preference.IGNORE_WHITESPACE_TM_REFERENCE
				.name()));
		ignoreWhitespaceTmxExtractCheck.setSelection(store.getDefaultBoolean(Preference.IGNORE_WHITESPACE_TMX_EXTRACT
				.name()));
		ignoreMnemonicKeyTmReferenceCheck.setSelection(store
				.getDefaultBoolean(Preference.IGNORE_MNEMONIC_KEY_TM_REFERENCE.name()));
		ignoreMnemonicKeyTmxExtractCheck.setSelection(store
				.getDefaultBoolean(Preference.IGNORE_MNEMONIC_KEY_TMX_EXTRACT.name()));
		updateTmxCheck();
	}

	@Override
	protected void setEnabled(final boolean enabled) {
		tmNameCombo.setEnabled(enabled);
		useTmxDefaultCheck.setEnabled(enabled);
		useTmxLocationCheck.setEnabled(enabled);
		tmxLocationText.setEnabled(enabled);
		ignoreWhitespaceTmReferenceCheck.setEnabled(enabled);
		ignoreWhitespaceTmxExtractCheck.setEnabled(enabled);
	}
}
