/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.model;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;

import org.xml.sax.SAXException;

import benten.core.io.Files;
import blanco.html.parser.util.BlancoHtmlParserUtil;

/**
 * HTML インライン・テキスト・ビルダー。
 *
 * <UL>
 * <LI>HTML ファイルからインライン・テキストを抽出、置換するためのクラスです。
 * <LI>抽出されるテキストに HTML ブロック・タグが含まれることはありません。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class HtmlInlineTextBuilder implements Iterable<HtmlInlineTextBuilder.InlineText> {

	/** 入力文字列 */
	private final String inputString;

	/** HTML ブロック・マッチャー */
	private final HtmlBlockMatcher htmlBlockMatcher;

	/** 出力文字列バッファー */
	private final StringBuilder outputBuffer = new StringBuilder();

	/**
	 * コンストラクター。
	 * @param file ファイル
	 * @throws IOException 入出力例外が発生した場合
	 */
	public HtmlInlineTextBuilder(final File file) throws IOException {
		this(load(file));
	}

	/**
	 * コンストラクター。
	 * @param inputString 入力文字列
	 */
	protected HtmlInlineTextBuilder(final String inputString) {
		this.inputString = inputString;
		htmlBlockMatcher = new HtmlBlockMatcher(inputString);
	}

	/**
	 * HTML ファイルのロード。
	 * @param file ファイル
	 * @return ロードしたファイルの内容文字列
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected static String load(final File file) throws IOException {
		final byte[] bytes = Files.readFileToByteArray(file);
		String encoding;
		try {
			encoding = BlancoHtmlParserUtil.decideEncoding(bytes);
		} catch (final SAXException e) {
			throw new IllegalArgumentException(e);
		}
		return new String(bytes, encoding);
	}

	/**
	 * インライン・テキストのイテレーターを取得。
	 * @return インライン・テキストを要素に持つイテレーター
	 */
	public Iterator<InlineText> iterator() {
		return new Iterator<InlineText>() {

			private InlineText inlineText;
			private int previousEnd;
			private boolean insideHtmlComment;

			public boolean hasNext() {
				if (!htmlBlockMatcher.find(previousEnd)) {
					outputBuffer.append(inputString.substring(previousEnd, inputString.length()));
					return false;
				}
				outputBuffer.append(inputString.substring(previousEnd, htmlBlockMatcher.start()));

				final HtmlBlock block = new HtmlBlock(htmlBlockMatcher.group());
				if (!block.isBlockTag()) {
					throw new IllegalArgumentException(
							"HtmlInlineTextBuilder#iterator: System Error：" + htmlBlockMatcher.group()); //$NON-NLS-1$
				}

				// HTML コメント判定
				if (block.containesHtmlCommentEndTag()) {
					insideHtmlComment = false;
				}
				if (insideHtmlComment) {
					return hasNext(block);
				}
				if (block.containesHtmlCommentStartTag()) {
					insideHtmlComment = true;
				}

				// インライン・テキストの取得
				final String text = block.getInlineText();

				if (text.equals("")) { //$NON-NLS-1$
					return hasNext(block);
				}
				final int index = block.startTag().length() + block.body().indexOf(text);
				outputBuffer.append(block.toString().substring(0, index));
				previousEnd = htmlBlockMatcher.start() + index + text.length();

				inlineText = new InlineText(text);
				return true;
			}

			public boolean hasNext(final HtmlBlock block) {
				outputBuffer.append(block.startTag() + block.body());
				previousEnd = htmlBlockMatcher.start() + block.endTagIndex();
				return hasNext();
			}

			public InlineText next() {
				return inlineText;
			}

			public void remove() {
				throw new UnsupportedOperationException();
			}
		};
	}

	/**
	 * インライン・テキストにセットした結果文字列を取得。
	 * @param encoding エンコーディング
	 * @return 結果文字列
	 */
	protected String getResultString(final String encoding) {
		// HTML の meta charset の値を指定されたエンコーディングに置換 (HTML が正規化されている前提)
		return outputBuffer.toString().replaceFirst("(?i)(?s)(text/html; charset=)[^\"]+", "$1" + encoding); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * インライン・テキストにセットした内容を指定したファイルに保管。
	 * @param outStream 出力ストリーム
	 * @param encoding エンコーディング
	 * @throws IOException 入出力例外が発生した場合
	 */
	public void writeTo(final OutputStream outStream, final String encoding) throws IOException {
		final String s = getResultString(encoding);
		Files.write(s, outStream, encoding);
	}

	/**
	 * インライン・テキスト・クラス。
	 */
	public class InlineText {

		private final String text;

		/**
		 * コンストラクター。
		 * @param text テキスト
		 */
		public InlineText(final String text) {
			this.text = text;
		}

		/**
		 * 置換文字列のセット。
		 *
		 * <UL>
		 * <LI>翻訳後の文字列をセットします。
		 * </UL>
		 *
		 * @param replacement 置換文字列
		 */
		public void setText(final String replacement) {
			outputBuffer.append(replacement.equals(getText()) ? text : replacement);
		}

		/**
		 * テキストの取得。
		 *
		 * <UL>
		 * <LI>翻訳対象となるテキスト文字列を取得します。
		 * </UL>
		 *
		 * @return テキスト
		 */
		public String getText() {
			return text;
		}
	}
}
