/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.File;
import java.io.IOException;

import benten.core.BentenConstants;
import benten.core.io.Files;
import benten.core.io.XliffFileFilter;
import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.valueobject.BentenConvertToXliffProcessInput;
import benten.twa.filter.messages.BentenConvertToXliffMessages;
import benten.twa.filter.model.HtmlInlineTextBuilder;
import benten.twa.filter.model.SentencePartitionList;
import benten.twa.filter.model.HtmlInlineTextBuilder.InlineText;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.xliff.BlancoXliffUtil;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffHeader;
import blanco.xliff.valueobject.BlancoXliffTool;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 翻訳対象を XLIFF に変換
 *
 * <pre>
 * 翻訳対象物を XLIFF に変換します。
 *   1.  プロジェクト内にインポートされた翻訳対象物を、XLIFF に変換します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTML－XLIFF変換機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class BentenConvertToXliffProcessImpl extends AbstractTraverseDir implements BentenConvertToXliffProcess {
	/**
	 * HTML－XLIFF変換機能のためのメッセージ。
	 */
	protected static final BentenConvertToXliffMessages fMsg = new BentenConvertToXliffMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenConvertToXliffProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenConvertToXliffProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenConvertToXliffProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenConvertToXliffProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirSource = new File(fInput.getSourcedir());
		if (dirSource.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getSourcedir()));
		}
		if (dirSource.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getSourcedir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenConvertToXliffProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				fCounter++;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
				// こちらはカウントしません。
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(dirSource);
		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirSource);

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".html") || file.getName().toLowerCase().endsWith(".htm")) { //$NON-NLS-1$ //$NON-NLS-2$
			return BentenTwaFilterHtmlUtil.isHtmlTagExist(file);
		} else {
			return false;
		}
	}

	@Override
	public void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenConvertToXliffProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		final File destFile = new File(fInput.getTargetdir(), Files.relativePath(new File(fInput.getSourcedir()), file));
		final File destDir = destFile.getParentFile();

		// 該当ディレクトリーが無い場合には、ディレクトリー作成。
		// すでに存在する場合は、それはそれで良い。

		if (destDir.exists() == false) {
			if (destDir.mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getTargetdir()));
			}
		} else {
			if (destDir.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getTargetdir()));
			}
		}

		final File outFile = XliffFileFilter.appendExtension(new File(destDir, file.getName()));

		final BentenXliff xliff = BentenXliff.newInstance(outFile);
		final BlancoXliffFile xliffFile = xliff.createBlancoXliffFile();
		xliffFile.setSourceLanguage(fInput.getTranssourcelang());
		xliffFile.setTargetLanguage(fInput.getTranstargetlang());
		xliffFile.setDatatype("html"); //$NON-NLS-1$

		final BlancoXliffHeader header = xliffFile.getHeader();
		final BlancoXliffBody body = xliffFile.getBody();

		final BlancoXliffTool tool = new BlancoXliffTool();
		tool.setToolId(BentenConstants.PROJECT_NAME_LOWER);
		tool.setToolName(BentenConstants.PROJECT_NAME);
		BlancoXliffUtil.setTool(header, tool);

		final String path = Files.relativePath(new File(fInput.getSourcedir()).getParentFile(), file);
		final HelpTransUnitId id = new HelpTransUnitId(fInput.getTranstargetid() + path);

		for (final InlineText inline : new HtmlInlineTextBuilder(file)) {
			final String text = inline.getText();
			final SentencePartitionList sentenceList = new SentencePartitionList(text);
			if (sentenceList.size() > 1) {
				for (final String sentence : sentenceList) {
					id.incrementSubSeq(sentenceList.size());
					createTransUnit(body, id, sentence);
				}
			} else {
				createTransUnit(body, id, text);
			}
			id.incrementSeq();
		}

		xliff.save();

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}

	/**
	 * trans-unit の作成。
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param source source 要素の値
	 */
	protected void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String source) {
		fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toString());
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}
}
