/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.engine.googletranslate;

import java.io.File;
import java.io.IOException;
import java.net.ProxySelector;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.StringTokenizer;

import net.sf.json.JSONArray;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.ProxySelectorRoutePlanner;

import benten.cat.glossary.core.BentenGlossaryEngine;
import benten.cat.glossary.core.BentenGlossarySearchResult;

/**
 * 正規表現用語集エンジン。
 *
 * <UL>
 * <LI>このクラスは、用語集エンジンの切り替えテストのためのテストクラスです。
 * <LI>実際の Benten 基本セットには含まれません。
 * </UL>
 * 
 * @author IGA Tosiki
 */
public class GoogleTranslateGlossaryEngine implements BentenGlossaryEngine {
	/**
	 * 翻訳元の言語。
	 */
	protected String fSourceLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳先の言語。
	 */
	protected String fTargetLang = "ja-JP";//$NON-NLS-1$

	/**
	 * 基軸言語。
	 */
	protected String fPivotLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳言語を設定。
	 * 
	 * @param source 翻訳元の言語。例: en-US。
	 * @param target 翻訳先の言語。例: ja-JP。
	 */
	public void setLang(final String source, final String target) {
		fSourceLang = source;
		fTargetLang = target;
	}

	/**
	 * 基軸言語を設定。
	 * 
	 * @param pivotLang 基軸言語。例: en-US。
	 */
	public void setPivotLang(final String pivotLang) {
		fPivotLang = pivotLang;
	}

	/**
	 * {@inheritDoc}
	 */
	public void loadTsv(final File glossaryDirectory) throws IOException {
		// 何もしません。
	}

	/**
	 * {@inheritDoc}
	 */
	public void unload() {
		// 何もしません。
	}

	/**
	 * {@inheritDoc}
	 */
	public List<BentenGlossarySearchResult> search(final String source) {
		final List<BentenGlossarySearchResult> list = new ArrayList<BentenGlossarySearchResult>();

		String sourceLang = fSourceLang;
		String targetLang = fTargetLang;

		if (sourceLang.indexOf('-') > 0) {
			sourceLang = sourceLang.substring(0, sourceLang.indexOf('-'));
		}
		if (targetLang.indexOf('-') > 0) {
			targetLang = targetLang.substring(0, targetLang.indexOf('-'));
		}

		// TODO 現状では、与えられた文字列そのままに翻訳しています。
		// TODO 文字列を単語単位に分割すれば、かなり翻訳されやすくなるとは考えます。

		try {
			json2Result(source, translateByGoogle(sourceLang, targetLang, source), list);
			if (list.size() == 0) {
				final StringTokenizer tokenizer = new StringTokenizer(source);
				for (; tokenizer.hasMoreTokens();) {
					final String token = tokenizer.nextToken();
					json2Result(token, translateByGoogle(sourceLang, targetLang, token), list);
				}
			}
		} catch (IOException e) {
			throw new IllegalArgumentException(e.toString());
		}

		// 文字列長さでソート。
		Collections.sort(list, new Comparator<BentenGlossarySearchResult>() {
			public int compare(final BentenGlossarySearchResult tuLeft, final BentenGlossarySearchResult tuRight) {
				String left = tuLeft.getSource();
				String right = tuRight.getSource();
				return left.length() - right.length();
			}
		});

		return list;
	}

	/**
	 * Google Translate をつかって翻訳。
	 * @param sourceLang 翻訳元の言語。
	 * @param targetLang 翻訳先の言語。
	 * @param input 入力文字列。
	 * @return 翻訳後文字列。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	static String translateByGoogle(final String sourceLang, final String targetLang, final String input)
			throws IOException {
		System.out.println("trans: source[" + sourceLang + "], target[" + targetLang + "], input[" + input + "]");

		final String sourceLine = URLEncoder.encode(input, "UTF-8");

		final DefaultHttpClient httpClient = new DefaultHttpClient();
		{
			// Proxy 設定を Java 実行環境にゆだねる。
			final ProxySelectorRoutePlanner routePlanner = new ProxySelectorRoutePlanner(httpClient
					.getConnectionManager().getSchemeRegistry(), ProxySelector.getDefault());
			httpClient.setRoutePlanner(routePlanner);
		}

		final String uri = "http://translate.google.co.jp/translate_a/t?client=t&text=" + sourceLine + "&hl=ja"
				+ "&sl=" + sourceLang + "&tl=" + targetLang + "&pc=0";

		final HttpGet httpGet = new HttpGet(uri);

		httpGet.setHeader("User-Agent", "Mozilla/5.0 (Java) Benten Ext.");
		httpGet.setHeader("Accept-Language", "en-us");
		httpGet.setHeader("Accept-Charset", "UTF-8");

		final BasicResponseHandler responseHandler = new BasicResponseHandler();

		return httpClient.execute(httpGet, responseHandler);
	}

	/**
	 * JSON 検索結果をプレーンテキストに変換します。
	 * @param source 翻訳元の文字列。
	 * @param input 入力 JSON 文字列。
	 * @param list 出力検索結果。
	 */
	static void json2Result(final String source, final String input, final List<BentenGlossarySearchResult> list) {
		try {
			final JSONArray sentences = JSONArray.fromObject(input);

			final JSONArray inner1 = (JSONArray) sentences.get(0);
			final JSONArray inner2 = (JSONArray) inner1.get(0);
			final String obj = (String) inner2.get(0);

			// TODO 複数検索結果への対応。
			final BentenGlossarySearchResult result = new BentenGlossarySearchResult();
			result.setSource(source);
			result.setTarget(obj);
			result.setNote("Translate by Google Translate.");
			list.add(result);
		} catch (Exception ex) {
			// 用語集検索結果なし。
			return;
		}
	}
}
