/*!
  \file
  \brief URG 

  \author Satofumi KAMIMURA

  $Id$

  \todo pointerDeque ̗pBƂAu
*/

#include "UrgCtrl.h"
#include "SerialCtrl.h"
#include "ScipHandler.h"
#include "ThreadCreator.h"
#include "Delay.h"
#include <SDL.h>
#include <deque>


struct UrgCtrl::pImpl {
  enum {
    Infinity = -1,
  };
  std::string error_message;
  bool auto_capture;
  SerialCtrl con;
  size_t buffer_num;
  bool state_changed;

  class DataBuffer {
    DataBuffer(void);

    bool auto_capture;

  public:
    SDL_mutex* mutex;
    SDL_sem* sem;
    ScipHandler scip;
    size_t data_length;
    std::deque<long*> data_deque;
    size_t last_timestamp;

    explicit DataBuffer(bool autoCapture)
      : auto_capture(autoCapture),
	mutex(SDL_CreateMutex()), sem(SDL_CreateSemaphore(0)),
	data_length(1), last_timestamp(0) {
    }

    ~DataBuffer(void) {
      SDL_DestroyMutex(mutex);
      SDL_DestroySemaphore(sem);

      for (std::deque<long*>::iterator it = data_deque.begin();
	   it != data_deque.end(); ++it) {
	delete [] *it;
      }
      //fprintf(stderr, "delete [] data_deque\n");
    }

    void setDataLength(size_t size) {
      data_length = size;
    }

    void resize(size_t size) {
      // !!! mutex

      int n = size - data_deque.size();
      if (n <= 0) {
	return;
      }
      for (int i = 0; i < n; ++i) {
	//fprintf(stderr, "deque new: %d\n", data_length);
	data_deque.push_back(new long[data_length]);
      }
    }
  };

  class ThreadArgs {
    ThreadArgs(void);

  public:
    DataBuffer data_buffer;
    int left_times;
    int recv_size;

    ThreadArgs(bool auto_capture)
      : data_buffer(auto_capture), left_times(0), recv_size(0) {
    }
  };
  ThreadArgs args;
  ThreadCreator thread;

  pImpl(CaptureMode mode)
    : error_message("no error."), auto_capture((mode == Auto) ? true : false),
      buffer_num(1), state_changed(true),
      args(auto_capture), thread(capture_thread, &args) {
  }

  ~pImpl(void) {
    thread.wait();
    disconnect();
  }

  static int capture_thread(void* args) {
    ThreadArgs* info = static_cast<ThreadArgs*>(args);

    if ((info->left_times < 0) || (info->left_times > 0)) {
      // f[^҂
      // !!! f[^擾֐̌Ăяo
      //fprintf(stderr, "data_length: %d\n", info->data_buffer.data_length);

      // !!! 10 ɒu
      // !!! rpm  1/10 炢H
      //fprintf(stderr, "<");
      info->recv_size = info->data_buffer.scip.
	recvCaptureData(info->data_buffer.data_deque.front(),
			info->data_buffer.data_length,
			info->data_buffer.last_timestamp, 10);
      //fprintf(stderr, ">");

      if (info->recv_size > 0) {
	// 擾
	SDL_SemPost(info->data_buffer.sem);
	//fprintf(stderr, "post!\n");
      }
    }
    return 0;
  }

  size_t getMaxDataLength(void) {
    return args.data_buffer.scip.getMaxDataLength();
  }

  void begin_autoCapture(void) {

    // 擾񐔂̐ݒ
    size_t capture_times = (auto_capture) ? 0 : 1;
    args.data_buffer.scip.setCaptureTimes(capture_times);

    // f[^擾R}h̑M
    if (auto_capture) {
      args.data_buffer.scip.sendCaptureMessage();
      args.left_times = Infinity;
      state_changed = false;
    }

    if (! thread.isRunning()) {
      thread.run();
      args.data_buffer.setDataLength(getMaxDataLength());
      args.data_buffer.resize(buffer_num);
    }
  }

  void disconnect(void) {
    if (! args.data_buffer.scip.isConnected()) {
      return;
    }
    args.data_buffer.scip.disconnect();
  }
};


UrgCtrl::UrgCtrl(CaptureMode mode) : pimpl(new pImpl(mode)) {
}


UrgCtrl::~UrgCtrl(void) {
}


const char* UrgCtrl::what(void) {
  return pimpl->error_message.c_str();
}


bool UrgCtrl::connect(const char* device, long baudrate) {

  pimpl->disconnect();
  if (pimpl->con.connect(device, baudrate)) {
    pimpl->error_message = pimpl->con.what();
    return false;
  }
  pimpl->args.data_buffer.scip.setConnection(&pimpl->con);

  // p{[[g킹
  if (! pimpl->args.data_buffer.scip.adjustBaudrate(baudrate)) {
    pimpl->error_message = pimpl->args.data_buffer.scip.what();
    return false;
  }

  // ZT̔f
  if (! loadSensorParameter(NULL)) {
    pimpl->error_message = pimpl->args.data_buffer.scip.what();
    return false;
  }

  // f[^擾Xbh̋N
  pimpl->begin_autoCapture();

  return true;
}


void UrgCtrl::disconnect(void) {
  pimpl->disconnect();
}


bool UrgCtrl::getVersionInfo(std::vector<std::string>& lines) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return false;
  }

  bool ret = pimpl->args.data_buffer.scip.getVersionInfo(lines);
  if (! ret) {
    pimpl->error_message = pimpl->args.data_buffer.scip.what();
  }
  return ret;
}


bool UrgCtrl::loadSensorParameter(SensorParameter* parameter) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return false;
  }

  bool ret = pimpl->args.data_buffer.scip.loadSensorParameter(parameter);
  if (! ret) {
    pimpl->error_message = pimpl->args.data_buffer.scip.what();
  }
  return ret;
}


int UrgCtrl::capture(long data[], size_t max_size) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return NotConnected;
  }

  // Ԃς擾R}h̍Ĕss
  if ((! pimpl->auto_capture) || (pimpl->state_changed)) {

    // 擾R}h̍Ĕs
    // !!! begin_autoCapture() Ƃ
    pimpl->args.data_buffer.scip.sendCaptureMessage();
    //pimpl->args.left_times = Infinity;
    // !!! }jÂƂ́Ả񐔂
    pimpl->state_changed = false;

    // deque ̃f[^𖳌
    while (SDL_SemValue(pimpl->args.data_buffer.sem) > 0) {
      SDL_SemTryWait(pimpl->args.data_buffer.sem);
    }
  }

  //fprintf(stderr, "sem: %d\n", SDL_SemValue(pimpl->args.data_buffer.sem));
  if (! pimpl->auto_capture) {
    // ManualCapture
    // !!! 130 ɒu
    return pimpl->args.data_buffer.scip.
      recvCaptureData(data, max_size,
		      pimpl->args.data_buffer.last_timestamp, 130);

  } else if (SDL_SemTryWait(pimpl->args.data_buffer.sem) == 0) {
    //== SDL_MUTEX_TIMEDOUT) {
    //return -9;
    int n = pimpl->args.recv_size;
    for (int i = 0; i < n; ++i) {
      data[i] = pimpl->args.data_buffer.data_deque[0][i];
    }
    return n;
  } else {
    return -1;
  }
}


void UrgCtrl::setBufferNum(size_t size) {
  pimpl->buffer_num = size;

  if (! pimpl->auto_capture) {
    pimpl->args.data_buffer.scip.setCaptureTimes(pimpl->buffer_num);
    pimpl->state_changed = true;
  }
  pimpl->args.data_buffer.resize(pimpl->buffer_num);
}


unsigned long UrgCtrl::getTimestamp(void) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return 0;
  }

  return pimpl->args.data_buffer.last_timestamp;
}


void UrgCtrl::setFrameSkipFrames(size_t skip_frames) {
  // !!!
}


void UrgCtrl::setDataGroups(size_t groups) {
  // !!!
}


void UrgCtrl::setCaptureRange(int first_index, int last_index) {
  // !!!
}


long UrgCtrl::getMinDistance(void) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return NotConnected;
  }

  return pimpl->args.data_buffer.scip.getMinDistance();
}


int UrgCtrl::getMaxDataLength(void) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return NotConnected;
  }
  return pimpl->getMaxDataLength();
}


void UrgCtrl::setLaserOutput(bool on) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return;
  }

  // !!!
}


double UrgCtrl::index2rad(const int index) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return NotConnected;
  }

  return pimpl->args.data_buffer.scip.index2rad(index);
}


int UrgCtrl::rad2index(const double radian) {
  if (! pimpl->args.data_buffer.scip.isConnected()) {
    return NotConnected;
  }

  return pimpl->args.data_buffer.scip.rad2index(radian);
}
