/*!
  \file
  \brief SCIP nh

  \author Satofumi KAMIMURA

  $Id$
*/

#include "ScipHandler.h"
#include "ConnectionInterface.h"
#include "SensorParameter.h"


struct ScipHandler::pImpl {
  enum {
    Timeout = 130,
    LineLength = 64 + 1 + 1 + 1,

    RecvIncomplete = -1,
    PacketInvalid = -2,
  };

  std::string error_message;
  ConnectionInterface* con;
  SensorParameter params;
  size_t first, last;
  size_t data_byte;
  size_t groups;
  size_t skipFrames;
  size_t times;
  std::vector<long> recv_data;
  int remain_byte;
  char remain_data[3];

  pImpl(void)
    : error_message("no error."), con(NULL),
      first(0), last(0), data_byte(2), groups(1), skipFrames(0), times(1),
      remain_byte(0) {
  }

  bool isLF(const char ch) {
    return ((ch == '\n') || (ch == '\r')) ? true : false;
  }

  long decode(const char* data, int data_byte) {
    long value = 0;
    for (int i = 0; i < data_byte; ++i) {
      value <<= 6;
      value &= ~0x3f;
      value |= data[i] - 0x30;
    }
    return value;
  }

  int readLine(char *buffer, int timeout = pImpl::Timeout) {
    //fprintf(stderr, "timeout = %d\n", timeout);

    size_t i;
    for (i = 0; i < LineLength -1; ++i) {
      char recv_ch;
      int n = con->recv(&recv_ch, 1, timeout);
      if (n <= 0) {
	if (i == 0) {
	  return -1;		// ^CAEg
	}
	break;
      }
      if (isLF(recv_ch)) {
	break;
      }
      buffer[i] = recv_ch;
    }

    buffer[i] = '\0';
    return i;
  }

  int sendTag(const char* tag) {
    if (! con) {
      return false;
    }

    char send_message[pImpl::LineLength];
    sprintf(send_message, "%s\n", tag);
    int send_size = strlen(send_message);
    int n = con->send(send_message, send_size);
    if (n < send_size) {
      // !!! error_message
      return -1;
    }
    return n;
  }

  // !!! \n\n ܂ł̑ҋ@֐ɂƂH
  int sendMessage(const char* tag, int timeout) {
    if (! con) {
      return -1;
    }

    // ݂̂Ԃ
    int send_size = sendTag(tag);
    int recv_size = send_size + 2 + 1 + 2;
    char buffer[LineLength];

    // !!! ̂AreadLine() Œu
    int n = con->recv(buffer, recv_size, timeout);
    //int n = readLine(buffer, timeout);
    //fprintf(stderr, "%s\n", buffer);
#if 0
    if (! strncmp(buffer, "QT", 2)) {
      for (int i = 0; i < n; ++i) {
	fprintf(stderr, "%c", buffer[i]);
      }
    }
#endif

    if (n < recv_size) {
      // !!! error_message
      error_message = "XX 1";
      return RecvIncomplete;
    }

    // tag }b`邩mF
    if (strncmp(buffer, tag, send_size -1)) {
      error_message = "XX 1.5";
      return PacketInvalid;
    }

    int data_length = recv_size - send_size;
    if (! checkSum(&buffer[send_size],
		   data_length - 3, buffer[send_size + data_length - 3])) {
      // !!! error_message
      error_message = "XX 2";
      return PacketInvalid;
    }

    // !!! K؂ȏ
    char reply_str[3] = "00";
    reply_str[0] = buffer[send_size];
    reply_str[1] = buffer[send_size + 1];
    int reply = strtol(reply_str, NULL, 16);

    return reply;
  }

  bool checkSum(const char* buffer, size_t size, char expected) {

    // !!! 얢mFBނAdlmFׂ
    char sum = 0;
    for (size_t i = 0; i < size; ++i) {
      sum += buffer[i];
    }
    sum = (sum & 0x3f) + 0x30;
    //fprintf(stderr, "check: %02x, %02x\n", sum, expected);

    return (sum == expected) ? true : false;
  }

  int addRecvData(const char buffer[]) {

    //fprintf(stderr, "data: %s\n", buffer);
    const char* pre_p = buffer;
    const char* p = pre_p;

    if (remain_byte > 0) {
      memmove(&remain_data[remain_byte], buffer, data_byte - remain_byte);
      recv_data.push_back(decode(remain_data, data_byte));
      pre_p = &buffer[data_byte - remain_byte];
      p = pre_p;

      remain_byte = 0;
    }

    do {
      ++p;
      if ((p - pre_p) >= static_cast<int>(data_byte)) {
	recv_data.push_back(decode(pre_p, data_byte));
	pre_p = p;
      }
    } while (*p != '\0');
    remain_byte = p - pre_p;
    memmove(remain_data, pre_p, remain_byte);

    return 0;
  }
};


ScipHandler::ScipHandler(void) : pimpl(new pImpl) {
}


ScipHandler::~ScipHandler(void) {
}


const char* ScipHandler::what(void) {
  return pimpl->error_message.c_str();
}


void ScipHandler::setConnection(ConnectionInterface* con) {
  pimpl->con = con;
}


void ScipHandler::disconnect(void) {
  if (pimpl->con == NULL) {
    return;
  }
  // M̒~
  pimpl->sendMessage("QT", pImpl::Timeout);

  pimpl->con->disconnect();
}


bool ScipHandler::isConnected(void) {

  return ((pimpl->con != NULL) && pimpl->con->isConnected()) ? true : false;
}


bool ScipHandler::adjustBaudrate(long baudrate) {

  // "SCIP2.0" p{[[g킹
  long try_baudrate[] = { 115200, 19200, 57600, };
  for (size_t i = 0; i < sizeof(try_baudrate)/sizeof(try_baudrate[0]); ++i) {
    int ret = pimpl->con->changeBaudrate(try_baudrate[i]);
    pimpl->con->skip(pImpl::Timeout);
    if (ret < 0) {
      pimpl->error_message = pimpl->con->what();
      return false;
    }

    // R}h̒~
    // !!! SCIP2.0 łȂꍇ́AKOpPbg̈ɂȂ_ɒ
    pimpl->sendMessage("QT", pImpl::Timeout);
    pimpl->con->skip(pImpl::Timeout);
#if 0
    int reply = pimpl->sendMessage("QT", pImpl::Timeout);
    if (reply < 0) {
      continue;
    }
#endif

    // SCIP2.0 R}h𑗐MA'00'  '0E' ̉ŒʐMłƌȂ
    int reply = pimpl->sendMessage("SCIP2.0", pImpl::Timeout);
    if (reply < 0) {
      continue;
    }
    if ((reply == 0x00) || (reply == 0x0e)) {
      return setBaudrate(baudrate);
    }
  }

  pimpl->error_message = "Cannot adjust baudrate.";
  return false;
}


bool ScipHandler::setBaudrate(long baudrate) {

  // SS R}hɂ{[[gύX
  char buffer[] = "SS000000";
  sprintf(buffer, "SS%06ld", baudrate);
  int reply = pimpl->sendMessage(buffer, pImpl::Timeout);

  return ((reply == 0) || (reply == 3)) ? true : false;
}


bool ScipHandler::getVersionInfo(std::vector<std::string>& lines) {

  pimpl->sendTag("VV");
  char buffer[pImpl::LineLength];
  int line_length;
  for (int i = 0; (line_length = pimpl->readLine(buffer)) > 0; ++i) {
    //fprintf(stderr, "%s\n", buffer);

    enum {
      TagReply = 0,
      DataReply,
      Other,
    };
    if (i == TagReply) {
      // !!!
      // !!! false

    } else if (i == DataReply) {
      // !!!
      // !!! G[ȂΖ߂B肦Ȃ͂...
      // !!! false

    } else if (i >= Other) {
      buffer[line_length -2] = '\0';
      lines.push_back(&buffer[5]);
    }
  }

  return true;
}


bool ScipHandler::loadSensorParameter(SensorParameter* parameter) {

  // p[^ǂݏoƔf
  pimpl->sendTag("PP");
  char buffer[pImpl::LineLength];
  int line_length;
  for (int i = 0; (line_length = pimpl->readLine(buffer)) > 0; ++i) {

    // `FbNT̊mF
    // !!!
    // !!! return false;

    enum {
      TagReply = 0,
      DataReply,
      Other,
    };
    if (i == TagReply) {
      // !!!

    } else if (i == DataReply) {
      // !!!
      // !!! G[ȂΖ߂B肦Ȃ͂...

    } else if (i == Other + SensorParameter::MODL) {
      buffer[line_length - 2] = '\0';
      pimpl->params.model = &buffer[5];

    } else if (i == Other + SensorParameter::DMIN) {
      pimpl->params.distance_min = atoi(&buffer[5]);

    } else if (i == Other + SensorParameter::DMAX) {
      pimpl->params.distance_max = atoi(&buffer[5]);
      if (pimpl->params.distance_max > 4095) {
	pimpl->data_byte = 3;
      }
    } else if (i == Other + SensorParameter::ARES) {
      pimpl->params.area_total = atoi(&buffer[5]);

    } else if (i == Other + SensorParameter::AMIN) {
      pimpl->params.area_min = atoi(&buffer[5]);
      pimpl->first = pimpl->params.area_min;

    } else if (i == Other + SensorParameter::AMAX) {
      pimpl->params.area_max = atoi(&buffer[5]);
      pimpl->last = pimpl->params.area_max;

    } else if (i == Other + SensorParameter::AFRT) {
      pimpl->params.area_front = atoi(&buffer[5]);

    } else if (i == Other + SensorParameter::SCAN) {
      pimpl->params.scan_rpm = atoi(&buffer[5]);
    }
  }

  if (parameter != NULL) {
    // p[^̎擾͍sAi[͍sȂꍇ
    *parameter = pimpl->params;
  }

  return true;
}


void ScipHandler::setCaptureTimes(size_t times) {
  pimpl->times = times;
}


void ScipHandler::sendCaptureMessage(void) {

  // !!! ̊֐ŁAQT ̑MKvłΑΏs

  // Timeout Ԃ́Ascan_rpm ̂P + 30 [msec] Ƃ
  // !!! pImpl ̃oƂĐ錾āAȂƂ銴ȁH

  pimpl->remain_byte = 0;

  char send_message[pImpl::LineLength];
  sprintf(send_message, "M%c%04d%04d%02d%01d%02d",
	  (pimpl->data_byte == 2) ? 'S' : 'D',
	  pimpl->first, pimpl->last,
	  pimpl->groups, pimpl->skipFrames, pimpl->times);
  //fprintf(stderr, "%s\n", send_message);
  // !!! ߂l̃`FbNׂBꉞ
  //int send_size = pimpl->sendTag(send_message);
  pimpl->sendTag(send_message);
}


int ScipHandler::recvCaptureData(long* data, size_t max_size,
				 size_t& timestamp, int timeout) {

  pimpl->recv_data.clear();
  char buffer[pImpl::LineLength];
  int line_length;
  size_t local_timestamp = 0;
  for (int i = 0; (line_length = pimpl->readLine(buffer)) >= 0; ++i) {

    //fprintf(stderr, "%02d: %s\n", i, buffer);

    // `FbNT̊mF
    // !!!
    // !!! return false;

    if ((i >= 6) && (line_length == 0)) {
	// f[^M̊
#if 0
      fprintf(stderr, "last line!\n");
      fprintf(stderr, "timestamp: %d\n", timestamp);
      fprintf(stderr, "expect size: %d\n", pimpl->last - pimpl->first + 1);
      fprintf(stderr, "size: %d\n", pimpl->recv_data.size());
#endif

      //fprintf(stderr, "data_size: %d, max_size: %d\n", pimpl->recv_data.size(), max_size);
      int data_size = pimpl->recv_data.size();
      size_t min_length =
	(static_cast<int>(max_size) < data_size) ? max_size : data_size;
      //fprintf(stderr, "data_size: %d, min_length: %d\n", pimpl->recv_data.size(), min_length);
      for (size_t i = 0; i < min_length; ++i) {
	//long length = pimpl->recv_data[i];
	// !!! G[R[hԂĂȂǁAȂH
	//data[i] = (length < 20) ? -1 : length;
	data[i] = pimpl->recv_data[i];
      }
      //fprintf(stderr, "last ticks: %d\n\n", SDL_GetTicks());
      timestamp = local_timestamp;
      return min_length;

    } else if (i == 0) {
      //fprintf(stderr, "echo back end\n");
#if 0
      // GR[obN
      if (strncmp(buffer, send_message, send_size -1)) {
	pimpl->error_message = "YY 01";
	return pImpl::PacketInvalid;
      }
#endif

    } else if (! strncmp(buffer, "99b", 3)) {
      i = 4;

    } else if (i == 2) {
      // MpPbgɑ΂鉞
      // !!!

    } else if (i == 4) {
      //fprintf(stderr, "99b end\n");
      // "99b" Œ
      if (strncmp(buffer, "99b", 3)) {
	pimpl->error_message = "YY 02";
	return pImpl::PacketInvalid;
      }

    } else if (i == 5) {
      // ^CX^v
      //fprintf(stderr, "timestamp end\n");
      local_timestamp = pimpl->decode(buffer, 4);
      //fprintf(stderr, "ticks ticks: %d\n", SDL_GetTicks());

    } else if (i >= 6) {
      // 擾f[^
      buffer[line_length -1] = '\0';
      int ret = pimpl->addRecvData(buffer);
      if (ret < 0) {
	// !!! G[bZ[W̍XV
	return ret;
      }
    }
  }
  return -1;
}


void ScipHandler::setFrameSkipFrames(size_t skip_frames) {
  if ((skip_frames >= 0) && (skip_frames <= 9)) {
    pimpl->skipFrames = skip_frames;
  }
}


void ScipHandler::setDataGroups(size_t groups) {
  if ((groups > 0) && (groups <= 99)) {
    pimpl->groups = groups;
  }
}


void ScipHandler::setCaptureRange(int first_index, int last_index) {
  if ((first_index >= 0) && (first_index <= pimpl->params.area_max)) {
    pimpl->first = first_index;
  }
  if ((last_index >= first_index) && (last_index <= pimpl->params.area_max)) {
    pimpl->last = last_index;
  }
}


long ScipHandler::getMinDistance(void) {
  return (isConnected()) ? pimpl->params.distance_min : -1;
}


int ScipHandler::getMaxDataLength(void) {
  if (! isConnected()) {
    return -1;
  } else {
    return pimpl->params.area_max - pimpl->params.area_min + 1;
  }
}


void ScipHandler::setLaserOutput(bool on) {
  // !!!
}


double ScipHandler::index2rad(const int index) {

  // !!! front ɒlvZ܂傤
  // !!!
  return 0.0;
}


int ScipHandler::rad2index(const double radian) {

  // !!! front ɒlvZ܂傤
  // !!!
  return 0;
}
