/**
 * Copyright (C) 2023 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
use rayon::iter::IntoParallelIterator;
use rayon::prelude::*;
use rayon::{ThreadPool, ThreadPoolBuilder};
extern crate rayon;

use crate::chmod;
use crate::files;
use crate::files::DD;
use crate::iomod;
use crate::thmod;

/*
    Rayonは、Iterator　の代わりに ParallelIterator を使用して並列プログラミングを行います。
    iter() → par_iter()
    iter_mut() → par_iter_mut()
    into_iter() → into_par_iter() （所有権が移動したイテレータ）
    に変更するだけで普通の Iterator と同じように使えるようになっています。
*/

// move occurs because `STACK` has type `Vec<DD>`,
//      which does not implement the `Copy` trait
// let _ = thmod::STACK.into_par_iter().for_each(|x| {

// Rayon スレッド
pub fn spawnx(threads: i32) {
    if threads <= 0 {
        exec();
    } else {
        create_thread_pool(threads as usize).install(|| {
            exec();
        });
    }
}

// Case 1 - STACK をダイレクトに読み込み
pub fn exec() {
    unsafe {
        let ve: &Vec<DD> = &thmod::STACK;

        // let _ = ve.par_iter().for_each(|x| { // 遅い
        let _ = ve.into_par_iter().for_each(|x| {
            task(x); // タスクを実行
        });
    }
}

// Case 2 - Range をインデックスとして使用
pub fn _2exec() {
    unsafe {
        let queue_size: usize = thmod::STACK.len();
        let range = 0..queue_size;

        let _ = range.into_par_iter().for_each(|i| {
            let dd: &DD = &thmod::STACK[i];
            task(dd); // タスクを実行
        });
    }
}

fn create_thread_pool(threads: usize) -> ThreadPool {
    ThreadPoolBuilder::new()
        .num_threads(threads)
        .build()
        .unwrap()
}

// copy/move/rename
fn task(dd: &DD) {
    thmod::progress(&dd.input); // プログレスを表示
    if dd.action != files::SKIP {
        // スキップ以外ならアクションを実行
        if dd.cmr_mode == files::_RENAME {
            iomod::rename_file(&dd.input, &dd.output); // Rename
        } else if dd.algorithm == files::_STD {
            iomod::copy(&dd.input, &dd.output); // Copy, Move
        } else if dd.algorithm == files::_MAXBUF {
            chmod::copymax(&dd.input, &dd.output); // maxbuf
        } else if dd.algorithm == files::_TEST {
            iomod::copy(&dd.input, &dd.output); // Test
        } else {
            chmod::copych(&dd.input, &dd.output); // channel
        }
    }
    if dd.cmr_mode != files::_COPY {
        // 入力ファイルを削除
        let _ = iomod::remove_file(&dd.input); // Move, Rename
    }
    let _: i32 = thmod::decrement_req_number(); // 処理要求(キュー数)を減算
}
