/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import org.jetbrains.annotations.NotNull;

import java.util.Arrays;

/**
 * Node.
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public final class Node {

    public static final String BOOL_OP = "?"; // 真値判定(Boolean|0|1).
    public static final String CONCAT_OP = "cat_"; // Built-in: 文字列結合.
    public static final String INDEX_OP = "INDEX"; // Built-in: 配列の添字.
    public static final String NOT_OP = "!"; // 論理否定(!).
    public static final String NUMBER_OP = "NUMBER"; // Built-in: 数値正規化

    /**
     * Don't let anyone instantiate this class.
     */
    private Node() {
        super();
    }

    /**
     * オブジェクトの文字列表現を返す.
     */
    public static String mkString(Object x) {
        return (x instanceof Object[]) ?
                Arrays.toString((Object[]) x) : x.toString();
    }

    /**
     * Annotation - 注釈.
     */
    public static class Annotation extends YyNop {
        public Annotation(String val) {
            super(val);
        }

        @Override
        public final String toString() {
            return super.value;
        }
    }

    /**
     * 注釈.
     */
    public static class Comment extends YyNop {
        public Comment(String val) {
            super(val);
        }
    }

    /**
     * 配列変数.
     */
    public static class Arr extends YyVariable {

        public Arr(String name, Object[] idx) {
            super(Keyword.SyyARRAY, name, idx);
        }

        @Override
        public final String toString() {
            return super.name + Arrays.toString(super.index);
        }
    }

    /**
     * AssignHelper.
     */
    public static class Ass extends YyStatement {

        public final String sType; // このオブジェクトの文字列表現タイプ.
        public final int nType; // このオブジェクトのタイプ.
        public final YyVariable left; // このオブジェクトの左の要素.
        public final Object right; // このオブジェクトの右の要素.
        @SuppressWarnings("CanBeFinal")
        public String name; // NOTE これは書き換え可能 ☆

        public Ass(Keyword id, String op,
                   YyVariable left, Object right, int ntype,
                   String stype) {
            super(id);
            this.name = op;
            this.left = left;
            this.right = right;
            this.nType = ntype;
            this.sType = stype;
        }

        @Override
        public final String toString() {
            return "(" + super.id + ' ' + this.left // + ": " + this.nType
                    + ' ' + this.name + ' ' + this.right + ')';
        }
    }

    /**
     * 真値判定(Boolean|0|1) 論理否定(!).
     */
    public static class B00l extends YyeValue {

        public final Object expr; // このオブジェクトの式.

        public B00l(String name, Object exp) {
            super(name, Flags.T02BOOL);
            this.expr = exp;
        }

        @Override
        public final String toString() {
            return super.name + '(' + mkString(this.expr) + ')';
        }
    }

    /**
     * 論理比較.
     */
    public static class Comp extends YyeValue {

        public final Object left; // このオブジェクトの左の要素.
        public final Object right; // このオブジェクトの右の要素.

        public Comp(String name, Object left, Object right) {
            super(name, Flags.T02BOOL);
            this.left = left;
            this.right = right;
        }

        @Override
        public final String toString() {
            return "(" + this.left + ' ' + super.name + ' ' + this.right + ')';
        }
    }

    /**
     * Call.
     */
    public static class Call extends YyCall {

        public Call(String name, Object[] args, int ntype) { //
            super(Keyword.SyyCALL, name, args, ntype);
        }

        @Override
        public final String toString() { //
            return super.name + Arrays.toString(super.args);
        }
    }

    /**
     * catch.
     */
    public static class Catch extends YyStatement {

        public final String name; // 例外変数名.
        public final String[] claz; // 例外クラス名.
        public final Object[] stmt; // このオブジェクトの文.

        public Catch(@NotNull String name, String[] claz, Object[] statement) {
            super(Keyword.SymCATCH);
            this.name = name;
            this.claz = claz;
            this.stmt = statement;
        }

        @Override
        public final String toString() { //
            return "catch " + Arrays.toString(this.claz) + Arrays.toString(this.stmt);
        }
    }

    /**
     * ++ -- 数値正規化.
     */
    public static class IncDec extends YyeValue {

        public final Object expr; // このオブジェクトの式.

        public IncDec(String name, Object exp) {
            super(name, Flags.T01NUM);
            this.expr = exp;
        }

        @Override
        public final String toString() {
            return super.name + '(' + this.expr + ')';
        }
    }

    /**
     * 四則演算.
     */
    public static class Calc extends YyeValue {

        public final Object left; // このオブジェクトの左の要素.
        public final Object right; // このオブジェクトの右の要素.

        public Calc(String name, Object left, Object right) {
            super(name, Flags.T01NUM);
            this.left = left;
            this.right = right;
        }

        @Override
        public final String toString() {
            return "("
                    + ((this.left instanceof Object[]) ? //
                    Arrays.toString((Object[]) this.left) : this.left)
                    + ' '
                    + super.name
                    + ' '
                    + ((this.right instanceof Object[]) ? //
                    Arrays.toString((Object[]) this.right) : this.right)
                    + ')';
        }
    }

    /**
     * delete.
     */
    public static class Del extends YyStatement {
        public final YyVariable e; // このオブジェクトの変数式.

        public Del(YyVariable expr) {
            super(Keyword.SymDELETE);
            this.e = expr;
        }

        @Override
        public final String toString() {
            return "delete(" + this.e + ')';
        }
    }

    /**
     * Function Declaration.
     */
    public static class Func extends YyStatement {

        public final String[] comment; // このオブジェクトの注釈.
        public final String[] annotation; // アノテーション.
        public final String name; // このオブジェクトの名前.
        public final FnP[] parm; // 仮引数 [+ ローカル変数].
        public final int argsLength; // 仮引数の長さ　-　length of the argument.
        public final int nType; // 復帰タイプ.
        public final String sType; // 復帰タイプ(文字列表現).
        public final Object[] stmt; // このオブジェクトの文.

        public Func(@NotNull final String name, FnP[] param,
                    int length, int ntype, String stype,
                    String[] comm, String[] anno, Object[] statement) {
            super(Keyword.SyyFN);
            this.name = name;
            this.parm = param;
            this.argsLength = length;
            this.nType = ntype;
            this.sType = stype;
            this.comment = comm;
            this.annotation = anno;
            this.stmt = statement;
        }

        @Override
        public final String toString() {
            return this.name + '(' + Arrays.toString(this.parm) + ','
                    + this.argsLength + ',' + this.nType + ','
                    + Arrays.toString(this.stmt) + ')';
        }
    }

    /**
     * The position where the function was defined.
     * REMIND root関数では、このノードは、出力されない.
     */
    public static class FnI extends YyStatement {

        public final String name; // 関数名: [Caller$]name.

        public FnI(@NotNull final String fullname) {
            super(Keyword.SyyFNI);
            this.name = fullname;
            // System.err.println("Node.FnI: " + fullname);
        }

        @Override
        public final String toString() {
            return name;
        }
    }

    /**
     * Function Parameter.
     */
    public static class FnP extends YyStatement {
        public final String name; // このオブジェクトの名前.
        public final int nType; // このオブジェクトのタイプ.
        public final String sType; // このオブジェクトの文字列表現タイプ.

        public FnP(@NotNull final String name, int ntype,
                   String stype) {
            super(Keyword.SyyFNP);
            this.name = name;
            this.nType = ntype;
            this.sType = stype;
        }

        @Override
        public final String toString() {
            return this.name
                    + ':'
                    + (this.sType.isEmpty() ? (Integer
                    .toHexString(this.nType)) : this.sType);
        }
    }

    /**
     * For.
     */
    public static class For extends YyStatement {
        public final Object e1; // このオブジェクトの式 1.
        public final Object e2; // このオブジェクトの式 2.
        public final Object e3; // このオブジェクトの式 3.
        public final Object[] stmt; // このオブジェクトの文.

        public For(Keyword x, Object expr1, Object expr2,
                   Object expr3, Object[] statement) {
            super(x);
            this.e1 = expr1;
            this.e2 = expr2;
            this.e3 = expr3;
            this.stmt = statement;
        }

        @Override
        public final String toString() {
            return super.id + "((" + this.e1 + ';' + this.e2 + ';' + this.e3
                    + ')' + Arrays.toString(this.stmt) + ')';
        }
    }

    /**
     * For in.
     */
    public static class ForIn extends YyStatement {
        public final YyVariable e1; // このオブジェクトの式 1.
        public final Object e2; // このオブジェクトの式 2.
        public final Object[] stmt; // このオブジェクトの文.

        public ForIn(Keyword x, YyVariable expr1,
                     Object expr2, Object[] statement) {
            super(x);
            this.e1 = expr1;
            this.e2 = expr2;
            this.stmt = statement;
        }

        @Override
        public final String toString() {
            return super.id + "((" + this.e1 + " in " + this.e2 + ')'
                    + Arrays.toString(this.stmt) + ')';
        }
    }

    /**
     * Get line.
     */
    public static class Getline extends YyCall {

        private static final Object[] $0 = // 省略値,$0
                new Object[]{new Arr("$", new Integer[]{0})};

        public final String rid; // このオブジェクトのリダイレクト識別子.
        public final Object filename; // このオブジェクトのファイル名.(式もある)

        public Getline(Object[] args, String rId, Object file) {
            super(Keyword.SyyCALL, "getline",
                    (0 == args.length) ? $0 : args, // 0件にしない.
                    Flags.T03INT);
            this.rid = rId;
            this.filename = file; // (式もある)
        }

        @Override
        public final String toString() {
            return super.name + '(' + Arrays.toString(super.args) + ": '"
                    + this.rid + '\'' + this.filename + ')';
        }
    }

    /**
     * if.
     */
    public static class If extends YyStatement {

        public final Object cc; // このオブジェクトの論理式.
        public final Object[] left; // このオブジェクトの左の要素.
        public final Object[] right; // このオブジェクトの右の要素.

        public If(Keyword x, Object c, Object[] left,
                  Object[] right) {
            super(x);
            this.cc = c;
            this.left = left;
            this.right = right;
        }

        @Override
        public final String toString() {
            String w = (Keyword.SymIF == super.id)
                    ? ("if(" + this.cc + ')') : ("(" + this.cc + ")?");
            return w + Arrays.toString(this.left) + ','
                    + Arrays.toString(this.right);
        }
    }

    /**
     * has, new, invoke, function value.
     */
    public static class Invoke extends YyCall {

        public final Object obj; // 処理対象のオブジェクト.
        public final String sType; // このオブジェクトの文字列表現タイプ.
        public final boolean isMethod; // メソドかどうか. コードに()が指定されている.

        public Invoke(Keyword id2, Object o, String name,
                      Object[] args, int ntype, String stype,
                      boolean isMethod) {
            super(id2, name, args, ntype);
            this.obj = o;
            this.sType = stype;
            this.isMethod = isMethod;
        }

        @Override
        public final String toString() {
            Object x = (this.obj instanceof Object[]) ?
                    Arrays.toString((Object[]) this.obj) : this.obj;
            return super.id + " " + x + '.' + super.name + Arrays.toString(super.args);
        }
    }

    /**
     * 単純変数.
     */
    public static class NAME extends YyVariable {

        private static final Object[] EMPTY_INDEX = {}; // 空の配列.

        public NAME(Keyword type, String name) {
            super(type, name, EMPTY_INDEX); // var, val
        }

        @Override
        public final String toString() {
            return super.name;
        }
    }

    /**
     * print[f].
     */
    public static class Print extends YyStatement {
        public final String name; // このオブジェクトの名前.
        public final Object[] args; // このオブジェクトの引数.
        public final String rid; // このオブジェクトのリダイレクト識別子.
        public final Object filename; // このオブジェクトのファイル名.(式もある)

        public Print(@NotNull final String name, Object[] args,
                     String rId, Object fileName) {
            super(Keyword.SyyCALL);
            this.name = name;
            this.args = args;
            this.rid = rId;
            this.filename = fileName; // (式もある)
        }

        @Override
        public final String toString() {
            return this.name + '(' + Arrays.toString(this.args) + ": `"
                    + this.rid + '`' + this.filename + ')';
        }
    }

    /**
     * このクラスのルートオブジェクト.
     */
    public abstract static class Root {

        public final boolean isEexpression; // このオブジェクトが式かどうか.
        public final int linenumber = Advance.yyLineNumber(); // スクリプト行番号.

        Root(boolean b) {
            this.isEexpression = b;
        }
    }

    /**
     * Simple Statement (break, continue, exit, next, nextfile, return).
     */
    public static class Stmt extends YyStatement {

        public final Object e; // このオブジェクトの変数式.
        public final int nType; // このオブジェクトのタイプ.

        public Stmt(Keyword x, Object expr, int ntype) {
            super(x);
            this.e = expr;
            this.nType = ntype;
        }

        @Override
        public final String toString() {
            return super.id.name() + ' ' + this.e;
        }
    }

    /**
     * throw.
     */
    public static class Throw extends YyStatement {

        public final Object expr; // このオブジェクトの変数式.

        public Throw(Object expr) {
            super(Keyword.SymTHROW);
            this.expr = expr;
        }

        @Override
        public final String toString() {
            return "throw(" + this.expr + ')';
        }
    }

    /**
     * try.
     */
    public static class Try extends YyStatement {

        public final Object[] e1; // Try.
        public final Catch[] e2; // Catch.
        public final Object[] e3; // Finally.

        public Try(Object[] ex1, Catch[] ex2, Object[] ex3) {
            super(Keyword.SymTRY);
            this.e1 = ex1;
            this.e2 = ex2;
            this.e3 = ex3;
        }

        @Override
        public final String toString() {
            return "try(" + Arrays.toString(this.e1) + Arrays.toString(this.e2)
                    + Arrays.toString(this.e3) + ')';
        }
    }

    /**
     * do while, while.
     */
    public static class While extends YyStatement {

        public final Object cc; // このオブジェクトの論理式.
        public final Object[] stmt; // このオブジェクトの文.

        public While(Keyword x, Object c, Object[] statement) {
            super(x);
            this.cc = c;
            this.stmt = statement;
        }

        @Override
        public final String toString() {
            return super.id +
                    "(" + this.cc + ',' + Arrays.toString(this.stmt) + ')';
        }
    }

    /**
     * 呼出しノード.
     */
    public abstract static class YyCall extends Root {

        public final Object[] args; // このオブジェクトの引数.
        public final Keyword id; // このオブジェクトの識別子.
        public final String name; // このオブジェクトの名前.
        public final int nType; // このオブジェクトのタイプ.

        YyCall(Keyword x, @NotNull final String name,
               @NotNull final Object[] args, int ntype) {
            super(true); // isEexpression
            this.id = x;
            this.name = name;
            this.args = args;
            this.nType = ntype;
        }
    }

    /**
     * NOP ノード.
     */
    public abstract static class YyNop extends Root {

        public final String value; // この value オブジェクト.

        YyNop(String val) {
            super(true); // isEexpression
            this.value = val;
        }
    }

    /**
     * 文ノード.
     */
    public abstract static class YyStatement extends Root {

        public final Keyword id; // このオブジェクトの識別子.

        YyStatement(Keyword x) {
            super(false); // isEexpression
            this.id = x;
        }
    }

    /**
     * 変数ノード.
     */
    public abstract static class YyVariable extends Root {

        public final Keyword id; // このオブジェクトの識別子.
        public final Object[] index; // このオブジェクトの添字.
        @SuppressWarnings("CanBeFinal")
        public String name; // NOTE これは書き換え可能 ☆

        YyVariable(Keyword x, @NotNull final String name, Object[] idx) {
            super(true); // isEexpression
            this.id = x;
            this.name = name;
            this.index = idx;
        }

        @Override
        public String toString() {
            return (0 == index.length) ? name : name + Arrays.toString(index);
        }
    }

    /**
     * 値を返すノード.
     */
    public abstract static class YyeValue extends Root {

        public final String name; // このオブジェクトの名前.
        public final int nType; // このオブジェクトのタイプ.

        YyeValue(@NotNull final String name, int type) {
            super(true); // isEexpression
            this.name = name;
            this.nType = type;
        }

        @Override
        public String toString() {
            return name;
        }
    }
}