/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import java.util.List;
import java.util.Map;
import java.util.concurrent.*;

/**
 * Futures Tools.
 * <p>
 * The class to which this annotation is applied is thread-safe.
 *
 * @author Kunio Himei.
 */
public final class Futures {

    /**
     * 関数起動時の同期を制御するコンテナ (Lambda).
     */
    private static final ConcurrentHashMap<String, CyclicBarrier> BARRIER =
            new ConcurrentHashMap<>();

    /**
     * ☆ 全てのパーティーが await を呼び出すまで待機する.
     * {@literal IllegalStateException} - バリアーが破壊された場合
     */
    @SuppressWarnings("UnusedReturnValue")
    private static Integer await(CyclicBarrier x)
            throws InterruptedException {
        try {
            return x.await();
        } catch (BrokenBarrierException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * ☆ ブロックを開放する.
     */
    public static void await(String id) throws InterruptedException {
        CyclicBarrier lock = BARRIER.get(id);
        if (null != lock) {
            await(lock);
        }
    }

    /**
     * ☆ タスクの実行の取り消しを試みる.
     */
    public static boolean cancel(Object x) {
        boolean rs = false;
        for (Object o : toArray(x)) {
            if (!(o instanceof Future<?> task)) {
                throw new IllegalArgumentException(o.getClass().getName() + ' '
                        + o);
            }
            if (!task.isDone() && !task.isCancelled() && task.cancel(true)) {
                rs = true;
            }
        }
        return rs;
    }

    /**
     * ☆ タスクの終了を待機して 計算結果を取得する.
     */
    @SuppressWarnings("UnusedReturnValue")
    public static Object getValue(Object x) throws InterruptedException {
        if (x instanceof Future<?>) {
            try {
                return ((Future<?>) x).get();

            } catch (ExecutionException e) { // メソッドが例外をスローする場合
                Throwable ex = e;
                while (null != ex.getCause()) {
                    ex = ex.getCause(); // ラップされた例外チェーンを解除する
                }
                if (ex instanceof RuntimeException) {
                    throw (RuntimeException) ex; // アンラップした例外を投げる
                }
                throw new RuntimeException(ex); // アンラップした例外を投げる
            }
        }
        throw new IllegalArgumentException(x.getClass().getName() + ' ' + x);
    }

    /**
     * ☆ タスクが生存しているかどうかを返す.
     */
    public static boolean isAlive(Object x) {
        for (Object o : toArray(x)) {
            if (o instanceof Future<?>) {
                if (!((Future<?>) o).isDone()) {
                    return true;
                }
            } else {
                throw new IllegalArgumentException(o.getClass().getName() + ' '
                        + o);
            }
        }
        return false;
    }

    /**
     * ☆ タスクの終了を待機する.
     */
    public static boolean join(Object x) throws InterruptedException {
        boolean rs = false;
        for (Object o : toArray(x)) {
            if (o instanceof Future<?>) {
                getValue(o);
                rs = true;
            } else if (o instanceof CyclicBarrier) {
                await((CyclicBarrier) o);
                rs = true;
            } else {
                throw new IllegalArgumentException(o.getClass().getName() + ' ' + o);
            }
        }
        return rs;
    }

    /**
     * ペアで待ち合わせる同期オブジェクトを構築する (barrier).
     */
    public static void pair(String id) {
        CyclicBarrier lock = BARRIER.get(id);
        if (null == lock) {
            CyclicBarrier x = new CyclicBarrier(2);
            BARRIER.putIfAbsent(id, x); // singleton
        }
    }

    /**
     * コンテナを配列に変換して返す.
     */
    private static Object[] toArray(Object x) {
        if (x instanceof Object[]) {
            return (Object[]) x;
        } else if (x instanceof Map<?, ?>) {
            return ((Map<?, ?>) x).values().toArray();
        } else if (x instanceof List<?>) {
            return ((List<?>) x).toArray();
        }
        return new Object[]{x};
    }
}