
/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Object used to request social information from the container.
 * This includes data for friends, profiles, app data, and activities.
 *
 * All apps that require access to people information should send a dataRequest
 * in order to receieve a dataResponse
 */


/**
 * @class
 * <p>
 * Used to request social information from the container.
 * This includes data for friends, profiles, app data, and activities.
 * All apps that require access to people information
 * should send a DataRequest.
 * </p>
 *
 * <p>
 * Here's an example of creating, initializing, sending, and handling
 * the results of a data request:
 * </p>
 *
 * <pre>function requestMe() {
  var req = opensocial.newDataRequest();
  req.add(req.newFetchPersonRequest(
            opensocial.IdSpec.PersonId.VIEWER),
          "viewer");
  req.send(handleRequestMe);
};

function handleRequestMe(data) {
  var viewer = data.get("viewer");
  if (viewer.hadError()) {
    <em>//Handle error using viewer.getError()...</em>
    return;
  }

  <em>//No error. Do something with viewer.getData()...</em>
}
</pre>
 * <p>
 * <b>See also:</b>
 * <a href="opensocial.html#newDataRequest"><code>
 * opensocial.newDataRequest()</code></a>
 * </p>
 *
 * @name opensocial.DataRequest
 */

/**
 * Do not create a DataRequest directly, instead use
 * opensocial.createDataRequest();
 *
 * @private
 * @constructor
 */
opensocial.DataRequest = function() {};


/**
 * Get the requested objects
 *
 * @return {Array.<{key:String, request:BaseDataRequest}>}
 *    requestObjects An array of data requests that the container should fetch
 *    data for
 * @private
 */
opensocial.DataRequest.prototype.getRequestObjects = function() {};


/**
 * Adds an item to fetch (get) or update (set) data from the server.
 * A single DataRequest object can have multiple items.
 * As a rule, each item is executed in the order it was added,
 * starting with the item that was added first.
 * However, items that can't collide might be executed in parallel.
 *
 * @param {Object} request Specifies which data to fetch or update
 * @param {String} opt_key A key to map the generated response data to
 */
opensocial.DataRequest.prototype.add = function(request, opt_key) {};


/**
 * Sends a data request to the server in order to get a data response.
 * Although the server may optimize these requests,
 * they will always be executed
 * as though they were serial.
 *
 * @param {Function} opt_callback The function to call with the
 *   <a href="opensocial.DataResponse.html">data response</a>
 *    generated by the server
 */
opensocial.DataRequest.prototype.send = function(opt_callback) {};


/**
 * @static
 * @class
 * The sort orders available for ordering person objects.
 *
 * @name opensocial.DataRequest.SortOrder
 */
opensocial.DataRequest.SortOrder = {
  /**
   * When used will sort people by the container's definition of top friends.
   * This field may be used interchangeably with the string 'topFriends'.
   * @member opensocial.DataRequest.SortOrder
   */
  TOP_FRIENDS : 'topFriends',
  /**
   * When used will sort people alphabetically by the name field.
   * This field may be used interchangeably with the string 'name'.
   * @member opensocial.DataRequest.SortOrder
   */
  NAME : 'name'
};


/**
 * @static
 * @class
 * The filters available for limiting person requests.
 *
 * @name opensocial.DataRequest.FilterType
 */
opensocial.DataRequest.FilterType = {
  /**
   * Retrieves all friends.
   * This field may be used interchangeably with the string 'all'.
   * @member opensocial.DataRequest.FilterType
   */
  ALL : 'all',
  /**
   * Retrieves all friends that use this application.
   *
   * Note: Containers may define "use" in any manner they deem appropriate for
   * their functionality, and it is not expected that this field will have
   * the exact same semantics across containers.
   * This field may be used interchangeably with the string 'hasApp'.
   * @member opensocial.DataRequest.FilterType
   */
  HAS_APP : 'hasApp',
  /**
   * Retrieves only the user's top friends as defined by the container.
   * Container support for this filter type is OPTIONAL.
   * This field may be used interchangeably with the string 'topFriends'.
   * @member opensocial.DataRequest.FilterType
   */
  TOP_FRIENDS : 'topFriends',
  /**
   * Will filter the people requested by checking if they are friends with
   * the given <a href="opensocial.IdSpec.html">idSpec</a>. Expects a filterOptions parameter to be passed with the
   *    following fields defined:
   *  - idSpec The <a href="opensocial.IdSpec.html">idSpec</a> that each person must be friends with.
   * This field may be used interchangeably with the string 'isFriendsWith'.
   * @member opensocial.DataRequest.FilterType
  */
  IS_FRIENDS_WITH: 'isFriendsWith'
};


/**
 * @static
 * @class
 * @name opensocial.DataRequest.PeopleRequestFields
 */
opensocial.DataRequest.PeopleRequestFields = {
  /**
   * An array of
   * <a href="opensocial.Person.Field.html">
   * <code>opensocial.Person.Field</code></a>
   * specifying what profile data to fetch
   * for each of the person objects. The server will always include
   * ID, NAME, and THUMBNAIL_URL.
   * This field may be used interchangeably with the string 'profileDetail'.
   * @member opensocial.DataRequest.PeopleRequestFields
   */
  PROFILE_DETAILS : 'profileDetail',

  /**
   * A sort order for the people objects; defaults to TOP_FRIENDS.
   * Possible values are defined by
   * <a href="opensocial.DataRequest.SortOrder.html">SortOrder</a>.
   * This field may be used interchangeably with the string 'sortOrder'.
   * @member opensocial.DataRequest.PeopleRequestFields
   */
  SORT_ORDER : 'sortOrder',

  /**
   * How to filter the people objects; defaults to ALL.
   * Possible values are defined by
   * <a href="opensocial.DataRequest.FilterType.html">FilterType</a>.
   * This field may be used interchangeably with the string 'filter'.
   * @member opensocial.DataRequest.PeopleRequestFields
   */
  FILTER : 'filter',

  /**
   * Additional options to be passed into the filter,
   * specified as a Map&lt;String, Object>.
   * This field may be used interchangeably with the string 'filterOptions'.
   * @member opensocial.DataRequest.PeopleRequestFields
   */
  FILTER_OPTIONS: 'filterOptions',

  /**
   * When paginating, the index of the first item to fetch;
   * specified as a number.
   * This field may be used interchangeably with the string 'first'.
   * @member opensocial.DataRequest.PeopleRequestFields
   */
  FIRST : 'first',

  /**
   * The maximum number of items to fetch,
   * specified as a number; defaults to 20. If set to a larger
   * number, a container may honor the request, or may limit the number to a
   * container-specified limit of at least 20.
   * This field may be used interchangeably with the string 'max'.
   * @member opensocial.DataRequest.PeopleRequestFields
   */
  MAX : 'max'
};


/**
 * Creates an item to request a profile for the specified person ID.
 * When processed, returns a
 * <a href="opensocial.Person.html"><code>Person</code></a> object.
 *
 * @param {String} id The ID of the person to fetch; can be the standard
 *    <a href="opensocial.DataRequest.PersonId.html">person ID</a>
 *    of VIEWER or OWNER
 * @param {Map.&lt;opensocial.DataRequest.PeopleRequestFields, Object&gt;}
 *  opt_params
 *    Additional
 *    <a href="opensocial.DataRequest.PeopleRequestFields.html">parameters</a>
 *    to pass to the request; this request supports PROFILE_DETAILS
 * @return {Object} A request object
 */
opensocial.DataRequest.prototype.newFetchPersonRequest = function(id,
    opt_params) {};


/**
 * Creates an item to request friends from the server.
 * When processed, returns a <a href="opensocial.Collection.html">Collection</a>
 * &lt;<a href="opensocial.Person.html">Person</a>&gt; object.
 *
 * @param {opensocial.IdSpec} idSpec An IdSpec used to specify
 *    which people to fetch. See also <a href="opensocial.IdSpec.html">IdSpec</a>.
 * @param {Map.&lt;opensocial.DataRequest.PeopleRequestFields, Object&gt;}
 *  opt_params
 *    Additional
 *    <a href="opensocial.DataRequest.PeopleRequestFields.html">params</a>
 *    to pass to the request
 * @return {Object} A request object
 */
opensocial.DataRequest.prototype.newFetchPeopleRequest = function(idSpec,
    opt_params) {};


/**
 * @static
 * @class
 * @name opensocial.DataRequest.DataRequestFields
 */
opensocial.DataRequest.DataRequestFields = {
  /**
   * How to escape person data returned from the server; defaults to HTML_ESCAPE.
   * Possible values are defined by
   * <a href="opensocial.EscapeType.html">EscapeType</a>.
   * This field may be used interchangeably with the string 'escapeType'.
   * @member opensocial.DataRequest.DataRequestFields
   */
  ESCAPE_TYPE : 'escapeType'
};


/**
 * Creates an item to request app data for the given people.
 * When processed, returns a Map&lt;
 * <a href="opensocial.DataRequest.PersonId.html">PersonId</a>,
 * Map&lt;String,
 * Object&gt;&gt; object. All of the data values returned will be valid json.
 *
 * @param {opensocial.IdSpec} idSpec An IdSpec used to specify
 *    which people to fetch. See also <a href="opensocial.IdSpec.html">IdSpec</a>.
 * @param {Array.&lt;String&gt; | String} keys The keys you want data for; this
 *     can be an array of key names, a single key name, or "*" to mean
 *     "all keys"
 * @param {Map.&lt;opensocial.DataRequest.DataRequestFields, Object&gt;}
 *  opt_params Additional
 *    <a href="opensocial.DataRequest.DataRequestFields.html">params</a>
 *    to pass to the request
 * @return {Object} A request object
 */
opensocial.DataRequest.prototype.newFetchPersonAppDataRequest = function(idSpec,
    keys, opt_params) {};


/**
 * Creates an item to request an update of an app field for the given person.
 * When processed, does not return any data.
 *
 * @param {String} id The ID of the person to update; only the
 *     special <code>VIEWER</code> ID is currently allowed.
 * @param {String} key The name of the key. This may only contain alphanumeric
 *     (A-Za-z0-9) characters, underscore(_), dot(.) or dash(-).
 * @param {Object} value The value, must be valid json
 * @return {Object} A request object
 */
opensocial.DataRequest.prototype.newUpdatePersonAppDataRequest = function(id,
    key, value) {};


/**
 * Deletes the given keys from the datastore for the given person.
 * When processed, does not return any data.
 *
 * @param {String} id The ID of the person to update; only the
 *     special <code>VIEWER</code> ID is currently allowed.
 * @param {Array.&lt;String&gt; | String} keys The keys you want to delete from
 *     the datastore; this can be an array of key names, a single key name,
 *     or "*" to mean "all keys"
 * @return {Object} A request object
 */
opensocial.DataRequest.prototype.newRemovePersonAppDataRequest = function(id,
    keys) {};


/**
 * Creates an item to request an activity stream from the server.
 *
 * <p>
 * When processed, returns a Collection&lt;Activity&gt;.
 * </p>
 *
 * @param {opensocial.IdSpec} idSpec An IdSpec used to specify
 *    which people to fetch. See also <a href="opensocial.IdSpec.html">IdSpec</a>.
 * @param {Map.&lt;opensocial.DataRequest.ActivityRequestFields, Object&gt;}
 *  opt_params
 *    Additional parameters
 *    to pass to the request; not currently used
 * @return {Object} A request object
 */
opensocial.DataRequest.prototype.newFetchActivitiesRequest = function(idSpec,
    opt_params) {};
