package jp.sfjp.armadillo.ui.console;

import java.io.*;
import java.util.*;
import jp.sfjp.armadillo.*;
import jp.sfjp.armadillo.archive.*;

public final class CommandLine {

    static final ResourceManager res = ResourceManager.getInstance("jp.sfjp.armadillo.ui.messages");

    static void usage(int messageId) {
        switch (messageId) {
            case 1:
                System.out.println(res.get("msg.warn-nofunction"));
                break;
            case 0:
            default:
                // no message
        }
        System.out.println(res.get("msg.usage"));
        System.exit(2);
    }

    static void execute(char function, String firstPath, List<String> paths) throws IOException {
        switch (function) {
            case 'c':
                // create archive
                if (paths.isEmpty())
                    usage(0);
                createArchive(new File(firstPath), paths);
                break;
            case 't':
                // list archive
                if (!paths.isEmpty())
                    System.out.println(res.get("msg.warn-ignoresecondfile"));
                listArchive(new File(firstPath));
                break;
            case 'x':
                // extract archive
                extractArchive(firstPath, paths);
                break;
            case 'z':
                // compress
                break;
            case 'd':
                // decompress
                break;
            case 'v':
                // validate
                if (ArmadilloCommands.validate(new File(firstPath)))
                    System.out.println("OK");
                else
                    System.out.println("NG");
                break;
            default:
                throw new IllegalStateException("function=" + function);
        }
    }

    static void createArchive(File afile, List<String> targets) throws IOException {
        ArchiveType type = ArchiveType.of(afile.getName());
        ArchiveCreator ac = ArmadilloCommands.getArchiveCreator(afile, type);
        try {
            for (final String name : targets) {
                if (name.length() == 0)
                    continue;
                final File file = new File(name);
                String s = name.replace('\\', '/');
                if (file.isDirectory() && !s.endsWith("/"))
                    s += "/";
                if (file.isDirectory())
                    for (final File child : ArmadilloCommands.getAllChildrenFile(file))
                        addEntry(ac, child, s);
                else
                    addEntry(ac, file, s);
            }
        }
        finally {
            ac.close();
        }
    }

    private static ArchiveEntry addEntry(ArchiveCreator ac, File file, String name) throws IOException {
        ArchiveEntry entry = ac.newEntry(name);
        entry.setSize(file.length());
        entry.setLastModified(file.lastModified());
        assert !entry.isDirectory() || entry.getSize() == 0L;
        ac.addEntry(entry, file);
        return entry;
    }

    static void listArchive(File afile) throws IOException {
        ArchiveType type = ArchiveType.of(afile.getName());
        ArchiveExtractor ax = ArmadilloCommands.getArchiveExtractor(afile, type);
        try {
            while (true) {
                ArchiveEntry entry = ax.nextEntry();
                if (entry == ArchiveEntry.NULL)
                    break;
                final String methodName = entry.getMethodName();
                System.out.print((methodName.length() < 8)
                                 ? String.format("%-8s", methodName)
                                         : methodName.substring(0, 8));
                System.out.printf("%9d", entry.getSize());
                System.out.printf(" %1$tF %1$tT ", new Date(entry.getLastModified()));
                System.out.println(entry.getName());
            }
        }
        finally {
            ax.close();
        }
    }

    static void extractArchive(String firstPath, List<String> paths) throws IOException {
        final String dstdir;
        if (paths.isEmpty())
            dstdir = firstPath + ".tmp";
        else {
            if (paths.size() > 1)
                System.out.println("msg.warn-ignorethirdfile");
            dstdir = paths.get(0);
        }
        ArmadilloCommands.extractAll(new File(firstPath), new File(dstdir));
    }

    public static void main(String[] args) {
        char function = ' ';
        int functionCount = 0;
        List<String> of = new ArrayList<String>();
        List<String> paths = new ArrayList<String>();
        for (final String arg : args)
            if (arg.equals("--")) {
                function = '-';
                ++functionCount;
            }
            else if (arg.startsWith("--")) {
                if (arg.equals("--validate")) {
                    function = 'v';
                    ++functionCount;
                }
                else if (arg.equals("--version")) {
                    System.out.println(res.get(".version", ArmadilloCommands.getVersionString(true)));
                    return;
                }
                else
                    usage(0);
            }
            else if (arg.equals("-"))
                usage(0);
            else if (arg.startsWith("-"))
                for (int i = 1; i < arg.length(); i++) {
                    final char c = arg.charAt(i);
                    switch (c) {
                        case 'c':
                        case 't':
                        case 'x':
                        case 'z':
                        case 'd':
                        case 'e':
                        case 'v':
                            function = c;
                            ++functionCount;
                            break;
                        default:
                            of.add(String.valueOf(c));
                    }
                }
            else if (functionCount == 0)
                usage(1);
            else
                paths.add(arg);
        if (functionCount != 1)
            usage(1);
        if (paths.isEmpty())
            usage(0);
        // main
        final String firstPath = paths.remove(0);
        try {
            execute(function, firstPath, paths);
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }
    }

}
