package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;

public final class ZipOutputStream extends ArchiveOutputStream {

    private ZipHeader header;
    private Deflater deflater;
    private CRC32 crc;
    private ZipEntry ongoingEntry;

    public ZipOutputStream(OutputStream os) {
        super(os);
        this.header = new ZipHeader();
        this.deflater = new Deflater(Deflater.DEFAULT_COMPRESSION, true);
        this.crc = new CRC32();
        frontStream = os;
    }

    public void putNextEntry(ZipEntry entry) throws IOException {
        ensureOpen();
        if (ongoingEntry != null)
            closeEntry();
        assert entry.method != -1;
        assert entry.uncompsize >= 0;
        if (entry.isDirectory()) {
            entry.version = 10;
            entry.flags &= 0xFFF7;
            entry.compsize = 0;
            entry.uncompsize = 0;
            entry.crc = 0;
        }
        else if (entry.method == ZipEntry.STORED) {
            final int size = (entry.uncompsize >= 0) ? entry.uncompsize : entry.compsize;
            entry.version = 10;
            entry.compsize = size;
            entry.uncompsize = size;
        }
        else if (entry.method == ZipEntry.DEFLATED) {
            entry.version = 20;
            if (entry.isOnePassMode())
                entry.flags |= 0x08; // EXT
            if (!entry.isDirectory())
                frontStream = new DeflaterOutputStream(out, deflater);
        }
        else
            throw new ZipException("unsupported compression method: " + entry.method);
        ongoingEntry = entry;
        header.write(out, entry);
    }

    public void closeEntry() throws IOException {
        ensureOpen();
        flush();
        if (frontStream instanceof DeflaterOutputStream) {
            DeflaterOutputStream deflaterOutputStream = (DeflaterOutputStream)frontStream;
            deflaterOutputStream.finish();
            deflaterOutputStream.flush();
            frontStream = out;
        }
        final int crc32 = (int)(crc.getValue() & 0xFFFFFFFF);
        ongoingEntry.crc = crc32;
        if (!ongoingEntry.isDirectory())
            if (ongoingEntry.hasEXT()) {
                ongoingEntry.compsize = deflater.getTotalOut();
                ongoingEntry.uncompsize = deflater.getTotalIn();
                header.writeEXT(out, ongoingEntry);
            }
            else if (ongoingEntry.method == ZipEntry.DEFLATED)
                if (deflater.getTotalOut() != ongoingEntry.compsize
                    || deflater.getTotalIn() != ongoingEntry.uncompsize)
                    throw new ZipException("invalid header info");
        deflater.reset();
        crc.reset();
        ongoingEntry = null;
    }

    @Override
    public void write(int b) throws IOException {
        super.write(b);
        crc.update(b);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        super.write(b, off, len);
        crc.update(b, off, len);
    }

    @Override
    public void close() throws IOException {
        try {
            flush();
            header.writeEND(out);
            deflater.end();
        }
        finally {
            header = null;
            deflater = null;
            crc = null;
            ongoingEntry = null;
            super.close();
        }
    }

}
