package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

public final class ZipInputStream extends ArchiveInputStream {

    private static final int BUFFER_SIZE = 16384;

    private ZipHeader header;
    private ZipEntry ongoingEntry;
    private Inflater inflater;

    public ZipInputStream(InputStream is) {
        super(new RewindableInputStream(is, BUFFER_SIZE));
        this.header = new ZipHeader();
        this.inflater = new Inflater(true);
    }

    public ZipEntry getNextEntry() throws IOException {
        ensureOpen();
        if (ongoingEntry != null)
            closeEntry();
        ZipEntry entry = header.readLOC(in);
        if (entry == null)
            return null;
        return openEntry(entry);
    }

    ZipEntry openEntry(ZipEntry entry) throws IOException {
        assert entry != null;
        if (entry.isDirectory() || entry.method == ZipEntry.STORED)
            frontStream = in;
        else if (entry.method == ZipEntry.DEFLATED)
            frontStream = new InflaterInputStream(in, inflater, 512);
        else {
            System.out.printf("method=%d, name=%s%n", entry.method, entry.getName());
            frontStream = in;
        }
        if (entry.hasEXT())
            if (entry.isDirectory() && !header.readEXT(in, entry))
                throw new ZipException("failed to read EXT header on stream mode (0)");
            else {
                byte[] buffer = new byte[BUFFER_SIZE];
                final int readSize = in.read(buffer);
                final int offset = findSIGEXT(buffer);
                if (offset < 0)
                    throw new ZipException("failed to read EXT header on stream mode (1)");
                ((RewindableInputStream)in).rewind(readSize);
                InputStream bis = new ByteArrayInputStream(buffer, offset, ZipHeader.LENGTH_EXT);
                if (!header.readEXT(bis, entry))
                    throw new ZipException("failed to read EXT header on stream mode (2)");
            }
        remaining = entry.getSize();
        return ongoingEntry = entry;
    }

    public void closeEntry() throws IOException {
        ZipEntry entry = ongoingEntry;
        ensureOpen();
        if (!entry.isDirectory()) {
            // jump to the head of next header if need
            if (remaining > 0) {
                final long compsize = entry.compsize;
                final long uncompsize = entry.uncompsize;
                if (remaining == uncompsize) {
                    long skip = compsize;
                    while (skip > 0)
                        skip -= in.skip(skip);
                    remaining -= uncompsize;
                    assert (skip != 0) : "skipped size should be zero, but was " + skip;
                }
            }
            final int inflaterRemaining = inflater.getRemaining();
            if (inflaterRemaining > 0) // overread
                ((RewindableInputStream)in).rewind(inflaterRemaining);
            assert remaining == 0 : "remaining should be zero, but was %d" + remaining;
            if (entry.hasEXT() && !header.readEXT(in, entry))
                throw new ZipException("failed to read EXT header on stream mode (3)");
        }
        // reset
        ongoingEntry = null;
        inflater.reset();
        frontStream = in;
    }

    private static int findSIGEXT(byte[] bytes) {
        // SIGEXT=0x504B0708
        for (int i = 0; i < bytes.length - 3; i++)
            if (bytes[i] == 0x50)
                if (bytes[i + 1] == 0x4B)
                    if (bytes[i + 2] == 0x07)
                        if (bytes[i + 3] == 0x08)
                            return i;
        return -1;
    }

    @Override
    public void close() throws IOException {
        header = null;
        inflater = null;
        ongoingEntry = null;
    }

}
