package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.time.*;

/**
 * ZIP archive entry.
 */
public final class ZipEntry extends ArchiveEntry {

    /**
     * <code>STORED</code>
     */
    public static final short STORED = 0;

    /**
     * <code>DEFLATED</code>
     */
    public static final short DEFLATED = 8;

    private static final long UINT_MAX = 0xFFFFFFFFL;
    private static final long USHORT_MAX = 0xFFFFL;
    private static final FTime FTIME = new FTime();

    // state
    boolean extOverwritten;
    int reloff;
    // header
    int signature;
    short version;
    short flags;
    short method;
    short mtime;
    short mdate;
    int crc;
    int compsize;
    int uncompsize;
    short extlen;

    public ZipEntry() {
        super(false);
        this.version = 10;
        this.flags = 0;
        this.method = DEFLATED;
        this.mdate = 0;
        this.mtime = 0;
        this.crc = -1;
        this.compsize = 0;
        this.uncompsize = 0;
    }

    public ZipEntry(String name) {
        this();
        setName(name);
    }

    public ZipEntry(String name, File file) {
        this(name);
        setFileInfo(file);
    }

    public boolean hasEXT() {
        return (flags & 8) == 8;
    }

    public boolean isOnePassMode() {
        return crc == -1;
    }

    @Override
    public long getSize() {
        if (isDirectory())
            return 0;
        return uncompsize & UINT_MAX;
    }

    @Override
    public void setSize(long size) {
        if (size > UINT_MAX)
            throw new IllegalArgumentException("max size is int.max: " + size);
        this.uncompsize = (int)(size & UINT_MAX);
    }

    @Override
    public long getCompressedSize() {
        if (isDirectory())
            return 0;
        return compsize & UINT_MAX;
    }

    @Override
    public void setCompressedSize(long compressedSize) {
        if (compressedSize > UINT_MAX)
            throw new IllegalArgumentException("max size is int-max: " + compressedSize);
        this.compsize = (int)(compressedSize & UINT_MAX);
    }

    @Override
    public long getLastModified() {
        return FTIME.toMillisecond(mdate, mtime);
    }

    @Override
    public void setLastModified(long lastModified) {
        final int ftime = FTIME.int32From(lastModified);
        this.mdate = (short)((ftime >> 16) & USHORT_MAX);
        this.mtime = (short)(ftime & USHORT_MAX);
    }

    @Override
    public String getMethodName() {
        return getMethodName(method);
    }

    private static String getMethodName(int method) {
        switch (method) {
            case ZipEntry.DEFLATED:
                return "DEFLATED";
            case ZipEntry.STORED:
                return "STORED";
            default:
                return String.valueOf(method);
        }
    }

}
