package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

public final class ZipArchiveCreator implements ArchiveCreator {

    private ZipOutputStream os;

    public ZipArchiveCreator(OutputStream os) {
        this.os = new ZipOutputStream(os);
    }

    @Override
    public ArchiveEntry newEntry(String name) {
        return new ZipEntry(name);
    }

    @Override
    public void addEntry(ArchiveEntry entry, File file) throws IOException {
        if (file.isDirectory()) {
            os.putNextEntry(toZipEntry(entry));
            os.closeEntry();
            entry.setAdded(true);
        }
        else
            addEntry(entry, file, null, file.length());
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        addEntry(entry, null, is, length);
    }

    private void addEntry(ArchiveEntry entry, File file, InputStream src, final long length) throws IOException {
        ZipEntry fileEntry = toZipEntry(entry);
        fileEntry.setSize(length);
        if (length == 0) {
            fileEntry.setCompressedSize(0L);
            fileEntry.method = ZipEntry.STORED;
        }
        os.putNextEntry(fileEntry);
        if (length > 0) {
            final long size;
            InputStream is = (src == null) ? new FileInputStream(file) : src;
            try {
                size = IOUtilities.transfer(is, os, length);
            }
            finally {
                if (src == null)
                    is.close();
            }
            assert size == fileEntry.getSize() : "file size";
            assert size == length : "file size";
            entry.setSize(size);
        }
        os.closeEntry();
        entry.setAdded(true);
    }

    static ZipEntry toZipEntry(ArchiveEntry entry) {
        if (entry instanceof ZipEntry)
            return (ZipEntry)entry;
        ZipEntry newEntry = new ZipEntry(entry.getName());
        entry.copyTo(newEntry);
        if (newEntry.isDirectory())
            newEntry.method = ZipEntry.STORED;
        return newEntry;
    }

    @Override
    public void close() throws IOException {
        os.close();
    }

}
