package jp.sfjp.armadillo.archive.tar;

import java.io.*;
import java.util.*;

public final class TarHeader {

    static final int BLOCK_SIZE = 512;

    private static final String MAGIC_USTAR = "ustar";

    private final byte[] buffer;
    private int position;

    public TarHeader() {
        this.buffer = new byte[BLOCK_SIZE];
        this.position = 0;
    }

    public TarEntry read(InputStream is) throws IOException {
        reset();
        final int readLength = draw(is, buffer);
        if (readLength == 0)
            return null;
        if (readLength != BLOCK_SIZE)
            throw new TarException("bad header: size=" + readLength);
        if (buffer[0] == 0x00 && isEmptyBlock()) {
            if (is.read(buffer) == BLOCK_SIZE && isEmptyBlock())
                return null;
            throw new TarException("bad end-of-archive");
        }
        try {
            if (isUstar())
                return readUstar(is);
            else
                return readTar(is);
        }
        catch (RuntimeException ex) {
            throw new TarException("bad header at " + position, ex);
        }
    }

    private boolean isUstar() {
        final String s = new String(buffer, 257, 5);
        return s.equals(MAGIC_USTAR);
    }

    @SuppressWarnings("unused")
    private TarEntry readTar(InputStream is) throws IOException {
        // Header Block (TAR Format)
        final byte[] name; // name of file (100bytes)
        final int mode; // file mode (8bytes)
        final int uid; // owner user ID (8bytes)
        final int gid; // owner group ID (8bytes)
        final long size; // length of file in bytes (12bytes)
        final long mtime; // modify time of file (12bytes)
        final int chksum; // checksum for header (8bytes)
        final int link; // indicator for links (1byte)
        final String linkname; // name of linked file (100bytes)
        // ---
        name = clipNameField(100);
        mode = clipAsInt(8);
        uid = clipAsInt(8);
        gid = clipAsInt(8);
        size = clipAsLong(12);
        mtime = clipAsLong(12);
        chksum = clipAsInt(8);
        link = clipAsInt(1);
        linkname = clipAsString(100);
        TarEntry entry = new TarEntry();
        entry.setName(name);
        entry.setMode(mode);
        entry.uid = uid;
        entry.gid = gid;
        entry.size = size;
        entry.mtime = mtime;
        entry.chksum = chksum;
        entry.linkname = linkname;
        return entry;
    }

    private TarEntry readUstar(InputStream is) throws IOException {
        TarEntry entry = new TarEntry();
        readUstar(is, entry);
        if (entry.typeflag == 'L') {
            // LongLink
            final int readLength = draw(is, buffer);
            assert readLength == BLOCK_SIZE;
            assert entry.size <= Integer.MAX_VALUE;
            position = 0;
            byte[] bytes = clipNameField((int)entry.size);
            position = 0;
            final int readLength2 = draw(is, buffer);
            assert readLength2 == BLOCK_SIZE;
            readUstar(is, entry);
            entry.setName(bytes);
        }
        return entry;
    }

    private void readUstar(InputStream is, TarEntry entry) throws IOException {
        // Header Block (USTAR Format)
        final byte[] name; // name of file (100bytes)
        final int mode; // file mode (8bytes)
        final int uid; // owner user ID (8bytes)
        final int gid; // owner group ID (8bytes)
        final long size; // length of file in bytes (12bytes)
        final long mtime; // modify time of file (12bytes)
        final int chksum; // checksum for header (8bytes)
        final char typeflag; // type of file (1byte)
        final String linkname; // name of linked file (100bytes)
        final String magic; // USTAR indicator (6bytes)
        final String version; // USTAR version (2bytes)
        final String uname; // owner user name (32bytes)
        final String gname; // owner group name (32bytes)
        final String devmajor; // device major number (8bytes)
        final String devminor; // device minor number (8bytes)
        final String prefix; // prefix for file name (155bytes)
        // ---
        name = clipNameField(100);
        mode = clipAsInt(8);
        uid = clipAsInt(8);
        gid = clipAsInt(8);
        size = clipAsLong(12);
        mtime = clipAsLong(12);
        chksum = clipAsInt(8);
        typeflag = clipChar();
        linkname = clipAsString(100);
        magic = clipAsString(6);
        version = clipAsString(2);
        uname = clipAsString(32);
        gname = clipAsString(32);
        devmajor = clipAsString(8);
        devminor = clipAsString(8);
        prefix = clipAsString(155);
        entry.setName(name);
        entry.setMode(mode);
        entry.uid = uid;
        entry.gid = gid;
        entry.size = size;
        entry.mtime = mtime;
        entry.chksum = chksum;
        entry.typeflag = typeflag;
        entry.linkname = linkname;
        entry.magic = magic;
        entry.version = version;
        entry.uname = uname;
        entry.gname = gname;
        entry.devmajor = devmajor;
        entry.devminor = devminor;
        entry.prefix = prefix;
    }

    private char clipChar() {
        final String s = clipAsString(1);
        if (s.isEmpty())
            return ' ';
        return s.charAt(0);
    }

    private static int draw(InputStream is, byte[] bytes) throws IOException {
        int readLength = 0;
        int offset = 0;
        while (readLength < BLOCK_SIZE) {
            int read = is.read(bytes, offset, BLOCK_SIZE - readLength);
            if (read <= 0)
                break;
            offset += read;
            readLength += read;
        }
        return readLength;
    }

    public void write(OutputStream os, TarEntry entry) throws IOException {
        try {
            reset();
            if (entry.magic.startsWith(MAGIC_USTAR))
                writeUstar(os, entry);
            else
                writeTar(os, entry);
        }
        catch (RuntimeException ex) {
            throw new TarException("bad header at " + position, ex);
        }
    }

    private void writeTar(OutputStream os, TarEntry entry) throws IOException {
        // Header Block (TAR Format)
        final String name; // name of file (100bytes)
        final int mode; // file mode (8bytes)
        final int uid; // owner user ID (8bytes)
        final int gid; // owner group ID (8bytes)
        final long size; // length of file in bytes (12bytes)
        final long mtime; // modify time of file (12bytes)
        final int chksum; // checksum for header (8bytes)
        final int link; // indicator for links (1byte)
        final String linkname; // name of linked file (100bytes)
        // ---
        name = entry.name();
        mode = entry.getMode();
        uid = entry.uid;
        gid = entry.gid;
        size = entry.size;
        mtime = entry.mtime;
        chksum = entry.chksum;
        link = 0;
        linkname = entry.linkname;
        patch(100, name);
        patch(8, mode);
        patch(8, uid);
        patch(8, gid);
        patch(12, size);
        patch(12, mtime);
        patch(8, chksum);
        patch(1, link);
        patch(100, linkname);
        os.write(buffer);
        throw new TarException("not impl yet (old Tar)");
    }

    @SuppressWarnings("unused")
    private void writeUstar(OutputStream os, TarEntry entry) throws IOException {
        // Header Block (USTAR Format)
        final String name; // name of file (100bytes)
        final int mode; // file mode (8bytes)
        final int uid; // owner user ID (8bytes)
        final int gid; // owner group ID (8bytes)
        final long size; // length of file in bytes (12bytes)
        final long mtime; // modify time of file (12bytes)
        final int chksum; // checksum for header (8bytes)
        final int typeflag; // type of file (1byte)
        final String linkname; // name of linked file (100bytes)
        final String magic; // USTAR indicator (6bytes)
        final String version; // USTAR version (2bytes)
        final String uname; // owner user name (32bytes)
        final String gname; // owner group name (32bytes)
        final String devmajor; // device major number (8bytes)
        final String devminor; // device minor number (8bytes)
        final String prefix; // prefix for file name (155bytes)
        // ---
        name = entry.name();
        mode = entry.getMode();
        uid = entry.uid;
        gid = entry.gid;
        size = entry.size;
        mtime = entry.mtime;
        chksum = entry.chksum;
        typeflag = entry.typeflag;
        linkname = entry.linkname;
        magic = entry.magic;
        version = entry.version;
        uname = entry.uname;
        gname = entry.gname;
        devmajor = entry.devmajor;
        devminor = entry.devminor;
        prefix = entry.prefix;
        patch(100, name);
        patch(8, mode);
        patch(8, uid);
        patch(8, gid);
        patch(12, size);
        patch(12, mtime);
        patch(8, "        "); // set after calculating checksum
        patch(1, typeflag);
        patch(100, linkname);
        patch(6, MAGIC_USTAR + ' '); // ignore input
        patch(2, version);
        patch(32, uname);
        patch(32, gname);
        patch(8, devmajor);
        patch(8, devminor);
        patch(155, prefix);
        // calculate checksum
        int checksum = 0;
        for (int i = 0; i < buffer.length; i++)
            checksum += (buffer[i] & 0xFF);
        position = 148;
        patch(6, checksum);
        entry.chksum = checksum;
        os.write(buffer);
    }

    public void writeEndOfArchive(OutputStream os) throws IOException {
        reset();
        os.write(buffer);
        os.write(buffer);
        os.flush();
    }

    public void reset() {
        position = 0;
        Arrays.fill(buffer, (byte)0);
    }

    private byte[] clipNameField(int length) {
        assert length <= BLOCK_SIZE - position;
        final int p = position;
        position += length;
        int availableLength = 0;
        for (int i = 0; i < length; i++) {
            if (buffer[p + i] == 0x00)
                break;
            ++availableLength;
        }
        byte[] nameb = new byte[availableLength];
        System.arraycopy(buffer, 0, nameb, 0, availableLength);
        return nameb;
    }

    private String clipAsString(int length) {
        assert length <= BLOCK_SIZE - position;
        final int p = position;
        position += length;
        int availableLength = 0;
        for (int i = 0; i < length; i++) {
            if (buffer[p + i] == 0x00)
                break;
            ++availableLength;
        }
        StringBuilder s = new StringBuilder(length);
        for (int i = 0; i < availableLength; i++) {
            final byte b = buffer[p + i];
            if (b < 0x20 || b > 0x7F)
                s.append(String.format("\\%o", (b & 0xFF)));
            else
                s.append((char)b);
        }
        return s.toString();
    }

    private int clipAsInt(int length) {
        final String s = clipAsNumberString(length);
        return s.isEmpty() ? 0 : Integer.parseInt(s, 8);
    }

    private long clipAsLong(int length) {
        final String s = clipAsNumberString(length);
        return s.isEmpty() ? 0 : Long.parseLong(s, 8);
    }

    private String clipAsNumberString(int length) {
        assert length <= BLOCK_SIZE - position;
        final int p = position;
        position += length;
        int i = 0;
        for (; i < length; i++) {
            final byte b = buffer[p + i];
            if (b == 0x00)
                break;
            assert b == 0x20 || b >= 0x30 && b <= 0x39;
            boolean x = b == 0x20 || b >= 0x30 && b <= 0x39;
            if (!x)
                System.out.print("");
        }
        return (new String(buffer, p, i)).trim();
    }

    private int patch(int length, String value) {
        final int p = position;
        position += length;
        byte[] data;
        if (value == null || value.isEmpty())
            data = new byte[0];
        else
            data = value.getBytes();
        final int width = (data.length > length) ? length : data.length;
        System.arraycopy(data, 0, buffer, p, width);
        return data.length;
    }

    private int patch(int length, int value) {
        final String s = padZero(Integer.toOctalString(value), length - 1);
        return patch(length, s);
    }

    private int patch(int length, long value) {
        final String s = padZero(Long.toOctalString(value), length - 1);
        return patch(length, s);
    }

    private String padZero(String value, int length) {
        final int valueLength = value.length();
        if (valueLength >= length)
            return value;
        char[] buffer = new char[length];
        int p = length - valueLength;
        for (int i = 0; i < p; i++)
            buffer[i] = '0';
        for (int i = p; i < length; i++)
            buffer[i] = value.charAt(i - p);
        return String.valueOf(buffer);
    }

    private boolean isEmptyBlock() {
        for (int i = 0; i < buffer.length; i++)
            if (buffer[i] != 0x00)
                return false;
        return true;
    }

    static long getSkipSize(long size) {
        if (size == 0 || size == BLOCK_SIZE || size % BLOCK_SIZE == 0)
            return 0;
        else
            return BLOCK_SIZE - ((size > BLOCK_SIZE) ? size % BLOCK_SIZE : size);

    }

}
