package jp.sfjp.armadillo.archive.tar;

import java.io.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.time.*;

/**
 * TAR entry.
 */
public final class TarEntry extends ArchiveEntry {

    /** default mode for dir */
    public static final int DEFAULT_MODE_DIR = 0755;

    /** default mode for file */
    public static final int DEFAULT_MODE_FILE = 0666;

    private static final TimeT TIME_T = new TimeT();

    private int mode;
    int uid;
    int gid;
    long size;
    long mtime;
    int chksum;
    char typeflag;
    String linkname;
    String magic;
    String version;
    String uname;
    String gname;
    String devmajor;
    String devminor;
    String prefix;

    TarEntry() {
        super(false);
        this.mode = 0;
        this.uid = -1;
        this.gid = -1;
        this.size = 0L;
        this.mtime = 0;
        this.chksum = 0;
        this.typeflag = 0;
        this.linkname = "";
        this.magic = "ustar"; // default format
        this.version = "";
        this.uname = "unknown";
        this.gname = "unknown";
        this.devmajor = "";
        this.devminor = "";
        this.prefix = "";
    }

    public TarEntry(String name) {
        this();
        setName(name);
        this.mode = (isDirectory()) ? DEFAULT_MODE_DIR : DEFAULT_MODE_FILE;
    }

    public TarEntry(String name, File file) {
        this(name);
        setFileInfo(file);
    }

    @Override
    public long getSize() {
        return size;
    }

    @Override
    public void setSize(long size) {
        if (!isDirectory())
            this.size = size;
    }

    @Override
    public long getCompressedSize() {
        return -1;
    }

    @Override
    public void setCompressedSize(long size) {
        // ignore
    }

    @Override
    public long getLastModified() {
        return TIME_T.toMilliseconds(mtime);
    }

    @Override
    public void setLastModified(long time) {
        this.mtime = TIME_T.int64From(time);
    }

    @Override
    public String getMethodName() {
        if (!this.magic.trim().isEmpty())
            return this.magic.trim();
        return "TAR";
    }

    public int getMode() {
        return mode;
    }

    public void setMode(int mode) {
        this.mode = mode;
    }

    public int getUid() {
        return uid;
    }

    public void setUid(int uid) {
        this.uid = uid;
    }

    public int getGid() {
        return gid;
    }

    public void setGid(int gid) {
        this.gid = gid;
    }

    public long getMtime() {
        return mtime;
    }

    public void setMtime(long mtime) {
        this.mtime = mtime;
    }

    public int getChksum() {
        return chksum;
    }

    public void setChksum(int chksum) {
        this.chksum = chksum;
    }

    public char getTypeflag() {
        return typeflag;
    }

    public void setTypeflag(char typeflag) {
        this.typeflag = typeflag;
    }

    public String getLinkname() {
        return linkname;
    }

    public void setLinkname(String linkname) {
        this.linkname = linkname;
    }

    public String getMagic() {
        return magic;
    }

    public void setMagic(String magic) {
        this.magic = magic;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public String getUname() {
        return uname;
    }

    public void setUname(String uname) {
        this.uname = uname;
    }

    public String getGname() {
        return gname;
    }

    public void setGname(String gname) {
        this.gname = gname;
    }

    public String getDevmajor() {
        return devmajor;
    }

    public void setDevmajor(String devmajor) {
        this.devmajor = devmajor;
    }

    public String getDevminor() {
        return devminor;
    }

    public void setDevminor(String devminor) {
        this.devminor = devminor;
    }

    public String getPrefix() {
        return prefix;
    }

    public void setPrefix(String prefix) {
        this.prefix = prefix;
    }

}
