package jp.sfjp.armadillo.archive.tar;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.time.*;

/**
 * Dump TAR archive header.
 */
public final class DumpTarHeader extends DumpArchiveHeader {

    private static final int BLOCK_SIZE = 512;
    private static final String MAGIC_USTAR = "ustar";
    private static final TimeT TIME_T = new TimeT();

    private static final String fmt1 = "  * %s = %s%n";
    private static final String fmt2 = "  %1$-16s = [0x%2$08X] [0o%2$012o] ( %2$d )%n";
    private static final String fmt3 = "  %-16s = \"%s\"%n";

    private final ByteBuffer buffer;

    public DumpTarHeader() {
        this.buffer = ByteBuffer.allocate(BLOCK_SIZE);
    }

    @Override
    public void dump(InputStream is, PrintWriter out) throws IOException {
        int p = 0;
        while (true) {
            printOffset(out, p);
            final int readLength = draw(is, buffer);
            if (readLength == 0)
                break;
            p += readLength;
            if (readLength != BLOCK_SIZE)
                warn(out, "bad header: size=%d", readLength);
            if (isEmptyBlock()) {
                out.println("  ( EMPTY BLOCK )");
                continue;
            }
            try {
                if (isUstar())
                    p += readUstar(is, out);
                else
                    p += readTar(is, out);
            }
            catch (Exception ex) {
                throw new IOException(ex);
            }
        }
        printEnd(out, "TAR", p);
    }

    private boolean isUstar() {
        final String s = new String(buffer.array(), 257, 5);
        return s.equals(MAGIC_USTAR);
    }

    private static int draw(InputStream is, ByteBuffer buffer) throws IOException {
        buffer.clear();
        buffer.limit(BLOCK_SIZE);
        Channels.newChannel(is).read(buffer);
        int length = buffer.position();
        buffer.rewind();
        return length;
    }

    private int readTar(InputStream is, PrintWriter out) throws IOException {
        int readSize = 0;
        // Header Block (TAR Format)
        final String name; // name of file (100bytes)
        final int mode; // file mode (8bytes)
        final int uid; // owner user ID (8bytes)
        final int gid; // owner group ID (8bytes)
        final long size; // length of file in bytes (12bytes)
        final long mtime; // modify time of file (12bytes)
        final int chksum; // checksum for header (8bytes)
        final int link; // indicator for links (1byte)
        name = clip(100);
        mode = clipAsInt(8);
        uid = clipAsInt(8);
        gid = clipAsInt(8);
        size = clipAsLong(12);
        mtime = clipAsLong(12);
        chksum = clipAsInt(8);
        link = clipAsInt(1);
        clip(100);
        printHeaderName(out, "TAR (old) format");
        out.printf(fmt1, "name", name);
        p(out, "mode", mode, 8);
        p(out, "uid", uid, 8);
        p(out, "gid", gid, 8);
        p(out, "size", size, 12);
        p(out, "mtime", mtime, 12);
        p(out, "chksum", chksum, 8);
        p(out, "link", link, 1);
        if (size > 0)
            readSize += skipBlock(is, size);
        return readSize;
    }

    private int readUstar(InputStream is, PrintWriter out) throws IOException {
        int readSize = 0;
        // Header Block (USTAR Format)
        final String name; // name of file (100bytes)
        final int mode; // file mode (8bytes)
        final int uid; // owner user ID (8bytes)
        final int gid; // owner group ID (8bytes)
        final long size; // length of file in bytes (12bytes)
        final long mtime; // modify time of file (12bytes)
        final int chksum; // checksum for header (8bytes)
        final char typeflag; // type of file (1byte)
        final String linkname; // name of linked file (100bytes)
        final String magic; // USTAR indicator (6bytes)
        final int version; // USTAR version (2bytes)
        final String uname; // owner user name (32bytes)
        final String gname; // owner group name (32bytes)
        final int devmajor; // device major number (8bytes)
        final int devminor; // device minor number (8bytes)
        final String prefix; // prefix for file name (155bytes)
        name = clip(100);
        mode = clipAsInt(8);
        uid = clipAsInt(8);
        gid = clipAsInt(8);
        size = clipAsLong(12);
        mtime = clipAsLong(12);
        chksum = clipAsInt(8);
        typeflag = clipAsChar();
        linkname = clip(100);
        magic = clip(6);
        version = clipAsInt(2);
        uname = clip(32);
        gname = clip(32);
        devmajor = clipAsInt(8);
        devminor = clipAsInt(8);
        prefix = clip(155);
        printHeaderName(out, "USTAR format");
        out.printf(fmt1, "name", name);
        out.printf(fmt1, "mtime as date", toDate(mtime));
        p(out, "mode", mode, 8);
        p(out, "uid", uid, 8);
        p(out, "gid", gid, 8);
        p(out, "size", size, 12);
        p(out, "mtime", mtime, 12);
        p(out, "chksum", chksum, 8);
        out.printf(fmt3, "typeflag", typeflag);
        out.printf(fmt3, "linkname", linkname);
        out.printf(fmt3, "magic", magic);
        out.printf(fmt2, "version", version);
        out.printf(fmt3, "uname", uname);
        out.printf(fmt3, "gname", gname);
        out.printf(fmt2, "devmajor", devmajor);
        out.printf(fmt2, "devminor", devminor);
        out.printf(fmt3, "prefix", prefix);
        if (typeflag == 'L') {
            // LongLink
            readSize += draw(is, buffer);
            final String s = clip(512);
            out.printf(fmt1, "LongLink", s);
        }
        else if (size > 0)
            readSize += skipBlock(is, size);
        return readSize;
    }

    private long skipBlock(InputStream is, long size) throws IOException {
        long skippedSize = 0;
        long skipCount = size;
        while (skipCount > 0) {
            final long skipped = is.skip(BLOCK_SIZE);
            skipCount -= skipped;
            if (skipped != BLOCK_SIZE && skipCount > 0)
                throw new IllegalStateException("bad skip size: " + skipped);
            skippedSize += skipped;
        }
        return skippedSize;
    }

    private String clip(int length) {
        final int p = buffer.position();
        int availableLength = 0;
        for (int i = 0; i < length; i++) {
            if (buffer.get() == 0x00)
                break;
            ++availableLength;
        }
        buffer.rewind();
        buffer.position(p);
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < availableLength; i++) {
            final int x = buffer.get() & 0xFF;
            if (x < 0x20 || x > 0x7F)
                sb.append(String.format("\\%o", x));
            else
                sb.append((char)x);
        }
        buffer.position(p + length);
        return sb.toString();
    }

    private char clipAsChar() {
        final String s = clip(1);
        assert s.length() == 1 && s.matches("^[A-Za-z0-9]$");
        return s.charAt(0);
    }

    private int clipAsInt(int length) {
        final String s = clipAsNumberString(length);
        return s.isEmpty() ? 0 : Integer.parseInt(s, 8);
    }

    private long clipAsLong(int length) {
        final String s = clipAsNumberString(length);
        return s.isEmpty() ? 0 : Long.parseLong(s, 8);
    }

    private String clipAsNumberString(int length) {
        byte[] bytes = new byte[length];
        buffer.get(bytes);
        int i = 0;
        for (; i < length; i++) {
            byte b = bytes[i];
            if (b == 0x00)
                break;
            assert b == 0x20 || b >= 0x30 && b <= 0x39;
        }
        return (new String(bytes, 0, i)).trim();
    }

    private boolean isEmptyBlock() {
        byte[] bytes = buffer.array();
        for (int i = 0; i < bytes.length; i++)
            if (bytes[i] != 0x00)
                return false;
        return true;
    }

    static long getSkipSize(long size) {
        if (size == 0 || size == BLOCK_SIZE || size % BLOCK_SIZE == 0)
            return 0;
        else
            return BLOCK_SIZE - ((size > BLOCK_SIZE) ? size % BLOCK_SIZE : size);

    }

    static InputStream getInputStream(File file) throws IOException {
        ArchiveType type = ArchiveType.of(file.getName());
        switch (type) {
            case TAR:
                return new FileInputStream(file);
            case TARGZ:
                return new GZIPInputStream(new FileInputStream(file));
            default:
                throw new UnsupportedOperationException("not TAR ? : " + type);
        }
    }

    static Date toDate(long timet) {
        return new Date(TIME_T.toMilliseconds(timet));
    }

    static <T> void p(PrintWriter out, String name, T value, int width) {
        final int wh;
        final int wo;
        switch (width) {
            case 1:
                wh = 2;
                wo = 2;
                break;
            case 8:
                wh = 6;
                wo = 8;
                break;
            case 12:
                wh = 6;
                wo = 8;
                break;
            default:
                wh = 8;
                wo = width;
        }
        final String fmt = "  %1$-16s = [0x%2$0" + wh + "X] [0o%2$0" + wo + "o] ( %2$d )%n";
        out.printf(fmt, name, value);
    }

}
