package jp.sfjp.armadillo.archive.lzh;

/**
 * LZH compression method.
 */
public final class LzhMethod {

    /** <code>LHD</code> */
    public static final String LHD = "-lhd-";
    /** <code>LH0</code> */
    public static final String LH0 = "-lh0-";
    /** <code>LH4</code> */
    public static final String LH4 = "-lh4-";
    /** <code>LH5</code> */
    public static final String LH5 = "-lh5-";
    /** <code>LH6</code> */
    public static final String LH6 = "-lh6-";
    /** <code>LH7</code> */
    public static final String LH7 = "-lh7-";

    private final String methodName;
    private final int dictionarySize;
    private final int matchSize;
    private final int threshold;

    public LzhMethod(String methodName) throws LzhException {
        this.methodName = methodName;
        this.dictionarySize = detectDictionarySize(methodName);
        this.matchSize = (dictionarySize == 0) ? 0 : 256;
        this.threshold = (dictionarySize == 0) ? 0 : 3;
    }

    private static int detectDictionarySize(String methodName) throws LzhException {
        if (methodName.matches("-lh[d0]-"))
            return 0;
        else if (methodName.matches("-lh[4567]-"))
            switch (methodName.charAt(3)) {
                case '4':
                    return 4096;
                case '5':
                default:
                    return 8192;
                case '6':
                    return 32768;
                case '7':
                    return 65536;
            }
        else
            throw new LzhException("unsupported method: " + methodName);
    }

    public String getMethodName() {
        return methodName;
    }

    public int getDictionarySize() {
        return dictionarySize;
    }

    public int getMatchSize() {
        return matchSize;
    }

    public int getThreshold() {
        return threshold;
    }

    public boolean isCompressing() {
        return dictionarySize > 0;
    }

    public static boolean isCompressing(String methodName) throws LzhException {
        return detectDictionarySize(methodName) > 0;
    }

}
