package jp.sfjp.armadillo.archive.cab;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.time.*;

/**
 * Dump CAB archive header.
 */
public final class DumpCabHeader extends DumpArchiveHeader {

    // signature 'MSCF'
    static final int SIGNATURE = 0x4D534346;

    private static final FTime FTIME = new FTime();
    private static final String fmt0 = "  -- %s[%d] --%n";
    private static final String fmt1 = "  * %s = %s%n";
    private static final String fmt2 = "  %1$-16s = [0x%2$08X] ( %2$d )%n";

    private static final int BUFFER_SIZE = 1024;

    private final ByteBuffer buffer;
    private short[] folderCompressType;

    public DumpCabHeader() {
        this.buffer = ByteBuffer.allocate(BUFFER_SIZE).order(ByteOrder.LITTLE_ENDIAN);
    }

    @SuppressWarnings("unused")
    @Override
    public void dump(InputStream is, PrintWriter out) throws IOException {
        // Cabinet File header
        final int signature; // cabinet file signature
        final int reserved1; // reserved
        final int cbCabinet; // size of this cabinet file in bytes
        final int reserved2; // reserved
        final int coffFiles; // offset of the first CFFILE entry
        final int reserved3; // reserved
        final byte versionMinor; // cabinet file format version, minor
        final byte versionMajor; // cabinet file format version, major
        final short cFolders; // number of CFFOLDER entries in this cabinet
        final short cFiles; // number of CFFILE entries in this cabinet
        final short flags; // cabinet file option indicators
        final short setID; // must be the same for all cabinets in a set
        final short iCabinet;// number of this cabinet file in a set
        buffer.clear();
        buffer.limit(36);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0)
            throw new CabException("reading archive header, but position is zero");
        buffer.rewind();
        buffer.order(ByteOrder.BIG_ENDIAN);
        signature = buffer.getInt();
        if (signature != SIGNATURE)
            throw new CabException("bad signature: 0x%X", signature);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        reserved1 = buffer.getInt();
        cbCabinet = buffer.getInt();
        reserved2 = buffer.getInt();
        coffFiles = buffer.getInt();
        reserved3 = buffer.getInt();
        versionMinor = buffer.get();
        versionMajor = buffer.get();
        cFolders = buffer.getShort();
        cFiles = buffer.getShort();
        flags = buffer.getShort();
        setID = buffer.getShort();
        iCabinet = buffer.getShort();
        printHeaderName(out, "Cabinet File header");
        out.printf(fmt2, "signature", signature);
        out.printf(fmt2, "reserved1", reserved1);
        out.printf(fmt2, "cbCabinet", cbCabinet);
        out.printf(fmt2, "reserved2", reserved2);
        out.printf(fmt2, "coffFiles", coffFiles);
        out.printf(fmt2, "reserved3", reserved3);
        out.printf(fmt2, "versionMinor", versionMinor);
        out.printf(fmt2, "versionMajor", versionMajor);
        out.printf(fmt2, "cFolders", cFolders);
        out.printf(fmt2, "cFiles", cFiles);
        out.printf(fmt2, "flags", flags);
        out.printf(fmt2, "setID", setID);
        out.printf(fmt2, "iCabinet", iCabinet);
        if (iCabinet != 0) {
            warn(out, "iCabinet not supported: 0x%d", iCabinet);
            return;
        }
        // array size of folderCompressType
        folderCompressType = new short[cFolders];
        // read CFFOLDERs
        printHeaderName(out, "CFFOLDER headers");
        int cfdatacount = 0;
        for (int i = 0; i < cFolders; i++) {
            // CFFOLDER header
            final int coffCabStart; // offset of the first CFDATA block in this folder
            final short cCFData; // number of CFDATA blocks in this folder
            final short typeCompress; // compression type indicator
            final byte[] abReserve; // (optional) per-folder reserved area
            buffer.clear();
            buffer.limit(8);
            Channels.newChannel(is).read(buffer);
            if (buffer.position() == 0) {
                warn(out, "reading CFFOLDER header, but position is zero");
                return;
            }
            buffer.rewind();
            coffCabStart = buffer.getInt();
            cCFData = buffer.getShort();
            typeCompress = buffer.getShort();
            // ignore optional data
            abReserve = new byte[0];
            folderCompressType[i] = typeCompress;
            out.printf(fmt0, "CFFOLDER", i);
            out.printf(fmt2, "coffCabStart", coffCabStart);
            out.printf(fmt2, "cCFData", cCFData);
            out.printf(fmt2, "typeCompress", typeCompress);
            //            out.printf(fmt2, "abReserve", abReserve);
            cfdatacount += cCFData;
        }
        // read CFFILEs
        printHeaderName(out, "CFFILE headers");
        for (int i = 0; i < cFiles; i++) {
            // CFFILE header
            final int cbFile; // uncompressed size of this file in bytes
            final int uoffFolderStart; // uncompressed offset of this file in the folder
            final short iFolder; // index into the CFFOLDER area
            final short date; // date stamp for this file
            final short time; // time stamp for this file
            final short attribs; // attribute flags for this file
            final byte[] szName; // name of this file
            buffer.clear();
            buffer.limit(16);
            Channels.newChannel(is).read(buffer);
            if (buffer.position() == 0)
                throw new CabException("reading CFFILE header, but position is zero");
            buffer.rewind();
            cbFile = buffer.getInt();
            uoffFolderStart = buffer.getInt();
            iFolder = buffer.getShort();
            date = buffer.getShort();
            time = buffer.getShort();
            attribs = buffer.getShort();
            szName = readName(is, 256);
            final String name = new String(szName);
            out.printf(fmt0, "CFFILE", i);
            out.printf(fmt1, "name", name);
            out.printf(fmt1, "mtime as date", toDate(date, time));
            out.printf(fmt2, "cbFile", cbFile);
            out.printf(fmt2, "uoffFolderStart", uoffFolderStart);
            out.printf(fmt2, "iFolder", iFolder);
            out.printf(fmt2, "date", date);
            out.printf(fmt2, "time", time);
            out.printf(fmt2, "attribs", attribs);
            out.printf(fmt2, "szName.length", szName.length);
        }
        // read CFDATAs
        printHeaderName(out, "CFDATA headers");
        for (int i = 0; i < cfdatacount; i++) {
            // CFDATA header
            final int csum; // checksum of this CFDATA entry
            final short cbData; // number of compressed bytes in this block
            final short cbUncomp; // number of uncompressed bytes in this block
            final byte[] abReserve; // (optional) per-datablock reserved area
            final byte[] ab; // compressed data bytes
            buffer.clear();
            buffer.limit(8);
            Channels.newChannel(is).read(buffer);
            if (buffer.position() < 8)
                break;
            buffer.rewind();
            csum = buffer.getInt();
            cbData = buffer.getShort();
            cbUncomp = buffer.getShort();
            out.printf(fmt0, "CFDATA", i);
            out.printf(fmt2, "csum", csum);
            out.printf(fmt2, "cbData", cbData);
            out.printf(fmt2, "cbUncomp", cbUncomp);
            int skipSize = cbData;
            while (skipSize > 0)
                skipSize -= is.skip(skipSize);
        }
        printEnd(out, "CAB", 0);
    }

    static Date toDate(int mdate, int mtime) {
        final int ftime = (mdate << 16) | mtime & 0xFFFF;
        return new Date(FTIME.toMilliseconds(ftime));
    }

    private byte[] readName(InputStream is, int limit) throws IOException {
        byte[] bytes = new byte[limit];
        int readSize = 0;
        for (int i = 0; i < limit; i++) {
            int b = is.read();
            if (b <= 0)
                break;
            bytes[i] = (byte)(b & 0xFF);
            ++readSize;
        }
        if (readSize < limit)
            return Arrays.copyOf(bytes, readSize);
        else
            return bytes;
    }

}
