package jp.sfjp.armadillo.archive;

import java.io.*;

/**
 * The InputStream for Archive.
 */
public abstract class ArchiveInputStream extends FilterInputStream {

    protected InputStream frontStream;
    protected long remaining;

    private boolean closed;

    public ArchiveInputStream(InputStream is) {
        super(is);
        this.closed = false;
        this.remaining = 0;
    }

    protected final void ensureOpen() throws IOException {
        if (closed)
            throw new IOException("stream closed");
    }

    @Override
    public int read() throws IOException {
        ensureOpen();
        if (frontStream == null)
            return super.read();
        else if (remaining <= 0) {
            assert remaining == 0;
            return -1;
        }
        else {
            int read = frontStream.read();
            if (read == -1)
                return -1;
            --remaining;
            return read;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        ensureOpen();
        if (frontStream == null)
            return super.read(b, off, len);
        else if (remaining <= 0) {
            assert remaining == 0;
            return -1;
        }
        else {
            long requiredLength = Math.min(len, remaining);
            assert requiredLength <= Integer.MAX_VALUE;
            int readLength = frontStream.read(b, off, (int)requiredLength);
            assert readLength != 0 : "Read Zero";
            if (readLength >= 0)
                remaining -= readLength;
            return readLength;
        }
    }

    @Override
    public long skip(long n) throws IOException {
        ensureOpen();
        if (frontStream == null)
            return super.skip(n);
        else if (remaining <= 0) {
            assert remaining == 0;
            return -1;
        }
        else {
            long skipped = frontStream.skip(n);
            if (skipped > 0)
                remaining -= skipped;
            return skipped;
        }
    }

    @Override
    public int available() throws IOException {
        return 0;
    }

    @Override
    public boolean markSupported() {
        return false;
    }

    @Override
    public synchronized void mark(int limit) {
        // not supported
    }

    @Override
    public synchronized void reset() throws IOException {
        throw new IOException("mark/reset not supported");
    }

    @Override
    public void close() throws IOException {
        ensureOpen();
        try {
            super.close();
        }
        finally {
            frontStream = null;
            closed = true;
        }
    }

}
