package jp.sfjp.armadillo.archive;

import java.io.*;
import java.util.*;

public abstract class ArchiveFile implements
                                 Iterable<ArchiveEntry>,
                                 ArchiveCreator,
                                 ArchiveExtractor {

    protected long currentPosition;
    protected boolean opened;
    private boolean closed;

    protected ArchiveFile() {
        this.currentPosition = 0L;
        this.opened = false;
        this.closed = false;
    }

    @Override
    public Iterator<ArchiveEntry> iterator() {
        return new Iterator<ArchiveEntry>() {
            boolean first = true;
            ArchiveEntry entry = null;

            @Override
            public boolean hasNext() {
                try {
                    if (first) {
                        reset();
                        first = false;
                    }
                    entry = nextEntry();
                }
                catch (IOException ex) {
                    handleException(ex);
                }
                assert entry != null;
                return entry != ArchiveEntry.NULL;
            }

            @Override
            public ArchiveEntry next() {
                return entry;
            }

            @Override
            public void remove() {
                try {
                    removeEntry(entry);
                }
                catch (IOException ex) {
                    handleException(ex);
                }
            }

            void handleException(Exception ex) {
                throw new RuntimeException(ex);
            }
        };
    }

    @Override
    public void addEntry(ArchiveEntry entry, File file) throws IOException {
        assert entry.isDirectory() == file.isDirectory();
        if (entry.isDirectory())
            addEntry(entry, null, 0L);
        else {
            FileInputStream fis = new FileInputStream(file);
            try {
                addEntry(entry, fis, file.length());
            }
            finally {
                fis.close();
            }
        }
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        throw new UnsupportedOperationException("ArchiveFile#addEntry");
    }

    public void updateEntry(ArchiveEntry entry, File file) throws IOException {
        assert entry.isDirectory() == file.isDirectory();
        if (entry.isDirectory())
            updateEntry(entry, null, 0);
        else {
            FileInputStream fis = new FileInputStream(file);
            try {
                updateEntry(entry, fis, file.length());
            }
            finally {
                fis.close();
            }
        }
    }

    public void updateEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        removeEntry(entry);
        addEntry(entry, is, length);
    }

    public void removeEntry(ArchiveEntry entry) throws IOException {
        throw new UnsupportedOperationException("ArchiveFile#removeEntry");
    }

    public boolean seek(ArchiveEntry entry) throws IOException {
        reset();
        while (true) {
            ArchiveEntry nextEntry = nextEntry();
            if (nextEntry == null)
                break;
            if (nextEntry.equalsName(entry))
                return true;
        }
        reset();
        return false;
    }

    @Override
    public ArchiveEntry nextEntry() throws IOException {
        throw new UnsupportedOperationException("ArchiveFile#nextEntry");
    }

    @Override
    public long extract(OutputStream os) throws IOException {
        throw new UnsupportedOperationException("ArchiveFile#extract");
    }

    public abstract void open() throws IOException;

    public abstract void reset() throws IOException;

    protected final void ensureOpen() throws IOException {
        if (!opened)
            throw new IOException("file is not opened yet");
        if (closed)
            throw new IOException("file was closed");
    }

    @Override
    public void close() throws IOException {
        currentPosition = 0L;
        closed = true;
    }

}
