package jp.sourceforge.armadillo.zip;

import java.io.*;
import java.nio.channels.*;
import java.util.zip.*;

import jp.sourceforge.armadillo.io.*;

/**
 * ZIP`A[JCu̓̓Xg[B
 */
public final class ZipInputStream extends ArchiveInputStream {

    private ZipHeader header;
    private Inflater inflater;
    private ZipEntry nextEntry;
    private RandomAccessFile raf;
    private long cenOffset;
    private boolean randomAccess;

    /**
     * ZipInputStream̐B
     * ftHgZbggpB
     * @param is InputStream
     */
    public ZipInputStream(InputStream is) {
        this(is, null);
    }

    /**
     * ZipInputStream̐B
     * @param is InputStream
     * @param charsetName Zbg
     *                    ftHgZbggpꍇ <code>null</code> w肷
     */
    public ZipInputStream(InputStream is, String charsetName) {
        super(new RewindableInputStream(is, 4096));
        this.randomAccess = false;
        this.header = new ZipHeader(charsetName);
        this.inflater = new Inflater(true);
    }

    /**
     * ZipInputStream̐B
     * @param zipFile ZIPt@C
     * @throws IOException o̓G[ꍇ 
     */
    public ZipInputStream(File zipFile) throws IOException {
        this(zipFile, null);
    }

    /**
     * ZipInputStream̐B
     * @param zipFile ZIPt@C
     * @param charsetName Zbg
     *                    ftHgZbggpꍇ <code>null</code> w肷
     * @throws IOException o̓G[ꍇ 
     */
    public ZipInputStream(File zipFile, String charsetName) throws IOException {
        super(null);
        RandomAccessFile raf = new RandomAccessFile(zipFile, "r");
        in = new RewindableInputStream(Channels.newInputStream(raf.getChannel()), 4096);
        this.header = new ZipHeader(charsetName);
        this.inflater = new Inflater(true);
        this.cenOffset = 0;
        raf.seek(raf.length() - ZipHeader.LENGTH_END);
        ZipEntry endHeader = header.readEND(in);
        if (endHeader == null) {
            raf.seek(0L);
            this.randomAccess = false;
        } else {
            this.raf = raf;
            this.cenOffset = endHeader.position;
            this.randomAccess = true;
        }
    }

    /**
     * ̃Gg擾B
     * @return ZipEntry ̃Ggꍇ <code>null</code>
     * @throws IOException o̓G[ꍇ
     */
    public ZipEntry getNextEntry() throws IOException {
        ensureOpen();
        if (nextEntry != null) {
            closeEntry();
        }
        ZipEntry entry;
        if (randomAccess) {
            raf.seek(cenOffset);
            entry = header.readCEN(in);
            cenOffset = raf.getFilePointer();
        } else {
            entry = header.read(in);
        }

        if (entry == null) {
            return null;
        }

        if (randomAccess) {
            raf.seek(entry.position);
            header.read(in);
        }
        final int method = entry.method;
        if (method == ZipEntry.DEFLATED) {
            frontStream = new InflaterInputStream(in, inflater, 512);
        } else if (method == ZipEntry.STORED) {
            frontStream = in;
        } else {
            throw new IOException("invalid method: " + method);
        }
        remaining = entry.getSize();
        return nextEntry = entry;
    }

    /**
     * ݂̃GgB
     * @throws IOException o̓G[ꍇ
     */
    public void closeEntry() throws IOException {
        ensureOpen();

        if (!randomAccess && !nextEntry.isDirectory()) {
            // KvΎ̃wb_܂ŃXLbv
            if (remaining > 0 && nextEntry.getCompressedSize() < nextEntry.getSize()) {
                while (remaining > 0) {
                    skip(remaining);
                }
            }
            // inflater̃obt@Ƀf[^cĂꍇ͏߂
            int inflaterRemaining = inflater.getRemaining();
            if (inflaterRemaining > 0) {
                ((RewindableInputStream)in).rewind(inflaterRemaining);
            }
        }

        if (randomAccess) {
            remaining = 0;
        } else if (nextEntry.hasEXT()) {
            // KvEXTwb_ǂݍ
            ZipEntry entry = header.readEXT(in);
            if (entry != null) {
                nextEntry.crc = entry.crc;
                nextEntry.compressedSize = entry.compressedSize;
                nextEntry.size = entry.size;
            }
            remaining = 0;
        } else if (!nextEntry.isDirectory()) {
            long rest = nextEntry.getCompressedSize() - inflater.getTotalIn();
            if (rest != 0) {
                // ǂݍ񂾃TCYkTCYƈقȂꍇ̑Ώ
                if (rest > 0) {
                    in.skip(rest);
                } else {
                    throw new IOException("unexpected state");
                }
            }
        }
        assert remaining == 0 : "remaining: " + remaining;

        // Zbg
        nextEntry = null;
        inflater.reset();
        frontStream = in;
    }

    /* (overridden)
     * @see java.io.FilterInputStream#close()
     */
    public void close() throws IOException {
        try {
            if (randomAccess) {
                raf.close();
            }
        } finally {
            header = null;
            inflater = null;
            nextEntry = null;
            super.close();
        }
    }

}
