package jp.sourceforge.armadillo.zip;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.nio.charset.*;
import java.util.*;
import java.util.zip.*;

import jp.sourceforge.armadillo.*;

/**
 * ZIP`A[JCũwb_B
 * 
 * <p>̃Cu̓dlƂāAŏIXV̓~b`Ƒ݂ɕϊB
 * GgIuWFNgɊi[ĂԂł́A~b`ƂB</p>
 * <p>GR[fBÓAftHgłJavãftHgGR[fBOKpB
 * Zbgw肳ꂽꍇ́A̕ZbgœǂݏB</p>
 * 
 * <h3>wb_̎dlɂ</h3>
 * <p>ɋLڂẮAȎdlł͂ȂƂɒӁB</p>
 * <ul>
 * <li>wb_̐擪́Awb_̎ނ4oCg̎ʎq(SIGNATURE)ɂȂĂB</li>
 * <li>f[^̒OȊOɔzuwb_(EXT,CEN,END)݂B</li>
 * <li>l2oCg܂4oCggGfBAŊi[B</li>
 * <li>ŏIXV́ADOS`Ŋi[B</li>
 * </ul>
 */
public final class ZipHeader {

    static final int SIGN_LOC = 0x04034B50;
    static final int SIGN_EXT = 0x08074B50;
    static final int SIGN_CEN = 0x02014B50;
    static final int SIGN_END = 0x06054B50;
    static final int LENGTH_LOC = 30;
    static final int LENGTH_EXT = 16;
    static final int LENGTH_CEN = 46;
    static final int LENGTH_END = 22;

    private static final short SHORT_0 = (short)0;

    private ByteBuffer buffer;
    private String charsetName;
    private boolean charsetEnables;
    private List entries;
    private long offset;

    /**
     * ZipHeader̐B
     * ftHgZbggpB
     */
    public ZipHeader() {
        this(null);
    }

    /**
     * ZipHeader̐B
     * @param charsetName Zbg
     *                    ftHgZbggpꍇ <code>null</code> w肷
     */
    public ZipHeader(String charsetName) {
        this.buffer = ByteBuffer.allocate(LENGTH_CEN).order(ByteOrder.LITTLE_ENDIAN);
        this.charsetName = charsetName;
        this.charsetEnables = charsetName != null && Charset.isSupported(charsetName);
        this.entries = new ArrayList();
        this.offset = 0L;
    }

    /**
     * wb_(LOC)ǂݍށB
     * @param is InputStream
     * @return ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public ZipEntry read(InputStream is) throws IOException {
        buffer.clear();
        buffer.limit(LENGTH_LOC);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0) {
            return null;
        }
        buffer.rewind();
        int signature = buffer.getInt();
        if (signature == SIGN_CEN || signature == SIGN_END) {
            return null;
        } else if (signature != SIGN_LOC) {
            throw new ZipException("invalid LOC header: signature="
                                   + Integer.toHexString(signature));
        }
        ZipEntry entry = new ZipEntry();
        entry.version = buffer.getShort();
        entry.option = buffer.getShort();
        entry.method = buffer.getShort();
        entry.lastModified = Utilities.DosTime.toMilliSeconds(buffer.getInt());
        entry.crc = buffer.getInt() & 0xFFFFFFFFL;
        entry.compressedSize = buffer.getInt() & 0xFFFFFFFFL;
        entry.size = buffer.getInt() & 0xFFFFFFFFL;
        short nameLength = buffer.getShort();
        short extraLength = buffer.getShort();
        byte[] nameBuffer = new byte[nameLength];
        if (is.read(nameBuffer) != nameLength) {
            throw new ZipException("invalid LOC header (name length)");
        }
        entry.name = (charsetEnables)
                ? new String(nameBuffer, charsetName)
                : new String(nameBuffer);
        if (extraLength > 0) {
            if (is.skip(extraLength) != extraLength) {
                throw new ZipException("invalid LOC header (extra length)");
            }
        }
        return entry;
    }

    /**
     * EXTwb_ǂݍށB
     * @param is InputStream
     * @return ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public ZipEntry readEXT(InputStream is) throws IOException {
        buffer.clear();
        buffer.limit(LENGTH_EXT);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0) {
            return null;
        }
        buffer.rewind();
        int signature = buffer.getInt();
        if (signature != SIGN_EXT) {
            throw new ZipException("invalid EXT header: signature="
                                   + Integer.toHexString(signature));
        }
        ZipEntry entry = new ZipEntry();
        entry.crc = buffer.getInt() & 0xFFFFFFFFL;
        entry.compressedSize = buffer.getInt() & 0xFFFFFFFFL;
        entry.size = buffer.getInt() & 0xFFFFFFFFL;
        return entry;
    }

    /**
     * CENwb_ǂݍށB
     * @param is InputStream
     * @return ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public ZipEntry readCEN(InputStream is) throws IOException {
        buffer.clear();
        buffer.limit(LENGTH_CEN);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0) {
            return null;
        }
        buffer.rewind();
        int signature = buffer.getInt();
        if (signature == SIGN_END) {
            return null;
        } else if (signature != SIGN_CEN) {
            throw new ZipException("invalid EXT header: signature="
                                   + Integer.toHexString(signature));
        }
        ZipEntry entry = new ZipEntry();
        entry.version = buffer.getShort();
        buffer.getShort(); // needver
        entry.option = buffer.getShort();
        entry.method = buffer.getShort();
        entry.lastModified = Utilities.DosTime.toMilliSeconds(buffer.getInt());
        entry.crc = buffer.getInt() & 0xFFFFFFFFL;
        entry.compressedSize = buffer.getInt() & 0xFFFFFFFFL;
        entry.size = buffer.getInt() & 0xFFFFFFFFL;
        short nameLength = buffer.getShort();
        short extraLength = buffer.getShort();
        buffer.getShort(); // comment
        buffer.getShort(); // disknum
        buffer.getShort(); // inattr
        buffer.getInt(); // outattr
        entry.position = buffer.getInt();
        byte[] nameBuffer = new byte[nameLength];
        if (is.read(nameBuffer) != nameLength) {
            throw new ZipException("invalid CEN header (name length)");
        }
        entry.name = (charsetEnables)
                ? new String(nameBuffer, charsetName)
                : new String(nameBuffer);
        if (extraLength > 0) {
            if (is.skip(extraLength) != extraLength) {
                throw new ZipException("invalid CEN header (extra length)");
            }
        }
        return entry;
    }

    /**
     * ENDwb_ǂݍށB
     * @param is InputStream
     * @return ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public ZipEntry readEND(InputStream is) throws IOException {
        buffer.clear();
        buffer.limit(LENGTH_END);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0) {
            return null;
        }
        buffer.rewind();
        int signature = buffer.getInt();
        if (signature != SIGN_END) {
            return null;
        }
        ZipEntry entry = new ZipEntry();
        buffer.position(16);
        entry.position = buffer.getInt();
        return entry;
    }

    /**
     * wb_(LOC)ށB
     * @param os OutputStream
     * @param entry ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public void write(OutputStream os, ZipEntry entry) throws IOException {
        entry.position = (int)offset;
        buffer.clear();
        buffer.putInt(SIGN_LOC);
        buffer.putShort(entry.version);
        buffer.putShort(entry.option);
        buffer.putShort((short)entry.getMethod());
        buffer.putInt(Utilities.DosTime.getValue(entry.lastModified));
        if (entry.hasEXT()) {
            buffer.putInt(0);
            buffer.putInt(0);
            buffer.putInt(0);
        } else {
            buffer.putInt((int)entry.crc);
            buffer.putInt((int)entry.compressedSize);
            buffer.putInt((int)entry.size);
            assert entry.compressedSize >= 0;
            offset += entry.compressedSize;
        }
        byte[] nameBytes = toBytes(entry.name);
        short nameLength = (short)nameBytes.length;
        if (nameLength > Short.MAX_VALUE) {
            throw new ZipException("too long name: length=" + nameLength);
        }
        short extraLength = 0;
        buffer.putShort(nameLength);
        buffer.putShort(extraLength);
        assert buffer.position() == LENGTH_LOC;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.write(nameBytes);
        os.flush();
        entries.add(entry);
        offset += LENGTH_LOC + nameLength + extraLength;
    }

    /**
     * EXTwb_ށB
     * @param os OutputStream
     * @param entry ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public void writeEXT(OutputStream os, ZipEntry entry) throws IOException {
        buffer.clear();
        buffer.putInt(SIGN_EXT);
        buffer.putInt((int)entry.crc);
        buffer.putInt((int)entry.compressedSize);
        buffer.putInt((int)entry.size);
        assert buffer.position() == LENGTH_EXT;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.flush();
        offset += LENGTH_EXT;
        assert entry.hasEXT();
        assert entry.compressedSize >= 0;
        offset += entry.compressedSize;
    }

    /**
     * CENwb_ށB
     * @param os OutputStream
     * @param entry ZipEntry
     * @throws IOException o̓G[ꍇ
     */
    public void writeCEN(OutputStream os, ZipEntry entry) throws IOException {
        buffer.clear();
        buffer.putInt(SIGN_CEN);
        buffer.putShort(entry.version);
        buffer.putShort(entry.version);
        buffer.putShort(entry.option);
        buffer.putShort((short)entry.method);
        buffer.putInt(Utilities.DosTime.getValue(entry.lastModified));
        buffer.putInt((int)entry.crc);
        buffer.putInt((int)entry.compressedSize);
        buffer.putInt((int)entry.size);
        byte[] nameBytes = toBytes(entry.name);
        short nameLength = (short)nameBytes.length;
        if (nameLength > Short.MAX_VALUE) {
            throw new ZipException("too long name: length=" + nameLength);
        }
        buffer.putShort(nameLength);
        buffer.putShort(SHORT_0); // extra
        buffer.putShort(SHORT_0); // comment
        buffer.putShort(SHORT_0); // disknum
        buffer.putShort(SHORT_0); // inattr
        buffer.putInt(0); // outattr
        buffer.putInt(entry.position);
        assert buffer.position() == LENGTH_CEN;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.write(nameBytes);
        os.flush();
        offset += LENGTH_CEN + nameLength;
    }

    /**
     * ENDwb_ށB
     * @param os OutputStream
     * @throws IOException o̓G[ꍇ 
     */
    public void writeEND(OutputStream os) throws IOException {
        long startPosition = offset;
        for (int i = 0, n = entries.size(); i < n; i++) {
            ZipEntry entry = (ZipEntry)entries.get(i);
            writeCEN(os, entry);
        }
        long dirsize = offset - startPosition;
        assert startPosition >= 0 && startPosition <= Integer.MAX_VALUE;
        assert dirsize >= 0 && dirsize <= Integer.MAX_VALUE;
        buffer.clear();
        buffer.putInt(SIGN_END);
        buffer.putShort(SHORT_0); // disknum
        buffer.putShort(SHORT_0); // startdisknum
        buffer.putShort((short)entries.size()); // diskdirentry
        buffer.putShort((short)entries.size()); // direntry
        buffer.putInt((int)dirsize); // dirsize
        buffer.putInt((int)startPosition); // startpos
        buffer.putShort(SHORT_0); // commentlen
        assert buffer.position() == LENGTH_END;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.flush();
        offset += LENGTH_END;
    }

    /**
     * o^ꂽZipEntry폜B
     */
    public void clearEntries() {
        entries.clear();
        offset = 0;
    }

    /**
     * oCgɕϊB
     * @param s 
     * @return oCg
     * @throws UnsupportedEncodingException GR[hT|[gĂȂꍇ
     */
    private byte[] toBytes(String s) throws UnsupportedEncodingException {
        if (charsetEnables) {
            return s.getBytes(charsetName);
        } else {
            return s.getBytes();
        }
    }

}
