package jp.sourceforge.armadillo;

import java.io.*;
import java.util.zip.*;

/**
 * t@CA[JCuB
 */
public final class FileArchiver implements Archiver {

    private Archiver archiver;
    private PrintWriter out;
    private boolean verbose;

    /**
     * FileArchiver̐B
     * @param dst o͐t@C
     * @throws IOException o̓G[ꍇ 
     */
    public FileArchiver(File dst) throws IOException {
        this(dst, ArchiveType.getType(dst));
    }

    /**
     * FileArchiver̐B
     * @param dst o͐t@C
     * @param type A[JCu^Cv
     * @throws IOException o̓G[ꍇ 
     */
    public FileArchiver(File dst, ArchiveType type) throws IOException {
        //        if (dst.exists()) {
        //            throw new IOException("file already exists");
        //        }
        OutputStream os = new FileOutputStream(dst);
        if (type == ArchiveType.TAR) {
            archiver = new TarArchiver(os);
        } else if (type == ArchiveType.TGZ) {
            archiver = new TarArchiver(new GZIPOutputStream(os));
        } else if (type == ArchiveType.ZIP) {
            archiver = new ZipArchiver(os);
        } else if (type == ArchiveType.LZH) {
            archiver = new LzhArchiver(os);
        } else {
            throw new IllegalStateException("ArchiverType: " + type);
        }
        this.out = Utilities.getDefaultPrintWriter();
    }

    /**
     * out(o̓Xg[)̐ݒB
     * @param out o̓Xg[
     */
    public void setOut(PrintWriter out) {
        if (out == null) {
            throw new IllegalArgumentException("out is null");
        }
        this.out = out;
    }

    /**
     * verbose̎擾B
     * @return verbose
     */
    public boolean isVerbose() {
        return verbose;
    }

    /**
     * verbose̐ݒB
     * @param verbose verbose
     */
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    /**
     * A[JCuB
     * @param src Ώۃt@C
     * @throws IOException o̓G[ꍇ 
     */
    public void archive(File src) throws IOException {
        archive(src, getRootPath(src));
    }

    /**
     * A[JCuB
     * @param src Ώۃt@C
     * @param rootPath Ώۂ̃[gpX
     * @throws IOException o̓G[ꍇ
     */
    private void archive(File src, String rootPath) throws IOException {
        boolean isDirectory = src.isDirectory();
        String entryName = getEntryName(src, rootPath, isDirectory);
        ArchiveEntry entry = new ArchiveEntry(entryName);
        try {
            entry.setLastModified(src.lastModified());
            if (isDirectory) {
                addEntry(entry, src);
                File[] files = src.listFiles();
                for (int i = 0, n = files.length; i < n; i++) {
                    archive(files[i], rootPath);
                }
            } else {
                addEntry(entry, src);
            }
        } catch (IOException ex) {
            IOException exception = new IOException(ex.getMessage() + " : entry=" + entryName);
            exception.setStackTrace(ex.getStackTrace());
            throw exception;
        }
        if (!entry.isAdded()) {
            return;
        }
        if (verbose) {
            long size = entry.getSize();
            long compressedSize = entry.getCompressedSize();
            assert compressedSize >= 0 && size >= 0;
            out.print("appending  " + src);
            if (size == 0 || compressedSize == size) {
                out.print(" (STORED)");
            } else {
                int rate = compressedSize == size ? 100 : (int)(compressedSize * 100 / size);
                out.print(" (" + rate + "%, " + size + " => " + compressedSize + ")");
            }
            out.println();
        }
    }

    /* (overridden)
     * @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.File)
     */
    public void addEntry(ArchiveEntry ae, File file) throws IOException {
        archiver.addEntry(ae, file);
    }

    /* (overridden)
     * @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.InputStream, long)
     */
    public void addEntry(ArchiveEntry ae, InputStream is, long length) throws IOException {
        archiver.addEntry(ae, is, length);
    }

    /**
     * RootPath̎擾B
     * @param file t@C
     * @return RootPath
     */
    private static String getRootPath(File file) {
        String parent = normalizePath(file.getAbsoluteFile().getParent());
        if (!parent.endsWith(String.valueOf('/'))) {
            parent += '/';
        }
        return parent;
    }

    /**
     * Gg̎擾B
     * @param file t@C
     * @param rootPath [gpX
     * @param isDirectory fBNg̏ꍇ <code>true</code> 
     * @return Gg
     */
    private static String getEntryName(File file, String rootPath, boolean isDirectory) {
        String entryName;
        String absolutePath = normalizePath(file.getAbsolutePath());
        if (absolutePath.startsWith(rootPath)) {
            entryName = absolutePath.substring(rootPath.length());
            if (file.isDirectory() && !entryName.endsWith(String.valueOf('/'))) {
                entryName += '/';
            }
        } else {
            entryName = file.getName();
        }
        return entryName;
    }

    /**
     * pXZp[^𐳋KB
     * @param path pX
     * @return KꂽpX
     */
    private static String normalizePath(String path) {
        return path.replace('\\', '/');
    }

    /**
     * IB
     * @throws IOException o̓G[ꍇ
     */
    public void close() throws IOException {
        archiver.close();
    }

}
